(*****************************************************************************)
(*                                                                           *)   
(*  Special thanks to:                                                       *)
(*  - David Moretti (dave@rimini.com);                                       *)
(*  - Alexey Gorbunov - AlGo (algo_kov@chat.ru);                             *)
(*  - Paul Barry (barryp@itcarlow.ie)                                        *)
(*                                                                           *)
(*****************************************************************************)

unit ZREscape;

interface

{$I ZRDefine.inc}

uses
  Windows, Messages,                               // WinAPI
  SysUtils,                                        // Delphi RTL
  Classes,                                         // Delphi VCL
  ZRStream;                                        // ZReport

type
  { Esc-codes }
  TZREscapeCode = (ecReset, ecFormFeed, ecPica, ecElite, ecCondensedOn, ecCondensedOff,
                   ecBoldOn, ecBoldOff, ecItalicOn, ecItalicOff, ecUnderlineOn, ecUnderlineOff,
                   ecSuperScriptOn, ecSuperScriptOff, ecSubScriptOn, ecSubScriptOff,
                   ecReportStart, ecReportFinish, ecPageStart, ecPageFinish);
  TZREscapeCodes = set of TZREscapeCode;

  { Esc-models }
  TZREscapeModel = (emCustom, emNone, emCannonF60, emCannonLaser,
                    emEpson, emHPDeskjet, emHPLaserjet, emHPThinkjet,
                    emIBMColorJet, emIBMPCGraphics, emIBMProprinter,
                    emNEC3500, emNECPinwriter);

  TZREscapeString     = String[32];
  TZREscapeStrings    = array[TZREscapeCode] of TZREscapeString;
  TZREscapeSpecifiers = array[TZREscapeCode] of Char;

const
  EscapeModelNames: array[TZREscapeModel] of String[64] = (
    '<custom>',
    '<none>',
    'Cannon F60',
    'Cannon Laser',
    'Epson',
    'HP Deskjet',
    'HP Laserjet',
    'HP Thinkjet',
    'IBM ColorJet',
    'IBM PCGraphics',
    'IBM Proprinter',
    'NEC 3500',
    'NEC Pinwriter'
  );

type
  { TZREscapes }
  TZREscapes = class(TPersistent)
  private
    fModel : TZREscapeModel;
    fValues: TZREscapeStrings;
    procedure SetModel(Value: TZREscapeModel);
    function  GetValue(Index: TZREscapeCode): String;
    procedure SetValue(Index: TZREscapeCode; const Value: String);
    procedure ReadValues(Reader: TReader);
    procedure WriteValues(Writer: TWriter);
  protected
    procedure DefineProperties(Filer: TFiler); override;
  public
    constructor Create;
    procedure Assign(Source: TPersistent); override;
    property Values : TZREscapeStrings read fValues;
  published
    property Model: TZREscapeModel read fModel write SetModel default emEpson;
    property BoldOn        : String index ecBoldOn         read GetValue write SetValue stored False;
    property BoldOff       : String index ecBoldOff        read GetValue write SetValue stored False;
    property CondensedOn   : String index ecCondensedOn    read GetValue write SetValue stored False;
    property CondensedOff  : String index ecCondensedOff   read GetValue write SetValue stored False;
    property Elite         : String index ecElite          read GetValue write SetValue stored False;
    property FormFeed      : String index ecFormFeed       read GetValue write SetValue stored False;
    property ItalicOn      : String index ecItalicOn       read GetValue write SetValue stored False;
    property ItalicOff     : String index ecItalicOff      read GetValue write SetValue stored False;
    property Pica          : String index ecPica           read GetValue write SetValue stored False;
    property Reset         : String index ecReset          read GetValue write SetValue stored False;
    property SubScriptOn   : String index ecSubScriptOn    read GetValue write SetValue stored False;
    property SubScriptOff  : String index ecSubScriptOff   read GetValue write SetValue stored False;
    property SuperScriptOn : String index ecSuperScriptOn  read GetValue write SetValue stored False;
    property SuperScriptOff: String index ecSuperScriptOff read GetValue write SetValue stored False;
    property UnderlineOn   : String index ecUnderlineOn    read GetValue write SetValue stored False;
    property UnderlineOff  : String index ecUnderlineOff   read GetValue write SetValue stored False;
    property ReportStart   : String index ecReportStart    read GetValue write SetValue stored False;
    property ReportFinish  : String index ecReportFinish   read GetValue write SetValue stored False;
    property PageStart     : String index ecPageStart      read GetValue write SetValue stored False;
    property PageFinish    : String index ecPageFinish     read GetValue write SetValue stored False;
  end;

  { TZREscapeStream }
  TZREscapeStream = class(TZStringStream)
  private
    fEscapes : TZREscapeStrings;
  public
    constructor Create(aStream: TStream; aEscapes: TZREscapeStrings);
    function  ReadString(var Value: String): Boolean; override;
    procedure WriteString(const Value: String); override;
    property Escapes : TZREscapeStrings read fEscapes;
  end;

type
  TZREscapeStyle = (esCondensed, esBold, esItalic, esUnderline, esSuperScript, esSubScript);
  TZREscapeStyles  = set of TZREscapeStyle;

{ Utility functions and classes }
function EscapeLength(const S: String): Integer;
function EscapeStyles(const S: String; Pos: Integer): TZREscapeStyles;
function EscapeFormat(const S: String; Codes: TZREscapeCodes; Styles: TZREscapeStyles): String;
function EscapeDeformat(const S: String): String;

function EscapePadLeft  (const S: String; N: Integer): String;
function EscapePadRight (const S: String; N: Integer): String;
function EscapePadCenter(const S: String; N: Integer): String;

function EscapeCopy(const S: String; Index, Count: Integer): String;
function EscapeInsert(const Pattern, S: String; Pos: Integer): String;
function EscapeDelete(const S: String; Index, Count: Integer): String;
function EscapeStuff(const Pattern, S: String; Pos: Integer): String;

type
  TZREscapeTokenizer = class(TObject)
  private
    fLine  : String;
    fCodes : TZREscapeCodes;
    fStyles: TZREscapeStyles;
    fStart : Integer;
    fFinish: Integer;
    fToken : String;
    procedure SetLine(const Value: String);
  public
    constructor Create;
    function EOL: Boolean;
    function NextToken: String;
    property Line  : String read fLine write SetLine;
    property Codes : TZREscapeCodes read fCodes;
    property Styles: TZREscapeStyles read fStyles;
    property Token : String read fToken;
  end;

const
  EscapeChar: Char = '\';
  EscapeSpecifiers: TZREscapeSpecifiers =
    ('@', '=', 'P', 'E', 'C', 'c', 'B', 'b', 'I', 'i', 'U', 'u', 'H', 'h', 'L', 'l', '<', '>', '{', '}');

implementation

uses
  ZRConst, ZRStrUtl;                                // ZReport

type
  TZRStyleCodes = record
    On, Off: TZREscapeCode;
  end;

const
  StyleCodes : array[TZREscapeStyle] of TZRStyleCodes = (
{esCondensed  } ( On: ecCondensedOn  ; Off: ecCondensedOff   ),
{esBold       } ( On: ecBoldOn       ; Off: ecBoldOff        ),
{esItalic     } ( On: ecItalicOn     ; Off: ecItalicOff      ),
{esUnderline  } ( On: ecUnderlineOn  ; Off: ecUnderlineOff   ),
{esSuperScript} ( On: ecSuperScriptOn; Off: ecSuperScriptOff ),
{esSubScript  } ( On: ecSubScriptOn  ; Off: ecSubScriptOff   ) );

var
  CharIsCodeMap      : array [Char] of Boolean;                 //   esc-
  CharToCodeMap      : array [Char] of TZREscapeCode;           //   

  CodeIsSwitchOnMap  : array [TZREscapeCode] of Boolean;        //    
  CodeIsSwitchOffMap : array [TZREscapeCode] of Boolean;        //    
  CodeToStyleMap     : array [TZREscapeCode] of TZREscapeStyle; //   

{!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!}
{!!!                       Utility functions and classes                    !!!}
{!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!}

//     S    Pos esc-?
function CharIsCode(const S: String; Pos: Integer): Boolean;
begin
  Result:= (Pos < length(S)) and (S[Pos] = EscapeChar) and CharIsCodeMap[S[Pos+1]];
end;

//     esc-
function CodeIsStyle(Code: TZREscapeCode; var Style: TZREscapeStyle; var On: Boolean): Boolean;
begin
  Result := (CodeIsSwitchOnMap[Code] or CodeIsSwitchOffMap[Code]);
  if Result then begin
    Style := CodeToStyleMap[Code];
    On    := CodeIsSwitchOnMap[Code];
  end;
end;

//     
function CodesToStyles(Codes: TZREscapeCodes): TZREscapeStyles;
var
  c: TZREscapeCode;
  s: TZREscapeStyle;
  o: Boolean;
begin
  Result := [];
  for c := Low(TZREscapeCode) to High(TZREscapeCode) do
    if (c in Codes) and CodeIsStyle(c, s, o) then begin
      if o then Include(Result, s) else Exclude(Result, s);
    end;
end;

//  esc-   S
function EscapeCount(const S: String): Integer;
var
  i: Integer;
begin
  Result:= 0;
  i     := 1;
  while i <= length(S) do begin
    if CharIsCode(S, i) then begin
      Inc(i);
      Inc(Result);
    end;
    Inc(i);
  end;
end;

//   S   esc-
function EscapeLength(const S: String): Integer;
begin
  Result:= length(S) - 2 * EscapeCount(S);
end;

//    
//  S    EscPos
// (  esc-)
function EscapePos(const S: String; EscPos: Integer): Integer;
var
  l: Integer;
  i: Integer;
  p: Integer;
begin
  Result:= EscPos;
  l     := length(S);
  i     := 1;
  p     := 1;
  while (i < l) and (p < EscPos) do begin
    if CharIsCode(S, i) then begin
      Inc(Result, 2);
      Inc(i);
    end else
      Inc(p);
    Inc(i);
  end;
  while (i < l) and CharIsCode(S, i) do begin
    Inc(Result, 2);
    Inc(i, 2);
  end;
end;



//    esc-   S    Pos
function EscapeCodes(const S: String; Pos: Integer): TZREscapeCodes;
var
  i: Integer;
  p: Integer;
  C: TZREscapeCode;
  E: TZREscapeStyle;
  O: Boolean;
begin
  Result:= [];
  i:= 1;
  p:= EscapePos(S, Pos);
  while (i < length(S)) and (i < p) do begin
    if CharIsCode(S, i) then begin
        Inc(i);
        C := CharToCodeMap[S[i]];
        if CodeIsStyle(C, E, O) then begin
          if O then begin
            Include(Result, StyleCodes[E].On );
            Exclude(Result, StyleCodes[E].Off);
          end else begin
            Exclude(Result, StyleCodes[E].On );
            Include(Result, StyleCodes[E].Off);
          end;
        end else
          Include(Result, C);
      end;
    Inc(i);
  end;
end;


//    S    Pos
function EscapeStyles(const S: String; Pos: Integer): TZREscapeStyles;
begin
  Result := CodesToStyles(EscapeCodes(S, Pos));
end;
{var
  i: Integer;
  p: Integer;
  C: TZREscapeCode;
  E: TZREscapeStyle;
  O: Boolean;
begin
  Result:= [];
  i:= 1;
  p:= EscapePos(S, Pos);
  while (i <= length(S)) and (i <= p) do begin
    if CharIsCode(S, i) then begin
        Inc(i);
        C := CharToCodeMap[S[i]];
        if CodeIsStyle(C, E, O) then begin
          if O then
            Include(Result, E)
          else
            Exclude(Result, E);
        end;
      end;
    Inc(i);
  end;
end;}


function EscapePrefix(const Codes: TZREscapeCodes; const Styles: TZREscapeStyles): String;
var
  C : TZREscapeCode;
  E : TZREscapeStyle;
  O : Boolean;
begin
  Result := '';
  for C:= Low(TZREscapeCode) to High(TZREscapeCode) do
    if (C in Codes) and not (CodeIsStyle(C, E, O) and (E in Styles)) then Result := Result + EscapeChar + EscapeSpecifiers[C];
  for E:= High(TZREscapeStyle) downto Low(TZREscapeStyle) do
    if E in Styles then Result := Result + EscapeChar + EscapeSpecifiers[StyleCodes[E].On];
end;

function EscapeSuffix(const Styles: TZREscapeStyles): String; 
var
  E : TZREscapeStyle;
begin
  Result := '';
  for E:= Low(TZREscapeStyle) to High(TZREscapeStyle) do
    if E in Styles then Result := Result + EscapeChar + EscapeSpecifiers[StyleCodes[E].Off];
end;

//   S esc-    Styles
function EscapeFormat(const S: String; Codes: TZREscapeCodes; Styles: TZREscapeStyles): String;
(*var
  C      : TZREscapeCode;
  E      : TZREscapeStyle;
  Start,
  Finish : String[80];*)
begin
  (*Start := EscapePrefix(Codes, Styles);
  Finish:= EscapeSuffix(Styles);
  for C:= Low(TZREscapeCode) to High(TZREscapeCode) do
    if (C in Codes) {and not CodeIsStyle(C, E, O)} then Start := Start + EscapeChar + EscapeSpecifiers[C];
  for E:= High(TZREscapeStyle) downto Low(TZREscapeStyle) do
    if E in Styles then Start := Start + EscapeChar + EscapeSpecifiers[StyleCodes[E].On];
  for E:= Low(TZREscapeStyle) to High(TZREscapeStyle) do
    if E in Styles then Finish := Finish + EscapeChar + EscapeSpecifiers[StyleCodes[E].Off];
  Result:= Start + S + Finish;*)
  Result:= EscapePrefix(Codes, Styles) + S + EscapeSuffix(Styles);
end;

//     esc-
function EscapeDeformat(const S: String): String;
var
  i: Integer;
begin
  Result:= S;
  i     := 1;
  while i <= length(Result) do
    if CharIsCode(Result, i) then
      Delete(Result, i, 2)
    else
      Inc(i);
end;


//    esc- ( )
function EscapeNormalize(const S: String): String;
var
  Codes  : TZREscapeCodes;
  Styles : TZREscapeStyles;
  E      : TZREscapeStyle;
  C      : TZREscapeCode;
  O      : Boolean;
  i      : Integer;
begin
  Result := S;
  Codes  := [];
  Styles := [];
  i      := 1;
  while i < length(Result) do
    if CharIsCode(Result, i) then begin
      C := CharToCodeMap[Result[i+1]];
      if CodeIsStyle(C, E, O) then begin
        if (E in Styles) then
          if O then
            Delete(Result, i, 2)
          else begin
            Exclude(Styles, E);
            Inc(i, 2);
          end
        else
          if O then begin
            Include(Styles, E);
            Inc(i, 2);
          end else
            Delete(Result, i, 2);
      end else
      if C in Codes then
        Delete(Result, i, 2)
      else begin
        Include(Codes, C);
        Inc(i, 2);
      end;
    end else
      Inc(i);
end;

function EscapePadLeft(const S: String; N: Integer): String;
var
  EL: Integer;
begin
  Result:= EscapeNormalize(S);
  EL    := EscapeLength(Result);
  if EL < N then Result:= Space(N-EL) + Result;
end;

function EscapePadRight(const S: String; N: Integer): String;
var
  EL: Integer;
begin
  Result:= EscapeNormalize(S);
  EL    := EscapeLength(Result);
  if EL < N then Result:= Result + Space(N-EL);
end;

function EscapePadCenter(const S: String; N: Integer): String;
var
  EL: Integer;
begin
  Result:= EscapeNormalize(S);
  EL    := EscapeLength(Result);
  if EL < N then begin
    Result := Space((N - EL) div 2) + S;
    Result := Result + Space(N - EscapeLength(Result));
  end;
end;



//   esc- S ,  
//   Index     Count
function EscapeCopy(const S: String; Index, Count: Integer): String;
{var
  si, ei : Integer;
  SC, EC : TZREscapeCodes;
  SE, EE : TZREscapeStyles;
  C      : TZREscapeCode;
  E      : TZREscapeStyle;
begin
  Result:= '';
  if Index > EscapeLength(S) then Exit;
  SetLength(Result, 255);
  si:= EscapePos(S, Index);
  ei:= EscapePos(S, Index+Count-1);
  SC:= EscapeCodes(S, Index);
  EC:= EscapeCodes(S, Index+Count-1);
  SE:= EscapeStyles(S, Index);
  EE:= EscapeStyles(S, Index+Count-1);
  Result := EscapeNormalize(EscapeFormat(copy(S, si, ei-si), SC, []));
end;}
{var
  Buffer  : array[0..1023] of Char;
  si, l, i: Integer;
begin
  Result:= '';
  if Index > EscapeLength(S) then Exit;
  si:= EscapePos(S, Index);
  i := 0;
  l := length(S);
  while (si+i <= l) and (Count > 0) do begin
    if CharIsCode(S, si+i) then begin
      Buffer[i]:= S[si+i];
      Inc(i);
    end else begin
      Dec(Count);
    end;
    Buffer[i] := S[si+i];
    Inc(i);
  end;
  Buffer[i] := #0;
  Result:= EscapeNormalize(Buffer);
end;}
{var
  Buffer  : array[0..1023] of Char;
  si, l, i: Integer;
begin
  Result:= '';
  if Index > EscapeLength(S) then Exit;
  si:= EscapePos(S, Index);
  while (si > 1) and CharIsCode(S, si-2) do Dec(si, 2);
  i := 0;
  l := length(S);
  while (si+i <= l) and (Count > 0) do begin
    if CharIsCode(S, si+i) then begin
      Buffer[i]:= S[si+i];
      Inc(i);
    end else begin
      Dec(Count);
    end;
    Buffer[i] := S[si+i];
    Inc(i);
  end;
  Buffer[i] := #0;
  Result:= EscapeNormalize(Buffer);
end;}
var
  EscLen : Integer;
  StartI : Integer;
  EndI   : Integer;
  StartC : TZREscapeCodes;
  EndS   : TZREscapeStyles;
begin
  EscLen  := EscapeLength(S);
  if Index > EscLen then
    Result:= ''
  else begin
    if Index+Count-1 > EscLen then Count := EscLen-Index+1;
    StartI := EscapePos(S, Index);
    StartC := EscapeCodes(S, Index);
    EndI   := EscapePos(S, Index+Count-1);
    EndS   := EscapeStyles(S, Index+Count-1);
    Result := {EscapeNormalize}(
                EscapePrefix(StartC, []) +
                copy(S, StartI, EndI-StartI+1) +
                EscapeSuffix(EndS) );
  end;
end;


//   esc- S ,  
//   Index     Count
function EscapeDelete(const S: String; Index, Count: Integer): String;
var
  Start,
  Finish: String;
begin
  Result := EscapePadRight(S, Index-1);
  Start  := EscapeCopy(Result, 1, Index-1);
  Finish := EscapeCopy(Result, Index+Count, length(Result));
  Result := EscapeNormalize(Start + Finish);
end;
{
var
  si, l, i: Integer;
begin
  Result := S;
  if Index > EscapeLength(S) then Exit;
  si:= EscapePos(S, Index);
  i := 0;
  l := length(S);
  while (si + i <= l) and (Count > 0) do begin
    if CharIsCode(S, si+i) then begin
      Inc(i, 2);
    end else begin
      Delete(Result, si+i, 1);
      Inc(i);
      Dec(Count);
    end;
  end;
  Result := EscapeNormalize(Result);
end;
}



//  esc- Pattern  esc- S ,
//     Pos
function EscapeInsert(const Pattern, S: String; Pos: Integer): String;
(*var
  SI: Integer;
  SF: TZREscapeCodes;
  C : TZREscapeCode;
  SS: String;
begin
  if Pos > EscapeLength(S) then
    Result := EscapePadRight(S, Pos)
  else
    Result := S;
  SI := EscapePos(Result, Pos);
  SF := EscapeCodes(Result, Pos);
  SS := Pattern; {EscapeNormalize(Pattern);}
  for C:= Low(TZREscapeCode) to High(TZREscapeCode) do
    if C in SF then SS:= EscapeChar + EscapeSpecifiers[C] + SS;
  for C:= High(TZREscapeCode) downto Low(TZREscapeCode) do
    if C in SF then SS:= SS + EscapeChar + EscapeSpecifiers[C];
  Insert(SS, Result, SI);
  Result := EscapeNormalize(Result);
end;*)
var
  Start,
  Finish: String;
begin
  Result := EscapePadRight(S, Pos-1);
  Start  := EscapeCopy(Result, 1, Pos-1);
  Finish := EscapeCopy(Result, Pos, length(Result));
  Result := EscapeNormalize(Start + Pattern + Finish);
end;



//  esc- Pattern  esc- S
// ,     Pos
function EscapeStuff(const Pattern, S: String; Pos: Integer): String;
begin
  Result := EscapePadRight(S, Pos-1);
  Result := EscapeDelete(Result, Pos, EscapeLength(Pattern));
  Result := EscapeInsert(Pattern, Result, Pos);
end;

{!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!}
{!!!                            TZREscapeTokenizer                          !!!}
{!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!}

constructor TZREscapeTokenizer.Create;
begin
  inherited;
  SetLine('');
end;

procedure TZREscapeTokenizer.SetLine(const Value: String);
begin
  fLine   := Value;
  fCodes  := [];
  fToken  := '';
  fStart  := 1;
  fFinish := 1;
  NextToken;
end;

function TZREscapeTokenizer.EOL: Boolean;
begin
  Result := fStart > EscapeLength(Line);
end;

function TZREscapeTokenizer.NextToken: String;
var
  si, l, i: Integer;
begin
  if EOL then
    fToken := ''
  else begin
    fStart := fFinish;
    fCodes := EscapeCodes(fLine, fStart);
    fStyles:= EscapeStyles(fLine, fStart);
    si := EscapePos(fLine, fStart);
    l  := length(fLine);
    i  := 0;
    while (si+i <= l) and not CharIsCode(fLine, si+i) do Inc(i);
    fToken  := copy(fLine, si, i);
    fFinish := fStart + i;
  end;
  Result := fToken;
end;


{!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!}
{!!!                              TZREscapeStream                           !!!}
{!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!}

constructor TZREscapeStream.Create(aStream: TStream; {aSpecifiers: TZREscapeSpecifiers; }aEscapes: TZREscapeStrings);
begin
  inherited Create(aStream);
//  fSpecifiers:= aSpecifiers;
  fEscapes   := aEscapes;
end;

function TZREscapeStream.ReadString(var Value: String): Boolean;
var
  e: TZREscapeCode;
  S: String;
begin
  Result:= inherited ReadString(S);
  if Result then
    for e:= Low(TZREscapeCode) to High(TZREscapeCode) do
      S:= StringReplace(S, Escapes[e], EscapeChar + EscapeSpecifiers[e], [rfReplaceAll]);
  Value:= S;
end;

procedure TZREscapeStream.WriteString(const Value: String);
var
  e: TZREscapeCode;
  S: String;
begin
  S:= Value;
  for e:= Low(TZREscapeCode) to High(TZREscapeCode) do
    S:= StringReplace(S, EscapeChar + EscapeSpecifiers[e], Escapes[e], [rfReplaceAll]);
  inherited WriteString(S);
end;

{!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!}
{!!!                                TZREscapes                              !!!}
{!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!}

constructor TZREscapes.Create;
begin
  Model:= emEpson;
end;

function TZREscapes.GetValue(Index: TZREscapeCode): String;
begin
  Result:= fValues[Index];
end;

procedure TZREscapes.SetValue(Index: TZREscapeCode; const Value: String);
begin
  if Value <> GetValue(Index) then begin
    fValues[Index]:= Value;
    Model:= emCustom;
  end;
end;

procedure TZREscapes.ReadValues(Reader: TReader);
var
  e: TZREscapeCode;
begin
  for e:= Low(TZREscapeCode) to High(TZREscapeCode) do fValues[e]:= Reader.ReadString;
end;

procedure TZREscapes.WriteValues(Writer: TWriter);
var
  e: TZREscapeCode;
begin
  for e:= Low(TZREscapeCode) to High(TZREscapeCode) do Writer.WriteString(fValues[e]);
end;

procedure TZREscapes.DefineProperties(Filer: TFiler);
begin
  inherited;
  Filer.DefineProperty('Values', ReadValues, WriteValues, Model = emCustom);
end;

(*
procedure TZREscapes.SetModel(Value: TZREscapeModel);
const
  EscapeMap: array[TZREscapeModel] of TZREscapeStrings = (
                 {ecReset, ecFormFeed, ecPica, ecElite, ecCondensedOn, ecCondensedOff, ecBoldOn, ecBoldOff, ecItalicOn, ecItalicOff, ecUnderlineOn, ecUnderlineOff, ecSuperScriptOn, ecSuperScriptOff, ecSubScriptOn, ecSubScriptOff, ecReportStart, ecReportFinish, ecPageStart, ecPageFinish}
{emCustom       }('',      '',         '',     '',      '',            '',             '',       '',        '',         '',          '',            '',             '',              '',               '',            ''            , ''           , ''            , ''         , ''           ),
{emNone         }('',      '',         '',     '',      '',            '',             '',       '',        '',         '',          '',            '',             '',              '',               '',            ''            , ''           , ''            , ''         , ''           ),
{emCannonF60    }('',      '',         '',     '',      '',            '',             '',       '',        '',         '',          '',            '',             '',              '',               '',            ''            , ''           , ''            , ''         , ''           ),
{emCannonLaser  }('',      '',         '',     '',      '',            '',             '',       '',        '',         '',          '',            '',             '',              '',               '',            ''            , ''           , ''            , ''         , ''           ),
{emEpson        }(#27#64,  #12,        #27#80, #27#77,  #15,           #18,            #27#71,   #27#72,    #27#52,     #27#53,      #27#45#49,     #27#45#48,      #27#83#01,       #27#84,           #27#83#00,     #27#84        , ''           , ''            , ''         , ''           ),
{emHPDeskjet    }('',      '',         '',     '',      '',            '',             '',       '',        '',         '',          '',            '',             '',              '',               '',            ''            , ''           , ''            , ''         , ''           ),
{emHPLaserjet   }('',      '',         '',     '',      '',            '',             '',       '',        '',         '',          '',            '',             '',              '',               '',            ''            , ''           , ''            , ''         , ''           ),
{emHPThinkjet   }('',      '',         '',     '',      '',            '',             '',       '',        '',         '',          '',            '',             '',              '',               '',            ''            , ''           , ''            , ''         , ''           ),
{emIBMColorJet  }('',      '',         '',     '',      '',            '',             '',       '',        '',         '',          '',            '',             '',              '',               '',            ''            , ''           , ''            , ''         , ''           ),
{emIBMPCGraphics}('',      '',         '',     '',      '',            '',             '',       '',        '',         '',          '',            '',             '',              '',               '',            ''            , ''           , ''            , ''         , ''           ),
{emIBMProprinter}('',      '',         '',     '',      '',            '',             '',       '',        '',         '',          '',            '',             '',              '',               '',            ''            , ''           , ''            , ''         , ''           ),
{emNEC3500      }('',      '',         '',     '',      '',            '',             '',       '',        '',         '',          '',            '',             '',              '',               '',            ''            , ''           , ''            , ''         , ''           ),
{emNECPinwriter }('',      '',         '',     '',      '',            '',             '',       '',        '',         '',          '',            '',             '',              '',               '',            ''            , ''           , ''            , ''         , ''           )
  );                                                                                                                                                                                                                                                                                             
var
  e: TZREscapeCode;
begin
  if Value <> fModel then begin
    fModel:= Value;
    if fModel = emCustom then
      for e:= Low(TZREscapeCode) to High(TZREscapeCode) do EscapeMap[fModel, e]:= fValues[e];
    for e:= Low(TZREscapeCode) to High(TZREscapeCode) do fValues[e]:= EscapeMap[fModel, e];
  end;
end;
*)

procedure TZREscapes.SetModel(Value: TZREscapeModel);
const
    (*
     * These codes added by PJB - Feb 15, 2000.
     *
     * Adding support for PCL5 printers (and compatibles) from HP.
     *
     * According to HP's web-site, these codes are common across their
     * range of printers, so we can use the same codes for the LaserJet,
     * DeskJet and ThinkJet.  Here's hoping ...
     *)
    PCL_Rst         = #27#69;             (* Reset *)
    PCL_FF          = #27#38#108#48#72;   (* Form Feed - "Eject Page" *)

    PCL_Pic         = #27#38#107#48#83;   (* Pica ???Not Sure??? 10 pitch *)
    PCL_Et          = #27#38#107#52#83;   (* Elite *)
    PCL_cOn         = #27#38#107#50#83;   (* CondensedOn/Compressed *)
    PCL_cOff        = PCL_Pic;            (* CondensedOff ???Not Sure??? *)

    PCL_Bold        = #27#40#115#51#66;   (* BoldOn *)
    PCL_NoBold      = #27#40#115#48#66;   (* BoldOff - "Medium Stroke" *)

    PCL_Italic      = #27#40#115#49#83;   (* ItalicOn *)
    PCL_ItalOff     = #27#40#115#48#83;   (* ItalicOff - "Upright Style *)

    PCL_Underline   = #27#38#100#48#68;   (* UnderlineOn *)
    PCL_UnderlOff   = #27#38#100#64;      (* UnderlineOff *)

    PCL_SuperOn     = #27#38#97#45#46#50#53#82; (* SuperScriptOn *)
    PCL_SubOn       = #27#38#97#43#46#50#53#82; (* SubScriptOn *)
    PCL_SuperOff    = PCL_SubOn;                (* SuperScriptOff *)
    PCL_SubOff      = PCL_SuperOn;              (* SubScriptOff *)

  EscapeMap: array[TZREscapeModel] of TZREscapeStrings = (
                 {ecReset, ecFormFeed, ecPica, ecElite, ecCondensedOn, ecCondensedOff, ecBoldOn, ecBoldOff, ecItalicOn, ecItalicOff, ecUnderlineOn, ecUnderlineOff, ecSuperScriptOn, ecSuperScriptOff, ecSubScriptOn, ecSubScriptOff, ecReportStart, ecReportFinish, ecPageStart, ecPageFinish}
{emCustom       }('',      '',         '',     '',      '',            '',             '',       '',        '',         '',          '',            '',             '',              '',               '',            ''            , ''           , ''            , ''         , ''          ),
{emNone         }('',      '',         '',     '',      '',            '',             '',       '',        '',         '',          '',            '',             '',              '',               '',            ''            , ''           , ''            , ''         , ''          ),
{emCannonF60    }('',      '',         '',     '',      '',            '',             '',       '',        '',         '',          '',            '',             '',              '',               '',            ''            , ''           , ''            , ''         , ''          ),
{emCannonLaser  }('',      '',         '',     '',      '',            '',             '',       '',        '',         '',          '',            '',             '',              '',               '',            ''            , ''           , ''            , ''         , ''          ),
{emEpson        }(#27#64,  #12,        #27#80, #27#77,  #15,           #18,            #27#71,   #27#72,    #27#52,     #27#53,      #27#45#49,     #27#45#48,      #27#83#01,       #27#84,           #27#83#00,     #27#84        , ''           , ''            , ''         , ''          ),
{emHPDeskjet    }(PCL_Rst, PCL_FF,     PCL_Pic,PCL_Et,  PCL_cOn,       PCL_cOff,       PCL_Bold, PCL_NoBold,PCL_Italic, PCL_ItalOff, PCL_Underline, PCL_UnderlOff,  PCL_SuperOn,     PCL_SuperOff,     PCL_SubOn,     PCL_SubOff    , ''           , ''            , ''         , ''          ),
{emHPLaserjet   }(PCL_Rst, PCL_FF,     PCL_Pic,PCL_Et,  PCL_cOn,       PCL_cOff,       PCL_Bold, PCL_NoBold,PCL_Italic, PCL_ItalOff, PCL_Underline, PCL_UnderlOff,  PCL_SuperOn,     PCL_SuperOff,     PCL_SubOn,     PCL_SubOff    , ''           , ''            , ''         , ''          ),
{emHPThinkjet   }(PCL_Rst, PCL_FF,     PCL_Pic,PCL_Et,  PCL_cOn,       PCL_cOff,       PCL_Bold, PCL_NoBold,PCL_Italic, PCL_ItalOff, PCL_Underline, PCL_UnderlOff,  PCL_SuperOn,     PCL_SuperOff,     PCL_SubOn,     PCL_SubOff    , ''           , ''            , ''         , ''          ),
{emIBMColorJet  }('',      '',         '',     '',      '',            '',             '',       '',        '',         '',          '',            '',             '',              '',               '',            ''            , ''           , ''            , ''         , ''          ),
{emIBMPCGraphics}('',      '',         '',     '',      '',            '',             '',       '',        '',         '',          '',            '',             '',              '',               '',            ''            , ''           , ''            , ''         , ''          ),
{emIBMProprinter}('',      '',         '',     '',      '',            '',             '',       '',        '',         '',          '',            '',             '',              '',               '',            ''            , ''           , ''            , ''         , ''          ),
{emNEC3500      }('',      '',         '',     '',      '',            '',             '',       '',        '',         '',          '',            '',             '',              '',               '',            ''            , ''           , ''            , ''         , ''          ),
{emNECPinwriter }('',      '',         '',     '',      '',            '',             '',       '',        '',         '',          '',            '',             '',              '',               '',            ''            , ''           , ''            , ''         , ''          )
  );
var
  e: TZREscapeCode;
begin
  if Value <> fModel then begin
    fModel:= Value;
    if fModel = emCustom then
      for e:= Low(TZREscapeCode) to High(TZREscapeCode) do EscapeMap[fModel, e]:= fValues[e];
    for e:= Low(TZREscapeCode) to High(TZREscapeCode) do fValues[e]:= EscapeMap[fModel, e];
  end;
end;


procedure TZREscapes.Assign(Source: TPersistent);
begin
  if Source is TZREscapes then begin
    fModel  := TZREscapes(Source).fModel;
    fValues := TZREscapes(Source).fValues;
  end else
    inherited;
end;



var
  ch: Char;
  ec: TZREscapeCode;
  es: TZREscapeStyle;
initialization
  EscapeModelNames[emCustom] := LoadStr(szrEscapeModelCustom);
  EscapeModelNames[emNone]   := LoadStr(szrEscapeModelNone);

  for ch:= #0 to #255 do begin
    CharIsCodeMap[ch] := False;
    for ec:= low(TZREscapeCode) to high(TZREscapeCode) do begin
      if ch = EscapeSpecifiers[ec] then begin
        CharIsCodeMap[ch]:= True;
        CharToCodeMap[ch]:= ec;
        continue;
      end;
    end;
  end;

  for ec:= low(TZREscapeCode) to high(TZREscapeCode) do begin
    CodeIsSwitchOnMap [ec] := False;
    CodeIsSwitchOffMap[ec] := False;
    for es:= low(TZREscapeStyle) to high(TZREscapeStyle) do
      if StyleCodes[es].On  = ec then begin
        CodeIsSwitchOnMap [ec] := True;
        CodeToStyleMap    [ec] := es;
      end else
      if StyleCodes[es].Off = ec then begin
        CodeIsSwitchOffMap[ec] := True;
        CodeToStyleMap    [ec] := es;
      end;
  end;

(*
  for ec:= low(TZREscapeCode) to high(TZREscapeCode) do begin
    StyleSwitchesOnFlags[ec] := False;
    StyleSwitchesOffFlags[ec]:= False;
  end;

  for ec:= low(TZREscapeCode) to high(TZREscapeCode) do
    for es:= low(TZREscapeStyle) to high(TZREscapeStyle) do begin
      if StyleSwitches[es].On  = ec then begin
        StyleSwitchesOn[ec]     := es;
        StyleSwitchesOnFlags[ec]:= True;
        continue;
      end;
    end;
  for ec:=low(TZREscapeCode) to high(TZREscapeCode) do
    for es:= low(TZREscapeStyle) to high(TZREscapeStyle) do begin
      if StyleSwitches[es].off  = ec then begin
        StyleSwitchesOff[ec]     := es;
        StyleSwitchesOffFlags[ec]:= True;
        continue;
      end;
    end;
*)
end.

