unit MDIBck;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs;

type
  TMDIPaint = procedure(Sender: TObject; ACanvas: TCanvas; var AClientRect: TRect) of object;

  TMDIBackground = class(TGraphicControl)
  private
    FMDICanvas: TCanvas;
    FBevelWidth: Integer;
    FMDIBackgroundPaint: TMDIPaint;
    FBackgroundColor: TColor;
    FOldClientProc: TFarProc;
    FNewClientProc: TFarProc;
    procedure ClientWndProc(var Mesg: TMessage);
    procedure SetBackgroundColor(const ANewColor: TColor);
    procedure SetBevelWidth(BW: Integer);
  protected
    procedure Paint; override;
    procedure DrawBackground(DC: HDC); dynamic;
    procedure FillClientRect(ACanvas: TCanvas; const ARect: TRect); dynamic;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property BevelWidth: Integer read FBevelWidth write SetBevelWidth default 2;
    property BackgroundColor: TColor read FBackgroundColor write SetBackgroundColor default clInactiveCaption;
    property OnPaint: TMDIPaint read FMDIBackgroundPaint write FMDIBackgroundPaint;
  end;

procedure Register;

implementation

uses ExtCtrls;

constructor TMDIBackground.Create(AOwner: TComponent);
begin
   // Create then background only if it's on a Form wich is MDI parent
   if (AOwner is TForm) and (TForm(AOwner).FormStyle = fsMDIForm) then begin
      inherited Create(AOwner);
      Width := 150;
      Height := 40;
      Align := alClient;
      FBackgroundColor := clInactiveCaption;
      FBevelWidth := 2;
      FMDICanvas := TCanvas.Create;

      if not (csDesigning in ComponentState) then begin
         TForm(Owner).HandleNeeded;
         FNewClientProc := MakeObjectInstance(ClientWndProc);
         FOldClientProc := Pointer(GetWindowLong(TForm(Owner).ClientHandle, GWL_WNDPROC));
         SetWindowLong(TForm(Owner).ClientHandle, GWL_WNDPROC, Longint(FNewClientProc));
      end; end
   else begin
      raise Exception.Create('TMDIBackground can only be used on a form '#10 +
                             'with FormStyle = fsMDIForm');
   end;
end;

destructor TMDIBackground.Destroy;
begin
   FMDICanvas.Free;
   if Assigned(FNewClientProc) then begin
      FreeObjectInstance(FNewClientProc);
      SetWindowLong(TForm(Owner).ClientHandle, GWL_WNDPROC, Longint(FOldClientProc));
   end;
   inherited Destroy;
end;

procedure TMDIBackground.ClientWndProc(var Mesg: TMessage);
begin
   with Mesg do begin
      case Msg of
         WM_ERASEBKGND: begin
            CallWindowProc(FOldClientProc, TForm(Owner).ClientHandle, Msg, wParam, lParam);
            DrawBackground(TWMEraseBkGnd(Mesg).DC);
            end;
         WM_VSCROLL, WM_HSCROLL: begin
            Result := CallWindowProc(FOldClientProc, TForm(Owner).ClientHandle, Msg, wParam, lParam);
            InvalidateRect(TForm(Owner).ClientHandle, nil, True);
            end;
         WM_WINDOWPOSCHANGING: begin
            with TWMWindowPosChanging(Mesg) do begin
               Inc(WindowPos^.x, FBevelWidth);
               Inc(WindowPos^.y, FBevelWidth);
               Dec(WindowPos^.cx, 2*FBevelWidth);
               Dec(WindowPos^.cy, 2*FBevelWidth);
            end;
            end;
      else
         Result := CallWindowProc(FOldClientProc, TForm(Owner).ClientHandle, Msg, wParam, lParam);
      end;
   end;
end;

procedure TMDIBackground.SetBackgroundColor(const ANewColor: TColor);
begin
   if (ANewColor <> FBackgroundColor) then begin
      FBackgroundColor := ANewColor;
      if not (csDesigning in ComponentState) then InvalidateRect(TForm(Owner).ClientHandle, nil, True);
      Repaint;
   end;
end;

procedure TMDIBackground.SetBevelWidth(BW: Integer);
var
   iDiff: Integer;
begin
   if (BW <> FBevelWidth) then begin
      iDiff := BW - FBevelWidth;
      FBevelWidth := BW;
      if not (csDesigning in ComponentState) then begin
         SetWindowPos(TForm(Owner).ClientHandle, 0, 0, 0, TForm(Owner).ClientWidth,
                      TForm(Owner).ClientHeight, SWP_NOZORDER + SWP_NOACTIVATE);
      end;
      Repaint;
   end;
end;


procedure TMDIBackground.Paint;
var
   Rect: TRect;
begin
   Rect := GetClientRect;
   if (csDesigning in ComponentState) then begin
      FillClientRect(Canvas, Rect);
   end;
   Frame3D(Canvas, Rect, clBlack, clBtnHighlight, FBevelWidth);
end;

procedure TMDIBackground.DrawBackground(DC: HDC);
var
   Rect: TRect;
begin
   FMDICanvas.Handle := DC;
   Windows.GetClientRect(TForm(Owner).ClientHandle, Rect);
   FillClientRect(FMDICanvas, Rect);
   if Assigned(FMDIBackgroundPaint) then FMDIBackgroundPaint(Self, FMDICanvas, Rect);
end;

procedure TMDIBackground.FillClientRect(ACanvas: TCanvas; const ARect: TRect);
begin
   with ACanvas do begin
      Brush.Style := bsSolid;
      Brush.Color := FBackgroundColor;
      FillRect(ARect);
   end;
end;

procedure Register;
begin
  RegisterComponents('Supplment', [TMDIBackground]);
end;

end.
