(*
  wtsFocusColor - Weber Tech Services Control Focus Color Component
  -----------------------------------------------------------------

  Author:  Richard Weber
  Date:    October 14, 2001
  Version: 1.7

  Email Questions and Comments to: RAWDirect@yahoo.com

  This component is a FREEWARE component, you can this component AT YOUR OWN RISK!!
  Just send me a email telling me how much you love this little component :p

  This simple component to change the color of contols as they receieve
  focus when selected.  The component hooks into the screen.OnActiveControlChange
  event and changes the control color as the event is call through delphi, then
  calls the default event if given.

  Development Notes:
  ----------------
  To provide more flexability to other Third-Party controls a InvalidClasses
  property is provided to allow programmers to specify controls to ignore.
  Also the InvalidClasses property doesnt force all the available controls
  in the uses clause.

  Coolness Factor:
  ----------------
  Even though the Color property defined in the TControl it is not Published.
  I used some Polymorphism to type cast the control to TwtsColorControlCustom,
  and all it does is make public the color property.  This allows the component to
  use any TControl decendant without having it in the uses clause.

  A nice little workaround even though it is a small hack is to create stringlists
  that is used to Ignore certain controls that is not compatible with this component.
  I just used the classnames control names found in the RTTI to perform checks.  Could
  slowdown the navigating of components some, but hell a lot faster than doing it
  all by hand!!!

  Revision History:
  1.7 - Removed InvalidClasses Property and used a single list for both names
  1.6 - Added Version Property
  1.51 - ReFixed bug to set the LastControl, if the active control is invalid
  1.5 - Fixed bug to set the LastControl, if the active control is invalid
  1.4 - Added InvalidNames property
  1.3 - Added InvalidClasses property
  1.2 - Added options to set the Font color
  1.1 - Color can only do the owner set for control owner, freeing forms when
        Last Control causes AccessViolations


Notes on using InvalidNames:
When you find a control that you wish the color and font not be changed
add the classname or component name to the InvalidNames list.
*)

unit wtsFocusColor;
{$X+}

interface

uses
   Forms, Classes, Controls, Graphics;

Const
  VersionString : String = 'Version 1.7 - wtsFocusColor';

type
  TwtsFocusColor = class(TComponent)
  private
    { Private declarations }
    FOnActiveControlChange      : TNotifyEvent;
    FLastControl                : TControl;
    FLastColor                  : TColor;
    FLastFontColor              : TColor;
    FEnabled                    : Boolean;
    FColor                      : TColor;
    FFontColor                  : TColor;
    FInvalidNames             : TStringList;
    FDefaultInvalid             : Boolean;
    procedure   SetEnabled(const Value: Boolean);
    procedure   SetColor(const Value: TColor);
    procedure   SetFontColor(const Value: TColor);
    procedure   SetInvalidNames(const Value: TStringlist);
    procedure   SetDefaultInvalid(const Value: Boolean);
    procedure   SetVersion(const Value: String);
    function    GetVersion: String;
  protected
    { Protected declarations }
    Procedure   ActiveControlChange(Sender : TObject);
    Procedure   SetControlColor(Control: TControl);
    Function    ValidControl(Control: TControl) : Boolean;
    Procedure   CheckDefaultInvalid;
  Public
    { Public declarations }
    Constructor Create(AOwner: TComponent); Override;
    Destructor  Destroy; Override;
    Property    LastControl     : TControl    Read FLastControl;
    Property    LastColor       : TColor      Read FColor;
    Property    LastFontColor   : TColor      Read FLastFontColor;
  published
    { Published declarations }
    Property    Enabled         : Boolean     Read FEnabled        Write SetEnabled;
    Property    FontColor       : TColor      Read FFontColor      Write SetFontColor;
    Property    Color           : TColor      Read FColor          Write SetColor;
    Property    DefaultInvalid  : Boolean     Read FDefaultInvalid Write SetDefaultInvalid;
    Property    InvalidNames    : TStringlist Read FInvalidNames   Write SetInvalidNames;
    Property    Version         : String      Read GetVersion      Write SetVersion;
  end;

procedure Register;

implementation

{$R wtsCustom.dcr}

Type
  TwtsColorControlCustom = Class(TControl)
    Published
      Property Font;
      Property Color;
  End;

procedure Register;
begin
  RegisterComponents('WTS', [TwtsFocusColor]);
end;

{ TFocusControl }

procedure TwtsFocusColor.ActiveControlChange(Sender: TObject);
begin
  Try
    If Not (csDesigning in ComponentState) then
    If (Screen <> nil) and (Screen.ActiveControl <> nil) and (Screen.ActiveControl is TControl) then
      SetControlColor(TwtsColorControlCustom(Screen.ActiveControl));
  Finally
    if Assigned(FOnActiveControlChange) then
       FOnActiveControlChange(Sender);
  End;
end;

Procedure  TwtsFocusColor.CheckDefaultInvalid;
Begin
  If FDefaultInvalid then
   Begin
     FInvalidNames.Add('TPageControl');
     FInvalidNames.Add('TTabControl');
     FInvalidNames.Add('TButton');
   End;
End;

constructor TwtsFocusColor.Create(AOwner: TComponent);
begin
  FInvalidNames := TStringList.Create;
  FLastControl := nil;
  FOnActiveControlChange := nil;
  FColor := clAqua;

  FInvalidNames.Duplicates := dupIgnore;
  FInvalidNames.Sorted := True;
  FEnabled := False;
  Inherited;
end;

destructor TwtsFocusColor.Destroy;
begin
  Enabled := False;
  FLastControl := nil;
  Screen.OnActiveControlChange := FOnActiveControlChange;
  FInvalidNames.Free;
  Inherited Destroy;
end;

function TwtsFocusColor.GetVersion: String;
begin
 Result := VersionString;
end;

procedure TwtsFocusColor.SetColor(const Value: TColor);
begin
  FColor := Value;
  If FColor <> Value then
    ActiveControlChange(Self);
end;

procedure TwtsFocusColor.SetControlColor(Control: TControl);
begin
 Try
  If (Control <> nil) and (Control.Owner = Owner) then
    Begin
      If (TwtsColorControlCustom(FLastControl) <> nil) then
        Begin
          TwtsColorControlCustom(FLastControl).Color := FLastColor;
          TwtsColorControlCustom(FLastControl).Font.Color := FLastFontColor;
          FLastControl       := nil;
        End;

      If ValidControl(Control) then
      Begin
        FLastControl       := Control;
        FLastColor         := TwtsColorControlCustom(FLastControl).Color;
        FLastFontColor     := TwtsColorControlCustom(FLastControl).Font.Color;
        TwtsColorControlCustom(Control).Color      := FColor;
        TwtsColorControlCustom(Control).Font.Color := FontColor;
      End;
    End;
 Except
  FLastControl       := nil;
 End;
end;

procedure TwtsFocusColor.SetDefaultInvalid(const Value: Boolean);
begin
  FDefaultInvalid := Value;
  CheckDefaultInvalid;
end;

procedure TwtsFocusColor.SetEnabled(const Value: Boolean);
begin
  If FEnabled <> Value then
  Begin
    FEnabled := Value;

    If Screen <> nil then
      Begin
        If Not FEnabled then
            Screen.OnActiveControlChange := FOnActiveControlChange
         else
          Begin
            FOnActiveControlChange := Screen.OnActiveControlChange;
            Screen.OnActiveControlChange := ActiveControlChange;
          End;
      End;
  End;
end;

procedure TwtsFocusColor.SetFontColor(const Value: TColor);
begin
  FFontColor := Value;
  If FFontColor <> Value then
    ActiveControlChange(Self);
end;

procedure TwtsFocusColor.SetInvalidNames(const Value: TStringlist);
begin
  FInvalidNames.Assign(Value);
end;

procedure TwtsFocusColor.SetVersion(const Value: String);
begin
end;

function TwtsFocusColor.ValidControl(Control: TControl): Boolean;
begin
  Result := (FInvalidNames.IndexOf(Control.ClassName) = -1) and
            (FInvalidNames.IndexOf(Control.Name) = -1);
end;

end.
