unit WndProp;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,TypInfo,Registry;


const
  wpLeft='Left';
  wpTop='Top';
  wpWidth='Width';
  wpHeight='Height';
  wpState='State';
  wpGrids='Grids';
  wpFontName='FontName';
  wpFontCharSet='FontCharSet';
  wpFontColor='FontColor';
  wpFontPitch='FontPitch';
  wpFontStyle='FontStyle';
  wpFontSize='FontSize';
  wpColor='Color';
  wpColumn='Column';
  wpTitle='Title';


type
  TWndProperties=(wpSize,wpPosition,wpWndState,wpControlSizes,wpCoolBar,wpGridSizes,wpFonts,wpColors);
  TWndProperty=Set of TWndProperties;
  TAction=(wpRead,wpWrite);


  TWndProp = class(TComponent)
  private
    FAbout   :String;
    Reg      :TRegIniFile;
    FKey     :String;
    FOptions :TWndProperty;
    FSection :String;
    FAutoLoad:Boolean;
    procedure SetKey(Value:String);
    procedure SetSection(Value:String);
    procedure SetOptions(Value:TWndProperty);
    procedure SetAutoLoad(Value:Boolean);
    function GetAbout:String;
  protected
    OnShowOrigin:TNotifyEvent;
    OnDestroyOrigin:TNotifyEvent;
    procedure OnShow(Sender: TObject);
    procedure OnDestroy(Sender: TObject);
    procedure LoadCoolBarDefaults;
    procedure SaveCoolBarDefaults;
    procedure LoadGridProps(AComponent:TComponent);
    procedure SaveGridProps(AComponent:TComponent);
    procedure ReadFont(AComponent:TComponent;AInfo:PPropInfo);
    procedure WriteFont(AComponent:TComponent;AInfo:PPropInfo);
    procedure ReadColor(AComponent:TComponent;AInfo:PPropInfo);
    procedure WriteColor(AComponent:TComponent;AInfo:PPropInfo);
    procedure EnumControls(Action:TAction);
  public
    constructor Create(AOwner:TComponent);override;
    destructor Destroy;override;
    procedure LoadWndProps;
    procedure SaveWndProps;
  published
    property About:string read GetAbout write FAbout;
    property Key:String read FKey write SetKey;
    property Section:String read FSection write SetSection;
    property Options:TWndProperty read FOptions write SetOptions;
    property AutoLoad:Boolean read FAutoLoad write SetAutoload;
  end;

procedure Register;

implementation
uses ComCtrls,extctrls,dbGrids;
procedure Register;
begin
  RegisterComponents('CWComponents', [TWndProp]);
end;

constructor TWndProp.Create(AOwner:TComponent);
begin
   inherited Create(AOwner);
   if AOwner is TForm then FSection:=TForm(AOwner).Caption else Section:='Section1';
   FKey:='Software';
   FOptions:=[wpSize,wpPosition,wpWndState];
   FAutoLoad:=True;
end;

destructor TWndProp.Destroy;
begin
  inherited Destroy;
end;

procedure TWndProp.SetKey(Value:String);
begin
  FKey:=Value;
end;

procedure TWndProp.SetSection(Value:String);
begin
  FSection:=Value;
end;

procedure TWndProp.SetOptions(Value:TWndProperty);
begin
  FOptions:=Value;
end;


procedure TWndProp.LoadWndProps;
begin
  if (not (csDesigning in ComponentState))and
     (Owner is TForm) then
  begin
    Reg:=TRegIniFile.Create(FKey);
    if wpPosition in FOptions then
    begin
      TForm(Owner).Left:=Reg.ReadInteger(Section,wpLeft,TForm(Owner).Left);
      TForm(Owner).Top:=Reg.ReadInteger(Section,wpTop,TForm(Owner).Top);
    end;
    if wpSize in FOptions then
    begin
      TForm(Owner).Width:=Reg.ReadInteger(Section,wpWidth,TForm(Owner).Width);
      TForm(Owner).Height:=Reg.ReadInteger(Section,wpHeight,TForm(Owner).Height);
    end;
    EnumControls(wpRead);
    if wpCoolBar in FOptions then LoadCoolBarDefaults;
    if wpWndState in FOptions then
    begin
      TForm(Owner).WindowState:=TWindowState(Reg.ReadInteger(Section,wpState,Integer(TForm(Owner).WindowState)));
    end;
    Reg.Free;
  end;
end;

procedure TWndProp.SaveWndProps;
begin
  if (not (csDesigning in ComponentState))and
     (Owner is TForm) then
  begin
    Reg:=TRegIniFile.Create(FKey);
    if wpWndState in FOptions then
    begin
      Reg.WriteInteger(Section,wpState,Integer(TForm(Owner).WindowState));
    end;
    if TForm(Owner).WindowState=wsNormal then
    begin
      if wpPosition in FOptions then
      begin
        Reg.WriteInteger(Section,wpLeft,TForm(Owner).Left);
        Reg.WriteInteger(Section,wpTop,TForm(Owner).Top);
      end;
      if wpSize in FOptions then
      begin
        Reg.WriteInteger(Section,wpWidth,TForm(Owner).Width);
        Reg.WriteInteger(Section,wpHeight,TForm(Owner).Height);
      end;
    end;
    EnumControls(wpWrite);
    if wpCoolBar in FOptions then SaveCoolBarDefaults;
    Reg.Free;
  end;
end;

procedure TWndProp.SetAutoLoad(Value:Boolean);
begin
  FAutoLoad:=Value;
  if (not (csDesigning in ComponentState))and FAutoLoad then
  begin
    OnShowOrigin:=TForm(Owner).OnShow;
    OnDestroyOrigin:=TForm(Owner).OnDestroy;
    TForm(Owner).OnShow:=OnShow;
    TForm(Owner).OnDestroy:=OnDestroy;
  end;
end;

procedure TWndProp.OnShow(Sender: TObject);
begin
  if Assigned(OnShowOrigin) then OnShowOrigin(Sender);
  LoadWndProps;
  TForm(Owner).OnShow:=OnShowOrigin;
end;

procedure TWndProp.OnDestroy(Sender: TObject);
begin
  if Assigned(OnDestroyOrigin) then OnDestroyOrigin(Sender);
  SaveWndProps;
  TForm(Owner).OnDestroy:=OnDestroyOrigin;
end;


procedure TWndProp.LoadCoolBarDefaults;
var i,j:Integer;
begin
  if Not(Owner is TForm) then Exit;
  for i:=0 to Owner.ComponentCount-1 do
    if Owner.Components[i] is TCoolBar then
      for j:=0 to TCoolBar(Owner.Components[i]).Bands.Count-1 do
      begin
        TCoolBar(Owner.Components[i]).Bands[j].Width:=Reg.ReadInteger(Section+'\'+TCoolBar(Owner.Components[i]).Name,'Band'+IntToStr(j),TCoolBar(Owner.Components[i]).Bands.Items[j].Width);
        TCoolBar(Owner.Components[i]).Bands[j].Break:=Boolean(Reg.ReadInteger(Section+'\'+TCoolBar(Owner.Components[i]).Name,'Band'+IntToStr(j)+'Break',Integer(TCoolBar(Owner.Components[i]).Bands.Items[j].Break)));
      end;
end;

procedure TWndProp.SaveCoolBarDefaults;
var i,j:Integer;
begin
  if Not(Owner is TForm) then Exit;
  for i:=0 to Owner.ComponentCount-1 do
    if Owner.Components[i] is TCoolBar then
      for j:=0 to TCoolBar(Owner.Components[i]).Bands.Count-1 do
      begin
        Reg.WriteInteger(Section+'\'+TCoolBar(Owner.Components[i]).Name,'Band'+IntToStr(j),TCoolBar(Owner.Components[i]).Bands.Items[j].Width);
        Reg.WriteInteger(Section+'\'+TCoolBar(Owner.Components[i]).Name,'Band'+IntToStr(j)+'Break',Integer(TCoolBar(Owner.Components[i]).Bands.Items[j].Break));        
      end;
end;

procedure TWndProp.EnumControls(Action:TAction);
var i:Integer;
    Info:PPropInfo;

begin
 if Not(Owner is TForm) then Exit;
   for i:=0 to Owner.ComponentCount-1 do
   begin
     if (wpControlSizes in FOptions)And
        ((Owner.Components[i] is TControl)and not (Owner.Components[i] is TSplitter)) then
     begin
       Info:=GetPropInfo(Owner.Components[i].ClassInfo,'Align');
       if Info<>nil then
       begin
         case TAlign(GetOrdProp(Owner.Components[i],Info)) of
         alTop,alBottom:
               begin
                 if Action = wpRead then
                   TControl(Owner.Components[i]).Height:=Reg.ReadInteger(Owner.Name+'\'+Owner.Components[i].Name,wpHeight,TControl(Owner.Components[i]).Height)
                     else
                   Reg.WriteInteger(Owner.Name+'\'+Owner.Components[i].Name,wpHeight,TControl(Owner.Components[i]).Height)
               end;
         alLeft,alRight:
               begin
                 if Action = wpRead then
                   TControl(Owner.Components[i]).Width:=Reg.ReadInteger(Owner.Name+'\'+Owner.Components[i].Name,wpWidth,TControl(Owner.Components[i]).Width)
                     else
                   Reg.WriteInteger(Owner.Name+'\'+Owner.Components[i].Name,wpWidth,TControl(Owner.Components[i]).Width)
               end;
         end;
       end;
     end;

     if wpFonts in FOptions then
     begin
       Info:=GetPropInfo(Owner.Components[i].ClassInfo,'ParentFont');
       if (Info<>nil)and
          not Boolean(GetOrdProp(Owner.Components[i],Info)) then
       begin
         Info:=GetPropInfo(Owner.Components[i].ClassInfo,'Font');
         if Action = wpRead then
            ReadFont(TControl(Owner.Components[i]),Info)
              else
            WriteFont(TControl(Owner.Components[i]),Info);
       end;
     end;

     if wpColors in FOptions then
     begin
       Info:=GetPropInfo(Owner.Components[i].ClassInfo,'ParentColor');
       if (Info<>nil)and
          not Boolean(GetOrdProp(Owner.Components[i],Info)) then
       begin
         Info:=GetPropInfo(Owner.Components[i].ClassInfo,'Color');
         if Action = wpRead then
            ReadColor(TControl(Owner.Components[i]),Info)
              else
            WriteColor(TControl(Owner.Components[i]),Info);
       end;
     end;

     if (wpGridSizes in FOptions) then
     begin
         if Action = wpRead then
           LoadGridProps(Owner.Components[i])
             else
           SaveGridProps(Owner.Components[i]);
     end;
   end;
end;

procedure TWndProp.LoadGridProps(AComponent:TComponent);
var j:Integer;
    S:Integer;
    FStyle:TFontStyles;
begin
  if AComponent is TDBGrid then
   for j:=0 to TDBGrid(AComponent).Columns.Count-1 do
   begin
      TDBGrid(AComponent).Columns.Items[j].Width:=Reg.ReadInteger(Owner.Name+'\'+TDBGrid(AComponent).Name,TDBGrid(AComponent).Columns.Items[j].FieldName,TDBGrid(AComponent).Columns.Items[j].Width);
      TDBGrid(AComponent).Columns.Items[j].Font.Name:=Reg.ReadString(Owner.Name+'\'+TDBGrid(AComponent).Name+'\'+TDBGrid(AComponent).Columns.Items[j].FieldName+'\'+wpColumn,wpFontName,'MS Sans Serif');
      TDBGrid(AComponent).Columns.Items[j].Font.CharSet:=Reg.ReadInteger(Owner.Name+'\'+TDBGrid(AComponent).Name+'\'+TDBGrid(AComponent).Columns.Items[j].FieldName+'\'+wpColumn,wpFontCharSet,1);
      TDBGrid(AComponent).Columns.Items[j].Font.Color:=Reg.ReadInteger(Owner.Name+'\'+TDBGrid(AComponent).Name+'\'+TDBGrid(AComponent).Columns.Items[j].FieldName+'\'+wpColumn,wpFontColor,0);
      TDBGrid(AComponent).Columns.Items[j].Font.Size:=Reg.ReadInteger(Owner.Name+'\'+TDBGrid(AComponent).Name+'\'+TDBGrid(AComponent).Columns.Items[j].FieldName+'\'+wpColumn,wpFontSize,8);
      TDBGrid(AComponent).Columns.Items[j].Color:=Reg.ReadInteger(Owner.Name+'\'+TDBGrid(AComponent).Name+'\'+TDBGrid(AComponent).Columns.Items[j].FieldName+'\'+wpColumn,wpColor,TDBGrid(AComponent).Columns.Items[j].Color);
      S:=Reg.ReadInteger(Owner.Name+'\'+TDBGrid(AComponent).Name+'\'+TDBGrid(AComponent).Columns.Items[j].FieldName+'\'+wpColumn,wpFontStyle,0);
      FStyle:=[];
      if (S and 1)<>0 then Include(FStyle,fsBold);
      if (S and 2)<>0 then Include(FStyle,fsItalic);
      if (S and 4)<>0 then Include(FStyle,fsUnderLine);
      if (S and 8)<>0 then Include(FStyle,fsStrikeOut);
      TDBGrid(AComponent).Columns.Items[j].Font.Style:=FStyle;
   end;
end;

procedure TWndProp.SaveGridProps(AComponent:TComponent);
var j:Integer;
    S:Integer;
begin
  if AComponent is TDBGrid then
  begin
    for j:=0 to TDBGrid(AComponent).Columns.Count-1 do
    begin
      Reg.WriteInteger(Owner.Name+'\'+TDBGrid(AComponent).Name,TDBGrid(AComponent).Columns.Items[j].FieldName,TDBGrid(AComponent).Columns.Items[j].Width);
      Reg.WriteString(Owner.Name+'\'+TDBGrid(AComponent).Name+'\'+TDBGrid(AComponent).Columns.Items[j].FieldName+'\'+wpColumn,wpFontName,TDBGrid(AComponent).Columns.Items[j].Font.Name);
      Reg.WriteInteger(Owner.Name+'\'+TDBGrid(AComponent).Name+'\'+TDBGrid(AComponent).Columns.Items[j].FieldName+'\'+wpColumn,wpFontCharSet,TDBGrid(AComponent).Columns.Items[j].Font.CharSet);
      Reg.WriteInteger(Owner.Name+'\'+TDBGrid(AComponent).Name+'\'+TDBGrid(AComponent).Columns.Items[j].FieldName+'\'+wpColumn,wpFontColor,TDBGrid(AComponent).Columns.Items[j].Font.Color);
      Reg.WriteInteger(Owner.Name+'\'+TDBGrid(AComponent).Name+'\'+TDBGrid(AComponent).Columns.Items[j].FieldName+'\'+wpColumn,wpFontSize,TDBGrid(AComponent).Columns.Items[j].Font.Size);
      Reg.WriteInteger(Owner.Name+'\'+TDBGrid(AComponent).Name+'\'+TDBGrid(AComponent).Columns.Items[j].FieldName+'\'+wpColumn,wpColor,TDBGrid(AComponent).Columns.Items[j].Color);      
      S:=0;
      if fsBold in TDBGrid(AComponent).Columns.Items[j].Font.Style then S:=S or 1;
      if fsItalic in TDBGrid(AComponent).Columns.Items[j].Font.Style then S:=S or 2;
      if fsUnderLine in TDBGrid(AComponent).Columns.Items[j].Font.Style then S:=S or 4;
      if fsStrikeOut in TDBGrid(AComponent).Columns.Items[j].Font.Style then S:=S or 8;
      Reg.WriteInteger(Owner.Name+'\'+TDBGrid(AComponent).Name+'\'+TDBGrid(AComponent).Columns.Items[j].FieldName+'\'+wpColumn,wpFontStyle,S);
    end;
  end;
end;

function TWndProp.GetAbout:String;
begin
   Result:='Copyright 1999 by V.Cechanowski';
end;


procedure TWndProp.ReadFont(AComponent:TComponent;AInfo:PPropInfo);
var F:TFont;
    S:Integer;
    FStyle:TFontStyles;
begin
  F:=TFont.Create;
  F.Name:=Reg.ReadString(AComponent.Owner.Name+'\'+AComponent.Name,wpFontName,'MS Sans Serif');
  F.CharSet:=Reg.ReadInteger(AComponent.Owner.Name+'\'+AComponent.Name,wpFontCharSet,1);
  F.Color:=Reg.ReadInteger(AComponent.Owner.Name+'\'+AComponent.Name,wpFontColor,0);
  F.Size:=Reg.ReadInteger(AComponent.Owner.Name+'\'+AComponent.Name,wpFontSize,8);
  S:=Reg.ReadInteger(AComponent.Owner.Name+'\'+AComponent.Name,wpFontStyle,0);
  FStyle:=[];
  if (S and 1)<>0 then Include(FStyle,fsBold);
  if (S and 2)<>0 then Include(FStyle,fsItalic);
  if (S and 4)<>0 then Include(FStyle,fsUnderLine);
  if (S and 8)<>0 then Include(FStyle,fsStrikeOut);
  F.Style:=FStyle;
  SetOrdProp(AComponent,AInfo,longint(F));
  F.Free;
end;

procedure TWndProp.WriteFont(AComponent:TComponent;AInfo:PPropInfo);
var F:TFont;
    S:Integer;
begin
  F:=TFont(GetOrdProp(AComponent,AInfo));
  S:=0;
  if fsBold in F.Style then S:=S or 1;
  if fsItalic in F.Style then S:=S or 2;
  if fsUnderLine in F.Style then S:=S or 4;
  if fsStrikeOut in F.Style then S:=S or 8;
  Reg.WriteString(AComponent.Owner.Name+'\'+AComponent.Name,wpFontName,F.Name);
  Reg.WriteInteger(AComponent.Owner.Name+'\'+AComponent.Name,wpFontCharSet,F.CharSet);
  Reg.WriteInteger(AComponent.Owner.Name+'\'+AComponent.Name,wpFontColor,F.Color);
  Reg.WriteInteger(AComponent.Owner.Name+'\'+AComponent.Name,wpFontSize,F.Size);
  Reg.WriteInteger(AComponent.Owner.Name+'\'+AComponent.Name,wpFontStyle,S);
end;

procedure TWndProp.ReadColor(AComponent:TComponent;AInfo:PPropInfo);
var C:TColor;
begin
  C:=Reg.ReadInteger(AComponent.Owner.Name+'\'+AComponent.Name,wpColor,(GetOrdProp(AComponent,AInfo)));
  SetOrdProp(AComponent,AInfo,longint(C));
end;

procedure TWndProp.WriteColor(AComponent:TComponent;AInfo:PPropInfo);
begin
  Reg.WriteInteger(AComponent.Owner.Name+'\'+AComponent.Name,wpColor,(GetOrdProp(AComponent,AInfo)));
end;


end.
