{*******************************************************************************
    TWinSplit
    Copyright  Bill Menees
    bmenees@usit.net
    http://www.public.usit.net/bmenees

    This is a window splitting component similar to the one used in the Win95
    Explorer.  To use it, you must assign a control to the TargetControl
    property.  This sets the Cursor property, and a bunch of private properties
    including the Align property.

    The TargetControl is the control that gets resized at the end of the window
    "split" operation.  Thus, TargetControl must have an alignment in [alLeft,
    alRight, alTop, alBottom].

    The other useful properties introduced are MinTargetSize and MaxTargetSize.
    These determine how small or large the Width or Height of the TargetControl
    can be.  If MaxTargetSize = 0 then no maximum size is enforced.

    Note 1: Even though TWinSplit is decended from TCustomPanel, don't think of
    it as a panel.  I only published the panel properties useful to TWinSplit
    and none of the panel events.  I even made it where it won't act as the
    container for controls placed on it at design time.

    Note 2: Some drawing code is from Borland's Resource Explorer example.

*******************************************************************************}
unit WinSplit;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls;

type
  TWinSplitOrientation = (wsVertical, wsHorizontal);
  TWinSplit = class(TCustomPanel)
  private
    fTarget: TWinControl;
    fOrientation: TWinSplitOrientation;
    fSizing: Boolean;
    fDelta: TPoint;
    fMinTargetSize, fMaxTargetSize: Cardinal;
    fOnBeginSplit, fOnEndSplit, fOnMoveSplit: TNotifyEvent;
    procedure DrawSizingLine;
    procedure SetTarget(Value: TWinControl);
    procedure SetMaxTargetSize(Value: Cardinal);
    procedure SetMinTargetSize(Value: Cardinal);
    
  protected
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    {This is here so we can update the TargetControl
    property if the target component is removed.}
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;

    procedure BeginSizing; virtual;
    procedure ChangeSizing(X, Y: Integer); virtual;
    procedure EndSizing; virtual;
    function CalculateTargetSize(var ViolatedConstraints: Boolean): Integer; virtual;

  public
    constructor Create(AOwner: TComponent); override;
    property Sizing: Boolean read fSizing;

  published
    property BevelInner;
    property BevelOuter;
    property BevelWidth;
    property BorderWidth;
    property BorderStyle;
    property Color;
    property Ctl3D;
    property Enabled;
    property Font;
    property MaxTargetSize: Cardinal read fMaxTargetSize write SetMaxTargetSize default 0;
    property MinTargetSize: Cardinal read fMinTargetSize write SetMinTargetSize default 0;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property ShowHint;
    property TargetControl: TWinControl read fTarget write SetTarget;
    property Visible;

    property OnBeginSplit: TNotifyEvent read fOnBeginSplit write fOnBeginSplit;
    property OnEndSplit: TNotifyEvent read fOnEndSplit write fOnEndSplit;
    property OnMoveSplit: TNotifyEvent read fOnMoveSplit write fOnMoveSplit;
  end;

procedure Register;

implementation

{$R WinSplit.res}

{******************************************************************************}
{** Non-Member Functions ******************************************************}
{******************************************************************************}

procedure Register;
begin
     RegisterComponents('Win95', [TWinSplit]);
end;

function CToC(C1, C2: TControl; P: TPoint): TPoint;
begin
     Result := C1.ScreenToClient(C2.ClientToScreen(P));
end;

{******************************************************************************}
{** TWinSplit Public Methods **************************************************}
{******************************************************************************}

constructor TWinSplit.Create(AOwner: TComponent);
begin
     inherited Create(AOwner);
     TargetControl:=nil;
     fOrientation:=wsVertical;
     fDelta:=Point(0,0);
     fSizing:=False;
     Caption:='';
     TabStop:=False;
     Height:=100;
     Width:=3;
     {BevelOuter:=bvNone;}
     ParentColor:=True;
     MinTargetSize:=0;
     MaxTargetSize:=0;
     {We don't want a TWinSplit to be a container like normal panels.
     We also don't ever want the Caption property to be non-empty.}
     ControlStyle:=ControlStyle - [csAcceptsControls, csSetCaption];
end;

{******************************************************************************}
{** TWinSplit Protected Methods ***********************************************}
{******************************************************************************}

procedure TWinSplit.BeginSizing;
var
   ParentForm: TForm;
begin
     ParentForm:=GetParentForm(Self);
     TargetControl:=fTarget; {Reset Align, etc.}
     if (ParentForm <> nil) and (TargetControl<>nil) then
     begin
          if ((fOrientation = wsVertical) and (Align in [alTop, alBottom])) or
             ((fOrientation = wsHorizontal) and (Align in [alLeft, alRight])) then
          begin
               if Assigned(fOnBeginSplit) then fOnBeginSplit(Self);
               fSizing:=True;
               SetCaptureControl(Self);
               if fOrientation = wsVertical then
                  fDelta := Point(0, Top)
               else
                   fDelta := Point(Self.Left, 0);
               with ParentForm.Canvas do
               begin
                    Handle := GetDCEx(Parent.Handle, 0, DCX_CACHE or DCX_CLIPSIBLINGS or DCX_LOCKWINDOWUPDATE);
                    Pen.Color := clWhite;
                    if fOrientation = wsVertical then
                       Pen.Width := Height
                    else
                        Pen.Width := Width;
                    Pen.Mode := pmXOR;
               end;
               DrawSizingLine;
          end;
     end;
end;

procedure TWinSplit.ChangeSizing(X, Y: Integer);
var
   OldfDelta: TPoint;
   Violated: Boolean;
begin
     if Sizing then
     begin
          if Assigned(fOnMoveSplit) then fOnMoveSplit(Self);
          DrawSizingLine;
          OldfDelta:=fDelta;
          if fOrientation = wsVertical then
             fDelta.Y := Y
          else
              fDelta.X := X;
          Violated:=False;
          CalculateTargetSize(Violated);
          if Violated then fDelta:=OldfDelta;
          DrawSizingLine;
     end;
end;

procedure TWinSplit.EndSizing;
var
  DC: HDC;
  ParentForm: TForm;
  Violated: Boolean;
begin
     ParentForm:=GetParentForm(Self);
     fSizing:=False;
     if Assigned(fOnEndSplit) then fOnEndSplit(Self);
     DrawSizingLine;
     SetCaptureControl(nil);
     if ParentForm <> nil then
     begin
          with ParentForm do
          begin
               DC := Canvas.Handle;
               Canvas.Handle := 0;
               ReleaseDC(Handle, DC);
          end;
     end;
     if TargetControl <> nil then
     begin
          {Here we don't care if it violates the constraints.}
          if fOrientation = wsVertical then
             TargetControl.Height:=CalculateTargetSize(Violated)
          else
              TargetControl.Width:=CalculateTargetSize(Violated);
     end;
     fDelta:=Point(0,0);
end;

procedure TWinSplit.MouseMove(Shift: TShiftState; X, Y: Integer);
var
   ParentForm: TForm;
begin
     ParentForm:=GetParentForm(Self);
     if ParentForm <> nil then
     begin
          if Sizing then ChangeSizing(X, Y);
     end;
     inherited MouseMove(Shift, X, Y);
end;

procedure TWinSplit.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
     if Sizing then EndSizing;
     inherited MouseUp(Button, Shift, X, Y);
end;

procedure TWinSplit.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
     inherited MouseDown(Button, Shift, X, Y);
     if (Button = mbLeft) and (Shift = [ssLeft]) then BeginSizing;
end;

function TWinSplit.CalculateTargetSize(var ViolatedConstraints: Boolean): Integer;
var
   TargetSize: Integer;
begin
     ViolatedConstraints:=False;
     {Get the suggested size.}
     if fOrientation = wsVertical then
     begin
          if Align = alTop then
             TargetSize:=TargetControl.Height + fDelta.Y
          else
              TargetSize:=TargetControl.Height - fDelta.Y;
     end
     else
     begin
          if Align = alLeft then
             TargetSize:=TargetControl.Width + fDelta.X
          else
              TargetSize:=TargetControl.Width - fDelta.X;
     end;
     {Check it against the constraints.}
     if TargetSize < MinTargetSize then
     begin
          ViolatedConstraints:=True;
          Result:=MinTargetSize;
     end
     else if (TargetSize > MaxTargetSize) and (MaxTargetSize <> 0) then
     begin
          ViolatedConstraints:=True;
          Result:=MaxTargetSize;
     end
     else
         Result:=TargetSize;
end;

procedure TWinSplit.Notification(AComponent: TComponent; Operation: TOperation);
begin
     if (Operation = opRemove) and (AComponent = TargetControl) then
     begin
          if Sizing then EndSizing;
          TargetControl:=nil;
     end;
     inherited Notification(AComponent, Operation);
end;

{******************************************************************************}
{** TWinSplit Private Methods *************************************************}
{******************************************************************************}

procedure TWinSplit.DrawSizingLine;
var
  P: TPoint;
  ParentForm: TForm;
begin
     ParentForm:=GetParentForm(Self);
     if ParentForm <> nil then
     begin
          P := CToC(Parent, Self, fDelta);
          with ParentForm.Canvas do
          begin
               MoveTo(P.X, P.Y);
               if fOrientation = wsVertical then
                  LineTo(CToC(Parent, Self, Point(Width, 0)).X, P.Y)
               else
                   LineTo(P.X, CToC(Parent, Self, Point(0, Height)).Y)
          end;
     end;
end;

procedure TWinSplit.SetTarget(Value: TWinControl);
begin
     if (Value = nil) or (Value = Self) then
     begin
          fTarget := nil;
          Align := alNone;
          Cursor := crDefault;
          Exit;
     end;
     if Value.Align in  [alNone, alClient] then
     begin
          SetTarget(nil);
          if csDesigning in ComponentState then
             raise Exception.Create('TargetControl''s align must be left, right, top or bottom');
     end;
     Align:=Value.Align;
     fTarget:=Value;
     if Align in [alTop, alBottom] then fOrientation := wsVertical
     else fOrientation := wsHorizontal;
     if fOrientation = wsVertical then Cursor:=crVSplit
     else Cursor:=crHSplit;
end;

procedure TWinSplit.SetMaxTargetSize(Value: Cardinal);
begin
     if (Value > MinTargetSize) or (Value = 0) then fMaxTargetSize := Value
     else fMaxTargetSize := MinTargetSize;
end;

procedure TWinSplit.SetMinTargetSize(Value: Cardinal);
begin
     if (Value < MaxTargetSize) or (MaxTargetSize = 0) then fMinTargetSize := Value
     else fMinTargetSize := MaxTargetSize;
end;

end.
