{$I DFS.INC}  { Standard defines for all Delphi Free Stuff components }

{-----------------------------------------------------------------------------}
{ TWebFileInfo v0.81                                                          }
{-----------------------------------------------------------------------------}
{ A component that provides information about a product from the product's    }
{ home page.  May be used to determine if the product has been updated.       }
{                                                                             }
{ Written by Jon M. Robertson.                                                }
{ Copyright 1998, Computer Solutions.  All Rights Reserved.                   }
{                                                                             }
{ This component can be freely used and distributed in commercial and private }
{ environments, provided this notice is not modified in any way and there is  }
{ no charge for it other than nomial handling fees.  Contact me directly for  }
{ modifications to this agreement.                                            }
{                                                                             }
{ DFS.INC is Copyright 1996, Brad Stowers.  All Rights Reserved.  DFS.INC     }
{ is included courtsey of Brad.  He may be contacted at bstowers@pobox.com    }
{                                                                             }
{ TWebFileInfo uses THttpCli from the freeware Internet Component Suite.      }
{ ICS is Copyright 1997-1998, Francois Piette, All Rights Reserved.           }
{ ICS is available at http://www.rtfm.be/fpiette/indexuk.htm                  }
{ In order for TWebFileInfo to work without ICS, some compiled ICS code is    }
{ distributed with TWebFileInfo.  This code is distributed with permission.   }
{-----------------------------------------------------------------------------}
{ Feel free to contact me if you have any questions, comments or suggestions  }
{ at touri@flash.net.                                                         }
{ The lateset version will always be available on the web at:                 }
{   http://www.pobox.com/~bstowers/delphi/                                    }
{ See WebFileInfo.txt for notes, known issues, and revision history.          }
{-----------------------------------------------------------------------------}
{ Date last modified:  October 20th, 1998                                     }
{-----------------------------------------------------------------------------}
{ This format was borrowed from Brad Stowers.  Primarily because he didn't    }
{ mind and I didn't see anything wrong with it.  Also, since he is kind       }
{ enough to provide a home for this component, I thought the DFS format       }
{ would be more familar to him and other visitors.  -- Jon                    }
{-----------------------------------------------------------------------------}

// BCB3 does not generate a duplicate resource warning if the component's icon
// is included twice.  So explicitly include here to make sure it makes it to
// the palette.

{$ifdef DFS_CPPB_3_UP}
  {$R WebFileInfo.dcr}
{$endif}

//---------------------------------------------------------------------------
unit WebFileInfo;

interface

uses
  Classes, Controls, ComCtrls, DsgnIntf, Forms,
  Graphics, StdCtrls, SysUtils, Windows;

type

  TDownloadProgressEvent = procedure(Sender: TObject;
                                     BytesReceived: LongInt;
                                     FileSize: LongInt) of object;

  TWfiDateFormat    = (wfidDFSDate, wfidNone);

  TWfiOption        = (wfioAlwaysOverwrite, wfioDisplayUI, wfioDisplayReadMe,
                       wfioRunUpdate, wfioExitDuringUpdate);
  TWfiOptions       = set of TWfiOption;

//  TWfiState         = (wfiReady, wfiGetInfo, wfiGetReadMe, wfiDownloading);

  TWfiVersionFormat = (wfivBuildNumber, wfivSimpleString);

  TWfiTags = class(TPersistent)
  public
    FAuthorTag:      String;
    FDateTag:        String;
    FDownloadTag:    String;
    FProductNameTag: String;
    FReadMeTag:      String;
    FReadMeExtTag:   String;
    FRedirectTag:    String;
    FRunParamsTag:   String;
    FVersionTag:     String;
  published
    property AuthorTag:      String  read FAuthorTag       write FAuthorTag;
    property DateTag:        String  read FDateTag         write FDateTag;
    property DownloadTag:    String  read FDownloadTag     write FDownloadTag;
    property ProductNameTag: String  read FProductNameTag  write FProductNameTag;
    property ReadMeTag:      String  read FReadMeTag       write FReadMeTag;
    property ReadMeExtTag:   String  read FReadMeExtTag    write FReadMeExtTag;
    property RedirectTag:    String  read FRedirectTag     write FRedirectTag;
    property RunParamsTag:   String  read FRunParamsTag    write FRunParamsTag;
    property VersionTag:     String  read FVersionTag      write FVersionTag;
  end;

  TWebDlProgress = class(TForm)
    pbarProgress:   TProgressBar;
    lblDownloading: TLabel;
    lblProgress:    TLabel;
    btnCancel:      TButton;
    procedure btnCancelClick(Sender: TObject);
  public
    userCancel: Boolean;
    constructor Create(AOwner: TComponent); override;
    destructor  Destroy; override;
  end;

//---------------------------------------------------------------------------
type
  TWebFileInfo = class(TComponent)
  protected
    FActive:             Boolean;
    FCurrentVersion:     String;
    FDateFormat:         TwfiDateFormat;
    FDownloadURL:        String;
    FHomeURL:            String;
    FInfoTags:           TWfiTags;
    FLastError:          String;
    FOptions:            TWfiOptions;
    FProductAuthor:      String;
    FProductDate:        TDateTime;
    FProductDateStr:     String;
    FProductName:        String;
    FProductReadMe:      String;
    FProductReadMeExt:   String;
    FProductVersion:     String;
    FProxyHost:          String;
    FProxyPort:          Integer;
    FRedirectURL:        String;
    FRunParams:          String;
//    FState:              TwfiState;
    FUIFont:             TFont;
    FUIOwnerFont:        Boolean;
    FVersionFormat:      TWfiVersionFormat;
    FOnDownloadProgress: TDownloadProgressEvent;

    exceptionHandler:    TExceptionEvent;
    lstDocument:         TStringList;
    hostFileName:        String;
    destFile:            String;

    frmDownloadProgress: TWebDlProgress;
    frmPleaseWait:       TForm;

    procedure SetFont(value: TFont);
    function  GetFont: TFont;
    procedure SetURL(value: String);
    procedure SetOptions(value: TWfiOptions);
    procedure SetActive(value: Boolean);

    procedure ParseDocument(doc: TStringList);
    function  ExtractContent(line: String): String;

    procedure HttpCliDocData(Sender: TObject; Buffer: Pointer; Len: Integer);
    procedure DoDownloadProgress(BytesReceived: LongInt; FileSize: LongInt);

    function  ConvertProductDate(srcDate: String): TDateTime;
    function  IsUpdateAvailable: Boolean;

    procedure DownloadProgress(Sender: TObject; BytesReceived, FileSize: LongInt);
    function  CreatePleaseWait(captionStr: String): TForm;
    function  CreateReadMe: TForm;
    function  CreateProgress: TForm;

    procedure SetExceptionHandler;
    procedure ClearExceptionHandler;
    procedure WfiException(Sender: TObject; E: Exception);

  public
    constructor Create(AOwner: TComponent); override;
    destructor  Destroy; override;

    function    CheckForUpdate: Boolean;
    function    GetFileInfo: Boolean;
    function    DisplayReadMe(displayPrompt: Boolean): Boolean;
    function    DownloadFile(destPath: String): Boolean;
    function    DoDownload(srcURL, destName, progressStr: String): Boolean;

  published
    property Active:           Boolean           read FActive           write SetActive stored False;
    property CurrentVersion:   String            read FCurrentVersion   write FCurrentVersion;
    property DateFormat:       TwfiDateFormat    read FDateFormat       write FDateFormat default wfidDFSDate;
    property DownloadURL:      String            read FDownloadURL      stored False;
    property HomeURL:          String            read FHomeURL          write SetURL;
    property InfoTags:         TWfiTags          read FInfoTags         write FInfoTags;
    property LastError:        String            read FLastError        stored False;
    property Options:          TWfiOptions       read FOptions          write  SetOptions;
    property ProductName:      String            read FProductName      stored False;
    property ProductAuthor:    String            read FProductAuthor    stored False;
    property ProductDate:      TDateTime         read FProductDate      stored False;
    property ProductDateStr:   String            read FProductDateStr   stored False;
    property ProductReadMe:    String            read FProductReadMe    stored False;
    property ProductReadMeExt: String            read FProductReadMeExt stored False;
    property ProductVersion:   String            read FProductVersion   stored False;
    property ProxyHost:        String            read FProxyHost        write FProxyHost;
    property ProxyPort:        Integer           read FProxyPort        write FProxyPort default 80;
    property RunParams:        String            read FRunParams        stored False;
//    property State:            TWfiState         read FState            stored False;
    property UIFont:           TFont             read GetFont           write SetFont;
    property UIOwnerFont:      Boolean           read FUIOwnerFont      write FUIOwnerFont default True;
    property VersionFormat:    TWfiVersionFormat read FVersionFormat    write FVersionFormat default wfivBuildNumber;
    property OnDownloadProgress: TDownloadProgressEvent read  FOnDownloadProgress
                                                        write FOnDownloadProgress;
  end;

procedure Register;

implementation

uses
  Dialogs, ShellAPI, HttpProt;

type

  THttpEvents = class(TObject)
  public
    wfiObject: TWebFileInfo;
    procedure HttpRequestDone(Sender: TObject; RqType: THttpRequest; Error: Word);
  end;

  // A component editor (not really) to display the version of the component.

  TWfiEditor = class(TDefaultEditor)
  public
    function GetVerb(Index : Integer): string; override;
    function GetVerbCount : Integer; override;
  end;

{$ifdef DFS_COMPILER_3_UP}
resourcestring
{$else}
const
{$endif}

  wfisBytesReceived        = '%s of %s bytes received.';

  wfisCheckingForUpdate    = 'Checking For Update...';

  wfisConnectionErrorTitle = 'Connection Error';
  wfisConnectionUnknownErrorMsg  =
        'Could not retrieve version information from the Internet.' + #13 +
        'Error unknown.';
  wfisConnectionKnownErrorMsg  =
        'Could not retrieve version information from the Internet.' + #13 +
        'Error: "%s."';

  wfisDownloadErrorTitle = 'Download Error';
  wfisDownloadUnknownErrorMsg  =
        'Error occurred while downloading update.' + #13 + 'Error unknown.';
  wfisDownloadKnownErrorMsg  =
        'Error occurred while downloading update.' + #13 + 'Error: "%s."';

  wfisDownloadProgress     = 'Download Progress';
  wfisDownloadingReadMe    = 'Downloading ReadMe';
  wfisDownloadingUpdate    = 'Downloading Update';

  wfisDownloadUpdateMsg    = 'Download the update?';

  wfisErrorStartingProgramTitle = 'Error';
  wfisErrorStartingProgramMsg   = 'Can not start program.';

  wfisExceptionTitle       = 'Exception';
  wfisExceptionMsg         = 'Error retrieving update information.' + #13 +
                             'Exception: %s.';

  wfisFileExistsTitle      = 'File Exists';
  wfisFileExistsMsg        = 'The file %s already exists.' + #13 +
                             'Do you want to overwrite it?';

  wfisFileFilters          = 'Application (*.EXE)|*.EXE|Zipped File|*.ZIP|All Files (*.*)|*.*';

  wfisInstallExitMsg       = 'Do you want to install the update now?' + #13 +
                             'This will close %s.';

  wfisInstallUpdateTitle   = 'Install Update';
  wfisInstallUpdateMsg     = 'Do you want to install the update now?';

  wfisMissingHomeUrlTitle  = 'Missing Home URL';
  wfisMissingHomeUrlMsg    = 'Can not retrieve version information' + #13 +
                             'because no home url was provided.';

  wfisMissingTagInfoTitle  = 'Missing Tag Information';
  wfisMissingTagInfoMsg    = 'Can not retrieve version information' + #13 +
                             'because no tag information was provided.';

  wfisNoUpdateTitle        = 'No Update Found';
  wfisNoUpdateMsg          = 'There is no update available.';

  wfisPleaseWait           = 'Please Wait';

  wfisReadMeTitle          = 'Read Me';

  wfisRetrievingReadMe     = 'Retrieving ReadMe...';

  wfisUpdateAvailableTitle = 'Update Available';

  wfisVersionAvailable     = 'Current version is %s.' + #13 +
                             'Version %s is available.';

  wfisViewReadMe           = 'Would you like to view the ReadMe file?';

const
  WEBFILEINFO_VERSION = '0.81';

var
  httpClient:          THttpCli;
  httpEvents:          THttpEvents;

procedure Register;
begin
  RegisterComponents('Internet', [TWebFileInfo]);
  RegisterComponentEditor(TWebFileInfo, TWfiEditor);
end;

//---------------------------------------------------------------------------
function CommaNumber(value: LongInt): String;
var
  valueStr: String;
  valueLen: Integer;
  loop:     Integer;

begin

  valueStr := IntToStr(value);
  valueLen := Length(valueStr);
  Result   := '';

  for loop := valueLen downto 1 do begin
    if (loop < valueLen) then begin
      if (((valueLen - loop) MOD 3) = 0) then Result := ',' + Result;
    end;
    Result := valueStr[loop] + Result;
  end;

end;
//---------------------------------------------------------------------------
// TWebDlProgress methods

constructor TWebDlProgress.Create(AOwner: TComponent);
begin
  inherited CreateNew(AOwner, 0);
  userCancel := False;
end;

destructor TWebDlProgress.Destroy;
begin
  inherited Destroy;
end;

procedure TWebDlProgress.btnCancelClick(Sender: TObject);
begin
  userCancel  := True;
  httpClient.Abort();
end;

//---------------------------------------------------------------------------
// TWfiVersionProperty methods
// Component Editor that displays version of component.

function TWfiEditor.GetVerbCount: Integer;
begin
  Result := 1;
end;

function TWfiEditor.GetVerb(Index: Integer): AnsiString;
begin
  Result := 'Version ' + WEBFILEINFO_VERSION;
end;

//---------------------------------------------------------------------------
// TWebFileInfo creation and destruction methods

constructor TWebFileInfo.Create(AOwner: TComponent);
begin

  // Call the inherited Create method.

  inherited Create(AOwner);

  // Init variables.

  exceptionHandler := nil;

  // Create the FInfoTag property's InfoTags object.

  FInfoTags := TWFITags.Create;

  // Create the FUIFont property's Font object.

  FUIFont   := TFont.Create;

  // Set property defaults.

  FActive           := False;
  FOptions          := [wfioDisplayUI, wfioDisplayReadMe, wfioRunUpdate, wfioExitDuringUpdate];
  FProxyPort        := 80;
  FUIOwnerFont      := True;
  FVersionFormat    := wfivBuildNumber;

end;

destructor TWebFileInfo.Destroy;
begin
  if (httpClient <> nil) then httpClient.Free;
  if (httpEvents <> nil) then httpEvents.Free;

  FInfoTags.Free;
  FUIFont.Free;

  inherited Destroy;
end;

//---------------------------------------------------------------------------
// TWebFileInfo exception handling

procedure TWebFileInfo.SetExceptionHandler;
begin

//  if Addr(Application.OnException) = Addr(WfiException) then
//    Exit;

  exceptionHandler        := Application.OnException;
  Application.OnException := WfiException;

end;

procedure TWebFileInfo.ClearExceptionHandler;
begin

//  if Application.OnException <> @WfiException then
//    Exit;

  Application.OnException := exceptionHandler;
  exceptionHandler        := nil;

end;

procedure TWebFileInfo.WfiException(Sender: TObject; E: Exception);
begin

  if E is EHttpException then begin

    FLastError := E.Message;

    if frmPleaseWait <> nil then begin
      frmPleaseWait.ModalResult := mrCancel;
      frmPleaseWait.Release;
      frmPleaseWait := nil;
    end;

    if frmDownloadProgress <> nil then begin
      frmDownloadProgress.ModalResult := mrCancel;
      frmDownloadProgress.Release;
      frmDownloadProgress := nil;
    end;

    if httpClient.RcvdStream <> nil then begin
      httpClient.RcvdStream.Free;
      httpClient.RcvdStream := nil;
    end;

    httpClient.OnDocData  := nil;

    ClearExceptionHandler;

    Application.MessageBox(PChar(Format(wfisExceptionMsg, [E.Message])),
                           PChar(wfisExceptionTitle), MB_ICONEXCLAMATION);

    Exit;

  end;

  if Assigned(exceptionHandler) then
    exceptionHandler(Sender, E);

end;

//---------------------------------------------------------------------------
// TWebFileInfo property access methods

function  TWebFileInfo.GetFont: TFont;
begin

  if FUIOwnerFont = True then
    if Owner is TForm then
      FUIFont.Assign(TForm(Owner).Font);

  Result := FUIFont;

end;

procedure TWebFileInfo.SetFont(value: TFont);
begin
  FUIFont.Assign(value);
  FUIOwnerFont := False;
end;

//---------------------------------------------------------------------------

procedure TWebFileInfo.SetURL(value: String);
begin

  // If the URL is not long enough, then exit.

  if Length(value) < 7 then Exit;

  // If the URL is long enough, make sure a valid protocol is specified.

  if Copy(value, 1, 7) <> 'http://' then Exit;

  // Reset the active flag.

  FActive := False;

  // Set the URL.

  FHomeURL := value;

end;

//---------------------------------------------------------------------------
procedure TWebFileInfo.SetOptions(value: TWfiOptions);
begin
  FOptions := value + [wfioDisplayUI];
end;

procedure TWebFileInfo.SetActive(value: Boolean);
begin
  if (value = True) then Exit;
  FActive := value;
end;

//---------------------------------------------------------------------------
function  TWebFileInfo.GetFileInfo: Boolean;
var
  offset:     Integer;
  lastOffset: Integer;

begin

  Result := FActive;

  // If the information has already been retrieved, then don't retrieve it again.

  if FActive = true then Exit;

  // If the URL to check has not been assigned, then exit.

  if Length(FHomeURL) = 0 then begin
    Application.MessageBox(PChar(wfisMissingHomeUrlMsg),
                           PChar(wfisMissingHomeUrlTitle),
                           MB_ICONEXCLAMATION);
    Exit;
  end;

  // If none of the tag names are provided, then there's no point in getting
  // file information.

  while (True) do begin
    with FInfoTags do begin
      if Length(FProductNameTag) > 0 then break;
      if Length(FReadMeTag)      > 0 then break;
      if Length(FReadMeExtTag)   > 0 then break;
      if Length(FVersionTag)     > 0 then break;
      if Length(FDateTag)        > 0 then break;
      if Length(FAuthorTag)      > 0 then break;
      if Length(FDownloadTag)    > 0 then break;
      if Length(FRedirectTag)    > 0 then break;
      if Length(FRunParamsTag)   > 0 then break;
    end;

    Application.MessageBox(PChar(wfisMissingTagInfoMsg),
                           PChar(wfisMissingTagInfoTitle),
                           MB_ICONEXCLAMATION);
    Exit;

  end;

  // Set the application's exception handler.

  SetExceptionHandler();

  // Create the HTTP component if necessary.

  if httpClient = nil then begin
    httpEvents  := THttpEvents.Create();
    httpEvents.wfiObject := Self;
    httpClient  := THttpCli.Create(Self);
    httpClient.OnRequestDone := httpEvents.HttpRequestDone;
  end;

  // Establish proxy settings.

  if Length(FProxyHost) > 0 then begin
    httpClient.Proxy     := FProxyHost;
    httpClient.ProxyPort := IntToStr(FProxyPort);
  end
  else begin
    httpClient.Proxy     := '';
    httpClient.ProxyPort := '80';
  end;

  // Create the stream used to store the document retrieved.

  httpClient.RcvdStream := TMemoryStream.Create;

  // Get the document using ICS.

  FLastError := '';

  if (Length(FRedirectURL) = 0) then
    httpClient.URL := FHomeURL
  else
    httpClient.URL := FRedirectURL;

  httpClient.GetASync();

  // Display a message box if the caller has requested an UI.

  if wfioDisplayUI in FOptions then begin
    if frmPleaseWait = nil then
      frmPleaseWait := CreatePleaseWait(wfisCheckingForUpdate);
    frmPleaseWait.ShowModal();
  end;

  if httpClient.StatusCode > 200 then begin
    FLastError := httpClient.ReasonPhrase;
    ShowMessage(IntToStr(httpClient.StatusCode));
  end;

  // Clear the properties which will be retrieved.

  FDownloadURL      := '';
  FProductName      := '';
  FProductAuthor    := '';
  FProductDate      := 0.0;
  FProductDateStr   := '';
  FProductReadMe    := '';
  FProductReadMeExt := '';
  FProductVersion   := '';
  FRedirectURL      := '';
  FRunParams        := '';

  // If the transfer was successful, then parse the retrieved document.

  if FLastError = '' then begin

    Result := True;

    // Allocate a string list to hold the document.

    lstDocument := TStringList.Create;

    // Rewind the document and load it into a string list for processing.

    httpClient.RcvdStream.Seek(0, soFromBeginning);
    lstDocument.LoadFromStream(httpClient.RcvdStream);

    // Free the stream used to store the document.

    httpClient.RcvdStream.Free;

    // Parse the document, looking for each InfoTag.

    ParseDocument(lstDocument);

    // Free the string list used to process the document.

    lstDocument.Free;

    // Extract the filename from the download URL.

    offset     := 0;
    lastOffset := 0;

    repeat
      lastOffset := lastOffset + offset;
      offset := Pos('/', Copy(FDownloadURL, lastOffset, 1000));
    until (offset = 0);

    hostFileName := Copy(FDownloadURL, lastOffset, 1000);

  end;

  // If no redirection was specified for the file information, then free any
  // dialogs and set the result value.

  if Length(FRedirectURL) = 0 then begin

    // Clear the message box if the caller has requested an UI.

    if wfioDisplayUI in FOptions then begin
      frmPleaseWait.Free;
      frmPleaseWait := nil;
    end;

    // Set the Active property based on the result of the transfer.

    FActive := Result;

  end

  // Otherwise, a redirection has been specified.  Get the file info from the
  // new page.

  else begin
    FActive := GetFileInfo();
  end;

end;

//---------------------------------------------------------------------------
procedure TWebFileInfo.ParseDocument(doc: TStringList);
var
  loop:      Integer;
  begTag:    Integer;
  tagLen:    Integer;
  tagName:   String;
  upperLine: String;

begin

  // Examine each line of the document, looking for DFS META tags.

  for loop := 0 to doc.Count - 1 do begin

    // Get a copy of the current line.  This makes the code easier to type and
    // easier to read.

    upperLine:= UpperCase(doc.Strings[loop]);

    // Determine the offset of the META tag.

    begTag := Pos('META HTTP-EQUIV', upperLine);

    // If the META tag isn't present, then continue the loop to the next line of
    // the document.

    if begTag = 0 then continue;

    // Adjust the offset past the META tag.

    INC(begTag, 17);

    // Determine the ending offset of the tag name.

    tagLen := Pos('"', Copy(upperLine, begTag, 1000)) - 1;

    // Extract the tag name.

    tagName := Copy(doc.Strings[loop], begTag, tagLen);

    // If this tag name matches one in InfoTags, then extract the content of
    // the META tag.

    with FInfoTags do begin
      if tagName = FProductNameTag then FProductName      := ExtractContent(doc.Strings[loop]);
      if tagName = FReadMeTag      then FProductReadMe    := ExtractContent(doc.Strings[loop]);
      if tagName = FReadMeExtTag   then FProductReadMeExt := ExtractContent(doc.Strings[loop]);
      if tagName = FVersionTag     then FProductVersion   := ExtractContent(doc.Strings[loop]);
      if tagName = FDateTag        then FProductDateStr   := ExtractContent(doc.Strings[loop]);
      if tagName = FAuthorTag      then FProductAuthor    := ExtractContent(doc.Strings[loop]);
      if tagName = FDownloadTag    then FDownloadURL      := ExtractContent(doc.Strings[loop]);
      if tagName = FRedirectTag    then FRedirectURL      := ExtractContent(doc.Strings[loop]);
      if tagName = FRunParamsTag   then FRunParams        := ExtractContent(doc.Strings[loop]);
    end;

  end;

  // If the date was provided, then convert the date to a TDateTime value.

  if Length(FProductDateStr) > 0 then
    FProductDate := ConvertProductDate(ProductDateStr);

end;

//---------------------------------------------------------------------------
function TWebFileInfo.ExtractContent(line: String): String;
var
  contentOffset: Integer;
  contentLength: Integer;

begin

  contentOffset := Pos('CONTENT=', line) + 9;
  contentLength := Pos('"', Copy(line, contentOffset, 1000)) - 1;

  Result := Copy(line, contentOffset, contentLength);

end;

//---------------------------------------------------------------------------
function TWebFileInfo.DisplayReadMe(displayPrompt: Boolean): Boolean;
var
  frmReadMe:    TForm;
  runResult:    Integer;
  tempName:     String;
  offset:       Integer;
  lastOffset:   Integer;

begin

  // Assume it doesn't work.

  Result := false;

  // If we don't have the file info, then get it.

  if FActive = false then
    if GetFileInfo = false then Exit;

  // If there is not a ReadMe file available, then exit the function.

  if (Length(FProductReadMe) = 0) and (Length(FProductReadMeExt) = 0) then Exit;

  // Display a prompt to the user if the caller desired.

  if displayPrompt = true then begin
    if (Application.MessageBox(
        PChar(Format(wfisVersionAvailable, [FCurrentVersion, ProductVersion])
              + #13 + wfisViewReadMe),
        PChar(wfisUpdateAvailableTitle),
        MB_ICONQUESTION or MB_YESNO) = IDNO) then Exit;
  end;

  if (Length(FProductReadMeExt) > 0) then begin

    SetLength(tempName, MAX_PATH);

    RunResult := 0;

    // Determine a temporary file name used to store the downloaded ReadMe.

    SetLength(tempName, GetTempPath(MAX_PATH, PChar(tempName)));

    // Extract the filename from the download URL.

    offset     := 0;
    lastOffset := 0;

    repeat
      lastOffset := lastOffset + offset;
      offset := Pos('/', Copy(FProductReadMeExt, lastOffset, 1000));
    until (offset = 0);

    tempName := tempName + Copy(FProductReadMeExt, lastOffset, 1000);

    // Download the file.

    if (DoDownload(FProductReadMeExt, tempName, wfisDownloadingReadMe) = True) then
      runResult := ShellExecute(Application.MainForm.Handle, PChar('open'),
                                PChar(tempName), PChar(''), PChar(''), SW_SHOWNORMAL);

    // If the ShellExecute did not work, then display an error message.

    if (runResult <= 32) then begin
      Application.MessageBox(PChar(wfisErrorStartingProgramMsg),
                             PChar(wfisErrorStartingProgramTitle),
                             MB_ICONEXCLAMATION)
    end

    // Otherwise, the ReadMeExt has been execute.  Exit this routine.

    else begin
      Result := True;
      Exit;
    end;

  end;

  // Display the ReadMe text to the user.  This should occur if either
  // ProductReadMeExt is empty or the ShellExecute call failed above.

  if (Length(FProductReadMe) > 0) then begin
    frmReadMe := CreateReadMe();
    if (frmReadMe <> nil) then begin
      frmReadMe.ShowModal();
      frmReadMe.Free;
    end;
  end;

  Result := True;

end;

//---------------------------------------------------------------------------
function TWebFileInfo.DownloadFile(destPath: String): Boolean;
begin

  // Assume it doesn't work.

  Result := false;

  // If we don't have the file info, then get it.

  if FActive = false then
    if GetFileInfo = false then Exit;

  // Determine if the caller has sent in a filename along with the path.  If so,
  // then use that name instead.

  if destPath[Length(destPath)] <> '\' then
    destFile := destPath

  // Otherwise, extract the filename from the download URL.

  else
    destFile := destPath + hostFileName;

  // If the file already exists, then make sure the user wants to overwrite it.
  // JMR!! This logic should be improved -- TSaveDialog clashes with this...

  if (not (wfioAlwaysOverwrite in FOptions)) and
     (not (wfioDisplayUI in FOptions)) then begin
    if FileExists(destFile) then
      if Application.MessageBox(PChar(Format(wfisFileExistsMsg, [destFile])),
                                PChar(wfisFileExistsTitle),
                                MB_ICONQUESTION or MB_YESNO) = ID_NO then Exit;
  end;

  // Download the file.

  Result := DoDownload(FDownloadURL, destFile, wfisDownloadingUpdate);

end;

//---------------------------------------------------------------------------
function TWebFileInfo.DoDownload(srcURL, destName, progressStr: String): Boolean;
begin

  // Get the document using the ICS control.  Don't need to set the proxy since
  // it was set in GetFileInfo.

  FLastError := '';

  httpClient.RcvdStream := TFileStream.Create(destName, fmCreate);
  httpClient.OnDocData  := HttpCliDocData;
  httpClient.URL        := srcURL;

  httpClient.GetASync();

  // If the caller has asked for a UI, then display download status.

  if wfioDisplayUI in FOptions then begin

    // Create the status display.

    frmDownloadProgress   := TWebDlProgress(CreateProgress());
    frmDownloadProgress.lblDownloading.Caption := progressStr;
    frmDownloadProgress.ShowModal();

  end;

  if httpClient.StatusCode <= 200 then begin
    Result     := True;
  end
  else begin
    Result     := False;
    FLastError := httpClient.ReasonPhrase;
  end;

  httpClient.OnDocData  := nil;

  httpClient.RcvdStream.Free;
  httpClient.RcvdStream := nil;

  // If the caller has asked for a UI, then remove the status display.

  if wfioDisplayUI in FOptions then begin
    frmDownloadProgress.Free;
    frmDownloadProgress := nil;
  end;

end;

//---------------------------------------------------------------------------
procedure TWebFileInfo.HttpCliDocData(Sender: TObject; Buffer: Pointer; Len: Integer);
begin
  DoDownloadProgress(httpClient.RcvdCount, httpClient.ContentLength);
end;

procedure TWebFileInfo.DoDownloadProgress(BytesReceived: LongInt; FileSize: LongInt);
begin

  // If the caller has asked for a UI, then use the internal DownloadProgress
  // procedure.

  if wfioDisplayUI in FOptions then
    DownloadProgress(Self, BytesReceived, FileSize);

  // Call the assigned DownloadProgress event handler, if there is one.

  if Assigned(FOnDownloadProgress) then
    FOnDownloadProgress(Self, BytesReceived, FileSize);

end;

procedure TWebFileInfo.DownloadProgress(Sender: TObject; BytesReceived, FileSize: LongInt);
var
  progress: LongInt;

begin

  if frmDownloadProgress.userCancel = True then Exit;

  progress := (BytesReceived * 100) div FileSize;

  frmDownloadProgress.lblProgress.Caption :=
    Format(wfisBytesReceived, [CommaNumber(BytesReceived), CommaNumber(FileSize)]);

  if (progress > frmDownloadProgress.pbarProgress.Position) then
    frmDownloadProgress.pbarProgress.Position := progress;

end;

procedure THttpEvents.HttpRequestDone(Sender: TObject; RqType: THttpRequest;
                                       Error: Word);
begin

  if (Error = 0) then wfiObject.FLastError := ''
  else                wfiObject.FLastError := httpClient.ReasonPhrase;

  if wfiObject.frmPleaseWait <> nil then
    wfiObject.frmPleaseWait.ModalResult := mrOk;

  if wfiObject.frmDownloadProgress <> nil then
    with wfiObject.frmDownloadProgress do
      if userCancel = True then ModalResult := mrCancel
    else                        ModalResult := mrOk;

end;

//---------------------------------------------------------------------------
function TWebFileInfo.ConvertProductDate(srcDate: String): TDateTime;
var
  year:  Word;
  month: Word;
  day:   Word;

begin

  Result := 0.0;

  case FDateFormat of
    wfidDFSDate: begin
      month   := (Pos(UpperCase(Copy(srcDate, 1, 3)),
                      'JANFEBMARAPRMAYJUNJULAUGSEPOCTNOVDEC') + 2) div 3;
      day     := StrToInt(Copy(srcDate, Pos(',', srcDate) - 2, 2));
      year    := StrToInt(Copy(srcDate, Length(srcDate) - 3, 4));
      Result  := EncodeDate(year, month, day);
    end;
  end;

end;

function TWebFileInfo.IsUpdateAvailable: Boolean;
var
  currentVer: Integer;
  productVer: Integer;
  currentPos: Integer;
  productPos: Integer;
  currentStr: String;
  productStr: String;

begin

  Result := False;

  case FVersionFormat of

    wfivSimpleString:  begin
      if (ProductVersion > FCurrentVersion) then Result := True
      else                                       Result := False;
    end;

    wfivBuildNumber:  begin

      currentStr := FCurrentVersion;
      productStr := ProductVersion;

      repeat

        currentPos := Pos('.', currentStr);
        productPos := Pos('.', productStr);

        if currentPos > 0 then
          currentVer := StrToIntDef(Copy(currentStr, 1, currentPos - 1), 0)
        else
          currentVer := StrToIntDef(currentStr, 0);

        if productPos > 0 then
          productVer := StrToIntDef(Copy(productStr, 1, productPos - 1), 0)
        else
          productVer := StrToIntDef(productStr, 0);

        if productVer > currentVer then begin
          Result := True;
          Exit;
        end;

        if productVer < currentVer then begin
          Result := False;
          Exit;
        end;

        System.Delete(currentStr, 1, currentPos);
        System.Delete(productStr, 1, productPos);

      until currentPos = 0;

    end;

  end;

end;

//---------------------------------------------------------------------------
function TWebFileInfo.CheckForUpdate: Boolean;
var
  dlgSaveAs:    TSaveDialog;
  newFile:      String;
  dialogStr:    String;

begin

  Result := False;

  // Get the current file info from the DFS page.

  if (GetFileInfo() = false) then begin
    if Length(FLastError) > 0 then
      dialogStr := Format(wfisConnectionKnownErrorMsg, [FLastError])
    else
      dialogStr := wfisConnectionUnknownErrorMsg;

    Application.MessageBox(PChar(dialogStr),
                           PChar(wfisConnectionErrorTitle),
                           MB_ICONEXCLAMATION);
    Exit;
  end;

  // Determine if a newer version exists on the server.

  if (IsUpdateAvailable() = False) then begin
    Application.MessageBox(PChar(wfisNoUpdateMsg),
                           PChar(wfisNoUpdateTitle),
                           MB_ICONINFORMATION);
    Exit;
  end;

  // Determine if the user wants to view the ReadMe file.

  if (wfioDisplayReadMe in FOptions) then
    DisplayReadMe(True);

  // Display a message to the user and determine whether to download the update.

  if (Application.MessageBox(
        PChar(Format(wfisVersionAvailable, [FCurrentVersion, ProductVersion])
              + #13 + wfisDownloadUpdateMsg),
        PChar(wfisUpdateAvailableTitle),
        MB_ICONQUESTION or MB_YESNO) = ID_NO) then Exit;

  // Determine where the user wants to place the file.

  dlgSaveAs := TSaveDialog.Create(Application);

  dlgSaveAs.Filter     := wfisFileFilters;
  dlgSaveAs.Options    := [ofOverwritePrompt, ofPathMustExist];
  dlgSaveAs.FileName   := hostFileName;

  if (dlgSaveAs.Execute() = false) then begin
    dlgSaveAs.Free;
    Exit;
  end;

  // Determine if the resulting filename contains an extension or not.

  newFile := dlgSaveAs.FileName;

  if ExtractFileExt(newFile) = '' then begin
    case dlgSaveAs.FilterIndex of
      1: newFile := newFile + '.EXE';
      2: newFile := newFile + '.ZIP';
    end;
  end;

  // Delete the browse directory dialog.

  dlgSaveAs.Free;

  // Download the file.

  Result := DownloadFile(newFile);

  // If the download did not succeed, then exit the function.

  if (Result = false) then begin

    if Length(FLastError) > 0 then
      dialogStr := Format(wfisDownloadKnownErrorMsg, [FLastError])
    else
      dialogStr := wfisDownloadUnknownErrorMsg;

    Application.MessageBox(PChar(dialogStr),
                           PChar(wfisDownloadErrorTitle),
                           MB_ICONEXCLAMATION);
    Exit;

  end;

  // If the user desires, then run the update.

  if wfioRunUpdate in FOptions then begin

    if wfioExitDuringUpdate in FOptions then
      dialogStr := Format(wfisInstallExitMsg, [Application.Title])
    else
      dialogStr := wfisInstallUpdateMsg;

    if Application.MessageBox(PChar(dialogStr),
                              PChar(wfisInstallUpdateTitle),
                              MB_ICONQUESTION or MB_YESNO) = IDYES then begin

      if (ShellExecute(Application.MainForm.Handle, PChar('open'), PChar(destFile),
                       PChar(FRunParams), PChar(''), SW_SHOWNORMAL) <= 32) then begin
        Application.MessageBox(PChar(wfisErrorStartingProgramMsg),
                               PChar(wfisErrorStartingProgramTitle),
                               MB_ICONEXCLAMATION);
        Exit;
      end;

      // If the user desires, then exit the application.

      if wfioExitDuringUpdate in FOptions then
        Application.Terminate;

    end;

  end;

  Result := True;

end;

//---------------------------------------------------------------------------
function TWebFileInfo.CreatePleaseWait(captionStr: String): TForm;
var
  frmWait:    TForm;
  msgLabel:   TLabel;

begin

  frmWait := TForm.Create(Application);
  with frmWait do begin
    BorderIcons := [];
    BorderStyle := bsDialog;
    Caption     := wfisPleaseWait;
    Cursor      := crHourGlass;
    Height      := 100;
    Position    := poScreenCenter;
    Width       := 240;
  end;

  msgLabel := TLabel.Create(frmWait);
  with msgLabel do begin
    Parent       := frmWait;
    Left         := 0;
    Top          := 27;
    Width        := 232;
    Height       := 23;
    Alignment    := taCenter;
    AutoSize     := False;
    Caption      := captionStr;
    ParentFont   := False;

    Font.Assign(FUIFont);
  end;

  Result := frmWait;

end;

//---------------------------------------------------------------------------
function TWebFileInfo.CreateReadMe: TForm;
var
  frmReadMe: TForm;
  memoText:  TMemo;
  btnOk:     TButton;

begin

  Result := nil;

  // Create the stream used to store the document retrieved.

  httpClient.RcvdStream := TMemoryStream.Create;

  // Get the document using the ICS control.

  FLastError := '';
  httpClient.URL := FProductReadMe;
  httpClient.GetASync();

  // Display a Please Wait message box to the user.

  frmPleaseWait := CreatePleaseWait(wfisRetrievingReadMe);
  frmPleaseWait.ShowModal();

  if httpClient.StatusCode > 200 then begin
    FLastError := httpClient.ReasonPhrase;
  end

  else begin

    // Create the read me form.

    frmReadMe := TForm.Create(Application);
    with frmReadMe do begin
      BorderIcons := [];
      BorderStyle := bsDialog;
      Caption     := wfisReadMeTitle;
      Height      := (Screen.Height * 4) div 5;
      Position    := poScreenCenter;
      Width       := (Screen.Width  * 4) div 5;

      Font.Assign(FUIFont);
    end;

    // Create the memo field to hold the read me text.

    memoText := TMemo.Create(frmReadMe);
    with memoText do begin
      Parent       := frmReadMe;
      Left         := 10;
      Top          := 10;
      Width        := frmReadMe.Width  - 25;
      Height       := frmReadMe.Height - 80;
      ScrollBars   := ssBoth;
      ParentFont   := False;

      Font.Assign(FUIFont);
    end;

    // Create the Ok button to close the form.

    btnOk := TButton.Create(frmReadMe);
    with btnOk do begin
      Parent       := frmReadMe;
      Left         := (frmReadMe.Width div 2) - 34;
      Top          := frmReadMe.Height - 56;
      Width        := 68;
      Height       := 22;
      Caption      := '&Ok';
      ModalResult  := mrOk;
    end;

    // Rewind the document and load it into a string list for processing.

    httpClient.RcvdStream.Seek(0, soFromBeginning);
    memoText.Lines.LoadFromStream(httpClient.RcvdStream);

    // Return the read me form back to the caller.

    Result := frmReadMe;

  end;

  // Free the stream used to store the document.

  httpClient.RcvdStream.Free;

  // Clear the Please Wait message box.

  frmPleaseWait.Free;
  frmPleaseWait := nil;

end;

//---------------------------------------------------------------------------
function TWebFileInfo.CreateProgress: TForm;
var
  WebDlProgress : TWebDlProgress;

begin

  WebDlProgress := TWebDlProgress.Create(Application);
  with WebDlProgress do begin
    Left          := 393;
    Top           := 111;
    Cursor        := crHourGlass;
    BorderIcons   := [];
    BorderStyle   := bsDialog;
    Caption       := wfisDownloadProgress;
    ClientHeight  := 123;
    ClientWidth   := 272;
    Position      := poScreenCenter;

    Font.Assign(FUIFont);
  end;

  WebDlProgress.lblDownloading := TLabel.Create(WebDlProgress);
  with WebDlProgress.lblDownloading do begin
    Parent     := WebDlProgress;
    Left       := 2;
    Top        := 10;
    Width      := 268;
    Height     := 18;
    Alignment  := taCenter;
    AutoSize   := False;
    ParentFont := True;
  end;

  WebDlProgress.lblProgress := TLabel.Create(WebDlProgress);
  with WebDlProgress.lblProgress do begin
    Parent     := WebDlProgress;
    Left       := 2;
    Top        := 60;
    Width      := 268;
    Height     := 22;
    Alignment  := taCenter;
    AutoSize   := False;
    Caption    := '';
    ParentFont := True;
  end;

  WebDlProgress.pbarProgress := TProgressBar.Create(WebDlProgress);
  with WebDlProgress.pbarProgress do begin
    Parent     := WebDlProgress;
    Left       := 36;
    Top        := 40;
    Width      := 200;
    Height     := 14;
    Cursor     := crHourGlass;
    Min        := 0;
    Max        := 100;
    TabOrder   := 0;
  end;

  WebDlProgress.btnCancel := TButton.Create(WebDlProgress);
  with WebDlProgress.btnCancel do begin
    Parent       := WebDlProgress;
    Left         := 102;
    Top          := 94;
    Width        := 68;
    Height       := 22;
    Caption      := '&Cancel';
    ParentFont   := True;
    TabOrder     := 1;
    OnClick      := WebDlProgress.btnCancelClick;
  end;

  Result := WebDlProgress;

end;

end.
