{
  wdwstate.pas
    v3.1 of 05/04/2000
      Components that can record read and restore a window's state, position and
      size using ini files or the registry.
}

{
  History
  - v1.0 of 15/08/1999
      Original version of TPJWdwState component that stores window state
      information in ini files.
  - v2.0 of 18/10/1999
      Re-write which added TPJRegWdwState component that stores window state
      information in registry. Extracted common functionality of TPJWdwState and
      TPJRegWdwState into new abstract base class TPJCustomWdwState. TPJWdwState
      presents same interface and functionality as version 1. TRegWdwState is
      not compile under Delphi 2.
  - v2.1 of 29/11/1999
      Renamed ReadState and SaveState methods to ReadWdwState and SaveWdwState
      respectively since ReadState was masking a method inherited from
      TComponent.
  - v3.0 of 05/04/2000
      + Added AutoSaveRestore property*
      + Modified window restoration code to avoid task bars in positions other
        than bottom of screen*
      + Prevented forms being restored as minimized from flashing on screen (by
        cheating!)
      Thanks to Stefan Winter for suggesting moficiations marked * and for the
      prototype coding.
      This version is no longer compatible with Delphi 1.
  - v3.1 of 29/04/2000
      + Bug fix curing problem of incorrect maximisation (had been using window
        size when "maximising" rather than maximising to full screen).
}

{
  Dependencies
      NONE
}

{
  Credits:
    Stefan Winter suggested (and provided prototype coding for) the
    AutoSaveRestore property
    Stefan also provided prototype of code that prevents windows overwriting
    task bar.
}

{
  Legal notice
    This component is copyright  P.D.Johnson, 1999-2000.
    The source code and help files can be freely distributed on a not-for-profit
    basis providing that:
      + the source code is not altered.
      + this readme file is distributed with it unchanged
    By not-for-profit I mean that you may recover out of pocket expenses
    incurred in distributing the code, but should not make a profit from this.
    If you discover any bugs in this implementation, or if you have any update
    suggestions, please contact me on peter.johnson@openlink.org.
    Please do modify the code for you own use. I'd like to see any changes you
    make - I could incorporate them into future versions. Please notify me of
    changes on at the above e-mail address.
    This software is provided as is - no warranty is given as to its suitability
    for any purposes to which you may wish to put it.
}


unit wdwstate;

interface

uses
  Messages, Classes, WinTypes, WinProcs, Forms, IniFiles, Registry;

type
  {Abstract base class for components that record window size, position and
  state between program executions}
  TPJCustomWdwState = class(TComponent)
  private {properties}
    FIgnoreState: Boolean;
    FAutoSaveRestore: Boolean;
    procedure SetAutoSaveRestore(const AValue: Boolean);
  private
    FWindow: TForm;
      {Instance of window on which to operate}
    FOnCreateAccessed: Boolean;
      {Flag set true once NewFormOnCreate method has been executed}
    FOldFormOnCreate: TNotifyEvent;
      {Reference to form's existing OnCreate event handler}
    FOldFormOnDestroy: TNotifyEvent;
      {Reference to form's existing OnDestroy event handler}
    FStorageID: string;
      {Identifies storage used for window information - eg ini file name or
      registry sub-key}
    procedure NewFormOnCreate(Sender:TObject);
      {Replaces form's OnCreate event handler with one that causes window state
      to be restored - used when AutoSaveRestore is true}
    procedure NewFormOnDestroy(Sender:TObject);
      {Replaces form's OnDestroy event handler with one that causes window state
      to be saved - used when AutoSaveRestore is true}
    procedure UpdateFormEventHandlers;
      {Updates form's OnCreate and OnDestroy event handlers. If AutoSaveRestore
      property is true then we link our save and restore event handlers into
      form's handlers - and if AutoSaveRestore is false we restore them if
      required}
  protected
    procedure Loaded; override;
      {Updates form's event handlers if AutoSaveRestore is true}
    procedure ReadWdwState(var Left, Top, Width, Height, State : Integer);
      virtual; abstract;
      {Read state of window from storage - user passes default values and method
      returns new values from storage, or defaults if no stored values found.
      Implementation depends on method of storage used by derived classes}
    procedure SaveWdwState(const Left, Top, Width, Height, State : Integer);
      virtual; abstract;
      {Save state of window to storage per given values. Implementation depends
      on method of storage used by derived classes}
    function GetStorageID : string;
      {Returns identification of where information is stored - can be ini file
      name or registry sub-key}
    procedure SetStorageID(AnID : string); virtual;
      {Sets identification of where window information is stored - can be ini
      file name or registry sub-key. Can be overriden to add additional
      processing}
  public
    constructor Create(AOwner: TComponent); override;
      {Class constructor}
    procedure Restore;
      {Read window placement and state from storage and set up the window}
    procedure Save;
      {Save window placement and size to storage}
  published
    property AutoSaveRestore : Boolean
      read FAutoSaveRestore write SetAutoSaveRestore default False;
      {When true component automatically restores window state on form creation
      and saves it on form destruction. Note - this property kooks into form's
      OnCreate and OnDestroy event handlers - if user changes assigns at run-
      time then this property may not work correctly unless it has its value
      (re)assigned after event handlers have been set}
    property IgnoreState : Boolean read FIgnoreState write FIgnoreState;
      {When true Restore method ignores the saved state of the window and leaves
      current state unchanged while still setting size and position. When false
      Restore also sets the window state according to the saved state}
  end;

  {Implements window state storage via ini file}
  TPJWdwState = class(TPJCustomWdwState)
  private {properties}
    FSection : string;
    procedure SetSection(AnID : string); virtual;
  protected
    procedure ReadWdwState(var Left, Top, Width, Height, State : Integer);
      override;
      {Reads window state from ini file using given values as defaults}
    procedure SaveWdwState(const Left, Top, Width, Height, State : Integer);
      override;
      {Writes window state to ini file}
    procedure SetStorageID(AnID : string); override;
      {Sets ini file name to given value - or to default name if AnID is false}
  public
    constructor Create(AOwner: TComponent); override;
      {Overridden class constructor sets default section name}
  published
    property IniFileName : string read GetStorageID write SetStorageID;
      {The name of the ini file in which to save window information - uses path
      and name of executable with extension replaced by .ini if set to empty
      string}
    property Section : string read FSection write SetSection;
      {The name of the section in ini file in which to save window information -
      uses "Main Window" if passed emtpy string}
  end;

  {Implements window state storage via registry}
  TPJRegWdwState = class(TPJCustomWdwState)
  private {properties}
    FRootKey : HKEY;
  private
    procedure WriteInt(const Reg: TRegistry; const AName : string;
      const AnInt : Integer);
      {Writes integer value to registry in current key with given name}
    function ReadInt(const Reg: TRegistry; const AName: string;
      const ADefault: Integer) : Integer;
      {Reads integer value from registry using given name in current key and
      returns it. If no such value exists returns given default value}
  protected
    procedure ReadWdwState(var Left, Top, Width, Height, State : Integer);
      override;
      {Reads window info from registry using given values as defaults}
    procedure SaveWdwState(const Left, Top, Width, Height, State : Integer);
      override;
      {Writes window info to registry}
    procedure SetStorageID(AnID : string); override;
      {Sets registry sub key below root entry to given name, or to a default
      value if empty string is passed}
  public
    constructor Create(AOwner: TComponent); override;
      {Overriden constructor - sets default root key}
  published
    property RootKey : HKEY read FRootKey write FRootKey
      default HKEY_CURRENT_USER;
      {The registry root key to use - must set to valid HKEY values}
    property SubKey : string read GetStorageID write SetStorageID;
      {The sub-key below root key where window information is to be stored - if
      set to empty string the value of
        '/Software/<EXECUTABLE FILE NAME>/Main Window'
      is used}
  end;


procedure Register;

implementation

uses
  SysUtils;


{ TPJWdwState }

constructor TPJWdwState.Create(AOwner: TComponent);
  {Overridden class constructor sets default section name}
begin
  inherited Create(AOwner);
  SetSection('');
end;

procedure TPJWdwState.ReadWdwState(var Left, Top, Width, Height, State : Integer);
  {Reads window state from ini file using given values as defaults}
var
  Ini : TIniFile;             {Instance of ini file class used to read info}
begin
  {Open .ini file and read window info from it}
  Ini := TIniFile.Create(GetStorageID);
  try
    Left := Ini.ReadInteger(Section, 'Left', Left);
    Top := Ini.ReadInteger(Section, 'Top', Top);
    Width := Ini.ReadInteger(Section, 'Width', Width);
    Height := Ini.ReadInteger(Section, 'Height', Height);
    State := Ini.ReadInteger(Section, 'State', State);
  finally
    Ini.Free;
  end;
end;

procedure TPJWdwState.SaveWdwState(const Left, Top, Width, Height, State
  : Integer);
  {Writes window state to ini file}
var
  Ini : TIniFile;             {Instance of ini file class used to read info}
begin
  {Open .ini file and write window info to it}
  Ini := TIniFile.Create(GetStorageID);
  try
    Ini.WriteInteger(Section, 'Left', Left);
    Ini.WriteInteger(Section, 'Top', Top);
    Ini.WriteInteger(Section, 'Width', Width);
    Ini.WriteInteger(Section, 'Height', Height);
    Ini.WriteInteger(Section, 'State', State);
  finally
    Ini.Free;
  end;
end;

procedure TPJWdwState.SetStorageID(AnID : string);
  {Sets ini file name to given value - or to default name if AnID is false}
begin
  if (AnID = '') and not (csDesigning in ComponentState) then
    {User provided empty string - set default (but only when we're not
    designing}
    AnID := ChangeFileExt(Application.ExeName, '.ini');
  {Now record adjusted ID}
  inherited SetStorageID(AnID);
end;

procedure TPJWdwState.SetSection(AnID : string);
  {Write access method for Section property - records section file name and
  substitutes default value of 'Main Window' if name provided is nul string.
  This substitution doesn't take place when designing the form}
begin
  FSection := AnID;
  if (AnID = '') and not (csDesigning in ComponentState) then
    FSection := 'Main Window';
end;


{ TPJRegWdwState }

constructor TPJRegWdwState.Create(AOwner: TComponent);
  {Overriden constructor - sets default root key}
begin
  inherited Create(AOwner);
  FRootKey := HKEY_CURRENT_USER;
end;

procedure TPJRegWdwState.ReadWdwState(var Left, Top, Width, Height, State
  : Integer);
  {Reads window info from registry using given values as defaults}
var
  Reg : TRegistry;      {Instance of registry ini file emulator class used to
                        read info}
begin
  {Read info from registry file}
  {open registry}
  Reg := TRegistry.Create;
  try
    {use required root and sub keys}
    Reg.RootKey := FRootKey;
    Reg.OpenKey(SubKey, True);
    {read position, size and state of window}
    Left := ReadInt(Reg, 'Left', Left);
    Top := ReadInt(Reg, 'Top', Top);
    Width := ReadInt(Reg, 'Width', Width);
    Height := ReadInt(Reg, 'Height', Height);
    State := ReadInt(Reg, 'State', State);
  finally
    {close registry}
    Reg.Free;
  end;
end;

function TPJRegWdwState.ReadInt(const Reg: TRegistry; const AName: string;
  const ADefault: Integer) : Integer;
  {Reads integer value from registry using given name in current key and returns
  it. If no such value exists returns given default value}
begin
  if Reg.ValueExists(AName) then
    Result := Reg.ReadInteger(AName)
  else
    Result := ADefault;
end;

procedure TPJRegWdwState.SaveWdwState(const Left, Top, Width, Height, State
  : Integer);
  {Writes window info to registry}
var
  Reg : TRegistry;      {Instance of registry ini file emulator class used to
                        write info}
begin
  {Read info from registry}
  {open registry}
  Reg := TRegistry.Create;
  try
    {use required root and sub keys}
    Reg.RootKey := FRootKey;
    Reg.OpenKey(SubKey, True);
    {read window size, position and state from registry}
    WriteInt(Reg, 'Left', Left);
    WriteInt(Reg, 'Top', Top);
    WriteInt(Reg, 'Width', Width);
    WriteInt(Reg, 'Height', Height);
    WriteInt(Reg, 'State', State);
  finally
    {close registry}
    Reg.Free;
  end;
end;

procedure TPJRegWdwState.WriteInt(const Reg: TRegistry; const AName : string;
  const AnInt : Integer);
  {Writes integer value to registry in current key with given name}
begin
  Reg.WriteInteger(AName, AnInt);
end;

procedure TPJRegWdwState.SetStorageID(AnID : string);
  {Sets registry sub key below root entry to given name, or to a default value
  if empty string is passed}
begin
  if (AnID = '') and not (csDesigning in ComponentState) then
    {User set to '' so use default value based on program name - but not if
    we're designing}
    AnID := Format('\Software\%s\Main Window',
      [ExtractFileName(Application.ExeName)]);
  {Record adjusted ID}
  inherited SetStorageID(AnID);
end;


{ TPJCustomWdwState }

constructor TPJCustomWdwState.Create(AOwner: TComponent);
  {Class constructor}
var
  TheForm : TComponent;       {The form that ultimately owns this component}
begin
  {Call inherited constructor}
  inherited Create(AOwner);

  {Check if owner is a form, if not, move up tree of owners until we either find
  a form or a nil reference}
  TheForm := AOwner;
  while (TheForm <> nil) and not (TheForm is TForm) do
    TheForm := TheForm.Owner;
  {Record form window reference appropriately}
  if TheForm <> nil then
    FWindow := TheForm as TForm
  else
    FWindow := nil;
  {Set default identifier for where window info is to be kept}
  SetStorageID('');
  {Set default state of AutoSaveRestore property}
  FAutoSaveRestore := False;
end;

procedure TPJCustomWdwState.Loaded;
  {Updates form's event handlers if AutoSaveRestore is true}
begin
  inherited Loaded;
  UpdateFormEventHandlers;
end;

procedure TPJCustomWdwState.NewFormOnCreate(Sender:TObject);
  {Replaces form's OnCreate event handler with one that causes window state to
  be restored - used when AutoSaveRestore is true}
begin
  {Restore window state if we haven't alreay done it}
  if not FOnCreateAccessed then
  begin
    Restore;
    FOnCreateAccessed := True;
  end;
  {Call original event handler, if assigned}
  if Assigned(FOldFormOnCreate) then
    FOldFormOnCreate(Sender);
end;

procedure TPJCustomWdwState.NewFormOnDestroy(Sender:TObject);
  {Replaces form's OnCreate event handler with one that causes window state to
  be saved - used when AutoSaveRestore is true}
begin
  {Save window state}
  Save;
  {Call original event handler, if assigned}
  if Assigned(FOldFormOnDestroy) then
    FOldFormOnDestroy(Sender);
end;

procedure TPJCustomWdwState.Restore;
  {Read window placement and state from ini file and set up the window}
var
  Left, Top : Integer;        {Position of window}
  Width, Height : integer;    {Dimensions of window}
  State : TWindowState;       {State of window}
  StateInt : Integer;         {State of window as integer}
  Pl : TWindowPlacement;      {Info structure for placement of window}
begin
  {Check if there is an owning window - get out if not}
  if FWindow = nil then Exit;

  {Read info from storage}
  {set default values in case any of items are not recorded in storage}
  Left := FWindow.Left;
  Top := FWindow.Top;
  Width := FWindow.Width;
  Height := FWindow.Height;
  StateInt := Ord(FWindow.WindowState);
  {read state and size of window from storage - using default set above}
  ReadWdwState(Left, Top, Width, Height, StateInt);
  State := TWindowState(StateInt);

  {Set up window placement record}
  FillChar(Pl, SizeOf(Pl), #0);       // makes window 0x0x0x0 until values set
  Pl.Length := SizeOf(TWindowPlacement);
  pl.ShowCmd := SW_HIDE;              // we set window's size without displaying

  {Now set window state if required}
  if not IgnoreState then
  begin
    {check if window was minimised}
    if State = wsMinimized then
    begin
      {we have special processing for minimized state since Delphi doesn't
      minimize windows - it uses application window instead - so we hide the
      form}
      {form has to be set to visible for this too work (don't know why) - so we
      temporarily make it 0 pixels wide so it doesn't flash on screen}
      SetWindowPlacement(FWindow.Handle, @Pl);    // zero size - zeroed Pl above
      FWindow.Visible := True;
      {now we miminize application}
      Application.Minimize;
    end
  end
  else
  begin
    {ignoring state - show window normally}
    State := wsNormal;
  end;

  {Set normal size of window}
  Pl.Length := SizeOf(TWindowPlacement);
  Pl.rcNormalPosition.Left := Left;
  Pl.rcNormalPosition.Top := Top;
  Pl.rcNormalPosition.Right := Left+Width;
  Pl.rcNormalPosition.Bottom := Top+Height;
  SetWindowPlacement(FWindow.Handle, @Pl);

  {Update window's state per State if it's not minimized - set to normal if it
  is}
  if State <> wsMinimized then
    FWindow.WindowState := State
  else
    FWindow.WindowState := wsNormal;
end;

procedure TPJCustomWdwState.Save;
  {Save window placement and size to ini file}
var
  Pl : TWindowPlacement;      {Info structure for placement of window}
  R : TRect;                  {Rectangle to hold normal position & size details}
  State : Integer;            {State of window}
begin
  {Check if there is an owning window - get out if not}
  if FWindow = nil then Exit;
  {Calculate window's normal size and position using Windows API call - the
  form's Width, Height, Top and Left properties will give actual window size if
  form is maximised, which is not what we want here}
  Pl.Length := SizeOf(TWindowPlacement);
  GetWindowPlacement(FWindow.Handle, @Pl);
  R := Pl.rcNormalPosition;
  {Record window state (maximised, minimised or normal) - special case when
  minimized since form window is simply hidden when minimised, and application
  window is actually the one minimised - so we check to see if application
  window *is* minimized and act accordingly}
  if IsIconic(Application.Handle) then
    {minimized - write that state}
    State := Ord(wsMinimized)
  else
    {not mimimized - we can rely on window state of form}
    State := Ord(FWindow.WindowState);
  {Save window info}
  SaveWdwState(R.Left, R.Top, R.Right-R.Left, R.Bottom-R.Top, State);
end;

function TPJCustomWdwState.GetStorageID : string;
  {Returns identification of where information is stored - can be ini file name
  or registry sub-key}
begin
  Result := FStorageID;
end;

procedure TPJCustomWdwState.SetStorageID(AnID : string);
  {Sets identification of where window information is stored - can be ini file
  name or registry sub-key. Can be overriden to add additional processing}
begin
  FStorageID := AnID;
end;

procedure TPJCustomWdwState.SetAutoSaveRestore(const AValue: Boolean);
  {Write access method for AutoSaveRestore property - updates form's event
  handlers according to state of property}
begin
  FAutoSaveRestore := AValue;
  UpdateFormEventHandlers;
end;

procedure TPJCustomWdwState.UpdateFormEventHandlers;
  {Updates form's OnCreate and OnDestroy event handlers. If AutoSaveRestore
  property is true then we link our save and restore event handlers into form's
  handlers - and if AutoSaveRestore is false we restore them if required}
begin
  if not (csDesigning in Componentstate)
    and not(csLoading in ComponentState) then
    if FAutoSaveRestore then
    begin
      {Link our special event handlers in to forms's OnCreate and OnDestroy
      event handlers}
      FOldFormOnCreate  := FWindow.OnCreate;
      FOldFormOnDestroy := FWindow.OnDestroy;
      FWindow.OnCreate  := NewFormOnCreate;
      FWindow.OnDestroy := NewFormOnDestroy;
    end
    else
    begin
      {Restore form's old event handlers, unless user has assigned new ones}
      if @TPJCustomWdwState.NewFormOnCreate = @FWindow.OnCreate  then
        FWindow.OnCreate := FOldFormOnCreate;
      if @TPJCustomWdwState.NewFormOnDestroy = @FWindow.OnDestroy then
        FWindow.OnDestroy := FOldFormOnDestroy;
    end;
end;

procedure Register;
begin
  RegisterComponents('PJ Stuff', [TPJWdwState]);
  RegisterComponents('PJ Stuff', [TPJRegWdwState]);
end;

end.
