unit Watch;
{
The TWatch Component V1.01

  MOST IMPORTANT :)
  =================
  This is NOT Freeware: It's PostCardWare. When you use
  this component or think it's useful, send me a post-card
  to: Florian Bmers, Colmarer Str.11, D - 28211 Bremen, Germany

  And of course, I am very interested in any application
  that uses this component (or any other application you wrote).
  If so, mail me (not the program, just an URL or similar) !
  (mail address below)

  Installation:
  =============
  1. Copy the File Watch.pas to the directory where
     you store your components (or let it where it is)
  2. The library watching.dll must be in the search path or in
     the directory of the program you created. It is loaded and
     unloaded dynamically when needed, in order to prevent Delphi
     to load it when installing the component.
  3. In Delphi, select Component|Install Component. In the
     following dialog enter the path and filename of
     Watch.pas and hit OK.
  3. Now the TWatch Component is available in the
     Component palette under Samples.

  Description
  ===========
  This component enables you to track some system events.
  Currently implemented are Shell events (i.e. Activation,
  Creation, Closure of Applications) and Keyboard events.

  The shell events have one drawback: The name parameter is
  identical for all instances of a watched program. You
  can only use the Handle parameter of the event to distinguish
  different instances of the same application.

  Note: the source for watching.dll is in the project watching.dpr.


  How to use it
  =============
  Properties:
  - NotifyOwnEvents: Whether an event shall be sent when
                     the own application receives a shell event
  - RepeatedEvents: When you activate different windows
                    of the same application, each time an
                    event is sent. If you set this to false,
                    only the first event will be sent.
  Events:
  I think, they do what they're named (?!)

  ideas for enhancements:
  - make it usable for multiple applications...
  - of course, add the other hooks
  - use toAscii in order to get an Ascii-Character of a key
  - let the dll get some information about the window
    in which are typed the keys
  - possibility to change messages
  - let the dll change specific things while in context
    of another program e.g. add to every menu an own menu...
    or just get some information about that program

  Copyright
  =========
  (c) 1997/1998 by Florian Bmers

  send any comments, proposals, enhancements etc. to:
  florian@bome.com

  new versions on:
  http://www.bome.com/

}
interface
uses
 SysUtils,
 Classes,
 Forms,
 Windows,
 Dialogs,
 Messages;

const
  watchdll='watching.dll';

type
  TWatchType=( wtSHELL,wtKEYBOARD,                  //implemented
               wtCALLWNDPROC,wtGETMESSAGE,          //not yet implemented
               wtMOUSE,wtMSGFILTER,wtSYSMSGFILTER); //not yet implemented

  TKeyInfo=record
   RepeatCount:Integer;
   ScanCode:Integer;
   Extended:Boolean;
   AltDown:Boolean;
   PreviousDown:Boolean;
   Down:Boolean;
   KeyNameText:String;
  end;

  //procedures in the dll
  TWatching=procedure(WatchType:TWatchType); stdcall;
  TStopAll=procedure; stdcall;
  TStillWatching=function:Boolean; stdcall;
  TSetReceiver=procedure(r:Integer); stdcall;

  // procedures for events
  TShellEvent=procedure( Sender:TObject;
                         Handle:THandle;
                         AppName:String) of Object;
  TKeyNativeEvent=procedure( Sender:TObject;
                             wParam,lParam:Integer) of Object;

  TWatch = class(TComponent)
  private
   // invisible window that receives messages from the dll
   HWNDHandle:THandle;
   // handle of the library
   LibHandle:THandle;
   DontLoadLib:Boolean;
   // message IDs for events of the dll
   SHELL_EVENT:Integer;
   KEY_EVENT:Integer;
   // procedures in the DLL
   StartWatching:TWatching;
   StopWatching:TWatching;
   StopAll:TStopAll;
   StillWatching:TStillWatching;
   SetReceiver:TSetReceiver;

   {----- SHELL EVENTS ------}
   FNotifyOwnEvents:Boolean;
   FRepeatedEvents:Boolean;
   LastActivated:String;
   FOnActivateShellWindow:TShellEvent;
   FOnWindowActivated: TShellEvent;
   FOnWindowCreated: TShellEvent;
   FOnWindowDestroyed: TShellEvent; //Window still lives when this call comes
   {----- KEY EVENTS --------}
   FOnKeyNative:TKeyNativeEvent;

   procedure ResetLibVars;
   procedure LoadLib;
   procedure UnLoadLib;

   procedure StartStop(start:Boolean; WatchType:TWatchType);
   procedure SetOnActivateShellWindow(h:TShellEvent);
   procedure SetOnWindowActivated(h:TShellEvent);
   procedure SetOnWindowCreated(h:TShellEvent);
   procedure SetOnWindowDestroyed(h:TShellEvent);
   procedure SetOnKeyNative(h:TKeyNativeEvent);


  protected
   function GetAppName(window:THandle):String;
   procedure OnShellHook(nCode: Integer; wParam: Longint);
   procedure OnKeyHook(wParam,lParam: Longint);

  public
   procedure OnHWNDEvent(var Message: TMessage);
   function getVKString(keyCode:Integer):String;
   procedure getKeyInfo(lParam:Integer; var KeyInfo:TKeyInfo);
   procedure ReleaseLib;
   constructor Create(AOwner:TComponent); override;
   destructor Destroy; override;

   procedure SetDLLReceiver(r:Integer);

  published
   property NotifyOwnEvents:Boolean read FNotifyOwnEvents write FNotifyOwnEvents default false;
   property RepeatedEvents:Boolean read FRepeatedEvents write FRepeatedEvents default false;
   property OnActivateShellWindow: TShellEvent
            read FOnActivateShellWindow write SetOnActivateShellWindow default nil;
   property OnWindowActivated: TShellEvent
            read FOnWindowActivated write SetOnWindowActivated default nil;
   property OnWindowCreated: TShellEvent
            read FOnWindowCreated write SetOnWindowCreated default nil;
   property OnWindowDestroyed: TShellEvent
            read FOnWindowDestroyed write SetOnWindowDestroyed default nil;
   property OnKeyNative: TKeyNativeEvent
            read FOnKeyNative write SetOnKeyNative default nil;
  end;

procedure Register;

implementation

constructor TWatch.Create(AOwner:TComponent);
begin
 inherited Create(AOwner);
 // get unique Message IDs for the Broadcast of the dll
 SHELL_EVENT:=RegisterWindowMessage(PChar('BOME_SHELL_EVENT'));
 KEY_EVENT:=RegisterWindowMessage(PChar('BOME_KEY_EVENT'));
 LastActivated:='';
 DontLoadLib:=false;
 ResetLibVars;
end;

destructor TWatch.Destroy;
begin
 ReleaseLib;
 inherited Destroy;
end;

procedure TWatch.ReleaseLib;
begin
 if assigned(StopAll) then
  StopAll;
 UnLoadLib;
 ResetLibVars;
end;

procedure TWatch.ResetLibVars;
begin
 LibHandle:=0;
 StartWatching:=nil;
 StopWatching:=nil;
 StopAll:=nil;
 StillWatching:=nil;
 HWNDHandle:=0;
end;

procedure TWatch.LoadLib;
begin
 if (not dontLoadLib)
    and (LibHandle=0)
    and  (not (csDesigning in ComponentState)) then
 begin
  LibHandle:=LoadLibrary(watchdll);
  if LibHandle<>0 then
  begin
   // Create the window for notifications of the dll
   HWNDHandle:=AllocateHwnd(OnHWNDEvent);
   @StartWatching:=GetProcAddress(LibHandle, 'StartWatching');
   @StopWatching:=GetProcAddress(LibHandle, 'StopWatching');
   @StillWatching:=GetProcAddress(LibHandle, 'StillWatching');
   @StopAll:=GetProcAddress(LibHandle, 'StopAll');
   @SetReceiver:=GetProcAddress(LibHandle, 'SetReceiver');
   if assigned(SetReceiver) then
    SetReceiver(HWNDHandle);
  end;
  if (LibHandle=0)
   or (not assigned(StartWatching))
   or (not assigned(StopWatching))
   or (not assigned(StillWatching))
   or (not assigned(StopAll)) then
  begin
   if LibHandle=0 then
    MessageDlg('Library '+watchdll+' not found.'+#13#10
     +'System watching is disabled.',mtError,[mbOK],0)
   else
    MessageDlg('Library '+watchdll+' is not valid.'+#13#10
     +'System watching is disabled.',mtError,[mbOK],0);
   DontLoadLib:=true;
   ResetLibVars;
  end;
 end;
end;

procedure TWatch.UnloadLib;
begin
 if HWNDHandle<>0 then
  DeallocateHwnd(HWNDHandle);
 if LibHandle<>0 then
  FreeLibrary(LibHandle);
 ResetLibVars;
end;

function TWatch.GetAppName(window:THandle):String;
//extra variable for clean conversion of PChar to String
var r:String;
begin
 setString(r,nil,300);
 if GetWindowText(window,PChar(r),300)=0 then
  result:=''
 else
  result:=PChar(r);
end;

procedure TWatch.OnShellHook(nCode: Integer; wParam: Longint);
var appName:String;
begin
 If NotifyOwnEvents or (Application.Handle<>wParam) then
 begin
  appName:=GetAppName(wParam);
  case nCode of
  HSHELL_ACTIVATESHELLWINDOW:
     begin
      if assigned(FOnActivateShellWindow) then
       FOnActivateShellWindow(Self,wParam,appName);
     end;
  HSHELL_WINDOWACTIVATED:
     begin
      if assigned(FOnWindowActivated) then
      begin
       if FRepeatedEvents or (appName<>LastActivated) then
        FOnWindowActivated(Self,wParam,appName);
       LastActivated:=appName;
      end;
     end;
  HSHELL_WINDOWCREATED:
     begin
      if assigned(FOnWindowCreated) then
       FOnWindowCreated(Self,wParam,appName);
     end;
  HSHELL_WINDOWDESTROYED:
     begin
      if assigned(FOnWindowDestroyed) then
       FOnWindowDestroyed(Self,wParam,appName);
     end;
  end;
 end;
end;

procedure TWatch.OnKeyHook(wParam,lParam: Longint);
begin
 if assigned(FOnKeyNative) then
  FOnKeyNative(Self,wParam,lParam);
end;

{This is the function to which the events of the dll are posted}
procedure TWatch.OnHWNDEvent(var Message: TMessage);
begin
 if Message.Msg=SHELL_EVENT then
   OnShellHook(Message.lParam,Message.wParam)
 else
  if Message.Msg=KEY_EVENT then
   OnKeyHook(Message.wParam,Message.lParam)
 else
  with Message do
   Result := DefWindowProc(HWNDHandle, Msg, wParam, lParam);
end;

// generic procedure that handles starting and stopping of the hooks
procedure TWatch.StartStop(start:boolean; WatchType:TWatchType);
begin
 if not (csDesigning in ComponentState) then
 begin
  if start then
  begin
   LoadLib;
   if assigned(StartWatching) then
    StartWatching(WatchType);
  end
  else
  begin
   if (self.libHandle<>0) and assigned(StopWatching) then
   begin
    StopWatching(WatchType);
    if not StillWatching then
     UnLoadLib;
   end;
  end;
 end;
end;

{--------------- Shell Properties ---------------}
procedure TWatch.SetOnActivateShellWindow(h:TShellEvent);
begin
 FOnActivateShellWindow:=h;
 StartStop(assigned(h),wtSHELL);
end;

procedure TWatch.SetOnWindowActivated(h:TShellEvent);
begin
 FOnWindowActivated:=h;
 StartStop(assigned(h),wtSHELL);
end;

procedure TWatch.SetOnWindowCreated(h:TShellEvent);
begin
 FOnWindowCreated:=h;
 StartStop(assigned(h),wtSHELL);
end;

procedure TWatch.SetOnWindowDestroyed(h:TShellEvent);
begin
 FOnWindowDestroyed:=h;
 StartStop(assigned(h),wtSHELL);
end;

{--------------- Keyboard Properties ---------------}
procedure TWatch.SetOnKeyNative(h:TKeyNativeEvent);
begin
 FOnKeyNative:=h;
 StartStop(assigned(h),wtKEYBOARD);
end;

function TWatch.getVKString(keyCode:Integer):String;
var i:Integer;
begin
 result:='undefined 0x'+IntToHex(keyCode,4);
 i:=MapVirtualKey(keyCode,2);
 if (i<>0) and (i<$FF) then
  result:=chr(byte(i))
 else
 if ((keyCode>=$30) and (keyCode<=$39))
  or ((keyCode>=$41) and (keyCode<=$5A)) then
  result:=chr(byte(keyCode))
 else
 if ((keyCode>=$70) and (keyCode<=$87)) then
  result:='F'+IntToStr(keyCode-$6F)
 else
 if ((keyCode>=$60) and (keyCode<=$69)) then
  result:='Numeric keypad '+IntToStr(keyCode-$60)
 else
 case keyCode of
  VK_BACK: result:='BACKSPACE';
  VK_TAB:  result:='TAB';
  VK_CLEAR: result:='CLEAR';
  VK_RETURN: result:='ENTER';
  VK_SHIFT: result:='SHIFT';
  VK_CONTROL: result:='CTRL';
  VK_MENU: result:='ALT';
  VK_PAUSE: result:='PAUSE';
  VK_CAPITAL: result:='CAPS LOCK';
  VK_ESCAPE: result:='ESC';
  VK_SPACE: result:='SPACEBAR';
  VK_PRIOR: result:='PAGE UP';
  VK_NEXT: result:='PAGE DOWN';
  VK_END: result:='END';
  VK_HOME: result:='HOME';
  VK_LEFT: result:='LEFT ARROW';
  VK_UP: result:='UP ARROW';
  VK_RIGHT: result:='RIGHT ARROW';
  VK_DOWN: result:='DOWN ARROW';
  VK_SELECT: result:='SELECT';
  VK_EXECUTE: result:='EXECUTE';
  VK_SNAPSHOT: result:='PRINT SCREEN';
  VK_INSERT: result:='INS';
  VK_DELETE: result:='DEL';
  VK_HELP: result:='HELP';
  VK_MULTIPLY: result:='MULTIPLY';
  VK_ADD: result:='ADD';
  VK_SEPARATOR: result:='SEPARATOR';
  VK_SUBTRACT: result:='SUBTRACT';
  VK_DECIMAL: result:='DECIMAL';
  VK_DIVIDE: result:='DIVIDE';
  VK_NUMLOCK: result:='NUM LOCK';
  VK_SCROLL: result:='SCROLL LOCK';
 end;
end;

procedure TWatch.getKeyInfo(lParam:Integer; var KeyInfo:TKeyInfo);
var s:String;
begin
 s:='';
 setString(s,nil,80);
 GetKeyNameText(lParam,PChar(S),79);
 setlength(s,strlen(PChar(s)));
 with KeyInfo do
 begin
  RepeatCount:=lParam and $FFFF;
  ScanCode:=(lParam and $FF0000) shr 16;
  Extended:=((lParam shr 24) and 1)=1;
  AltDown:=((lParam shr 29) and 1)=1;
  PreviousDown:=((lParam shr 30) and 1)=1;
  Down:=((lParam shr 31) and 1)=0;
  KeyNameText:=s;
 end;
end;

procedure TWatch.SetDLLReceiver(r:Integer);
begin
 if assigned(SetReceiver) then
  SetReceiver(r);
end;

procedure Register;
begin
 RegisterComponents('Samples',[TWatch]);
end;

end.
