unit Version;

interface
{D-,L-}
uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, DsgnIntf, StdCtrls;

type
	EUnrecognizedCharacterSet = Exception;
   EUnrecognizedLanguage = Exception;

	TFilename = string;
	TReadOnlyString = string;

	TLanguage = (lnArabic, lnBulgarian, lnCatalan, lnTraditionalChinese, lnCzech,
		lnDanish, lnGerman, lnGreek, lnUSEnglish, lnCastilianSpanish, lnFinnish,
		lnFrench, lnHebrew, lnHungarian, lnIcelandic, lnItalian, lnJapanese,
		lnKorean, lnDutch, lnNorwegianBokmal, lnPolish, lnBrazilianPortuguese,
		lnRhaetoRomanic, lnRomanian, lnRussian, lnCroatoSerbian, lnSlovak,
		lnAlbanian, lnSwedish, lnThai, lnTurkish, lnUrdu, lnBahasa,
		lnSimplifiedChinese, lnSwissGerman, lnUKEnglish, lnMexicanSpanish,
		lnBelgianFrench, lnSwissItalian, lnBelgianDutch, lnNorwegianNynorsk,
		lnPortuguese, lnSerboCroatian, lnCanadianFrench, lnSwissFrench);
	TCharacterSet = (csASCII, csJapan, csKorea, csTaiwan, csUnicode, csLatin2,
		csCyrillic,	csMultilingual, csGreek, csTurkish, csHebrew, csArabic);

	TReadOnlyStringProperty = class(TStringProperty)
	public
   	{ Public Declarations }
		function GetAttributes : TPropertyAttributes; Override;
	end;

	TVersionInfo = class(TPersistent)
	private
   	{ Private Declarations }
		FComment : TReadOnlyString;
		FCompanyName : TReadOnlyString;
		FFileDescription : TReadOnlyString;
		FFileVersion : TReadOnlyString;
		FInternalName : TReadOnlyString;
		FLegalCopyright : TReadOnlyString;
		FLegalTradeMarks : TReadOnlyString;
		FOriginalFileName : TReadOnlyString;
		FPrivateBuild : TReadOnlyString;
		FProductName : TReadOnlyString;
		FProductVersion : TReadOnlyString;
		FSpecialBuild : TReadOnlyString;
		FFileOS : TReadOnlyString;
		FFileType : TReadOnlyString;
		FFileSubType : TReadOnlyString;
	public
   	{ Public Declarations }
	published
   	{ Published Declarations }
		Property Comment : string read FComment write FComment stored FALSE;
		Property CompanyName : TReadOnlyString read FCompanyName
			write FCompanyName stored FALSE;
		Property FileDescription : TReadOnlyString read FFileDescription
			write FFileDescription stored FALSE;
		Property FileVersion : TReadOnlyString read FFileVersion
			write FFileVersion stored FALSE;
		Property InternalName : TReadOnlyString read FInternalName
			write FInternalName stored FALSE;
		Property LegalCopyright : TReadOnlyString read FLegalCopyright
			write FLegalCopyright stored FALSE;
		Property LegalTradeMarks : TReadOnlyString read FLegalTradeMarks
			write FLegalTradeMarks stored FALSE;
		Property OriginalFileName : TReadOnlyString read FOriginalFileName
			write FOriginalFileName stored FALSE;
		Property PrivateBuild : TReadOnlyString read FPrivateBuild
			write FPrivateBuild stored FALSE;
		Property ProductName : TReadOnlyString read FProductName
			write FProductName stored FALSE;
		Property ProductVersion : TReadOnlyString read FProductVersion
			write FProductVersion stored FALSE;
		Property SpecialBuild : TReadOnlyString read FSpecialBuild
			write FSpecialBuild stored FALSE;
		Property FileOS : TReadOnlyString read FFileOS write FFileOS
			stored FALSE;
		Property FileType : TReadOnlyString read FFileType write FFileType
			stored FALSE;
		Property FileSubType : TReadOnlyString read FFileSubType
			write FFileSubType stored FALSE;
	end;

	TFilenameProperty = class(TStringProperty)
	public
   	{ Public Declarations }
		procedure Edit; override;
		function GetAttributes: TPropertyAttributes; override;
	end;

	TVersionResource = class(TComponent)
	private
  		{ Private declarations }
		StoreFileName : boolean;

		FFileName : TFilename;
		FLanguage : TLanguage;
		FCharacterSet : TCharacterSet;
		FVersionInfo : TVersionInfo;

		FOnChange : TNotifyEvent;

		procedure ResetVersion;
		procedure ReadVersionInfo;
		procedure DecodeVersionInfo(pInfo : pointer);
		procedure SetFileName(Value : TFilename);
		procedure SetLanguage(Value : TLanguage);
		procedure SetCharacterSet(Value : TCharacterSet);
		function DecodeCharacterSet(CharSet : TCharacterSet) : word;
		function DecodeLanguage(Lang : TLanguage) : word;
	protected
		{ Protected declarations }
	public
		{ Public declarations }
		constructor create(AOwner : TComponent); Override;
		destructor destroy; Override;
	published
		{ Published declarations }
		Property FileName : string read FFileName write SetFilename stored StoreFileName;
		Property Language : TLanguage read FLanguage write SetLanguage default lnUSEnglish;
		Property CharacterSet : TCharacterSet read FCharacterSet write SetCharacterSet default csMultilingual;

		Property VersionInfo : TVersionInfo read FVersionInfo write FVersionInfo;

		Property OnChange: TNotifyEvent read FOnChange write FOnChange;
end;

procedure Register;

implementation

constructor TVersionResource.create(AOwner : TComponent);
begin
	inherited create(AOwner);
	VersionInfo := TVersionInfo.create;
	StoreFileName := FALSE;
	FFileName := application.exeName;
	FLanguage := lnUSEnglish;
	FCharacterSet := csMultilingual;
	ReadVersionInfo;
end;

destructor TVersionResource.destroy;
begin
	versionInfo.free;
	inherited destroy;
end;

procedure TVersionResource.SetFileName(Value : TFileName);
begin
	FFileName := Value;
	StoreFilename := FFileName <> Application.ExeName;
	ReadVersionInfo;
	if assigned(FOnChange) then
		FOnChange(Self);
end;

procedure TVersionResource.SetLanguage(Value : TLanguage);
begin
	FLanguage := Value;
	ReadVersionInfo;
	if assigned(FOnChange) then
		FOnChange(Self);
end;

procedure TVersionResource.SetCharacterSet(Value : TCharacterSet);
begin
	FCharacterSet := Value;
	ReadVersionInfo;
	if assigned(FOnChange) then
		FOnChange(Self);
end;

function TVersionResource.DecodeLanguage(lang : TLanguage) : word;
begin
	case lang of
		lnArabic : result := $401;
		lnBulgarian : result := $402;
		lnCatalan : result := $403;
		lnTraditionalChinese : result := $404;
		lnCzech : result := $405;
		lnDanish : result := $406;
		lnGerman : result := $407;
		lnGreek : result := $408;
		lnUSEnglish : result := $409;
		lnCastilianSpanish : result := $40A;
		lnFinnish : result := $40B;
		lnFrench : result := $40C;
		lnHebrew : result := $40D;
		lnHungarian : result := $40E;
		lnIcelandic : result := $40F;
		lnItalian : result := $410;
		lnJapanese : result := $411;
		lnKorean : result := $412;
		lnDutch : result := $413;
		lnNorwegianBokmal : result := $414;
		lnPolish : result := $415;
		lnBrazilianPortuguese : result := $416;
		lnRhaetoRomanic : result := $417;
		lnRomanian : result := $418;
		lnRussian : result := $419;
		lnCroatoSerbian : result := $41A;
		lnSlovak : result := $41B;
		lnAlbanian : result := $41C;
		lnSwedish : result := $41D;
		lnThai : result := $41E;
		lnTurkish : result := $41F;
		lnUrdu : result := $420;
		lnBahasa : result := $421;
		lnSimplifiedChinese : result := $804;
		lnSwissGerman : result := $807;
		lnUKEnglish : result := $809;
		lnMexicanSpanish : result := $80A;
		lnBelgianFrench : result := $80C;
		lnSwissItalian : result := $810;
		lnBelgianDutch : result := $813;
		lnNorwegianNynorsk : result := $814;
		lnPortuguese : result := $816;
		lnSerboCroatian : result := $81A;
		lnCanadianFrench : result := $C0C;
		lnSwissFrench : result := $100C;
   else
   	raise EUnrecognizedLanguage.create('');
	end;
end;

function TVersionResource.DecodeCharacterSet(charSet : TCharacterSet) : word;
begin
	case charSet of
		csASCII : result := 0;
		csJapan : result := 932;
		csKorea : result := 949;
		csTaiwan : result := 950;
		csUnicode : result := 1200;
		csLatin2 : result := 1250;
		csCyrillic : result := 1251;
		csMultilingual : result := 1252;
		csGreek : result := 1253;
		csTurkish : result := 1254;
		csHebrew : result := 1255;
		csArabic : result := 1256;
   else
   	raise EUnrecognizedCharacterSet.create('');
	end;
end;

procedure TVersionResource.ResetVersion;
begin
		versionInfo.FileOS := '';
		versionInfo.FileType := '';
		versionInfo.FileSubType := '';
		versionInfo.Comment := '';
		versionInfo.CompanyName := '';
		versionInfo.FileDescription := '';
		versionInfo.FileVersion := '';
		versionInfo.InternalName := '';
		versionInfo.LegalCopyright := '';
		versionInfo.LegalTradeMarks := '';
		versionInfo.OriginalFileName := '';
		versionInfo.PrivateBuild := '';
		versionInfo.ProductName := '';
		versionInfo.ProductVersion := '';
		versionInfo.SpecialBuild := '';
end;

procedure TVersionResource.ReadVersionInfo;
var versionHandle, versionSize : DWord;
	pVersionInfo : pointer;
begin
	ResetVersion;
	versionSize := GetFileVersionInfoSize(PChar(FFileName), versionHandle);
	if versionSize = 0 then
		exit;
	getMem(pVersionInfo, versionSize);
	try
		if GetFileVersionInfo(PChar(FFileName), versionHandle, versionSize, pVersionInfo) then
			DecodeVersionInfo(pVersionInfo);
	finally
		freeMem(pVersionInfo, versionSize);
	end;
end;

procedure TVersionResource.decodeVersionInfo(pInfo : pointer);
var pItem : pointer;
	itemLen : UInt;
	FixedFileInfo : PVSFixedFileInfo;
	VersionSection : string;
begin
	if VerQueryValue(pInfo, '\', Pointer(FixedFileInfo), itemLen) then begin

   	{File Version}
		versionInfo.FileVersion :=
			IntToStr(HIWORD(FixedFileInfo^.dwFileVersionMS)) +
			'.' + IntToStr(LOWORD(FixedFileInfo^.dwFileVersionMS)) +
			'.' +	IntToStr(HIWORD(FixedFileInfo^.dwFileVersionLS)) +
			'.' + IntToStr(LOWORD(FixedFileInfo^.dwFileVersionLS));

      {Product Version}
		versionInfo.ProductVersion :=
			IntToStr(HIWORD(FixedFileInfo^.dwProductVersionMS)) +
			'.' +	IntToStr(LOWORD(FixedFileInfo^.dwProductVersionMS)) +
			'.' + IntToStr(HIWORD(FixedFileInfo^.dwProductVersionLS)) +
			'.' + IntToStr(LOWORD(FixedFileInfo^.dwProductVersionLS));

      {Operating System}
		case FixedFileInfo^.dwFileOS of
      	VOS_DOS : VersionInfo.FileOS := 'MS-Dos';
         VOS_OS216 : VersionInfo.FileOS := '16 bit OS/2';
         VOS_OS232 : VersionInfo.FileOS := '32 bit OS/2';
         VOS_NT : VersionInfo.FileOS := 'Windows NT';
         VOS__WINDOWS16 : VersionInfo.FileOS := 'Windows 3.0 or later';
         VOS__PM16 : VersionInfo.FileOS := '16 bit OS/2 Presentation Manager';
         VOS__PM32 : VersionInfo.FileOS := '32 bit OS/2 Presentation Manager';
         VOS__WINDOWS32 : VersionInfo.FileOS := '32 bit Windows';
         VOS_DOS_WINDOWS16 : VersionInfo.FileOS := 'Windows 3.0 or later with MS-Dos';
         VOS_DOS_WINDOWS32 : VersionInfo.FileOS := '32 bit Windows with MS-Dos';
         VOS_OS216_PM16 : VersionInfo.FileOS := '16 bit OS/2 with Presentation Manager';
         VOS_OS232_PM32 : VersionInfo.FileOS := '32 bit OS/2 with Presentation Manager';
         VOS_NT_WINDOWS32 : VersionInfo.FileOS := 'Windows NT with 32 bit Windows';
      else
      	VersionInfo.FileOS := 'Unknown';
      end; {case}

      {File Type}
		case FixedFileInfo^.dwFileType of
      	VFT_APP : VersionInfo.FileType := 'Application';
         VFT_DLL : VersionInfo.FileType := 'Dynamic-Link Library';
         VFT_DRV : VersionInfo.FileType := 'Device Driver';
         VFT_FONT : VersionInfo.FileType := 'Font';
         VFT_VXD : VersionInfo.FileType := 'Virtual Device Driver';
         VFT_STATIC_LIB : VersionInfo.FileType := 'Static-Link Library';
      else
      	VersionInfo.FileType := 'Unknown';
      end; {case}

      {File SubType}
		if FixedFileInfo^.dwFileType = VFT_DRV then
      	case FixedFileInfo^.dwFileSubType of
         	VFT2_DRV_PRINTER : VersionInfo.FileSubType := 'Printer';
            VFT2_DRV_KEYBOARD : VersionInfo.FileSubType := 'Keyboard';
            VFT2_DRV_LANGUAGE : VersionInfo.FileSubType := 'Language';
				VFT2_DRV_DISPLAY : VersionInfo.FileSubType := 'Display';
            VFT2_DRV_MOUSE : VersionInfo.FileSubType := 'Mouse';
				VFT2_DRV_NETWORK : VersionInfo.FileSubType := 'Network';
				VFT2_DRV_SYSTEM :VersionInfo.FileSubType := 'System';
				VFT2_DRV_INSTALLABLE : VersionInfo.FileSubType := 'Installable';
				VFT2_DRV_SOUND : VersionInfo.FileSubType := 'Sound';
				VFT2_DRV_COMM : VersionInfo.FileSubType := 'Communications';
         else
				VersionInfo.FileSubType := 'Unknown';
         end; {case}

      {File SubType}
		if FixedFileInfo^.dwFileType = VFT_FONT then
      	case FixedFileInfo^.dwFileSubType of
				VFT2_FONT_RASTER : VersionInfo.FileSubType := 'Raster';
				VFT2_FONT_VECTOR : VersionInfo.FileSubType := 'Vector';
				VFT2_FONT_TRUETYPE : VersionInfo.FileSubType := 'TrueType';
         else
				VersionInfo.FileSubType := 'Unknown';
         end; {case}

      {Version Strings}
 		VersionSection := '\StringFileInfo\' +
      	IntToHex(decodeLanguage(Language), 4) +
      	IntToHex(decodeCharacterSet(CharacterSet), 4) + '\';
 		if VerQueryValue(pInfo, PChar(VersionSection + 'Comment'), pItem, itemLen) then
 			VersionInfo.Comment := PChar(pItem);
 		if VerQueryValue(pInfo, PChar(VersionSection + 'CompanyName'), pItem, itemLen) then
 			VersionInfo.CompanyName := PChar(pItem);
 		if VerQueryValue(pInfo, PChar(VersionSection + 'FileDescription'), pItem, itemLen) then
 			VersionInfo.FileDescription := PChar(pItem);
 		if VerQueryValue(pInfo, PChar(VersionSection + 'InternalName'), pItem, itemLen) then
 			VersionInfo.InternalName := PChar(pItem);
 		if VerQueryValue(pInfo, PChar(VersionSection + 'LegalCopyright'), pItem, itemLen) then
 			VersionInfo.LegalCopyright := PChar(pItem);
 		if VerQueryValue(pInfo, PChar(VersionSection + 'LegalTradeMarks'), pItem, itemLen) then
 			VersionInfo.LegalTradeMarks := PChar(pItem);
 		if VerQueryValue(pInfo, PChar(VersionSection + 'OriginalFileName'), pItem, itemLen) then
 			VersionInfo.OriginalFileName := PChar(pItem);
 		if VerQueryValue(pInfo, PChar(VersionSection + 'PrivateBuild'), pItem, itemLen) then
 			VersionInfo.PrivateBuild := PChar(pItem);
 		if VerQueryValue(pInfo, PChar(VersionSection + 'ProductName'), pItem, itemLen) then
 			VersionInfo.ProductName := PChar(pItem);
 		if VerQueryValue(pInfo, PChar(VersionSection + 'ProductVersion'), pItem, itemLen) then
 			VersionInfo.ProductVersion := PChar(pItem);
 		if VerQueryValue(pInfo, PChar(VersionSection + 'SpecialBuild'), pItem, itemLen) then
 			VersionInfo.SpecialBuild := PChar(pItem);
	end; {if Valid Root Block of Version Resource}
end;

procedure TFilenameProperty.Edit;
var OpenDialog : TOpenDialog;
begin
	OpenDialog := TOpenDialog.create(Application);
	try
   	OpenDialog.filename := GetValue;
		OpenDialog.filter := 'All Files (*.*)|*.*|' +
			'DLL' + chr(39) + 's (*.dll)|*.dll|' +
			'Drivers (*.drv;*.vxd)|*.drv;*.vxd|' +
			'Executables (*.exe)|*.exe|' +
			'Fonts (*.ttf;*.fot;*.fon)|*.ttf;*.fot*.fon|';
   	OpenDialog.options := [ofFileMustExist, ofPathMustExist];
   	if OpenDialog.execute then
			SetValue(OpenDialog.Filename);
	finally
		OpenDialog.free;
	end;
end;

function TFilenameProperty.GetAttributes : TPropertyAttributes;
begin
  Result := [paDialog];
end;

function TReadOnlyStringProperty.GetAttributes : TPropertyAttributes;
begin
	Result := inherited getAttributes + [paReadOnly];
end;

procedure Register;
begin
	RegisterComponents('CCI', [TVersionResource]);
	RegisterPropertyEditor(TypeInfo(TFileName), TVersionResource, 'FileName',
		TFilenameProperty);
	RegisterPropertyEditor(TypeInfo(TReadOnlyString), TVersionInfo, '',
		TReadOnlyStringProperty);
end;

end.

