{*******************************************************************}
{                                                                   }
{  Delphi Variant Grid Component  -  Version 1.10                   }
{                                                                   }
{  written by T.Bednarek                                            }
{                                                                   }
{*******************************************************************}

unit VarGrid;

interface

uses
  SysUtils, Messages, Windows, Classes, Graphics, Controls, Grids;

const
  VGMaxRows = 32767;
  VGMaxCols = 255;

type
  TValueType = (vtInteger, vtFloat, vtString, vtDate, vtTime);
  TStyle = (stNone, stBump, stEtched, stRaised, stSunken);
  TFloat = Double;
  TSortDirection = (sdAsc, sdDesc);

  TVarGridOption = (vgFixedVertLine, vgFixedHorzLine, vgVertLine, vgHorzLine, vgDrawFocusSelected,
                    vgEditing, vgEditKeys, vgTabs, vgRowSelect, vgAlwaysShowEditor, vgAutoSize,
                    vgSorting, vgRangeSelect);

  TVarGridOptions = set of TVarGridOption;

  TCellData = packed Record
    DataModified: Boolean;
    AnObject: TObject;
    VType: TValueType;
    Case TValueType of
      vtInteger: (cInteger: Integer);
      vtFloat:   (cFloat: TFloat);
      vtString:  (cString: Pointer);
      vtDate:    (cDate: Double);
      vtTime:    (cTime: Double);
  End;

  PCellDataArray = ^TCellDataArray;
  TCellDataArray = packed Array[0..255] of TCellData;

  EVargrid = class(Exception)
  end;

  TColumn = class(TCollectionItem)
  private
    FTitle: String;
    FTitleFont: TFont;
    FTitleColor: TColor;
    FTitleAlignment: TAlignment;
    FValueType: TValueType;
    FFont: TFont;
    FColor: TColor;
    FAlignment: TAlignment;
    FEditMask: String;
    FDisplayFormat: String;
    FReadOnly: Boolean;
    FWidth: Integer;
    FStyle: TStyle;
    procedure SetTitle(const value: String);
    procedure SetTitleFont(const value: TFont);
    procedure SetTitleColor(const value: TColor);
    procedure SetTitleAlignment(const value: TAlignment);
    procedure SetValueType(const value: TValueType);
    procedure SetFont(const value: TFont);
    procedure SetColor(const value: TColor);
    procedure SetAlignment(const value: TAlignment);
    procedure SetEditMask(const value: String);
    procedure SetDisplayFormat(const value: String);
    procedure SetReadOnly(const value: Boolean);
    procedure SetWidth(const value: Integer);
    procedure SetStyle(const value: TStyle);
  public
    constructor Create(Collection: TCollection); override;
    destructor Destroy; override;
    procedure Assign(Source: TPersistent); override;
  published
    property Title: String read FTitle write SetTitle;
    property TitleFont: TFont read FTitleFont write SetTitleFont;
    property TitleColor: TColor read FTitleColor write SetTitleColor;
    property TitleAlignment: TAlignment read FTitleAlignment write SetTitleAlignment;
    property ValueType: TValueType read FValueType write SetValueType;
    property Font: TFont read FFont write SetFont;
    property Color: TColor read FColor write SetColor;
    property Alignment: TAlignment read FAlignment write SetAlignment;
    property EditMask: String read FEditMask write SetEditMask;
    property DisplayFormat: String read FDisplayFormat write SetDisplayFormat;
    property ReadOnly: Boolean read FReadOnly write SetReadOnly;
    property Width: Integer read FWidth write SetWidth;
    property Style: TStyle read FStyle write SetStyle;
  end;

  TVarGrid = class;

  TColumns = class(TCollection)
  private
    FVarGrid: TVarGrid;
    function GetItem(Index: Integer): TColumn;
    procedure SetItem(Index: Integer; Value: TColumn);
  public
    constructor Create(VarGrid: TVarGrid);
    function Add: TColumn;
    property VarGrid: TVarGrid read FVarGrid;
    property Items[Index: Integer]: TColumn read GetItem write SetItem;
  end;

  TCellCols = class(TObject)
  private
    FCols: Word;
    FCellDataArray: PCellDataArray;
    procedure SetModified(index: Integer; value: Boolean);
    procedure SetAsString(index: Integer; value: String);
    procedure SetAsFloat(index: Integer; value: TFloat);
    procedure SetAsInteger(index: Integer; value: Longint);
    procedure SetAsDate(index: Integer; value: TDateTime);
    procedure SetAsTime(index: Integer; value: TDateTime);
    procedure SetObject(index: Integer; value: TObject);
    function GetModified(index: Integer): Boolean;
    function GetAsString(index: Integer): String;
    function GetAsFloat(index: Integer): TFloat;
    function GetAsInteger(index: Integer): Longint;
    function GetAsDate(index: Integer): TDateTime;
    function GetAsTime(index: Integer): TDateTime;
    function GetObject(index: Integer): TObject;
  public
    constructor Create(Columns: TColumns);
    destructor Destroy; override;
    property Modified[index: Integer]: Boolean read GetModified write SetModified;
    property AsString[index: Integer]: String read GetAsString write SetAsString;
    property AsFloat[index: Integer]: TFloat read GetAsFloat write SetAsFloat;
    property AsInteger[index: Integer]: LongInt read GetAsInteger write SetAsInteger;
    property AsDate[index: Integer]: TDateTime read GetAsDate write SetAsDate;
    property AsTime[index: Integer]: TDateTime read GetAsTime write SetAsTime;
    property AnObject[index: Integer]: TObject read GetObject write SetObject;
  end;

  TCellRows = class(TObject)
  private
    SortIndex: Longint;
    SortDirection: Integer;
    FMaxCols: Integer;
    FItems: TList;
    FColumns: TColumns;
    function GetValue(aRow: Integer): TCellCols;
    function GetRowcount: Longint;
    function ColsCompare(Item1, Item2: Pointer): Integer;
    procedure QuickSort(L, R: Integer);
  public
    constructor Create(Columns: TColumns);
    destructor Destroy; override;
    procedure Insert(aRow: Integer);
    procedure Delete(aRow: Integer);
    function Add: Integer;
    procedure Sort(Col: Longint; Direction: Integer);
    property RowCount: Longint read GetRowcount;
    property Row[index: Integer]: TCellCols read GetValue;
  end;

  TCellEvent = procedure(Sender: TObject; Row, Col: Longint; Var AllowAction: Boolean) of object;
  TRowEvent = procedure(Sender: TObject; Row: Longint; Var AllowAction: Boolean) of object;
  TDefaultDrawEvent = procedure(Sender: TObject; Row, Col: Longint; Var Column: TColumn) of object;

  TVarGrid = class(TDrawGrid)
  private
    FColumns: TColumns;
    FRows: TCellRows;
    FDefaultDrawing: Boolean;
    FEditMode: Boolean;
    FOptions: TVarGridOptions;
    FEditText: String;
    FUpdateLock: Boolean;
    FBeforeEdit: TCellEvent;
    FAfterEdit: TCellEvent;
    FOnCancelEdit: TCellEvent;
    FOnInsertRow: TRowEvent;
    FOnDeleteRow: TRowEvent;
    FOnAddRow: TRowEvent;
    FOnSelectCell: TCellEvent;
    FOnDrawCell: TDrawCellEvent;
    FOnDefaultDraw: TDefaultDrawEvent;
    FFixedCols: Longint;
    FFixedRows: Longint;
    LastSortCol: Longint;
    LastSortDirection: TSortDirection;
    procedure SetColumns(const Value: TColumns);
    procedure SetCellValue(aRow, aCol: Longint; const value: Variant);
    procedure SetOptions(value: TVarGridOptions);
    procedure SetUpdateLock(value: Boolean);
    procedure SetCellText(aRow, aCol: Longint; value: String);
    procedure SetCellObject(aRow, aCol: Longint; value: TObject);
    procedure SetCellModify(aRow, aCol: Longint; value: Boolean);
    procedure SetRow(value: Longint);
    procedure SetSelection(Value: TGridRect);
    procedure SetVersion(value: String);
    function GetCellValue(aRow, aCol: Longint): Variant;
    function GetColumns: TColumns;
    function GetRowCount: LongInt;
    function GetColCount: LongInt;
    function GetCellText(aRow, aCol: Longint): String;
    function GetCellObject(aRow, aCol: Longint): TObject;
    function GetCellModify(aRow, aCol: Longint): Boolean;
    function GetRow: Longint;
    function GetSelection: TGridRect;
    function GetVersion: String;
    procedure CMExit(var Message: TMessage); message CM_EXIT;
    procedure WMSize(var Message: TMessage); message WM_SIZE;
  protected
    procedure Loaded; override;
    procedure DrawCell(ACol, ARow: Longint; ARect: TRect; AState: TGridDrawState); override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure KeyPress(var Key: Char); override;
    procedure SetEditText(ACol, ARow: Longint; const Value: string); override;
    function CanEditAcceptKey(Key: Char): Boolean; override;
    function CanEditModify: Boolean; override;
    function CanEditShow: Boolean; override;
    function GetEditMask(ACol, ARow: Longint): string; override;
    function GetEditText(ACol, ARow: Longint): string; override;
    function SelectCell(ACol, ARow: Longint): Boolean; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Insert;
    procedure Delete;
    procedure LoadFromFile(FileName: String);
    procedure SaveToFile(FileName: String);
    procedure Clear;
    procedure Sort(Col: Longint; Direction: TSortDirection);
    procedure UpdateCols;
    procedure MouseToCell(X, Y: Integer; var ACol, ARow: Longint);
    function Add: LongInt;
    property CellValue[aRow, aCol: Longint]: Variant read GetCellValue write SetCellValue;
    property CellText[aRow, aCol: Longint]: String read GetCellText write SetCellText;
    property CellObject[aRow, aCol: Longint]: TObject read GetCellObject write SetCellObject;
    property CellModified[aRow, aCol: Longint]: Boolean read GetCellModify write SetCellModify;
    property Col;
    property Row: Longint read GetRow write SetRow;
    property Selection read GetSelection write SetSelection;
  published
    property Options: TVarGridOptions read FOptions write SetOptions
      default [vgFixedVertLine, vgFixedHorzLine, vgVertLine, vgHorzLine, vgDrawFocusSelected,
               vgEditing, vgDrawFocusSelected, vgEditKeys];
    property UpdateLock: Boolean read FUpdateLock write SetUpdateLock default false;
    property Columns: TColumns read GetColumns write SetColumns;
    property DefaultDrawing: Boolean read FDefaultDrawing write FDefaultDrawing default True;
    property RowCount: LongInt read GetRowCount;
    property ColCount: Longint read GetColCount;
    property BeforeEdit: TCellEvent read FBeforeEdit write FBeforeEdit;
    property AfterEdit: TCellEvent read FAfterEdit write FAfterEdit;
    property OnCancelEdit: TCellEvent read FOnCancelEdit write FOnCancelEdit;
    property OnInsertRow: TRowEvent read FOnInsertRow write FOnInsertRow;
    property OnDeleteRow: TRowEvent read FOnDeleteRow write FOnDeleteRow;
    property OnAddRow: TRowEvent read FOnAddRow write FOnAddRow;
    property OnSelectCell: TCellEvent read FOnSelectCell write FOnSelectCell;
    property OnDrawCell: TDrawCellEvent read FOnDrawCell write FOnDrawCell;
    property OnDefaultDraw: TDefaultDrawEvent read FOnDefaultDraw write FOnDefaultDraw;
    property FixedCols: Longint read FFixedCols write FFixedCols;
    property FixedRows: Longint read FFixedRows write FFixedRows;
    property Version: String read GetVersion write SetVersion;
  End;

implementation

Uses
  UStrings;

Const
  VGVersion = '1.10';
  RowMoveKeys = [VK_UP, VK_PRIOR, VK_DOWN, VK_NEXT];

var
  DrawBitmap: TBitmap;
  UserCount: Integer;

procedure UsesBitmap;
begin
  if UserCount = 0 then DrawBitmap := TBitmap.Create;
  Inc(UserCount);
end;

procedure ReleaseBitmap;
begin
  Dec(UserCount);
  if UserCount = 0 then DrawBitmap.Free;
end;

function Max(X, Y: Integer): Integer;
begin
  Result := Y;
  if X > Y then Result := X;
end;

procedure WriteText(ACanvas: TCanvas; ARect: TRect; DX, DY: Integer;
                    const Text: string; Alignment: TAlignment);
const
  AlignFlags : array [TAlignment] of Integer =
    ( DT_LEFT or DT_WORDBREAK or DT_EXPANDTABS or DT_NOPREFIX,
      DT_RIGHT or DT_WORDBREAK or DT_EXPANDTABS or DT_NOPREFIX,
      DT_CENTER or DT_WORDBREAK or DT_EXPANDTABS or DT_NOPREFIX );
var
  B, R: TRect;
  I, Left: Integer;
begin
  I := ColorToRGB(ACanvas.Brush.Color);

  if GetNearestColor(ACanvas.Handle, I) = I then
    begin
      case Alignment of
        taLeftJustify:
          Left := ARect.Left + DX;
        taRightJustify:
          Left := ARect.Right - ACanvas.TextWidth(Text) - 3;
        else
          Left := ARect.Left + (ARect.Right - ARect.Left) shr 1
                  - (ACanvas.TextWidth(Text) shr 1);
      end;

      ExtTextOut(ACanvas.Handle, Left, ARect.Top + DY, ETO_OPAQUE or
                 ETO_CLIPPED, @ARect, PChar(Text), Length(Text), nil);
    end
  else
    begin
      with DrawBitmap, ARect do
        begin
          Width := Max(Width, Right - Left);
          Height := Max(Height, Bottom - Top);
          R := Rect(DX, DY, Right - Left - 1, Bottom - Top - 1);
          B := Rect(0, 0, Right - Left, Bottom - Top);
        end;

      with DrawBitmap.Canvas do
        begin
          Font := ACanvas.Font;
          Font.Color := ACanvas.Font.Color;
          Brush := ACanvas.Brush;
          Brush.Style := bsSolid;
          FillRect(B);
          SetBkMode(Handle, TRANSPARENT);
          DrawText(Handle, PChar(Text), Length(Text), R, AlignFlags[Alignment]);
        end;

      ACanvas.CopyRect(ARect, DrawBitmap.Canvas, B);
    end;
end;

{ --- TCellCols ---------------------------------------------------- }

constructor TCellCols.Create(Columns: TColumns);
Var
  i: Integer;
begin
  inherited Create;
  GetMem(FCellDataArray, (SizeOf(TCellData) * Columns.Count));
  FCols := Columns.Count;

  For i := 0 to FCols - 1 do
    begin
      FCellDataArray^[i].DataModified := False;
      FCellDataArray^[i].AnObject := nil;
      FCellDataArray^[i].VType := Columns.Items[i].ValueType;

      case FCellDataArray^[i].VType of
        vtInteger:
          FCellDataArray^[i].cInteger := 0;
        vtFloat:
          FCellDataArray^[i].cFloat := 0;
        vtString:
          FCellDataArray^[i].cString := nil;
        vtDate:
          FCellDataArray^[i].cDate := 0;
        vtTime:
          FCellDataArray^[i].cTime := 0;
      end;
    end;
end;

destructor TCellCols.Destroy;
Var
  i: Integer;
begin
  For i := 0 to FCols - 1 do
    begin
      If FCellDataArray^[i].VType = vtString then ReallocMem(FCellDataArray^[i].cString, 0);
      If Assigned(FCellDataArray^[i].AnObject) then FCellDataArray^[i].AnObject.Destroy;
    end;

  FreeMem(FCellDataArray);
  inherited Destroy;
end;

procedure TCellCols.SetModified(index: Integer; value: Boolean);
begin
  FCellDataArray^[index].DataModified := value;
end;

procedure TCellCols.SetAsString(index: Integer; value: String);
begin
  If Length(value) = 0 then
    ReallocMem(FCellDataArray^[index].cString, 0)
  else
    begin
      ReallocMem(FCellDataArray^[index].cString, Length(value)+1);
      StrPCopy(FCellDataArray^[index].cString, value);
    end;
end;

procedure TCellCols.SetAsFloat(index: Integer; value: TFloat);
begin
  FCellDataArray^[index].cFloat := value;
end;

procedure TCellCols.SetAsInteger(index: Integer; value: Longint);
begin
  FCellDataArray^[index].cInteger := value;
end;

procedure TCellCols.SetAsDate(index: Integer; value: TDateTime);
begin
  FCellDataArray^[index].cDate := value;
end;

procedure TCellCols.SetAsTime(index: Integer; value: TDateTime);
begin
  FCellDataArray^[index].cTime := value;
end;

procedure TCellCols.SetObject(index: Integer; value: TObject);
begin
  FCellDataArray^[index].AnObject := value;
end;

function TCellCols.GetModified(index: Integer): Boolean;
begin
  Result := FCellDataArray^[index].DataModified;
end;

function TCellCols.GetAsString(index: Integer): String;
begin
  Result := StrPas(FCellDataArray^[index].cString);
end;

function TCellCols.GetAsFloat(index: Integer): TFloat;
begin
  Result := FCellDataArray^[index].cFloat;
end;

function TCellCols.GetAsInteger(index: Integer): Longint;
begin
  Result := FCellDataArray^[index].cInteger;
end;

function TCellCols.GetAsDate(index: Integer): TDateTime;
begin
  Result := FCellDataArray^[index].cDate;
end;

function TCellCols.GetAsTime(index: Integer): TDateTime;
begin
  Result := FCellDataArray^[index].cTime;
end;

function TCellCols.GetObject(index: Integer): TObject;
begin
  Result := FCellDataArray^[index].AnObject;
end;

{ --- TCellRows ---------------------------------------------------- }

constructor TCellRows.Create(Columns: TColumns);
begin
  FItems := TList.Create;
  FColumns := Columns;
  FMaxCols := FColumns.Count;
end;

destructor TCellRows.Destroy;
Var
  i: Integer;
begin
  For i := 0 to FItems.Count - 1 do TCellCols(FItems.Items[i]).Destroy;
  FItems.Destroy;
  inherited Destroy;
end;

function TCellRows.GetValue(aRow: Integer): TCellCols;
begin
  Result := TCellCols(FItems.Items[aRow]);
end;

function TCellRows.GetRowcount: Longint;
begin
  Result := FItems.Count;
end;

procedure TCellRows.Insert(aRow: Integer);
Var
  CellCols: TCellCols;
begin
  If FItems.Count = VGMaxRows then raise EVarGrid.Create('max. no of rows: ' + IntToStr(VGMaxRows));
  CellCols := TCellCols.Create(FColumns);
  FItems.Insert(aRow, CellCols);
end;

procedure TCellRows.Delete(aRow: Integer);
begin
  TCellCols(FItems.Items[aRow]).Destroy;
  FItems.Delete(aRow);
  FItems.Pack;
end;

function TCellRows.Add: Integer;
Var
  CellCols: TCellCols;
begin
  If FItems.Count = VGMaxRows then raise EVarGrid.Create('max. no of rows: ' + IntToStr(VGMaxRows));
  CellCols := TCellCols.Create(FColumns);
  Result := FItems.Add(CellCols);
end;

function TCellRows.ColsCompare(Item1, Item2: Pointer): Integer;
begin
  Case FColumns.Items[SortIndex].ValueType of
    vtInteger:
      If TCellCols(Item1).AsInteger[SortIndex] < TCellCols(Item2).AsInteger[SortIndex] then
        ColsCompare := -1 * SortDirection
      else
        If TCellCols(Item1).AsInteger[SortIndex] > TCellCols(Item2).AsInteger[SortIndex] then
          ColsCompare := 1 * SortDirection
        else
          ColsCompare := 0;
    vtFloat:
      If TCellCols(Item1).AsFloat[SortIndex] < TCellCols(Item2).AsFloat[SortIndex] then
        ColsCompare := -1 * SortDirection
      else
        If TCellCols(Item1).AsFloat[SortIndex] > TCellCols(Item2).AsFloat[SortIndex] then
          ColsCompare := 1 * SortDirection
        else
          ColsCompare := 0;
    vtString:
      ColsCompare := CompareText(TCellCols(Item1).AsString[SortIndex], TCellCols(Item2).AsString[SortIndex]) *SortDirection;
    vtDate:
      If TCellCols(Item1).AsDate[SortIndex] < TCellCols(Item2).AsDate[SortIndex] then
        ColsCompare := -1 * SortDirection
      else
        If TCellCols(Item1).AsDate[SortIndex] > TCellCols(Item2).AsDate[SortIndex] then
          ColsCompare := 1 * SortDirection
        else
          ColsCompare := 0;
    vtTime:
      If TCellCols(Item1).AsTime[SortIndex] < TCellCols(Item2).AsTime[SortIndex] then
        ColsCompare := -1 * SortDirection
      else
        If TCellCols(Item1).AsTime[SortIndex] > TCellCols(Item2).AsTime[SortIndex] then
          ColsCompare := 1 * SortDirection
        else
          ColsCompare := 0;
    else
      ColsCompare := 0;
  end;
end;

procedure TCellRows.QuickSort(L, R: Integer);
var
  I, J: Integer;
  P, T: Pointer;
begin
  repeat
    I := L;
    J := R;
    P := FItems[(L + R) shr 1];
    repeat
      while ColsCompare(FItems[I], P) < 0 do Inc(I);
      while ColsCompare(FItems[J], P) > 0 do Dec(J);
      if I <= J then
      begin
        T := FItems[I];
        FItems[I] := FItems[J];
        FItems[J] := T;
        Inc(I);
        Dec(J);
      end;
    until I > J;
    if L < J then QuickSort(L, J);
    L := I;
  until I >= R;
end;

procedure TCellRows.Sort(Col: Longint; Direction: Integer);
begin
  SortIndex := Col;
  SortDirection := Direction;
  QuickSort(0, FItems.Count - 1);
end;

{ --- TColumn ------------------------------------------------------ }

constructor TColumn.Create(Collection: TCollection);
begin
  inherited Create(Collection);
  FTitleFont := TFont.Create;
  FFont := TFont.Create;
end;

destructor TColumn.Destroy;
begin
  FFont.Destroy;
  inherited Destroy;
end;

procedure TColumn.SetTitle(const value: String);
begin
  FTitle := value;
  Changed(false);
end;

procedure TColumn.SetTitleFont(const value: TFont);
begin
  FTitleFont.Assign(value);
  Changed(False);
end;

procedure TColumn.SetTitleColor(const value: TColor);
begin
  FTitleColor := value;
  Changed(false);
end;

procedure TColumn.SetTitleAlignment(const value: TAlignment);
begin
  FTitleAlignment := value;
  Changed(False);
end;

procedure TColumn.SetValueType(const value: TValueType);
begin
  FValueType := value;
  Changed(False);
end;

procedure TColumn.SetFont(const value: TFont);
begin
  FFont.Assign(value);
  Changed(False);
end;

procedure TColumn.SetColor(const value: TColor);
begin
  FColor := value;
  Changed(False);
end;

procedure TColumn.SetAlignment(const value: TAlignment);
begin
  FAlignment := value;
  Changed(False);
end;

procedure TColumn.SetEditMask(const value: String);
begin
  FEditMask := value;
  Changed(False);
end;

procedure TColumn.SetDisplayFormat(const value: String);
begin
  FDisplayformat := value;
  Changed(False);
end;

procedure TColumn.SetReadOnly(const value: Boolean);
begin
  FReadOnly := value;
  Changed(False);
end;

procedure TColumn.SetWidth(const value: Integer);
begin
  FWidth := value;
  Changed(False);
end;

procedure TColumn.SetStyle(const value: TStyle);
begin
  FStyle := value;
  Changed(False);
end;

procedure TColumn.Assign(Source: TPersistent);
begin
  if Source is TColumn then
    begin
      Title := TColumn(Source).Title;
      TitleFont := TColumn(Source).TitleFont;
      TitleColor := TColumn(Source).TitleColor;
      TitleAlignment := TColumn(Source).TitleAlignment;
      ValueType := TColumn(Source).ValueType;
      Font := TColumn(Source).Font;
      Color := TColumn(Source).Color;
      Alignment := TColumn(Source).Alignment;
      EditMask := TColumn(Source).EditMask;
      DisplayFormat := TColumn(Source).DisplayFormat;
      ReadOnly := TColumn(Source).ReadOnly;
      Width := TColumn(Source).Width;
      Style := TColumn(Source).Style;
    end
  else
    inherited Assign(Source);
end;

{ --- TColumns ----------------------------------------------------- }

constructor TColumns.Create(VarGrid: TVarGrid);
begin
  inherited Create(TColumn);
  FVarGrid := VarGrid;
end;

function TColumns.GetItem(Index: Integer): TColumn;
begin
  Result := TColumn(inherited GetItem(Index));
end;

procedure TColumns.SetItem(Index: Integer; Value: TColumn);
begin
  inherited SetItem(Index, Value);
end;

function TColumns.Add: TColumn;
begin
  If Count = VGMaxCols then raise EVarGrid.Create('max. no of colums: ' + IntToStr(VGMaxCols));
  Result := TColumn(inherited Add);
end;

{ --- TVarGrid ----------------------------------------------------- }

constructor TVarGrid.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  inherited RowCount := 0;
  inherited ColCount := 0;
  inherited FixedRows := 0;
  inherited FixedCols := 0;
  FixedRows := 1;
  DefaultRowHeight := 18;
  DefaultColWidth := 70;
  FColumns := TColumns.Create(self);
  FRows := Nil;
  FOnCancelEdit := Nil;
  FOnInsertRow := Nil;
  FOnDeleteRow := Nil;
  FOnAddRow := Nil;
  FOnSelectCell := Nil;
  FOnDrawCell := Nil;
  FEditMode := False;
  FUpdateLock := False;
  FDefaultDrawing := True;
  FSaveCellExtents := False;
  UsesBitmap;
  HideEditor;
  LastSortCol := 0;
  LastSortDirection := sdDesc;
  Options := [vgFixedVertLine, vgFixedHorzLine, vgVertLine, vgHorzLine, vgDrawFocusSelected,
              vgEditing, vgDrawFocusSelected, vgEditKeys];
end;

destructor TVarGrid.Destroy;
begin
  FColumns.Destroy;
  If Assigned(FRows) then FRows.Destroy;
  ReleaseBitmap;
  inherited Destroy;
end;

procedure TVarGrid.Loaded;
Var
  i: Integer;
begin
  inherited Loaded;

  If FColumns.Count > 0 then
    begin
      inherited RowCount := 2;
      inherited ColCount := FColumns.Count;

      If not (csDesigning in ComponentState) then
        begin
          FRows := TCellRows.Create(FColumns);
          inherited FixedCols := FixedCols;
          inherited Row := 1;
          inherited FixedRows := 1;
          inherited Row := 1;
          TopRow := 1;
        end;
    end;

  For i := 0 to FColumns.Count - 1 do ColWidths[i] := FColumns.Items[i].Width;
end;

function TVarGrid.GetEditText(ACol, ARow: Longint): string;
begin
  If FRows.Rowcount = 0 then FRows.Add;
  Result := CellText[aRow-1, aCol];
end;

procedure TVarGrid.SetEditText(ACol, ARow: Longint; const Value: string);
begin
  FEditText := Value;
end;

function TVarGrid.CanEditAcceptKey(Key: Char): Boolean;
begin
  Case FColumns.Items[Col].ValueType of
    vtInteger:
      if not (Key in ['-','0'..'9']) then
        result := false
      else
        result := true;
    vtFloat:
      if not (Key in [DecimalSeparator,'-','0'..'9']) then
        result := false
      else
        result := true;
    vtDate:
      if not (Key in [DateSeparator, '0'..'9']) then
        result := false
      else
        result := true;
    vtTime:
      if not (Key in [TimeSeparator, '0'..'9']) then
        result := false
      else
        result := true;
    else
      result := true;
    End;
end;

function TVarGrid.CanEditModify: Boolean;
begin
  If (vgEditing in Options) and not (FColumns.Items[Col].ReadOnly) then
    Result := true
  else
    Result := false;

  FEditMode := Result;
  If FEditMode and Assigned(FBeforeEdit) then FBeforeEdit(self, Row, Col, FEditMode);
end;

function TVarGrid.CanEditShow: Boolean;
begin
  If inherited CanEditShow and
     (vgEditing in Options) and
     not (FColumns.Items[Col].ReadOnly) then
    Result := true
  else
    Result := false;
end;

function TVarGrid.GetEditMask(ACol, ARow: Longint): string;
begin
  Result := FColumns.Items[Col].EditMask;
end;

procedure TVarGrid.CMExit(var Message: TMessage);
Var
  Dummy: Boolean;
begin
  If FEditMode then
    begin
      FEditMode := False;
      CellModified[Row, Col] := true;
      CellText[Row, Col] := FEditText;
      If Assigned(FAfterEdit) then FAfterEdit(self, Row, Col, Dummy);
    end;

  inherited;
end;

procedure TVarGrid.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
Var
  Dummy: Boolean;
begin
  If FEditMode then
    begin
      FEditMode := False;
      CellModified[Row, Col] := true;
      CellText[Row, Col] := FEditText;
      If Assigned(FAfterEdit) then FAfterEdit(self, Row, Col, Dummy);
    end;

  If (vgSorting in Options) and
     (MouseCoord(X, Y).Y = 0) and
     (FRows.RowCount > 0) then
    begin
      If LastSortCol = MouseCoord(X, Y).X then
        If LastSortDirection = sdAsc then
          LastSortDirection := sdDesc
        else
          LastSortDirection := sdAsc
      else
        begin
          LastSortCol := MouseCoord(X, y).X;
          LastSortDirection := sdAsc;
        end;

      Sort(LastSortCol, LastSortDirection);
      inherited MouseDown(Button, Shift, X, Y);
    end;

  If MouseCoord(X, Y).Y > 0 then inherited MouseDown(Button, Shift, X, Y);
end;

procedure TVarGrid.KeyPress(var Key: Char);
Var
  Dummy: Boolean;
begin
  If FEditMode and (Key = #13) then
    begin
      FEditMode := False;
      CellModified[Row, Col] := true;
      CellText[Row, Col] := FEditText;
      If Assigned(FAfterEdit) then FAfterEdit(self, Row, Col, Dummy);
    end;

  inherited KeyPress(Key);
end;

procedure TVarGrid.KeyDown(var Key: Word; Shift: TShiftState);
Var
  Dummy: Boolean;
begin
  If FEditMode and ((Key in RowMoveKeys) or (Key = VK_ESCAPE)) then
    begin
      FEditMode := False;

      If (Key = VK_ESCAPE) then
        begin
          If Assigned(FOnCancelEdit) then FOnCancelEdit(self, Row, Col, Dummy);
          If not (vgAlwaysShowEditor in Options) then HideEditor;
        end
      else
        begin
          CellModified[Row, Col] := true;
          CellText[Row, Col] := FEditText;
          If Assigned(FAfterEdit) then FAfterEdit(self, Row, Col, Dummy);
        end;
    end
  else
    If (vgEditing in Options) and
       (vgEditKeys in Options) then
    Case Key of
      VK_INSERT:
        Insert;
      VK_DELETE:
        Delete;
      VK_DOWN:
        If (ssCtrl in Shift) then
          begin
            Add;
            KeyDown(Key, []);
          end;
    End;

  inherited KeyDown(Key, Shift);
end;

procedure TVarGrid.WMSize(var Message: TMessage);
Var
  i: Integer;
  szCols: Integer;
  szLastCol: Integer;
begin
  If (FColumns.Count > 0) and (vgAutoSize in Options) then
    begin
      szCols := 0;
      For i := 0 to ColCount - 2 do szCols := szCols + ColWidths[i] + GridLineWidth;
      szLastCol := ClientWidth - szCols;

      If szLastCol > FColumns.Items[ColCount-1].Width then
        ColWidths[ColCount-1] := szLastCol
      else
        ColWidths[ColCount-1] := FColumns.Items[ColCount-1].Width;
    end;

  inherited;
end;

procedure TVarGrid.DrawCell(ACol, ARow: Longint; ARect: TRect; AState: TGridDrawState);
Var
  DrawColumn: TColumn;
  DrawStyle: Word;
begin
  if DefaultDrawing and (FColumns.Count > 0) then
    with Canvas do
    begin
      DrawColumn := FColumns.Items[ACol];

      If ARow = 0 then
        begin
          Brush.Color := DrawColumn.TitleColor;
          Font := DrawColumn.TitleFont;
          FillRect(ARect);
          WriteText(Canvas, ARect, 1, 3, DrawColumn.Title, DrawColumn.TitleAlignment);
          DrawEdge(Canvas.Handle, ARect, BDR_RAISEDINNER, BF_BOTTOMRIGHT);
          DrawEdge(Canvas.Handle, ARect, BDR_RAISEDINNER, BF_TOPLEFT);
        end;

      If ARow > 0 then
        begin
          If Assigned(FOnDefaultDraw) then FOnDefaultDraw(Self, ARow-1, ACol, DrawColumn);
          Brush.Color := DrawColumn.Color;
          Font := DrawColumn.Font;

          If (gdSelected in AState) and
             (vgDrawFocusSelected in Options) and
              not(vgRowSelect in Options) then
            begin
              Windows.DrawFocusRect(Handle, ARect);
              Brush.Color := clHighlight;
              Font.Color := clHighlightText;
            end;

          If (gdSelected in AState) and
             (vgRowSelect in Options) then
            begin
              Brush.Color := clHighlight;
              Font.Color := clHighlightText;
            end;

          If (gdFixed in AState) then
            begin
              Brush.Color := DrawColumn.TitleColor;
            end;

          FillRect(ARect);

          If Assigned(FRows) and
            (ARow <= FRows.Rowcount) then
            WriteText(Canvas, ARect, 1, 3, CellText[ARow-1, ACol], DrawColumn.Alignment);

          If (DrawColumn.Style <> stNone) then
            begin
              Case DrawColumn.Style of
                stBump:
                  DrawStyle := EDGE_BUMP;
                stEtched:
                  DrawStyle := EDGE_ETCHED;
                stRaised:
                  DrawStyle := EDGE_RAISED;
                stSunken:
                  DrawStyle := EDGE_SUNKEN;
                else
                  DrawStyle := EDGE_BUMP;
              End;

              DrawEdge(Canvas.Handle, ARect, DrawStyle, BF_BOTTOMRIGHT);
              DrawEdge(Canvas.Handle, ARect, DrawStyle, BF_TOPLEFT);
            end;
        end;
    end;

  if Assigned(FOnDrawCell) then FOnDrawCell(Self, ACol, ARow, ARect, AState);
end;

procedure TVarGrid.Delete;
Var
  value: Boolean;
begin;
  If FRows.RowCount = 0 then Exit;
  value := true;
  If Assigned(FOnDeleteRow) then FOnDeleteRow(self, Row, value);

  If value then
    begin
      FRows.Delete(Row);
      If inherited RowCount > 2 then inherited RowCount := FRows.RowCount + 1;
      If FRows.RowCount < FixedRows then inherited FixedRows := 1;
      If not UpdateLock then Refresh;
    end;
end;

procedure TVarGrid.Insert;
Var
  value: Boolean;
begin
  value := true;
  If Assigned(FOnInsertRow) then FOnInsertRow(self, Row, value);

  If value then
   begin
     FRows.Insert(Row);
     inherited RowCount := FRows.RowCount + 1;
     If FRows.RowCount = 1 then TopRow := RowCount;
     If FRows.RowCount > FixedRows then inherited FixedRows := FixedRows;
     If not UpdateLock then Refresh;
   end;
end;

function TVarGrid.Add: LongInt;
Var
  value: Boolean;
begin
  value := true;
  If Assigned(FOnAddRow) then FOnAddRow(self, Row, value);

  If value then
    begin
      Result := FRows.Add;
      inherited RowCount := FRows.RowCount + 1;
      If FRows.RowCount = 1 then TopRow := RowCount;
      If FRows.RowCount > FixedRows then inherited FixedRows := FixedRows;
      If not UpdateLock then inherited Row := RowCount;
    end
  else
    Result := 0;
end;

procedure TVarGrid.SetColumns(const Value: TColumns);
begin;
  FColumns.Assign(Value);
end;

procedure TVarGrid.SetCellText(aRow, aCol: Longint; value: String);
begin
  With FColumns.Items[aCol] do
  Case ValueType of
    vtInteger:
      CellValue[aRow, aCol] := StrToInt(StripStr(ThousandSeparator, value));
    vtFloat:
      CellValue[aRow, aCol] := StrToFloat(StripStr(ThousandSeparator, value));
    vtString:
      CellValue[aRow, aCol] := value;
    vtDate:
      CellValue[aRow, aCol] := StrToDate(value);
    vtTime:
      CellValue[aRow, aCol] := StrToTime(value);
  End;
end;

procedure TVarGrid.SetCellObject(aRow, aCol: Longint; value: TObject);
begin
  FRows.Row[aRow].AnObject[aCol] := value;
end;

function TVarGrid.GetCellObject(aRow, aCol: Longint): TObject;
begin
  Result := FRows.Row[aRow].AnObject[aCol];
end;

function TVarGrid.GetCellText(aRow, aCol: Longint): String;
begin
  With FColumns.Items[aCol] do
  Case ValueType of
    vtInteger:
      Result := FormatFloat(Displayformat, CellValue[aRow, aCol]);
    vtFloat:
      Result := FormatFloat(Displayformat, CellValue[aRow, aCol]);
    vtString:
      Result := CellValue[aRow, aCol];
    vtDate:
      If CellValue[aRow, aCol] = 0 then
        Result := ''
      else
        Result := FormatDateTime('dd.mm.yyyy', CellValue[aRow, aCol]);
    vtTime:
      If CellValue[aRow, aCol] = 0 then
        Result := ''
      else
        Result := FormatDateTime('hh:nn', CellValue[aRow, aCol]);
  end;
end;

procedure TVarGrid.SetCellValue(aRow, aCol: Longint; const value: Variant);
begin
  With FColumns.Items[aCol] do
  Case ValueType of
    vtInteger:
      FRows.Row[aRow].AsInteger[aCol] := VarAsType(value, varInteger);
    vtFloat:
      FRows.Row[aRow].AsFloat[aCol] := VarAsType(value, varDouble);
    vtString:
      FRows.Row[aRow].AsString[aCol] := VarAsType(value, varString);
    vtDate:
      FRows.Row[aRow].AsDate[aCol] := VarAsType(value, varDate);
    vtTime:
      FRows.Row[aRow].AsTime[aCol] := VarAsType(value, varDate);
  end;

  InvalidateCell(aCol, aRow+1);
end;

function TVarGrid.GetCellValue(aRow, aCol: Longint): Variant;
begin
  With FColumns.Items[aCol] do
  Case ValueType of
    vtInteger:
      Result := FRows.Row[aRow].AsInteger[aCol];
    vtFloat:
      Result := FRows.Row[aRow].AsFloat[aCol];
    vtString:
      Result := FRows.Row[aRow].AsString[aCol];
    vtDate:
      Result := FRows.Row[aRow].AsDate[aCol];
    vtTime:
      Result := FRows.Row[aRow].AsTime[aCol];
  end;
end;

function TVarGrid.GetColumns: TColumns;
begin;
  Result := FColumns;
end;

function TVarGrid.GetRowCount: LongInt;
begin
  Result := FRows.RowCount;
end;

function TVarGrid.GetColCount: LongInt;
begin
  Result := inherited ColCount;
end;

procedure TVarGrid.SetOptions(value: TVarGridOptions);
Var
  InheritOptions: TGridOptions;
begin
  If FOptions <> Value then
    begin
      FOptions := value;
      InheritOptions := [];
      If (vgFixedVertLine in Options) then Include(InheritOptions, goFixedVertLine);
      If (vgFixedHorzLine in Options) then Include(InheritOptions, goFixedHorzLine);
      If (vgVertLine in Options) then Include(InheritOptions, goVertLine);
      If (vgHorzLine in Options) then Include(InheritOptions,goHorzLine);
      If (vgDrawFocusSelected in Options) then Include(InheritOptions, goDrawFocusSelected);
      If (vgEditing in Options) then Include(InheritOptions, goEditing);
      If (vgTabs in Options) then Include(InheritOptions, goTabs);
      If (vgRowSelect in Options) then Include(InheritOptions, goRowSelect);
      If (vgAlwaysShowEditor in Options) then Include(InheritOptions, goAlwaysShowEditor);
      If (vgDrawFocusSelected in Options) then Include(InheritOptions, goDrawFocusSelected);
      If (vgRangeSelect in Options) then Include(InheritOptions, goRangeSelect);
      inherited Options := InheritOptions;
    end;
end;

procedure TVarGrid.Clear;
Var
  i: Integer;
begin
  FRows.Destroy;
  inherited RowCount := 2;
  inherited ColCount := FColumns.Count;
  inherited FixedRows := 1;
  inherited Row := 1;
  TopRow := 1;
  FRows := TCellRows.Create(FColumns);
  For i := 0 to FColumns.Count - 1 do ColWidths[i] := FColumns.Items[i].Width;
  If not UpdateLock then Refresh;
end;

procedure TVarGrid.SetUpdateLock(value: Boolean);
begin;
  If value <> FUpdateLock then
    begin
      FUpdateLock := value;
      If not value then Refresh;
    end;
end;

procedure TVarGrid.LoadFromFile(FileName: String);
Var
  Txt: TextFile;
  Data: String;
  r, c: Longint;
  p: Integer;
  Eod: Boolean;
  OldLock: Boolean;
begin
  OldLock := UpdateLock;
  UpdateLock := True;
  Clear;
  AssignFile(Txt, FileName);
  {$I+} Reset(Txt); {$I-}
  r := 0;

  Try
    While not Eof(Txt) do
      begin
        ReadLn(Txt, Data);
        Add;
        c := 0;
        Eod := False;

        While not Eod and (c < Colcount) do
          begin
            p := Pos(Char(9), Data);

            If p = 0 then
              begin
                Eod := true;
                p := Length(Data);
              end;

            Try
              CellText[r, c] := Copy(Data, 1, p - 1);
            Except
              CellText[r, c] := '';
            End;

            System.Delete(Data, 1, p);
            inc(c);
          end;
        Inc(r);
      end;
  Finally
    CloseFile(Txt);
    inherited Row := 1;
    TopRow := 1;
    UpdateLock := False;
    UpdateLock := OldLock;
  End;
end;

procedure TVarGrid.SaveToFile(FileName: String);
Var
  Txt: TextFile;
  Data: String;
  r, c: Longint;
begin
  AssignFile(Txt, FileName);
  {$I+} Rewrite(Txt); {$I-}

  Try
    For r := 0 to RowCount - 1 do
      begin
        Data := '';
        For c := 0 to ColCount - 2 do Data := Data + CellText[r, c] + Char(9);
        c := Colcount - 1;
        Data := Data + CellText[r, c] + Char(9);
        WriteLn(Txt, Data);
      end;
  Finally
    CloseFile(Txt);
  End;
end;

procedure TVarGrid.Sort(Col: Longint; Direction: TSortDirection);
begin
  UpdateLock := True;

  If Direction = sdAsc then
    FRows.Sort(Col, 1)
  else
    FRows.Sort(Col, -1);

  inherited Row := 1;
  TopRow := 1;
  UpdateLock := False;
end;

procedure TVarGrid.SetCellModify(aRow, aCol: Longint; value: Boolean);
begin
  FRows.Row[aRow].Modified[aCol] := value;
end;

function TVarGrid.GetCellModify(aRow, aCol: Longint): Boolean;
begin
  Result := FRows.Row[aRow].Modified[aCol];
end;

function TVarGrid.SelectCell(ACol, ARow: Longint): Boolean;
begin
  Result := True;
  If Assigned(FOnSelectCell) then FOnSelectCell(Self, ARow-1, ACol, Result);
end;

procedure TVarGrid.SetRow(value: Longint);
begin
  inherited Row := value + 1;
end;

function TVarGrid.GetRow: Longint;
begin
  Result := inherited Row - 1;
end;

procedure TVarGrid.SetSelection(Value: TGridRect);
begin
  inherited Selection := Value;
end;

function TVarGrid.GetSelection: TGridRect;
Var
  Sel: TGridRect;
begin
  Sel := inherited Selection;
  Dec(Sel.Top);
  Dec(Sel.Bottom);
  Result := Sel;
end;

procedure TVarGrid.SetVersion(value: String);
begin
end;

function TVarGrid.GetVersion: String;
begin
  Result := VGVersion;
end;

procedure TVarGrid.UpdateCols;
Var
  i: Integer;
begin
  If FColumns.Count > 0 then
    begin
      inherited RowCount := 2;
      inherited ColCount := FColumns.Count;
      inherited FixedRows := FixedRows;
      For i := 0 to FColumns.Count - 1 do ColWidths[i] := FColumns.Items[i].Width;
    end
  else
    begin
      inherited RowCount := 0;
      inherited ColCount := 0;
      inherited FixedRows := 0;
      inherited FixedCols := 0;
    end;

  Refresh;
end;

procedure TVarGrid.MouseToCell(X, Y: Integer; var ACol, ARow: Longint);
var
  Coord: TGridCoord;
begin
  Coord := MouseCoord(X, Y);
  ACol := Coord.X;
  ARow := Coord.Y - 1;
end;

end.
