{*******************************************************************}
{                                                                   }
{  Delphi 2.0                                                       }
{  Numeric Components                                               }
{                                                                   }
{  Copyright (c) 1996 Thomas Bednarek                               }
{                                                                   }
{*******************************************************************}

unit Numctrls;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Controls,
  Forms, Graphics, Mask, StdCtrls;

type
  TNumLabel = class(TLabel)
  private
    FDisplayFormat: String;
    FValue: Double;
    procedure SetDisplayFormat(value: string);
    function GetDisplayFormat: string;
    procedure SetAsInteger(value: LongInt);
    procedure SetAsFloat(value: Double);
    function GetAsInteger: LongInt;
    function GetAsFloat: Double;
    procedure FormatDisplay;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property AsInteger: LongInt read GetAsInteger write SetAsInteger;
    property AsFloat: Double read GetAsFloat write SetAsFloat;
    property DisplayFormat: string read GetDisplayFormat write SetDisplayFormat;
  end;

  TNumEdit = class(TCustomMaskEdit)
  private
    FDisplayFormat: String;
    FEditFormat: String;
    FCanvas: TControlCanvas;
    FAlignment: TAlignment;
    FTextMargin: Integer;
    FValue: Double;
    procedure SetDisplayFormat(value: string);
    procedure SetEditFormat(value: String);
    procedure SetAsInteger(value: LongInt);
    procedure SetAsFloat(value: Double);
    function GetAsFloat: Double;
    function GetAsInteger: LongInt;
    function GetDisplayFormat: string;
    function GetEditFormat: string;
    function GetTextTop: Integer;
    procedure FormatDisplay;
    procedure FormatEdit;
    procedure CMEnter(var Message: TCMEnter); message CM_ENTER;
    procedure CMExit(var Message: TCMExit); message CM_EXIT;
    procedure WMPaint(var Message: TWMPaint); message WM_PAINT;
    procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property DisplayFormat: string read GetDisplayFormat write SetDisplayFormat;
    property EditFormat: string read GetEditFormat write SetEditFormat;
    property AutoSelect;
    property AutoSize;
    property BorderStyle;
    property CharCase;
    property Color;
    property Ctl3D;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
    property MaxLength;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PasswordChar;
    property ReadOnly;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

  TIntegerEdit = class(TNumEdit)
  protected
    procedure KeyPress(var Key: Char); override;
  published
    property Value: LongInt read GetAsInteger write SetAsInteger;
  end;

  TFloatEdit = class(TNumEdit)
  protected
    procedure KeyPress(var Key: Char); override;
  published
    property Value: Double read GetAsFloat write SetAsFloat;
  end;

procedure Register;

implementation

Uses
  UStrings;

procedure Register;
begin
  RegisterComponents('Extra',[TNumLabel,TIntegerEdit,TFloatEdit]);
end;

{ --- TNumLabel ----------------------------------------------------}

constructor TNumLabel.Create(AOwner: TComponent);
begin;
  inherited Create(AOwner);
  Alignment := taRightJustify;
  FDisplayFormat := '';
  FValue := 0;
end;

destructor TNumLabel.Destroy;
begin;
  inherited Destroy;
end;

Procedure TNumLabel.SetDisplayFormat(value: String);
begin;
  if FDisplayFormat <> Value then FDisplayFormat := Value;
  FormatDisplay;
end;

Procedure TNumLabel.SetAsInteger(value: LongInt);
begin;
  FValue := value;
  FormatDisplay;
end;

Procedure TNumLabel.SetAsFloat(value: Double);
begin;
  FValue := value;
  FormatDisplay;
end;

function TNumLabel.GetAsInteger: LongInt;
begin;
  GetAsInteger := Trunc(FValue);
end;

function TNumLabel.GetAsFloat: Double;
begin;
  GetAsFloat := FValue;
end;

Function TNumLabel.GetDisplayFormat: String;
begin;
  GetDisplayFormat := FDisplayFormat;
end;

procedure TNumLabel.FormatDisplay;
begin;
  Caption := FormatFloat(FDisplayFormat,FValue);
end;

{ --- TNumEdit ------------------------------------------------------}

constructor TNumEdit.Create(AOwner: TComponent);
begin;
  inherited Create(AOwner);
  FValue := 0;
  FDisplayFormat := '';
  FEditFormat := '';
  FTextMargin := GetTextTop;
  FAlignment := taRightJustify;
end;

destructor TNumEdit.Destroy;
begin;
  if Assigned(FCanvas) then FCanvas.Destroy;
  inherited Destroy;
end;

procedure TNumEdit.SetDisplayFormat(value: string);
begin;
  if FDisplayFormat <> Value then FDisplayFormat := Value;
  FormatDisplay;
end;

procedure TNumEdit.SetEditFormat(value: String);
begin;
  if FEditFormat <> Value then FEditFormat := Value;
end;

function TNumEdit.GetDisplayFormat: string;
begin;
  GetDisplayFormat := FDisplayFormat;
end;

function TNumEdit.GetEditFormat: string;
begin;
  GetEditFormat := FEditFormat;
end;

procedure TNumEdit.CMEnter(var Message: TCMEnter);
begin
  FAlignment := taLeftJustify;
  FormatEdit;
  inherited;
end;

procedure TNumEdit.CMExit(var Message: TCMExit);
begin
  Try
    If Trim(Text) = '' then
      FValue := 0
    else
      FValue := StrToFloat(StripStr(ThousandSeparator,Text));
    FormatDisplay;
  except
    SelectAll;
    SetFocus;
    raise;
  end;
  SetCursor(0);
  FAlignment := taRightJustify;
  Invalidate;
  DoExit;
end;

procedure TNumEdit.WMPaint(var Message: TWMPaint);
var
  PS: TPaintStruct;
  ARect: TRect;
  S: array[0..255] of Char;
  TextLeft, TextTop: Integer;
  TextWid: Integer;
const
  Formats: array[TAlignment] of Word = (DT_LEFT, DT_RIGHT,
    DT_CENTER or DT_WORDBREAK or DT_EXPANDTABS or DT_NOPREFIX);
begin
  if (FAlignment = taLeftJustify) then
    inherited
  else
  begin
    if FCanvas = nil then
    begin
      FCanvas := TControlCanvas.Create;
      FCanvas.Control := Self;
    end;
    FCanvas.Handle := BeginPaint(Handle, PS);
    try
      FCanvas.Font := Font;
      with FCanvas do
      begin
        ARect := ClientRect;
        if not NewStyleControls and (BorderStyle = bsSingle) then
        begin
          Brush.Color := clWindowFrame;
          FrameRect(ARect);
          Inc(ARect.Left);
          Inc(ARect.Top);
          Dec(ARect.Bottom);
          Dec(ARect.Right);
        end;
        Brush.Style := bsSolid;
        Brush.Color := Color;
        FillRect (ARect);
        TextTop := FTextMargin;
        if BorderStyle = bsNone then TextTop := 0;
        ARect.Left := ARect.Left + 2;
        ARect.Right := ARect.Right - 2;
        StrPCopy (S, Text);
        TextWid := TextWidth(Text);
        if FAlignment = taRightJustify then
          TextLeft := ARect.Right - TextWid - 3
        else
          TextLeft := (ARect.Right - TextWid) div 2;
        if not Enabled then Font.Color := clGrayText;
        ExtTextOut(FCanvas.Handle, TextLeft,
          TextTop, ETO_OPAQUE or ETO_CLIPPED, @ARect,
          S, StrLen(S), nil);
      end;
    finally
      FCanvas.Handle := 0;
      EndPaint(Handle, PS);
    end;
  end;
end;

procedure TNumEdit.CMFontChanged(var Message: TMessage);
begin
  inherited;
  FTextMargin := GetTextTop;
end;

function TNumEdit.GetTextTop: Integer;
var
  DC: HDC;
  SaveFont: HFont;
  I: Integer;
  SysMetrics, Metrics: TTextMetric;
begin
  DC := GetDC(0);
  GetTextMetrics(DC, SysMetrics);
  SaveFont := SelectObject(DC, Font.Handle);
  GetTextMetrics(DC, Metrics);
  SelectObject(DC, SaveFont);
  ReleaseDC(0, DC);
  I := SysMetrics.tmHeight;
  if I > Metrics.tmHeight then I := Metrics.tmHeight;
  Result := I div 4;
  if BorderStyle = bsNone then Result := 2;
end;

procedure TNumEdit.SetAsInteger(value: LongInt);
begin
  FValue := value;
  FormatDisplay;
end;

procedure TNumEdit.SetAsFloat(value: Double);
begin
  FValue := value;
  FormatDisplay;
end;

function TNumEdit.GetAsFloat: Double;
begin
  GetAsFloat := FValue;
end;

function TNumEdit.GetAsInteger: LongInt;
begin
  GetAsInteger := Trunc(FValue);
end;

procedure TNumEdit.FormatDisplay;
begin;
  Text := FormatFloat(FDisplayFormat,FValue);
end;

procedure TNumEdit.FormatEdit;
begin;
  If FValue = 0.0 then Exit;
  Text := FormatFloat(FDisplayFormat,FValue);
end;

{ --- TIntegerEdit -------------------------------------------------}

procedure TIntegerEdit.KeyPress(var Key: Char);
begin;
  inherited KeyPress(Key);
  if not (Key in [^H,'-','0'..'9']) then Key := #0;
end;

{ --- TFloatEdit ---------------------------------------------------}

procedure TFloatEdit.KeyPress(var Key: Char);
begin;
  inherited KeyPress(Key);
  if not (Key in [DecimalSeparator,^H,'-','0'..'9']) then Key := #0;
end;

end.
