(************************************
 *** TSoniqueVis 1.1 by Baerware ****
 *** (c) 2003 Sebastian Brhausen ***
 *** e-mail: baerware@web.de ********
 ************************************)


(*
!!!IMPORTANT!!!
To be able to receive waveforms and spectrums from Sonique and to pass these values
to the plugin rendered by TSoniqueVis, it is necessary to copy the 'Baerware_Dummy_Plugin'
directory to the 'vis' subdirectory of the Sonique path (e.g. 'C:\Programs\Sonique\vis').
Then start Sonique, set the visual plugin to 'Baerware Dummy Plugin' and close Sonique again.

Version history:
- 1.1; 19.09.03: - TSoniqueVis is now derived from TCustomControl
								 - Minor bugfixes like OnResize
								 - Problem with missing VisData solved when Sonique is not in Vis Mode
								 - Event OnRender added for custom paint actions after rendering the Visual Plugin
								 - Events OnKeyDown, OnKeyPress and OnKeyUp added
								 - Register procedure moved to TSoniquePackRegister
- 1.0; 04.11.02: Initial release


New designtime properties:
- CursorFullscreen: cursor in fullscreen window mode (default is crNone = -1)
- FramesPerSecond: desired frame rate; very large numbers (e.g. 10000) cause fastest possible
	rendering (default is 20)
- Priority: priority of the rendering thread (default is tpLower to prevent Sonique from
	interrupting MP3 playback on slower computers and to smoothen the waveform animations)
- ShowFramesPerSecond: flag whether to show the current FPS in the upper right corner
  (default is False)
- ShowSongName: property for showing the song name never, only in normal window mode,
	only in fullscreen mode or always in the upper left corner (default is ssnFullscreen)
- SoniqueRemote: connection to a TSoniqueRemote object for getting the current song name
	and the path to Sonique.exe (default is nil)
- VisFileName: name of a *.svp Sonique Visual Plugin file (default is '')
- VisINIFileName: name of the vis.ini file for the plugin options (default is the
	Sonique path + '\vis.ini' if SoniqueRemote is assigned)

New runtime properties:
- Fullscreen: sets fullscreen or normal window mode (default is False)
- VisName: name of the current vis (read only)

New events:
- OnFullscreenChanged: triggered if the window mode changes to fullscreen or back

New methods:
- Clear: draw blank black background
- GetVisNameFromFile: gets the name of a Sonique Visual Plugin
- VisDataInitialized: tests whether the VisData structures for receiving waveforms and
	spectrums from Sonique are initialized
- VisInitialized: True if the Sonique Visual Plugin is loaded and initialized

Interface methods:
- OutlinedTextOut: draws an outlined text on a canvas

At the moment only the old header for Sonique Visual Plugins is supported. The new header
(available at http://sonique.lycos.com/customize/developer.php) adds some new functions like
mouse events and sending such strings like the current song name. Perhaps these features are
implemented in a newer version.
*)


unit SoniqueVis;

interface

uses
	Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs, ExtCtrls, SoniqueRemote, SyncObjs;

{$I SoniquePlugin.inc}

const
	DUMMY_PLUGIN_NAME = 'Baerware Dummy Plugin';
	SHARED_VISDATA_NAME = 'BaerwareDummyPluginVisData';
	FPS_AVERAGE_TIME = 2000;

type
	TRenderEvent = procedure(Sender: TObject; RenderBitmap: TBitmap) of object;
	TShowSongName = (ssnNever, ssnNormal, ssnFullscreen, ssnAlways);

	TRenderThread = class;

	TSoniqueVis = class(TCustomControl)
	private
		FCursorFullscreen: TCursor;
		FFramesPerSecond: Word;
		FFullscreenForm: TForm;
		FLastVisDataMillSec: Cardinal; //1.1: added
		FOnFullscreenChanged: TNotifyEvent; //1.1: changed
		FOnRender: TRenderEvent; //1.1: added
		FOnResize: TNotifyEvent; //1.1: added
		FRenderBitmap: TBitmap;
		FRenderInterval: Word;
		FRenderThread: TRenderThread;
		FRenderThreadPriority: TThreadPriority;
		FShowFramesPerSecond: Boolean;
		FShowSongName: TShowSongName;
		FSoniqueRemote: TSoniqueRemote;
		FVisData: PVisData;
		FVisDataHandle: THandle;
		FVisFileName: String;
		FVisHandle: THandle;
		FVisInfo: PVisInfo;
		FVisIniFileName: String;
		FWaitTerminated: TEvent; //1.1: added
	protected
		procedure DeInitializeVisData;
		procedure DoFullscreenClose(Sender: TObject; var Action: TCloseAction);
		procedure DoFullscreenKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
		procedure DoFullscreenKeyPress(Sender: TObject; var Key: Char); //1.1: added
		procedure DoFullscreenKeyUp(Sender: TObject; var Key: Word; Shift: TShiftState); //1.1: added
		procedure DoFullscreenPaint(Sender: TObject);
		procedure DoResize(Sender: TObject);
		procedure DrawFramesPerSecond(FPS: Real);
		procedure DrawSongName;
		function GetFullscreen: Boolean;
		function GetVisName: String;
		procedure InitializeVisData;
		procedure Loaded; override;
		procedure Render;
		procedure SetCursorFullscreen(const Value: TCursor);
		procedure SetEnabled(Value: Boolean); override;
		procedure SetFramesPerSecond(const Value: Word);
		procedure SetFullscreen(const Value: Boolean);
		procedure SetHeight(const Value: Integer);
		procedure SetRenderThreadPriority(const Value: TThreadPriority);
		procedure SetSoniqueRemote(const Value: TSoniqueRemote);
		procedure SetVisFileName(const Value: String);
		procedure SetVisible(const Value: Boolean);
		procedure SetWidth(const Value: Integer);
		procedure StartRendering;
	public
		procedure Clear;
		constructor Create(AOwner: TComponent); override;
		destructor Destroy; override;
		function GetVisNameFromFile(const FileName: String): String;
		procedure Paint; override;
		function VisDataInitialized: Boolean;
		function VisInitialized: Boolean;

		property Fullscreen: Boolean read GetFullscreen write SetFullscreen;
	published
		property Align;
		property Anchors;
		property Constraints;
		property Enabled write SetEnabled;
		property FramesPerSecond: Word read FFramesPerSecond write SetFramesPerSecond;
		property CursorFullscreen: TCursor read FCursorFullscreen write SetCursorFullscreen;
		property Height write SetHeight;
		property OnClick;
		property OnDblClick;
		property OnFullscreenChanged: TNotifyevent read FOnFullscreenChanged write FOnFullscreenChanged; //1.1: changed
		property OnKeyDown; //1.1: added
		property OnKeyPress; //1.1: added
		property OnKeyUp; //1.1: added
		property OnMouseDown;
		property OnMouseMove;
		property OnMouseUp;
		property OnRender: TRenderEvent read FOnRender write FOnRender; //1.1: added
		property OnResize read FOnResize write FOnResize; //1.1: changed
		property Priority: TThreadPriority read FRenderThreadPriority write SetRenderThreadPriority;
		property ShowFramesPerSecond: Boolean read FShowFramesPerSecond write FShowFramesPerSecond;
		property ShowSongName: TShowSongName read FShowSongName write FShowSongName;
		property SoniqueRemote: TSoniqueRemote read FSoniqueRemote write SetSoniqueRemote;
		property VisFileName: String read FVisFileName write SetVisFileName;
		property Visible write SetVisible;
		property VisINIFileName: String read FVisIniFileName write FVisIniFileName;
		property VisName: String read GetVisName;
		property Width write SetWidth;
	end;

	TRenderThread = class(TThread)
	private
		FSoniqueVis: TSoniqueVis;
	protected
		procedure Execute; override;
	public
		constructor Create(SoniqueVis: TSoniqueVis; NewPriority: TThreadPriority);
	published
	end;

procedure OutlinedTextOut(Canvas: TCanvas; X, Y: Integer; const Text: string;
	OutlineColor: TColor);

implementation

procedure OutlinedTextOut(Canvas: TCanvas; X, Y: Integer; const Text: string;
	OutlineColor: TColor);
var OldBrushStyle: TBrushStyle;
		OldFontColor: TColor;
		DX, DY: Integer;
begin
	with Canvas do
	begin
		Lock;
		OldFontColor := Font.Color;
		Font.Color := OutlineColor;
		OldBrushStyle := Brush.Style;
		Brush.Style := bsClear;
		for DX := -1 to 1 do for DY := -1 to 1 do TextOut(X + DX, Y + DY, Text);
		Font.Color := OldFontColor;
		TextOut(X, Y, Text);
		Brush.Style := OldBrushStyle;
		Unlock;
	end;
end;


{ TSoniqueVis }

procedure TSoniqueVis.Clear;
begin
	FRenderBitmap.Canvas.Lock;
	FRenderBitmap.Canvas.Brush.Style := bsSolid;
	FRenderBitmap.Canvas.Brush.Color := clBlack;
	FRenderBitmap.Canvas.Pen.Color := clBlack;
	FRenderBitmap.Canvas.Pen.Style := psSolid;
	FRenderBitmap.Canvas.Rectangle(0, 0, FRenderBitmap.Width, FRenderBitmap.Height);
	FRenderBitmap.Canvas.Unlock;
	if Fullscreen then FFullscreenForm.Repaint else	Repaint;
end;

constructor TSoniqueVis.Create(AOwner: TComponent);
begin
	inherited;
	FWaitTerminated := TEvent.Create(nil, True, True, '');
	InitializeVisData;
	FLastVisDataMillSec := 0;
	FRenderBitmap := TBitmap.Create;
	FRenderBitmap.PixelFormat := pf32Bit;
	inherited OnResize := DoResize;
	Priority := tpLower;

	ControlStyle := [csCaptureMouse, csClickEvents, csDoubleClicks, csOpaque];
	CursorFullscreen := crNone;
	DoubleBuffered := True;
	FramesPerSecond := 20;
	ShowSongName := ssnFullscreen;
	TabStop := False;
	Width := 320;
	Height := 240;
end;

procedure TSoniqueVis.DeInitializeVisData;
begin
	if VisDataInitialized then
	begin
		if Assigned(FVisData) then UnmapViewOfFile(FVisData);
		FVisData := nil;
		CloseHandle(FVisDataHandle);
		FVisDataHandle := 0;
	end;
	if Assigned(FVisData) then Dispose(FVisData);
	FVisData := nil;
end;

destructor TSoniqueVis.Destroy;
begin
	FWaitTerminated.Free;
	VisFileName := '';
	FRenderBitmap.Free;
	DeInitializeVisData;
	inherited;
end;

procedure TSoniqueVis.DoFullscreenClose(Sender: TObject; var Action: TCloseAction);
var OldEnabled: Boolean;
begin
	OldEnabled := Enabled;
	Enabled := False;
	FWaitTerminated.WaitFor(10000);
	Action := caFree;
	FFullscreenForm := nil;
	FRenderBitmap.Width := Width;
	FRenderBitmap.Height := Height;
	Clear;
	Enabled := OldEnabled;
	if Assigned(OnFullscreenChanged) then OnFullscreenChanged(Self);
end;


procedure TSoniqueVis.DoFullscreenKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
	if Key = VK_ESCAPE then Fullscreen := False;
	if Assigned(OnKeyDown) then OnKeyDown(Sender, Key, Shift);
end;

procedure TSoniqueVis.DoFullscreenKeyPress(Sender: TObject; var Key: Char);
begin
	if Assigned(OnKeyPress) then OnKeyPress(Sender, Key);
end;

procedure TSoniqueVis.DoFullscreenKeyUp(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
	if Assigned(OnKeyUp) then OnKeyUp(Sender, Key, Shift);
end;

procedure TSoniqueVis.DoFullscreenPaint(Sender: TObject);
begin
	if not Assigned(FRenderThread) then	FFullscreenForm.Canvas.Draw(0, 0, FRenderBitmap);
end;

procedure TSoniqueVis.DoResize(Sender: TObject);
var OldEnabled: Boolean;
begin
	if Fullscreen then Exit;
	OldEnabled := Enabled;
	Enabled := False;
	FWaitTerminated.WaitFor(10000);
	FRenderBitmap.Width := Width;
	FRenderBitmap.Height := Height;
	Clear;
	Enabled := OldEnabled;
	if Assigned(OnResize) then OnResize(Sender);
end;

procedure TSoniqueVis.DrawFramesPerSecond(FPS: Real);
var Text: String;
begin
	if not ShowFramesPerSecond or (FPS <= 0) then Exit;
	Text := FloatToStrF(FPS, ffFixed, 10, 2) + ' FPS';
	with FRenderBitmap.Canvas do
	begin
		Lock;
		Font.Size := 10;
		Font.Name := 'Arial';
		Font.Color := clWhite;
		OutlinedTextOut(FRenderBitmap.Canvas, FRenderBitmap.Width - TextWidth(Text) - 7, 5, Text, clBlack);
		Unlock;
	end;
end;

procedure TSoniqueVis.DrawSongName;
var Text: String;
begin
	if Assigned(SoniqueRemote) and ((ShowSongName = ssnAlways) or ((ShowSongName = ssnNormal) and not Fullscreen) or ((ShowSongName = ssnFullscreen) and Fullscreen)) then with FRenderBitmap.Canvas do
	begin
		Lock;
		if SoniqueRemote.Artist <> '' then Text := SoniqueRemote.Artist + ' - ' else Text := '';
		Text := Text + SoniqueRemote.Title;
		Font.Size := 10;
		Font.Name := 'Arial';
		Brush.Style := bsClear;
		Font.Color := clWhite;
		OutlinedTextOut(FRenderBitmap.Canvas, 7, 5, Text, clBlack);
		Unlock;
	end;
end;

function TSoniqueVis.GetFullscreen: Boolean;
begin
	GetFullscreen := Assigned(FFullscreenForm);
end;

function TSoniqueVis.GetVisName: String;
begin
	if VisInitialized then GetVisName := StrPas(FVisInfo^.PluginName) else GetVisName := '';
end;

function TSoniqueVis.GetVisNameFromFile(const FileName: String): String;
var QueryInstanceProc: function: PVisInfo; cdecl;
		Buffer: array[0..255] of Char;
		TempVisHandle: THandle;
		TempVisInfo: PVisInfo;
begin
	GetVisNameFromFile := '';
	TempVisHandle := LoadLibrary(StrPCopy(Buffer, FileName));
	if TempVisHandle <> 0 then
	begin
		@QueryInstanceProc := GetProcAddress(TempVisHandle, 'QueryModule');
		if Assigned(QueryInstanceProc) then
		begin
			TempVisInfo := QueryInstanceProc;
			if Assigned(TempVisInfo) then if TempVisInfo^.PluginName <> DUMMY_PLUGIN_NAME then
				GetVisNameFromFile := TempVisInfo^.PluginName;
		end;
		FreeLibrary(TempVisHandle);
	end;
end;

procedure TSoniqueVis.InitializeVisData;
var SampleNr: Word;
		SoniqueHandle: THandle;
begin
	if csDesigning in ComponentState then Exit;
	FVisDataHandle := CreateFileMapping(THandle($FFFFFFFF), nil,
		PAGE_READWRITE or SEC_COMMIT, 0, SizeOf(TVisData) + 1, SHARED_VISDATA_NAME);
	if FVisDataHandle <> 0 then FVisData := MapViewOfFile(FVisDataHandle, FILE_MAP_ALL_ACCESS, 0, 0, 0);
	if not Assigned(FVisData) then
	begin
		FVisData := New(PVisData);
		for SampleNr := 0 to 511 do
		begin
			FVisData^.Waveform[0][SampleNr] := Round(Sin(SampleNr / 511 * 2 * PI) * 127);
			FVisData^.Waveform[1][SampleNr] := FVisData^.Waveform[0][SampleNr];
		end;
		if FVisDataHandle <> 0 then
		begin
			CloseHandle(FVisDataHandle);
			FVisDataHandle := 0;
		end;
	end;
	SoniqueHandle := FindWindow(SONIQUE_CLASS_NAME, nil);
	if SoniqueHandle <> 0 then
	begin
		SendKeyEvent(SoniqueHandle, 80, 112);
		SendKeyEvent(SoniqueHandle, 9, 9);
	end;
end;

procedure TSoniqueVis.Loaded;
var TempVisFileName: String;
begin
	inherited;
	if VisFileName <> '' then
	begin
		TempVisFileName := VisFileName;
		FVisFileName := '';
		VisFileName := TempVisFileName;
	end;
end;

procedure TSoniqueVis.Paint;
begin
	inherited;
	if (not Enabled or not Assigned(FRenderThread)) and not Fullscreen then Canvas.Draw(0, 0, FRenderBitmap);
end;

procedure TSoniqueVis.Render;
var CurrentMilliSeconds: Cardinal;
		FPS: Real;
		FPSCounter: Word;
		FPSTime: Cardinal;
		RenderTime: Cardinal;
begin
	FWaitTerminated.ResetEvent;
	FPS := 0;
	FPSCounter := 0;
	FPSTime := 0;
	RenderTime := GetTickCount;
	while Enabled and Visible and VisInitialized and not (csDesigning in ComponentState) and not (csLoading in ComponentState) do
	begin
		CurrentMilliSeconds := GetTickCount;
		if CurrentMilliSeconds - RenderTime >= FRenderInterval then
		begin
			RenderTime := CurrentMilliSeconds;
			if FPSTime = 0 then	FPSTime := CurrentMilliSeconds;
			if CurrentMilliSeconds - FPSTime >= FPS_AVERAGE_TIME then
			begin
				FPS := FPSCounter / (CurrentMilliSeconds - FPSTime) * 1000;
				FPSTime := CurrentMilliSeconds;
				FPSCounter := 0;
			end;
			FRenderBitmap.Canvas.Lock;
			if not VisDataInitialized then FVisData.MillSec := GetTickCount;
			try
				FVisInfo^.RenderProc(FRenderBitmap.ScanLine[0], FRenderBitmap.Width, FRenderBitmap.Height, -FRenderBitmap.Width, FVisData);
			except on Exception do end;
			DrawSongName;
			DrawFramesPerSecond(FPS);
			try
				if Assigned(OnRender) then OnRender(Self, FRenderBitmap);
				if Fullscreen then
				begin
					FFullscreenForm.Canvas.Lock;
					FFullscreenForm.Canvas.Draw(0, 0, FRenderBitmap);
					FFullscreenForm.Canvas.Unlock;
				end
				else
				begin
					Canvas.Lock;
					Canvas.Draw(0, 0, FRenderBitmap);
					Canvas.Unlock;
				end;
			except on Exception do end;
			FRenderBitmap.Canvas.Unlock;
			Inc(FPSCounter);
		end;
		Application.ProcessMessages;
	end;
	FRenderThread := nil;
	FWaitTerminated.SetEvent;
end;

procedure TSoniqueVis.SetCursorFullscreen(const Value: TCursor);
begin
	if Value = CursorFullscreen then Exit;
	FCursorFullscreen := Value;
	if Fullscreen then FFullscreenForm.Cursor := CursorFullscreen;
end;

procedure TSoniqueVis.SetEnabled(Value: Boolean);
begin
	if Value = Enabled then Exit;
	inherited;
	StartRendering;
end;

procedure TSoniqueVis.SetFramesPerSecond(const Value: Word);
begin
	if Value = FFramesPerSecond then Exit;
	FFramesPerSecond := Value;
	FRenderInterval := 1000 div FramesPerSecond;
end;

procedure TSoniqueVis.SetFullscreen(const Value: Boolean);
var OldEnabled: Boolean;
begin
	if Value = Fullscreen then Exit;
	OldEnabled := Enabled;
	Enabled := False;
	FWaitTerminated.WaitFor(10000); 
	if Value then
	begin
		FFullscreenForm := TForm.Create(Self);
		with FFullscreenForm do
		begin
			BorderStyle := bsNone;
			FormStyle := fsStayOnTop;
			Cursor := crNone;
			Left := 0;
			Top := 0;
			ClientWidth := Screen.Width;
			ClientHeight := Screen.Height;
			Cursor := CursorFullscreen;
			KeyPreview := True;
			OnClick := Self.OnClick;
			OnClose := DoFullscreenClose;
			OnDblClick := Self.OnDblClick;
			OnMouseDown := Self.OnMouseDown;
			OnMouseMove := Self.OnMouseMove;
			OnMouseUp := Self.OnMouseUp;
			OnPaint := DoFullscreenPaint;
			OnKeyDown := DoFullscreenKeyDown;
			OnKeyPress := DoFullscreenKeyPress;
			OnKeyUp := DoFullscreenKeyUp;
			FRenderBitmap.Width := Screen.Width;
			FRenderBitmap.Height := Screen.Height;
			Show;
			SetForegroundWindow(FFullscreenForm.Handle);
			Clear;
		end;
	end
	else FFullscreenForm.Close;
	Enabled := OldEnabled;
	if Assigned(OnFullscreenChanged) then OnFullscreenChanged(Self);
end;

procedure TSoniqueVis.SetHeight(const Value: Integer);
begin
	if Value = Height then Exit;
	inherited Height := Value;
	DoResize(Self);
end;

procedure TSoniqueVis.SetRenderThreadPriority(const Value: TThreadPriority);
begin
	if Value = FRenderThreadPriority then Exit;
	FRenderThreadPriority := Value;
	if Assigned(FRenderThread) then FRenderThread.Priority := Priority;
end;

procedure TSoniqueVis.SetSoniqueRemote(const Value: TSoniqueRemote);
begin
	if Value = SoniqueRemote then Exit;
	FSoniqueRemote := Value;
	if Assigned(SoniqueRemote) and (VisINIFileName = '') and not (csDesigning in ComponentState) then VisINIFileName := ExtractFileDir(SoniqueRemote.SoniqueEXE) + '\vis.ini';
end;

procedure TSoniqueVis.SetVisFileName(const Value: String);
var QueryInstanceProc: function: PVisInfo; cdecl;
		Buffer: array[0..255] of Char;
		OldDir: String;
begin
	if (csDesigning in ComponentState) or (csLoading in ComponentState) then
	begin
		FVisFileName := Value;
		Exit;
	end;
	if (Value = FVisFileName) then Exit;
	FVisFileName := '';
	FWaitTerminated.WaitFor(10000);
	if FVisHandle <> 0 then
	begin
		if Assigned(FVisInfo) then
		begin
			if VisINIFileName <> '' then FVisInfo^.SaveSettingsProc(StrPCopy(Buffer, VisINIFileName));
		end;
		FreeLibrary(FVisHandle);
		FVisHandle := 0;
	end;
	if Value = '' then Exit;
	FVisHandle := LoadLibrary(StrPCopy(Buffer, Value));
	if FVisHandle <> 0 then
	begin
		@QueryInstanceProc := GetProcAddress(FVisHandle, 'QueryModule');
		if Assigned(QueryInstanceProc) then
		begin
			FVisInfo := QueryInstanceProc;
			if Assigned(FVisInfo) then
			begin
				OldDir := GetCurrentDir;
				ChDir(ExtractFileDir(Value));
				if VisINIFileName <> '' then FVisInfo^.OpenSettingsProc(StrPCopy(Buffer, VisINIFileName));
				FVisInfo^.InitializeProc;
				ChDir(OldDir);
				FVisFileName := Value;
				Clear;
				StartRendering;
			end;
		end;
	end;
end;

procedure TSoniqueVis.SetVisible(const Value: Boolean);
begin
	if Value = Visible then Exit;
	inherited Visible := Value;
	StartRendering;
end;

procedure TSoniqueVis.SetWidth(const Value: Integer);
begin
	if Value = Width then Exit;
	inherited Width := Value;
	DoResize(Self);
end;

procedure TSoniqueVis.StartRendering;
begin
	if not Assigned(FRenderThread) and Enabled and Visible and VisInitialized and not (csDesigning in ComponentState) and not (csLoading in ComponentState) then FRenderThread := TRenderThread.Create(Self, Priority);
end;

function TSoniqueVis.VisDataInitialized: Boolean;
begin
	VisDataInitialized := FVisDataHandle <> 0;
end;

function TSoniqueVis.VisInitialized: Boolean;
begin
	VisInitialized := VisFileName <> '';
end;


{ TRenderThread }

constructor TRenderThread.Create(SoniqueVis: TSoniqueVis; NewPriority: TThreadPriority);
begin
	inherited Create(False);
	Priority := NewPriority;
	FreeOnTerminate := True;
	FSoniqueVis := SoniqueVis;
end;

procedure TRenderThread.Execute;
begin
	if not Assigned(FSoniqueVis) then Exit;
	FSoniqueVis.Render;
end;

end.
