(**************************************
 *** TSoniqueRemote 1.1 by Baerware ***
 *** (c) 2003 Sebastian Brhausen *****
 *** e-mail: baerware@web.de **********
 **************************************)


(*
!!!IMPORTANT!!!
To ensure that TSoniqueRemote works properly, please DISABLE the following settings in the
Sonique setup options:
- System -> Always on top (not necessary but perhaps needed if AutoHideSonique is False)
- System -> Show Sonique on taskbar (not necessary but a nice feature)
- Playlist -> Append to playlist (necessary)
- Playlist -> Play added songs immediately (necessary)
- set play mode to normal (NOT shuffle or repeat; necessary)

Version history:
- 1.1; 19.09.03: - More reliable registry key for Sonique.exe used
								 - Minor changes
								 - New interface methods SavePlaylist and LoadPlaylist
								 - SendKeyEvent is now an interface method
								 - Register procedure moved to TSoniquePackRegister
- 1.0; 04.11.02: Initial release


New designtime properties:
- AutoHideSonique: flag whether to hide the Sonique window automatically after finding it (default is True)
- AutoStartSonique: if enabled, starts Sonique automatically when application is launched (default is True)
- OnInfoChanged: triggered if Sonique writes new important information in the SoniqueInfo structure

New runtime properties:
- Album: album name read from the MP3 tag of the current song (read only)
- Artist: artist name read from the MP3 tag of the current song (read only)
- EqualizerEnabled: flag whether the equalizer is enabled (initial value is read from the registry)
- FileName: file name of the current song (read only)
- Genre: genre name identified by the genre number read from the MP3 tag of the current song (read only)
- PlayState: play state of Sonique: psStopped, psPlaying, psPaused
- SongLength: length of the current song in ms (read only)
- SongPosition: position in the current song in ms (read only)
- SoniqueEXE: full path and file name of the Sonique executable file (read from the registry; read only)
- SoniqueHandle: handle of the Sonique window (read only)
- SoniqueStarted: tests whether Sonique is running (read only)
- Title: title read from the MP3 tag of the current song (read only)
- Year: year read from the MP3 tag of the current song (read only)

New methods:
- AddFile: adds an MP3 file / playlist to the Sonique playlist
- CloseSonique: closes Sonique
- DeleteSong: deletes the current song from the Sonique playlist
- HideSonique: hides the Sonique window
- InfoInitialized: tests whether the SoniqueInfo structures for receiving
  song information like the current file name from Sonique are initialized
- NextSong: activates the next song in the Sonique playlist
- Pause: pauses Sonique
- Play: plays the active song
- PlayFile: replaces the Sonique playlist with an MP3 file / playlist and starts playing
- PreviousSong: activates the previous song in the Sonique playlist
- ShowSonique: shows the Sonique window in its original position
- StartSonique: starts Sonique if it isn't launched yet
- Stop: stops Sonique

Interface methods:
- ReadID3Tag: reads an MP3 tag from a file
- WriteID3Tag: writes an MP3 tag to a file
*)


unit SoniqueRemote;

interface

uses
	Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs, Registry, IniFiles;

const
	ID3TAG_SIZE = 128;
	MP3TAG = 'TAG';
	MAX_GENRES = 114;
	GENRES: array [0..MAX_GENRES] of String =
	(
		'Blues','Classic Rock','Country','Dance','Disco','Funk','Grunge','Hip-Hop','Jazz','Metal','New Age','Oldies',
		'Other','Pop','R&B','Rap','Reggae','Rock','Techno','Industrial','Alternative','Ska','Death Metal','Pranks',
		'Soundtrack','Euro-Techno','Ambient','Trip-Hop','Vocal','Jazz+Funk','Fusion','Trance','Classical','Instrumental',
		'Acid','House','Game','Sound Clip','Gospel','Noise','AlternRock','Bass','Soul','Punk','Space','Meditative',
		'Instrumental Pop','Instrumental Rock','Ethnic','Gothic','Darkwave','Techno-Industrial','Electronic','Pop-Folk',
		'Eurodance','Dream','Southern Rock','Comedy','Cult','Gangsta','Top 40','Christian Rap','Pop/Funk','Jungle',
		'Native American','Cabaret','New Wave','Psychadelic','Rave','Showtunes','Trailer','Lo-Fi','Tribal','Acid Punk',
		'Acid Jazz','Polka','Retro','Musical','Rock & Roll','Hard Rock','Folk','Folk/Rock','National Folk','Swing','Bebob',
		'Latin','Revival','Celtic','Bluegrass','Avantgarde','Gothic Rock','Progressive Rock','Psychedelic Rock','Symphonic Rock',
		'Slow Rock','Big Band','Chorus','Easy Listening','Acoustic','Humour','Speech','Chanson','Opera','Chamber Music','Sonata',
		'Symphony','Booty Bass','Primus','Porn Groove','Satire','Slow Jam','Club','Tango','Samba','Folklore'
	);

	SHARED_INFO_NAME	= 'SoniqueInformation';
	INFO_EVENT_NAME = 'SoniqueInformationEvent';

	SONIQUE_EXE_REGISTRY_ENTRY = '\Sonique.File\shell\Open\command';
	SONIQUE_PREFS_REGISTRY_ENTRY = '\Software\MediaScience\Sonique\General Preferences 0.80';
	SONIQUE_CLASS_NAME = 'Sonique Window Class';
//	SONIQUE_CLASS_NAME = 'Scarab WindowClass'; //Sonique 2.x

type
	TID3Tag = record
		Tag: array[0..2] of Char;
		Title: array[0..29] of Char;
		Artist: array[0..29] of Char;
		Album: array[0..29] of Char;
		Year: array[0..3] of Char;
		Comment: array[0..29] of Char;
		Genre: Byte;
	end;

	PSoniqueInfo = ^TSoniqueInfo;
	TSoniqueInfo = record
		Size: Cardinal;
		State: Cardinal;
		SongLengthMS: Cardinal;
		SongPositionMS: Cardinal;
		Artist: array[0..255] of Char;
		Song: array[0..255] of Char;
		SkinColor: array[0..3] of Cardinal;
		SkinName: array[0..255] of Char;
		VisName: array[0..255] of Char;
		FileName: array[0..259] of Char;
	end;

	TWaitInfoChangedThread = class(TThread)
	private
		FChangeEvent: THandle;
		FOnInfoChanged: TNotifyEvent; //1.1: changed
	protected
		procedure Execute; override;
	public
		constructor Create(CreateSuspended: Boolean);
		destructor Destroy; override;
		procedure SetChangeEvent;
	published
		property OnInfoChanged: TNotifyEvent read FOnInfoChanged write FOnInfoChanged; //1.1: changed
	end;

	TPlayState = (psStopped, psPlaying, psPaused);

	TSoniqueRemote = class(TComponent)
	private
		FAlbum: String;
		FArtist: String;
		FAutoHideSonique: Boolean;
		FAutoStartSonique: Boolean;
		FComment: String;
		FEqualizerEnabled: Boolean;
		FFileName: String;
		FGenre: String;
		FOnInfoChanged: TNotifyEvent; //1.1: changed
		FPlayState: TPlayState;
		FSoniqueEXE: String;
		FSoniqueHandle: THandle;
		FSoniqueInfo: PSoniqueInfo;
		FSoniqueInfoHandle: THandle;
		FSoniqueWindowPlacement: TWindowPlacement;
		FTitle: String;
		FWaitInfoChangedThread: TWaitInfoChangedThread;
		FYear: String;
		procedure SetPlayState(const Value: TPlayState);
	protected
		procedure DeInitializeInfo;
		procedure DoInfoChanged(Sender: TObject);
		function GetSongLength: Integer; //1.1: changed
		function GetSongPosition: Integer; //1.1: changed
		function GetSoniqueStarted: Boolean;
		function InitializeInfo: Boolean;
		procedure InitializeSonique; //1.1: added
		procedure Loaded; override;
		procedure SetEqualizerEnabled(const Value: Boolean);
	public
		procedure AddFile(const NewFile: String);
		procedure CloseSonique;
		constructor Create(AOwner: TComponent); override;
		procedure DeleteSong;
		destructor Destroy; override;
		procedure HideSonique;
		function InfoInitialized: Boolean;
		procedure NextSong;
		procedure Pause;
		procedure Play;
		procedure PlayFile(const NewFile: String);
		procedure PreviousSong;
		procedure ShowSonique;
		function StartSonique: Boolean;
		procedure Stop;

		property Album: String read FAlbum;
		property Artist: String read FArtist;
		property Comment: String read FComment;
		property EqualizerEnabled: Boolean read FEqualizerEnabled write SetEqualizerEnabled;
		property FileName: String read FFileName;
		property Genre: String read FGenre;
		property PlayState: TPlayState read FPlayState write SetPlayState;
		property SongLength: Integer read GetSongLength; //1.1: changed
		property SongPosition: Integer read GetSongPosition; //1.1: changed
		property SoniqueEXE: String read FSoniqueEXE;
		property SoniqueHandle: THandle read FSoniqueHandle;
		property SoniqueStarted: Boolean read GetSoniqueStarted;
		property Title: String read FTitle;
		property Year: String read FYear;
	published
		property AutoHideSonique: Boolean read FAutoHideSonique write FAutoHideSonique;
		property AutoStartSonique: Boolean read FAutoStartSonique write FAutoStartSonique;
		property OnInfoChanged: TNotifyEvent read FOnInfoChanged write FOnInfoChanged; //1.1: changed
	end;

function SavePlaylist(FileName: String; Playlist: TStringList): Boolean; //1.1: added
function LoadPlaylist(FileName: String; var Playlist: TStringlist): Boolean; //1.1: added
function ReadID3Tag(FileName: String; var Tag: TID3Tag): Boolean;
procedure SendKeyEvent(WindowHandle: THandle; VirtualKey, Key: Word); //1.1: changed
Procedure WriteID3Tag(FileName: String; Tag: TID3Tag);

implementation

function SavePlaylist(FileName: String; Playlist: TStringList): Boolean;
var PlaylistFile: Text;
		Counter: Integer;
begin
	SavePlaylist := False;
	AssignFile(PlaylistFile, FileName);
	{$i-}
	Rewrite(PlaylistFile);
	{$i+}
	if IOResult <> 0 then Exit;
	SavePlaylist := True;
	Writeln(PlaylistFile, '[playlist]');
	for Counter := 0 to Playlist.Count - 1 do
		Writeln(PlaylistFile, 'File' + IntToStr(Counter + 1) + '=' + Playlist.Strings[Counter]);
	Writeln(PlaylistFile, 'NumberOfEntries=' + IntToStr(Playlist.Count));
	CloseFile(PlaylistFile);
end;

function LoadPlaylist(FileName: String; var Playlist: TStringlist): Boolean;
var Counter, NumberOfEntries: Integer;
		PlaylistINI: TINIFile;
		PlaylistDir, Song: String;
begin
	LoadPlaylist := False;
	if not FileExists(FileName) then Exit;
	LoadPlaylist := True;
	PlaylistDir := ExtractFileDir(PlaylistDir);
	if (Length(PlaylistDir) > 0) and (PlaylistDir[Length(PlaylistDir)] <> '\') then PlaylistDir := PlaylistDir + '\';
	PlaylistINI := TINIFile.Create(FileName);
	NumberOfEntries := PlaylistINI.ReadInteger('playlist', 'NumberOfEntries', 0);
	for Counter := 1 to NumberOfEntries do
	begin
		Song := PlaylistINI.ReadString('playlist', 'File' + IntToStr(Counter), '');
		if Song <> '' then
		begin
			if ExtractFileDir(Song) = '' then Song := PlaylistDir + Song;
			if FileExists(Song) then Playlist.Add(Song);
		end;
	end;
	PlaylistINI.Free;
end;

function ReadID3Tag(FileName: String; var Tag: TID3Tag): Boolean;
var FileStream: TFileStream;
begin
	if not FileExists(FileName) then Result := False
	else
	begin
		FileStream := TFileStream.Create(FileName, fmOpenRead or fmShareDenyWrite);
		FileStream.Seek(-ID3TAG_SIZE, soFromEnd);
		FileStream.Read(Tag, ID3TAG_SIZE);
		FileStream.Free;
		Result := Tag.tag = MP3TAG;
	end;
	if not Result then
	begin
		FillChar(Tag, ID3TAG_SIZE, ' ');
		Tag.Genre := 0;
	end;
end;

procedure SendKeyEvent(WindowHandle: THandle; VirtualKey, Key: Word);
begin
	PostMessage(WindowHandle, WM_KEYDOWN, VirtualKey, 1);
	if Key <> 0 then PostMessage(WindowHandle, WM_CHAR, Key, 1);
	PostMessage(WindowHandle, WM_KEYUP, VirtualKey, Longint($C0000001));
end;

Procedure WriteID3Tag(FileName: String; Tag: TID3Tag);
var FileStream: TFileStream;
		OldTag : array[0..2] of Char;
begin
	if not FileExists(FileName) then Exit;
	FileStream := TFileStream.Create(FileName, fmOpenReadWrite or fmShareExclusive);
	FileStream.Seek(-ID3TAG_SIZE, soFromEnd);
	FileStream.Read(OldTag, 3);
	if OldTag <> MP3TAG then FileStream.Seek(0, soFromEnd)
	else FileStream.Seek(-ID3TAG_SIZE, soFromEnd);
	FileStream.Write(Tag, ID3TAG_SIZE);
	FileStream.Free;
end;


{ TWaitInfoChangedThread }

constructor TWaitInfoChangedThread.Create(CreateSuspended: Boolean);
begin
	inherited;
	FChangeEvent := CreateEvent(nil, False, False, INFO_EVENT_NAME);
end;

destructor TWaitInfoChangedThread.Destroy;
begin
	if FChangeEvent <> 0 then CloseHandle(FChangeEvent);
	inherited;
end;

procedure TWaitInfoChangedThread.Execute;
begin
	if FChangeEvent = 0 then Exit;
	while not Terminated do
	begin
		if WaitForSingleObject(FChangeEvent, 100) = WAIT_OBJECT_0 then
			if Assigned(OnInfoChanged) then OnInfoChanged(Self);
	end;
end;

procedure TWaitInfoChangedThread.SetChangeEvent;
begin
	SetEvent(FChangeEvent);
end;


{ TSoniqueRemote }

procedure TSoniqueRemote.AddFile(const NewFile: String);
var Buf: array[0..255] of Char;
begin
	if SoniqueStarted then WinExec(StrPCopy(Buf, FSoniqueEXE + ' -appendonly "' + NewFile + '"'), SW_SHOWNOACTIVATE);
end;

procedure TSoniqueRemote.CloseSonique;
begin
	PostMessage(SoniqueHandle, WM_CLOSE, 0, 0);
end;

constructor TSoniqueRemote.Create(AOwner: TComponent);
var Registry: TRegistry;
begin
	inherited;
	Registry := TRegistry.Create(KEY_QUERY_VALUE);
	Registry.RootKey := HKEY_CLASSES_ROOT;
	if Registry.OpenKey(SONIQUE_EXE_REGISTRY_ENTRY, False) then
	begin
		FSoniqueEXE := Registry.ReadString('');
		FSoniqueEXE := Copy(FSoniqueEXE, 2, Length(FSoniqueEXE) - 7);
		Registry.CloseKey;
	end;
	Registry.RootKey := HKEY_CURRENT_USER;
	if Registry.OpenKey(SONIQUE_PREFS_REGISTRY_ENTRY, False) then
	begin
		FEqualizerEnabled := Registry.ReadString('EnableEQ') = '1';
		Registry.CloseKey;
	end;
	Registry.Free;
	FSoniqueWindowPlacement.Length := SizeOf(FSoniqueWindowPlacement);
	AutoHideSonique := True;
	AutoStartSonique := True;
end;

procedure TSoniqueRemote.DeInitializeInfo;
begin
	if InfoInitialized then
	begin
		FWaitInfoChangedThread.Free;
		if Assigned(FSoniqueInfo) then UnmapViewOfFile(FSoniqueInfo);
		FSoniqueInfo := nil;
		CloseHandle(FSoniqueInfoHandle);
	end;
end;

procedure TSoniqueRemote.DeleteSong;
begin
	if SoniqueStarted then SendKeyEvent(SoniqueHandle, 68, 100);
end;

destructor TSoniqueRemote.Destroy;
begin
	DeInitializeInfo;
	if AutoStartSonique and SoniqueStarted then CloseSonique;
	if AutoHideSonique and not AutoStartSonique then ShowSonique;
	inherited;
end;

procedure TSoniqueRemote.DoInfoChanged(Sender: TObject);
var Tag: TID3Tag;
begin
	FFileName := StrPas(FSoniqueInfo^.FileName);
	ReadID3Tag(FileName, Tag);
	FAlbum := Trim(Tag.Album);
	FArtist := Trim(Tag.Artist);
	FComment := Trim(Tag.Comment);
	if Tag.Genre > MAX_GENRES then FGenre := 'Unknown' else FGenre := GENRES[Tag.Genre];
	FTitle := Trim(Tag.Title);
	FYear := Trim(Tag.Year);
	if (Artist = '') and (Title = '') then
	begin
		FTitle := ExtractFileName(FileName);
		FTitle := Copy(Title, 1, Length(Title) - Length(ExtractFileExt(Title)));
	end;
	if Assigned(OnInfoChanged) and (Sender = FWaitInfoChangedThread) then OnInfoChanged(Sender);
end;

function TSoniqueRemote.GetSongLength: Integer;
begin
	if InfoInitialized then GetSongLength := FSoniqueInfo^.SongLengthMS
	else GetSongLength := 0;
end;

function TSoniqueRemote.GetSongPosition: Integer;
begin
	if InfoInitialized then GetSongPosition := FSoniqueInfo^.SongPositionMS
	else GetSongPosition := 0;
end;

function TSoniqueRemote.GetSoniqueStarted: Boolean;
begin
	if SoniqueHandle = 0 then
	begin
		FSoniqueHandle := FindWindow(SONIQUE_CLASS_NAME, nil);
		if SoniqueHandle <> 0 then
		begin
			ShowWindow(SoniqueHandle, SW_SHOWNOACTIVATE);
			InitializeSonique;
		end;
	end;
	GetSoniqueStarted := SoniqueHandle <> 0;
end;

procedure TSoniqueRemote.HideSonique;
begin
	if SoniqueStarted then MoveWindow(SoniqueHandle, -10000, -10000, 0, 0, True);
end;

function TSoniqueRemote.InfoInitialized: Boolean;
begin
	InfoInitialized := FSoniqueInfoHandle <> 0;
end;

function TSoniqueRemote.InitializeInfo: Boolean;
begin
	InitializeInfo := InfoInitialized;
	if InfoInitialized or (csDesigning in ComponentState) then Exit;
	FSoniqueInfoHandle := CreateFileMapping(THandle($FFFFFFFF), nil,
		PAGE_READWRITE or SEC_COMMIT, 0, SizeOf(TSoniqueInfo) + 1, SHARED_INFO_NAME);
	if FSoniqueInfoHandle = 0 then Exit;
	FSoniqueInfo := MapViewOfFile(FSoniqueInfoHandle, FILE_MAP_ALL_ACCESS, 0, 0, 0);
	if not Assigned(FSoniqueInfo) then
	begin
		CloseHandle(FSoniqueInfoHandle);
		Exit;
	end;
	InitializeInfo := True;
	FSoniqueInfo^.Size := SizeOf(TSoniqueInfo);
	FWaitInfoChangedThread := TWaitInfoChangedThread.Create(False);
	FWaitInfoChangedThread.OnInfoChanged := DoInfoChanged;
	DoInfoChanged(Self);
end;

procedure TSoniqueRemote.InitializeSonique;
begin
	if SoniqueHandle = 0 then Exit;
	GetWindowPlacement(SoniqueHandle, @FSoniqueWindowPlacement);
	if AutoHideSonique then HideSonique;
	FPlayState := psStopped;
	if SoniqueStarted then
	begin
		Sleep(200);
		SendKeyEvent(SoniqueHandle, 86, 118);
		Sleep(200);
		SendKeyEvent(SoniqueHandle, 80, 112);
		Sleep(200);
		SendKeyEvent(SoniqueHandle, 9, 9);
	end;
end;

procedure TSoniqueRemote.Loaded;
begin
	inherited;
	if AutoStartSonique and not (csDesigning in ComponentState) then StartSonique else SoniqueStarted;
	if not (csDesigning in ComponentState) then InitializeInfo;
end;

procedure TSoniqueRemote.NextSong;
begin
	if SoniqueStarted then SendKeyEvent(SoniqueHandle, 66, 98);
end;

procedure TSoniqueRemote.Pause;
begin
	if PlayState <> psPlaying then Exit;
	if SoniqueStarted then SendKeyEvent(SoniqueHandle, 67, 99);
	FPlayState := psPaused;
end;

procedure TSoniqueRemote.Play;
begin
	if PlayState = psPlaying then Exit;
	if SoniqueStarted then SendKeyEvent(SoniqueHandle, 88, 120);
	FPlayState := psPlaying;
end;

procedure TSoniqueRemote.PlayFile(const NewFile: String);
var Buf: array[0..255] of Char;
begin
	if SoniqueStarted then
	begin
		Stop;
		WinExec(StrPCopy(Buf, FSoniqueEXE + ' "' + NewFile + '"'), SW_SHOWNOACTIVATE);
		Sleep(500);
		Play;
	end;
end;

procedure TSoniqueRemote.PreviousSong;
begin
	if SoniqueStarted then SendKeyEvent(SoniqueHandle, 90, 122);
end;

procedure TSoniqueRemote.SetEqualizerEnabled(const Value: Boolean);
begin
	if Value = FEqualizerEnabled then Exit;
	FEqualizerEnabled := Value;
	if SoniqueStarted then SendKeyEvent(SoniqueHandle, 81, 113);
end;

procedure TSoniqueRemote.SetPlayState(const Value: TPlayState);
begin
	if Value = PlayState then Exit;
	case Value of
		psStopped: Stop;
		psPlaying: Play;
		psPaused: Pause;
	end;
end;

procedure TSoniqueRemote.ShowSonique;
begin
	if SoniqueStarted then SetWindowPlacement(SoniqueHandle, @FSoniqueWindowPlacement);
end;

function TSoniqueRemote.StartSonique: Boolean;
var Buf: array[0..255] of Char;
		TimeOut: Cardinal;
		TempHandle: THandle;
begin
	TempHandle := FindWindow(SONIQUE_CLASS_NAME, nil);
	if TempHandle <> 0 then
	begin
		StartSonique := True;
		if SoniqueHandle = 0 then
		begin
			FSoniqueHandle := TempHandle;
			InitializeSonique;
		end;
		Exit;
	end;
	StartSonique := False;
	if WinExec(StrPCopy(Buf, FSoniqueEXE), SW_SHOWNOACTIVATE) > 31 then
	begin
		TimeOut := GetTickCount;
		repeat
			Sleep(100);
			FSoniqueHandle := FindWindow(SONIQUE_CLASS_NAME, nil);
		until (GetTickCount - TimeOut > 10000) or (SoniqueHandle <> 0);
		if SoniqueHandle <> 0 then
		begin
			StartSonique := True;
			ShowWindow(SoniqueHandle, SW_SHOWNOACTIVATE);
			PostMessage(SoniqueHandle, WM_LBUTTONDOWN, MK_LBUTTON, 0);
			PostMessage(SoniqueHandle, WM_LBUTTONUP, 0, 0);
			Sleep(200);
			InitializeSonique;
		end;
	end;
end;

procedure TSoniqueRemote.Stop;
begin
	if SoniqueStarted then SendKeyEvent(SoniqueHandle, 86, 118);
	FPlayState := psStopped;
end;

end.
