unit treecoll;
{/*****************************************************************/
 /  Copyright (c) 1998 IMCA SYSTEMS
 /  All Rights Reserved.
 /  Creation Date: 06.01.99
 /  Version 0.31b
 /  MODULE:
 /  TTreeCollection - implementation Collection with
 /         hierarchical structure
 // TCollection and TCollectionItem united in one Class
 /  The given class will allow more simple to use a collection,
 /  creating a collection in a collection and compatibility with TreeNodes class
 // Sorting and Recursive Sorting
 // Loading From File,stream,Strings
 /  NOTES:
 /       The changes and additions are expected:
 //      a) Save and Load Collection From stream and File - Released!
 //      b) Sort Collection  - Released!
 //

 /  HISTORY:
 /*****************************************************************/}
{
 I take no responsibility for any losses suffered
 resulting from use or misuse of this code.  It is
 freeware, with source code, and so you use (and
 maintain) it at your own risk.
 Please forward any comments or suggestions to: imcasys@hotmail.com
 }
interface
{$I DFS.inc}
uses Windows, Messages, SysUtils,Classes;

type
  TTreeCollection = class;
  TTreeCollectionSortCompare = function (Item1, Item2: TTreeCollection): Integer;
  ETreeCollectionError = class(Exception);
  TTreeCollection = class(TPersistent)
  private
    //from item
    FCollection: TTreeCollection;
    FID: Integer;
    FType : Integer;
    FCaption : String;

    FItems: TList;
    FUpdateCount: Integer;
    FNextID: Integer;
    function GetCount: Integer;
    procedure InsertItem(Item: TTreeCollection);
    procedure RemoveItem(Item: TTreeCollection);
    //from item
    function GetIndex: Integer;
    procedure SetTreeCollection(Value: TTreeCollection);
    //new
    function GetAbsoluteIndex: Integer;
    function GetLevel: Integer;
    function GetParent: TTreeCollection;
    function GetChildren: Boolean;

  protected
    //from item
    procedure Changed(AllItems: Boolean);
    {$ifdef DFS_COMPILER_4}
    function GetOwner: TPersistent; override;
    {$endif}
    procedure SetIndex(Value: Integer); virtual;
    function GetItem(Index: Integer): TTreeCollection;
    procedure SetItem(Index: Integer; Value: TTreeCollection);
    //procedure SetItemName(Item: TTreeCollection); virtual;
    procedure Update(Item: TTreeCollection); virtual;
    procedure SetCaption(value : string);
    property UpdateCount: Integer read FUpdateCount;

    //read & write procedures
    procedure DefineProperties(Filer: TFiler); override;
    procedure WriteData(Writer: TWriter); virtual;
    procedure ReadData(Reader : TReader); virtual;
    function GetString(var S: PChar): String;
    function  ReadItemStrings(var S: PChar; CurrentLevel: Integer): Integer;
  public
    //new
    function GetFirstChild: TTreeCollection;
    function GetLastChild: TTreeCollection;
    function GetNext: TTreeCollection;
    function getNextSibling: TTreeCollection;
    function GetPrev: TTreeCollection;
    function getPrevSibling: TTreeCollection;

    procedure Sort(Compare: TTreeCollectionSortCompare; Recursive : boolean);
    procedure CustomSort(SortProc: TTreeCollectionSortCompare; Recursive : boolean);
    procedure AlphaSort(Recursive : boolean);
    procedure Exchange(Index1, index2 : integer);
    procedure LoadFromFile(const FileName: string);
    procedure LoadFromStream(Stream: TStream);
    procedure SaveToFile(const FileName: string);
    procedure SaveToStream(Stream: TStream);
    procedure ReadStrings(Stream: TStream);
    procedure WriteStrings(Stream: TStream);
    //from item
    //function CustomSort(SortProc: TTVCompare; Data: Longint): Boolean;
    property TreeCollection: TTreeCollection read FCollection write SetTreeCollection;
    property ID: Integer read FID;
    property TypeCol : integer Read Ftype write Ftype;//for different types of Collections
    property Caption : string read Fcaption write SetCaption;
    property Index: Integer read GetIndex write SetIndex;
    constructor Create(aOwner: TTreeCollection);
    destructor Destroy; override;
    function Add(aCaption : string): TTreeCollection;
    procedure DeleteItem(Item : TTreeCollection);
    procedure Assign(Source: TPersistent); override;
    procedure BeginUpdate; virtual;
    procedure Clear;
    procedure EndUpdate; virtual;
    function FindItemID(ID: Integer): TTreeCollection;
    function Insert(Index: Integer): TTreeCollection;
    property Count: Integer read GetCount;
    property Level : integer read GetLevel;
    property Items[Index: Integer]: TTreeCollection read GetItem write SetItem; default;
  end;


implementation

uses Consts,ComStrs, TypInfo;

{$ifdef DFS_COMPILER_3_UP}

  procedure TreeCollectionError(Msg: String);

  begin
    raise ETreeCollectionError.Create(Msg);
  end;

{$else}

  procedure TreeCollectionError(MsgID: Integer);

  begin
    raise ETreeCollectionError.CreateRes(MsgID);
  end;

{$endif}

procedure QuickSort(SortList: TList; L, R: Integer; SCompare: TTreeCollectionSortCompare);

var I, J: Integer;
    P, T: Pointer;

begin
  repeat
    I:=L;
    J:=R;
    P:=SortList[(L+R) shr 1];
    repeat
      while SCompare(TTreeCollection(SortList[I]),TTreeCollection(P)) < 0 do Inc(I);
      while SCompare(TTreeCollection(SortList[J]),TTreeCollection(P)) > 0 do Dec(J);
      if I <= J then
      begin
        T:=SortList[I];
        SortList[I]:=SortList[J];
        SortList[J]:=T;
        Inc(I);
        Dec(J);
      end;
    until I > J;
    if L < J then QuickSort(SortList,L,J,SCompare);
    L := I;
  until I >= R;


end;

function DefaultSort(Item1, Item2: TTreeCollection): Integer; 
begin
    Result:=lstrcmp(PChar(item1.Caption),PChar(item2.Caption));
end;

constructor TTreeCollection.Create(aOwner: TTreeCollection);
begin
  FItems := TList.Create;
  fCaption:='TreeCollection';
  if aOwner<> nil then SetTreeCollection(aOwner);//treecollitem
end;

destructor TTreeCollection.Destroy;
begin
  FUpdateCount := 1;
  if FItems <> nil then Clear;
  FItems.Free;
  SetTreeCollection(nil);
  inherited Destroy;
end;

function TTreeCollection.Add(aCaption : string): TTreeCollection;
begin
  Result := TTreeCollection.Create(Self);
  Result.FCaption:=aCaption;
end;

procedure TTreeCollection.DeleteItem(Item : TTreeCollection);
begin
   BeginUpdate;
   RemoveItem(Item);
   Item.Clear;
   Item.Free;
   EndUpdate;
end;

procedure TTreeCollection.Assign(Source: TPersistent);
var
  I: Integer;
begin
  if Source is TTreeCollection then
  begin
    BeginUpdate;
    try
      Fcaption:=TTreeCollection(Source).Fcaption;
      Ftype:=TTreeCollection(Source).Ftype;
      Clear;
      for I := 0 to TTreeCollection(Source).Count - 1 do
        Add('').Assign(TTreeCollection(Source).Items[I]);
    finally
      EndUpdate;
    end;
    Exit;
  end;
  inherited Assign(Source);
end;

procedure TTreeCollection.BeginUpdate;
begin
  Inc(FUpdateCount);
end;

procedure TTreeCollection.Changed(AllItems: Boolean);
var
  Item: TTreeCollection;
begin
  //update owner collection
  if (FCollection <> nil) and (FCollection.FUpdateCount = 0) then
  begin
    if AllItems then Item := nil else Item := Self;
    FCollection.Update(Item);
  end;
end;

procedure TTreeCollection.Clear;
begin
  if FItems.Count > 0 then
  begin
    BeginUpdate;
    try
      while FItems.Count > 0 do TTreeCollection(FItems.Last).Free;
    finally
      EndUpdate;
    end;
  end;
end;

procedure TTreeCollection.EndUpdate;
begin
  Dec(FUpdateCount);
  Changed(false);
end;

function TTreeCollection.FindItemID(ID: Integer): TTreeCollection;
var
  I: Integer;
begin
  for I := 0 to FItems.Count-1 do
  begin
    Result := TTreeCollection(FItems[I]);
    if Result.ID = ID then Exit;
  end;
  Result := nil;
end;

function TTreeCollection.GetCount: Integer;
begin
  Result := FItems.Count;
end;

function TTreeCollection.GetItem(Index: Integer): TTreeCollection;
begin
  Result := FItems[Index];
end;



function TTreeCollection.Insert(Index: Integer): TTreeCollection;
begin
  Result := Add('');
  Result.Index := Index;
end;


procedure TTreeCollection.InsertItem(Item: TTreeCollection);
begin

  FItems.Add(Item);
  Item.FCollection := Self;
  Item.FID := FNextID;
  Inc(FNextID);
  //SetItemName(Item);
  Changed(false);

end;

procedure TTreeCollection.RemoveItem(Item: TTreeCollection);
begin
  FItems.Remove(Item);
  Item.FCollection := nil;
  Changed(false);
end;

function TTreeCollection.GetIndex: Integer;
begin
  if FCollection <> nil then
    Result := FCollection.FItems.IndexOf(Self)
  else
    Result := -1;
end;

procedure TTreeCollection.SetItem(Index: Integer; Value: TTreeCollection);
begin
  TTreeCollection(FItems[Index]).Assign(Value);
end;


procedure TTreeCollection.Update(Item: TTreeCollection);
begin
end;

{$ifdef DFS_COMPILER_4}
function TTreeCollection.GetOwner: TPersistent;
begin
  Result := FCollection;
end;
{$endif}

procedure TTreeCollection.SetTreeCollection(Value: TTreeCollection);
begin
  if FCollection <> Value then
  begin
    if FCollection <> nil then FCollection.RemoveItem(Self);
    if Value <> nil then Value.InsertItem(Self);
  end;
end;

procedure TTreeCollection.Exchange(Index1, index2 : integer);
begin
   if (Index1>=0) and (index2>=0) and (index1 < Count) and (index2 < Count) then
   begin
       FItems.Exchange(Index1, Index2);
       Changed(True);
   end;
end;

procedure TTreeCollection.SetIndex(Value: Integer);
var
  CurIndex: Integer;
begin
  CurIndex := GetIndex;
  if (CurIndex >= 0) and (CurIndex <> Value) then
  begin
    FCollection.FItems.Move(CurIndex, Value);
    Changed(True);
  end;
end;

//new
function TTreeCollection.GetAbsoluteIndex: Integer;
var
  Collection: TTreeCollection;
begin
    Result := -1;
    Collection := Self;
    while Collection <> nil do
    begin
      Inc(Result);
      Collection := Collection.GetPrev;
    end;
end;

function TTreeCollection.GetLevel: Integer;
var
  Collection: TTreeCollection;
begin
  Result := 0;
  Collection := FCollection;
  while Collection <> nil do
  begin
    Inc(Result);
    Collection := Collection.FCollection;
  end;

end;

function TTreeCollection.GetParent: TTreeCollection;
begin
  Result := FCollection;
end;

function TTreeCollection.GetChildren: Boolean;
begin
  Result := FItems.Count > 0;
end;

function TTreeCollection.GetFirstChild: TTreeCollection;
begin
  if Fitems.Count>0 then
    Result := items[0]
  else
    Result:=nil;
end;

function TTreeCollection.GetLastChild: TTreeCollection;
begin
  if Fitems.Count >0 then
    Result := items[Fitems.Count-1]
  else
    Result:=nil;
end;


function TTreeCollection.GetNext: TTreeCollection;
var
   ParentCollection, Collection : TTreeCollection;
begin
   Collection:=GetFirstChild;
   if Collection = nil then
      Collection:=GetNextSibling;
   ParentCollection:=FCollection;//parent
   while (Collection=nil) and (ParentCollection<>nil) do
   begin
       ParentCollection:=ParentCollection.Fcollection;
       Collection:=ParentCollection.GetNextSibling;
   end;
   Result:=Collection;
end;

function TTreeCollection.GetPrev: TTreeCollection;
var Collection : TTreeCollection;
begin
   Result:=GetPrevSibling;
   if Result <> nil then
   begin
    Collection := Result;
    repeat
      Result := Collection;
      Collection := Result.GetLastChild;
    until Collection = nil;
  end
  else
    Result := FCollection;

end;

function TTreeCollection.getNextSibling: TTreeCollection;
begin
   Result:=FCollection.items[index+1];
end;

function TTreeCollection.getPrevSibling: TTreeCollection;
var aindex : integer;
begin
   aindex:=GetIndex;
   if aindex<>0 then
      Result:=FCollection.items[aindex-1]
   else
      Result:=nil;
end;
procedure TTreeCollection.SetCaption(value : string);
begin
   Fcaption:=Value;
end;

procedure TTreeCollection.ReadData(Reader : TReader); 
var item : TTreeCollection;
begin
  Reader.ReadListBegin;
  BeginUpdate;
  try
    if not Reader.EndOfList then Clear;
    Fcaption :=Reader.readString;
    Ftype:=Reader.ReadInteger;
    while not Reader.EndOfList do
    begin
      Item := Add('');
      while not Reader.EndOfList do Item.ReadData(Reader);
    end;
    Reader.ReadListEnd;
  finally
    EndUpdate;
  end;
end;

procedure TTreeCollection.WriteData(Writer: TWriter);
var
   I : Integer;
begin
  Writer.WriteListBegin;
  Writer.WriteString(Fcaption);
  Writer.WriteInteger(FType);
  for I := 0 to Count - 1 do
       items[i].Writedata(Writer);
  Writer.WriteListEnd;
end;

procedure TTreeCollection.DefineProperties(Filer: TFiler);
begin
  Filer.DefineProperty('TreeCollection', ReadData, WriteData, Count>1);
end;

//------------------------------------------------------------------------------
procedure TTreeCollection.SaveToFile(const FileName: String);
var Stream : TStream;
begin
  Stream:=TFileStream.Create(FileName,fmCreate);
  try
    SaveToStream(Stream);
  finally
    Stream.Free;
  end;
end;
//------------------------------------------------------------------------------
procedure TTreeCollection.SaveToStream(Stream: TStream);
begin
    WriteStrings(Stream);
end;

procedure TTreeCollection.WriteStrings(Stream: TStream);
var I : Integer;
    Buffer : String;
    Level : integer;
begin
  Level:= GetLevel;
  if Level > 0 then//for filling tab chars
  begin
    Buffer:='';
    for I:=0 to Level-1 do Buffer:=Buffer+#9;
    Stream.Write(PChar(Buffer)^,Length(Buffer));
  end;

  //Stream.Write(PChar(FCaption)^,Length(FCaption));
    Stream.Write(PChar(FCaption)^,Length(FCaption));
  Buffer:=#13#10;
  Stream.Write(PChar(Buffer)^,2);

  for I:=0 to Count-1 do Items[I].WriteStrings(Stream);
end;

//------------------------------------------------------------------------------
procedure TTreeCollection.LoadFromFile(const FileName: String);
var Stream : TStream;

begin
  Stream:=TFileStream.Create(FileName, fmOpenRead);
  try
    LoadFromStream(Stream);
  finally
    Stream.Free;
  end;
end;

//------------------------------------------------------------------------------

procedure TTreeCollection.LoadFromStream(Stream: TStream);
begin
    BeginUpdate;
    Clear;
    ReadStrings(Stream);
    EndUpdate;
end;

procedure TTreeCollection.ReadStrings(Stream: TStream);

var Size,newlevel   : Integer;
    Start,
    S      : PChar;

begin
  Size:=Stream.Size-Stream.Position;
  if Size > 0 then
  begin
    S:=AllocMem(Size+1);
    try
      Stream.Read(S^,Size);
      Start:=S;
      newlevel:=0;
      fCaption:=GetString(Start);
      while (Start^ <> #0) and (Start^ in [' ',#9]) do
      begin
        Inc(Start);
        Inc(NewLevel);
      end;
      ReaditemStrings(Start,newlevel);
    finally
      FreeMem(S);
    end;
  end;
end;

function TTreeCollection.GetString(var S: PChar): String;
var Start : PChar;
begin
    Start:=S;
    while not (S^ in [#0, #10, #13]) do Inc(S);
    SetString(Result,Start,S-Start);
    while (S^ <> #0) and (S^ in [#13,#10]) do Inc(S);
end;

//------------------------------------------------------------------------------
function TTreeCollection.ReadItemStrings(var S: PChar; CurrentLevel: Integer): Integer;
 var NewLevel   : Integer;
    NewText    : String;
    NewChild   : TTreeCollection;
begin
  NewChild:=nil;
  repeat
    NewLevel:=0;
    NewText:=GetString(S);
    if Length(NewText) > 0 then
    begin
       NewChild:=Add(newText);
    end
    else
       Break;
    while (S^ <> #0) and (S^ in [' ',#9]) do
    begin
      Inc(S);
      Inc(NewLevel);
    end;
    if NewLevel-CurrentLevel > 1 then
       TreeCollectionError(sInvalidLevel);
    if (NewLevel > CurrentLevel) and assigned(NewChild) then
       NewLevel:=NewChild.ReadItemStrings(S,NewLevel);
  until NewLevel < CurrentLevel;
  Result:=NewLevel;
 end;
//------------------------------------------------------------------------------

procedure TTreeCollection.Sort(Compare: TTreeCollectionSortCompare; Recursive : boolean);
var i : integer;
begin
  BeginUpdate;
  try
    if (FItems <> nil) and (Count > 0) then QuickSort(FItems,0,Count-1,Compare);
    if Recursive then
      for i:=0 to Count-1 do items[i].Sort(Compare, Recursive);
  finally
    EndUpdate;
  end;
end;

procedure TTreeCollection.CustomSort(SortProc: TTreeCollectionSortCompare; Recursive : boolean);
begin
  if not assigned(SortProc) then
      Sort(@DefaultSort,Recursive)
  else
      Sort(@SortProc,Recursive);
end;

procedure TTreeCollection.AlphaSort(Recursive : boolean);
begin
   CustomSort(nil,Recursive);
end;


end.
