{*********************************************************************}
{                                                                     }
{       TKeyboard component                                           }
{                                                                     }
{       Copyright (c) 1997 Andrea Carli                               }
{       Portions Copyright (c) 1995,96 Borland International          }
{                                                                     }
{ Relevant design-runtime properties:   KeyCaptions: TStrings;        }
{                                       KeyIndex: TKeyboardBtn; [R/O] }
{                                       NullValue: String;            }
{                                                                     }
{ Relevant runtime properties:          Value: String;          [R/O] }
{                                                                     }
{ Relevant runtime methods:                                           }
{                        Enable(Index: TKeyboardBtn; Value: Boolean); }
{                          Down(Index: TKeyboardBtn; Value: Boolean); }
{                                                                     }
{ KeyIndex as KeyCaptions.Strings[index] start from 0.                }
{                                                                     }
{ Limitations: Max 30 keys (TKeyboardBtn goes from 0 to 29) may be    }
{ increased changing the source code and recompiling the component.   }
{ Change the 'MaxKeys' constant.                                      }
{                                                                     }
{ This component display a keyboard in wich each key had a caption    }
{ given by the 'KeyCaptions' property; setting this property make the }
{ corresponding button visible and enabled.                           }
{ Only a key remain pressed at time or none at all, in the first      }
{ case the 'Value' property (runtime) of the component is the caption }
{ of the pressed key; otherwise the 'NullValue' property is returned. }
{ 'KeyIndex' property (runtime) represent the current button selected }
{ (not necessarily pressed!).                                         }
{ The method 'Enable' Enable/Disable the button specified             }
{ The method 'Down'   Push/Pull the button specified                  }
{ The component could be expanded for the use of: Glyphs, Hints, ecc. }
{ but i wrote this component only to have a Keyboard to simplify and  }
{ speed up user input in some applications.                           }
{                                                                     }
{ I would be pleased to receive all modifications of this source code }
{ expecially those who mantain the purpose I have create it for (look }
{ the example acclosed).                                              }
{                                                                     }
{                                                                     }
{ For information, BUG REPORT and suggestions (very appreciated)      }
{ e-mail me at:                                                       }
{                       a.carli@leonet.it                             }
{                                                                     }
{                                                                     }
{ I DECIDED TO MAKE THIS COMPONENT FREEWARE SINCE I BELIEVE IN THIS   }
{ KIND OF SOFTWARE DISTRIBUTION, I HAVE SOLVE LOT OF PROBLEMS, BUGS   }
{ AND LEARN MORE ABOUT THIS LANGUAGE THANKS TO SOURCE LISTINGS FOUNDED}
{ IN FREEWARE WAY.                                                    }
{ LET'S GIVE OUR CONTRIBUTION TO THIS CAUSE!                          }
{                                                                     }
{ SINCE I AM NOT A COMPONENT OR LOW LEVEL PROGRAMMER EXPERT I DO NOT  }
{ GRANT YOU THAT THIS COMPONENT WILL FUNCTION AS DESCRIBED OR MAY NOT }
{ CAUSE DAMAGES OF ANY KIND, IT'S THE FIRST COMPONENT I WROTE!        }
{                                                                     }
{ I PLEASE EVERY ONE WHO USE THIS SOURCE CODE TO MENTION MY NAME AND  }
{ BORLAND COPYRIGHT IN THEIR MODIFICATIONS.                           }
{                                                                     }
{*********************************************************************}

unit Keyboard;

{$R-}

interface

uses SysUtils, Windows, Messages, Classes, Controls, Forms, Graphics, Menus,
     StdCtrls, ExtCtrls, Mask, Buttons;

const
  MaxKeys = 29;

type
  TKeyButton = class;

  TKeyboardBtn = 0..MaxKeys;

  TButtonSet = set of TKeyboardBtn;

  TKeyArray = array[TKeyboardBtn] of TKeyButton;

  EKeyClick = procedure (Sender: TObject; Button: TKeyboardBtn) of object;

{ TKeyboard }

  TKeyboard = class (TCustomPanel)
  private
    FCaptions: TStrings;
    FCapTmp: TStrings;
    ButtonWidth: Integer;
    MinBtnSize: TPoint;
    FOnKeyClick: EKeyClick;
    FFocusedButton: TKeyboardBtn;
    FValue: String;
    FNullValue: String;
    procedure InitButtons;
    procedure InitCaptions;
    procedure Click(Sender: TObject);
    procedure BtnMouseDown (Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure AdjustSize (var W: Integer; var H: Integer);
    procedure SetCaptions(Value: TStrings);
    procedure SetNullValue(Value: String);
    procedure WMSize(var Message: TWMSize);  message WM_SIZE;
    procedure WMSetFocus(var Message: TWMSetFocus); message WM_SETFOCUS;
    procedure WMKillFocus(var Message: TWMKillFocus); message WM_KILLFOCUS;
    procedure WMGetDlgCode(var Message: TWMGetDlgCode); message WM_GETDLGCODE;
  protected
    Buttons: TKeyArray;
    procedure Loaded; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure GetChildren(Proc: TGetChildProc); override;
  public
    property Value: String read FValue;      { Value returned by the control }

    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
    procedure BtnClick(Index: TKeyboardBtn);
    procedure Enable(Index: TKeyboardBtn; Value: Boolean);
    procedure Down(Index: TKeyboardBtn; Value: Boolean);
  published
    property KeyCaptions: TStrings read FCaptions write SetCaptions;
    property KeyIndex: TKeyboardBtn read FFocusedButton write FFocusedButton;
    property NullValue: String read FNullValue write SetNullValue;
    property Align;
    property Ctl3D;
    property DragCursor;
    property DragMode;
    property Enabled;
    property ParentCtl3D;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnClick: EKeyClick read FOnKeyClick write FOnKeyClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnResize;
    property OnStartDrag;
  end;

{ TKeyButton }

{ TKeyButton }

  TKeyButton = class(TSpeedButton)
  private
    FIndex: TKeyboardBtn;
  protected
    procedure Paint; override;
  public
    destructor Destroy; override;
    property Index : TKeyboardBtn read FIndex write FIndex;
  end;

procedure Register;

implementation

uses Clipbrd, Dialogs;


procedure Register;
begin
  RegisterComponents('Freeware',[TKeyboard]);
end;


{ TKeyboard }

constructor TKeyboard.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle - [csAcceptsControls, csSetCaption] + [csOpaque];
  if not NewStyleControls then ControlStyle := ControlStyle + [csFramed];

  FCaptions := TStringList.Create;
  FCapTmp := TStringList.Create;
  FValue := '';
  FNullValue := '';
  InitButtons;
  BevelOuter := bvNone;
  BevelInner := bvNone;
  Width := 20;
  Height := 20;
  ButtonWidth := 0;
  FFocusedButton := 0;
end;

destructor TKeyboard.Destroy;
begin
  FCaptions.Free;
  FCapTmp.Free;
  inherited Destroy;
end;

procedure TKeyboard.InitButtons;
var
  I: TKeyboardBtn;
  Btn: TKeyButton;
  X: Integer;
begin
  MinBtnSize := Point(20, 20);
  X := 0;
  for I := Low(Buttons) to High(Buttons) do
  begin
    Btn := TKeyButton.Create (Self);
    Btn.AllowAllUp := True;
    Btn.GroupIndex := 1;
    Btn.Index := I;
    Btn.Visible := False;
    Btn.Enabled := True;
    Btn.SetBounds (X, 0, MinBtnSize.X, MinBtnSize.Y);
    Btn.OnClick := Click;
    Btn.OnMouseDown := BtnMouseDown;
    Btn.Parent := Self;
    Buttons[I] := Btn;
    X := X + MinBtnSize.X;
  end;
  InitCaptions;
end;

procedure TKeyboard.InitCaptions;
var
  I, Y, Z: Integer;
  J: TKeyboardBtn;
  W, H: Integer;

begin
  W := Width;
  H := Height;

  for J := Low(Buttons) to High(Buttons) do begin
    Buttons[J].Caption := '';
    Buttons[J].Visible := False;
  end;

  J := Low(Buttons);

  for I := 0 to (FCaptions.Count - 1) do
  begin
    if FCaptions.Strings[I] <> '' then begin
       Buttons[J].Caption := FCaptions.Strings[I];
       Buttons[J].Visible := True;
    end;

    if J = High(Buttons) then begin
{      for Y := I+1 to (FCaptions.Count - 1) do          old - see SetCaptions
        FCaptions.Strings[Y] := '';
}
      Break;
    end
    else
      Inc(J);
  end;

  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds (Left, Top, W, H);
  Invalidate;
end;

procedure TKeyboard.SetCaptions(Value: TStrings);
var I: Integer;
begin
  FCaptions.Clear;
  FCapTmp.Assign(Value);
  if (FCapTmp.Count-1) > MaxKeys then
    for I := 0 to MaxKeys do begin
      if FCapTmp.Strings[I] <> '' then FCaptions.Add(FCapTmp.Strings[I])
    end
  else
    for I := 0 to (FCapTmp.Count-1) do begin
      if FCapTmp.Strings[I] <> '' then FCaptions.Add(FCapTmp.Strings[I]);
    end;

//    FCaptions.Assign(Value);  old
  InitCaptions;
end;

procedure TKeyboard.SetNullValue(Value: String);
begin
  if FNullValue <> Value then
  begin
    FNullValue := Value;

    if FValue = '' then
       FValue := FNullValue;
  end;
end;

procedure TKeyboard.GetChildren(Proc: TGetChildProc);
begin
end;

procedure TKeyboard.AdjustSize (var W: Integer; var H: Integer);
var
  Count: Integer;
  MinW: Integer;
  I: TKeyboardBtn;
  Space, Temp, Remain: Integer;
  X: Integer;
begin
  if (csLoading in ComponentState) then Exit;
  if Buttons[0] = nil then Exit;

  Count := 0;
  for I := Low(Buttons) to High(Buttons) do
  begin
    if Buttons[I].Visible then
    begin
      Inc(Count);
    end;
  end;
  if Count = 0 then Inc(Count);

  MinW := Count * MinBtnSize.X;
  if W < MinW then W := MinW;
  if H < MinBtnSize.Y then H := MinBtnSize.Y;

  ButtonWidth := W div Count;
  Temp := Count * ButtonWidth;
  if Align = alNone then W := Temp;

  X := 0;
  Remain := W - Temp;
  Temp := Count div 2;
  for I := Low(Buttons) to High(Buttons) do
  begin
    if Buttons[I].Visible then
    begin
      Space := 0;
      if Remain <> 0 then
      begin
        Dec(Temp, Remain);
        if Temp < 0 then
        begin
          Inc(Temp, Count);
          Space := 1;
        end;
      end;
      Buttons[I].SetBounds(X, 0, ButtonWidth + Space, Height);
      Inc(X, ButtonWidth + Space);
    end
    else
      Buttons[I].SetBounds (Width + 1, 0, ButtonWidth, Height);
  end;
end;

procedure TKeyboard.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
var
  W, H: Integer;
begin
  W := AWidth;
  H := AHeight;
  AdjustSize (W, H);
  inherited SetBounds (ALeft, ATop, W, H);
end;

procedure TKeyboard.WMSize(var Message: TWMSize);
var
  W, H: Integer;
begin
  inherited;

  { check for minimum size }
  W := Width;
  H := Height;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds(Left, Top, W, H);
  Message.Result := 0;
end;

procedure TKeyboard.Click(Sender: TObject);
begin
  BtnClick (TKeyButton (Sender).Index);
end;

procedure TKeyboard.BtnMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  OldFocus: TKeyboardBtn;
begin
  OldFocus := FFocusedButton;
  FFocusedButton := TKeyButton (Sender).Index;
  if TabStop and (GetFocus <> Handle) and CanFocus then
  begin
    SetFocus;
    if (GetFocus <> Handle) then
      Exit;
  end
  else if TabStop and (GetFocus = Handle) and (OldFocus <> FFocusedButton) then
  begin
    Buttons[OldFocus].Invalidate;
    Buttons[FFocusedButton].Invalidate;
  end;
end;

procedure TKeyboard.BtnClick(Index: TKeyboardBtn);
begin
  if Buttons[Index].Down then
    FValue := Buttons[Index].Caption
  else
    FValue := FNullValue;

  if not (csDesigning in ComponentState) and Assigned(FOnKeyClick) then
    FOnKeyClick(Self, Index);
end;

procedure TKeyboard.Enable(Index: TKeyboardBtn; Value: Boolean);
begin
  if (not Value) and (Buttons[Index].Down) then
    Down(Index,False);

  Buttons[Index].Enabled := Value;
end;

procedure TKeyboard.Down(Index: TKeyboardBtn; Value: Boolean);
begin
  if (Buttons[Index].Down <> Value) and (Buttons[Index].Enabled) then begin

    Buttons[Index].Down := Value;
    FFocusedButton := Index;

    if Value then
      FValue := Buttons[Index].Caption
    else
      FValue := FNullValue;

  end;
end;



procedure TKeyboard.WMSetFocus(var Message: TWMSetFocus);
begin
  Buttons[FFocusedButton].Invalidate;
end;

procedure TKeyboard.WMKillFocus(var Message: TWMKillFocus);
begin
  Buttons[FFocusedButton].Invalidate;
end;

procedure TKeyboard.KeyDown(var Key: Word; Shift: TShiftState);
var
  NewFocus: TKeyboardBtn;
  OldFocus: TKeyboardBtn;
begin
  OldFocus := FFocusedButton;
  case Key of
    VK_RIGHT:
      begin
        NewFocus := FFocusedButton;

        if NewFocus < High(Buttons) then
          NewFocus := Succ(NewFocus);

        if not Buttons[NewFocus].Visible then
          NewFocus := Pred(NewFocus);

        if NewFocus <> FFocusedButton then
        begin
          FFocusedButton := NewFocus;
          Buttons[OldFocus].Invalidate;
          Buttons[FFocusedButton].Invalidate;
        end;
      end;
    VK_LEFT:
      begin
        NewFocus := FFocusedButton;

        if NewFocus > Low(Buttons) then
          NewFocus := Pred(NewFocus);

        if not Buttons[NewFocus].Visible then
          NewFocus := Succ(NewFocus);

        if NewFocus <> FFocusedButton then
        begin
          FFocusedButton := NewFocus;
          Buttons[OldFocus].Invalidate;
          Buttons[FFocusedButton].Invalidate;
        end;
      end;
    VK_SPACE:
      begin
        if Buttons[FFocusedButton].Enabled then

          if not Buttons[FFocusedButton].Down then       { Simulate Mouse Click }
            Buttons[FFocusedButton].Down := True
          else
            Buttons[FFocusedButton].Down := False;

          Buttons[FFocusedButton].Click;
          Buttons[FFocusedButton].Invalidate;
      end;
  end;
  Invalidate;
end;

procedure TKeyboard.WMGetDlgCode(var Message: TWMGetDlgCode);
begin
  Message.Result := DLGC_WANTARROWS;
end;

procedure TKeyboard.Loaded;
var
  W, H: Integer;
begin
  inherited Loaded;
  W := Width;
  H := Height;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds (Left, Top, W, H);
  InitCaptions;
end;



{TKeyButton}

destructor TKeyButton.Destroy;
begin
  inherited Destroy;
end;

procedure TKeyButton.Paint;
var
  R: TRect;
begin
  inherited Paint;
  if (GetFocus = Parent.Handle) and
     (FIndex = TKeyboard (Parent).FFocusedButton) then
  begin
    R := Bounds(0, 0, Width, Height);

    { Set the brush color otherwise it does now draw at all! }
    Canvas.Brush.Color := clBtnHighLight;

    InflateRect(R, -3, -3);
    if FState = bsDown then
      OffsetRect(R, 1, 1);
    DrawFocusRect(Canvas.Handle, R);
  end;
end;


end.
