unit TimePanel;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls;

type

  { Exception for TTimeBoxControl Create method }

  ENotATimePanel = class(Exception);

  { Custom types used by TTimePanel } 

  TTPDays = 1..2;
  TTPHours = 0..23;
  TTPDoubleHours = 0..47;
  TTPMins = 0..59;
  TTPArrowDirection = (adUp, adDown);
  TTPScrollDirection = (sdUp, sdDown, sdNotScrolling);
  TTPStartFinish = (sfStart, sfFinish);
  TTPTimeControlType = (ttStart, ttFinish, ttDuration);

  { TTimePanel forward declaration }

  TTimePanel = class;

  { TTimeBoxControl }

  TTimeBoxControl = class(TGraphicControl)
  protected
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    property OnMouseMove;
  end;

  { TTimePanel }

  TTimePanel = class(TCustomPanel)
  private
    { Private declarations }
    FBottomArrowRect: TRect;
    FBottomDay: TTPDays;
    FBottomHour: TTPHours;
    FDurationColor: TColor;
    FDurationControl: TTimeBoxControl;
    FDurationFormat: String;
    FDurationRect: TRect;
    FDurationVisible: Boolean;
    FFinishColor: TColor;
    FFinishControl: TTimeBoxControl;
    FFinishDay: TTPDays;
    FFinishHours: TTPHours;
    FFinishMins: TTPMins;
    FFinishRect: TRect;
    FFinishVisible: Boolean;
    FTextHeight: Integer;
    FTimeColor: TColor;
    FTimeFormat: String;
    FTopArrowRect: TRect;
    FTopDay: TTPDays;
    FTopHour: TTPHours;
    FTopMin: TTPMins;
    FScrollAmount: Single;
    FScrolling: TTPScrollDirection;
    FSelColor: TColor;
    FShowConnect: Boolean;
    FShowDays: TTPDays;
    FShowDuration: Boolean;
    FStartColor: TColor;
    FStartControl: TTimeBoxControl;
    FStartHours: TTPHours;
    FStartMins: TTPMins;
    FStartRect: TRect;
    FStartVisible: Boolean;
    FWorkFinish: TTPHours;
    FWorkStart: TTPHours;
    FWorkTimeColor: TColor;
    FOnTimeChange: TNotifyEvent;
    function GetDurationHours: TTPHours;
    function GetDurationMins: TTPMins;
    function GetFinishTime: TDateTime;
    function GetStartTime: TDateTime;
    function IsInRect(ARect: TRect; X, Y: Integer): Boolean;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer);
      override;
    procedure StartMouseMove(Sender: TObject; Shift: TShiftState;
      X, Y: Integer);
    procedure FinishMouseMove(Sender: TObject; Shift: TShiftState;
      X, Y: Integer);
    procedure DurationMouseMove(Sender: TObject; Shift: TShiftState;
      X, Y: Integer);
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
      override;
    procedure SetDurationColor(Value: TColor);
    procedure SetDurationFormat(Value: String);
    procedure SetDurationHours(Value: TTPHours);
    procedure SetDurationMins(Value: TTPMins);
    procedure SetFinishColor(Value: TColor);
    procedure SetFinishDay(Value: TTPDays);
    procedure SetFinishHours(Value: TTPHours);
    procedure SetFinishMins(Value: TTPMins);
    procedure SetFinishTime(Value: TDateTime);
    procedure SetSelColor(Value: TColor);
    procedure SetShowConnect(Value: Boolean);
    procedure SetShowDays(Value: TTPDays);
    procedure SetShowDuration(Value: Boolean);
    procedure SetStartColor(Value: TColor);
    procedure SetStartHours(Value: TTPHours);
    procedure SetStartMins(Value: TTPMins);
    procedure SetStartTime(Value: TDateTime);
    procedure SetTimeColor(Value: TColor);
    procedure SetTimeFormat(Value: String);
    procedure SetTopDay(Value: TTPDays);
    procedure SetTopHour(Value: TTPHours);
    procedure SetTopMin(Value: TTPMins);
    procedure SetWorkFinish(Value: TTPHours);
    procedure SetWorkStart(Value: TTPHours);
    procedure SetWorkTimeColor(Value: TColor);
    procedure Scroll(Dir: TTPScrollDirection; Hours: TTPHours; Mins: TTPMins);
    procedure DoTimeChange;
  protected
    { Protected declarations }
    procedure CalcRange;
    procedure DrawArrow(C: TCanvas; Dir: TTPArrowDirection; AColor: TColor);
      virtual;
    procedure DrawConnect(C: TCanvas); virtual;
    procedure DrawFrame(C: TCanvas); virtual;
    function DrawTimeBox(C: TCanvas; Day: TTPDays; Hours: TTPHours;
      Mins: TTPMins; AColor: TColor; BoxText: String): TRect; virtual;
    procedure DrawTimeBoxes(C: TCanvas); virtual;
    procedure DrawTimeNumbers(C: TCanvas); virtual;
    procedure Paint; override;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure IncTime(StartFinish: TTPStartFinish; Hours: TTPHours;
      Mins: TTPMins);
    procedure DecTime(StartFinish: TTPStartFinish; Hours: TTPHours;
      Mins: TTPMins);
    property FinishTime: TDateTime read GetFinishTime write SetFinishTime;
    property StartTime: TDateTime read GetStartTime write SetStartTime;
  published
    { Published declarations }
    property BevelInner default bvNone;
    property BevelOuter default bvRaised;
    property BevelWidth default 1;
    property BorderStyle default bsNone;
    property BorderWidth default 0;
    property Color default clBtnFace;
    property Cursor;
    property DurationColor: TColor read FDurationColor write SetDurationColor
      default clYellow;
    property DurationFormat: String read FDurationFormat
      write SetDurationFormat;
    property DurationHours: TTPHours read GetDurationHours
      write SetDurationHours stored False;
    property DurationMins: TTPMins read GetDurationMins write SetDurationMins
      stored False;
    property DurationVisible: Boolean read FDurationVisible;
    property FinishColor: TColor read FFinishColor write SetFinishColor
      default clRed;
    property FinishDay: TTPDays read FFinishDay write SetFinishDay default 1;
    property FinishHours: TTPHours read FFinishHours write SetFinishHours
      default 11;
    property FinishMins: TTPMins read FFinishMins write SetFinishMins
      default 30;
    property FinishVisible: Boolean read FFinishVisible;
    property Font;
    property ParentShowHint;
    property SelColor: TColor read FSelColor write SetSelColor default clTeal;
    property ShowConnect: Boolean read FShowConnect write SetShowConnect
      default True;
    property ShowDays: TTPDays read FShowDays write SetShowDays default 1;
    property ShowDuration: Boolean read FShowDuration write SetShowDuration
      default True;
    property ShowHint;
    property StartColor: TColor read FStartColor write SetStartColor
      default clLime;
    property StartHours: TTPHours read FStartHours write SetStartHours
      default 9;
    property StartMins: TTPMins read FStartMins write SetStartMins default 0;
    property StartVisible: Boolean read FStartVisible;
    property TabOrder;
    property TabStop default False;
    property TimeColor: TColor read FTimeColor write SetTimeColor
      default clGray;
    property TimeFormat: String read FTimeFormat write SetTimeFormat;
    property TopDay: TTPDays read FTopDay write SetTopDay default 1;
    property TopHour: TTPHours read FTopHour write SetTopHour default 7;
    property TopMin: TTPMins read FTopMin write SetTopMin default 0;
    property WorkFinish: TTPHours read FWorkFinish write SetWorkFinish
      default 17;
    property WorkStart: TTPHours read FWorkStart write SetWorkStart default 9;
    property WorkTimeColor: TColor read FWorkTimeColor write SetWorkTimeColor
      default clBlack;
    property OnTimeChange: TNotifyEvent read FOnTimeChange write FOnTimeChange;
  end;

procedure Register;

implementation

const
  ArrowHeight = 12;
  TimeNumMargin = 4;

{ TTimeBoxControl }

constructor TTimeBoxControl.Create(AOwner: TComponent);
begin
  if AOwner is TTimePanel then
  begin
    inherited Create(AOwner);
    ControlStyle := ControlStyle - [csOpaque];
    Visible := True;
  end
  else
    raise ENotATimePanel.Create('Parent is not a TTimePanel');
end;

{ TTimePanel }

constructor TTimePanel.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle + [csReplicatable];

  // Set defaults
  FTopDay := 1;
  FTopHour := 7;
  FTopMin := 0;
  FShowConnect := True;
  FShowDays := 1;
  FStartHours := 9;
  FStartMins := 0;
  FStartColor := clLime;
  FStartControl := TTimeBoxControl.Create(Self);
  FStartControl.Parent := Self;
  FStartControl.OnMouseMove := StartMouseMove;
  FFinishDay := 1;
  FFinishHours := 11;
  FFinishMins := 30;
  FFinishColor := clRed;
  FFinishControl := TTimeBoxControl.Create(Self);
  FFinishControl.Parent := Self;
  FFinishControl.OnMouseMove := FinishMouseMove;
  FTimeColor := clGray;
  FTimeFormat := 'HH:mm';
  FWorkTimeColor := clBlack;
  FWorkStart := 9;
  FWorkFinish := 17;
  FDurationColor := clYellow;
  FShowDuration := True;
  FDurationControl := TTimeBoxControl.Create(Self);
  FDurationControl.Parent := Self;
  FDurationControl.OnMouseMove := DurationMouseMove;
  FDurationFormat := '%dh %.2dm';
  FSelColor := clTeal;
  FScrolling := sdNotScrolling;

  // Set inherited properties defaults
  Caption := '';
  Width := 128;
  Height := 200;
  BevelInner := bvNone;
  BevelOuter := bvRaised;
  BevelWidth := 1;
  BorderStyle := bsNone;
  BorderWidth := 0;
  Color := clBtnFace;
  TabStop := False;
  Font.Name := 'MS Sans Serif';
  Font.Size := 8;
  Font.Style := [];
end;

destructor TTimePanel.Destroy;
begin
  FDurationControl.Free;
  FFinishControl.Free;
  FStartControl.Free;
  inherited Destroy;
end;

procedure TTimePanel.DrawArrow(C: TCanvas; Dir: TTPArrowDirection;
  AColor: TColor);
var
  Arrow: Array [1..3] of TPoint;
begin
  with C do
  begin
    Brush.Color := AColor;
    Brush.Style := bsSolid;
    Pen.Color := AColor;
    Pen.Style := psSolid;
  end;
  if Dir = adUp then
  begin
    Arrow[1] := Point(2, 8);
    Arrow[2] := Point(8, 2);
    Arrow[3] := Point(14, 8);
    with FTopArrowRect do
    begin
      Top := 2;
      Left := 2;
      Right := 14;
      Bottom := 8;
    end;
  end
  else
  begin
    if BorderStyle = bsSingle then
    begin
      Arrow[1] := Point(2, Height - 13);
      Arrow[2] := Point(8, Height - 7);
      Arrow[3] := Point(14, Height - 13);
      with FBottomArrowRect do
      begin
        Top := Height - 13;
        Left := 2;
        Bottom := Height - 7;
        Right := 14;
      end;
    end
    else
    begin
      Arrow[1] := Point(2, Height - 8);
      Arrow[2] := Point(8, Height - 2);
      Arrow[3] := Point(14, Height - 8);
      with FBottomArrowRect do
      begin
        Top := Height - 8;
        Left := 2;
        Bottom := Height - 2;
        Right := 14;
      end;
    end;
  end;
  C.Polygon(Arrow);
end;

procedure TTimePanel.DrawConnect(C: TCanvas);
const
  LineWidth = 3;
var
  RHours, RTopHours: TTPDoubleHours;
  LineTop, LineBottom, LineLeft: Integer;
begin
  CalcRange;
  RTopHours := (FTopHour + ((FTopDay - 1) * 24));
  // Calculate top of line
  with C do
  begin
    RHours := FStartHours;
    LineTop := ArrowHeight + ((RHours - RTopHours) * FTextHeight) -
     (Trunc(FTextHeight * (FTopMin / 60.0))) + (Trunc(FTextHeight *
     (FStartMins / 60.0))) + (TextHeight('88:88') div 2);
    RHours := FFinishHours + ((FFinishDay - 1) * 24);
    LineBottom := ArrowHeight + ((RHours - RTopHours) * FTextHeight) -
     (Trunc(FTextHeight * (FTopMin / 60.0))) + (Trunc(FTextHeight *
     (FFinishMins / 60.0))) + (TextHeight('88:88') div 2);
    LineLeft := Trunc(TextWidth('88:88') * 1.5) + 26;
    Pen.Color := clBlack;
    Pen.Style := psSolid;
    Pen.Width := LineWidth;
    MoveTo(LineLeft, LineTop);
    LineTo(LineLeft, LineBottom);
    Pen.Width := 1;
  end;
end;

function TTimePanel.DrawTimeBox(C: TCanvas; Day: TTPDays; Hours: TTPHours;
  Mins: TTPMins; AColor: TColor; BoxText: String): TRect;
const
  VMargin = 2;
  HMargin = 8;
var
  BoxRect: TRect;
  RHours, RTopHours: TTPDoubleHours;
begin
  CalcRange;
  RTopHours := (FTopHour + ((FTopDay - 1) * 24));
  RHours := (Hours + ((Day - 1) * 24));
  with C do
  begin
    Brush.Color := AColor;
    Brush.Style := bsSolid;
    Pen.Color := clBlack;
    Pen.Style := psSolid;
    BoxRect.Left := TextWidth('88:88') + 18;
    BoxRect.Right := BoxRect.Left + TextWidth(BoxText) + (2 * HMargin);
    BoxRect.Top := 12 + ((RHours - RTopHours) * FTextHeight) -
      (Trunc(FTextHeight * (FTopMin / 60.0))) + (Trunc(FTextHeight *
      (Mins / 60.0))) - VMargin;
    BoxRect.Bottom := BoxRect.Top + TextHeight(BoxText) + VMargin;
    with BoxRect do
      Rectangle(Left, Top, Right, Bottom);
    Brush.Style := bsClear;
    Font.Color := clBlack;
    TextOut(BoxRect.Left + HMargin, BoxRect.Top + (VMargin div 2), BoxText);
  end;
  Result := BoxRect;
end;

procedure TTimePanel.DrawTimeBoxes(C: TCanvas);
var
  STTPMins, FinMins, DMins: Integer;
  DurDay: TTPDays;
  DurHours: TTPHours;
  DurMins: TTPMins;
begin
  // Start time box
  FStartRect := DrawTimeBox(C, 1, FStartHours, FStartMins, FStartColor,
    FormatDateTime(FTimeFormat, StartTime));
  with FStartRect do
  begin
    if (Bottom > 0) and (Top < Height) then
      FStartVisible := True
    else
      FStartVisible := False;
    FStartControl.Top := Top;
    FStartControl.Left := Left;
    FStartControl.Height := (Bottom - Top);
    FStartControl.Width := (Right - Left);
  end;
  // Finish time box
  FFinishRect := DrawTimeBox(C, FFinishDay, FFinishHours, FFinishMins,
    FFinishColor, FormatDateTime(FTimeFormat, FinishTime));
  with FFinishRect do
  begin
    if (Bottom > 0) and (Top < Height) then
      FFinishVisible := True
    else
      FFinishVisible := False;
    FFinishControl.Top := Top;
    FFinishControl.Left := Left;
    FFinishControl.Height := (Bottom - Top);
    FFinishControl.Width := (Right - Left);
  end;
  // Duration time box
  if FShowDuration then
  begin
    STTPMins := ((FStartHours * 60) + FStartMins);
    FinMins := (((FFinishDay - 1) * (24 * 60)) + (FFinishHours * 60) +
      FFinishMins);
    DMins := STTPMins + ((FinMins - STTPMins) div 2);
    DurDay := (DMins div (24 * 60)) + 1;
    DurHours := (DMins div 60) mod 24;
    DurMins := DMins mod 60;
    FDurationRect := DrawTimeBox(C, DurDay, DurHours, DurMins,
      FDurationColor, Format(FDurationFormat, [GetDurationHours,
      GetDurationMins]));
    with FDurationRect do
    begin
      if (Bottom > 0) and (Top < Height) then
        FDurationVisible := True
      else
        FDurationVisible := False;
      FDurationControl.Top := Top;
      FDurationControl.Left := Left;
      FDurationControl.Height := (Bottom - Top);
      FDurationControl.Width := (Right - Left);
    end;
  end
  else
    FDurationVisible := False;
end;

procedure TTimePanel.DrawTimeNumbers(C: TCanvas);
var
  LineRect, TextClipRect: TRect;
  TextTop: Integer;
  HourNum, RTopHours, RBottomHours: TTPDoubleHours;
begin
  C.Font := Font;
  with C do
  begin
    Brush.Style := bsClear;
    with TextClipRect do
    begin
      Left := 0;
      Top := ArrowHeight;
      Right := TextWidth('88:88') + (2 * TimeNumMargin);
      if BorderStyle = bsSingle then
        Bottom := Height - ArrowHeight - 5
      else
        Bottom := Height - ArrowHeight;
    end;
    CalcRange;
    RTopHours := (FTopHour + ((FTopDay - 1) * 24));
    RBottomHours := (FBottomHour + ((FBottomDay - 1) * 24));
    for HourNum := RTopHours to RBottomHours do
    begin
      TextTop := TextClipRect.Top + ((HourNum - RTopHours) * FTextHeight) -
        (Trunc(FTextHeight * (FTopMin / 60.0)));
      if ((HourNum mod 24) >= FWorkStart) and
        ((HourNum mod 24) <= FWorkFinish) then
        Font.Color := FWorkTimeColor
      else
        Font.Color := FTimeColor;
      TextRect(TextClipRect, TimeNumMargin, TextTop, Format('%.2d:00',
        [HourNum mod 24]));
    end;
  end;
  LineRect := GetClientRect;
  LineRect.Left := C.TextWidth('88:88') + (2 * TimeNumMargin);
  LineRect.Right := LineRect.Left + 2;
  Frame3D(C, LineRect, clBtnShadow, clBtnHighlight, 1);
end;

procedure TTimePanel.CalcRange;
var
  Metric: TTextMetric;
  Margin, CalcBottom: Integer;
begin
  Margin := ArrowHeight * 2;
  if BorderStyle = bsSingle then
    Margin := Margin + 5;
  GetTextMetrics(Canvas.Handle, Metric);
  FTextHeight := Metric.tmHeight * 2;
  CalcBottom := (FTopHour + ((FTopDay - 1) * 24)) +
    ((Height - Margin) div FTextHeight) + 1;
  if CalcBottom > 47 then
  begin
    FBottomDay := FShowDays;
    FBottomHour := 23;
  end
  else
  begin
    if CalcBottom > 23 then
    begin
      FBottomDay := FShowDays;
      case FShowDays of
        1: FBottomHour := 23;
        2: FBottomHour := CalcBottom;
      end;
    end
    else
    begin
      FBottomDay := 1;
      FBottomHour := CalcBottom;
    end;
  end;
end;

procedure TTimePanel.DrawFrame(C: TCanvas);
var
  Rect: TRect;
  TopColor, BottomColor: TColor;

  procedure AdjustColors(Bevel: TPanelBevel);
  begin
    TopColor := clBtnHighlight;
    if Bevel = bvLowered then TopColor := clBtnShadow;
    BottomColor := clBtnShadow;
    if Bevel = bvLowered then BottomColor := clBtnHighlight;
  end;

begin
  Rect := GetClientRect;
  if BevelOuter <> bvNone then
  begin
    AdjustColors(BevelOuter);
    Frame3D(C, Rect, TopColor, BottomColor, BevelWidth);
  end;
  Frame3D(C, Rect, Color, Color, BorderWidth);
  if BevelInner <> bvNone then
  begin
    AdjustColors(BevelInner);
    Frame3D(C, Rect, TopColor, BottomColor, BevelWidth);
  end;
  with C do
  begin
    Brush.Color := Color;
    FillRect(Rect);
  end;
end;

procedure TTimePanel.Paint;
var
  B: TBitmap;
begin
  B := TBitmap.Create;
  try
    B.Width := Width;
    B.Height := Height;
    Canvas.Font := Font;
    B.Canvas.Font := Font;
    // Draw frame
    DrawFrame(B.Canvas);
    // Draw left hand side numbers
    DrawTimeNumbers(B.Canvas);
    // Draw connecting line
    if FShowConnect then
      DrawConnect(B.Canvas);
    // Draw start, finish and duration boxes
    DrawTimeBoxes(B.Canvas);
    // Draw arrows
    if FScrolling = sdNotScrolling then
    begin
      DrawArrow(B.Canvas, adUp, clBlack);
      DrawArrow(B.Canvas, adDown, clBlack);
    end
    else
    begin
      if FScrolling = sdUp then
      begin
        DrawArrow(B.Canvas, adUp, FSelColor);
        DrawArrow(B.Canvas, adDown, clBlack);
      end
      else
      begin
        DrawArrow(B.Canvas, adUp, clBlack);
        DrawArrow(B.Canvas, adDown, FSelColor);
      end;
    end;
    // Draw temporary bitmap onto control canvas
    Canvas.Draw(0, 0, B);
  finally
    // Destroy temporary bitmap
    B.Free;
  end;
  // Deal with scrolling
  if FScrolling <> sdNotScrolling then
  begin
    if FScrolling = sdUp then
    begin
      Scroll(sdUp, 0, Trunc(FScrollAmount));
      Invalidate;
      Sleep(5);
    end
    else
    begin
      Scroll(sdDown, 0, Trunc(FScrollAmount));
      Invalidate;
      Sleep(5);
    end;
    FScrollAmount := FScrollAmount + 0.5;
    if FScrollAmount > 15.0 then
      FScrollAmount := 15.0;
  end;
end;

procedure TTimePanel.SetTimeColor(Value: TColor);
begin
  if Value <> FTimeColor then
  begin
    FTimeColor := Value;
    Invalidate;
  end;
end;

procedure TTimePanel.SetDurationColor(Value: TColor);
begin
  if Value <> FDurationColor then
  begin
    FDurationColor := Value;
    Invalidate;
  end;
end;

procedure TTimePanel.SetDurationHours(Value: TTPHours);
var
  FinMins: Integer;
  CalcFinishMins: TTPMins;
  CalcFinishHours: TTPHours;
  CalcFinishDay: TTPDays;
begin
  if Value <> GetDurationHours then
  begin
    FinMins := ((FStartHours * 60) + FStartMins) +
      ((Value * 60) + GetDurationMins);
    CalcFinishMins := FinMins mod 60;
    CalcFinishHours := (FinMins div 60) mod 24;
    CalcFinishDay := (FinMins div (24 * 60)) + 1;
    if Value < GetDurationHours then
    begin
      FFinishDay := CalcFinishDay;
      FFinishHours := CalcFinishHours;
      FFinishMins := CalcFinishMins;
    end
    else
    begin
      if CalcFinishDay > FShowDays then
      begin
        FFinishDay := 1;
        FFinishHours := 23;
        FFinishMins := 59;
      end
      else
      begin
        FFinishDay := CalcFinishDay;
        FFinishHours := CalcFinishHours;
        FFinishMins := CalcFinishMins;
      end;
    end;
    Invalidate;
    DoTimeChange;
  end;
end;

procedure TTimePanel.SetDurationMins(Value: TTPMins);
var
  FinMins: Integer;
  CalcFinishMins: TTPMins;
  CalcFinishHours: TTPHours;
  CalcFinishDay: TTPDays;
begin
  if Value <> GetDurationMins then
  begin
    FinMins := ((FStartHours * 60) + FStartMins) +
      ((GetDurationHours * 60) + Value);
    CalcFinishMins := FinMins mod 60;
    CalcFinishHours := (FinMins div 60) mod 24;
    CalcFinishDay := (FinMins div (24 * 60)) + 1;
    if Value < GetDurationMins then
    begin
      FFinishDay := CalcFinishDay;
      FFinishHours := CalcFinishHours;
      FFinishMins := CalcFinishMins;
    end
    else
    begin
      if CalcFinishDay > FShowDays then
      begin
        FFinishDay := 1;
        FFinishHours := 23;
        FFinishMins := 59;
      end
      else
      begin
        FFinishDay := CalcFinishDay;
        FFinishHours := CalcFinishHours;
        FFinishMins := CalcFinishMins;
      end;
    end;
    Invalidate;
    DoTimeChange;
  end;
end;

function TTimePanel.GetDurationHours: TTPHours;
var
  STTPMins, FinMins: Integer;
begin
  STTPMins := ((FStartHours * 60) + FStartMins);
  FinMins := (((FFinishDay - 1) * (24 * 60)) + (FFinishHours * 60) +
    FFinishMins);
  Result := (FinMins - STTPMins) div 60;
end;

function TTimePanel.GetDurationMins: TTPMins;
var
  STTPMins, FinMins: Integer;
begin
  STTPMins := ((FStartHours * 60) + FStartMins);
  FinMins := (((FFinishDay - 1) * (24 * 60)) + (FFinishHours * 60) +
    FFinishMins);
  Result := (FinMins - STTPMins) mod 60;
end;

procedure TTimePanel.SetStartColor(Value: TColor);
begin
  if Value <> FStartColor then
  begin
    FStartColor := Value;
    Invalidate;
  end;
end;

procedure TTimePanel.SetStartHours(Value: TTPHours);
var
  STTPMins, FinMins: Integer;
begin
  if Value <> FStartHours then
  begin
    STTPMins := ((Value * 60) + FStartMins);
    FinMins := (((FFinishDay - 1) * (24 * 60)) + (FFinishHours * 60) +
      FFinishMins);
    if STTPMins > FinMins then
    begin
      FStartHours := FFinishHours;
      FStartMins := FFinishMins;
    end
    else
    begin
      FStartHours := STTPMins div 60;
      FStartMins := STTPMins mod 60;
      if (FinMins - STTPMins) >= (24 * 60) then
      begin
        FinMins := STTPMins + ((24 * 60) - 1);
        FFinishDay := 2;
        FFinishHours := (FinMins div 60) mod 24;
        FFinishMins := FinMins mod 60;
      end;
    end;
    Invalidate;
    DoTimeChange;
  end;
end;

procedure TTimePanel.SetStartMins(Value: TTPMins);
var
  STTPMins, FinMins: Integer;
begin
  if Value <> FStartMins then
  begin
    STTPMins := ((FStartHours * 60) + Value);
    FinMins := (((FFinishDay - 1) * (24 * 60)) + (FFinishHours * 60) +
      FFinishMins);
    if STTPMins > FinMins then
    begin
      FStartHours := FFinishHours;
      FStartMins := FFinishMins;
    end
    else
    begin
      FStartHours := STTPMins div 60;
      FStartMins := STTPMins mod 60;
      if (FinMins - STTPMins) >= (24 * 60) then
      begin
        FinMins := STTPMins + ((24 * 60) - 1);
        FFinishDay := 2;
        FFinishHours := (FinMins div 60) mod 24;
        FFinishMins := FinMins mod 60;
      end;
    end;
    Invalidate;
    DoTimeChange;
  end;
end;

procedure TTimePanel.SetFinishColor(Value: TColor);
begin
  if Value <> FFinishColor then
  begin
    FFinishColor := Value;
    Invalidate;
  end;
end;

procedure TTimePanel.SetFinishDay(Value: TTPDays);
var
  STTPMins, FinMins: Integer;
begin
  if Value <> FFinishDay then
  begin
    if Value <= FShowDays then
    begin
      STTPMins := ((FStartHours * 60) + FStartMins);
      FinMins := (((Value - 1) * (24 * 60)) + (FFinishHours * 60) +
        FFinishMins);
      if FinMins < STTPMins then
      begin
        FFinishDay := 1;
        FFinishHours := FStartHours;
        FFinishMins := FStartMins;
      end
      else
      begin
        FFinishDay := (FinMins div (24 * 60)) + 1;
        FFinishHours := (FinMins div 60) mod 24;
        FFinishMins := FinMins mod 60;
        if (FinMins - STTPMins) >= (24 * 60) then
        begin
          STTPMins := FinMins - ((24 * 60) - 1);
          FStartHours := STTPMins div 60;
          FStartMins := STTPMins mod 60;
        end;
      end;
      Invalidate;
      DoTimeChange;
    end;
  end;
end;

procedure TTimePanel.SetFinishHours(Value: TTPHours);
var
  STTPMins, FinMins: Integer;
begin
  if Value <> FFinishHours then
  begin
    STTPMins := ((FStartHours * 60) + FStartMins);
    FinMins := (((FFinishDay - 1) * (24 * 60)) + (Value * 60) +
      FFinishMins);
    if FinMins < STTPMins then
    begin
      FFinishDay := 1;
      FFinishHours := FStartHours;
      FFinishMins := FStartMins;
    end
    else
    begin
      FFinishDay := (FinMins div (24 * 60)) + 1;
      FFinishHours := (FinMins div 60) mod 24;
      FFinishMins := FinMins mod 60;
      if (FinMins - STTPMins) >= (24 * 60) then
      begin
        STTPMins := FinMins - ((24 * 60) - 1);
        FStartHours := STTPMins div 60;
        FStartMins := STTPMins mod 60;
      end;
    end;
    Invalidate;
    DoTimeChange;
  end;
end;

procedure TTimePanel.SetFinishMins(Value: TTPMins);
var
  STTPMins, FinMins: Integer;
begin
  if Value <> FFinishMins then
  begin
    STTPMins := ((FStartHours * 60) + FStartMins);
    FinMins := (((FFinishDay - 1) * (24 * 60)) + (FFinishHours * 60) +
      Value);
    if FinMins < STTPMins then
    begin
      FFinishDay := 1;
      FFinishHours := FStartHours;
      FFinishMins := FStartMins;
    end
    else
    begin
      FFinishDay := (FinMins div (24 * 60)) + 1;
      FFinishHours := (FinMins div 60) mod 24;
      FFinishMins := FinMins mod 60;
      if (FinMins - STTPMins) >= (24 * 60) then
      begin
        STTPMins := FinMins - ((24 * 60) - 1);
        FStartHours := STTPMins div 60;
        FStartMins := STTPMins mod 60;
      end;
    end;
    Invalidate;
    DoTimeChange;
  end;
end;

procedure TTimePanel.SetTopDay(Value: TTPDays);
begin
  if Value <> FTopDay then
  begin
    if Value <= FShowDays then
    begin
      FTopDay := Value;
      Invalidate;
    end;
  end;
end;

procedure TTimePanel.SetTopHour(Value: TTPHours);
begin
  if Value <> FTopHour then
  begin
    FTopHour := Value;
    Invalidate;
  end;
end;

procedure TTimePanel.SetTopMin(Value: TTPMins);
begin
  if Value <> FTopMin then
  begin
    FTopMin := Value;
    Invalidate;
  end;
end;

procedure TTimePanel.SetShowDays(Value: TTPDays);
var
  STTPMins, FinMins: Integer;
begin
  if Value <> FShowDays then
  begin
    FShowDays := Value;
    if (FFinishDay = 2) and (FShowDays = 1) then
    begin
      STTPMins := ((FStartHours * 60) + FStartMins);
      FinMins := ((FFinishHours * 60) + FFinishMins);
      FFinishDay := 1;
      if FinMins < STTPMins then
      begin
        FFinishHours := 23;
        FFinishMins := 59;
      end;
    end;
    Invalidate;
    DoTimeChange;
  end;
end;

procedure TTimePanel.SetShowDuration(Value: Boolean);
begin
  if Value <> FShowDuration then
  begin
    FShowDuration := Value;
    Invalidate;
  end;
end;

procedure TTimePanel.SetWorkTimeColor(Value: TColor);
begin
  if Value <> FWorkTimeColor then
  begin
    FWorkTimeColor := Value;
    Invalidate;
  end;
end;

procedure TTimePanel.SetWorkStart(Value: TTPHours);
begin
  if Value <> FWorkStart then
  begin
    if Value <= FWorkFinish then
    begin
      FWorkStart := Value;
    end
    else
    begin
      FWorkStart := FWorkFinish;
    end;
    Invalidate;
    DoTimeChange;
  end;
end;

procedure TTimePanel.SetWorkFinish(Value: TTPHours);
begin
  if Value <> FWorkFinish then
  begin
    if Value >= FWorkStart then
    begin
      FWorkFinish := Value;
    end
    else
    begin
      FWorkFinish := FWorkStart;
    end;
    Invalidate;
    DoTimeChange;
  end;
end;

procedure TTimePanel.SetSelColor(Value: TColor);
begin
  if Value <> FSelColor then
  begin
    FSelColor := Value;
    Invalidate;
  end;
end;

function TTimePanel.IsInRect(ARect: TRect; X, Y: Integer): Boolean;
begin
  with ARect do
    Result := (X >= Left) and (X <= Right) and (Y >= Top) and (Y <= Bottom);
end;

procedure TTimePanel.Scroll(Dir: TTPScrollDirection; Hours: TTPHours;
  Mins: TTPMins);
var
  TopMins, ScrollMins: Integer;
begin
  TopMins := (((FTopDay - 1) * (24 * 60)) + (FTopHour * 60) + FTopMin);
  ScrollMins := ((Hours * 60) + Mins);
  if Dir = sdUp then
  begin
    Dec(TopMins, ScrollMins);
    if TopMins < 0 then
      TopMins := 0;
  end
  else
  begin
    Inc(TopMins, ScrollMins);
    if FShowDays = 1 then
    begin
      if TopMins > ((24 * 60) - 1) then
        TopMins := (24 * 60) - 1;
    end
    else
    begin
      if TopMins > ((24 * 60 * 2) - 1) then
        TopMins := (24 * 60 * 2) - 1;
    end;
  end;
  FTopDay := (TopMins div (24 * 60)) + 1;
  FTopHour := (TopMins div 60) mod 24;
  FTopMin := TopMins mod 60;
  Invalidate;
end;

procedure TTimePanel.IncTime(StartFinish: TTPStartFinish; Hours: TTPHours;
  Mins: TTPMins);
var
  ToTTPMins, CompMins: Integer;
begin
  if StartFinish = sfStart then
  begin
    ToTTPMins := ((FStartHours * 60) + FStartMins);
    CompMins := (((FFinishDay - 1) * (24 * 60)) + (FFinishHours * 60) +
      FFinishMins);
  end
  else
  begin
    ToTTPMins := (((FFinishDay - 1) * (24 * 60)) + (FFinishHours * 60) +
      FFinishMins);
    CompMins := ((FStartHours * 60) + FStartMins);
  end;
  Inc(ToTTPMins, ((Hours * 60) + Mins));
  if StartFinish = sfStart then
  begin
    if ToTTPMins > ((24 * 60) - 1) then
      ToTTPMins := ((24 * 60) - 1);
    if (ToTTPMins <= CompMins) or (CompMins > ((24 * 60) - 1)) then
    begin
      FStartHours := ToTTPMins div 60;
      FStartMins := ToTTPMins mod 60;
    end
    else
    begin
      FStartHours := FFinishHours;
      FStartMins := FFinishMins;
    end;
  end
  else
  begin
    if ToTTPMins > ((24 * 60 * FShowDays) - 1) then
      ToTTPMins := ((24 * 60 * FShowDays) - 1);
    FFinishDay := (ToTTPMins div (24 * 60)) + 1;
    FFinishHours := (ToTTPMins div 60) mod 24;
    FFinishMins := ToTTPMins mod 60;
  end;
  Invalidate;
  DoTimeChange;
end;

procedure TTimePanel.DecTime(StartFinish: TTPStartFinish; Hours: TTPHours;
  Mins: TTPMins);
var
  ToTTPMins, CompMins: Integer;
begin
  if StartFinish = sfStart then
  begin
    ToTTPMins := ((FStartHours * 60) + FStartMins);
    CompMins := (((FFinishDay - 1) * (24 * 60)) + (FFinishHours * 60) +
      FFinishMins);
  end
  else
  begin
    ToTTPMins := (((FFinishDay - 1) * (24 * 60)) + (FFinishHours * 60) +
      FFinishMins);
    CompMins := ((FStartHours * 60) + FStartMins);
  end;
  Dec(ToTTPMins, ((Hours * 60) + Mins));
  if ToTTPMins < 0 then
    ToTTPMins := 0;
  if StartFinish = sfStart then
  begin
    FStartHours := ToTTPMins div 60;
    FStartMins := ToTTPMins mod 60;
    Invalidate;
    DoTimeChange;
  end
  else
  begin
    if ToTTPMins >= CompMins then
    begin
      FFinishDay := (ToTTPMins div (24 * 60)) + 1;
      FFinishHours := (ToTTPMins div 60) mod 24;
      FFinishMins := ToTTPMins mod 60;
      Invalidate;
      DoTimeChange;
    end;
  end;
end;

procedure TTimePanel.MouseDown(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
begin
  inherited MouseDown(Button, Shift, X, Y);
  if IsInRect(FTopArrowRect, X, Y) and (Button = mbLeft) then
  begin
    FScrolling := sdUp;
    FScrollAmount := 1.0;
    Invalidate;
  end
  else if IsInRect(FBottomArrowRect, X, Y) and (Button = mbLeft) then
  begin
    FScrolling := sdDown;
    FScrollAmount := 1.0;
    Invalidate;
  end
  else
    FScrolling := sdNotScrolling;
end;

procedure TTimePanel.MouseUp(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
begin
  inherited MouseUp(Button, Shift, X, Y);
  if (FScrolling <> sdNotScrolling) and (Button = mbLeft) then
    FScrolling := sdNotScrolling;
end;

procedure TTimePanel.MouseMove(Shift: TShiftState; X, Y: Integer);
begin
  inherited MouseMove(Shift, X, Y);
  // Deal with scrolling arrows
  if IsInRect(FTopArrowRect, X, Y) then
    DrawArrow(Canvas, adUp, SelColor)
  else
    DrawArrow(Canvas, adUp, clBlack);
  if IsInRect(FBottomArrowRect, X, Y) then
    DrawArrow(Canvas, adDown, SelColor)
  else
    DrawArrow(Canvas, adDown, clBlack);
end;

procedure TTimePanel.StartMouseMove(Sender: TObject; Shift: TShiftState;
  X, Y: Integer);
var
  OldTop, NewTop: Integer;
  Hours: TTPHours;
  Mins: TTPMins;
begin
  if Shift = [ssLeft] then
    if (Y < 0) then
    begin
      OldTop := FStartControl.Top;
      NewTop := FStartControl.Top + Y;
      FStartControl.Top := NewTop;
      Hours := ((OldTop - NewTop) div FTextHeight) mod 24;
      Mins := Trunc(((OldTop - NewTop) / FTextHeight) * 12) * 5;
      DecTime(sfStart, Hours, Mins + (FStartMins mod 5));
    end
    else if (Y > FStartControl.Height) then
    begin
      OldTop := FStartControl.Top;
      NewTop := FStartControl.Top + (Y - FStartControl.Height);
      FStartControl.Top := NewTop;
      Hours := ((NewTop - OldTop) div FTextHeight) mod 24;
      Mins := Trunc(((NewTop - OldTop) / FTextHeight) * 12) * 5;
      IncTime(sfStart, Hours, Mins + (5 - (FStartMins mod 5)));
    end;
end;

procedure TTimePanel.FinishMouseMove(Sender: TObject; Shift: TShiftState;
  X, Y: Integer);
var
  OldTop, NewTop: Integer;
  Hours: TTPHours;
  Mins: TTPMins;
begin
  if Shift = [ssLeft] then
    if (Y < 0) then
    begin
      OldTop := FFinishControl.Top;
      NewTop := FFinishControl.Top + Y;
      FFinishControl.Top := NewTop;
      Hours := ((OldTop - NewTop) div FTextHeight) mod 24;
      Mins := Trunc(((OldTop - NewTop) / FTextHeight) * 12) * 5;
      DecTime(sfFinish, Hours, Mins + (FFinishMins mod 5));
    end
    else if (Y > FFinishControl.Height) then
    begin
      OldTop := FFinishControl.Top;
      NewTop := FFinishControl.Top + (Y - FFinishControl.Height);
      FFinishControl.Top := NewTop;
      Hours := ((NewTop - OldTop) div FTextHeight) mod 24;
      Mins := Trunc(((NewTop - OldTop) / FTextHeight) * 12) * 5;
      IncTime(sfFinish, Hours, Mins + (5 - (FFinishMins mod 5)));
    end;
end;

procedure TTimePanel.DurationMouseMove(Sender: TObject; Shift: TShiftState;
  X, Y: Integer);
var
  OldTop, NewTop: Integer;
  Hours: TTPHours;
  Mins: TTPMins;
begin
  if (Shift = [ssLeft]) and (FShowDuration) then
    if (Y < 0) then
    begin
      OldTop := FDurationControl.Top;
      NewTop := FDurationControl.Top + Y;
      FDurationControl.Top := NewTop;
      Hours := ((OldTop - NewTop) div FTextHeight) mod 24;
      Mins := Trunc(((OldTop - NewTop) / FTextHeight) * 12) * 5;
      DecTime(sfStart, Hours, Mins);
      DecTime(sfFinish, Hours, Mins);
    end
    else if (Y > FDurationControl.Height) then
    begin
      OldTop := FDurationControl.Top;
      NewTop := FDurationControl.Top + (Y - FDurationControl.Height);
      FDurationControl.Top := NewTop;
      Hours := ((NewTop - OldTop) div FTextHeight) mod 24;
      Mins := Trunc(((NewTop - OldTop) / FTextHeight) * 12) * 5;
      IncTime(sfFinish, Hours, Mins);
      IncTime(sfStart, Hours, Mins);
    end;
end;

function TTimePanel.GetFinishTime: TDateTime;
begin
  Result := EncodeTime(FFinishHours, FFinishMins, 0, 0);
end;

function TTimePanel.GetStartTime: TDateTime;
begin
  Result := EncodeTime(FStartHours, FStartMins, 0, 0);
end;

procedure TTimePanel.SetFinishTime(Value: TDateTime);
var
  Hours, Mins, Secs, MSecs: Word;
begin
  if Value <> GetFinishTime then
  begin
    DecodeTime(Value, Hours, Mins, Secs, MSecs);
    if Value >= GetStartTime then
    begin
      FFinishHours := Hours;
      FFinishMins := Mins;
    end
    else
      if FShowDays = 2 then
      begin
        FFinishDay := 2;
        FFinishHours := Hours;
        FFinishMins := Mins;
      end
      else
      begin
        FFinishHours := FStartHours;
        FFinishMins := FStartMins;
      end;
    Invalidate;
  end;
end;

procedure TTimePanel.SetStartTime(Value: TDateTime);
var
  Hours, Mins, Secs, MSecs: Word;
begin
  if Value <> GetStartTime then
  begin
    DecodeTime(Value, Hours, Mins, Secs, MSecs);
    if Value <= GetFinishTime then
    begin
      FStartHours := Hours;
      FStartMins := Mins;
    end
    else
      if (FShowDays = 2) and (FFinishDay = 2) then
      begin
        FStartHours := Hours;
        FStartMins := Mins;
      end
      else
      begin
        FStartHours := FFinishHours;
        FStartMins := FFinishMins;
      end;
    Invalidate;
  end;
end;

procedure TTimePanel.SetDurationFormat(Value: String);
begin
  if Value <> FDurationFormat then
  begin
    FDurationFormat := Value;
    Invalidate;
  end;
end;

procedure TTimePanel.SetTimeFormat(Value: String);
begin
  if Value <> FTimeFormat then
  begin
    FTimeFormat := Value;
    Invalidate;
  end;
end;

procedure TTimePanel.SetShowConnect(Value: Boolean);
begin
  if Value <> FShowConnect then
  begin
    FShowConnect := Value;
    Invalidate;
  end;
end;

procedure TTimePanel.DoTimeChange;
begin
  if Assigned(FOnTimeChange) then FOnTimeChange(Self);
end;

procedure Register;
begin
  RegisterComponents('Time', [TTimePanel]);
end;

end.
