{ ****************************************************************** }
{                                                                    }
{   CCthread.pas:  Main unit file for thread component framework     }
{                                                                    }
{   Copyright  1997 by David A. Price                               }
{                                                                    }
{   Distributed with Thread Component Toolset (tm) v. 1.0            }
{   http://www.compcreate.com                                        }
{                                                                    }
{ ****************************************************************** }

{ ****************************************************************** }
{   TNewThread, the parent class for thread components, creates      }
{   and runs a new thread within an application.                     }
{   It requires Delphi 2.0 or later.                                 }
{ ****************************************************************** }

unit Ccthread;

interface

uses SysUtils, WinProcs, WinTypes, Classes;

type
  ThreadStateType = (thdInactive, thdSuspended, thdActive);
  
  TPrivateThread = class(TThread)
    private
        { Pointer to OnExecute handler, if any }
        FOnExecute : TNotifyEvent;
    public
        procedure Execute; override;
        { Expose the Synchronize method, protected in TThread }
        procedure Synchronize(Method: TThreadMethod);
        { Enable assignment of handler for Execute, protected in TThread }
        property OnExecute : TNotifyEvent read FOnExecute write FOnExecute;
        { Expose the Terminated property, protected in TThread }
        property Terminated;
  end;

  TNewThread = class(TComponent)
    private
      { Private fields of TNewThread }
        { Storage for property ExceptionName }
        FExceptionName : String;
        { Storage for property ExceptionString }
        FExceptionString : String;
        { Storage for property StopRequested }
        FStopRequested : Boolean;
        { Storage for property Priority }
        FPriority : TThreadPriority;
        { Storage for property Running }
        FRunning : Boolean;
        { Storage for property ThreadState }
        FThreadState : ThreadStateType;
        { Pointer to application's OnTerminate handler, if any }
        FOnTerminate : TNotifyEvent;
        { How many times has thread been suspended? }
        SuspendCount : Integer;
        { Internal thread object }
        Thread : TPrivateThread;

      { Private methods of TNewThread }
        { Method to set variable and property values and create objects }
        procedure AutoInitialize;
        { Method to free any objects created by AutoInitialize }
        procedure AutoDestroy;
        { Method to wrap around Execute method and trap any exceptions }
        procedure PrivateExecute(Sender : TObject);
        { Read method for property StopRequested }
        function GetStopRequested : Boolean;
        { Read method for property Priority }
        function GetPriority : TThreadPriority;
        { Write method for property Priority }
        procedure SetPriority(Value : TThreadPriority);

    protected
      { Protected fields of TNewThread }

      { Protected methods of TNewThread }
        { Method for thread to execute when started }
        procedure Execute(Sender : TObject); virtual;
        { Execute a method that makes VCL calls }
        procedure Synchronize(Method : TThreadMethod);
        { Method executed when thread terminates }
        procedure Terminate(Sender : TObject); virtual;

    public
      { Public fields and properties of TNewThread }
        { Name of exception not handled by Execute method, if any }
        property ExceptionName : String read FExceptionName;
        { Message string of exception not handled by Execute method, if any }
        property ExceptionString : String read FExceptionString;
        { Is the thread started and not terminated? }
        property Running : Boolean read FRunning;
        { Has the thread been requested to stop yet? }
        property StopRequested : Boolean read GetStopRequested;

      { Public methods of TNewThread }
        constructor Create(AOwner: TComponent); override;
        destructor Destroy; override;
        { Start the thread process }
        procedure Start; virtual;
        { Resume the thread process }
        procedure Resume;
        { Suspend the thread process }
        procedure Suspend;
        { Request the thread process to terminate }
        procedure RequestStop;

    published
      { Published properties of TNewThread }
        { Thread's priority level }
        property Priority : TThreadPriority
             read GetPriority write SetPriority
             default tpNormal;
        { Thread's execution status }
        property ThreadState : ThreadStateType read FThreadState;
        { OnTerminate handler runs when thread terminates }
        property OnTerminate : TNotifyEvent read FOnTerminate write FOnTerminate;
  end;

  TResourceFlag = class(TComponent)
    private
      { Private fields of TResourceFlag }
        FCriticalSectionRecord : TRTLCriticalSection;
        FHaveCriticalSectionRecord : Boolean;

    public
      { Public fields and properties of TResourceFlag }
        property CriticalSectionRecord : TRTLCriticalSection
             read FCriticalSectionRecord;

      { Public methods of TResourceFlag }
        constructor Create(AOwner: TComponent); override;
        destructor Destroy; override;
  end;

  TSharingThread = class(TNewThread)
    private
      { Private fields of TSharingThread }
        FResourceFlag : TResourceFlag;
        FCriticalSectionRecord : TRTLCriticalSection;
        FHaveCriticalSectionRecord : Boolean;

      { Private methods of TSharingThread }
        { Method to set variable and property values and create objects }
        procedure AutoInitialize;

    protected
      { Protected methods of TSharingThread }
        { Write method for ResourceFlag }
        procedure SetResourceFlag(Value : TResourceFlag); virtual;
        { Resets prop if referenced TResourceFlag deleted }
        procedure Notification(AComponent : TComponent; Operation : TOperation);
                  override;
        procedure WaitForResource;
        procedure DoneWithResource;

    public
      { Public methods of TSharingThread }
        constructor Create(AOwner: TComponent); override;
        destructor Destroy; override;
        procedure Start; override;

    published
      { Published properties of TSharingThread }
        { TResourceFlag component, if any, signalling
          status of a shared resource }
        property ResourceFlag : TResourceFlag
             read FResourceFlag write SetResourceFlag;
  end;

type
  ThreadError = class(Exception);

procedure Register;

implementation

procedure Register;
begin
     { Register TResourceFlag with Thread as its
       default page on the Delphi component palette }
     RegisterComponents('Thread', [TResourceFlag]);
end;

{ ****************************************************************** }
{   TPrivateThread object                                            }
{ ****************************************************************** }

{ Enable assignment of handler for Execute, protected in TThread }
procedure TPrivateThread.Execute;
begin
     if Assigned(FOnExecute) then
        FOnExecute(Self);
end;

{ Expose the Synchronize method, protected in TThread }
procedure TPrivateThread.Synchronize(Method: TThreadMethod);
begin
     inherited Synchronize(Method)
end;

{ ****************************************************************** }
{   TNewThread component                                             }
{ ****************************************************************** }

{ Method to set variable and property values and create objects }
procedure TNewThread.AutoInitialize;
begin
     SuspendCount := 0;
     FExceptionName := '';
     FExceptionString := '';
     FPriority := tpNormal;
     FRunning := False;
     FThreadState := thdInactive;
end; { of AutoInitialize }

{ Method to free any objects created by AutoInitialize }
procedure TNewThread.AutoDestroy;
begin
     if FRunning then
        begin
        Thread.Suspend;
        Thread.Free
        end;
end; { of AutoDestroy }

{ Read method for property StopRequested }
function TNewThread.GetStopRequested : Boolean;
begin
     if FRunning then
        Result := Thread.Terminated
     else
         Result := False
end;

{ Read method for property Priority }
function TNewThread.GetPriority : TThreadPriority;
begin
     if FRunning then
        Result := Thread.Priority
     else
         Result := FPriority
end;

{ Write method for property Priority }
procedure TNewThread.SetPriority(Value : TThreadPriority);
begin
     FPriority := Value;
     if FRunning then
        Thread.Priority := Value;
end;

{ Method to wrap around Execute method and trap any exceptions }
procedure TNewThread.PrivateExecute(Sender : TObject);
begin
     try
        Execute(Sender);
     except
        on Ex : Exception do
           begin
           FExceptionName := UpperCase(Ex.ClassName);
           FExceptionString := Ex.Message
           end
     end;
end;

{ Method for thread to execute when started }
procedure TNewThread.Execute(Sender : TObject);
begin
end;

{ Method executed when thread terminates }
procedure TNewThread.Terminate;
begin
     if not FRunning then
        Exit;
     FRunning := False;
     FThreadState := thdInactive;
     { Has the application assigned a method to OnTerminate, whether
       via the Object Inspector or a run-time assignment?  If so,
       execute that method }
     if Assigned(FOnTerminate) then
        FOnTerminate(Self)
end;

constructor TNewThread.Create(AOwner: TComponent);
begin
     inherited Create(AOwner);
     AutoInitialize;
end;

destructor TNewThread.Destroy;
begin
     AutoDestroy;
     inherited Destroy;
end;

{ Start the thread process }
procedure TNewThread.Start;
begin
     if FRunning then
        begin
        raise ThreadError.Create('Start called when thread already started');
        Exit
        end;
     { Create the thread in a suspended state }
     Thread := TPrivateThread.Create(True);
     FExceptionName := '';
     FExceptionString := '';
     Thread.FreeOnTerminate := True;
     { Assign PrivateExecute method as the OnExecute handler of the thread }
     Thread.OnExecute := PrivateExecute;
     { Assign Terminate method as the OnTerminate handler of the thread }
     Thread.OnTerminate := Terminate;
     FRunning := True;
     FThreadState := thdActive;
     { Start the thread process }
     Thread.Resume;
end;

{ Resume the thread process }
procedure TNewThread.Resume;
var
   I : Integer;
begin
     if not FRunning then
        begin
        raise ThreadError.Create('Resume called when thread not started');
        Exit
        end;
     { Execute thread's Resume method the number of times
       needed to balance calls to thread's Suspend method }
     for I := SuspendCount downto 1 do
         Thread.Resume;
     { Update ThreadState property }
     FThreadState := thdActive;
     { Reset counter for Suspend calls }
     SuspendCount := 0;
end;

{ Suspend the thread process }
procedure TNewThread.Suspend;
begin
     if not FRunning then
        begin
        raise ThreadError.Create('Suspend called when thread not started');
        Exit
        end;
     SuspendCount := SuspendCount + 1;
     FThreadState := thdSuspended;
     Thread.Suspend
end;

{ Request the thread process to terminate }
procedure TNewThread.RequestStop;
begin
     if not FRunning then
        begin
        raise ThreadError.Create('Stop called when thread not started');
        Exit
        end;
     Thread.Terminate;
end;

{ Execute a method that makes VCL calls }
procedure TNewThread.Synchronize(Method : TThreadMethod);
begin
    if not FRunning then
       begin
       raise ThreadError.Create('Synchronize called when thread not started');
       Exit
       end;
    Thread.Synchronize(Method)
end;

{ ****************************************************************** }
{   TResourceFlag component                                          }
{ ****************************************************************** }

constructor TResourceFlag.Create(AOwner: TComponent);
begin
     inherited Create(AOwner);
     { Create the critical resource object if we are executing }
     if csDesigning in ComponentState then
        FHaveCriticalSectionRecord := False
     else
        begin
        InitializeCriticalSection(FCriticalSectionRecord);
        FHaveCriticalSectionRecord := True
        end
end;

destructor TResourceFlag.Destroy;
begin
     if FHaveCriticalSectionRecord then
        DeleteCriticalSection(FCriticalSectionRecord);
     inherited Destroy;
end;

{ ****************************************************************** }
{   TSharingThread component                                         }
{ ****************************************************************** }

{ Method to set variable and property values and create objects }
procedure TSharingThread.AutoInitialize;
begin
     FResourceFlag := nil;
     FHaveCriticalSectionRecord := False;
end; { of AutoInitialize }

{ Resets prop of component type if referenced component deleted }
procedure TSharingThread.Notification(AComponent : TComponent; Operation : TOperation);
begin
     inherited Notification(AComponent, Operation);
     if Operation <> opRemove then
        Exit;
     { Has a TResourceFlag component referenced by
       property ResourceFlag of this component been deleted?
       If so, update the property. }
     if AComponent = FResourceFlag then
        begin
        FResourceFlag := nil;
        FHaveCriticalSectionRecord := False
        end
end;

procedure TSharingThread.SetResourceFlag(Value : TResourceFlag);
begin
     FResourceFlag := Value;
     if Value <> nil then
        Value.FreeNotification(Self)
end;

constructor TSharingThread.Create(AOwner: TComponent);
begin
     inherited Create(AOwner);
     AutoInitialize;
end;

destructor TSharingThread.Destroy;
begin
     { The critical section record will be deleted by the
       TResourceFlag component that initialized it,
       not by us. }
     inherited Destroy;
end;

{ Method to start the thread }
procedure TSharingThread.Start;
begin
     { Load the critical resource object from the TResourceFlag
       component, if we are linked to one }
     if FResourceFlag <> nil then
        begin
        FCriticalSectionRecord := FResourceFlag.CriticalSectionRecord;
        FHaveCriticalSectionRecord := True
        end
     else
         FHaveCriticalSectionRecord := False;

     { Run the Execute method as a separate thread }
     inherited Start;
end;

procedure TSharingThread.WaitForResource;
begin
     if FHaveCriticalSectionRecord then
        EnterCriticalSection(FCriticalSectionRecord)
end;

procedure TSharingThread.DoneWithResource;
begin
     if FHaveCriticalSectionRecord then
        LeaveCriticalSection(FCriticalSectionRecord)
end;


end.
