// -----------------------------------------------------------------------------
// +--------------------------------------------------------+
// | Filename       : TextEdit.PAS                          |
// | Delphi Version : 4.0 (3.57)                            |
// | System Version : 1.22                                  |
// | Author         : Lincoln Birnie                        |
// | Date           :  8/10/1998                            |
// | Overview       : Based upon the TEdit component, this  |
// |                  implementation overrides several      |
// |                  methods and adds several extra        |
// |                  properties to provide both a generic  |
// |                  or numeric datum entry interface.     |
// |                  This component is intended to replace |
// |                  the TEdit component that ships with   |
// |                  Delphi.                               |
// |                                                        |
// |                  Overriden Methods:                    |
// |                    - Change             - DoEnter      |
// |                    - KeyPress           - DoExit       |
// |                  Additional Properties:                |
// |                    - Alert          : TAlertData       |
// |                    - Range          : TRangeData       |
// |                    - ActiveFocus    : Boolean          |
// |                    - EditStyle      : TEditType        |
// |                    - DecimalPlaces  : TDecWidth [0..9] |
// |                    - Value          : GetValue(...)    |
// |                    - OORError       : Boolean          |
// +--------------------------------------------------------+
// -----------------------------------------------------------------------------

unit TextEdit;

interface

uses
  SysUtils, WinTypes, WinProcs, Classes, Controls, Graphics, StdCtrls,
  ExtCtrls, Dialogs, Messages;

type
// -----------------------------------------------------------------------------
// -- Supporting Objects
  TEditType = (esGeneral, esInteger, esFloat, esCurrency, esPercentage);
  TDecWidth = 0..9;

  TAlertData = class(TPersistent)
    private
      FControl               : TControl;
      FActive                : Boolean;
      FColor                 : TColor;
    public
      constructor Create(Control: TControl);
      destructor Destroy; override;
      procedure Assign(Source: TPersistent); override;
    private
      procedure UpdateParent;
      procedure SetActive(Value: Boolean);
      procedure SetColor(Value: TColor);
    published
      property Active        : Boolean          read FActive   write SetActive;
      property Color         : TColor           read FColor    write SetColor;
   end;

  TRangeData = class(TPersistent)
    private
      FControl               : TControl;
      FEnforce               : Boolean;
      FMaxValue              : Double;
      FMinValue              : Double;
      FColor                 : TColor;
    public
      constructor Create(Control: TControl);
      destructor Destroy; override;
      procedure Assign(Source: TPersistent); override;
    private
      procedure UpdateParent;
      procedure SetEnforce(Value: Boolean);
      procedure SetValue(Index: Integer; Value: Double);
      procedure SetColor(Value: TColor);
    published
      property Enforce       : Boolean          read FEnforce  write SetEnforce;
      property MinValue      : Double   index 1 read FMinValue write SetValue;
      property MaxValue      : Double   index 2 read FMaxValue write SetValue;
      property Color         : TColor           read FColor    write SetColor;
   end;

// -- VCL Object
  TExtendEdit = class(TEdit)
    private      // -- Private declarations --
      FEditStyle             : TEditType;
      FActiveFocus           : Boolean;
      FDecPlaces             : TDecWidth; (* 0 = any, 1 - 6 *)
      FOORError              : Boolean;
      FAlertData             : TAlertData;
      FRangeData             : TRangeData;
   protected     // -- Protected declarations --
      procedure Change; override;
      procedure DoEnter; override;
      procedure DoExit; override;
      procedure KeyPress(var Key: Char); override;
    private      // -- Private declarations --
      procedure DisplayControl;
      procedure DisplayActive;
      procedure DisplayInActive;
      procedure InvalidateDisplay;
      procedure InvalidateVisuals;
      procedure InvalidateValue;
      procedure SetEditStyle(DStyle: TEditType);
      procedure SetDecimalPlaces(DWidth: TDecWidth);
      procedure SetValue(DValue: Double);
      function GetValue : Double;
      function ConvNumeric(DValue: string) : Double;
    public       // -- Public declarations --
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;
    published    // -- Published declarations --
      // -- Properties
      property ActiveFocus   : Boolean             read FActiveFocus  write FActiveFocus     default True;
      property Alert         : TAlertData          read FAlertData    write FAlertData;
      property DecimalPlaces : TDecWidth           read FDecPlaces    write SetDecimalPlaces default 0;
      property EditStyle     : TEditType           read FEditStyle    write SetEditStyle     default esGeneral;
      property OORError      : Boolean             read FOORError     write FOORError;
      property Range         : TRangeData          read FRangeData    write FRangeData;
      property Value         : Double              read GetValue      write SetValue;
      // -- Events
   end;


// -----------------------------------------------------------------------------
// -- Supporting Data-Types and Structures
// -----------------------------------------------------------------------------
procedure Register;

// -----------------------------------------------------------------------------
implementation
{ $X+ }

// -----------------------------------------------------------------------------
procedure Register;
 begin
   RegisterComponents('Additional', [TExtendEdit]);
 end;

// -----------------------------------------------------------------------------
// -- TExtendEdit : AlertData Source Code
// -----------------------------------------------------------------------------
constructor TAlertData.Create(Control: TControl);
begin
  inherited Create;
  FControl := Control;
  FActive := False;
  FColor := clWindow;
end;

destructor TAlertData.Destroy;
begin
  inherited Destroy;
end;

procedure TAlertData.Assign(Source: TPersistent);
begin
  with TAlertData(Source) do
  begin
    FActive := Active;
    FColor := Color;
  end;
end;

procedure TAlertData.UpdateParent;
begin
  if (FControl <> nil) then TExtendEdit(FControl).InvalidateDisplay;
end;

procedure TAlertData.SetActive(Value: Boolean);
begin
  if (FActive <> Value) then
  begin
    FActive := Value;
    UpdateParent;
  end;
end;

procedure TAlertData.SetColor(Value: TColor);
begin
  if (FColor <> Value) then
  begin
    FColor := Value;
    UpdateParent;
  end;
end;

// -----------------------------------------------------------------------------
// -- TExtendEdit : RangeData Source Code
// -----------------------------------------------------------------------------
constructor TRangeData.Create(Control: TControl);
begin
  inherited Create;
  FControl := Control;
  FColor := clWindow;
  FEnforce := False;
  FMinValue := -100;
  FMaxValue := 100;
end;

destructor TRangeData.Destroy;
begin
  inherited Destroy;
end;

procedure TRangeData.Assign(Source: TPersistent);
begin
  with TRangeData(Source) do
  begin
    FColor := Color;
    FEnforce := Enforce;
    FMinValue := MinValue;
    FMaxValue := MaxValue;
  end;
end;

procedure TRangeData.UpdateParent;
begin
  if (FControl <> nil) then TExtendEdit(FControl).InvalidateDisplay;
end;

procedure TRangeData.SetEnforce(Value: Boolean);
begin
  if (FEnforce <> Value) then
  begin
    FEnforce := Value;
    UpdateParent;
  end;
end;

procedure TRangeData.SetValue(Index: Integer; Value: Double);
begin
  case Index of
  1 : FMinValue := Value;
  2 : FMaxValue := Value;
  end;
  if (FMinValue > FMaxValue) then FMinValue := FMaxValue;
  UpdateParent;
end;

procedure TRangeData.SetColor(Value: TColor);
begin
  if (FColor <> Value) then
  begin
    FColor := Value;
    UpdateParent;
  end;
end;

// -----------------------------------------------------------------------------
// -- TExtendEdit Component Source Code
// -----------------------------------------------------------------------------
constructor TExtendEdit.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FAlertData := TAlertData.Create(Self);
  FRangeData := TRangeData.Create(Self);
  FActiveFocus := True;
  FOORError := False;
end;

destructor TExtendEdit.Destroy;
begin
  FRangeData.Free;
  FAlertData.Free;
  inherited Destroy;
end;

// -----------------------------------------------------------------------------
procedure TExtendEdit.Change;
begin
  inherited Change;
  InvalidateDisplay;
end;

procedure TExtendEdit.DoEnter;
begin
  DisplayActive;
  inherited DoEnter;
end;

procedure TExtendEdit.DoExit;
begin
  InvalidateDisplay;
  DisplayInActive;
  inherited DoExit;
end;

procedure TExtendEdit.KeyPress(var Key: Char);
var
  cvSearch, cvxValue, cvDec  : string;
  cvKeyOk                    : Boolean;
begin
  if (FEditStyle <> esGeneral) then
  begin
    cvDec := DecimalSeparator;
    cvxValue := Self.Text;
    if (Self.SelLength > 0) then Delete(cvxValue,Self.SelStart+1,Self.SelLength);
    case Ord(Key) of
    8 : if (Self.SelLength = 0) then Delete(cvxValue,Self.SelStart,1); // BKSP
    else Insert(Key,cvxValue,Self.SelStart+1);
    end;
    cvSearch := '0123456789-()';
    case FEditStyle of
         esFloat : cvSearch := (cvSearch + DecimalSeparator);
      esCurrency : cvSearch := (cvSearch + DecimalSeparator + CurrencyString[1]);
    esPercentage : cvSearch := (cvSearch + DecimalSeparator + '%');
    end;
    case FEditStyle of
    esGeneral : cvKeyOk := True;
    else begin
           cvKeyOk := (pos(Key,cvSearch) > 0);
           if (pos(cvDec,Self.Text) > 0) then
           begin
             cvKeyOk := (cvKeyOk and (length(copy(cvxValue,pos(cvDec,cvxValue)+1,99)) <= FDecPlaces));
             cvKeyOk := (cvKeyOk and (not (Key = cvDec[1])));
           end;
           cvKeyOk := (cvKeyOk or (Ord(Key) = vk_Back));
         end;
    end;
    if (not cvKeyOk) then Key := Chr(0);
  end;
  inherited KeyPress(Key);
end;

// -----------------------------------------------------------------------------
procedure TExtendEdit.DisplayControl;
begin
  InvalidateDisplay;
  if (Self.Focused and Self.ActiveFocus) then DisplayActive
  else DisplayInActive;
end;

procedure TExtendEdit.DisplayActive;
var
  cvValue                    : Double;
begin
  cvValue := GetValue;
  if ((not FActiveFocus) or (not Self.Focused)) then DisplayInActive
  else case FEditStyle of
          esGeneral : ;
          esInteger : Self.Text := IntToStr(Trunc(cvValue));
       esPercentage : Self.Text := FloatToStrF(cvValue*100,ffFixed,15,FDecPlaces);
       else Self.Text := FloatToStrF(cvValue,ffFixed,15,FDecPlaces);
       end;
  if (AutoSelect and Self.Focused) then Self.SelectAll;
end;

procedure TExtendEdit.DisplayInActive;
var
  cvValue                    : Double;
begin
  cvValue := GetValue;
  case FEditStyle of
     esInteger : Self.Text := IntToStr(Trunc(cvValue));
       esFloat : Self.Text := FloatToStrF(cvValue,ffFixed,15,FDecPlaces);
    esCurrency : Self.Text := FloatToStrF(cvValue,ffCurrency,15,FDecPlaces);
  esPercentage : Self.Text := (FloatToStrF(cvValue*100,ffFixed,15,FDecPlaces)+'%');
  end;
end;

procedure TExtendEdit.InvalidateDisplay;
begin
  InvalidateValue;
  InvalidateVisuals;
end;

procedure TExtendEdit.InvalidateVisuals;
var
  cvValue                    : Double;
begin
  cvValue := GetValue;
  FOORError := ((FRangeData.MinValue > cvValue) or (FRangeData.MaxValue < cvValue));
  case (FOORError and FAlertData.Active and (FEditStyle <> esGeneral)) of
   True : Self.Color := FAlertData.Color;
  False : Self.Color := FRangeData.Color;
  end;
end;

procedure TExtendEdit.InvalidateValue;
var
  cvValue                    : Double;
begin
  cvValue := GetValue;
  if (FRangeData.Enforce and (FEditStyle <> esGeneral)) then
  begin
    if (FRangeData.MinValue > cvValue) then SetValue(FRangeData.MinValue);
    if (FRangeData.MaxValue < cvValue) then SetValue(FRangeData.MaxValue);
  end;
  FOORError := ((FRangeData.MinValue > cvValue) or (FRangeData.MaxValue < cvValue));
end;

// -----------------------------------------------------------------------------
procedure TExtendEdit.SetEditStyle(DStyle: TEditType);
begin
  if (FEditStyle <> DStyle) then FEditStyle := DStyle;
  DisplayControl;
end;

procedure TExtendEdit.SetDecimalPlaces(DWidth: TDecWidth);
begin
  if (FDecPlaces <> DWidth) then FDecPlaces := DWidth;
  DisplayControl;
end;

// -----------------------------------------------------------------------------
procedure TExtendEdit.SetValue(DValue: Double);
begin
  if (FEditStyle <> esGeneral) then Self.Text := FloatToStr(DValue);
  DisplayControl;
end;

function TExtendEdit.GetValue : Double;
begin
  case FEditStyle of
     esGeneral : Result := 0;
  esPercentage : Result := (ConvNumeric(Self.Text)/100);
  else Result := ConvNumeric(Self.Text);
  end;
end;

// -----------------------------------------------------------------------------
function TExtendEdit.ConvNumeric(DValue: string) : Double;
var
  cvIndex, cvMaxDec          : Integer;
  cvValue, cvSearch          : string;
  cvNegative                 : Boolean;
begin
  try
    cvValue := '';
    cvNegative := ((pos('(',DValue) > 0) and (pos(')',DValue) > 0)) or (pos('-',DValue) > 0);
    cvSearch := ('0123456789' + DecimalSeparator);
    case FEditStyle of
    esGeneral : cvMaxDec := -1;
    esInteger : cvMaxDec := 0;
    else cvMaxDec := 1;
    end;
    for cvIndex := 1 to (Length(DValue)) do
    begin
      if (DecimalSeparator = DValue[cvIndex]) then Dec(cvMaxDec);
      if (pos(DValue[cvIndex],cvSearch) > 0) and (cvMaxDec > -1) then
        cvValue := (cvValue + DValue[cvIndex]);
    end;
    cvValue := Trim(cvValue);
    if (cvNegative) then cvValue := ('-' + cvValue);
    if (cvValue = '') or (cvValue = '-') then cvValue := '0';
    Result := StrToFloat(cvValue);
  except on
     E : Exception do Result := 0;
  end;
end;

// -----------------------------------------------------------------------------

end.
