unit AddOnTB97;

////////////////////////////////////////////////////////////////////////////////
//
//  ADDONTB97.PAS
//  Version 1.8 (30/jan/00)   for Delphi 2-3-4-5 and C++ 3-4
//  written by Jacques VOIRIN
//  E-Mail: jacques.voirin@iname.com
//
////////////////////////////////////////////////////////////////////////////////
//
//  AddOnTB97 is designed to work with TB97 written by Jordan Russel (jordanr@iname.com)
//  Tested with TB97.pas v1.75d                            (http://www.jordanr.dhs.org/)
//
//  It provides seven components:
//
//     -TPlusEdit97-
//        Enhanced Russel's TEdit97 with more properties :
//           - Alignment: TAlignment;
//           - Color;
//           - KeyboardFilter = (kfNone, kfNumeric, kfAlphabetic, kfAll);
//           - MouseBehaviour: TBehaviour; (mb97Style or mbEasyStyle)
//              when the active component is a TEdit97 or descendant
//              (i.e. TPlusEdit97, TSpinEdit97,TLinkEdit, TComboBox97,etc)
//              and the mouse cursor goes over another TEdit97 familly,
//              - mb97Style: nothing happend
//              - mbEasy Style: component under mouse cursor get "Focus Style"
//
//     -TNumEdit97-
//        Derived from TPlusEdit
//           - DisplayFormat: TFloatFormat;
//           - Decimal: Byte;
//           - Value: Double;
//           - MaxValue, MinValue: Double ;
//           - KeyboardFilter = kfNumeric for '0..9' keys;
//
//     -TSpin97-
//        A Flat Spin Component using 2 TToolbarButton97s working like VCL TSpin
//           - Fully customizable trougth Caption, Glyph, NumGlyph, Margin, Layout,
//             Mask and Spacing properties for each Up and Down Buttons;
//           - MouseLeft or MouseEntered
//              - True parameter force the Up Button to be raised or not
//              - False, act over Down Button.
//           - Repeating, RepeatingDelay and RepeatingInterval properties;
//           - Increment: Double;
//           - Kind: TSpin97Kind; (skVertical or skHorizontal)
//           - FocusControl: TWinControl;
//
//     -TSpinEdit97-
//        A Flat Spin Edit Component using one TNumEdit97 and one TSpin97
//           - SpinButton fully customizable using TSpin97 properties;
//
//     -TLinkEdit-
//        A Flat Edit using a TPlusEdit and a TTolbarButton97
//            - Button fully customizable using TTolbarButton97 Properties;
//            - Many Button Mouse Events implemented;
//
//     -TComboBox97-
//        Derived form TCustomCombobox97, a Flat ComboBox
//
//     -TFontComboBox97-
//        Derived form TCustomCombobox97, a Flat Font ComboBox
//            - Options: TFontComboBox97Options = (foTTOnly, foUseItemFont);
//                  foTTOnly: show only TT Fonts
//                  foUseItemFont: display listed Fonts in their way
//
//
////////////////////////////////////////////////////////////////////////////////
//
//  You can heritate from TCustomLinkedit97 and TCustomCombobox97 for your own
//  components.
//
////////////////////////////////////////////////////////////////////////////////
//
//
//  These components and source are FREEWARE.
//
//  You are free to use AddOnTB97.pas.
//
//  If you use it for personnal purpose,
//
//  However, use in commercial or shareware applications requires
//  MY EXPRESS WRITTEN PERMISSION.
//
//  ENJOY CODING!
//
////////////////////////////////////////////////////////////////////////////////
//
//  History:
//
//   v0.9b: (Springs 98...)
//           First release
//
//   v1.5:  (20/jul/99)
//         - Add TLinkEdit, TComboBox97 and TFontComboBox97
//         - Many bugs fixed
//
//
//   v1.7: (15/oct/99)
//         - Tb97 v1.74 compatible (use of the new TB97's ControlIs97Control prop)
//         - added Version property  (sorry for Russel...)
//         - added Flat: Boolean and NoBorder: Boolean for TSpin97
//         - code design: added CustomLinkEdit97 and TCustomCombobox97
//         - fixed TFontCombobox.ItemIndex goes GPF when assigned
//         - TFontCombobox now filtered fonts names (no more doubles)
//         - added TFontCombobox Options property
//         - remove TFontCombobox's TrueTypeFontsOnly. Use Options
//         - remove TFontCombobox's OnFontChange. Use OnChange (more classic)
//         - new Focus code for TSpinEdit97, TCustomLinkEdit97
//            and TCustomCombobox97 (works better, I think...)
//         - removed ChangeColorOnFocus prop. (too buggy)
//         - added History
//
//   v1.8: (30/jan/00)
//         - Multi Delphi versions compatible.
//         - TCustomComboBox97.Destroy fix.
//         - TSpinEdit Alignment fix.
//         - other minor drawing routines bug fix.
//
////////////////////////////////////////////////////////////////////////////////
//
//
//  Know bugs: (only drawing bugs...)
//             - at design time, buttons in TSpinEdit97, TLinkEdit, TComboBox97
//               and TFontComboBox97 are drawing with DOUBLE up/left white border (?)
//             - Button still visible at design time when TCombobox97 set to csSimple
//             - problem with default bitmap in TSpin and TLinkEdit when you don't want it
//             - problem of drawing buttons 3D aspect when a Form is activated
//               for the first time and if Form.ActiveControl is TSpinEdit97 or
//               TLinkEdit97 or TCombobox97.
//             - obviously others...
//
//  Please notice bugs or problems, suggestions and improvements at
//
//  jacques.voirin@iname.com
//
////////////////////////////////////////////////////////////////////////////////


 {$R AddOnTB97.res}


interface


 {$I AddOnTB97Ver.inc}


uses
  Windows, Classes, Controls, Messages, Graphics, Buttons, SysUtils, Forms,
  StdCtrls, Tb97Ctls;

const
  TB97AddOnVersion = '1.8';
  TB97AddOnPropText = 'TB97AddOn v' + TB97AddOnVersion;

type


////////////////////////////////////////////////////////////////////////////////
//
//  TPlusEdit97
//
////////////////////////////////////////////////////////////////////////////////


  TKeyboardFilter = (kfNone, kfNumeric, kfAlphabetic, kfAll);
  TBehaviour = (mb97Style, mbEasyStyle);

  TPlusEdit97 = class(TEdit97)
  private
    FAlignment: TAlignment;
    FEditorEnabled: Boolean;
    FKeyboardFilter: TKeyboardFilter;
    FBehaviour: TBehaviour;
    FOnMouseEnter: TNotifyEvent;
    procedure WMSetFocus (var Message: TWMSetFocus);  message WM_SETFOCUS;
    procedure WMSetCursor(var Msg: TWMSetCursor); message WM_SETCURSOR;
    procedure SetAlignment(Value: TAlignment);
    procedure SetBehaviour(Value: TBehaviour);
    function  GetVersion: String;
    procedure SetVersion (const Value: String);
    procedure CMMouseEnter (var Message: TMessage); message CM_MOUSEENTER;

  protected
    procedure CreateParams(var Params: TCreateParams); override;
    function  IsValidChar(Key: Char): Boolean; virtual;
    procedure KeyPress(var Key: Char); override;
  public
    constructor Create(AOwner: TComponent); override;
  published
    property Alignment: TAlignment read FAlignment write SetAlignment default taLeftJustify;
    property Color;
    property EditorEnabled: Boolean read FEditorEnabled write FEditorEnabled default True;
    property KeyboardFilter: TKeyboardFilter read FKeyboardFilter write FKeyboardFilter default kfNone;
    property MouseBehaviour: TBehaviour read FBehaviour write SetBehaviour default mb97Style;
    property Version: String read GetVersion write SetVersion stored False;
  end;



////////////////////////////////////////////////////////////////////////////////
//
//  TNumEdit97
//
////////////////////////////////////////////////////////////////////////////////


  TNumEdit97 = class(TPlusEdit97)
  private
    FValue, FMinValue, FMaxValue: Double;
    FDigits: Byte;
    FDisplayFormat: TFloatFormat;
    procedure SetDigits(NewValue: Byte);
    procedure SetDisplayFormat(NewValue: TFloatFormat);
    procedure SetValue(NewValue: Double);
    function  GetValue: Double;
    function  CheckValue (NewValue: Double): Double;
  protected
    procedure CMExit(var Message: TCMExit);   message CM_EXIT;
  public
    constructor Create(AOwner: TComponent); override;
  published
    property DisplayFormat: TFloatFormat read FDisplayFormat write SetDisplayFormat;
    property Decimal: Byte read FDigits write SetDigits default 0;
    property Value: Double read GetValue write SetValue;// default 0;
    property MaxValue: Double read FMaxValue write FMaxValue;
    property MinValue: Double read FMinValue write FMinValue;
  end;



////////////////////////////////////////////////////////////////////////////////
//
//  TSpin97
//
////////////////////////////////////////////////////////////////////////////////


  TSpin97Kind = (skVertical, skHorizontal);

  TSpin97 = class(TWinControl)
   private
    FUpButton, FDownButton, FActiveBUtton: TToolBarButton97;
    FFocusControl: TWinControl;
    FOnUpClick, FOnDownClick, FOnMouseEnter, FOnMouseExit: TNotifyEvent;
    FOnMouseMove: TMouseMoveEvent;
    FRepeating: Boolean;
    FRepeatDelay, FRepeatInterval: Integer;
    FIncrement: Double;
    FFlat: Boolean;
    FNoBorder: Boolean;
    FKind: TSpin97Kind;
    FirstTime: Boolean;

    function  GetUpCaption: TCaption;
    procedure SetUpCaption(Value: TCaption);
    function  GetUpGlyph: TBitmap;
    procedure SetUpGlyph(Value: TBitmap);
    function  GetUpLayout: TButtonLayout;
    procedure SetUpLayout(Value: TButtonLayout);
    function  GetUpMargin: Integer;
    procedure SetUpMargin(Value:Integer);
    function  GetUpMask: TBitmap;
    procedure SetUpMask (Value: TBitmap);
    function  GetUpNumGlyphs: TNumGlyphs;
    procedure SetUpNumGlyphs(Value: TNumGlyphs);
    function  GetUpSpacing: Integer;
    procedure SetUpSpacing(Value: Integer);

    function  GetDownCaption: TCaption;
    procedure SetDownCaption(Value: TCaption);
    function  GetDownGlyph: TBitmap;
    procedure SetDownGlyph(Value: TBitmap);
    function  GetDownLayout: TButtonLayout;
    procedure SetDownLayout(Value: TButtonLayout);
    function  GetDownMargin: Integer;
    procedure SetDownMargin(Value:Integer);
    function  GetDownMask: TBitmap;
    procedure SetDownMask (Value: TBitmap);
    procedure SetDownNumGlyphs(Value: TNumGlyphs);
    function  GetDownNumGlyphs: TNumGlyphs;
    function  GetDownSpacing: Integer;
    procedure SetDownSpacing(Value: Integer);

    function  GetVersion: String;
    procedure SetVersion (const Value: String);

    procedure WMSize(var Message: TWMSize);  message WM_SIZE;

    procedure ArrangeButtons;
    procedure SetKind(Value: TSpin97Kind);
    procedure SetFlat(Value: Boolean);
    procedure SetNoBorder(Value: Boolean);
  protected
    function  CreateButton: TToolBarButton97;
    procedure BtnMouseEnter(Sender: TObject);
    procedure BtnMouseExit(Sender: TObject);
    procedure BtnMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure BtnMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure BtnClick(Sender: TObject);
    procedure BtnMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);

  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure MouseEntered(Value: Boolean);
    procedure MouseLeft(Value: Boolean);
    property OnMouseEnter: TNotifyEvent read FOnMouseEnter write FOnMouseEnter;
  published
    property Align;
  {$IFNDEF Delphi4OrHigher}
    property Anchors;
    property BiDiMode;
    property Constraints;
    {$ENDIF}
    property DownCaption: TCaption read GetDownCaption write SetDownCaption;
    property DownGlyph: TBitmap read GetDownGlyph write SetDownGlyph;
    property DownLayout: TButtonLayout read GetDownLayout write SetDownLayout;
    property DownMargin: Integer read GetDownMargin write SetDownMargin default -1;
    property DownMask: TBitmap read GetDownMask write SetDownMask;
    property DownNumGlyphs: TNumGlyphs read GetDownNumGlyphs write SetDownNumGlyphs default 1;
    property DownSpacing: Integer read GetDownSpacing write SetDownSpacing default 4;
    property DragCursor;
   {$IFNDEF Delphi4OrHigher}
    property DragKind;
    {$ENDIF}
    property DragMode;
    property Enabled;
    property Flat: Boolean read FFlat write SetFlat default True;
    property Font;
    property FocusControl: TWinControl read FFocusControl write FFocusControl;
   {$IFNDEF Delphi4OrHigher}
    property ImeMode;
    property ImeName;
    {$ENDIF}
    property Increment: Double read FIncrement write FIncrement;// default 0;
    property Kind: TSpin97Kind read FKind write SetKind default skVertical;
    property NoBorder: Boolean read FNoBorder write SetNoBorder default False;
   {$IFNDEF Delphi4OrHigher}
    property ParentBiDiMode;
    {$ENDIF}
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property Repeating: Boolean read FRepeating write FRepeating default False;
    property RepeatDelay: Integer read FRepeatDelay write FRepeatDelay default 100;
    property RepeatInterval: Integer read FRepeatInterval write FRepeatInterval default 100;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property UpCaption: TCaption read GetUpCaption write SetUpCaption;
    property UpGlyph: TBitmap read GetUpGlyph write SetUpGlyph;
    property UpLayout: TButtonLayout read GetUpLayout write SetUpLayout;
    property UpMargin: Integer read GetUpMargin write SetUpMargin default -1;
    property UpMask: TBitmap read GetUpMask write SetUpMask;
    property UpNumGlyphs: TNumGlyphs read GetUpNumGlyphs write SetUpNumGlyphs default 1;
    property UpSpacing: Integer read GetUpSpacing write SetUpSpacing default 4;
    property Version: String read GetVersion write SetVersion stored False;
    property Visible;
    property OnDownClick: TNotifyEvent read FOnDownClick write FOnDownClick;
    property OnUpClick: TNotifyEvent read FOnUpClick write FOnUpClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDock;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnStartDock;
    property OnStartDrag;
  end;



////////////////////////////////////////////////////////////////////////////////
//
//  TSpinEdit97
//
////////////////////////////////////////////////////////////////////////////////


 TSpinEdit97 = class(TNumEdit97)
  private
    FButton: TSpin97;
    FSeparator: TWinControl;
    MouseInControl: Boolean;

    function  GetUpCaption: TCaption;
    procedure SetUpCaption(Value: TCaption);
    function  GetUpGlyph: TBitmap;
    procedure SetUpGlyph(Value: TBitmap);
    function  GetUpLayout: TButtonLayout;
    procedure SetUpLayout(Value: TButtonLayout);
    function  GetUpMargin: Integer;
    procedure SetUpMargin(Value:Integer);
    function  GetUpMask: TBitmap;
    procedure SetupMask (Value: TBitmap);
    function  GetUpNumGlyphs: TNumGlyphs;
    procedure SetUpNumGlyphs(Value: TNumGlyphs);
    function  GetUpSpacing: Integer;
    procedure SetUpSpacing(Value: Integer);

    function  GetDownCaption: TCaption;
    procedure SetDownCaption(Value: TCaption);
    function  GetDownGlyph: TBitmap;
    procedure SetDownGlyph(Value: TBitmap);
    function  GetDownLayout: TButtonLayout;
    procedure SetDownLayout(Value: TButtonLayout);
    function  GetDownMargin: Integer;
    procedure SetDownMargin(Value:Integer);
    function  GetDownMask: TBitmap;
    procedure SetDownMask (Value: TBitmap);
    procedure SetDownNumGlyphs(Value: TNumGlyphs);
    function  GetDownNumGlyphs: TNumGlyphs;
    function  GetDownSpacing: Integer;
    procedure SetDownSpacing(Value: Integer);

    procedure SetEditRect;

    procedure SetIncrement(Value: Double);
    function  GetIncrement: Double;

    procedure SetSpinWidth(Value: Integer);
    function  GetSpinWidth: Integer;
    procedure SetSpinKind(Value: TSpin97Kind);
    function  GetSpinKind: TSpin97Kind;

    procedure WMSize(var Message: TWMSize);  message WM_SIZE;
    procedure CMEnter(var Message: TCMGotFocus);  message CM_ENTER;
    procedure CMExit(var Message: TCMExit);   message CM_EXIT;
    procedure CMMouseEnter (var Message: TMessage);  message CM_MOUSEENTER;
    procedure CMMouseLeave (var Message: TMessage);  message CM_MOUSELEAVE;
    procedure WMSetFocus (var Message: TWMSetFocus);  message WM_SETFOCUS;
    procedure WMKillFocus (var Message: TWMKillFocus);  message WM_KILLFOCUS;
    procedure WMNCPaint (var Message: TMessage);  message WM_NCPAINT;


  protected
    procedure ButtonMouseEnter(Sender: TObject);
    procedure UpClick (Sender: TObject); virtual;
    procedure DownClick (Sender: TObject); virtual;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure CreateWnd; override;
    procedure RedrawButton;
  public
    constructor Create(AOwner: TComponent); override;
    destructor  Destroy; override;
  published
    property DownCaption: TCaption read GetDownCaption write SetDownCaption;
    property DownGlyph: TBitmap read GetDownGlyph write SetDownGlyph;
    property DownLayout: TButtonLayout read GetDownLayout write SetDownLayout;
    property DownMargin: Integer read GetDownMargin write SetDownMargin default -1;
    property DownMask: TBitmap read GetDownMask write SetDownMask;
    property DownNumGlyphs: TNumGlyphs read GetDownNumGlyphs write SetDownNumGlyphs default 1;
    property DownSpacing: Integer read GetDownSpacing write SetDownSpacing default 4;
    property Increment: Double read GetIncrement write SetIncrement;// default 1;
    property SpinKind: TSpin97Kind read GetSpinKind write SetSpinKind default skVertical;
    property SpinWidth: Integer read GetSpinWidth write SetSpinWidth;
    property UpCaption: TCaption read GetUpCaption write SetUpCaption;
    property UpGlyph: TBitmap read GetUpGlyph write SetUpGlyph;
    property UpLayout: TButtonLayout read GetUpLayout write SetUpLayout;
    property UpMargin: Integer read GetUpMargin write SetUpMargin default -1;
    property UpMask: TBitmap read GetUpMask write SetUpMask;
    property UpNumGlyphs: TNumGlyphs read GetUpNumGlyphs write SetUpNumGlyphs default 1;
    property UpSpacing: Integer read GetUpSpacing write SetUpSpacing default 4;
    property Value: Double read GetValue write SetValue;
  end;



////////////////////////////////////////////////////////////////////////////////
//
//  TCustomLinkEdit97
//
////////////////////////////////////////////////////////////////////////////////


  TCustomLinkEdit97 = class(TPlusEdit97)
   private
    FButton: TToolbarButton97;
    FSeparator: TWinControl;
    FBtnAllowDown: Boolean;
    FButtonClick, FButtonDblClick, FButtonMouseEnter,
    FButtonMouseExit: TNotifyEvent;
    FButtonMouseDown, FButtonMouseUp: TMouseEvent;
    FButtonMouseMove: TMouseMoveEvent;
    MouseInControl: Boolean;

    procedure SetBtnAllowDown(Value: Boolean);
    function  GetBtnCaption: TCaption;
    procedure SetBtnCaption(Value: TCaption);
    function  GetBtnGlyph: TBitmap;
    procedure SetBtnGlyph(Value: TBitmap);
    function  GetBtnLayout: TButtonLayout;
    procedure SetBtnLayout(Value: TButtonLayout);
    function  GetBtnMargin: Integer;
    procedure SetBtnMargin(Value:Integer);
    function  GetBtnMask: TBitmap;
    procedure SetBtnMask (Value: TBitmap);
    function  GetBtnNumGlyphs: TNumGlyphs;
    procedure SetBtnNumGlyphs(Value: TNumGlyphs);
    function  GetBtnSpacing: Integer;
    procedure SetBtnSpacing(Value: Integer);
    function  GetBtnVisible: Boolean;
    procedure SetBtnVisible(Value:Boolean);
    function  GetBtnWidth: Integer;
    procedure SetBtnwidth(Value: Integer);

    procedure SetEditRect;

    procedure WMSetCursor(var Msg: TWMSetCursor); message WM_SETCURSOR;
    procedure WMSize(var Message: TWMSize); message WM_SIZE;
    procedure CMEnter(var Message: TCMGotFocus); message CM_ENTER;
    procedure CMExit(var Message: TCMExit);  message CM_EXIT;
    procedure CMMouseEnter(var Message: TMessage); message CM_MOUSEENTER;
    procedure CMMouseLeave(var Message: TMessage); message CM_MOUSELEAVE;
    procedure WMSetFocus(var Message: TWMSetFocus);  message WM_SETFOCUS;
    procedure WMKillFocus(var Message: TWMKillFocus); message WM_KILLFOCUS;
    procedure WMNCPaint (var Message: TMessage); message WM_NCPAINT;

  protected
    procedure ButtonClick(Sender: TObject);
    procedure ButtonDblClick(Sender: TObject);
    procedure ButtonMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure ButtonMouseEnter(Sender: TObject);
    procedure ButtonMouseExit(Sender: TObject);
    procedure ButtonMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
    procedure ButtonMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure CreateWnd; override;
    procedure RedrawButton;
  public
    constructor Create(AOwner: TComponent); override;
    destructor  Destroy; override;
    procedure BtnUp;
    procedure BtnDown;
    property BtnAllowDown: Boolean read FBtnAllowDown write SetBtnAllowDown default False;
    property BtnCaption: TCaption read GetBtnCaption write SetBtnCaption;
    property BtnGlyph: TBitmap read GetBtnGlyph write SetBtnGlyph;
    property BtnLayout: TButtonLayout read GetBtnLayout write SetBtnLayout;
    property BtnMargin: Integer read GetBtnMargin write SetBtnMargin default -1;
    property BtnMask: TBitmap read GetBtnMask write SetBtnMask;
    property BtnNumGlyphs: TNumGlyphs read GetBtnNumGlyphs write SetBtnNumGlyphs default 1;
    property BtnSpacing: Integer read GetBtnSpacing write SetBtnSpacing default 4;
    property BtnVisible: Boolean read GetBtnVisible write SetBtnVisible default True;
    property BtnWidth: Integer read GetBtnWidth write SetBtnWidth default 15;
    property OnButtonClick: TNotifyEvent read FButtonClick write FButtonClick;
    property OnButtonDblClick: TNotifyEvent read FButtonDblClick write FButtonDblClick;
    property OnButtonMouseDown: TMouseEvent read FButtonMouseDown write FButtonMouseDown;
    property OnButtonMouseEnter: TNotifyEvent read FButtonMouseEnter write FButtonMouseEnter;
    property OnButtonMouseExit: TNotifyEvent read FButtonMouseExit write FButtonMouseExit;
    property OnButtonMouseMove: TMouseMoveEvent read FButtonMouseMove write FButtonMouseMove;
    property OnButtonMouseUp: TMouseEvent read FButtonMouseUp write FButtonMouseUp;
  end;



////////////////////////////////////////////////////////////////////////////////
//
//  TLinkEdit97
//
////////////////////////////////////////////////////////////////////////////////


   TLinkEdit97 = class(TCustomLinkEdit97)
   published
    property BtnAllowDown;
    property BtnCaption;
    property BtnGlyph;
    property BtnLayout;
    property BtnMargin;
    property BtnMask;
    property BtnNumGlyphs;
    property BtnSpacing;
    property BtnWidth;
    property OnButtonClick;
    property OnButtonDblClick;
    property OnButtonMouseDown;
    property OnButtonMouseEnter;
    property OnButtonMouseExit;
    property OnButtonMouseMove;
    property OnButtonMouseUp;
   end;



////////////////////////////////////////////////////////////////////////////////
//
//  TCustomComboBox97
//
////////////////////////////////////////////////////////////////////////////////


type
  TCustomComboBox97 = class;

  TCBListView = class(TListBox)
  private
    FMouseInControl: Boolean;
    procedure CMMouseEnter(var Message: TMessage); message CM_MOUSEENTER;
    procedure CMMouseLeave(var Message: TMessage); message CM_MOUSELEAVE;
    procedure WMMouseMove(var Message: TWMMouseMove); message WM_MOUSEMOVE;
    procedure WMLButtonDown(var Message: TWMLButtonDown); message WM_LBUTTONDOWN;
  protected
    procedure CreateParams(var Params: TCreateParams); override;
    procedure CreateWnd; override;
  end;

  TCustomComboBox97 = class(TCustomLinkEdit97)
    List: TListBox;
    procedure BtnClick(Sender: TObject);
  private
    FDropDownCount: Integer;
    FStyle: TComboBoxStyle;
    FCanvas: TControlCanvas;
    FItemsIndex: Integer;
    FOnDropDown: TNotifyEvent;
    FOnDrawItem: TDrawItemEvent;
    FOnMeasureItem: TMeasureItemEvent;
    function GetItemIndex: Integer;
    procedure SetItemIndex(Value: Integer);
    function  GetItems: TStrings;
    procedure SetItems(Value: TStrings);
    function  GetSorted: Boolean;
    procedure SetSorted(Value: Boolean);
    function  GetItemHeight: Integer;
    procedure SetItemHeight(Value: Integer);
    procedure SetListVisible(Value: Boolean);
    function  GetListVisible: Boolean;
    procedure SetStyle(Value: TComboBoxStyle);
    procedure WMKillFocus (var Message: TWMKillFocus);  message WM_KILLFOCUS;
  protected
    procedure CreateParams(var Params: TCreateParams); override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure ListOnDrawItem(Control: TWinControl; Index: Integer;
      Rect: TRect; State: TOwnerDrawState);
    procedure ListOnMeasureItem(Control: TWinControl; Index: Integer;
      var Height: Integer);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property Canvas: TControlCanvas read FCanvas write FCanvas;
    property DropDownCount: Integer read FDropDownCount write FDropDownCount default 8;
    property ItemIndex: Integer read GetItemIndex write SetItemIndex default -1;
    property Items: TStrings read GetItems write SetItems;
    property ItemHeight: Integer read GetItemHeight write SetItemHeight default 13;
    property Sorted: Boolean read GetSorted write SetSorted default False;
    property Style: TComboBoxStyle read FStyle write SetStyle default csDropDown;
    property ListVisible: Boolean read GetListVisible write SetListVisible default False;
    property OnDropDown: TNotifyEvent read FOnDropDown write FOnDropDown;
    property OnDrawItem: TDrawItemEvent read FOnDrawItem write FOnDrawItem;
    property OnMeasureItem: TMeasureItemEvent read FOnMeasureItem write FOnMeasureItem;
  end;


////////////////////////////////////////////////////////////////////////////////
//
//  TComboBox97
//
////////////////////////////////////////////////////////////////////////////////


type
  TComboBox97 = class(TCustomComboBox97)
  published
    property BtnCaption;
    property BtnGlyph;
    property BtnLayout;
    property BtnMargin;
    property BtnMask;
    property BtnNumGlyphs;
    property BtnSpacing;
    property BtnWidth;
    property DropDownCount;
    property ItemIndex;
    property Items;
    property ItemHeight;
    property Sorted;
    property Style;
    property OnDropDown;
    property OnDrawItem;
    property OnMeasureItem;
  end;



////////////////////////////////////////////////////////////////////////////////
//
//  TFontComboBox97
//
////////////////////////////////////////////////////////////////////////////////

  TFontComboBox97Option = (foTTOnly, foUseItemFont);
  TFontComboBox97Options = set of TFontComboBox97Option;


  TFontComboBox97 = class(TCustomComboBox97)
  protected
    TrueTypeSign: TBitmap;
    FOnChange: TNotifyEvent;
    FOptions: TFontComboBox97Options;
    procedure CreateFontList;
    procedure PurgeFontList;
    procedure SetOptions(Value: TFontComboBox97Options);
    procedure DrawFontItem(Control: TWinControl; Index: Integer; Rect: TRect; State: TOwnerDrawState);
    procedure CreateWnd; override;
    procedure Change; override;
  public
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
  published
    property DropDownCount;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
    property Options: TFontComboBox97Options read FOptions write SetOptions default [];
  end;




procedure Register;


implementation

const
     ACTIVE = True;
     INACTIVE = False;

     UP_SPIN_BUTTON = True;
     DOWN_SPIN_BUTTON = False;



////////////////////////////////////////////////////////////////////////////////
//
//  TPlusEdit97
//
////////////////////////////////////////////////////////////////////////////////


constructor TPlusEdit97.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FBehaviour := mb97Style;
  FEditorEnabled := True;
end;


procedure TPlusEdit97.WMSetFocus (var Message: TWMSetFocus);
begin
  inherited;
  if not FEditorEnabled then DestroyCaret;
end;


procedure TPlusEdit97.WMSetCursor(var Msg: TWMSetCursor);
begin
  inherited;
  if not FEditorEnabled then Cursor := crArrow;
end;


procedure TPlusEdit97.SetAlignment(Value: TAlignment);
begin
  if Value <> FAlignment
  then begin
         FAlignment := Value;
         RecreateWnd;
         end;
end;


procedure TPlusEdit97.SetBehaviour(Value: TBehaviour);
begin
  if Value <> FBehaviour
  then FBehaviour := Value;
end;


function TPlusEdit97.GetVersion: String;
begin
  Result := TB97AddOnPropText;
end;


procedure TPlusEdit97.SetVersion (const Value: String);
begin
  // just for syntax
end;


procedure TPlusEdit97.CreateParams(var Params: TCreateParams);
const
  Alignments:  array[TAlignment] of UINT = (DT_LEFT, DT_RIGHT, DT_CENTER);
begin
  inherited CreateParams(Params);
  Params.Style := Params.Style or ES_MULTILINE or WS_CLIPCHILDREN
    or Alignments[FAlignment];
end;


procedure TPlusEdit97.KeyPress(var Key: Char);
begin
  if not IsValidChar(Key)
  then begin
         Key := #0;
         Beep;
         end;
  if Key <> #0
  then inherited KeyPress(Key);
end;


function TPlusEdit97.IsValidChar(Key: Char): Boolean;
begin
  case FKeyboardFilter of
   kfNone:       Result := True;
   kfNumeric:    begin
                   Result := (Key in [DecimalSeparator, '+', '-', '0'..'9']) or
                   ((Key < #32) and (Key <> Chr(VK_RETURN)));
                   if not Result and ((Key >= #32) or (Key = Char(VK_BACK)) or (Key = Char(VK_DELETE)))
                   then Result := False;
                 end;
   kfAlphabetic: begin
                   Result := (Key in ['a'..'z', 'A'..'Z']) or (Key = Char(VK_BACK)) or
                   ((Key < #32) and (Key <> Chr(VK_RETURN)));
                   if not Result and ((Key >= #32) or (Key = Char(VK_BACK)) or (Key = Char(VK_DELETE)))
                   then Result := False;
                 end;
   else Result := False;   // FKeyboardFilter := kfAll
  end;
end;


procedure TPlusEdit97.CMMouseEnter (var Message: TMessage);

var
  DC: HDC;
  R: TRect;
  BtnFaceBrush, WindowBrush: HBRUSH;
begin
  DC := GetDC(0);
  if ((MouseBehaviour = mbEasyStyle) or not(ControlIs97Control(Screen.ActiveControl)))
  then begin
         inherited ;
         DC := GetWindowDC(Handle);
         GetWindowRect (Handle, R);
         OffsetRect (R, -R.Left, -R.Top);
         BtnFaceBrush := GetSysColorBrush(COLOR_BTNFACE);
         WindowBrush := GetSysColorBrush(COLOR_WINDOW);
         DrawEdge (DC, R, BDR_SUNKENOUTER, BF_RECT or BF_ADJUST);
         with R do begin
                     FillRect (DC, Rect(Left, Top, Left+1, Bottom-1), BtnFaceBrush);
                     FillRect (DC, Rect(Left, Top, Right-1, Top+1), BtnFaceBrush);
                     end;
         DrawEdge (DC, R, BDR_SUNKENINNER, BF_BOTTOMRIGHT);
         InflateRect (R, -1, -1);
         FrameRect (DC, R, WindowBrush);
         end;

  ReleaseDC(Handle, DC);

  if Assigned(FOnMouseEnter)
  then FOnMouseEnter(Self);
end;



////////////////////////////////////////////////////////////////////////////////
//
//  TNumEdit97
//
////////////////////////////////////////////////////////////////////////////////


constructor TNumEdit97.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle - [csSetCaption];
  Value := 0;
end;


procedure TNumEdit97.CMExit(var Message: TCMExit);
begin
  inherited;
  if CheckValue(Value) <> Value
  then SetValue(Value);
end;


procedure TNumEdit97.SetValue(NewValue: Double);
begin
  FValue := CheckValue(NewValue);
  Caption := FloatToStrF(FValue, FDisplayFormat, 15, FDigits);
end;


function TNumEdit97.GetValue: Double;
var s: String;
begin
  if FDisplayFormat = ffCurrency
  then s := FloatToStrF(FValue, ffGeneral, 15, FDigits)
  else s := Text;
  try
    Result := StrToFloat(s);
  except
    Result := FMinValue;
  end;
end;


function TNumEdit97.CheckValue (NewValue: Double): Double;
begin
  Result := NewValue;
  if FMaxValue <> FMinValue
  then begin
         if NewValue < FMinValue
         then Result := FMinValue
         else if NewValue > FMaxValue
              then Result := FMaxValue;
         end;
  if Result <> NewValue
  then Beep;
end;


procedure TNumEdit97.SetDigits(NewValue: Byte);
begin
  if FDigits <> NewValue
  then begin
         FDigits := NewValue;
         SetValue(FValue);
         end;
end;


procedure TNumEdit97.SetDisplayFormat(NewValue: TFloatFormat);
begin
  if FDisplayFormat <> NewValue
  then begin
         FDisplayFormat := NewValue;
         SetValue(FValue);
         end;
end;




////////////////////////////////////////////////////////////////////////////////
//
//  TSpin97
//
////////////////////////////////////////////////////////////////////////////////


constructor TSpin97.Create(AOwner: TComponent);
begin

  inherited Create(AOwner);

  FUpButton := CreateButton;
  if (csDesigning in ComponentState)
  then FUpButton.Glyph.Handle := LoadBitmap(HInstance, 'UPBITMAP');

  FDownButton := CreateButton;
  if (csDesigning in ComponentState)
  then FDownButton.Glyph.Handle := LoadBitmap(HInstance, 'DOWNBITMAP');

  Width := 20;
  Height := 25;
  RepeatDelay := 100;
  RepeatInterval := 100;
  Repeating := False;
end;


destructor TSpin97.Destroy;
begin
  FUpButton.Free;
  FDownButton.Free;
  inherited Destroy;
end;


function TSpin97.CreateButton: TToolBarButton97;
begin
  Result := TToolBarButton97.Create(Self);
  Result.OnMouseDown := BtnMouseDown;
  Result.OnMouseUp := BtnMouseUp;
  Result.OnMouseEnter := BtnMouseEnter;
  Result.OnMouseExit := BtnMouseExit;
  Result.OnMouseMove := BtnMouseMove;
  Result.OnClick := BtnClick;
  Result.Visible := True;
  Result.Enabled := True;
  Result.Parent := Self;
end;


procedure TSpin97.MouseEntered(Value: Boolean);
begin
  if Value
  then FUpButton.MouseEntered
  else FDownButton.MouseEntered;
end;


procedure TSpin97.MouseLeft(Value: Boolean);
begin
  if Value
  then FUpButton.MouseLeft
  else FDownButton.MouseLeft;
end;


procedure TSpin97.ArrangeButtons;
begin
  if FKind = skVertical
  then begin
         FUpButton.SetBounds(0, 0, Width, (Height div 2));
         FDownButton.SetBounds(0, FUpButton.Height, Width, Height - FUpButton.Height);
         end
  else begin
          FUpButton.SetBounds(0, 0, (Width div 2), Height);
          FDownButton.SetBounds(FUpButton.Width, 0, Width - FUpButton.Width, Height);
         end;
end;


procedure TSpin97.WMSize(var Message: TWMSize);
begin
  inherited;
  ArrangeButtons;
end;


procedure TSpin97.SetKind(Value: TSpin97Kind);
begin
  if FKind <> Value
  then begin
         FKind := Value;
         ArrangeButtons;
         end;
end;


procedure TSpin97.SetFlat(Value: Boolean);
begin
  if FFlat <> Value
  then begin
         FFlat := Value;
         FUpButton.Flat := Value;
         FDownButton.Flat := Value;
         end;
end;


procedure TSpin97.SetNoBorder(Value: Boolean);
begin
  if FNoBorder <> Value
  then begin
         FNoBorder := Value;
         FUpButton.NoBorder := Value;
         FDownButton.NoBorder := Value;
         end;
end;


procedure TSpin97.BtnMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  if ((FocusControl <> nil) and (not FocusControl.Focused))
  then try FocusControl.SetFocus; except; raise; end;

  if Repeating
  then begin
       FActiveButton := Sender as TToolbarButton97;
       FirstTime := True;
       with FActiveButton do begin
                               RepeatInterval := FRepeatInterval;
                               RepeatDelay := FRepeatDelay;
                               Repeating := True;
                               end;
       end;
end;


procedure TSpin97.BtnMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  if Repeating
  then with FActiveButton do Repeating := False;
end;


procedure TSpin97.BtnClick(Sender: TObject);
begin
  if not FirstTime
  then begin
         if Sender=FUpButton
         then begin
                if Assigned(FOnUpClick)
                then FOnUpClick(Self)
                end
         else if Assigned(FOnDownClick)
              then FOnDownClick(Self);
         end;
  FirstTime := False;
end;


procedure TSpin97.BtnMouseEnter(Sender: TObject);
begin
  if Assigned(FOnMouseEnter)
  then FOnMouseEnter(Self);
end;


procedure TSpin97.BtnMouseExit(Sender: TObject);
begin
  if Assigned(FOnMouseExit)
  then FOnMouseExit(Self);
end;


procedure TSpin97.BtnMouseMove(Sender: TObject; Shift: TShiftState; X,
  Y: Integer);
begin
  if Assigned(FOnMouseMove)
  then FOnMouseMove(Self, Shift, X, Y);
end;


function TSpin97.GetVersion: String;
begin
  Result := TB97AddOnPropText;
end;


procedure TSpin97.SetVersion (const Value: String);
begin
  // just for syntax
end;

////////////////////////////////////////////////////////////////////////////////
// TSpin97
// Procedures & Functions for UP-Button
////////////////////////////////////////////////////////////////////////////////

function TSpin97.GetUpCaption: TCaption;
begin
  Result := FUpButton.Caption;
end;


procedure TSpin97.SetUpCaption(Value: TCaption);
begin
  FUpButton.Caption := Value;
end;


function TSpin97.GetUpGlyph: TBitmap;
begin
  Result := FUpButton.Glyph;
end;


procedure TSpin97.SetUpGlyph(Value: TBitmap);
begin
  FUpButton.Glyph := Value ;
end;


function TSpin97.GetUpLayout: TButtonLayout;
begin
  Result := FUpButton.Layout;
end;

procedure TSpin97.SetUpLayout(Value: TButtonLayout);
begin
  FUpButton.Layout := Value;
end;


function TSpin97.GetUpMargin: Integer;
begin
  Result := FUpButton.Margin;
end;


procedure TSpin97.SetUpMargin(Value: Integer);
begin
  FUpButton.Margin := Value;
end;


function TSpin97.GetUpMask: TBitmap;
begin
  Result := FUpButton.GlyphMask;
end;


procedure TSpin97.SetUpMask(Value: TBitmap);
begin
  FUpButton.GlyphMask := Value;
end;


function TSpin97.GetUpNumGlyphs: TNumGlyphs;
begin
  Result := FUpButton.NumGlyphs;
end;


procedure TSpin97.SetUpNumGlyphs(Value: TNumGlyphs);
begin
  FUpButton.NumGlyphs := Value;
end;


function TSpin97.GetUpSpacing: Integer;
begin
  Result := FUpButton.Spacing;
end;


procedure TSpin97.SetUpSpacing(Value: Integer);
begin
  FUpButton.Spacing := Value;
end;


////////////////////////////////////////////////////////////////////////////////
// TSpin97
// Procedures & Functions for DOWN-Button
////////////////////////////////////////////////////////////////////////////////


function TSpin97.GetDownCaption: TCaption;
begin
  Result := FDownButton.Caption;
end;


procedure TSpin97.SetDownCaption(Value: TCaption);
begin
  FDownButton.Caption := Value;
end;


function TSpin97.GetDownGlyph: TBitmap;
begin
  Result := FDownButton.Glyph;
end;


procedure TSpin97.SetDownGlyph(Value: TBitmap);
begin
  FDownButton.Glyph := Value;
end;


function TSpin97.GetDownLayout: TButtonLayout;
begin
  Result := FDownButton.Layout;
end;


procedure TSpin97.SetDownLayout(Value: TButtonLayout);
begin
  FDownButton.Layout := Value;
end;


function TSpin97.GetDownMargin: Integer;
begin
  Result := FDownButton.Margin;
end;

procedure TSpin97.SetDownMargin(Value: Integer);
begin
  FDownButton.Margin := Value;
end;


function TSpin97.GetDownMask: TBitmap;
begin
  Result := FDownButton.GlyphMask;
end;


procedure TSpin97.SetDownMask(Value: TBitmap);
begin
  FDownButton.GlyphMask := Value;
end;


function TSpin97.GetDownNumGlyphs: TNumGlyphs;
begin
  Result := FDownButton.NumGlyphs;
end;


procedure TSpin97.SetDownNumGlyphs(Value: TNumGlyphs);
begin
  FDownButton.NumGlyphs := Value;
end;


function TSpin97.GetDownSpacing: Integer;
begin
  Result := FDownButton.Spacing;
end;


procedure TSpin97.SetDownSpacing(Value: Integer);
begin
  FDownButton.Spacing := Value;
end;



////////////////////////////////////////////////////////////////////////////////
//
//  TSpinEdit97
//
////////////////////////////////////////////////////////////////////////////////


constructor TSpinEdit97.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  FButton := TSpin97.Create(Self);
  with FButton do begin
                    Parent := Self;
                    Visible := True;
                    FocusControl := Parent;
                    OnUpClick := UpClick;
                    OnDownClick := DownClick;
                    OnMouseEnter := ButtonMouseEnter;
                    Repeating := True;
                    Kind := skVertical;
                    end;

  FSeparator := TWinControl.Create(Self);
  with FSeparator do begin
                       Parent := Self;
                       Width := 2;
                       Brush.Color := clWhite;
                       end;
end;


destructor TSpinEdit97.Destroy;
begin
  FButton.Free;
  FSeparator.Free;
  inherited Destroy;
end;


procedure TSpinEdit97.CreateWnd;
begin
  inherited CreateWnd;
  SetEditRect;
end;


procedure TSpinEdit97.SetEditRect;
var
  R: TRect;
begin
  SetRect(R, 0, 0, FSeparator.Left - 1, ClientHeight + 1);
  SendMessage(Handle, EM_SETRECTNP, 0, LongInt(@R));
end;


procedure TSpinEdit97.KeyDown(var Key: Word; Shift: TShiftState);
begin
  case Key of
    VK_UP  : UpClick(Self);
    VK_DOWN: DownClick(Self);
    end;
  inherited KeyDown(Key, Shift);
end;


procedure TSpinEdit97.SetIncrement(Value: Double);
begin
  if FButton.Increment <> Value
  then FButton.Increment := Value;
end;


function TSpinEdit97.GetIncrement: Double;
begin
  Result := FButton.Increment;
end;


procedure TSpinEdit97.SetSpinWidth(Value: Integer);
begin
  if FButton.Width <> Value
  then begin
         FButton.SetBounds(Width - Value - 5, 0, Value, Height - 5);
         RecreateWnd;
         SetEditRect;
         end;
end;


function TSpinEdit97.GetSpinWidth: Integer;
begin
  Result := FButton.Width;
end;


procedure TSpinEdit97.SetSpinKind(Value: TSpin97Kind);
begin
  if FButton.Kind <> Value
  then FButton.Kind := Value;
end;


function  TSpinEdit97.GetSpinKind: TSpin97Kind;
begin
  Result := FButton.Kind;
end;


procedure TSpinEdit97.UpClick (Sender: TObject);
begin
  if ReadOnly
  then Beep
  else Value := Value + FButton.Increment;
end;


procedure TSpinEdit97.DownClick (Sender: TObject);
begin
  if ReadOnly
  then Beep
  else Value := Value - FButton.Increment;
end;



procedure TSpinEdit97.WMSize(var Message: TWMSize);
begin
  inherited;
  if Width < FButton.Width + 4
  then Width := FButton.Width + 4;
  FButton.SetBounds(Width - FButton.Width - 5, 0, FButton.Width, Height - 5);
  FSeparator.Left := FButton.Left - 2;
  FSeparator.Height := FButton.Height;
  SetEditRect;
end;


procedure TSpinEdit97.CMEnter(var Message: TCMEnter);
begin
  inherited;
  MouseInControl := True;
  RedrawButton;
end;


procedure TSpinEdit97.CMExit(var Message: TCMExit);
begin
  inherited;
  RedrawButton;
end;


procedure TSpinEdit97.WMSetFocus (var Message: TWMSetFocus);
begin
  inherited;
  MouseInControl := True;
  RedrawButton;
end;


procedure TSpinEdit97.WMKillFocus (var Message: TWMKillFocus);
begin
  inherited;
  MouseInControl := False;
  RedrawButton;
end;


procedure TSpinEdit97.CMMouseEnter (var Message: TMessage);
begin
  inherited;
  MouseInControl := True;
  RedrawButton;
end;


procedure TSpinEdit97.CMMouseLeave (var Message: TMessage);
begin
  inherited;
  MouseInControl := False;
  RedrawButton;
end;


procedure TSpinEdit97.WMNCPaint (var Message: TMessage);
begin
  inherited;
  RedrawButton;
end;


procedure TSpinEdit97.RedrawButton;
var
  DC: HDC;
  R: TRect;
begin
  DC := GetWindowDC(Handle);
  R := FButton.BoundsRect;
  Inc(R.Top, 2);
  Inc(R.Left);
  Inc(R.Right, 3);
  Inc(R.Bottom, 3);

  if (((MouseBehaviour = mbEasyStyle) or not ControlIs97Control(Screen.ActiveControl))
      and MouseInControl) or Focused or (csDesigning in ComponentState)
  then begin
         FrameRect(DC, R, GetSysColorBrush(COLOR_3DFACE));
         Inc(R.Left);
         DrawEdge(DC, R,BDR_RAISEDINNER, BF_TOPLEFT);
         DrawEdge(DC, R, BDR_RAISEDINNER, BF_BOTTOMRIGHT);
         end
  else begin
         DrawEdge(DC, R, BDR_RAISEDINNER, BF_LEFT);
         Inc(R.Left);
         FrameRect(DC, R, GetSysColorBrush(COLOR_BTNHILIGHT));
         end;
  ReleaseDC(Handle, DC);
end;


procedure TSpinEdit97.ButtonMouseEnter(Sender: TObject);
begin
  FButton.MouseLeft(UP_SPIN_BUTTON);
  FButton.MouseLeft(DOWN_SPIN_BUTTON);
end;




////////////////////////////////////////////////////////////////////////////////
// TSpinEdit97
// Procedures & Functions for UP-Button
////////////////////////////////////////////////////////////////////////////////


function TSpinEdit97.GetUpCaption: TCaption;
begin
  Result := FButton.UpCaption;
end;


procedure TSpinEdit97.SetUpCaption(Value: TCaption);
begin
  FButton.UpCaption := Value;
end;


function TSpinEdit97.GetUpGlyph: TBitmap;
begin
  Result := FButton.UpGlyph;
end;


procedure TSpinEdit97.SetUpGlyph(Value: TBitmap);
begin
  FButton.UpGlyph := Value;
end;


function TSpinEdit97.GetUpLayout: TButtonLayout;
begin
  Result := FButton.UpLayout;
end;

procedure TSpinEdit97.SetUpLayout(Value: TButtonLayout);
begin
  FButton.UpLayout := Value;
end;


function TSpinEdit97.GetUpMargin: Integer;
begin
  Result := FButton.UpMargin;
end;


procedure TSpinEdit97.SetUpMargin(Value: Integer);
begin
  FButton.UpMargin := Value;
end;


function TSpinEdit97.GetUpMask: TBitmap;
begin
  Result := FButton.UpMask;
end;


procedure TSpinEdit97.SetUpMask(Value: TBitmap);
begin
  FButton.UpMask := Value;
end;


function TSpinEdit97.GetUpNumGlyphs: TNumGlyphs;
begin
  Result := FButton.UpNumGlyphs;
end;


procedure TSpinEdit97.SetUpNumGlyphs(Value: TNumGlyphs);
begin
  FButton.UpNumGlyphs := Value;
end;


function TSpinEdit97.GetUpSpacing: Integer;
begin
  Result := FButton.UpSpacing;
end;


procedure TSpinEdit97.SetUpSpacing(Value: Integer);
begin
  FButton.UpSpacing := Value;
end;


////////////////////////////////////////////////////////////////////////////////
// TSpinEdit97
// Procedures & Functions for DOWN-Button
////////////////////////////////////////////////////////////////////////////////


function TSpinEdit97.GetDownCaption: TCaption;
begin
  Result := FButton.DownCaption;
end;


procedure TSpinEdit97.SetDownCaption(Value: TCaption);
begin
  FButton.DownCaption := Value;
end;


function TSpinEdit97.GetDownGlyph: TBitmap;
begin
  Result := FButton.DownGlyph;
end;


procedure TSpinEdit97.SetDownGlyph(Value: TBitmap);
begin
  FButton.DownGlyph := Value;
end;


function TSpinEdit97.GetDownLayout: TButtonLayout;
begin
  Result := FButton.DownLayout;
end;


procedure TSpinEdit97.SetDownLayout(Value: TButtonLayout);
begin
  FButton.DownLayout := Value;
end;


function TSpinEdit97.GetDownMargin: Integer;
begin
  Result := FButton.DownMargin;
end;

procedure TSpinEdit97.SetDownMargin(Value: Integer);
begin
  FButton.DownMargin := Value;
end;


function TSpinEdit97.GetDownMask: TBitmap;
begin
  Result := FButton.DownMask;
end;


procedure TSpinEdit97.SetDownMask(Value: TBitmap);
begin
  FButton.DownMask := Value;
end;


function TSpinEdit97.GetDownNumGlyphs: TNumGlyphs;
begin
  Result := FButton.DownNumGlyphs;
end;


procedure TSpinEdit97.SetDownNumGlyphs(Value: TNumGlyphs);
begin
  FButton.DownNumGlyphs := Value;
end;


function TSpinEdit97.GetDownSpacing: Integer;
begin
  Result := FButton.DownSpacing;
end;


procedure TSpinEdit97.SetDownSpacing(Value: Integer);
begin
  FButton.DownSpacing := Value;
end;


////////////////////////////////////////////////////////////////////////////////
//
//  TCustomLinkEdit97
//
////////////////////////////////////////////////////////////////////////////////


constructor TCustomLinkEdit97.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  FButton := TToolbarButton97.Create(Self);
  with FButton do begin
                    Parent := Self;
                    BtnAllowDown := False;
                    AllowAllUp := True;
                    if (csDesigning in ComponentState)
                    then Glyph.Handle := LoadBitmap(HInstance, 'ELLIPSIS');
                    Width := 15;
                    OnClick := ButtonClick;
                    OnDblClick := ButtonDblClick;
                    OnMouseDown := ButtonMouseDown;
                    OnMouseEnter := ButtonMouseEnter;
                    OnMouseExit := ButtonMouseExit;
                    OnMouseMove := ButtonMouseMove;
                    OnMouseUp := ButtonMouseUp;
                    end;

  FSeparator := TWinControl.Create(Self);
  with FSeparator do begin
                       Parent := Self;
                       Width := 2;
                       Brush.Color := clWhite;
                       end;
end;


destructor TCustomLinkEdit97.Destroy;
begin
  FButton.Free;
  FSeparator.Free;
  inherited Destroy;
end;


procedure TCustomLinkEdit97.CreateWnd;
begin
  inherited CreateWnd;
  SetEditRect;
end;


procedure TCustomLinkEdit97.SetEditRect;
var
  R: TRect;
begin
  if FButton.Visible
  then SetRect(R, 0, 0, FSeparator.Left - 1, ClientHeight + 1)
  else SetRect(R, 0, 0, ClientWidth, ClientHeight + 1);
  SendMessage(Handle, EM_SETRECTNP, 0, LongInt(@R));
end;


procedure TCustomLinkEdit97.WMSetCursor(var Msg: TWMSetCursor);
var P: TPoint;
begin
  GetCursorPos(P);
  if PtInRect( Rect(FSeparator.Left, 0, ClientWidth, ClientHeight), ScreenToClient(P))
  and FButton.Visible
  then Windows.SetCursor(LoadCursor(0, idc_Arrow))
  else inherited;
end;


procedure TCustomLinkEdit97.WMSize(var Message: TWMSize);
begin
  inherited;
  if Width < FButton.Width + 4
  then Width := FButton.Width + 4;
  FButton.SetBounds(Width - FButton.Width - 5, 0, FButton.Width, Height - 5);
  FSeparator.Left := FButton.Left - 2;
  FSeparator.Height := FButton.Height;// + 4;
  SetEditRect;
end;


procedure TCustomLinkEdit97.CMEnter(var Message: TCMEnter);
begin
  inherited;
  MouseInControl := True;
  RedrawButton;
end;


procedure TCustomLinkEdit97.CMExit(var Message: TCMExit);
begin
  inherited;
  RedrawButton;
end;


procedure TCustomLinkEdit97.WMSetFocus (var Message: TWMSetFocus);
begin
  inherited;
  MouseInControl := True;
  RedrawButton;
end;


procedure TCustomLinkEdit97.WMKillFocus (var Message: TWMKillFocus);
begin
  inherited;
  MouseInControl := False;
  RedrawButton;
end;


procedure TCustomLinkEdit97.CMMouseEnter (var Message: TMessage);
begin
  inherited;
  MouseInControl := True;
  RedrawButton;
end;


procedure TCustomLinkEdit97.CMMouseLeave (var Message: TMessage);
begin
  inherited;
  MouseInControl := False;
  RedrawButton;
end;


procedure TCustomLinkEdit97.WMNCPaint (var Message: TMessage);
begin
  inherited;
  RedrawButton;
end;


procedure TCustomLinkEdit97.RedrawButton;
var
  DC: HDC;
  R: TRect;
begin
  DC := GetWindowDC(Handle);
  R := FButton.BoundsRect;
  Inc(R.Top, 2);
  Inc(R.Left);
  Inc(R.Right, 3);
  Inc(R.Bottom, 3);

  if FButton.Visible
  then begin
    if (((MouseBehaviour = mbEasyStyle) or not ControlIs97Control(Screen.ActiveControl))
      and MouseInControl) or Focused or (csDesigning in ComponentState)
    then begin
           FrameRect(DC, R, GetSysColorBrush(COLOR_3DFACE));
           Inc(R.Left);
           DrawEdge(DC, R,BDR_RAISEDINNER, BF_TOPLEFT);
           DrawEdge(DC, R, BDR_RAISEDINNER, BF_BOTTOMRIGHT);
           end
    else begin
           DrawEdge(DC, R, BDR_RAISEDINNER, BF_LEFT);
           Inc(R.Left);
           FrameRect(DC, R, GetSysColorBrush(COLOR_BTNHILIGHT));
           end;
    end;
  ReleaseDC(Handle, DC);
end;


////////////////////////////////////////////////////////////////////////////////
// TCustomLinkEdit97
// Events for FButton
////////////////////////////////////////////////////////////////////////////////

procedure TCustomLinkEdit97.ButtonClick(Sender: TObject);
begin
  FButton.Invalidate;
  if Assigned(FButtonClick)
  then FButtonClick(Self);
end;


procedure TCustomLinkEdit97.ButtonDblClick(Sender: TObject);
begin
  if Assigned(FButtonDblClick)
  then FButtonDblClick(Self);
end;


procedure TCustomLinkEdit97.ButtonMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
begin
  if not Focused
  then SetFocus;
  if Assigned(FButtonMouseDown)
  then FButtonMouseDown(Self, Button, Shift, X, Y);
end;


procedure TCustomLinkEdit97.ButtonMouseEnter(Sender: TObject);
begin
  FButton.Mouseleft;
  if Assigned(FButtonMouseEnter)
  then FButtonMouseEnter(Self);
end;


procedure TCustomLinkEdit97.ButtonMouseExit(Sender: TObject);
begin
  if Assigned(FButtonMouseExit)
  then FButtonMouseExit(Self);
end;


procedure TCustomLinkEdit97.ButtonMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
begin
  if Assigned(FButtonMouseMove)
  then FButtonMouseMove(Self, Shift, X, Y);
end;


procedure TCustomLinkEdit97.ButtonMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
begin
  if Assigned(FButtonMouseUp)
  then FButtonMouseUp(Self, Button, Shift, X, Y);
end;


////////////////////////////////////////////////////////////////////////////////
// TCustomLinkEdit97
// Procedures & Functions for FButton
////////////////////////////////////////////////////////////////////////////////


procedure TCustomLinkEdit97.SetBtnAllowDown(Value : Boolean);
begin
  if FBtnAllowDown <> Value
  then begin
         FBtnAllowDown := Value;
         if FBtnAllowDown then FButton.GroupIndex := 1
                          else FButton.GroupIndex := 0;
         end;
end;


function TCustomLinkEdit97.GetBtnCaption: TCaption;
begin
  Result := FButton.Caption;
end;


procedure TCustomLinkEdit97.SetBtnCaption(Value: TCaption);
begin
  FButton.Caption := Value;
end;


function TCustomLinkEdit97.GetBtnGlyph: TBitmap;
begin
  Result := FButton.Glyph;
end;


procedure TCustomLinkEdit97.SetBtnGlyph(Value: TBitmap);
begin
  FButton.Glyph := Value;
end;


function TCustomLinkEdit97.GetBtnLayout: TButtonLayout;
begin
  Result := FButton.Layout;
end;

procedure TCustomLinkEdit97.SetBtnLayout(Value: TButtonLayout);
begin
  FButton.Layout := Value;
end;


function TCustomLinkEdit97.GetBtnMargin: Integer;
begin
  Result := FButton.Margin;
end;


procedure TCustomLinkEdit97.SetBtnMargin(Value: Integer);
begin
  FButton.Margin := Value;
end;


function TCustomLinkEdit97.GetBtnMask: TBitmap;
begin
  Result := FButton.GlyphMask;
end;


procedure TCustomLinkEdit97.SetBtnMask(Value: TBitmap);
begin
  FButton.GlyphMask := Value;
end;


function TCustomLinkEdit97.GetBtnNumGlyphs: TNumGlyphs;
begin
  Result := FButton.NumGlyphs;
end;


procedure TCustomLinkEdit97.SetBtnNumGlyphs(Value: TNumGlyphs);
begin
  FButton.NumGlyphs := Value;
end;


function TCustomLinkEdit97.GetBtnSpacing: Integer;
begin
  Result := FButton.Spacing;
end;


procedure TCustomLinkEdit97.SetBtnSpacing(Value: Integer);
begin
  FButton.Spacing := Value;
end;


function TCustomLinkEdit97.GetBtnVisible: Boolean;
begin
  Result := FButton.Visible;
end;


procedure TCustomLinkEdit97.SetBtnVisible(Value:Boolean);
begin
  If FButton.Visible <> Value
  then begin
         FButton.Visible := Value;
         FSeparator.Visible := Value;
         RecreateWnd;
         SetEditRect;
         end;
end;


function TCustomLinkEdit97.GetBtnWidth: Integer;
begin
  Result := FButton.Width;
end;


procedure TCustomLinkEdit97.SetBtnWidth(Value: Integer);
begin
  if FButton.Width <> Value
  then begin
         FButton.SetBounds(Width - Value - 5, 0, Value, Height - 5);
         RecreateWnd;
         SetEditRect;
         end;
end;


procedure TCustomLinkEdit97.BtnUp;
begin
  FButton.Down := False;
end;


procedure TCustomLinkEdit97.BtnDown;
begin
   FButton.Down := True;
end;


////////////////////////////////////////////////////////////////////////////////
//
//  UTILITY ROUTINE: calculate the height of a Font
//                   used by TCustomComboBox97 and TFontComboBox97
//
////////////////////////////////////////////////////////////////////////////////


function GetFontHeight(Font: TFont): Integer;
var
  DC: HDC;
  SaveFont: HFont;
  Metrics: TTextMetric;
begin
  DC := GetDC(0);
  try
    SaveFont := SelectObject(DC, Font.Handle);
    GetTextMetrics(DC, Metrics);
    SelectObject(DC, SaveFont);
  finally
    ReleaseDC(0, DC);
  end;
  GetFontHeight := Metrics.tmHeight;// + 1;
end;


///////////////////////////////////////////////////////////////////////////////
//
//  TCustomComboBox97
//
//  TCBListView
//
////////////////////////////////////////////////////////////////////////////////



procedure TCBListView.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  with Params do
  begin
    ExStyle :=  WS_EX_TOOLWINDOW or WS_EX_TOPMOST or WS_EX_STATICEDGE;
    WindowClass.Style := CS_SAVEBITS;
    end;
end;


procedure TCBListView.CreateWnd;
begin
  inherited CreateWnd;
  Windows.SetParent(Handle, 0);
end;


procedure TCBListView.CMMouseEnter(var Message: TMessage);
begin
  inherited;
  FMouseInControl := True;
  ReleaseCapture;
end;


procedure TCBListView.CMMouseLeave(var Message: TMessage);
begin
  inherited;
  FMouseInControl := False;
  SetCaptureControl(Self);
end;


procedure TCBListView.WMMouseMove(var Message: TWMMouseMove);
var
  ListHitTest: Integer;
begin
  inherited;
  if FMouseInControl and Enabled
  then begin
         ListHitTest := ItemAtPos(Point(Message.XPos, Message.YPos), False);
         if (ListHitTest <> -1)
         then ItemIndex := ListHitTest;
         end;
end;


procedure TCBListView.WMLButtonDown(var Message: TWMLButtonDown);
var
  HitTest: Integer;
begin
  inherited;
  case FMouseInControl of
    False: begin
             ReleaseCapture;
             Enabled := False;
             Visible := False;
             end;
    True: begin
            SendMessage((Owner.Owner as TWinControl).Handle, WM_LBUTTONDOWN, 0, 0);
            HitTest := ItemAtPos(Point(Message.XPos, Message.YPos), False);
            if HitTest<>-1
            then begin
                   ItemIndex := HitTest;
                   (Owner as TCustomComboBox97).BtnUp;
                   (Owner as TCustomComboBox97).Text := Items[ItemIndex];
                   (Owner as TCustomComboBox97).SelectAll;
                   Enabled := False;
                   Visible := False;
                   end;
            end;
  end;
  (Owner as TCustomComboBox97).BtnUp;
end;


////////////////////////////////////////////////////////////////////////////////
//
//  TCustomComboBox97
//
////////////////////////////////////////////////////////////////////////////////

constructor TCustomComboBox97.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  BtnGlyph.Handle := LoadBitmap(HInstance, 'DOWNBITMAP');

  BtnAllowDown := True;
  FDropDownCount := 8;

  OnButtonClick:=BtnClick;

  List := TCBListView.Create(Self);
  List.Parent := Self;
  with List do
  begin
    Left := 0;
    Top := 0;
    Width := 0;
    Enabled := False;
    Visible := False;
  end;

  FCanvas := TControlCanvas.Create;
  FCanvas.Control := List;

end;


destructor TCustomComboBox97.Destroy;
begin
  FCanvas.Free;
  List.Free;
  inherited Destroy;
end;


procedure TCustomComboBox97.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  Params.Style := Params.Style or ES_MULTILINE or WS_CLIPCHILDREN;
end;


procedure TCustomComboBox97.BtnClick(Sender: TObject);
var
  CP, SP: TPoint;
  DropHeight: Integer;
begin
  inherited;
  if Assigned(FOnDropDown)
  then FOnDropDown(Self);

  if FButton.Down
  then begin
         CP.X := Left;
         CP.Y := Top + Height;
         SP := Parent.ClientToScreen(CP);

         with List do begin
                        Left := SP.X;
                        Top := SP.Y;
                        Width := Self.Width;
                        end;

         if FDropDownCount > List.Items.Count
           then DropHeight := List.Items.Count
           else DropHeight := FDropDownCount;

         case List.Style of
       lbStandard         : List.Height := DropHeight * GetFontHeight(Font) + 2;
       lbOwnerDrawFixed,
       lbOwnerDrawVariable: List.Height := DropHeight * ItemHeight + 2;
         end;

         List.Enabled := True;
         List.Visible := True;
         end
  else begin
         List.Enabled := False;
         List.Visible := False;
         end;
end;


function TCustomComboBox97.GetItems: TStrings;
begin
  Result := List.Items;
end;


procedure TCustomComboBox97.SetItems(Value: TStrings);
begin
  List.Items := Value;
end;

function TCustomComboBox97.GetItemIndex: Integer;
begin
  Result := List.ItemIndex;
end;


procedure TCustomComboBox97.SetItemIndex(Value: Integer);
begin
  if GetItemIndex <> Value
  then List.ItemIndex := Value;
end;


function TCustomComboBox97.GetSorted: Boolean;
begin
  Result := List.Sorted;
end;


procedure TCustomComboBox97.SetSorted(Value: Boolean);
begin
  List.Sorted := Value;
end;


function TCustomComboBox97.GetItemHeight: Integer;
begin
  Result := List.ItemHeight;
end;


procedure TCustomComboBox97.SetItemHeight(Value: Integer);
begin
  List.ItemHeight := Value;
end;


function TCustomComboBox97.GetListVisible: Boolean;
begin
  Result := List.Visible;
end;


procedure TCustomComboBox97.SetListVisible(Value: Boolean);
begin
  List.Visible := Value;
end;


procedure TCustomComboBox97.ListOnDrawItem(Control: TWinControl; Index: Integer;
  Rect: TRect; State: TOwnerDrawState);
begin
  if Assigned(FOnDrawItem)
  then FOnDrawItem(Control, Index, Rect, State);
end;


procedure TCustomComboBox97.ListOnMeasureItem(Control: TWinControl; Index: Integer;
  var Height: Integer);
begin
  if Assigned(FOnMeasureItem)
  then FOnMeasureItem(Control, Index, Height);
end;


procedure TCustomComboBox97.SetStyle(Value: TComboBoxStyle);
begin
  if FStyle <> Value
  then begin
    FStyle := Value;
    case FStyle of
     csSimple:         begin
                         BtnVisible := False;
                         EditorEnabled := True;
                         KeyboardFilter := kfNone;
                         List.OnDrawItem := nil;
                         List.OnMeasureItem := nil;
                         List.Style := lbStandard;
                         end;

     csDropDown:       begin
                         BtnVisible := True;
                         EditorEnabled := True;
                         KeyboardFilter := kfNone;
                         List.OnDrawItem := nil;
                         List.OnMeasureItem := nil;
                         List.Style := lbStandard;
                         end;

     csDropDownList:   begin
                         BtnVisible := True;
                         EditorEnabled := False;
                         KeyboardFilter := kfAll;
                         List.OnDrawItem := nil;
                         List.OnMeasureItem := nil;
                         List.Style := lbStandard;
                         end;

     csOwnerDrawFixed: begin
                         BtnVisible := True;
                         EditorEnabled := True;
                         KeyboardFilter := kfNone;
                         List.OnDrawItem := ListOnDrawItem;
                         List.OnMeasureItem := nil;
                         List.Style := lbOwnerDrawFixed;
                         end;

  csOwnerDrawVariable: begin
                         BtnVisible := True;
                         EditorEnabled := True;
                         KeyboardFilter := kfNone;
                         List.OnDrawItem := ListOnDrawItem;
                         List.OnMeasureItem := ListOnMeasureItem;
                         List.Style := lbOwnerDrawVariable;
                         end;
    end;
    end;
end;


procedure TCustomComboBox97.KeyDown(var Key: Word; Shift: TShiftState);
var NeedTextUpdate: Boolean;
begin
  NeedTextUpdate := False;

  case Key of
   vk_Return: begin
                ReleaseCapture;
                List.Enabled := False;
                List.Hide;
                BtnUp;
                NeedTextUpdate := True;
                end;
   vk_Escape:
              begin
                ReleaseCapture;
                List.Enabled := False;
                List.Hide;
                BtnUp;
                end;
   vk_Prior:  begin
                List.ItemIndex := 0;
                NeedTextUpdate := True;
                end;
   vk_Next:   begin
                List.ItemIndex := List.Items.Count - 1;
                NeedTextUpdate := True;
                end;
   vk_Up:     begin
                if (Shift = [ssAlt]) and (FStyle <> csSimple)
                then begin
                       ReleaseCapture;
                       List.Enabled := False;
                       List.Hide;
                       BtnUp;
                       end
                else begin
                       if List.ItemIndex = - 1
                       then List.ItemIndex := 0
                       else if List.ItemIndex <> 0
                            then List.ItemIndex := List.ItemIndex - 1;
                       NeedTextUpdate := True;
                       end;
                end;
   vk_Down:   begin
                if (Shift = [ssAlt]) and (FStyle <> csSimple)
                then begin
                       BtnDown;
                       BtnClick(Self);
                       end
                else begin
                       if List.ItemIndex = - 1
                       then List.ItemIndex := 0
                       else if List.ItemIndex <> List.Items.Count - 1
                            then List.ItemIndex := List.ItemIndex + 1;
                       NeedTextUpdate := True;
                       end;
                end;
   end;

  if NeedTextUpdate and (List.ItemIndex <> -1)
  then begin
         try Text := List.Items[List.ItemIndex];
             FItemsIndex := List.ItemIndex;
         except; end;
         end;
end;



procedure TCustomComboBox97.WMKillFocus (var Message: TWMKillFocus);
begin
  inherited;
  if FButton.Down then begin
                         ReleaseCapture;
                         List.Enabled := False;
                         List.Hide;
                         BtnUp;
                         end;
end;


////////////////////////////////////////////////////////////////////////////////
//
//  TFontComboBox97
//
////////////////////////////////////////////////////////////////////////////////

type
  TEnumFontData = record
    ListItems    : TStrings;
    OnlyTrueType : Boolean;
    end;

var
  GlobalFontData : TEnumFontData;



constructor TFontComboBox97.Create(AOwner : TComponent);
begin
  inherited Create(AOwner);
  TrueTypeSign := TBitmap.Create;
  TrueTypeSign.Handle := LoadBitmap(hInstance,'TRUETYPESIGN');
end;


procedure TFontComboBox97.CreateWnd;
begin
  inherited CreateWnd;
  Sorted := True;
  OnDrawItem := DrawFontItem;
  Style := csOwnerDrawFixed;
  CreateFontList;
  PurgeFontList;
  ItemHeight := GetFontHeight(Font);
end;


destructor TFontComboBox97.Destroy;
begin
  TrueTypeSign.Free;
  inherited Destroy;
end;


function MyFontEnumProc(lpelfe: PEnumLogFontEx; P: Pointer;
                        FontType,LParam: Integer): Integer; StdCall;
begin
  with GlobalFontData,ListItems
  do begin
     if FontType = TrueType_FontType
            then Add(lpelfe^.elfLogFont.lfFaceName + '*')
            else if not OnlyTrueType then Add(lpelfe^.elfLogFont.lfFaceName);
     end;
  Result := 1;
end;


procedure TFontComboBox97.CreateFontList;
var
  DC: hDC;
  LF: TLogFont;
begin
  Items.Clear;
  with LF do begin
               lfCharSet := Default_CharSet;
               lfFaceName := '';
               lfPitchAndFamily := 0;
               end;
  with GlobalFontData do begin
                           ListItems := Items;
                           OnlyTrueType := (foTTOnly in Options);
                           end;
  DC := GetDC(0);
  EnumFontFamiliesEx(DC, LF, @MyFontEnumProc, 0, 0);
  ReleaseDC(0, DC);
end;


procedure TFontComboBox97.PurgeFontList;
var
  i: Integer;
begin
  i:=0;
  repeat
    if Items[i] = Items[i+1]
    then begin
           Items.Delete(i+1);
           Dec(i);
           end;
    Inc(i);
   until i >= Items.Count - 2;
end;


procedure TFontComboBox97.Change;
var S: String;
begin
  S := Text;
  if Pos('*',S) <> 0
  then begin
         Delete(S, Length(S), 1);
         Text := S;
         end;
  if Assigned(FOnChange)
  then FOnChange(Self);
end;


procedure TFontComboBox97.SetOptions(Value: TFontComboBox97Options);
begin
  if Value <> Options
  then begin
         FOptions := Value;
         CreateFontList;
         PurgeFontList;
         ItemHeight := GetFontHeight(Font);
         end;
end;


procedure TFontComboBox97.DrawFontItem(Control : TWinControl; Index : Integer;
                                       Rect : TRect; State : TOwnerDrawState);
var S: string;
begin
  with (Control as TListBox).Canvas
  do begin
       FillRect(Rect);
       S := Items[Index];
       if State =[odSelected]
       then DrawFocusRect(Rect);
       if Pos('*',S) <> 0
       then begin
              Delete(S, Length(S), 1);
              BrushCopy(Bounds(Rect.Left + 2, (Rect.Top + Rect.Bottom - TrueTypeSign.Height) div 2,
                        TrueTypeSign.Width, TrueTypeSign.Height),TrueTypeSign,
                        Bounds(0, 0, TrueTypeSign.Width, TrueTypeSign.Height),
                        TrueTypeSign.TransparentColor);
              end;

       if foUseItemFont in Options
       then Font.Name := S;

       TextOut(Rect.Left + TrueTypeSign.Width + 4, Rect.Top, S)
       end;
end;


////////////////////////////////////////////////////////////////////////////////

procedure Register;

////////////////////////////////////////////////////////////////////////////////


begin
  RegisterComponents('Tb97AddOn', [TPlusEdit97, TNumEdit97, TSpin97, TSpinEdit97,
                                   TLinkEdit97, TComboBox97, TFontComboBox97]);
end;


////////////////////////////////////////////////////////////////////////////////
end.
////////////////////////////////////////////////////////////////////////////////
