unit SysInfo;

(*******************************************************************************

 Name:     SysInfo.Pas
 Project:  TSysInfo component stuff
 Version:	 1.4
 Coder:    Christophe Porteneuve aka TDD
					 URL: http://www.epita.fr/~porten_c - E-mail: porten_c@epita.fr

 Started:  12/15/96
 Effici.:  12/15/96
 Latest:   03/19/98

 ** USE THE APPROPRIATE DFM/PAS FILE FOR F_SysInf WITH IT! **

 ******************************************************************************)

interface

uses
	Windows, Messages, SysUtils, Classes, Graphics, Forms, DsgnIntf;

const
	VERSION_ID = '1.4';

type
	TLanguage  = (lgFrench, lgEnglish, lgItalian);
	TSpaceUnit = (suByte, suKB, suMB);

	TInfoStr	= (isVer32s, isVer9532, isVerNT, isUnknown, isArchBlocked, isModel,
							 isStepPass, isNoRoot, isRemovable, isFixed, isRemote, isCDRom,
							 isRamDisk, isByte, isKB, isMB);

	TUniqStr  = (usPAIntel, usPAMips, usPAAlpha, usPAPPC, usI386, usI486, usI586,
							 usR4000, usA21046, usK386, usK486, usK586, usK4000, usK21064,
							 usK21066, usK21164, usK601, usK603, usK604, usK603p, usK604p,
							 usK620);

	// TDiskInfo

	TDiskInfo = class(TPersistent)
	private
		FGotErrors: boolean;
		FBPS, FSPC: DWORD;
		FDFS, FDTS: string;
		FLang:			TLanguage;
		FSpaceUnit: TSpaceUnit;
		procedure SetLanguage(Value: TLanguage);
		procedure SetSpaceUnit(Value: TSpaceUnit);
		procedure InitDiskInfo;
	public
		constructor Create;
		procedure		Update;
		property GotErrors: boolean 	    read FGotErrors;
		property BytesPerSector: DWORD 		read FBPS;
		property SectorsPerCluster: DWORD read FSPC;
		property DiskFreeSpace: string		read FDFS;
		property DiskTotalSpace: string		read FDTS;
	published
		property Language: TLanguage 			read FLang write SetLanguage
																			default lgFrench;
		property SpaceUnit: TSpaceUnit    read FSpaceUnit write SetSpaceUnit
																			default suMB;
	end; // TDiskInfo

	// TProcInfo

	TBoolMask = array[0..31] of boolean;
	TProcInfo = class(TPersistent)
	private
		FArchi: string;
		FAPrcs: TBoolMask;
		FNb: 		word;
		FKind: 	string;
		FLang:	TLanguage;
		FLevel:	string;
		FRev:		string;
		function  GetAPrcs(Index: Integer): boolean;
		procedure	SetLanguage(Value: TLanguage);
		procedure	InitProcessors;
	public
		constructor Create;
		procedure   Update;
		property ActiveProcessors[Index: Integer]: boolean
																	read getAPrcs;
		property Architecture: string read FArchi;
		property Number: 			 word		read FNb;
		property Kind: 				 string	read FKind;
		property Level:				 string	read FLevel;
		property Revision:		 string	read FRev;
	published
		property Language: TLanguage read FLang write SetLanguage default lgFrench;
	end; // TProcInfo

	// TVersionInfo

	TVersionInfo = class(TPersistent)
	private
		FGotErrors: boolean;
		FLang:		  TLanguage;
		FMajor:     word;
		FMinor:	   	word;
		FFull:		 	string;
		FBuild:	   	DWORD;
		FPlatform: 	string;
		procedure InitVersionInfo;
		procedure SetLanguage(Value: TLanguage);
	public
		constructor Create;
		procedure Update;
		property Build:	   	DWORD	 	 read FBuild;
		property Full:		 	string	 read FFull;
		property GotErrors: boolean  read FGotErrors;
		property Major:     word		 read FMajor;
		property Minor:	   	word		 read FMinor;
		property Platform: 	string	 read FPlatform;
	published
		property Language: TLanguage read FLang write SetLanguage default lgFrench;
	end; // TVersionInfo

	// TVolInfo

	TFSFlag  = (ffCasePreserved, ffCaseSensitive, ffUnicodeStored,
							ffFileCompression, ffCompressedVolume);
	TFSFlags = set of TFSFlag;

	TVolInfo = class(TPersistent)
	private
		FGotErrors: boolean;
		FLang:			TLanguage;
		FVolName:  	string;
		FSerial:    string;
		FLongN:     boolean;
		FFSName: 		string;
		FFSInfo: 		TFSFlags;
		procedure InitVolInfo;
		procedure SetLanguage(Value: TLanguage);
	public
		constructor Create;
		procedure   Update;
		property GotErrors:      boolean   read FGotErrors;
		property VolumeName:   	 string	 	 read FVolName;
		property Serial:    		 string	 	 read FSerial;
		property LongNames:      boolean	 read FLongN;
		property FileSystemName: string	 	 read FFSName;
		property FileSystemInfo: TFSFlags  read FFSInfo;
	published
		property Language:			 TLanguage read FLang write SetLanguage
																			 default lgFrench;
	end; // TVolInfo

	// TMemoryInfo

	TMemoryInfo = class(TPersistent)
	private
		FMemLoad: Integer;
		FTotPhys: real;
		FAvlPhys: real;
		FTotPage: real;
		FAvlPage: real;
		FPagRate: Integer;
		FTotVirt: real;
		FAvlVirt: real;
		FSpcUnit: TSpaceUnit;
		procedure InitMemInfo;
		procedure SetSpaceUnit(Value: TSpaceUnit);
	public
		constructor Create;
		procedure   Update;
		property MemoryLoad:    Integer read FMemLoad;
		property TotalPhysical: real 		read FTotPhys;
		property AvailPhysical: real 		read FAvlPhys;
		property TotalPageFile: real 		read FTotPage;
		property AvailPageFile: real 		read FAvlPage;
		property PageFileRatio: Integer read FPagRate;
		property TotalVirtual:  real 		read FTotVirt;
		property AvailVirtual:  real 		read FAvlVirt;
	published
		property SpaceUnit: TSpaceUnit  read FSpcUnit write SetSpaceUnit
																		default suMB;
	end; // TMemoryInfo

	// TSysInfo

	TSysInfo = class(TComponent)
	private
		FCName:		  string;
		FDiskInfo:  TDiskInfo;
		FDrvTypes:	TStringList;
		FInformation: Cardinal;
		FLang:	  	TLanguage;
		FMemInfo:		TMemoryInfo;
		FProcssr: 	TProcInfo;
		FUName:			string;
		FVersion: 	TVersionInfo;
		FVolInfo:	  TVolInfo;
		FWinDir:		string;
		procedure SetLanguage(Value: TLanguage);
	protected
		procedure InitDriveTypes;
		procedure InitNames;
		procedure InitWinDir;
		procedure LittleUpdate;
	public
		constructor Create(AOwner: TComponent); override;
		destructor  Destroy; override;
		procedure   ShowInfo;
		procedure   Update;
		property ComputerName: string    		read FCName;
		property DiskInfo:   	 TDiskInfo	 	read FDiskInfo;
		property DriveTypes: 	 TStringList 	read FDrvTypes;
		property MemoryInfo:   TMemoryInfo 	read FMemInfo;
		property Processors: 	 TProcInfo   	read FProcssr;
		property UserName: 		 string 		  read FUName;
		property Version: 	 	 TVersionInfo read FVersion;
		property VolumeInfo: 	 TVolInfo 	 	read FVolInfo;
		property WinDir: 			 string				read FWinDir;
	published
		property Information:  Cardinal     read FInformation write FInformation;
		property Language: 		 TLanguage   	read FLang write SetLanguage
																				default lgFrench;
	end; // TSysInfo

	TInformationProperty = class(TPropertyEditor)
	public
		function GetAttributes: TPropertyAttributes; override;
		function GetName: string; override;
		function GetValue: string; override;
		procedure Edit; override;
	end; // TInformationProperty

	TInformationEditor = class(TComponentEditor)
	public
		function GetVerbCount: Integer; override;
		function GetVerb(Index: Integer): string; override;
		procedure ExecuteVerb(Index: Integer); override;
		procedure Edit; override;
	end; // TInformationEditor

const
	DS_SIZES:  array[TSpaceUnit] of DWORD = (1, 1024, 1048576);
	INFO_STRS: array[TLanguage, TInfoStr] of string = (
							 ('Windows 3.1x avec Win32s','Windows 95 ou Win32',
								'Windows NT','Introuvable','Introuvable car l''architecture est'+
								' inconnue','Modle','Passe','Racine inexistante','Amovible/'+
								'Disquette','Fixe','Lecteur rseau','Lecteur CD-ROM','Disque '+
								'RAM','octet(s)','Ko','Mo'
							 ),
							 ('Windows 3.1x with Win32s','Windows 95 or Win32',
								'Windows NT','Unknown','Unknown because architecture is missing',
								'Model','Pass','Inexistant root','Removable/Floppy','Fixed',
								'Remote drive','CD-ROM drive','RAM Disk','byte(s)','Kb','Mb'
							 ),
							 ('Windows 3.1x con Win32s','Windows 95 o Win32',
								'Windows NT','Sconosciuto','Architettura sconosciuta',
								'Modello','Num.','Root inesistente','Rimovibile/Floppy','Fisso',
								'Drive remoto','Drive CD','Disco RAM','byte(s)','Kb','Mb'
							 )
						 );
	UNIQ_STRS: array[TUniqStr] of string = (
						 'Intel','MIPS','Alpha','Power PC',
						 'Intel 386','Intel 486','Intel Pentium','MIPS R4000','Alpha 21046',
						 '386','486','Pentium','R4000','21064','21066','21164','601','603',
						 '604','603+','604+','620'
						 );

	PROCESSOR_ARCHITECTURE_INTEL = 0;
	PROCESSOR_ARCHITECTURE_MIPS  = 1; // Assumed
	PROCESSOR_ARCHITECTURE_ALPHA = 2; // Assumed
	PROCESSOR_ARCHITECTURE_PPC	 = 3; // Assumed

	PROCESSOR_INTEL_386     = 386;    // Assumed
	PROCESSOR_INTEL_486			= 486;		// Assumed
	PROCESSOR_INTEL_PENTIUM	= 586;
	PROCESSOR_MIPS_R4000		= 4000; 	// Assumed
	PROCESSOR_ALPHA_21046		= 21046;	// Assumed

procedure Register;

implementation

uses Math, F_SysInf;

//******************************** TDiskInfo ***********************************

constructor TDiskInfo.Create;
begin
	inherited;
	FLang	     := lgFrench;
	FSpaceUnit := suMB;
end; // Create

procedure TDiskInfo.SetLanguage(Value: TLanguage);
begin
	if (FLang = Value) then Exit;
	FLang := Value;
	Update;
end; // SetLanguage

procedure TDiskInfo.SetSpaceUnit(Value: TSpaceUnit);
begin
	if (FSpaceUnit = Value) then Exit;
	FSpaceUnit := Value;
	Update;
end; // SetSpaceUnit

procedure TDiskInfo.InitDiskInfo;
var
	CurDir: string;
	BPC, NOFC, TNOC: DWORD;
	s: string;
begin
	GetDir(0, CurDir);
	CurDir := Copy(CurDir, 1, pos('\', CurDir));
	FGotErrors := not GetDiskFreeSpace(PChar(CurDir), FSPC, FBPS, NOFC, TNOC);
	BPC  := FBPS * FSPC;
	FDFS := Format('%.2f',[BPC*(NOFC/DS_SIZES[FSpaceUnit])]);
	FDTS := Format('%.2f',[BPC*(TNOC/DS_SIZES[FSpaceUnit])]);
	case FSpaceUnit of
		suByte: s := INFO_STRS[FLang, isByte];
		suKB:   s := INFO_STRS[FLang, isKB];
		suMB:   s := INFO_STRS[FLang, isMB];
	end;
	s := ' '+s;
	FDFS := FDFS+s;
	FDTS := FDTS+s;
end; // InitDiskInfo

procedure TDiskInfo.Update;
begin
	InitDiskInfo;
end; // Update

//******************************** TProcInfo ***********************************

constructor TProcInfo.Create;
begin
	inherited;
	FLang := lgFrench;
end; // Create

procedure	TProcInfo.SetLanguage(Value: TLanguage);
begin
	if (FLang = Value) then Exit;
	FLang := Value;
	Update;
end; // SetLanguage

function TProcInfo.getAPrcs(Index: Integer): boolean;
begin
	if (Index in [0..31]) then
		Result := FAprcs[Index]
	else
		raise Exception.Create('Index must be between 0 and 31');
end; // GetAPrcs

procedure TProcInfo.InitProcessors;
var
	PData:   TSystemInfo;
	i:			 Integer;
begin
	GetSystemInfo(PData);
	with PData do
	begin
		// Architecture
		case wProcessorArchitecture of
			PROCESSOR_ARCHITECTURE_INTEL: FArchi := UNIQ_STRS[usPAIntel];
			PROCESSOR_ARCHITECTURE_MIPS:	FArchi := UNIQ_STRS[usPAMips];
			PROCESSOR_ARCHITECTURE_ALPHA: FArchi := UNIQ_STRS[usPAAlpha];
			PROCESSOR_ARCHITECTURE_PPC:		FArchi := UNIQ_STRS[usPAPPC];
			else													FArchi := 'Code '+
																			intToStr(wProcessorArchitecture);
		end;

		// Active processors
		for i := 0 to 31 do
		begin
			FAPrcs[i] := (dwActiveProcessorMask and 1 = 1);
			dwActiveProcessorMask := dwActiveProcessorMask shr 1;
		end;

		// Number of processors
		FNb := dwNumberOfProcessors;

		// Kind of processor
		case dwProcessorType of
			PROCESSOR_INTEL_386:     FKind := UNIQ_STRS[usI386];
			PROCESSOR_INTEL_486:		 FKind := UNIQ_STRS[usI486];
			PROCESSOR_INTEL_PENTIUM: FKind := UNIQ_STRS[usI586];
			PROCESSOR_MIPS_R4000:		 FKind := UNIQ_STRS[usR4000];
			PROCESSOR_ALPHA_21046:	 FKind := UNIQ_STRS[usA21046];
			else										 FKind := 'Standard';
		end;

		// Level of processor
		case wProcessorArchitecture of
			PROCESSOR_ARCHITECTURE_INTEL: case wProcessorLevel of
																			3: FLevel := UNIQ_STRS[usK386];
																			4: FLevel := UNIQ_STRS[usK486];
																			5: FLevel := UNIQ_STRS[usK586];
																			else FLevel := INFO_STRS[FLang,isUnknown];
																		end;
			PROCESSOR_ARCHITECTURE_MIPS:  case wProcessorLevel of
																			4: FLevel := UNIQ_STRS[usK4000];
																			else FLevel := INFO_STRS[FLang,isUnknown];
																		end;
			PROCESSOR_ARCHITECTURE_ALPHA: case wProcessorLevel of
																			21064: FLevel := UNIQ_STRS[usK21064];
																			21066: FLevel := UNIQ_STRS[usK21066];
																			21164: FLevel := UNIQ_STRS[usK21164];
																			else FLevel := INFO_STRS[FLang,isUnknown];
																		end;
			PROCESSOR_ARCHITECTURE_PPC:   case wProcessorLevel of
																			1: FLevel := UNIQ_STRS[usK601];
																			3: FLevel := UNIQ_STRS[usK603];
																			4: FLevel := UNIQ_STRS[usK604];
																			6: FLevel := UNIQ_STRS[usK603p];
																			9: FLevel := UNIQ_STRS[usK604p];
																			20: FLevel := UNIQ_STRS[usK620];
																			else FLevel := INFO_STRS[FLang,isUnknown];
																		end;
			else														FLevel := INFO_STRS[FLang, isArchBlocked];
		end; // case wProcessorArchitecture

		// Revision of processor
		case wProcessorArchitecture of
			PROCESSOR_ARCHITECTURE_INTEL:
			begin
				case dwProcessorType of
					PROCESSOR_INTEL_386,
					PROCESSOR_INTEL_486:
						FRev := chr(Hi(wProcessorRevision)+ord('A'))+
										intToStr(Lo(wProcessorRevision));
					else
						FRev := INFO_STRS[FLang, isModel]+' '+
										intToStr(Hi(wProcessorRevision))+' '+
										INFO_STRS[FLang, isStepPass]+' '+
										intToStr(Lo(wProcessorRevision));
				end; // case dwProcessorType
			end;
			PROCESSOR_ARCHITECTURE_MIPS:
				FRev := intToStr(Lo(wProcessorRevision));
			PROCESSOR_ARCHITECTURE_ALPHA:
				FRev := INFO_STRS[FLang, isModel]+' '+
								chr(Hi(wProcessorRevision)+ord('A'))+' '+
								INFO_STRS[FLang, isStepPass]+' '+
								intToStr(Lo(wProcessorRevision));
			PROCESSOR_ARCHITECTURE_PPC:
				FRev := intToStr(Hi(wProcessorRevision))+'.'+
								intToStr(Hi(wProcessorRevision));
			else
				FRev := 'Code '+intToStr(wProcessorRevision);
		end; // case wProcessorArchitecture
	end; // with PData, FProcssr
end; // InitProcessors

procedure TProcInfo.Update;
begin
	InitProcessors;
end; // TProcInfo

//****************************** TVersionInfo **********************************

constructor TVersionInfo.Create;
begin
	inherited;
	FLang := lgFrench;
end; // Create

procedure	TVersionInfo.SetLanguage(Value: TLanguage);
begin
	if (FLang = Value) then Exit;
	FLang := Value;
	Update;
end; // SetLanguage

procedure TVersionInfo.InitVersionInfo;
var
	OSVersion: TOSVersionInfo;
begin
	with OSVersion do
	begin
		dwOSVersionInfoSize := SizeOf(TOSVersionInfo);
		FGotErrors := not GetVersionEx(OSVersion);
		FMajor     := dwMajorVersion;
		FMinor 	 	 := dwMinorVersion;
		FFull		 	 := IntToStr(Major) + '.' + IntToStr(Minor);
		FBuild 	   := dwBuildNumber;
		case dwPlatformId of
			VER_PLATFORM_WIN32S:         FPlatform := INFO_STRS[FLang, isVer32s];
			VER_PLATFORM_WIN32_WINDOWS:	 FPlatform := INFO_STRS[FLang, isVer9532];
			VER_PLATFORM_WIN32_NT:			 FPlatform := INFO_STRS[FLang, isVerNT];
		end;
	end;
end; // InitVersionInfo

procedure TVersionInfo.Update;
begin
	InitVersionInfo;
end; // Update

//******************************** TVolInfo ************************************

constructor TVolInfo.Create;
begin
	inherited;
	FLang := lgFrench;
end; // Create

procedure TVolInfo.SetLanguage(Value: TLanguage);
begin
	if (FLang = Value) then Exit;
	FLang := Value;
	Update;
end; // SetLanguage

procedure TVolInfo.InitVolInfo;
var
	CurDir: string;
	Volpc,
	FSNpc:  PChar;
	VSN:    PDWORD;
	MCL,FSF,
	Tmp:	  DWORD;

	function BitTest(AVal: DWORD; ABit: DWORD): Boolean;
	begin
		Result := (AVal and ABit = ABit);
	end;

	function HighWord(Orig: DWORD): Word;
	begin
		Result := (Orig and $FFFF0000) shr $10;
	end;

	function LowWord(Orig: DWORD): Word;
	begin
		Result := Orig and $0000FFFF;
	end;

begin // InitVolInfo
	GetDir(0,CurDir);
	CurDir := Copy(CurDir, 1, pos('\', CurDir));
	VolPC := StrAlloc(50);
	FSNpc := StrAlloc(50);
	New(VSN);
	FGotErrors :=
		not GetVolumeInformation(PChar(CurDir),Volpc,50,VSN,MCL,FSF,FSNpc,50);
	FVolName := Volpc;
	try
		Tmp := DWORD(VSN^);
	except
		Tmp := 0;
	end;
	FSerial := IntToHex(HighWord(Tmp),4)+'-'+
						 IntToHex(LowWord(Tmp),4);
	FLongN  := (MCL > 12);
	FFSName := FSNpc;
	FFSInfo := [];
	if BitTest(FSF,FS_CASE_IS_PRESERVED) then
		FFSInfo := FFSInfo + [ffCasePreserved];
	if BitTest(FSF,FS_CASE_SENSITIVE) then
		FFSInfo := FFSInfo + [ffCaseSensitive];
	if BitTest(FSF,FS_UNICODE_STORED_ON_DISK) then
		FFSInfo := FFSInfo + [ffUnicodeStored];
	if BitTest(FSF,FS_FILE_COMPRESSION) then
		FFSInfo := FFSInfo + [ffFileCompression];
	if BitTest(FSF,FS_VOL_IS_COMPRESSED) then
		FFSInfo := FFSInfo + [ffCompressedVolume];
	StrDispose(Volpc);
	StrDispose(FSNpc);
	Dispose(VSN);
end; // InitVolInfo

procedure TVolInfo.Update;
begin
	InitVolInfo;
end; // Update

//****************************** TMemoryInfo ***********************************

constructor TMemoryInfo.Create;
begin
	inherited;
	FSpcUnit := suMB;
end; // Create

procedure TMemoryInfo.SetSpaceUnit(Value: TSpaceUnit);
begin
	if (FSpcUnit = Value) then exit;
	FSpcUnit := Value;
	Update;
end; // SetSpaceUnit

procedure TMemoryInfo.InitMemInfo;
var
	MemSt: TMemoryStatus;
begin
	MemSt.dwLength := SizeOf(MemSt);
	GlobalMemoryStatus(MemSt);
	with MemSt do
	begin
		FMemLoad := dwMemoryLoad;
		FTotPhys := dwTotalPhys/DS_SIZES[FSpcUnit];
		FAvlPhys := dwAvailPhys/DS_SIZES[FSpcUnit];
		FTotPage := dwTotalPageFile/DS_SIZES[FSpcUnit];
		FAvlPage := dwAvailPageFile/DS_SIZES[FSpcUnit];
		FPagRate := trunc((dwTotalPageFile-dwAvailPageFile)/dwTotalPageFile*100);
		FTotVirt := dwTotalVirtual/DS_SIZES[FSpcUnit];
		FAvlVirt := dwAvailVirtual/DS_SIZES[FSpcUnit];
	end;
end; // InitMemInfo

procedure TMemoryInfo.Update;
begin
	InitMemInfo;
end; // Update

//******************************** TSysInfo ************************************

constructor TSysInfo.Create(AOwner: TComponent);
begin
	inherited Create(AOwner);
	FCName     := '';
	FDiskInfo	 := TDiskInfo.Create;
	FDrvTypes  := TStringList.Create;
	FLang    	 := lgFrench;
	FMemInfo	 := TMemoryInfo.Create;
	FProcssr   := TProcInfo.Create;
	FUName	   := '';
	FVersion   := TVersionInfo.Create;
	FVolInfo   := TVolInfo.Create;
	FWinDir		 := '';
	Update;
end; // Create

destructor TSysInfo.Destroy;
begin
	FDiskInfo.Free;
	FDrvTypes.Free;
	FMemInfo.Free;
	FProcssr.Free;
	FVersion.Free;
	FVolInfo.Free;
	inherited;
end; // Destroy

procedure TSysInfo.SetLanguage(Value: TLanguage);
begin
	if (FLang = Value) then Exit;
	FLang := Value;
	FDiskInfo.Language := Value;
	FProcssr.Language  := Value;
	FVersion.Language  := Value;
	FVolInfo.Language	 := Value;
	LittleUpdate;
end; // SetLanguage

procedure TSysInfo.InitDriveTypes;
var
	c: Char;
begin
	with FDrvTypes do
	begin
		Clear;
		for c := 'A' to 'Z' do
		begin
			case GetDriveType(PChar(c+':\')) of
				DRIVE_REMOVABLE:	 Values[c+':'] := INFO_STRS[FLang,isRemovable];
				DRIVE_FIXED:       Values[c+':'] := INFO_STRS[FLang,isFixed];
				DRIVE_REMOTE:      Values[c+':'] := INFO_STRS[FLang,isRemote];
				DRIVE_CDROM:       Values[c+':'] := INFO_STRS[FLang,isCDRom];
				DRIVE_RAMDISK:     Values[c+':'] := INFO_STRS[FLang,isRamDisk];
			end;
		end;
	end;
end; // InitDriveTypes

procedure TSysInfo.InitNames;
var
	Pc: PChar;
	Sz: DWORD;
begin
	Sz := MAX_COMPUTERNAME_LENGTH+1;
	Pc := StrAlloc(Sz);
	try
		// Computer name
		if GetComputerName(Pc, Sz) then
			FCName := Pc
		else
			FCName := INFO_STRS[FLang, isUnknown];
		StrDispose(Pc);
		// User name
		Sz := 50+1;
		Pc := StrAlloc(Sz);
		if GetUserName(Pc, Sz) then
			FUName := Pc
		else
			FUName := INFO_STRS[FLang, isUnknown];
	finally
		StrDispose(Pc);
	end;
end; // InitNames

procedure TSysInfo.InitWinDir;
var
	Pc: PChar;
begin
	Pc := StrAlloc(MAX_PATH);
	GetWindowsDirectory(Pc, MAX_PATH);
	FWinDir := Pc;
	StrDispose(Pc);
end; // InitWinDir

procedure TSysInfo.ShowInfo;
begin
	frmSysInfo := TfrmSysInfo.Create(Application);
	try
		frmSysInfo.FillUp(Self);
		frmSysInfo.ShowModal;
	finally
		frmSysInfo.Free;
	end;
end; // ShowInfo

procedure TSysInfo.Update;
begin
	FDiskInfo.Update;
	InitDriveTypes;
	InitNames;
	FMemInfo.Update;
	FProcssr.Update;
	FVersion.Update;
	FVolInfo.Update;
	InitWinDir;
end; // Update

procedure TSysInfo.LittleUpdate;
begin
	InitDriveTypes;
	InitNames;
	InitWinDir;
end; // LittleUpdate

//***************************** TInformationProperty ***************************

function TInformationProperty.GetAttributes: TPropertyAttributes;
begin
	Result := inherited GetAttributes+[paReadOnly, paDialog];
end; // GetAttributes

function TInformationProperty.GetName: string;
begin
	Result := 'See System Info';
end; // GetName

function TInformationProperty.GetValue: string;
begin
	Result := '(Double-click here)';
end; // GetValue

procedure TInformationProperty.Edit;
begin
	frmSysInfo := TfrmSysInfo.Create(Application);
	try
		frmSysInfo.FillUp(TComponent(GetComponent(0)));
		frmSysInfo.ShowModal;
	finally
		frmSysInfo.Free;
	end;
end; // Edit

//****************************** TInformationEditor ****************************

function TInformationEditor.GetVerbCount: Integer;
begin
	Result := 4;
end; // GetVerbCount

function TInformationEditor.GetVerb(Index: Integer): string;
begin
	case Index of
		0: Result := 'TSysInfo v. '+VERSION_ID;
		1: Result := '(c) 1997,98 by Christophe Porteneuve';
		2: Result := '-';
		3: Result := 'Get System Information...';
	end;
end; // GetVerb

procedure TInformationEditor.ExecuteVerb(Index: Integer);
begin
	if (Index = 3) then
		Edit;
end; // ExecuteVerb

procedure TInformationEditor.Edit;
begin
	frmSysInfo := TfrmSysInfo.Create(Application);
	try
		frmSysInfo.FillUp(Component);
		frmSysInfo.ShowModal;
	finally
		frmSysInfo.Free;
	end;
end; // Edit

procedure Register;
begin
	RegisterComponents('TDDians', [TSysInfo]);
	RegisterPropertyEditor(TypeInfo(Cardinal), TSysInfo, 'Information',
		TInformationProperty);
	RegisterComponentEditor(TSysInfo, TInformationEditor);
end;

end.
