unit SummInfo;

interface

uses Dialogs, SysUtils, Ole2, Oleauto, Windows, classes;

const
  PID_TITLE = $00000002;
  PID_SUBJECT = $00000003;
  PID_AUTHOR = $00000004;
  PID_KEYWORDS = $00000005;
  PID_COMMENTS = $00000006;
  PID_TEMPLATE = $00000007;
  PID_LASTAUTHOR = $00000008;
  PID_REVNUMBER = $00000009;
  PID_EDITTIME = $0000000A;
  PID_LASTPRINTED = $0000000B;
  PID_CRAETE_DTM = $0000000C;
  PID_LASTSAVE_DTM = $0000000D;
  PID_PAGECOUNT = $0000000E;
  PID_WORDCOUNT = $0000000F;
  PID_CHARCOUNT = $00000010;
  PID_THUMBAIL = $00000011;
  PID_APPNAME = $00000012;
  PID_SECURITY = $00000013;

type
  PPropertySetHeader = ^TPropertySetHeader;
  TPropertySetHeader = record
    wByteOrder: Word;   // Always 0xFFFE
    wFormat: Word ;     // Always 0
    dwOSVer: DWORD;     // System version
    clsid: TCLSID;      // Application CLSID
    dwReserved: DWORD;  // Should be 1
  end;

  TFMTID = TCLSID;

  PFormatIDOffset = ^TFormatIDOffset;
  TFormatIDOffset = record
    fmtid: TFMTID;      // Semantic name of a section
    dwOffset: DWORD;    // Offset from start of whole property set
                        // stream to the section
  end;

  PPropertySectionHeader = ^TPropertySectionHeader;
  TPropertySectionHeader = record
    cbSection: DWORD;    // Size of section
    cProperties: DWORD;  // Count of properties in section
  end;

  PPropertyIDOffset = ^TPropertyIDOffset;
  TPropertyIDOffset = record
    propid: DWORD;      // Name of a property
    dwOffset: DWORD;    // Offset from the start of the section to that
                        // property type/value pair
  end;

  PPropertyIDOffsetList = ^TPropertyIDOffsetList;
  TPropertyIDOffsetList = array[0..255] of TPropertyIDOffset;

  PSerializedPropertyValue = ^TSerializedPropertyValue;
  TSerializedPropertyValue = record
    dwType: DWORD;       // Type tag
    prgb: PBYTE;         // The actual property value
  end;

  PSerializedPropertyValueList = ^TSerializedPropertyValueList;
  TSerializedPropertyValueList = array[0..255] of TSerializedPropertyValue;

  PStringProperty = ^TStringProperty;
  TStringProperty = record
    propid: DWORD;
    Value: AnsiString;
  end;

  PIntegerProperty = ^TIntegerProperty;
  TIntegerProperty = record
    propid: DWORD;
    Value: Integer;
  end;

  PFileTimeProperty = ^TFileTimeProperty;
  TFileTimeProperty = record
    propid: DWORD;
    Value: TFileTime;
  end;

  TSummInfo = class(TComponent)
  private
    FActive: Boolean;
    FDocumentName: string;
    FTitle: string;
    FSubject: string;
    FAuthor: string;
    FKeywords: string;
    FComments: string;
    FTemplate: string;
    FLastAuthor: string;
    FRevNumber: string;
    FEditTime: Integer;
    FLastPrinted: TDateTime;
    FCreate_DTM: TDateTime;
    FLastSave_DTM: TDateTime;
    FPageCount: Integer;
    FWordCount: Integer;
    FCharCount: Integer;

    stgOpen: IStorage;
    stm: IStream;
    PropertySetHeader: TPropertySetHeader;
    FormatIDOffset: TFormatIDOffset;
    PropertySectionHeader: TPropertySectionHeader;
    prgPropIDOffset: PPropertyIDOffsetList;
    prgPropertyValue: PSerializedPropertyValueList;

    procedure InternalOpen;
    procedure InternalClose;
    procedure InternalInitPropertyDefs;
    procedure AddProperty(propid: DWORD; Value: Pointer);
    procedure SetActive(Value: Boolean);
    procedure SetDocumentName(AFileName: TFileName);

    function OpenStorage: HResult;
    function OpenStream: HResult;
    function ReadPropertySetHeader: HResult;
    function ReadFormatIdOffset: HResult;
    function ReadPropertySectionHeader: HResult;
    function ReadPropertyIdOffset: HResult;
    function ReadPropertySet: HResult;
    function FileTimeToDateTime(FileTime: TFileTime): TDateTime;
    function FileTimeToElapsedTime(FileTime: TFileTime): Integer;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    procedure Open;
    procedure Close;

    function IsStorageFile: Boolean;
  published
    property Active: Boolean read FActive write SetActive default False;
    property DocumentName: string read FDocumentName write SetDocumentName;
    property Title: string read FTitle;
    property Subject: string read FSubject;
    property Author: string read FAuthor;
    property Keywords: string read FKeywords;
    property Comments: string read FComments;
    property Template: string read FTemplate;
    property LastAuthor: string read FLastAuthor;
    property RevNumber: string read FRevNumber;
    property EditTime: Integer read FEditTime;
    property LastPrinted: TDateTime read FLastPrinted;
    property Create_DTM: TDateTime read FCreate_DTM;
    property LastSave_DTM: TDateTime read FLastSave_DTM;
    property PageCount: Integer read FPageCount;
    property WordCount: Integer read FWordCount;
    property CharCount: Integer read FCharCount;
  end;

procedure Register;

implementation

constructor TSummInfo.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  stgOpen:=nil;
  stm:=nil;
  prgPropIDOffset:=nil;
  prgPropertyValue:=nil;
end;

destructor TSummInfo.Destroy;
begin
  if FActive then InternalClose;
  inherited Destroy;
end;

procedure TSummInfo.Open;
begin
  Active:=True;
end;

procedure TSummInfo.Close;
begin
  Active:=False;
end;

function TSummInfo.IsStorageFile: Boolean;
var
  awcName: array[0..MAX_PATH-1] of WideChar;
begin
  StringToWideChar(DocumentName,awcName,MAX_PATH);
  Result:=StgIsStorageFile(awcName) = S_OK;
end;

procedure TSummInfo.InternalOpen;
begin
  if FDocumentName <> '' then
  begin
    OpenStorage;
    OpenStream;
    InternalInitPropertyDefs;
  end;
end;

procedure TSummInfo.InternalClose;
begin
  if prgPropertyValue <> nil then FreeMem(prgPropertyValue);
  if prgPropIDOffset <> nil then FreeMem(prgPropIDOffset);
  if stm <> nil then stm.Release;
  if stgOpen <> nil then stgOpen.Release;
  stgOpen:=nil;
  stm:=nil;
  prgPropIDOffset:=nil;
  prgPropertyValue:=nil;
end;

procedure TSummInfo.InternalInitPropertyDefs;
begin
  ReadPropertySetHeader;
  ReadFormatIdOffset;
  ReadPropertySectionHeader;
  ReadPropertyIdOffset;
  ReadPropertySet;
end;

function TSummInfo.OpenStorage: HResult;
var
  awcName: array[0..MAX_PATH-1] of WideChar;
begin
  StringToWideChar(DocumentName,awcName,MAX_PATH);
  Result:=StgOpenStorage(awcName,               //Points to the pathname of the file containing storage object
                         nil,                   //Points to a previous opening of a root storage object
                         STGM_READ or           //Specifies the access mode for the object
                         STGM_SHARE_EXCLUSIVE,
                         nil,                   //Points to an SNB structure specifying elements to be excluded
                         0,                     //Reserved; must be zero
                         stgOpen	        //Points to location for returning the storage object
                        );

  OleCheck(Result);
end;

function TSummInfo.OpenStream: HResult;
var
  awcName: array[0..MAX_PATH-1] of WideChar;
begin
  StringToWideChar(#5'SummaryInformation',awcName,MAX_PATH);
  Result:=stgOpen.OpenStream(awcName,               //Points to name of stream to open
                             nil,                   //Reserved; must be NULL
                             STGM_READ or           //Access mode for the new stream
                             STGM_SHARE_EXCLUSIVE,
                             0, 	              //Reserved; must be zero
                             stm	              //Points to opened stream object
                            );

  OleCheck(Result);
end;

function TSummInfo.ReadPropertySetHeader: HResult;
var
  cbRead: Longint;
begin
  Result:=stm.Read(@PropertySetHeader,        //Pointer to buffer into which the stream is read
                   SizeOf(PropertySetHeader), //Specifies the number of bytes to read
                   @cbRead                    //Pointer to location that contains actual number of bytes read
                  );

  OleCheck(Result);
end;

function TSummInfo.ReadFormatIdOffset: HResult;
var
  cbRead: Longint;
begin
  Result:=stm.Read(@FormatIDOffset,        //Pointer to buffer into which the stream is read
                   SizeOf(FormatIDOffset), //Specifies the number of bytes to read
                   @cbRead                 //Pointer to location that contains actual number of bytes read
                  );

  OleCheck(Result);
end;

function TSummInfo.ReadPropertySectionHeader: HResult;
var
  cbRead: Longint;
  libNewPosition: Largeint;
begin
  Result:=Stm.Seek(FormatIDOffset.dwOffset, //Offset relative to dwOrigin
                   STREAM_SEEK_SET,         //Specifies the origin for the offset
                   libNewPosition           //Pointer to location containing new seek pointer
                  );

  OleCheck(Result);

  Result:=stm.Read(@PropertySectionHeader,        //Pointer to buffer into which the stream is read
                   SizeOf(PropertySectionHeader), //Specifies the number of bytes to read
                   @cbRead                        //Pointer to location that contains actual number of bytes read
                  );

  OleCheck(Result);
end;

function TSummInfo.ReadPropertyIdOffset: HResult;
var
  Size: Cardinal;
  cbRead: Longint;
begin
  Size:=PropertySectionHeader.cProperties*SizeOf(prgPropIDOffset^);
  GetMem(prgPropIDOffset,Size);
  Result:=stm.Read(prgPropIDOffset, //Pointer to buffer into which the stream is read
                   Size,            //Specifies the number of bytes to read
                   @cbRead          //Pointer to location that contains actual number of bytes read
                  );

  OleCheck(Result);
end;

function TSummInfo.ReadPropertySet: HResult;
var
  I: Integer;
  Buffer: PChar;
  I4: Integer;
  dwType: DWORD;
  Size: Cardinal;
  cb, cbRead: Longint;
  FileTime: TFileTime;
  dlibMove, libNewPosition: Largeint;
begin
  Result:=S_OK;
  Size:=PropertySectionHeader.cProperties*SizeOf(prgPropertyValue^);
  GetMem(prgPropertyValue,Size);
  for I:=0 to PropertySectionHeader.cProperties-1 do
  begin
    dlibMove:=FormatIDOffset.dwOffset+prgPropIDOffset^[I].dwOffset;
    Result:=Stm.Seek(dlibMove,        //Offset relative to dwOrigin
                     STREAM_SEEK_SET, //Specifies the origin for the offset
                     libNewPosition   //Pointer to location containing new seek pointer
                    );

    OleCheck(Result);

    Result:=stm.Read(@dwType,        //Pointer to buffer into which the stream is read
                     SizeOf(dwType), //Specifies the number of bytes to read
                     @cbRead         //Pointer to location that contains actual number of bytes read
                    );

    OleCheck(Result);

    case dwType of
      VT_EMPTY:               ;{ [V]   [P]  nothing                     }
      VT_NULL:                ;{ [V]        SQL style Null              }
      VT_I2:                  ;{ [V][T][P]  2 byte signed int           }
      VT_I4:                   { [V][T][P]  4 byte signed int           }
      begin
        Result:=stm.Read(@I4,        //Pointer to buffer into which the stream is read
                         SizeOf(I4), //Specifies the number of bytes to read
                         @cbRead     //Pointer to location that contains actual number of bytes read
                        );

        OleCheck(Result);

        AddProperty(prgPropIDOffset^[I].propid,@I4);
      end;
      VT_R4:                  ;{ [V][T][P]  4 byte real                 }
      VT_R8:                  ;{ [V][T][P]  8 byte real                 }
      VT_CY:                  ;{ [V][T][P]  currency                    }
      VT_DATE:                ;{ [V][T][P]  date                        }
      VT_BSTR:                ;{ [V][T][P]  binary string               }
      VT_DISPATCH:            ;{ [V][T]     IDispatch FAR*              }
      VT_ERROR:               ;{ [V][T]     SCODE                       }
      VT_BOOL:                ;{ [V][T][P]  True=-1, False=0            }
      VT_VARIANT:             ;{ [V][T][P]  VARIANT FAR*                }
      VT_UNKNOWN:             ;{ [V][T]     IUnknown FAR*               }

      VT_I1:                  ;{    [T]     signed char                 }
      VT_UI1:                 ;{    [T]     unsigned char               }
      VT_UI2:                 ;{    [T]     unsigned short              }
      VT_UI4:                 ;{    [T]     unsigned short              }
      VT_I8:                  ;{    [T][P]  signed 64-bit int           }
      VT_UI8:                 ;{    [T]     unsigned 64-bit int         }
      VT_INT:                 ;{    [T]     signed machine int          }
      VT_UINT:                ;{    [T]     unsigned machine int        }
      VT_VOID:                ;{    [T]     C style void                }
      VT_HRESULT:             ;{    [T]                                 }
      VT_PTR:                 ;{    [T]     pointer type                }
      VT_SAFEARRAY:           ;{    [T]     (use VT_ARRAY in VARIANT)   }
      VT_CARRAY:              ;{    [T]     C style array               }
      VT_USERDEFINED:         ;{    [T]     user defined type           }
      VT_LPSTR:                {    [T][P]  null terminated string      }
      begin
        Result:=stm.Read(@cb,        //Pointer to buffer into which the stream is read
                         SizeOf(cb), //Specifies the number of bytes to read
                         @cbRead     //Pointer to location that contains actual number of bytes read
                        );

        OleCheck(Result);

        GetMem(Buffer,cb*SizeOf(Char));
        try
          Result:=stm.Read(Buffer, //Pointer to buffer into which the stream is read
                           cb,     //Specifies the number of bytes to read
                           @cbRead //Pointer to location that contains actual number of bytes read
                          );

          OleCheck(Result);

          AddProperty(prgPropIDOffset^[I].propid,Buffer);
        finally
          FreeMem(Buffer);
        end;
      end;
      VT_LPWSTR:              ;{    [T][P]  wide null terminated string }

      VT_FILETIME:             {       [P]  FILETIME                    }
      begin
        Result:=stm.Read(@FileTime,        //Pointer to buffer into which the stream is read
                         SizeOf(FileTime), //Specifies the number of bytes to read
                         @cbRead           //Pointer to location that contains actual number of bytes read
                        );

        OleCheck(Result);

        AddProperty(prgPropIDOffset^[I].propid,@FileTime);
      end;
      VT_BLOB:                ;{       [P]  Length prefixed bytes       }
      VT_STREAM:              ;{       [P]  Name of the stream follows  }
      VT_STORAGE:             ;{       [P]  Name of the storage follows }
      VT_STREAMED_OBJECT:     ;{       [P]  Stream contains an object   }
      VT_STORED_OBJECT:       ;{       [P]  Storage contains an object  }
      VT_BLOB_OBJECT:         ;{       [P]  Blob contains an object     }
      VT_CF:                  ;{       [P]  Clipboard format            }
      VT_CLSID:               ;{       [P]  A Class ID                  }

      VT_VECTOR:              ;{       [P]  simple counted array        }
      VT_ARRAY:               ;{ [V]        SAFEARRAY*                  }
      VT_BYREF:               ;{ [V]                                    }
      VT_RESERVED:            ;
    end;
  end;
end;

procedure TSummInfo.AddProperty(propid: DWORD; Value: Pointer);
var
  FileTime: TFileTime;
begin
  case propid of
    PID_TITLE:
      FTitle:=PChar(Value);
    PID_SUBJECT:
      FSubject:=PChar(Value);
    PID_AUTHOR:
      FAuthor:=PChar(Value);
    PID_KEYWORDS:
      FKeywords:=PChar(Value);
    PID_COMMENTS:
      FComments:=PChar(Value);
    PID_TEMPLATE:
      FTemplate:=PChar(Value);
    PID_LASTAUTHOR:
      FLastAuthor:=PChar(Value);
    PID_REVNUMBER:
      FRevNumber:=PChar(Value);
    PID_EDITTIME:
    begin
      CopyMemory(@FileTime,Value,SizeOf(FileTime));
      FEditTime:=FileTimeToElapsedTime(FileTime);
    end;
    PID_LASTPRINTED:
    begin
      CopyMemory(@FileTime,Value,SizeOf(FileTime));
      FLastPrinted:=FileTimeToDateTime(FileTime);
    end;
    PID_CRAETE_DTM:
    begin
      CopyMemory(@FileTime,Value,SizeOf(FileTime));
      FCreate_DTM:=FileTimeToDateTime(FileTime);
    end;
    PID_LASTSAVE_DTM:
    begin
      CopyMemory(@FileTime,Value,SizeOf(FileTime));
      FLastSave_DTM:=FileTimeToDateTime(FileTime);
    end;
    PID_PAGECOUNT:
      CopyMemory(@FPageCount,Value,SizeOf(FPageCount));
    PID_WORDCOUNT:
      CopyMemory(@FWordCount,Value,SizeOf(FWordCount));
    PID_CHARCOUNT:
      CopyMemory(@FCharCount,Value,SizeOf(FCharCount));
    PID_THUMBAIL: ;
    PID_APPNAME: ;
    PID_SECURITY: ;
  end;
end;

procedure TSummInfo.SetActive(Value: Boolean);
begin
  if FActive <> Value then
  begin
    if Value then InternalOpen
    else InternalClose;
    FActive:=Value;
  end;
end;

procedure TSummInfo.SetDocumentName(AFileName: TFileName);
begin
  if (not FActive) and (FDocumentName <> AFileName) then FDocumentName:=AFileName;
end;

function TSummInfo.FileTimeToElapsedTime(FileTime: TFileTime): Integer;
var
  SystemTime: TSystemTime;
  LocalFileTime: TFileTime;
begin
  Result:=0;
  if FileTimeToLocalFileTime(FileTime, LocalFileTime) and
     FileTimeToSystemTime(LocalFileTime, SystemTime)
  then
    Result:=SystemTime.wMinute;
end;

function TSummInfo.FileTimeToDateTime(FileTime: TFileTime): TDateTime;
var
  FileDate: Integer;
  LocalFileTime: TFileTime;
begin
  Result:=0;
  if FileTimeToLocalFileTime(FileTime, LocalFileTime) and
     FileTimeToDosDateTime(LocalFileTime,
                           LongRec(FileDate).Hi, LongRec(FileDate).Lo)
  then
    try Result:=FileDateToDateTime(FileDate); except Result:=0; end;
end;

procedure Register;
begin
  RegisterComponents('SummInfo', [TSummInfo]);
end;

end.
