{
      Dmitry Streblechenko
      Department of Physics and Astronomy
      Arizona State University
      Box 871504
      Tempe, AZ 85287-1504

      E-mail: dmitrys@asu.edu
}

unit dsstream;

interface

uses Classes,Windows,SysUtils;

type

 TMemoryMappedFileStream = class (TStream)
  private
   FFileHandle:THandle;
   FMapHandle:THandle;
   FFileName:string;
   FMemory:pointer;
   FCapacity,FSize,FPosition:longint;
   FMode:integer;
   //Name of the file mapping object
   FMapName:string;
   //file size increment
   FMemoryDelta:longint;
   procedure SetCapacity(NewCapacity:longint);
   procedure CloseHandles;
  public
   constructor Create(const FileName, MapName: string; Mode: Word);
   constructor Open(const MapName:string; Mode:word);
   destructor Destroy; override;
   function Read(var Buffer;Count:longint):longint;override;
   function Write(const Buffer;Count:longint):longint;override;
   procedure SetSize(NewSize:longint);override;
   function Seek(Offset:longint;Origin:word):longint;override;
   procedure Flush;virtual;
   property Memory:pointer read FMemory;
   property FileName:string read FFileName;
   property MapName:string read FMapName;
   property MemoryDelta:longint read FMemoryDelta write FMemoryDelta;
 end;

 PRequest = ^TRequest;
 TRequest = record
  Overlapped:TOverlapped; //Overlapped structure
  Count:integer;          //Number of bytes in the Buffer^
  Buffer:pointer;         //Data
 end;

 TRequestsArray=array[0..$77777777 div SizeOf(PRequest)] of PRequest;

 TAsyncFileStream = class(THandleStream)
 private
 protected
  FCapacity:integer;
  FNumRequests:integer;
  FRequests:^TRequestsArray;
  FPosition:integer;
  FSize:integer;
  procedure AddRequest(Request:PRequest);
  function IsRequestPending(Request:TRequest):boolean;
  procedure DeleteRequest(var Request:PRequest);
 public
  CapacityDelta:integer;
  constructor Create(const FileName: string; Mode: Word);
  destructor Destroy; override;
  function Read(var Buffer;Count:longint):longint;override;
  function Write(const Buffer;Count:longint):longint;override;
  procedure SetSize(NewSize:longint);override;
  function Seek(Offset: Longint; Origin: Word): Longint; override;
  function ReadAsync(var Buffer;Count:longint):longint;virtual;
  function WriteSync(var Buffer;Count:longint):longint;virtual;
  function NumPendingRequests:integer;
  function Busy:boolean;
  procedure Wait;
  function Cancel:boolean;
 end;

implementation

const
  AccessMode: array[0..2] of Integer = (
    GENERIC_READ,
    GENERIC_WRITE,
    GENERIC_READ or GENERIC_WRITE);
  ShareMode: array[0..4] of Integer = (
    0,
    0,
    FILE_SHARE_READ,
    FILE_SHARE_WRITE,
    FILE_SHARE_READ or FILE_SHARE_WRITE);



constructor TMemoryMappedFileStream.Create(const FileName, MapName: string; Mode: Word);
begin
  FMemoryDelta:=$FFFF;
  FFileName:=FileName;
  FMapName:=MapName;
  FMode:=Mode;
  if Mode = fmCreate then
  //Create new file
  begin
     FFileHandle:=CreateFile(PChar(FFileName), GENERIC_READ or GENERIC_WRITE,
                  0, nil, CREATE_ALWAYS,
                  FILE_ATTRIBUTE_NORMAL, 0);
    if FFileHandle = 0 then
      raise EFCreateError.CreateFmt('Error creating %s', [FFileName])
    else begin
     FSize:=FMemoryDelta;
     FCapacity:=FMemoryDelta;
    end;
  end else
  //Open existing file
  begin
    FFileHandle:=CreateFile(PChar(FFileName), AccessMode[Mode and 3],
                 ShareMode[(Mode and $F0) shr 4], nil, OPEN_EXISTING,
                 FILE_ATTRIBUTE_NORMAL, 0);
    FSize:=GetFileSize(FFileHandle,nil);
    FCapacity:=FSize;
    if (FFileHandle = 0) or (FSize=longint($FFFFFFFF)) then
      raise EFOpenError.CreateFmt('Error opening %s', [FFileName])
  end;
  FMapHandle:=CreateFileMapping(FFileHandle,nil,PAGE_READWRITE,0,FCapacity,PChar(FMapName));
  if FMapHandle = 0 then
      raise EFOpenError.CreateFmt('Error creating file mapping object %s', [FMapName]);
  FMemory:=MapViewOfFile(FmapHandle,FILE_MAP_WRITE,0,0,0);
  FPosition:=0;
end;

constructor TMemoryMappedFileStream.Open(const MapName:string; Mode:word);
begin
  if Mode = fmCreate then
      raise EFCreateError.Create('Use "Create" constructor with fmCreate mode');
  FMemoryDelta:=$FFFF;
  FFileName:='';
  FMapName:=MapName;
  FMode:=Mode;
  FSize:=FMemoryDelta;
  FCapacity:=FMemoryDelta;
  FFileHandle := 0;
  FMapHandle:=OpenFileMapping(FILE_MAP_ALL_ACCESS,FALSE,PChar(FMapName));
  if FMapHandle = 0 then
      raise EFOpenError.CreateFmt('Error opening file mapping object', [FMapName]);
  FMemory:=MapViewOfFile(FmapHandle,FILE_MAP_WRITE,0,0,0);
  FPosition:=0;
end;

procedure TMemoryMappedFileStream.CloseHandles;
begin
  if FMemory <> nil then begin
   FlushViewOfFile(FMemory,0);
   UnmapViewOfFile(FMemory);
  end;
  if FMapHandle <> 0 then CloseHandle(FMapHandle);
  if FFileHandle <> 0 then CloseHandle(FFileHandle);
end;

destructor TMemoryMappedFileStream.Destroy;
begin
  CloseHandles;
end;

function TMemoryMappedFileStream.Read(var Buffer;Count:longint):longint;
begin
  if (FPosition >= 0) and (Count >= 0) then
  begin
    Result := FSize - FPosition;
    if Result > 0 then
    begin
      if Result > Count then Result := Count;
      Move(Pointer(Longint(FMemory) + FPosition)^, Buffer, Result);
      Inc(FPosition, Result);
      Exit;
    end;
  end;
  Result := 0;
end;

function TMemoryMappedFileStream.Write(const Buffer;Count:longint):longint;
var
  Pos: Longint;
begin
  if (FPosition >= 0) and (Count >= 0) then
  begin
    Pos := FPosition + Count;
    if Pos > 0 then
    begin
      if Pos > FSize then
      begin
        if Pos > FCapacity then
          SetCapacity(Pos);
        FSize := Pos;
      end;
      System.Move(Buffer, Pointer(Longint(FMemory) + FPosition)^, Count);
      FPosition := Pos;
      Result := Count;
      Exit;
    end;
  end;
  Result := 0;
end;


procedure TMemoryMappedFileStream.SetSize(NewSize:longint);
begin
 if NewSize < FSize then SetCapacity(NewSize)
  else if NewSize >= FSize then
     if NewSize <= FCapacity then
                               FSize:=NewSize
                              else begin
                                SetCapacity(NewSize);
                                if NewSize <= FCapacity then FSize:=NewSize;
                              end;
end;

procedure TMemoryMappedFileStream.SetCapacity(NewCapacity:longint);
begin
 if NewCapacity=0 then begin
  Flush;
  FSize:=0;
  FCapacity:=0;
 end
 else begin
  CloseHandles;
  //if FFileHandle=0, then we're using somebody's else
  //mapping object, cannot change its size
  if FFileHandle=0 then
    raise EFOpenError.CreateFmt('Cannot change size of an open'#13#10'mapping file object %s', [FMapName]);
  //recreate the mapping object with the new size
  //Is there a better way to do it?
  if FFileHandle <> 0 then begin //FFileHandle=0 for the open named mapping object
   FFileHandle:=CreateFile(PChar(FFileName), AccessMode[FMode and 3],
                 ShareMode[(FMode and $F0) shr 4], nil, OPEN_EXISTING,
                 FILE_ATTRIBUTE_NORMAL, 0);
   if (FFileHandle = 0) or (FSize=longint($FFFFFFFF)) then
       raise EFOpenError.CreateFmt('Error opening %s', [FileName]);
  end;
  FCapacity:=NewCapacity+FMemoryDElta;
  FMapHandle:=CreateFileMapping(FFileHandle,nil,PAGE_READWRITE,0,FCapacity,PChar(FMapName));
  FMemory:=MapViewOfFile(FMapHandle,FILE_MAP_WRITE,0,0,0);
  FPosition:=0;
 end;
end;

function TMemoryMappedFileStream.Seek(Offset:longint;Origin:word):longint;
begin
 case Origin of
  soFromBeginning : FPosition:=Offset;
  soFromCurrent   : FPosition:=FPosition+Offset;
  soFromEnd       : FPosition:=FSize+Offset;
 end;
 Result:=FPosition;
end;

procedure TMemoryMappedFileStream.Flush;
begin
 FlushViewOfFile(FMemory,0);
end;

//*******************************************************************
//                       TAsyncFileStream
//*******************************************************************

constructor TAsyncFileStream.Create(const FileName: string; Mode: Word);
var h:THandle;
begin
  if (GetVersion shr 31)=1 then  //=0 for NT
    raise EFCreateError.Create('Asynchronous I/O is supported by Windows NT only');
  if Mode = fmCreate then
  begin
    h := CreateFile(PChar(FileName), GENERIC_READ or GENERIC_WRITE,
         0, nil, CREATE_ALWAYS,
         FILE_ATTRIBUTE_NORMAL or FILE_FLAG_OVERLAPPED, 0);
    if h < 0 then
      raise EFCreateError.CreateFmt('Error creating %s', [FileName]);
  end else
  begin
    h := CreateFile(PChar(FileName), AccessMode[Mode and 3],
         ShareMode[(Mode and $F0) shr 4], nil, OPEN_EXISTING,
         FILE_ATTRIBUTE_NORMAL or FILE_FLAG_OVERLAPPED, 0);
    if h < 0 then
      raise EFOpenError.CreateFmt('Error opening %s', [FileName]);
  end;
  inherited Create(h);
  FSize:=GetFileSize(Handle,nil);
  FPosition:=0;
  FNumRequests:=0;
  FCapacity:=0;
  CapacityDelta:=$1000;
  FRequests:=nil;
end;

destructor TAsyncFileStream.Destroy;
var i:integer;
begin
 Wait;
 for i:=0 to FNumRequests-1 do DeleteRequest(FRequests^[i]);
 if FRequests<>nil then FreeMem(FRequests,SizeOf(PRequest)*FCapacity);
 CloseHandle(Handle);
 inherited;
end;

procedure TAsyncFileStream.SetSize(NewSize:longint);
begin
 Wait;
 FSize:=SetFilePointer(Handle, NewSize, nil, FILE_BEGIN);
 SetEndOfFile(Handle);
end;

procedure TAsyncFileStream.AddRequest(Request:PRequest);
var i:integer;
begin
 for i:=0 to FNumRequests-1 do if FRequests^[i]=nil then
  begin
   FRequests^[i]:=Request;
   Exit;
  end
  else
   if not IsRequestPending(FRequests^[i]^) then begin
    DeleteRequest(FRequests^[i]);
    FRequests^[i]:=Request;
    Exit;
   end;
 if FNumRequests >= FCapacity then begin
   FCapacity:=FCapacity+CapacityDelta;
   ReallocMem(FRequests,FCapacity*SizeOf(PRequest));
 end;
 FRequests^[FNumRequests]:=Request;
 inc(FNumRequests);
end;

function TAsyncFileStream.IsRequestPending(Request:TRequest):boolean;
var dw:integer;
begin
 Result:=not GetOverlappedResult(Handle,Request.Overlapped,dw,FALSE);
end;

procedure TAsyncFileStream.DeleteRequest(var Request:PRequest);
begin
 if Request=nil then Exit;
 PulseEvent(Request^.Overlapped.hEvent);
 ResetEvent(Request^.Overlapped.hEvent);
 CloseHandle(Request^.Overlapped.hEvent);
 if Request^.Buffer<>nil then GlobalFreePtr(Request^.Buffer);
 GlobalFreePtr(Request);
 Request:=nil;
end;

//synchronous read
function TAsyncFileStream.Read(var Buffer;Count:longint):longint;
var h:THandle;
    CountRead,dw,err:integer;
    Overlapped:POverlapped;
    smin,smax:integer;
begin
 if FPosition+Count > FSize then Count:=FSize-FPosition;
 h:=CreateEvent(nil,TRUE,FALSE,nil);
 GetMem(Overlapped,SizeOf(Overlapped^));
 with Overlapped^ do begin
  Offset:=FPosition;
  OffsetHigh:=0;
  hEvent:=h;
 end;
 repeat
  ReadFile(Handle,Buffer,Count,CountRead,Overlapped);
  err:=GetLastError;
  if (err=ERROR_WORKING_SET_QUOTA) or
     (err=ERROR_INVALID_USER_BUFFER) or
     (err=ERROR_NOT_ENOUGH_MEMORY) or
     (err=ERROR_NOT_ENOUGH_QUOTA)  then begin
   GetProcessWorkingSetSize(GetCurrentProcess,smin,smax);
   if not SetProcessWorkingSetSize(GetCurrentProcess,smin,smax+Count) then begin
     if NumPendingRequests >= 1 then Wait
      else raise EFOpenError.Create('Not enough memory for the request')
   end;
  end;
 until (err=ERROR_SUCCESS) or (err=ERROR_IO_PENDING) or
       (err=ERROR_IO_INCOMPLETE);
 GetOverlappedResult(Handle,Overlapped^,dw,TRUE);
 FPosition:=FPosition+Overlapped^.InternalHigh;
 Result:=Overlapped^.InternalHigh;
 ResetEvent(h);
 CloseHandle(h);
 FreeMem(Overlapped,SizeOf(Overlapped^));
end;

//asynchronous read
function TAsyncFileStream.ReadAsync(var Buffer;Count:longint):longint;
var h:THandle;
    CountRead,err:integer;
    Request:PRequest;
    smin,smax:integer;
begin
 if FPosition+Count > FSize then Count:=FSize-FPosition;
 h:=CreateEvent(nil,TRUE,FALSE,nil);
 Request:=GlobalAllocPtr(GMEM_MOVEABLE,SizeOf(TRequest));
 with Request^.Overlapped do begin
  Offset:=FPosition;
  OffsetHigh:=0;
  hEvent:=h;
 end;
 Request^.Buffer:=nil;
 Request^.Count:=Count;
 repeat
  ReadFile(Handle,Buffer,Count,CountRead,@Request^.Overlapped);
  err:=GetLastError;
  if (err=ERROR_WORKING_SET_QUOTA) or
     (err=ERROR_INVALID_USER_BUFFER) or
     (err=ERROR_NOT_ENOUGH_MEMORY) or
     (err=ERROR_NOT_ENOUGH_QUOTA)  then begin
   GetProcessWorkingSetSize(GetCurrentProcess,smin,smax);
   if not SetProcessWorkingSetSize(GetCurrentProcess,smin,smax+Count) then begin
     if NumPendingRequests >= 1 then Wait
      else raise EFOpenError.Create('Not enough memory for the request')
   end;
  end;
 until (err=ERROR_SUCCESS) or (err=ERROR_IO_PENDING) or
       (err=ERROR_IO_INCOMPLETE);

 AddRequest(Request);
 FPosition:=FPosition+Count;
 Result:=Count;
end;

//asynchronous write
function TAsyncFileStream.Write(const Buffer;Count:longint):longint;
var h:THandle;
    CountRead,err:integer;
    Request:PRequest;
    smin,smax:integer;
begin
 try
 h:=CreateEvent(nil,TRUE,FALSE,nil);
 Request:=GlobalAllocPtr(GMEM_MOVEABLE,SizeOf(TRequest));
 with Request^.Overlapped do begin
  Offset:=FPosition;
  OffsetHigh:=0;
  hEvent:=h;
 end;
 repeat
  Request^.Buffer:=GlobalAllocPtr(GMEM_MOVEABLE,Count);
  if Request^.Buffer=nil then Wait;
 until Request^.Buffer<>nil;
 Request^.Count:=Count;
 Move(Buffer,Request^.Buffer^,Count);
 repeat
  WriteFile(Handle,Request^.Buffer^,Count,CountRead,@Request^.Overlapped);
  err:=GetLastError;
  if (err=ERROR_WORKING_SET_QUOTA) or
     (err=ERROR_INVALID_USER_BUFFER) or
     (err=ERROR_NOT_ENOUGH_MEMORY) or
     (err=ERROR_NOT_ENOUGH_QUOTA)  then begin
   GetProcessWorkingSetSize(GetCurrentProcess,smin,smax);
   if not SetProcessWorkingSetSize(GetCurrentProcess,smin,smax+Count) then begin
     if NumPendingRequests >= 1 then Wait
      else raise EFOpenError.Create('Not enough memory for the request')
   end;
  end;
 until (err=ERROR_SUCCESS) or (err=ERROR_IO_PENDING) or
       (err=ERROR_IO_INCOMPLETE);
 AddRequest(Request);
 FPosition:=FPosition+Count;
 if FPosition > FSize then FSize:=FPosition;
 Result:=Count;
 except
  DeleteRequest(Request);
  Result:=0;
 end;
end;                          

//synchronous write
function TAsyncFileStream.WriteSync(var Buffer;Count:longint):longint;
var h:THandle;
    CountRead,dw,err:integer;
    Overlapped:POverlapped;
    smin,smax:integer;
begin
 if FPosition+Count > FSize then Count:=FSize-FPosition;
 h:=CreateEvent(nil,TRUE,FALSE,nil);
 GetMem(Overlapped,SizeOf(Overlapped^));
 with Overlapped^ do begin
  Offset:=FPosition;
  OffsetHigh:=0;
  hEvent:=h;
 end;
 repeat
  WriteFile(Handle,Buffer,Count,CountRead,Overlapped);
  err:=GetLastError;
  if (err=ERROR_WORKING_SET_QUOTA) or
     (err=ERROR_INVALID_USER_BUFFER) or
     (err=ERROR_NOT_ENOUGH_MEMORY) or
     (err=ERROR_NOT_ENOUGH_QUOTA)  then begin
   GetProcessWorkingSetSize(GetCurrentProcess,smin,smax);
   if not SetProcessWorkingSetSize(GetCurrentProcess,smin,smax+Count) then begin
     if NumPendingRequests >= 1 then Wait
      else raise EFOpenError.Create('Not enough memory for the request')
   end;
  end;
 until (err=ERROR_SUCCESS) or (err=ERROR_IO_PENDING) or
       (err=ERROR_IO_INCOMPLETE);
 GetOverlappedResult(Handle,Overlapped^,dw,TRUE);
 FPosition:=FPosition+Overlapped^.InternalHigh;
 Result:=Overlapped^.InternalHigh;
 ResetEvent(h);
 CloseHandle(h);
 FreeMem(Overlapped,SizeOf(Overlapped^));
end;

function TAsyncFileStream.NumPendingRequests:integer;
var i,dw:integer;
begin
 Result:=0;
 for i:=0 to FNumRequests-1 do if FRequests^[i]<>nil then
   if not GetOverlappedResult(Handle,FRequests^[i].Overlapped,dw,FALSE) then begin
    DeleteRequest(FRequests^[i]);
   end else if GetLastError=ERROR_IO_INCOMPLETE then begin
    inc(Result);
   end;
end;

function TAsyncFileStream.Seek(Offset: Longint; Origin: Word): Longint;
begin
 case Origin of
  soFromBeginning : FPosition:=Offset;
  soFromCurrent   : FPosition:=FPosition+Offset;
  soFromEnd       : FPosition:=FSize+Offset;
 end;
 Result:=FPosition;
end;

function TAsyncFileStream.Busy:boolean;
var i:integer;
begin
 Result:=false;
 for i:=0 to FNumRequests-1 do if FRequests^[i]<>nil then
   if not IsRequestPending(FRequests^[i]^) then begin
    DeleteRequest(FRequests^[i]);
   end else if GetLastError=ERROR_IO_INCOMPLETE then begin
    Result:=true;
    Exit;
   end;
end;

procedure TAsyncFileStream.Wait;
var i,dw:integer;
begin
 for i:=0 to FNumRequests-1 do if FRequests^[i]<>nil then
   if GetOverlappedResult(Handle,FRequests^[i].Overlapped,dw,FALSE) then
     DeleteRequest(FRequests^[i])
   else begin
     GetOverlappedResult(Handle,FRequests^[i].Overlapped,dw,TRUE);
     DeleteRequest(FRequests^[i]);
   end;
end;

function TAsyncFileStream.Cancel:boolean;
var i:integer;
begin
 Result:=CancelIO(Handle);
 if Result then
  for i:=0 to FNumRequests-1 do if FRequests^[i]<>nil then
      DeleteRequest(FRequests^[i]);
end;

end.

