unit Logomain;

{
Program to show disk usage as a pie-chart

Revision history:

1.0    1993 Feb 08  First version for Borland's Turbo Pascal for Windows
2.0.0  1996 Apr 14  Version for Borland's Delphi 2.0
2.0.2  1996 Apr 16  Pre-load Open dialog with '*.*' file name
                    Add number of files and directories display
2.0.4  1996 May 26  Add E-mail address to About box
2.1.0  1996 Jul 20  Add auto-registration as Folder Shell Extension
                    Collect size, allocated and slack simultaneously
                    Add option for displaying in bytes, kilobytes or megabytes
                    Save current displayed folder across option selection
2.1.2  1996 Oct 20  Don't count directories as files (they are, but...)
2.1.4  1996 Dec 08  Change to "standard" main form (was ToolWin)
                    Revise logo to mimic PieChart colours better
}

interface

uses Windows, Classes, Graphics, Forms, Controls, Menus,
  Dialogs, StdCtrls, Buttons, ExtCtrls, ComCtrls, PieChart,
  OptnDlg, FileInfo;

type
  TMainForm = class(TForm)
    MainMenu: TMainMenu;
    FileMenu: TMenuItem;
    FileOpenItem: TMenuItem;
    FileExitItem: TMenuItem;
    OpenDialog: TOpenDialog;
    Help1: TMenuItem;
    AboutItem: TMenuItem;
    SpeedPanel: TPanel;
    OpenBtn: TSpeedButton;
    ExitBtn: TSpeedButton;
    StatusBar: TStatusBar;
    PieChart1: TPieChart;
    ListBox1: TListBox;
    Timer1: TTimer;
    btnStop: TButton;
    BitBtn1: TBitBtn;
    btnUp: TButton;
    View1: TMenuItem;
    Options1: TMenuItem;
    Refresh1: TMenuItem;
    N2: TMenuItem;
    cbxUnits: TComboBox;
    Label1: TLabel;
    cbxDisplay: TComboBox;
    Label2: TLabel;
    procedure FormCreate(Sender: TObject);
    procedure FileExit(Sender: TObject);
    procedure FileOpen(Sender: TObject);
    procedure About(Sender: TObject);
    procedure ShowHint(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure Timer1Timer(Sender: TObject);
    procedure PieChart1DblClick(Sender: TObject);
    procedure ListBox1DblClick(Sender: TObject);
    procedure btnStopClick(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure btnUpClick(Sender: TObject);
    procedure Options1Click(Sender: TObject);
    procedure Refresh1Click(Sender: TObject);
    procedure cbxUnitsChange(Sender: TObject);
    procedure cbxDisplayChange(Sender: TObject);
  private
    dir_list: TDirectoryList;
    current_folder: TDirectoryList;
    scanning: boolean;
    stop_requested: boolean;
    display_mode: Tdisplay_mode;
    display_units: Tdisplay_units;
    procedure scan_tree;
    procedure display_list (const list: TDirectoryList);
    procedure handle_double_click (entry: TDirectoryData);
  public
    { Public declarations }
  end;

procedure set_status_text (const s: string);  stdcall;

var
  MainForm: TMainForm;

implementation

uses SysUtils, About, LogoStrs, Registry;

const
  product_name = 'David''s ShowMan program';
  product_version = 'Version 2.1.4';
  product_copyright = 'Copyright '#169' David J Taylor, Edinburgh, 1993-1996';
  product_comments = 'Version for Delphi 2.0'#10'Author: david.taylor@gecm.com';

{$R *.DFM}

{$R version.res}

procedure set_status_text (const s: string);
begin
  MainForm.StatusBar.Panels[1].Text := s;
end;

procedure TMainForm.FormCreate(Sender: TObject);
var
  reg: TRegistry;
begin
  Application.OnHint := ShowHint;
  dir_list := TDirectoryList.Create (nil, '.');
  current_folder := nil;
  scanning := False;
  stop_requested := False;

  // set defaults for the display
  display_mode := show_allocated;    // initial choice is allocated space
  display_units := show_KB;          // and to display it in kilobytes
  cbxUnits.ItemIndex := 1;
  cbxDisplay.ItemIndex := 1;

  // code for Windows NT and 95 - add Folder right-click menu item
  // add appropriate pointer to the .EXE and a parameter string placeholder
  reg := TRegistry.Create;
  with reg do
    begin
    RootKey := HKEY_CLASSES_ROOT;
    if OpenKey ('\Folder\shell', False) then
      if OpenKey ('\Folder\shell\Usage Pie Chart\command', True) then
        WriteString ('', Application.ExeName + ' %1');
    end;
  reg.Free;

  // prepare for first scan, triggered by the timer
  Timer1.Enabled := True;
end;


procedure TMainForm.FormDestroy(Sender: TObject);
begin
  dir_list.Free;
end;

procedure TMainForm.FileOpen(Sender: TObject);
begin
  // rely on the OpenDialog function changing the current directory
  with OpenDialog do
    begin
    FileName := '*.*';
    if Execute then scan_tree;
    end;
end;

procedure TMainForm.FileExit(Sender: TObject);
begin
  Close;
end;

procedure TMainForm.About(Sender: TObject);
begin
  with AboutBox do
    begin
    ProductName.Caption := product_name;
    Version.Caption := product_version;
    Copyright.Caption := product_copyright;
    Comments.Caption := product_comments;
    ProgramIcon.Picture.Icon := Application.Icon;
    ShowModal;
    end;
end;

procedure TMainForm.ShowHint(Sender: TObject);
begin
  StatusBar.Panels[0].Text := Application.Hint;
end;

procedure TMainForm.scan_tree;
var
  sectors_per_cluster: integer;
  bytes_per_sector: integer;
  free_clusters: integer;
  total_clusters: integer;
  cluster_bytes: integer;
  disk: array [0..3] of char;
  root: string;
begin
  if scanning then Exit;
  PieChart1.Clear;
  Caption := 'ShowMan';
  root := GetCurrentDir;
  // ensure the directory to be scanned ends in a trailing backslash
  if root [Length (root)] <> '\' then root := root + '\';
  // get the C:\ part, this code is not UNC compliant....
  StrPLcopy (disk, root, 3);
  // try and get disk parameters - assume cluster size is one otherwise
  if GetDiskFreeSpace (disk, sectors_per_cluster, bytes_per_sector,
                       free_clusters, total_clusters)
    then cluster_bytes := bytes_per_sector * sectors_per_cluster
    else cluster_bytes := 1;
  try
    stop_requested := false;
    btnStop.Enabled := True;
    dir_list.SetDirectoryName (root);
    scanning := true;
    dir_list.scan (stop_requested, cluster_bytes, set_status_text);
    display_list (dir_list);
    scanning := false;
    stop_requested := false;
    btnStop.Enabled := False;
  finally
  end;
end;

procedure TMainForm.Timer1Timer(Sender: TObject);
begin
  // one-shot code at program start time, scan something to
  // show the user what sort of display to expect
  Timer1.Enabled := False;
  if ParamCount > 0 then SetCurrentDir (ParamStr (1));
  scan_tree;
end;

procedure TMainForm.display_list (const list: TDirectoryList);
var
  str_list: TStringList;
  index: integer;
  size: integer;
  total_size: integer;
  units: Double;
  s: string;
begin
  if list = nil
    then Exit                        // nothing to do
    else current_folder := list;     // make a note of where we are

  // prepare a string list with the sizes (numeric) and names
  // of the files and directories in this part of the tree
  str_list := TStringList.Create;
  total_size := 0;
  // sort out the user's chosen units for the display
  units := 1;
  case display_units of
    show_KB: units := 1024.0;
    show_MB: units := 1024.0 * 1024.0;
  end;
  for index := 0 to list.Count-1 do
    begin
    size := 0;           // only required to stop the compiler warning
    // choose which number to display
    case display_mode of
       show_actual: size := TDirectoryData (list.Objects[index]).GetBytes;
    show_allocated: size := TDirectoryData (list.Objects[index]).GetAllocated;
        show_slack: size := TDirectoryData (list.Objects[index]).GetSlack;
    end;
    // quantise the display units
    case display_units of
      show_bytes: Str (size, s);
      show_KB: Str (size / units:3:1, s);
      show_MB: Str (size / units:3:3, s);
    end;
    str_list.AddObject (s + '   ' + LowerCase (list.Strings[index]),
                        list.Objects[index]);
    Inc (total_size, size);
    end;
  // compute and show the pie-chart
  PieChart1.SetDataAndLabels (str_list);
  str_list.Free;
  // create and display the status line
  with list do
    begin
    case display_mode of
       show_actual: total_size := GetTotalBytes;
    show_allocated: total_size := GetTotalAllocated;
        show_slack: total_size := GetTotalSlack;
    end;
    StatusBar.Panels[1].Text :=
      Format ('%s ... contains %1.n bytes in %1.n files and %1.n directories',
      [GetDirectoryName, total_size + 0.0,
       GetTotalFiles + 0.0, GetTotalDirectories + 0.0]);
    Caption := 'ShowMan - ' + GetDirectoryName;
    // only enable the "Up" button if there's somewhere to go....
    btnUp.Enabled := GetParentDirectoryList <> nil;
    end;
end;

procedure TMainForm.PieChart1DblClick(Sender: TObject);
begin
  handle_double_click (PieChart1.ClickedObject as TDirectoryData);
end;

procedure TMainForm.ListBox1DblClick(Sender: TObject);
var
  entry: TDirectoryData;
begin
  // determine what item was clicked
  entry := TDirectoryData (ListBox1.Items.Objects [ListBox1.ItemIndex]);
  handle_double_click (entry);
end;

procedure TMainForm.handle_double_click (entry: TDirectoryData);
var
  lst: TDirectoryList;
begin
  // On a double-click, if the item is a sub-directory, go down
  // to it.  If the item a file, then try and go up a level.
  if entry <> nil then
    with entry do
      begin
      lst := GetSubDirectoryList;
      if lst <> nil
      then
        display_list (lst)              // we got a sub-directory
      else
        begin
        lst := GetParentDirectoryList;
        if lst <> nil then
          display_list (lst)            // we got a parent directory
        end;
      end;
end;

procedure TMainForm.btnStopClick(Sender: TObject);
begin
  stop_requested := True;
end;

procedure TMainForm.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
begin
  stop_requested := True;
end;

procedure TMainForm.btnUpClick(Sender: TObject);
// go up a directory level if there is one
var
  lst: TDirectoryList;
begin
  if ListBox1.Items.Count <> 0 then
    with TDirectoryData (ListBox1.Items.Objects [0]) do
    begin
    lst := GetParentDirectoryList;
    if lst <> nil then
      display_list (lst)
    end;
end;

procedure TMainForm.Options1Click(Sender: TObject);
begin
  // use present setting of DISPLAY_MODE and DISPLAY_UNITS
  // for the initial settings in the options dialog
  OptionsDialog.display_mode := display_mode;
  OptionsDialog.display_units := display_units;

  if OptionsDialog.ShowModal = mrOK then
    if (display_mode <> OptionsDialog.display_mode) or
       (display_units <> OptionsDialog.display_units) then
      begin
      // settings have changed, save the new settings and re-display
      display_mode := OptionsDialog.display_mode;
      display_units := OptionsDialog.display_units;
      display_list (current_folder);
      end;
end;

procedure TMainForm.Refresh1Click(Sender: TObject);
begin
  scan_tree;
end;

procedure TMainForm.cbxUnitsChange(Sender: TObject);
begin
  case cbxUnits.ItemIndex of
    0: display_units := show_bytes;
    1: display_units := show_KB;
    2: display_units := show_MB;
  end;
  display_list (current_folder);
end;

procedure TMainForm.cbxDisplayChange(Sender: TObject);
begin
  case cbxDisplay.ItemIndex of
    0: display_mode := show_actual;
    1: display_mode := show_allocated;
    2: display_mode := show_slack;
  end;
  display_list (current_folder);
end;

end.

