unit Shell_DDE_reg;
(***************************************************************************************
 Copyright Ken_Miles 2000

 For conditions of use please see TShellCommandInterface_ReadMe.Txt or
 copyright in TShellCommandInterface.hlp

 9 July - Added code to Tell IDE form had been modified
****************************************************************************************)
interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs, Shell_DDE,
  StdCtrls, Buttons, ComCtrls, EditFileAssociation, DsgnIntf,
  DupExtDialog, Menus, ExtCtrls;

type
(***************************************************************************************
  Object:   TEditFileAssociations

  Description: Form for editing collection of FileAssociations
               File Associations are passed as strings from which
               they are created.  When editing is completed they
               are returned as strings - this makes it easier to store
               in the form (TSTrings storage is already defined) and
               minimises creating of objects when not required

  1.02         Now tells IDE that value has been modified (oops)

  Version:  1.01
  Date last Modified: 9 July 2000
****************************************************************************************)
  TEditFileAssociations = class(TForm)
    lvFileAssocs: TListView;
    pmlvFileAssocs: TPopupMenu;
    New1: TMenuItem;
    Edit1: TMenuItem;
    Delete1: TMenuItem;
    pnlButtons: TPanel;
    btnNew: TBitBtn;
    btnEdit: TBitBtn;
    btnDelete: TBitBtn;
    btnOk: TBitBtn;
    btnCancel: TBitBtn;
    procedure editSelected(Sender: TObject);
    procedure NewFileAssoc(Sender: TObject);
    procedure lvFileAssocsChange(Sender: TObject; Item: TListItem;
      Change: TItemChange);
    procedure DeleteSelected(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure btnOkClick(Sender: TObject);
    procedure pmlvFileAssocsPopup(Sender: TObject);
    procedure pnlButtonsResize(Sender: TObject);
  private
    { Private declarations }
    Function  CheckForDuplicatesFileTypes: Boolean;
    Function  CheckForDuplicatesExts: Boolean;
  public
    { Public declarations }
    Function EditFileAssociations(FileAssocs: TStrings): Boolean;
  end;

(***************************************************************************************
  Object:   TShellCommandInterfaceEditor

  Description: A Delphi IDE component editor for TShellCommandInterface
               - double clicking on a TShellCommandInterface will default
               to the openfile command event

  Version:  1.00
  Date last Modified: 28 May 2000
****************************************************************************************)

  TShellCommandInterfaceEditor = class(TDefaultEditor)
  Private
  protected
    procedure EditProperty(PropertyEditor: TPropertyEditor; Var Continue, FreeEditor: Boolean); override;
  End;


(***************************************************************************************
  Object:   TFileAssociationsProperty

  Description: A Delphi IDE property editor for TFileAssociations
               When the TStrings property called FileAssociations is editied
               then a TEditFileAssociations dialog is displayed to allow
               editing

  Version:  1.00
  Date last Modified: 28 May 2000
****************************************************************************************)

  TFileAssociationsProperty = class(TClassProperty)
  public
    procedure Edit; override;
    function GetAttributes: TPropertyAttributes; override;
  end;

(***************************************************************************************
  Function: EditFileAssociations

  Description: Edits a list of strings as FileAssociations
****************************************************************************************)

Function EditFileAssociations(FileAssocs: TStrings): Boolean;

(***************************************************************************************
  Function: Register

  Description: Registers component and editors with Delphi
****************************************************************************************)

Procedure Register;

{---------------------------------------------------------------------------

---------------------------------------------------------------------------}
// Change the Constant PutOnTab to the Tab you wish to place  on
Const
  PutOnTab = 'System';

implementation

{$R *.DFM}

{---------------------------------------------------------------------------

---------------------------------------------------------------------------}

(***************************************************************************************
  Object:   TShellCommandInterfaceEditor
  Procedure: EditProperty

  Description: Blocks all but 'OnCmdOpenFile' ensuring that it becomes the
               default designer double click function.
               Check the deliberate mistake in the D5 help file example

  Version:  1.00
  Date last Modified: 28 May 2000
****************************************************************************************)

procedure TShellCommandInterfaceEditor.EditProperty(PropertyEditor: TPropertyEditor; Var Continue, FreeEditor: Boolean);

begin
  if (PropertyEditor.ClassName = 'TMethodProperty') and
    (PropertyEditor.GetName = 'OnCmdOpenFile') then
    inherited EditProperty(PropertyEditor, Continue, FreeEditor);
end;


(***************************************************************************************
  Object:   TFileAssociationsProperty

  Procedure: Edit

  Description: Edit function called from Delphi IDE object inspector

  1.02 Now tells the IDE that we have modified the value

  Version:  1.02
  Date last Modified: 9 July 2000
****************************************************************************************)

procedure TFileAssociationsProperty.Edit;

Var
  Values: TStrings;

begin
  Values:= TStringList.Create;
  Try
    // Read the FileAssociations Strings
    Values.Assign(TStrings(GetOrdValue));
    // Edit the Associations
    If EditFileAssociations(Values) Then Begin
      // Return updated associations
      TStrings(GetOrdValue).Assign(Values);
      // Tell IDE that file has been modified
      Modified;
    End;
  Finally
    Values.Free;
  End;
end;

(***************************************************************************************
  Object:   TFileAssociationsProperty

  Procedure: GetAttributes

  Description: Tell Delphi what type of parameter/editor

  Version:  1.00
  Date last Modified: 28 May 2000
****************************************************************************************)
Function TFileAssociationsProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paDialog];
end;

(***************************************************************************************
  Object:   TEditFileAssociations

  Procedure: EditSelected

  Description: Edit the file association in the selected listitem

  Version:  1.00
  Date last Modified: 28 May 2000
****************************************************************************************)
procedure TEditFileAssociations.editSelected(Sender: TObject);

Var
  FileAss: TFileAssociation;

begin
  // Check something is selected and it is a fileassociation object
  If (lvFileAssocs.Selected <> Nil) and (TObject(lvFileAssocs.Selected.Data) is TFileAssociation) Then Begin
    FileAss:= TFileAssociation(lvFileAssocs.Selected.Data);
    // Edit the file association
    If Edit_File_Association(Self, FileAss) Then Begin
      // Delete old list item
      lvFileAssocs.Selected.Free;
      // display modified item
      FileAss.DisplayOnListView(lvFileAssocs);
    End;
  End;
end;

(***************************************************************************************
  Object:   TEditFileAssociations

  Procedure: NewFileAssoc

  Description: Create a new fileassoc and edit it

  Version:  1.00
  Date last Modified: 28 May 2000
****************************************************************************************)
procedure TEditFileAssociations.NewFileAssoc(Sender: TObject);

Var
  NewFileAss: TFileAssociation;

begin
 //Create a default association
 NewFileAss:= TFileAssociation.CreateFrom('Name=New.FileType,Exts=xxx,"cmds=Open"');
 // Edit it
 If Edit_File_Association(Self, NewFileAss) Then Begin
   // Add to list
   NewFileAss.DisplayOnListView(lvFileAssocs);
 End
 Else Begin
  // Free object - user did not press ok
  NewFileAss.Free;
 End;
end;

(***************************************************************************************
  Object:      TEditFileAssociations/pmlvFileAssocsPopup

  Procedure:   lvFileAssocsChange

  Description: When user changes the selection/popups menu
               work out whether to enable edit/delete buttons/menu items

  Version:  1.00
  Date last Modified: 28 May 2000
****************************************************************************************)
procedure TEditFileAssociations.lvFileAssocsChange(Sender: TObject; Item: TListItem;
  Change: TItemChange);
begin
  btnEdit.Enabled:= lvFileAssocs.SelCount = 1;
  btnDelete.Enabled:= lvFileAssocs.SelCount = 1;
end;

procedure TEditFileAssociations.pmlvFileAssocsPopup(Sender: TObject);
begin
  Edit1.Enabled:= lvFileAssocs.SelCount = 1;
  Delete1.Enabled:= lvFileAssocs.SelCount = 1;
end;


(***************************************************************************************
  Object:   TEditFileAssociations

  Procedure: DeleteSelected

  Description: Deletes the item selected in the list view

  Version:  1.00
  Date last Modified: 28 May 2000
****************************************************************************************)
procedure TEditFileAssociations.DeleteSelected(Sender: TObject);
begin
  If (lvFileAssocs.Selected <> Nil) Then Begin
    Try
      // ensure fileassociation is deleted from list view item
      TObject(lvFileAssocs.Selected.Data).Free;
    Except
    End;
    // Delete listview item
    lvFileAssocs.Selected.Free;
  End;
end;

(***************************************************************************************
  Object:   TEditFileAssociations

  Procedure: FormDestroy

  Description: Clears the

  Version:  1.00
  Date last Modified: 28 May 2000
****************************************************************************************)

procedure TEditFileAssociations.FormDestroy(Sender: TObject);

Var
  Idx: Integer;

Begin
  // Free all File associations objects in listview
  Idx:= lvFileAssocs.Items.Count;
  While Idx > 0 Do Begin
    Dec(Idx);
    Try
      TObject(lvFileAssocs.Items[Idx].Data).Free;
    Except
    End;
  End;
End;

(***************************************************************************************
  Object:   TEditFileAssociations

  Procedure: CheckForDuplicatesFileTypes

  Description: Checks that no duplicate names

  Version:  1.00
  Date last Modified: 28 May 2000
****************************************************************************************)
Function  TEditFileAssociations.CheckForDuplicatesFileTypes: Boolean;

Var
  iCheckUpTo, Idx: Integer;
  CapCache, Duplicates: String;

Begin
  Idx:= lvFileAssocs.Items.Count;
  Duplicates:= '';
  // loop around all (but last)
  While Idx > 1 Do Begin
    Dec(Idx);
    // Cache caption of what checking
    CapCache:= lowercase(lvFileAssocs.Items[Idx].Caption);
    // loop around all upto the one checking
    iCheckUpTo:= 0;
    While iCheckUpTo < Idx Do Begin
      // Does the item match
      If lowercase(lvFileAssocs.Items[iCheckUpTo].Caption) = CapCache Then Begin
        // Add to duplicates
        If  Duplicates = ''
          Then Duplicates:= lvFileAssocs.Items[iCheckUpTo].Caption
          Else Duplicates:= Duplicates + ',' + lvFileAssocs.Items[iCheckUpTo].Caption;
        // break out look - found one duplicate, no point checking further
        Break;
      End;
      Inc(iCheckUpTo);
    End;
  End;
  // If duplicates found then tell user
  Result:=  Duplicates <> '';
  If Result Then MessageBox(Handle, PChar('Duplicated File Associations: ' + Duplicates), 'Duplicates', MB_ICONERROR);
End;

(***************************************************************************************
  Object:   TEditFileAssociations

  Procedure: CheckForDuplicatesExts

  Description: Checks that no duplicate extensions are found

  Version:  1.00
  Date last Modified: 28 May 2000
****************************************************************************************)

Function  TEditFileAssociations.CheckForDuplicatesExts: Boolean;

Var
  Ext, FIn: String;
  Exts, FoundIn: TStringList;
  iExtsStore, iExts, iCheckUpTo, Idx: Integer;
  CheckAgainst, Check: TFileAssociation;


Begin
  // Create lists for storing errors - note these are twinned pairs and
  // will have same number of rows
  Exts:= TStringList.Create;
  FoundIn:= TStringList.Create;
  Try
    // Sweep all
    Idx:= lvFileAssocs.Items.Count;
    While Idx > 1 Do Begin
      Dec(Idx);
      // Get File association object
      CheckAgainst:= TFileAssociation(lvFileAssocs.Items[Idx].Data);
      // Sweep all upto
      iCheckUpTo:= 0;
      While iCheckUpTo < Idx Do Begin
        Check:= TFileAssociation(lvFileAssocs.Items[iCheckUpTo].Data);
        // Sweep exetension
        iExts:= CheckAgainst.Exts.Count;
        While iExts > 0 Do Begin
          Dec(iExts);
          // Cache extension
          Ext:= CheckAgainst.Exts[iExts];
          // Does check have same extension?
          If Check.Exts.IndexOf(Ext) >= 0 Then Begin
            // Record ext and associations
            iExtsStore:= Exts.IndexOf(Ext);
            // Previously found
            If iExtsStore >= 0 Then Begin
              // Add, if required, to the found items
              FIn:= FoundIn[iExtsStore];
              if pos('"' + CheckAgainst.Name + '"', Fin) = 0 Then Fin:= Fin + ', "' + CheckAgainst.Name + '"';
              if pos('"' + Check.Name + '"', Fin) = 0 Then Fin:= Fin + ', "' + Check.Name + '"';
              FoundIn[iExtsStore]:= FIn;
            End
            Else Begin
              // Add new duplicate ext + associations
              Exts.Add(Ext);
              FoundIn.Add('"' + CheckAgainst.Name + '", "' + Check.Name + '"');
            End;
          End;
        End;
        Inc(iCheckUpTo);
      End;
    End;
    // Any duplicates found - tell user
    Result:=  Exts.Count > 0;
    If Result Then  DisplayDuplicates(exts, FoundIn);
  Finally
    Exts.Free;
    FoundIn.Free;
  End;
End;

(***************************************************************************************
  Object:   TEditFileAssociations

  Procedure: btnOkClick

  Description: User pressed ok - check things, only closing if

  Version:  1.00
  Date last Modified: 28 May 2000
****************************************************************************************)
procedure TEditFileAssociations.btnOkClick(Sender: TObject);

Var
  Check1, Check2: Boolean;

begin
  Check1:=  CheckForDuplicatesFileTypes;
  Check2:=  CheckForDuplicatesExts;
  If Check1 or Check2 Then ModalResult:= mrNone;
end;

(***************************************************************************************
  Object:   TEditFileAssociations

  Procedure: EditFileAssociations

  Description: Converts strings to FileAssociations and edits them
               saving them back to the TStrings

  Version:  1.00
  Date last Modified: 28 May 2000
****************************************************************************************)

Function TEditFileAssociations.EditFileAssociations(FileAssocs: TStrings): Boolean;

Var
  Idx: Integer;
  NewAsscs: TStringList;

Begin
  // Convert each string to a file association and display them in list view
  Idx:= FileAssocs.Count;
  While Idx > 0 Do Begin
    Dec(Idx);
    With TFileAssociation.CreateFrom(FileAssocs[Idx]) Do Begin
      DisplayOnListView(lvFileAssocs);
    End;
  End;
  // Show dialog
  Result:= ShowModal = mrOk;
  If Result Then Begin
    // Read each list item's File association and get the setup string
    Idx:= lvFileAssocs.Items.Count;
    NewAsscs:= TStringList.Create;
    Try
      While Idx > 0 Do Begin
        Dec(Idx);
        If TObject(lvFileAssocs.Items[Idx].Data) is TFileAssociation Then Begin
          NewAsscs.Add(TFileAssociation(lvFileAssocs.Items[Idx].Data).CSVConfig);
        End;
      End;
      // Copy Strings to object passed
      FileAssocs.Assign(NewAsscs);
    Finally
      NewAsscs.Free;
    End;
  End;
End;



(***************************************************************************************
  Function: EditFileAssociations

  Description: Edits a list of strings as FileAssociations

  Version:  1.00
  Date last Modified: 28 May 2000
****************************************************************************************)

Function EditFileAssociations(FileAssocs: TStrings): Boolean;

Begin
  With TEditFileAssociations.Create(Application) Do Begin
    Try
      Result:= EditFileAssociations(FileAssocs);
    Finally
      Free;
    End;
  End;
End;

(***************************************************************************************
  Function: Register

  Description: Registers component and editors with Delphi

  Version:  1.00
  Date last Modified: 28 May 2000
****************************************************************************************)

procedure Register;

begin
  RegisterComponents(PutOnTab, [TShellCommandInterface]);
  RegisterComponentEditor(TShellCommandInterface, TShellCommandInterfaceEditor);
  RegisterPropertyEditor(TypeInfo(TStrings), TShellCommandInterface, 'FileAssociations', TFileAssociationsProperty);
end;


procedure TEditFileAssociations.pnlButtonsResize(Sender: TObject);
begin
  btnCancel.Left:= pnlButtons.ClientWidth - btnCancel.Width - 10;
  btnOk.Left:= btnCancel.Left - btnOk.Width - 10;
end;

end.
