unit Shell_DDE;
(***************************************************************************************
 Copyright Ken_Miles 2000

 For conditions of use please see TShellCommandInterface_ReadMe.Txt or
 copyright in TShellCommandInterface.hlp

1 June 2000 - Mod TFileAssociation RegisterAssociations
9 July      - Stop the stealing of FormCreate in designtime
****************************************************************************************)

interface

Uses DDEMan, Windows, SysUtils, Classes, Registry, Forms, grids,
     ComCtrls;

{$I ObjectPascalCapabilities.inc}
Type
(******************************************************************************
  Command Record

  Description: Store for Command and String

  Date last Modified: 28 May 2000
  Version 1.0
******************************************************************************)

  TCommand = Record
    Command,
    Option: String;
  End;

 {$IFNDEF DYNAMIC_ARRAYS}
   {$R-}
   TCommandArray = Array[0..0] of TCommand;
   PCommandArray = ^TCommandArray;
 {$ENDIF}

(******************************************************************************
  Object: TFileAssociation

  Function of: For handling registering/un-registering of File Extensions and
               and commands for explorer

  Date last Modified: 9 July 2000
  Version 1.02

  1.01: Modified register Associations to cope with commands with spaces in name
  1.02: Stop it taking the OnFormCreate when in the design stage
  ******************************************************************************)

  TFileAssociation    = Class(TObject)
  Private
    fExts: TStrings; // List of file Extensions associated with file type
    fName,           // Filetype
    fDescription,    // Description (to be displayed in explorer)
    fBlankStartOpt: String; // Options to add to cmdline for starting from explorer
    fIconIndex: Integer;    // Which Icon;

    fOwnAssociation: Boolean; // whether in full control or just add commands

    fCapacity, fCount: Integer;
 {$IFDEF DYNAMIC_ARRAYS}
    fCommands: Array of TCommand; // List of commands to add to explorer
 {$ELSE}
    fCommands: PCommandArray; // List of commands to add to explorer
    Procedure SetLength(Var CommandArray: PCommandArray; NewLength: Integer);
    Procedure Finalize(Var CommandArray: PCommandArray);
 {$ENDIF}

    // Command handling routines
    Function   GetCommand(Index: Integer): TCommand;
    Function   Add_Cmd(csvCommandOption: String ): Integer;

    // Create a string to enable store/recreate from
    Function   GetCSVConfig: String;

  Protected

    // Get the index of/where add of the command
    Function   Find(Command: String): Integer;

    // Options to add to cmdline for starting from explorer
    Property   BlankCmdLineStartOpt: String Read fBlankStartOpt Write fBlankStartOpt;

    // Register/unregister in registry
    Function   RegisterAssociation(Topic: String; Var AutoReplace: Boolean): Boolean;
    Function   UnRegisterAssociation: Boolean;

  Public
    Constructor Create;                     // Create blank
    Constructor CreateFrom(csvValue: String);  // Create from string provided by CommandasText
    Destructor  Destroy; Override;

    // Add commands custom/standard
 {$IFDEF DEFAULT_VALUES}
    Function   AddCommand(Command: String; Option: String = ''): Integer;
    Function   AddFileOpen(Option: String = ''): Integer;
    Function   AddFileNew(Option: String = ''): Integer;
    Function   AddFilePrint(Option: String = ''): Integer;
    Function   AddFilePrintTo(Option: String = ''): Integer;
 {$ELSE}
    Function   AddCommand(Command: String; Option: String): Integer;
    Function   AddFileOpen(Option: String): Integer;
    Function   AddFileNew(Option: String): Integer;
    Function   AddFilePrint(Option: String): Integer;
    Function   AddFilePrintTo(Option: String): Integer;
 {$ENDIF}

    // Delete commands
    Function   DeleteCommand(Command: String): Boolean;
    Function   Delete(Idx: Integer): Boolean;

    // Index of command -1 if not found
    Function   Index(Command: String): Integer;

    // Functions for displaying to allow editing
    Function   DisplayOnListView(ListView: TListView): TListItem;
    Procedure  FillCmdGrid(CmdGrid: TStringGrid);
    Procedure  GetFromCmdGrid(CmdGrid: TStringGrid);

    // String to enable store/recreate from
    Property   CSVConfig: String Read GetCSVConfig;

    // List of commands
    Property   Command[Index: Integer]: TCommand Read GetCommand; Default;
    // Number of commands
    Property   Count: Integer Read fCount;
    // List of file Extensions associated with file type
    Property   Exts: TStrings Read fExts;
    // File Type
    Property   Name: String Read fName Write fName;
    // Description (to be displayed in explorer)
    Property   Description: String Read fDescription Write fDescription;
    // whether in full control or just add commands
    Property   OwnAssociation: Boolean Read fOwnAssociation Write fOwnAssociation;

    Property   IconIndex: Integer Read fIconIndex Write fIconIndex;

  End;

(******************************************************************************
  Event types for when DDE marco executed from Explorer - e.g. Open files etc

  Date last Modified: 28 May 2000
  Version 1.0
******************************************************************************)

  TShellStdCommand    = Procedure(FileName, Option: String) of Object;
  TShellCustomCommand = Procedure(Command, FileName, Option: String) of Object;
  TCommandSwitch      = Procedure(Switch: String) of Object;

(******************************************************************************
  Object: TShellCommandInterface

  Function of:  Handling DDE marco executed from Explorer - e.g. Open files etc
                and the registration in the Registry

  Date last Modified: 28 May 2000
  Version 1.0
******************************************************************************)

  TShellCommandInterface = Class(TDdeServerConv)

  Private
    // Events to call on macro DDE execution
    fOpenFile, fNewFile, fPrint, fPrintTo: TShellStdCommand;
    fCustomDefaultCommand: TShellCustomCommand;
    // Store for handling old MacroExecute Function
    fOldMacroExecute: TMacroEvent;
    // List of commands - note need to convert to TFileAssociation (CreateFrom) to make sense
    fCommandStrings: TStrings;
    // Switch to add tell it being started by a double click on file
    fBlankStartOption: String;
    // When a command line switch is found
    fCommandSwitch: TCommandSwitch;

    // Remember use's on Form Create
    fOnFormCreate: TNotifyEvent;
    // Automatic processing command line?
    fAutoProcessCmdLine: Boolean;

    // Procedure/Event to be executed when a DDE Macro is recieved - triggers appropriate macros
    Procedure MacroExecute(Sender: TObject; Msg: TStrings);

    // Store new FileAssociations
    Procedure SetCommandStrings(NewValues: TStrings);


  Protected
    Function  ProcessCommand(Command, FileName, Option: String): Boolean;
    procedure SetParentComponent(Value: TComponent); Override;

    // Process the command line to open files etc
    Procedure   OnParentFormCreate(Sender: TObject);

  Public
    Constructor Create(AOwner: TComponent); override;
    Destructor  Destroy; Override;

    // Register/unregister commands with registry for explorer
    Procedure   RegisterFileAssociations(AutoReplace: Boolean);
    Procedure   UnRegisterFileAssociations;

    // Process the command line to open files etc
    Procedure   ProcessCommandLine;

  Published

    Property AutoProcessCommandLine: Boolean read fAutoProcessCmdLine write fAutoProcessCmdLine default True;
    // Switch to add tell it being started by a double click on file
    Property BlankStartOption: String Read fBlankStartOption Write fBlankStartOption;

    // List of FileAssociations - note stored as a list of strings
    Property  FileAssociations: TStrings Read fCommandStrings Write SetCommandStrings;

    // Override old OnMacroExecute event
    Property  OnExecuteMacro Read fOldMacroExecute Write fOldMacroExecute;

    // Events to call on macro DDE execution
    Property  OnCmdCustom:      TShellCustomCommand Read fCustomDefaultCommand Write fCustomDefaultCommand;
    Property  OnCmdNewFile:     TShellStdCommand Read fNewFile  Write fNewFile;
    Property  OnCmdOpenFile:    TShellStdCommand Read fOpenFile Write fOpenFile;
    Property  OnCmdPrintFile:   TShellStdCommand Read fPrint    Write fPrint;
    Property  OnCmdPrintFileTo: TShellStdCommand Read fPrintTo  Write fPrintTo;

    // When a command line switch is found
    Property  OnCommandSwitch: TCommandSwitch Read fCommandSwitch Write fCommandSwitch;
  End;


{------------------------------------------------------------------------------}

{------------------------------------------------------------------------------}



(******************************************************************************
  Default strings for Standard explorer commands

  Date last Modified: 28 May 2000
******************************************************************************)

Const
  FileOpenCmdStr     = 'Open';
  FileNewCmdStr      = 'New';
  FilePrintCmdStr    = 'Print';
  FilePrintToCmdStr  = 'PrintTo';


implementation

{------------------------------------------------------------------------------}

{------------------------------------------------------------------------------}

// Type of setup info for TFileAssociation
Type InputDataType = (Invalid, SetName, SetDescription, SetExts, AddCmd, OwnsAssocation, SetIconIndex);


(******************************************************************************
  Function: WhichCommand

  Description: identify command type and data out to enable set up info
               Note: Could have been a "local procedure" of Createfrom
               but this is advised against for speed issues

  Date last Modified: 28 May 2000
******************************************************************************)

Function WhichCommand(Input: String; Var Data: String): InputDataType;

Var
  Cache: String;

Begin
  // Assume invalid until proven otherwise
  Result:= Invalid;
  // Cmd lengths all 4 chars + "=" therefore split along this boundry
  Data:= Copy(Input, 5 + 1, Length(Input) - 5);
  Cache:= LowerCase(Copy(Input, 1, 5));
  // compare command against options - exit when found to speed up
  // looks neater than If then Else if Then else
  If Cache ='exts=' Then Begin
    Result:= SetExts;
    Exit;
  End;
  If Cache ='name=' Then Begin
    Result:= SetName;
    Exit;
  End;
  If Cache ='cmds=' Then Begin
    Result:= AddCmd;
    Exit;
  End;
  If Cache ='owns=' Then Begin
    Result:= OwnsAssocation;
    Exit;
  End;
  If Cache ='desc=' Then Begin
    Result:= SetDescription;
    Exit;
  End;
  If Cache ='icon=' Then Begin
    Result:= SetIconIndex;
    Exit;
  End;
End;

(******************************************************************************
  Object: TFileAssociation
  Constructor: Create - Set up

  Date last Modified: 28 May 2000
******************************************************************************)

Constructor TFileAssociation.Create;

Begin
  Inherited Create;
  // Setup Exts
  fExts:= TStringList.Create;
  // Setup command array
  fCommands:= Nil;
  fCount:= 0;
  fCapacity:= 0;
  // Set defaults
  fOwnAssociation:= True;
  fName:= '';
  fDescription:= '';
  fBlankStartOpt:= '';
  fIconIndex:= 0;
End;

(******************************************************************************
  Object: TFileAssociation
  Constructor: CreateFrom

  Description:  setup with values as defined by value string

  Date last Modified: 28 May 2000
******************************************************************************)
Constructor TFileAssociation.CreateFrom(csvValue: String);

Var
  Decode: TStrings;
  Data: String;
  Idx:  Integer;

Begin
  // Setup defaults/arrays
  Create;
  // create a string list to decode the comma seperated details (csv) passed in Value
  Decode:= TStringList.Create;
  Try
    // seperate csv into indvidual strings
    Decode.CommaText:= csvValue;
    Idx:= Decode.Count;
    // Perform each cmd
    While Idx > 0 Do Begin
      Dec(Idx);
      // Decode command  and take appropriate action
      Case WhichCommand(Decode[Idx], Data) of
        SetName:        fName:= Data;        // Set File Type from data
        SetDescription: Description:= Data;  // Set description from data
        SetExts:        fExts.CommaText:= Data; // Set Extensions from data
        AddCmd:         Add_Cmd(Data);          // Add command
        OwnsAssocation: fOwnAssociation:= lowercase(Data) = 'true'; // Set owns Association
        SetIconIndex:   Try fIconIndex:= StrToInt(Data); Except End;
        Invalid: ;      // No Action
      End;
    End;
  Finally
    // Tidy up
    Decode.Free;
  End;
End;

Destructor  TFileAssociation.Destroy;

Begin
  Finalize(fCommands);
  Inherited Destroy;
End;

(******************************************************************************
  Object: TFileAssociation
  Procedure  FillCmdGrid

  Description:  Fills a string grid component with the list of commands
                - for IDE, object inspector editing

  Date last Modified: 28 May 2000
******************************************************************************)

Procedure  TFileAssociation.FillCmdGrid(CmdGrid: TStringGrid);

Var
  Idx: Integer;

Begin
  // If no commands set length to 2 but add blank line
  // Dropping to single line when fixed rows = 1 then add can cause problems
  If fCount <= 0 Then Begin
    CmdGrid.RowCount:= 2;
    CmdGrid.Rows[1].Text:= '';
  End
  Else Begin
    // put each command in a row - not first row is title line
    CmdGrid.RowCount:= fCount + 1;
    Idx:= fCount;
    While Idx > 0 Do Begin
      Dec(Idx);
      CmdGrid.Rows[Idx + 1].CommaText:= '"' + fCommands[Idx].Command + '","' + fCommands[Idx].Option + '"';
    End;
  End;
End;


(******************************************************************************
  Object: TFileAssociation
  Procedure  GetFromCmdGrid

  Description:  Reads commands from a string grid component
                - for IDE, object inspector editing
                NOTE: No Error Checking - all done by Edit Form

  Date last Modified: 28 May 2000
******************************************************************************)
Procedure  TFileAssociation.GetFromCmdGrid(CmdGrid: TStringGrid);

Var
  Idx: Integer;

Begin
  // Clear commands
  fCount:= 0;
  // Read all (but first) row in string grid and convert to command
  Idx:= CmdGrid.RowCount;
  While Idx > 1 Do Begin
    Dec(Idx);
    // Do not add empty string command
    If CmdGrid.Rows[Idx][0] <> '' Then Begin
      Add_Cmd(CmdGrid.Rows[Idx].CommaText);
    End;
  End;
End;

(******************************************************************************
  Object: TFileAssociation
  Procedure  DisplayOnListView

  Description:  Displays self on list view - for IDE object inspector editing

  Date last Modified: 28 May 2000
******************************************************************************)

Function   TFileAssociation.DisplayOnListView(ListView: TListView): TListItem;

Var
  Cmds: String;
  iCmds: Integer;

Begin
  // Make a string with all commands listed
  iCmds:= 0;
  Cmds:= '';
  While iCmds < fCount Do Begin
    If Cmds <> ''
      Then Cmds:= Cmds + ', ' + fCommands[iCmds].Command
      Else Cmds:= fCommands[iCmds].Command;
    Inc(iCmds);
  End;

  // Add a new List item
  Result:= ListView.Items.Add;
  Result.Caption:= Name;
  // Point List item data at self
  Result.Data:= Self;

  // Add description, Own Association, Exts and command columns
  With Result.SubItems Do Begin
    Add(fDescription);
    If fOwnAssociation  Then Add('Yes') Else Add('No');
    Add(fExts.CommaText);
    Add(IntToStr(IconIndex));
    Add(Cmds);
  End;
End;

(******************************************************************************
  Object: TFileAssociation
  Procedure  GetCommand

  Description:  Returns the Value of the Command at Index IDX
                - note read only

  Date last Modified: 28 May 2000
******************************************************************************)

Function   TFileAssociation.GetCommand(Index: Integer): TCommand;

Begin
  If (Index >= 0) and (Index <= fCount)
    Then Result:= fCommands[Index]
    Else Raise Exception.Create('Index Out of Range');
End;

(******************************************************************************
  Object: TFileAssociation
  Procedure  AddCommand

  Description:  Adds a command to the command list
                Note if command already exists it is replaced

  Date last Modified: 28 May 2000
******************************************************************************)
{$IFNDEF DYNAMIC_ARRAYS}

Procedure TFileAssociation.SetLength(Var CommandArray: PCommandArray; NewLength: Integer);

Var
  Temp: PCommandArray;

Begin
  // Get new memory for Array
  GetMem(Temp, NewLength * SizeOf(TCommand));
  // Ensure default condition of 0
  FillChar(Temp^, NewLength * SizeOf(TCommand), 0);
  // Copy old values and free memory (If there were any)
  If fCount > 0 Then Begin
    Move(CommandArray^, Temp^, fCount * SizeOf(TCommand));
    Freemem(CommandArray);
  End;
  // Switch to new array
  CommandArray:= Temp;
End;

Procedure TFileAssociation.Finalize(Var CommandArray: PCommandArray);

Begin
  FreeMem(CommandArray);
  CommandArray:= Nil;
End;
{$ENDIF}

{$IFDEF DEFAULT_VALUES}
Function   TFileAssociation.AddCommand(Command: String; Option: String = ''): Integer;
{$ELSE}
Function   TFileAssociation.AddCommand(Command: String; Option: String): Integer;
{$ENDIF}

Var
  TempCap, Idx, iRipple: Integer;

Begin
  // If no command then return fail
  Result:= -1;
  If Command = '' Then Exit;

  // Find where to add
  Idx:= Find(Command);
  // If after last command or  command is different at IDX
  If ((Idx >= fCount) or (Lowercase(fCommands[Idx].Command) <> Lowercase(Command))) Then Begin
    // Check size - increase if required, unlikely more than once - ten commands would be
    // Excessive
    If fCount >= fCapacity Then Begin
      // Setting a temp variable first, ensures that if an exception is raised (e.g.) out of memory
      // invalid capacity is NOT set
      TempCap:= fCapacity + 10;
      SetLength(fCommands, TempCap);
      fCapacity:= TempCap;
    End;
    // Insert space (if required) by copying each row up (start from top)
    iRipple:= Count;
    While iRipple > Idx Do Begin
      fCommands[iRipple]:= fCommands[iRipple - 1];
      Dec(iRipple);
    End;
    // We have added a new item therefore increment count to show this
    Inc(fCount);
  End;
  // Set values
  fCommands[Idx].Command:= Command;
  fCommands[Idx].Option:=  Option;
  // return where the value was added
  Result:= Idx;
End;

(******************************************************************************
  Object: TFileAssociation
  Procedure  Add_Cmd

  Description:  Adds a command passed in a comma seperated string

  Date last Modified: 28 May 2000
******************************************************************************)

Function   TFileAssociation.Add_Cmd(csvCommandOption: String ): Integer;

Var
  Cmd, Opt: String;
  Decode: TStringList;

Begin
  Result:= -1; // If no value to add then return failed condition
  Decode:= TStringList.Create;  // Create a TStrings List to decode the csv string
  Try
    Decode.CommaText:= csvCommandOption;  // seperate csv into indvidual strings
    If (Decode.count > 0) Then Begin      // If not an empty string
      Cmd:= Decode[0];                    // Command is first line
      If Decode.count > 1 Then Opt:= Decode[1] Else Opt:= ''; // Next (if any) is option
      Result:= AddCommand(Cmd, Opt);      // Add command
    End;
  // Tidy up
  Finally
    Decode.Free;
  End;
End;

(******************************************************************************
  Object: TFileAssociation
  Procedure  AddFileOpen, AddFileNew, AddFilePrint, AddFilePrintTo

  Description:  Adds standard commands

  Date last Modified: 28 May 2000
******************************************************************************)

{$IFDEF DEFAULT_VALUES}
Function   TFileAssociation.AddFileOpen(Option: String = ''): Integer;

Begin
  Result:= AddCommand(FileOpenCmdStr, Option);
End;

Function   TFileAssociation.AddFileNew(Option: String = ''): Integer;

Begin
  Result:= AddCommand(FileNewCmdStr, Option);
End;

Function   TFileAssociation.AddFilePrint(Option: String = ''): Integer;

Begin
  Result:= AddCommand(FilePrintCmdStr, Option);
End;

Function   TFileAssociation.AddFilePrintTo(Option: String = ''): Integer;

Begin
  Result:= AddCommand(FilePrintToCmdStr, Option);
End;
{$ELSE}
Function   TFileAssociation.AddFileOpen(Option: String): Integer;

Begin
  Result:= AddCommand(FileOpenCmdStr, Option);
End;

Function   TFileAssociation.AddFileNew(Option: String): Integer;

Begin
  Result:= AddCommand(FileNewCmdStr, Option);
End;

Function   TFileAssociation.AddFilePrint(Option: String): Integer;

Begin
  Result:= AddCommand(FilePrintCmdStr, Option);
End;

Function   TFileAssociation.AddFilePrintTo(Option: String): Integer;

Begin
  Result:= AddCommand(FilePrintToCmdStr, Option);
End;
{$ENDIF}


(******************************************************************************
  Object: TFileAssociation
  Procedure  DeleteCommand

  Description:  Deletes the command (if present)

  Date last Modified: 28 May 2000
******************************************************************************)

Function   TFileAssociation.DeleteCommand(Command: String): Boolean;

Var
  Idx: Integer;

Begin
  Result:= False;
  // Get index of command
  Idx:= Index(Command);
  // If exists then delete
  If Idx >= 0 Then Result:= Delete(Idx);
End;

(******************************************************************************
  Object: TFileAssociation
  Procedure  Delete

  Description:  Deletes the command at index

  Date last Modified: 28 May 2000
******************************************************************************)

Function   TFileAssociation.Delete(Idx: Integer): Boolean;

Begin
  Result:= False;
  // Is the index valid
  If (Idx >= 0) and (Idx < fCount) Then Begin
    Result:= True;
    // Reduce number of commands
    Dec(fCount);
    // Copy commands down to fill gap
    While Idx < fCount Do Begin
      fCommands[Idx]:= fCommands[Idx + 1];
      Inc(Idx);
    End;

  End;
End;

(******************************************************************************
  Object: TFileAssociation
  Procedure  Find

  Description:  returns the index at which the command exists or should be inserted

  Date last Modified: 28 May 2000
******************************************************************************)

Function   TFileAssociation.Find(Command: String): Integer;

Var
  CheckRes: Integer;

Begin
  // Go through all positions until result found - it is unlikely to exceed 5/6 commands,
  // therefore a quick sort/search will probably not be worth it
  Result:= 0;
  While (Result < fCount) Do Begin
    // Compare command
    CheckRes:= AnsiCompareText(fCommands[Result].Command, Command);
    // If position found then break out
    If CheckRes >= 0 Then Exit;
    // Not found, then next
    Inc(Result);
  End;
End;

(******************************************************************************
  Object: TFileAssociation
  Procedure  Find

  Description:  returns the index at which the command exists
                -1 if not exists

  Date last Modified: 28 May 2000
******************************************************************************)

Function TFileAssociation.Index(Command: String): Integer;

Var
  Idx: Integer;

Begin
  // Find where should be insert
  Idx:= Find(Command);
  // check that a match is (not) found at index
  If ((Idx >= fCount) or (Lowercase(fCommands[Idx].Command) <> Lowercase(Command))) Then Begin
    Result:= -1; // Not found
  End
  Else Begin
    Result:= Idx; // Found
  End;
End;

(******************************************************************************
  Object: TFileAssociation
  Procedure  GetCSVConfig

  Description:  returns a configuration string - used to store/recreate object

  Date last Modified: 28 May 2000
******************************************************************************)

Function   TFileAssociation.GetCSVConfig: String;

Var
  Idx: Integer;

Begin
  // Create string list to create
  With TStringList.Create Do Begin
    Try
      // Add Name, description and Extenstions and Owns
      Add('Name=' + fName);
      Add('Desc='+Description);
      Add('Exts=' + fExts.CommaText);
      If fOwnAssociation  Then Add('Owns=True') Else Add('Owns=False');
      Add('Icon=' + IntToStr(fIconIndex));
      // Add commands
      Idx:= 0;
      While Idx < fCount Do Begin
         // add a csv type file
         Add('cmds="'+fCommands[Idx].Command + '","' + fCommands[Idx].Option + '"');
         Inc(Idx);
      End;
      // return comma text files
      Result:= CommaText;
    Finally
      Free;
    End;
  End;
End;

(******************************************************************************
  Object: TFileAssociation
  Procedure  RegisterAssociation

  Description: Registers the File Association and Commands

  Date last Modified: 28 May 2000
******************************************************************************)

Function   TFileAssociation.RegisterAssociation(Topic: String; Var AutoReplace: Boolean): Boolean;

Var
  Response, Idx: Integer;
  Reg: TRegistry;
  Replace: Boolean;
  OldValue, DDEAppTitle, TempExt: String;

  (****************************************************************************
    Procedure  WriteShellCommand

    Description: Writes a shell command to registry
                 A "local procedure" - on balance due to the limited calls
                 this is likely to make (and mainly on initial install)
                 we will leave this as a local procedure and the potential extra
                 work for the compiler/code

    Mod: add quotes surrounding command when a space is included - enables commands
    with spaces to funcion correctly.

    Date last Modified: 1 June 2000
  *****************************************************************************)
  Procedure WriteShellCommand(RootKey, Command, Opt, Application, Topic, StartCmd: String);

  Var
    CmdKey: String;

  Begin
    CmdKey:= RootKey + '\' + Command + '\';
    // Mod 1 June 2000
    // Check for no spaces in command - surround in spaces otherwise
    If pos(' ', Command) > 0 Then Command:= '"' + Command + '"';
    // Write command line needed to start application
    If Reg.OpenKey(cmdKey +'\Command', True) Then Reg.WriteString('', Trim(StartCmd));
    // Add quotes around Opt if needed
    If Opt <> '' Then Opt:= ' "' + Opt + '"';
    // write Shell DDE commands to registry
    If Reg.OpenKey(cmdKey +'\ddeexec', True) Then Begin
      Reg.WriteString('', '[' + Command + ' "%1"' + Opt + ']');
      Reg.OpenKey(cmdKey +'\ddeexec\Application', True);
      Reg.WriteString('', Application);
      Reg.OpenKey(cmdKey +'\ddeexec\Topic', True);
      Reg.WriteString('', Topic);
    End;
    Reg.CloseKey;
  End;



Begin
  // Open Registry in HKey_Classes
  Reg:= TRegistry.Create;
  Try
    Reg.RootKey:= HKEY_CLASSES_ROOT;
    Result:= False;

    // For each Extensions
    Idx:= fExts.Count;
    While Idx > 0 Do Begin
      Dec(Idx);
      // Open Ext Key - existing
      If Reg.OpenKey('\.'+ fExts[Idx], False) Then Begin
        // Read current value
        OldValue:= Reg.ReadString('');
        // Check whether to replace
        Replace:= AutoReplace or (OldValue = fName);
        // If not automatic then ask user
        If Not (Replace) Then Begin
          Response:= MessageBox(GetActiveWindow, PChar('Override File type for ext "'+fExts[Idx]+'"'), 'Override File Association', MB_ICONQUESTION + MB_YESNO);
          Replace:= Response = idYes;
        End;
        // Write old value to allow recovery
        Try
          If (OldValue <> '') and (OldValue <> fName) and Not Reg.ValueExists('Old Value') Then Begin
            Reg.WriteString('Old Value', OldValue);
          End;
        Except
        End;
        Reg.CloseKey;
      End
      Else Replace:= True;
      // Create Ext entry
      If Replace Then Begin
        If Reg.OpenKey('\.'+ fExts[Idx], True) Then Begin
          Reg.WriteString('', fName);
          Reg.CloseKey;
        End;
      End;
    End;

    // If own the Association then delete all previous - ensure clean slate
    If fOwnAssociation and Reg.KeyExists('\'+ fName) Then Reg.DeleteKey('\'+ fName);

    // Build A string for DDE Title
    DDEAppTitle:= ExtractFileName(ParamStr(0));
    TempExt:= ExtractFileExt(DDEAppTitle);
    DDEAppTitle:= copy(DDEAppTitle, 1, Length(DDEAppTitle) - length(TempExt));

    // Open/Create File Type entry
    If Reg.OpenKey('\'+ fName, fOwnAssociation) Then Begin
      // Write description (only if own Association)
      If fOwnAssociation Then Begin
        If fDescription = ''
          Then Reg.WriteString('', fDescription)
          Else Reg.WriteString('', fName);
        If (Reg.OpenKey('\'+ fName + '\DefaultIcon', True)) Then Begin
          Reg.WriteString('', ParamStr(0) +',' + IntToStr(fIconIndex));
        End
      End;
      // Create File Asscoiation commands entries
      If Reg.OpenKey('\'+ fName +'\shell', True) Then Begin
        // No Commands then add OpenFile
        If (fCount = 0) and fOwnAssociation Then AddFileOpen('');
        //  Do for each command - create extensions
        For Idx:= 0 to fCount - 1 Do Begin
          WriteShellCommand('\'+ fName +'\shell\', fCommands[Idx].Command, fCommands[Idx].Option, DDEAppTitle,  Topic, ParamStr(0) + ' ' + fBlankStartOpt);
        End;
      End;
    End;
  Finally
    Reg.Free;
  End;
End;

(******************************************************************************
  Object: TFileAssociation
  Procedure  UnRegisterAssociation

  Description: Unregisters the File Association and Commands

  Date last Modified: 28 May 2000
******************************************************************************)

Function   TFileAssociation.UnRegisterAssociation: Boolean;

Var
  Idx: Integer;
  Reg: TRegistry;

Begin
  Reg:= TRegistry.Create;
  Try
    Reg.RootKey:= HKEY_CLASSES_ROOT;
    Result:= True;

    // Check each extensions
    Idx:= fExts.Count;
    While Idx > 0 Do Begin
      Dec(Idx);
      // Does Key exists
      If Reg.OpenKey('\.'+ fExts[Idx], False) Then Begin
        // If Not this app - leave alone
        If lowercase(Reg.ReadString('')) = lowercase(fName) Then Begin
          // Is there an old value to restore
          If Reg.ValueExists('Old Value') and (Reg.GetDataType('Old Value') = rdString) Then Begin
            // Restore old value
            Reg.WriteString('', Reg.ReadString('Old Value'));
            Reg.DeleteValue('Old Value');
            Reg.CloseKey;
          End
          Else Begin
            // No old then Delete Key
            Reg.CloseKey;
            Reg.DeleteKey('\.'+ fExts[Idx]);
          End;
        End;
      End;
    End;

    // Check still exists
    If Reg.KeyExists('\'+ fName) Then Begin
      // Check whether owner
      If fOwnAssociation Then Begin
        // If Owner - Delete the Key
        Result:= Reg.DeleteKey('\'+ fName);
      End
      Else Begin
        // Not owner just delete command keys owned by self
        Idx:= 0;
        While Idx < fCount Do Begin
          If Reg.KeyExists('\'+ fName +'\shell\' + fCommands[Idx].Command) Then Begin
            Reg.DeleteKey('\'+ fName +'\shell\' + fCommands[Idx].Command);
          End;
          Inc(Idx);
        End;
      End
    End;
  Finally
    Reg.Free;
  End;
End;






(******************************************************************************
  Object: TShellCommandInterface
  Constructor: Create - Set up

  Date last Modified: 28 May 2000 30 June 1999
******************************************************************************)

Constructor TShellCommandInterface.Create(AOwner: TComponent);

Begin
  Inherited Create(AOwner);
  // Remember old - probably not needed since unlikey to be assigned yet
  fOldMacroExecute:= TDdeServerConv(Self).OnExecuteMacro;
  // Set onExecute - Note we have redifined onMacroExecute therefore must
  // Ensure that old objects
  TDdeServerConv(Self).OnExecuteMacro:= MacroExecute;
  // list of commands (Create/Store Strings any way)
  fCommandStrings:= TStringList.Create;
  // Ensure blank
  fOnFormCreate:= Nil;

  fBlankStartOption:= '';
  fAutoProcessCmdLine:= True;
End;

Destructor TShellCommandInterface.Destroy;

Begin
  fCommandStrings.Free;
  Inherited Destroy;
End;

(******************************************************************************
  Object: TShellCommandInterface
  Procedure  SetCommandStrings

  Description: Copys the strings into the Command Strings

  Date last Modified: 28 May 2000
******************************************************************************)

Procedure TShellCommandInterface.SetCommandStrings(NewValues: TStrings);

Begin
  fCommandStrings.Assign(NewValues);
End;

(******************************************************************************
  Object: TShellCommandInterface
  Procedure  ProcessCommand

  Description: Triggers the appropiate command event

  Date last Modified: 28 May 2000
******************************************************************************)
Function TShellCommandInterface.ProcessCommand(Command, FileName, Option: String): Boolean;

Begin
  Result:= False;
  Try
    // Check if command is standard and event is assigned
    If Assigned(fOpenFile) and (Lowercase(Command) = LowerCase(FileOpenCmdStr)) Then Begin
      fOpenFile(FileName, Option);
      Result:= True;
    End;
    If Assigned(fNewFile) and (Lowercase(Command) = LowerCase(FileNewCmdStr)) Then Begin
      fNewFile(FileName, Option);
      Result:= True;
    End;
    If Assigned(fPrint) and (Lowercase(Command) = LowerCase(FilePrintCmdStr)) Then Begin
      fPrint(FileName, Option);
      Result:= True;
    End;
    If Assigned(fPrintTo) and (Lowercase(Command) = LowerCase(FilePrintToCmdStr)) Then Begin
      fPrintTo(FileName, Option);
      Result:= True;
    End;
    // If not handled and custom is assigned then pass to custom
    // e.g. if cmd is open but fOpenFile not assigned then fcustom will be tried
    If Not(Result) and Assigned(fCustomDefaultCommand) Then Begin
      fCustomDefaultCommand(Command, FileName, Option);
      Result:= True;
    End;
  Except
  End;
End;

Procedure TShellCommandInterface.MacroExecute(Sender: TObject; Msg: TStrings);

Var
  Decode, PassMsgs: TStrings;
  Stp, iMacros, iOptions: Integer;
  Command, FileName, Options, Cmd: String;

begin
  Decode:=   TStringList.Create; // to Split command into Command, File Name and Options
  PassMsgs:= TStringList.Create; // Non Commands to pass back to old OnMacroExecute
  Try
    // For each command
    Stp:= Msg.Count;
    IMacros:= 0;
    While iMacros < Stp Do Begin
      Command:= ''; FileName:= ''; Options:= '';
      // Cache Cmd - save calling Object function too often
      Cmd:= Msg[iMacros];
      // Commands surrounded by '[' and ']'
      If (Length(Cmd) > 2) and (Cmd[1] = '[') and (Cmd[Length(Cmd)] = ']') Then Begin
        // Split command up into - Command, File Name and Options
        Decode.CommaText:= Copy(Cmd, 2, Length(Cmd) - 2);
        If Decode.Count > 0 Then Begin
          Command:= Decode[0];
          If Decode.Count > 1 Then Begin
            FileName:= Decode[1];
            iOptions:= 2;
            While iOptions < Decode.Count Do Begin
              If  Options <> ''
                Then Options:= Options + ' ' + Decode[iOptions]
                Else Options:= Decode[iOptions];
              Inc(iOptions);
            End;
          End;
          If Not ProcessCommand(Command, FileName, Options) Then Begin
            PassMsgs.Add(Cmd);
          End;
        End
      End
      Else PassMsgs.Add(Cmd); // Command not valid - pass on to old procedure
      Inc(iMacros);
    End;
    // If needed try old OnmacroExecute
    If Assigned(fOldMacroExecute) and  (PassMsgs.Count > 0) Then Begin
      Try
        fOldMacroExecute(Sender, PassMsgs);
      Except
      End;
    End;
  // Tidy up
  Finally
    PassMsgs.Free;
    Decode.Free;
  End;
End;

(******************************************************************************
  Object: TShellCommandInterface
  Procedure  RegisterFileAssociations

  Description: Writes FileAssociations to registry

  Date last Modified: 28 May 2000
******************************************************************************)

Procedure   TShellCommandInterface.RegisterFileAssociations(AutoReplace: Boolean);

Var
  Idx: Integer;
  SingleAssociation: Boolean;
  Topic: String;

Begin
  Idx:= fCommandStrings.Count;
  Topic:= Name;  // Advoid repated function calls
  If Topic = '' Then raise Exception.Create('No File Type');
  // Check whether single association only
  SingleAssociation:= Idx = 1;
  While Idx > 0 Do Begin
    Dec(Idx);
    // Decode String into FileAssociation details
    With TFileAssociation.CreateFrom(fCommandStrings[Idx]) Do Begin
      Try
        fBlankStartOpt:= BlankStartOption;
        // Help user - if only a single association and owns then add an commands for assigned
        // events - these will automatically be deleted when fileassociation is unregistered
        If SingleAssociation and OwnAssociation Then Begin
          If Assigned(fOpenFile) and (Index(FileOpenCmdStr) < 0)    Then AddFileOpen('');
          If Assigned(fNewFile)  and (Index(FileNewCmdStr) < 0)     Then AddFileNew('');
          If Assigned(fPrint)    and (Index(FilePrintCmdStr) < 0)   Then AddFilePrint('');
          If Assigned(fPrintTo)  and (Index(FilePrintToCmdStr) < 0) Then AddFilePrintTo('');
        End;
        // Register association
        RegisterAssociation(Topic, AutoReplace);
      Finally
         // Finished with
         Free;
      End;
    End;
  End;
End;

(******************************************************************************
  Object: TShellCommandInterface
  Procedure  UnRegisterFileAssociations

  Description: Deletes FileAssociation from registry

  Date last Modified: 28 May 2000
******************************************************************************)

Procedure   TShellCommandInterface.UnRegisterFileAssociations;

Var
  Idx: Integer;

Begin
  Idx:= fCommandStrings.Count;
  While Idx > 0 Do Begin
    Dec(Idx);
    // Decode String into FileAssociation details
    With TFileAssociation.CreateFrom(fCommandStrings[Idx]) Do Begin
      Try
        // remove from registry
        UnRegisterAssociation;
      Finally
         // Finished with
         Free;
      End;
    End;
  End;
End;

(******************************************************************************
  Function Split

  Description: Split the line into Command, Filename and option

  Date last Modified: 28 May 2000
******************************************************************************)

Function Split(Param: String; Var Command, FileName, Option: String): Boolean;

Var
  idx: Integer;

Begin
  // Set defaults
  Result:= False;
  Command:= '';
  FileName:= '';
  Option:= '';
  // check whether is the option
  If (Param <> '') and ((Param[1] = '/') or (Param[1] = '-')) Then Begin
    // Find filename start
    Idx:= pos(':', Param);
    If Idx > 0 Then Begin
      // Get Command
      Command:= Copy(Param, 2, idx - 2);
      // Split off filename and Option
      Param:= Copy(Param, idx + 1, Length(Param) - idx);
      // Find Option start
      idx:= pos('/', Param);
      If idx < 1 Then idx:= pos('-', Param);
      If idx < 1 Then idx:= Length(Param) + 1;  // No Option
      // Get FileName and Option
      FileName:= Copy(Param, 1, idx - 1);
      Option:= Copy(Param, idx + 1, Length(Param) - idx);
      Result:= True;
    End;
  End;
End;

(******************************************************************************
  Object: TShellCommandInterface
  Procedure  ProcessCommandLine

  Description: Processes the command line to execute commands

  Date last Modified: 28 May 2000
******************************************************************************)
Procedure   TShellCommandInterface.ProcessCommandLine;

Var
  Idx: Integer;
  Param, Command, FileName, Option: String;

Begin
  // Loop around all commandline paramters (except 0 - exe name)
  Idx:= ParamCount;
  While Idx > 0 Do Begin
    // Cache parameter
    Param:= ParamStr(Idx);
    // Is it a Command
    If Split(Param, Command, FileName, Option) Then Begin
      ProcessCommand(Command, FileName, Option);
    End
    Else Begin
      If (Param <> '') Then Begin
        // Is it a switch
        If (Param[1] <> '/') and (Param[1] <> '-') Then Begin
          // Not then assume it is open file
          ProcessCommand(FileOpenCmdStr, Param, '');
        End
        Else Begin
          // Process switch
          If Assigned(fCommandSwitch)
            Then fCommandSwitch(Copy(Param, 2, Length(Param) -1));
        End;
      End;
    End;
    Dec(Idx);
  End;
End;

(******************************************************************************
  Object: TShellCommandInterface
  Procedure  OnParentFormCreate

  Description: Action to occur when for Form.OnCreate
               If designer wants automatically process command line

  Date last Modified: 28 May 2000
******************************************************************************)

Procedure   TShellCommandInterface.OnParentFormCreate;

Begin
  // Call up user FormCreate (if assigned)
  If Assigned(fOnFormCreate) Then fOnFormCreate(Sender);
  // if required process the Command Line
  If fAutoProcessCmdLine Then ProcessCommandLine;
End;

(******************************************************************************
  Object: TShellCommandInterface
  Procedure  SetParentComponent

  Description: Intercept the form loading to intercept the form's on Create event

  1.02 - Stop it taking the OnFormCreate when in the design stage

  Date last Modified: 9 July 2000
******************************************************************************)

Procedure TShellCommandInterface.SetParentComponent(Value: TComponent);

Begin
  // Check it a TForm (or decendant) and not the design stage
  If (Value is TForm) and not (csDesigning In ComponentState)  Then Begin
    // Remember user on create
    fOnFormCreate:= TForm(Value).OnCreate;
    // Sub our own
    TForm(Value).OnCreate:= OnParentFormCreate;
  End;
End;


end.
