unit EditFileAssociation;

(***************************************************************************************
 Copyright Ken_Miles 2000

 For conditions of use please see TShellCommandInterface_ReadMe.Txt or
 copyright in TShellCommandInterface.hlp
****************************************************************************************)

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Grids, StdCtrls, Buttons, Menus, Shell_DDE, ComCtrls;

type

(***************************************************************************************
  Object:  TEdtFileAssociation

  Comments: A dialog for editing the file association details
            Note: checking occurs when user presses ok button - errors will stop dialog
            closing on OK

  Version:  1.00
  Date Last Modified: 28 May 2000
****************************************************************************************)

  TEdtFileAssociation = class(TForm)
    memExts: TMemo;
    edtType: TEdit;
    edtDescription: TEdit;
    Label1: TLabel;
    Label2: TLabel;
    Label3: TLabel;
    Label4: TLabel;
    Label5: TLabel;
    btnOK: TBitBtn;
    btnCancel: TBitBtn;
    sgCommands: TStringGrid;
    btnAdd: TBitBtn;
    btnDelete: TBitBtn;
    pmAddButton: TPopupMenu;
    FileOpen1: TMenuItem;
    FileNew1: TMenuItem;
    Print1: TMenuItem;
    PrintTo1: TMenuItem;
    Custom1: TMenuItem;
    pmCommands: TPopupMenu;
    AddFileOpen1: TMenuItem;
    AddFileNew1: TMenuItem;
    AddPrint1: TMenuItem;
    AddPrintTo1: TMenuItem;
    AddCustomCommand1: TMenuItem;
    N1: TMenuItem;
    DeleteSelected1: TMenuItem;
    Label6: TLabel;
    cbOwnsAssociation: TCheckBox;
    Label7: TLabel;
    edtInconIndex: TEdit;
    udIconIndex: TUpDown;
    procedure btnAddClick(Sender: TObject);
    procedure pmAddButtonPopup(Sender: TObject);
    procedure AddCommandClick(Sender: TObject);
    procedure btnDeleteClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure btnOKClick(Sender: TObject);
    procedure pmCommandsPopup(Sender: TObject);
  private
    Procedure AddNewCommand(Name: String);
    Function  DeleteCommand(Idx: Integer): Boolean;
    Function  CheckDuplicates(FailMsg, FailTitle: String; Check: TStrings): Boolean;
    Function  Enable(CheckFor: String; MenuItem: TMenuItem): Boolean;
    Function  EnableNew(Open, New, Print, PrintTo: TMenuItem): Boolean;
    Function  CheckExts: Boolean;
    Function  CheckCmds: Boolean;
  public
    { Public declarations }
    Function  EditAssociation(FileAssociation: TFileAssociation): Boolean;
  end;

// Create Dialog and allow user to edit file association details
Function  Edit_File_Association(Parent: TForm; FileAssociation: TFileAssociation): Boolean;

{---------------------------------------------------------------------------------------

---------------------------------------------------------------------------------------}
implementation

{$R *.DFM}
{---------------------------------------------------------------------------------------

---------------------------------------------------------------------------------------}


(***************************************************************************************
  Object:   TEdtFileAssociation
  Function: FormCreate

  Comments: fill in first/title row of string grid control

  Version:  1.00
  Date Last Modified: 28 May 2000
****************************************************************************************)

Procedure TEdtFileAssociation.FormCreate(Sender: TObject);

begin
  sgCommands.Rows[0].CommaText:= 'Command,Options';
end;

(***************************************************************************************
  Object:   TEdtFileAssociation
  Function: Enable

  Comments: Check to see whether command already exists - if so disable appropriate
            menu item

  Version:  1.00
  Date Last Modified: 28 May 2000
****************************************************************************************)

Function  TEdtFileAssociation.Enable(CheckFor: String; MenuItem: TMenuItem): Boolean;

Begin
  // Search list for command - not found = -1
  Result:= sgCommands.Cols[0].IndexOf(CheckFor) <= 0;
  // Disable menuitem - if one has been passed
  If MenuItem <> Nil then MenuItem.Enabled:= Result;
End;

(***************************************************************************************
  Object:   TEdtFileAssociation
  Function: Enable

  Comments: Check to see whether list of command already exists - if so disable appropriate
            menu item and return false if all already exist

  Version:  1.00
  Date Last Modified: 28 May 2000
****************************************************************************************)
Function  TEdtFileAssociation.EnableNew(Open, New, Print, PrintTo: TMenuItem): Boolean;

Begin
  // Assume that all exists
  Result:= False;
  // check for standard commands - if one (or more) not found then return true
  If Enable(FileOpenCmdStr, Open) Then Result:= True;
  If Enable(FileNewCmdStr, New) Then Result:= True;
  If Enable(FilePrintCmdStr, Print) Then Result:= True;
  If Enable(FilePrintToCmdStr, PrintTo) Then Result:= True;
End;


(***************************************************************************************
  Object:   TEdtFileAssociation
  Function: AddNewCommand

  Comments: Add new command to the list in the string grid

  Version:  1.00
  Date Last Modified: 28 May 2000
****************************************************************************************)
Procedure TEdtFileAssociation.AddNewCommand(Name: String);

Begin
  // Check that first line and only line is not blank - if so place command there
  If (sgCommands.RowCount = 2) and (sgCommands.Cols[0].Strings[1] = '')
    Then sgCommands.Cols[0].Strings[1]:= Name
    Else Begin
      // Insert new line at end and add command
      sgCommands.RowCount:= sgCommands.RowCount + 1;
      sgCommands.Cols[0].Strings[sgCommands.RowCount - 1]:= Name;
    End;
  // Check that new command has not triggered scrollbars
  sgCommands.DefaultColWidth:= sgCommands.ClientWidth div 2;
  // must now be something to delete
  btnDelete.Enabled:= True;
End;

(***************************************************************************************
  Object:   TEdtFileAssociation
  Function: btnNewClick

  Comments: Event when user clicks button new
            - brings up a menu if not all standard commands implementeted otherwise adds
            a new custom command

  Version:  1.00
  Date Last Modified: 28 May 2000
****************************************************************************************)
procedure TEdtFileAssociation.btnAddClick(Sender: TObject);

Var
  ScrPos, CliPos: TPoint;

begin
  // Check whether standard commands have all gone
  If EnableNew(Nil, Nil, Nil, Nil) Then Begin
    // If not - work out there button is on screen
    CliPos.x:= 0;
    CliPos.y:= btnAdd.Height;
    ScrPos:= btnAdd.ClientToScreen(CliPos);
    // Popup menu under button
    pmAddButton.Popup(ScrPos.x,ScrPos.y);
  End
  // new custom command
  Else AddNewCommand('Custom_Command');
end;

(***************************************************************************************
  Object:   TEdtFileAssociation
  Function: pmAddButtonPopup/pmCommandsPopup

  Comments: Event when the popup menu asscoaciated with the add button/Commands String Grid
            - enables/disables the menuitems as appropriate

  Version:  1.00
  Date Last Modified: 28 May 2000
****************************************************************************************)

procedure TEdtFileAssociation.pmAddButtonPopup(Sender: TObject);

Begin
  EnableNew(FileOpen1, FileNew1, Print1, PrintTo1);
end;

Procedure TEdtFileAssociation.pmCommandsPopup(Sender: TObject);

Var
  Idx: Integer;

begin
  // Enable New commands
  EnableNew(AddFileOpen1, AddFileNew1, AddPrint1, AddPrintTo1);
  // Set delete enabled to match detelete button
  DeleteSelected1.Enabled:= btnDelete.Enabled;
  // get selected title and add it to delete menu item's caption
  Idx:= sgCommands.Selection.Top;
  If (Idx > 0) and btnDelete.Enabled
    Then DeleteSelected1.Caption:= 'Delete ' + sgCommands.Rows[Idx].Strings[0]
    Else DeleteSelected1.Caption:= 'Delete Selected';
End;


(***************************************************************************************
  Object:   TEdtFileAssociation
  Function: AddCommandClick

  Comments: Event when an Add command menuitem is clicked - commands added is dependant
            upon menuitems tag value

  Version:  1.00
  Date Last Modified: 28 May 2000
****************************************************************************************)
Procedure TEdtFileAssociation.AddCommandClick(Sender: TObject);

Var
  Cmd: String;

begin
  // Set Default command
  Cmd:= 'Custom_Command';
  // Get the command approriate for the tag value
  If Sender is TComponent Then Begin
    Case TComponent(Sender).Tag of
      1: Cmd:= FileOpenCmdStr;
      2: Cmd:= FileNewCmdStr;
      3: Cmd:= FilePrintCmdStr;
      4: Cmd:= FilePrintToCmdStr;
    End;
  End;
  // Add the command
  AddNewCommand(Cmd);
end;

(***************************************************************************************
  Object:   TEdtFileAssociation
  Function: DeleteCommand

  Comments: Deletes the command at index Idx

  Version:  1.00
  Date Last Modified: 28 May 2000
****************************************************************************************)
Function  TEdtFileAssociation.DeleteCommand(Idx: Integer): Boolean;

Var
  Stp: Integer;

Begin
  // Check valid index passed - note 0 row is title so not valid index
  Result:= False;
  If (0 >= Idx) or (Idx >= sgCommands.RowCount) Then Exit;
  // ripple commands down to fill hole created
  Stp:= sgCommands.RowCount - 1;
  While (Idx < Stp)  Do Begin
    sgCommands.Rows[Idx].CommaText:= sgCommands.Rows[Idx + 1].CommaText;
    Inc(Idx);
  End;
  // Do not drop to no commands - causes problems with string grid and fixed rows
  If sgCommands.RowCount > 2
    Then sgCommands.RowCount:= sgCommands.RowCount -1 // Chop last line off
    Else Begin
      // Deleting only command
      // blank command out
      sgCommands.Rows[Idx].CommaText:= '';
      // nothing left to delete
      btnDelete.Enabled:= False;
    End;
  // Actually deleted something
  Result:= True;
end;


(***************************************************************************************
  Object:   TEdtFileAssociation
  Function: btnDeleteClick

  Comments: Event when user clicks delete button/menu item

  Version:  1.00
  Date Last Modified: 28 May 2000
****************************************************************************************)
Procedure TEdtFileAssociation.btnDeleteClick(Sender: TObject);

begin
  // Delete selected - out of index handled by delete command
  DeleteCommand(sgCommands.Selection.Top);
end;

(***************************************************************************************
  Function: CheckForIllegalChar/IllegalChar

  Comments: Check string for illegal characters (OS files, etc.)

  Version:  1.00
  Date Last Modified: 28 May 2000
****************************************************************************************)

Function  IllegalChar(Input: Char): Boolean;

Begin
  // Is character is one of these
  Result:=   (Input = '\') or (Input = '/')
          or (Input = ':') or (Input = '*')
          or (Input = '?') or (Input = '"')
          or (Input = '<') or (Input = '>')
          or (Input = '|') or (Input = ' ')
          or (Input = '.') or (Ord(Input) < 33);
End;

Function  CheckForIllegalChar(Input: String): Boolean;

Var Idx: Integer;

Begin
  Result:= False; // Assume no illegal characters
  If Input = '' Then Exit; // No string - no illegal chars
  // Search until end of string (Working right to left) or illegal char
  Idx:= length(Input);
  While (Idx > 0) and Not Result Do Begin
    Result:= IllegalChar(Input[Idx]);
    Dec(Idx);
  End;
End;


(***************************************************************************************
  Object:   TEdtFileAssociation
  Function: CheckDuplicates

  Comments: Checks a TStrings object for any duplicate entries

  Version:  1.00
  Date Last Modified: 28 May 2000
****************************************************************************************)

Function  TEdtFileAssociation.CheckDuplicates(FailMsg, FailTitle: String; Check: TStrings): Boolean;

Var
  Idx: Integer;
  Errs: String;

Begin
  Errs:= '';
  Result:= True;
  // Note: Working backwards, i.e. last to first
  Idx:= Check.Count;
  While Idx > 1 Do Begin
    Dec(Idx);
    { If we search a Tstrings object for a string at index Idx then if no duplicates then
      the first position should be Idx, a lower value means that a duplicate was found
      no find isn't an option
      since we have already checked against all before them - any duplicates after IDX have
      already been found and do not need recording (and we can forget the index 0, since none before)
      - tripplicate entries add more than one entry - what a shame}
    If Check.IndexOf(Check[Idx]) < Idx Then Begin
      Result:= False;
      // Record duplicates so as to identify them to the user
      If Errs = '' Then Errs:= Check[Idx] Else Errs:= Errs + ', ' + Check[Idx];
    End;
  End;
  If Not Result Then Begin
    // Tell user about duplicates found
    MessageBoxA(Handle, PChar(FailMsg + ' ' + errs), PChar(FailTitle), MB_ICONERROR);
  End;
End;

(***************************************************************************************
  Object:   TEdtFileAssociation
  Function: CheckExts

  Comments: Checks user has entered a valid set of extensions
            - by valid we mean no doggy charaters
            leading "." will be trimed as will leading/trailing blank spaces and empty strings

  Version:  1.00
  Date Last Modified: 28 May 2000
****************************************************************************************)

Function  TEdtFileAssociation.CheckExts: Boolean;

Var
  Idx: Integer;
  Ext, Errs: String;

Begin
  Result:= True;
  Errs:= '';  // No errors yet
  // Note Working last to first - when lines deleted no clever tricks needed with index
  Idx:= memExts.Lines.Count;
  While Idx > 0 Do Begin
    Dec(Idx);
    //Cache Ext for working on - do not keep going to tstringlist
    Ext:= Trim(memExts.Lines[Idx]);
    // Strip leading trailing spaces and leading "."
    If Ext[1] = '.' Then Ext:= Trim(Copy(Ext, 2, Length(Ext) - 1));
    // Check for empty string
    If (Ext = '') or (Ext = '.') Then Begin
      // Delete empty strings
      memExts.Lines.Delete(Idx);
    End
    Else Begin
      // Check for illegal characters and record as appropriate
      If CheckForIllegalChar(Ext) Then If Errs = '' Then Errs:= Ext Else Errs:= Errs + ', ' + Ext;
      // Save modified string
      memExts.Lines[Idx]:= Ext;
    End;
  End;

  // If the error string conatins details then tell user - and of course set failed
  If Errs <> '' Then Begin
    Result:= False;
    MessageBoxA(Handle, PChar('There is a problem with the extensions: ' + errs), 'Errors', MB_ICONERROR);
  End;
  // Check whether any duplicate extensions - fail if so
  // Note: If line was Result:= Result and CheckDuplicates( ..) then CheckDuplicates would only be run
  //       when no errors in illegal characters in extensions
  Result:= CheckDuplicates('Duplicate Extensions:', 'Duplicate Extensions', memExts.Lines) and Result;
End;

(***************************************************************************************
  Object:   TEdtFileAssociation
  Function: CheckCmds

  Comments: Check the commands for blanks (delete them) and duplicates - tell user

  Version:  1.00
  Date Last Modified: 28 May 2000
****************************************************************************************)
Function  TEdtFileAssociation.CheckCmds: Boolean;

Var
  Idx: Integer;

Begin
  // Work though commands and delete any blanks - note work backwards to remove any need to
  // to jump when deleting commands
  Idx:= sgCommands.RowCount;
  While Idx > 1 Do Begin
    Dec(Idx);
    // If there are no commands then there being a blankline ensures that this is always called
    // not a problem though
    If sgCommands.Cols[0][idx] = '' Then DeleteCommand(Idx);
  End;
  // Check for any duplicates - tell user about them
  Result:= CheckDuplicates('Duplicate Commands: ', 'Duplicate Commands', sgCommands.Cols[0]);
End;


(***************************************************************************************
  Object:   TEdtFileAssociation
  Function: btnOKClick

  Comments: Event when user clicks the ok button - check things before closing form

  Version:  1.00
  Date Last Modified: 28 May 2000
****************************************************************************************)

procedure TEdtFileAssociation.btnOKClick(Sender: TObject);

Var
  OkType: Boolean;
  CacheType: String;

Begin
  // Cache the type - do not want to keep calling a edit property
  CacheType:= edtType.Text;
  // Check that valid type entered - by valid type we mean checking for empty strings
  // or illegal characters -  note different characters than above - '.' particually
  // Only one string to search, keep it simple (multiple scans of string using pos might be slower)
  OkType:= (CacheType <> '') and (CacheType[1] <> '.') and
           (Pos('/', CacheType) = 0) and (Pos('\', CacheType) = 0) and
           (Pos('*', CacheType) = 0) and (Pos('?', CacheType) = 0) and
           (Pos('"', CacheType) = 0) and (Pos('''', CacheType) = 0) and
           (Pos('<', CacheType) = 0) and (Pos('>', CacheType) = 0) and
           (Pos(':', CacheType) = 0) and (Pos('|', CacheType) = 0);
  // Tell user if an error was found
  If Not OkType Then MessageBoxA(Handle, 'Error in file type', 'Error File Type', MB_ICONERROR);

  // Check whether it there is a problem - do not shut if there is
  If Not(CheckExts and CheckCmds and OkType) Then ModalResult:= mrNone;
end;


(***************************************************************************************
  Object:   TEdtFileAssociation
  Function: EditAssociation

  Comments: Reads the values from the Association
            - returns modified values if user presses ok
            Result is whether user pressed ok or cancel

  Version:  1.00
  Date Last Modified: 28 May 2000
****************************************************************************************)


Function  TEdtFileAssociation.EditAssociation(FileAssociation: TFileAssociation): Boolean;

Begin
  // Get Association details
  FileAssociation.FillCmdGrid(sgCommands);
  edtType.Text:=  FileAssociation.Name;
  edtDescription.Text:= FileAssociation.Description;
  memExts.Lines.Assign(FileAssociation.Exts);
  btnDelete.Enabled:= FileAssociation.Count > 0;
  cbOwnsAssociation.Checked:= FileAssociation.OwnAssociation;
  udIconIndex.Position:= FileAssociation.IconIndex;
  // Show dialog
  Result:=ShowModal = mrOk;
  If Result Then Begin
    // update association if user presses ok
    FileAssociation.GetFromCmdGrid(sgCommands);
    FileAssociation.Name:= edtType.Text;
    FileAssociation.Description:= edtDescription.Text;
    FileAssociation.Exts.Assign(memExts.Lines);
    FileAssociation.OwnAssociation:= cbOwnsAssociation.Checked;
    FileAssociation.IconIndex := udIconIndex.Position;
  End;
End;

(***************************************************************************************
  Function: Edit_File_Association

  Comments: Creates a dialog box (TEdtFileAssociation) allows it to edit the
            Association and frees it afterwards
            Result is whether user pressed ok or cancel

  Version:  1.00
  Date Last Modified: 28 May 2000
****************************************************************************************)

Function  Edit_File_Association(Parent: TForm; FileAssociation: TFileAssociation): Boolean;

Begin
  With TEdtFileAssociation.Create(Parent) Do Begin
    Try
      Result:= EditAssociation(FileAssociation);
    Finally
      Free;
    End;
  End;
End;


end.
