unit sgr_scale;
{(c) S.P.Pod'yachev 1998}
{***************************************************}
{ Auxiliary persistent object for using             }
{ in components with scale and axis                 }
{***************************************************}
interface
uses
   Windows, Messages, SysUtils, Classes, Graphics, Controls, ExtCtrls,
   Forms, Dialogs,
   sgr_misc;
Type
   Tsp_CustomLineAttr = class(TPersistent)
   private
      fColor: TColor;
      fStyle: TPenStyle;
      fWidth: word;
      fMode: TPenMode;
      fVisible: boolean;
      fOnChange: TNotifyEvent;
      procedure SetColor(V: TColor);
      procedure SetStyle(V: TPenStyle);
      procedure SetWidth(V: word);
      procedure SetMode(V: TPenMode);
      procedure SetVisible(V: boolean);
   protected
      procedure Changed; virtual;
   public
      constructor Create;
      procedure Assign(Source: TPersistent); override;
      procedure AssignTo(Dest: TPersistent); override;
      procedure SetPenAttr(const APen: TPen);
      property Color: TColor read fColor write SetColor;
      property Style: TPenStyle read fStyle write SetStyle;
      property Width: word read fWidth write SetWidth;
      property Mode: TPenMode read fMode write SetMode;
      property Visible: boolean read fVisible write SetVisible;
      property OnChange: TNotifyEvent read FOnChange write FOnChange;
   end;
   Tsp_LineAttr = class(Tsp_CustomLineAttr)
   public
      function IsSame(const LA: Tsp_CustomLineAttr): boolean;
   published
      property Color;
      property Style default psSolid;
      property Width default 1;
      property Visible;
   end;
Const
   MaxTicksCount = 21;
   fbposVertical = 0;
   fbposInversed = 1;
   fbposNegMetr = 2;
   fbposNoTicks = 3;
   fbposNoTicksLabel = 4;
   fbposRevertTicks = 5;
   fbposAutoMin = 6;
   fbposAutoMax = 7;
   sdfVertical = 1 shl fbposVertical;
   sdfInversed = 1 shl fbposInversed;
   sdfNegMetr = 1 shl fbposNegMetr;
   sdfNoTicks = 1 shl fbposNoTicks;
   sdfNoTicksLabel = 1 shl fbposNoTicksLabel;
   sdfRevertTicks = 1 shl fbposRevertTicks;
   sdfLabelAtTop = sdfRevertTicks;
   sdfLabelOnRight = sdfRevertTicks;
   sdfAutoMin = 1 shl fbposAutoMin;
   sdfAutoMax = 1 shl fbposAutoMax;
Type
   Tsp_Scale = class(TPersistent)
   private
      IMin, IMax: double;
      IntFactor: double;
      IStep: integer;
      rTksCount: Byte;
      fOPos: integer;
      fOVal  : double;
      fM: Double;
      procedure ReadFlags(Reader: TReader);
      procedure WriteFlags(Writer: TWriter);
      procedure ReadLinePos(Reader: TReader);
      procedure WriteLinePos(Writer: TWriter);
      function  GetVisible: boolean;
   protected
      O: TPoint;
      fLen: integer;
      fLineAttr: Tsp_LineAttr;
      fFlags: integer;
      fMin, fMax: double;
      fTicksCount: Byte;
      fLabelFormat: string;
      procedure DefineProperties(Filer: TFiler); override;
      procedure CalcMetr;
      procedure CalcTicksPos;
      procedure TreatMinMax(Mi, Ma: double);
      function  TickLabel(tickNum: integer): string; virtual;
      function  GetTicksCount: byte;
      procedure SetFlagBit(BN: integer; On: boolean);
      function  GetFlagBit(BN: integer): boolean;
      procedure FlagsChanged(BN: integer; On: boolean); virtual;
   public
      TksPos: array [0..MaxTicksCount - 1] of smallInt;
      TksDbl: array [0..MaxTicksCount - 1] of double;
      constructor Create(Flags: integer);
      destructor Destroy; override;
      function BandWidth(FntWidth, FntHeight: integer): integer;
      function OrgIndent(FntWidth, FntHeight: integer): integer;
      function EndIndent(FntWidth, FntHeight: integer): integer;
      function CalcDrawBounds(fCanvas: TCanvas): TRect;
      procedure DrawLine(fCanvas: TCanvas; odec, einc: word);
      procedure DrawTicks(fCanvas: TCanvas);
      procedure SetMetrics(Min, Max: double; oX, oY, lLen: integer);
      procedure SetLine(oX, oY, lLen: integer);
      procedure SetMinMax(aMin, aMax: double);
      procedure ScrollBy(delta: integer);
      procedure OnScrollEnd;
      function V2P(const V: double): integer;
      function P2V(const V: integer): double;
      property OPos: integer read fOPos;
      property OVal: double  read fOVal;
      property SM: double read fM;
      property Visible: boolean read GetVisible;
      property OX: Longint read O.x;
      property OY: Longint read O.y;
      property Len: integer read fLen;
   published
      property Inversed: boolean index fbposInversed read GetFlagBit
      write SetFlagBit stored False;
      property NoTicksLabel: boolean index fbposNoTicksLabel read GetFlagBit
      write SetFlagBit stored False;
      property NoTicks: boolean index fbposNoTicks read GetFlagBit
      write SetFlagBit stored False;
   end;
   TTestScale = class(TGraphicControl)
   private
      A, B: Tsp_Scale;
      procedure SetA(V: Tsp_Scale);
   public
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;
      procedure Paint; override;
   published
      property Align;
      property Axis: Tsp_Scale read A write SetA;
   end;
IMPLEMENTATION
procedure Tsp_CustomLineAttr.Changed;
begin
   if Assigned(fOnChange) then fOnChange(Self);
end;
procedure Tsp_CustomLineAttr.SetColor(V: TColor);
begin
   if V<>fColor then fColor := V;
   Changed;
end;
procedure Tsp_CustomLineAttr.SetStyle(V: TPenStyle);
begin
   if V<>fStyle then fStyle := V;
   Changed;
end;
procedure Tsp_CustomLineAttr.SetWidth(V: word);
begin
   if V<>fWidth then fWidth := V;
   Changed;
end;
procedure Tsp_CustomLineAttr.SetVisible(V: boolean);
begin
   if V<>fVisible then fVisible := V;
   Changed;
end;
procedure Tsp_CustomLineAttr.SetMode(V: TPenMode);
begin
   if V<>fMode then fMode := V;
   Changed;
end;
constructor Tsp_CustomLineAttr.Create;
begin
   inherited Create;
   fOnChange := nil;
   fColor := clBlack;
   fStyle := psSolid;
   fWidth := 1;
   fVisible := True;
end;
procedure Tsp_CustomLineAttr.Assign(Source: TPersistent);
var ss: Tsp_CustomLineAttr;
begin
   if Source is Tsp_CustomLineAttr then
   begin
      ss := Tsp_CustomLineAttr(Source);
      fColor := ss.fColor;
      fStyle := ss.fStyle;
      fWidth := ss.fWidth;
      fVisible := ss.fVisible;
end else inherited Assign(Source);
end;
procedure Tsp_CustomLineAttr.AssignTo(Dest: TPersistent);
begin
   if Dest is Tsp_CustomLineAttr then Dest.Assign(Self)
   else inherited AssignTo(Dest);
end;
procedure Tsp_CustomLineAttr.SetPenAttr(const APen: TPen);
begin
   with APen do begin
      Color := fColor;
      Style := fStyle;
      Width := fWidth;
   end;
end;
function Tsp_LineAttr.IsSame(const LA: Tsp_CustomLineAttr): boolean;
begin
   with LA do
      Result := (fColor = Color) and (fStyle = Style) and
      (fWidth = Width) and (fVisible = Visible);
end;
Const
   Ln10 = 2.30258509299405;
   sdfLineOnly = sdfNoTicks or sdfNoTicksLabel;
procedure Tsp_Scale.CalcMetr;
begin
   if (fFlags and sdfInversed) = 0 then fOVal := fMin else fOVal := fMax;
   if (fFlags and sdfVertical) = 0 then fOPos := O.x else fOPos := O.y;
   if (fFlags and sdfInversed) = 0 then fM := fLen / (fMax - fMin)
   else fM := fLen / (fMin - fMax);
   if (FFlags and sdfVertical)<>0 then fM := - fM;
end;
procedure Tsp_Scale.CalcTicksPos;
procedure LabeledTicks;
var j, VStep: integer; oid: double;
begin
   if fTicksCount>1 then begin
      j := Round(IMax - IMin);
      IStep := j div (fTicksCount - 1);
      if IStep = 0
      then inc(IStep);
      rTksCount := (j div IStep) + 1;
      if rTksCount>MaxTicksCount then rTksCount := MaxTicksCount;
   end else begin
      IStep := Round(IMax - IMin);
      rTksCount := fTicksCount;
   end;
if (fFlags and sdfInversed) = 0 then begin oid := IMin; VStep := IStep; end
else begin oid := IMax; VStep := - IStep; end;
   for j := 0 to rTksCount - 1 do TksDbl[j] := Int(oid + VStep * j) / IntFactor;
   for j := 0 to rTksCount - 1 do TksPos[j] := V2P(TksDbl[j]);
end;
procedure TicksOnly;
var j: integer; PStep: extended;
begin
   rTksCount := fTicksCount;
   if fTicksCount>1 then begin
      PStep := fLen / (fTicksCount - 1);
      if (fFlags and sdfVertical)<>0 then PStep := - PStep;
      for j := 0 to rTksCount - 1 do TksPos[j] := fOPos + round(PStep * j);
      for j := 0 to rTksCount - 1 do TksDbl[j] := P2V(TksPos[j]);
   end else begin
      TksDbl[0] := fOVal;
      TksPos[0] := fOPos;
      if (fFlags and sdfVertical) = 0 then TksPos[1] := fOPos + fLen
      else TksPos[1] := fOPos + fLen;
   end;
end;
begin
if fTicksCount<1 then begin rTksCount := fTicksCount; Exit end;
   if (fFlags and sdfNoTicksLabel) = 0 then LabeledTicks else TicksOnly;
end;
procedure Tsp_Scale.TreatMinMax(Mi, Ma: double);
var  DecmlPos: integer;
procedure Separate;
begin
   if abs(fMax)<10 then begin
      fMax := fMax + 1; fMin := fMax - 1
   end
   else begin
      fMax := fMin + abs(fMax) / 10;
      fMin := fMax - abs(fMax) / 10;
   end;
end;
begin
if Mi>Ma then begin fMin := Ma; fMax := Mi end
else begin fMin := Mi; fMax := Ma end;
   if fMin = fMax then Separate;
   IntFactor := Ln(fMax - fMin) / Ln10;
   DecmlPos := Trunc(IntFactor); if Frac(IntFactor)<0 then dec(DecmlPos);
   if DecmlPos>0 then IntFactor := 1 / Int(exp((DecmlPos - 1) * Ln10))
   else IntFactor := Int(exp( - (DecmlPos - 1) * Ln10));
   IMin := fMin * IntFactor;
   if Frac(IMin)>0 then IMin := Int(IMin) + 1 else IMin := Int(IMin);
   IMax := fMax * IntFactor;
   if Frac(IMax)<0 then IMax := Int(IMax) - 1 else IMax := Int(IMax);
   if (fFlags and sdfInversed) = 0 then fOVal := fMin else fOVal := fMax;
end;
procedure Tsp_Scale.DefineProperties(Filer: TFiler);
begin
   inherited DefineProperties(filer);
   Filer.DefineProperty('SFlags', ReadFlags, WriteFlags, true);
   Filer.DefineProperty('SLinePos', ReadLinePos, WriteLinePos, true);
end;
procedure Tsp_Scale.ReadFlags(Reader: TReader);
begin
   fFlags := Reader.ReadInteger;
end;
procedure Tsp_Scale.WriteFlags(Writer: TWriter);
begin
   Writer.WriteInteger(fFlags)
end;
procedure Tsp_Scale.ReadLinePos(Reader: TReader);
begin
   with Reader do begin
      ReadListBegin;
      SetLine(ReadInteger, ReadInteger, ReadInteger);
      ReadListEnd;
   end
end;
procedure Tsp_Scale.WriteLinePos(writer: TWriter);
begin
   with Writer do begin
      WriteListBegin;
      WriteInteger(O.x);
      WriteInteger(O.y);
      WriteInteger(fLen);
      WriteListEnd;
   end;
end;
function Tsp_Scale.GetVisible: boolean;
begin
   Result := ((fFlags and sdfLineOnly)<>sdfLineOnly) or fLineAttr.Visible;
end;
function Tsp_Scale.TickLabel(tickNum: integer): string;
begin
   Result := FormatFloat(fLabelFormat, TksDbl[tickNum]);
end;
function Tsp_Scale.GetTicksCount: byte;
begin
   Result := rTksCount;
end;
procedure Tsp_Scale.SetFlagBit(BN: integer; On: boolean);
var Mask: integer;
begin
   Mask := 1 shl BN;
   if On then
   begin
      if ((fFlags and Mask) = 0) then begin
         fFlags := fFlags or Mask;
         FlagsChanged(BN, On);
      end;
   end else
      if ((fFlags and Mask)<>0) then begin
      fFlags := fFlags and Not Mask;
      FlagsChanged(BN, On);
   end
end;
function Tsp_Scale.GetFlagBit(BN: integer): boolean;
begin
   Result := (fFlags and (1 shl BN))<>0;
end;
procedure Tsp_Scale.FlagsChanged(BN: integer; On: boolean);
begin
end;
constructor Tsp_Scale.Create(Flags: integer);
begin
   inherited Create;
   fLineAttr := Tsp_LineAttr.Create;
   fTicksCount := 3;
   FFlags := Flags;
   SetMetrics(0.0, 10.0, 10, 30, 25);
end;
destructor Tsp_Scale.Destroy;
begin
   if Assigned(fLineAttr) then fLineAttr.Free;
   inherited Destroy;
end;
procedure Tsp_Scale.SetMetrics(Min, Max: double; oX, oY, lLen: integer);
begin
   TreatMinMax(Min, Max);
   O.x := oX; O.y := oY;
   if (fFlags and sdfVertical) = 0 then fOPos := O.x else fOPos := O.y;
   fLen := lLen;
   CalcMetr;
   CalcTicksPos;
end;
procedure Tsp_Scale.SetLine(oX, oY, lLen: integer);
begin
   if (O.x<>oX) or (O.y<>oY) or (lLen<>fLen) then
   begin
      O.x := oX; O.y := oY;
      if (fFlags and sdfVertical) = 0 then fOPos := O.x else fOPos := O.y;
      if lLen = 0 then inc(lLen) else if lLen<0 then lLen := - lLen;
      fLen := lLen;
      CalcMetr;
      CalcTicksPos;
   end;
end;
procedure Tsp_Scale.SetMinMax(aMin, aMax: double);
begin
   if (aMin<>fMin) or (aMax<>fMax)then begin
      TreatMinMax(aMin, aMax);
      CalcMetr;
      CalcTicksPos;
   end;
end;
procedure Tsp_Scale.ScrollBy(delta: integer);
var dd: double; Step: integer;
procedure SLabeledTicks;
var j, VStep: integer; oid: double; mi, ma: double;
begin
   mi := fMin * IntFactor;
   ma := mi + IStep;
   while IMin<mi do IMin := Int(IMin + IStep);
   while IMin>ma do IMin := Int(IMin - IStep);
   ma := fMax * IntFactor;
   mi := ma - IStep;
   while IMax>ma do IMax := Int(IMax - IStep);
   while IMax<mi do IMax := Int(IMax + IStep);
   if fTicksCount>0 then rTksCount := (Round(ma - IMin) div IStep) + 1
   else rTksCount := fTicksCount;
   if rTksCount>MaxTicksCount then rTksCount := MaxTicksCount;
if (fFlags and sdfInversed) = 0 then begin oid := IMin; VStep := IStep; end
else begin oid := IMax; VStep := - IStep; end;
   for j := 0 to rTksCount - 1 do TksDbl[j] := Int(oid + VStep * j) / IntFactor;
   for j := 0 to rTksCount - 1 do TksPos[j] := V2P(TksDbl[j]);
end;
procedure STicksOnly;
var j, sl: integer; PStep: extended;
begin
   if fTicksCount>1 then PStep := fLen / (fTicksCount - 1) else PStep := fLen;
   sl := round(PStep);
   TksDbl[0] := TksDbl[0] + dd;
   dec(TksPos[0], delta);
   if (fFlags and sdfVertical) = 0 then begin
      while TksPos[0]<fOPos do inc(TksPos[0], sl);
      while TksPos[0]>(fOPos + sl) do dec(TksPos[0], sl);
   end
   else  begin
      while TksPos[0]>fOPos do dec(TksPos[0], sl);
      while TksPos[0]<(fOPos - sl) do inc(TksPos[0], sl);
      PStep := - PStep;
   end;
   if fTicksCount>0 then rTksCount := ((fLen - abs(TksPos[0] - fOPos) - 1) div sl) + 1
   else rTksCount := 0;
   if rTksCount>MaxTicksCount then rTksCount := MaxTicksCount;
   TksDbl[0] := P2V(TksPos[0]);
   for j := 1 to rTksCount - 1 do TksPos[j] := TksPos[0] + round(PStep * j);
   for j := 1 to rTksCount - 1 do TksDbl[j] := P2V(TksPos[j]);
end;
begin
   if delta = 0 then Exit;
   dd := delta / fM;
   if (fTicksCount>1) then Step := round(fLen / (fTicksCount - 1)) else Step := fLen;
   if (abs(delta)>Step) or (fTicksCount<1) then
   begin
      SetMinMax(fMin + dd, fMax + dd);
      Exit;
   end;
if fTicksCount<1 then begin rTksCount := fTicksCount; Exit end;
   fMin := fMin + dd; fMax := fMax + dd;
   if (fFlags and sdfInversed) = 0 then fOVal := fMin else fOVal := fMax;
   if (fFlags and sdfNoTicksLabel) = 0 then SLabeledTicks else STicksOnly;
end;
procedure Tsp_Scale.OnScrollEnd;
begin
   TreatMinMax(fMin, fMax);
   CalcMetr;
   CalcTicksPos;
end;
function Tsp_Scale.V2P(const V: double): integer;
var rr: double;
begin
   rr := fOPos + (fM * (V - fOVal));
   if rr>16383 then Result := 16383
   else if rr< - 16383 then Result := - 16383
      else Result := round(rr);
end;
function Tsp_Scale.P2V(const V: integer): double;
begin
   Result := fOVal + (V - fOPos) / fM;
end;
Const
   TickOfs = 0;
   MnTick = 1;
   MjTick = 4;
   LblOfs = 1;
function Tsp_Scale.BandWidth(FntWidth, FntHeight: integer): integer;
var j, tw: integer;
begin
   Result := fLineAttr.Width;
   if (FFlags and sdfVertical) = 0 then
   begin
      if (rTksCount>0) then begin
         if ((FFlags and sdfNoTicksLabel) = 0) then
            inc(Result, TickOfs + MjTick + LblOfs + FntHeight)
         else if (FFlags and sdfNoTicks) = 0 then inc(Result, TickOfs + MjTick);
      end;
      end else
      begin
         Result := fLineAttr.Width;
         if (rTksCount>0) then begin
            if ((FFlags and sdfNoTicksLabel) = 0) then
            begin
               tw := Length(TickLabel(0));
               for j := 1 to rTksCount - 1 do
                  if tw < Length(TickLabel(j)) then
                  tw := Length(TickLabel(j));
               inc(Result, TickOfs + MjTick + LblOfs + tw * FntWidth);
            end
            else if (FFlags and sdfNoTicks) = 0 then inc(Result, TickOfs + MjTick);
         end;
      end;
   end;
function Tsp_Scale.OrgIndent(FntWidth, FntHeight: integer): integer;
var tp: integer;
begin
   if (rTksCount>0) and ((FFlags and sdfNoTicksLabel) = 0) then
   begin
      tp := abs(round((fOVal - TksDbl[0]) * fM));
      if (FFlags and sdfVertical) = 0 then
         Result := FntWidth * Length(TickLabel(0)) div 2 - tp
      else
         Result := FntHeight div 2 - tp;
      if Result<0 then Result := 0;
end else Result := 0;
end;
function Tsp_Scale.EndIndent(FntWidth, FntHeight: integer): integer;
var tp: integer;
begin
   if (rTksCount>0) and ((FFlags and sdfNoTicksLabel) = 0) then
   begin
      if (fFlags and sdfInversed) = 0
      then tp := abs(round((fMax - TksDbl[rTksCount - 1]) * fM))
      else tp := abs(round((fMin - TksDbl[rTksCount - 1]) * fM));
      if (FFlags and sdfVertical) = 0 then
         Result := FntWidth * Length(TickLabel(rTksCount - 1)) div 2 - tp
      else
         Result := FntHeight div 2 - tp;
      if Result<0 then Result := 0;
end else Result := 0;
end;
function Tsp_Scale.CalcDrawBounds(fCanvas: TCanvas): TRect;
var j, ti: integer;
begin
   with Result do with fCanvas do
   begin
      if (FFlags and sdfVertical) = 0 then
      begin
         ti := fLineAttr.Width;
         if (rTksCount>0) and ((FFlags and sdfNoTicksLabel) = 0) then
         begin
            inc(ti, TickOfs + MjTick + LblOfs + TextHeight('8'));
            Left := TksPos[0] - TextWidth(TickLabel(0)) div 2;
            Right := TksPos[rTksCount - 1] + TextWidth(TickLabel(rTksCount - 1)) div 2;
         end
         else if (FFlags and sdfNoTicks) = 0 then inc(ti, TickOfs + MjTick);
         if (FFlags and sdfLabelAtTop) = 0 then Top := O.y else Top := O.y - ti + 1;
         Bottom := Top + ti;
         if Left>O.x then Left := O.x;
         if Right<O.x + fLen then Right := O.x + fLen + 1;
         end else
         begin
            ti := fLineAttr.Width;
            if (rTksCount>0) and ((FFlags and sdfNoTicksLabel) = 0) then
            begin
               ti := TextWidth(TickLabel(0));
               for j := 1 to rTksCount - 1 do
                  if ti< TextWidth(TickLabel(j)) then
                  ti := TextWidth(TickLabel(j));
               inc(ti, TickOfs + MjTick + LblOfs);
               Top := TksPos[rTksCount - 1] - TextHeight('8')div 2;
               Bottom := TksPos[0] + TextHeight('8')div 2;
            end
            else if (FFlags and sdfNoTicks) = 0 then inc(ti, TickOfs + MjTick);
            if (FFlags and sdfLabelOnRight) = 0 then Left := O.x - ti + 1 else Left := O.x;
            Right := Left + ti;
            if Top>(O.y - fLen) then Top := O.y - fLen;
            if Bottom<O.y then Bottom := O.y + 1;
         end;
      end;
   end;
procedure Tsp_Scale.DrawLine;
var j, st, w, b, e: integer;
begin
   with fLineAttr do if Visible then with fCanvas do
   begin
      Pen.Color := fLineAttr.Color;
      Pen.Style := Style;
      Pen.Width := 1;
      Pen.Mode := pmCopy;
      if (fFlags and sdfRevertTicks) = 0 then st := 1 else st := - 1;
      if (FFlags and sdfVertical) = 0 then
      begin
         w := O.y;
         b := O.x - odec; e := O.x + fLen + 1 + einc;
         for j := 1 to Width do begin
            MoveTo(b, w);
            LineTo(e, w);
            inc(w, st);
         end;
      end
      else
      begin
         w := O.x;
         e := O.y + 1 + odec; b := O.y - fLen - einc;
         for j := 1 to Width do begin
            MoveTo(w, b);
            LineTo(w, e);
            dec(w, st);
         end;
      end;
   end;
end;
procedure Tsp_Scale.DrawTicks;
procedure DrawVert;
var j: word;
   x, l: integer;
   LS: String; LW: integer;
begin
   with fCanvas do
   begin
      if ((FFlags and sdfNoTicks) = 0) and (rTksCount>0)then
      begin
         if (FFlags and sdfLabelOnRight) = 0 then begin
            x := O.x - TickOfs - fLineAttr.Width; l := x - MjTick;
         end else begin
            x := O.x + TickOfs + fLineAttr.Width; l := x + MjTick;
         end;
         for j := 0 to rTksCount - 1 do begin
            MoveTo(x, TksPos[j]);
            LineTo(l, TksPos[j]);
         end;
      end;
      if ((FFlags and sdfNoTicksLabel) = 0) and (rTksCount>0) then
      begin
         l := TextHeight('8') div 2;
         if (FFlags and sdfLabelOnRight) = 0 then
         begin
            x := O.x - TickOfs - fLineAttr.Width - MjTick - LblOfs;
            for j := 0 to rTksCount - 1 do begin
               LS := TickLabel(j);
               LW := TextWidth(LS);
               TextOut(x - LW, TksPos[j] - l, LS);
            end;
         end
         else
         begin
            x := O.x + TickOfs + fLineAttr.Width + MjTick + LblOfs;
            for j := 0 to rTksCount - 1 do begin
               TextOut(x, TksPos[j] - l, TickLabel(j));
            end;
         end;
      end;
   end;
end;
procedure DrawHoriz;
var j: word;
   y, l: integer;
   LS: String;   LW: integer;
begin
   with fCanvas do
   begin
      if ((FFlags and sdfNoTicks) = 0) and (rTksCount>0)then
      begin
         if (FFlags and sdfLabelAtTop) = 0 then begin
            y := O.y + TickOfs + fLineAttr.Width; l := y + MjTick;
         end else begin
            y := O.y - TickOfs - fLineAttr.Width; l := y - MjTick;
         end;
         for j := 0 to rTksCount - 1 do begin
            MoveTo(TksPos[j], y);
            LineTo(TksPos[j], l);
         end;
      end;
      if ((FFlags and sdfNoTicksLabel) = 0) and (rTksCount>0)then
      begin
         if (FFlags and sdfLabelAtTop) = 0 then
            y := O.y + TickOfs + fLineAttr.Width + MjTick + LblOfs
         else y := O.y - TickOfs - fLineAttr.Width - MjTick - LblOfs - TextHeight('8');
         for j := 0 to rTksCount - 1 do begin
            LS := TickLabel(j);
            LW := TextWidth(LS);
            TextOut(TksPos[j] - LW div 2, y, LS);
         end;
      end;
   end;
end;
begin
   if (fFlags and sdfLineOnly) = sdfLineOnly then Exit;
   if (FFlags and sdfVertical) = 0 then DrawHoriz
   else DrawVert;
end;
procedure TTestScale.SetA(V: Tsp_Scale);
begin
   if V<>A then A.Assign(V);
end;
constructor TTestScale.Create(AOwner: TComponent);
begin
   Inherited Create(AOwner);
   Width := 120;
   Height := 40;
   A := Tsp_Scale.Create(0);
   A.fFlags := A.fFlags or sdfVertical;
   A.fTicksCount := 3;
   B := Tsp_Scale.Create(0);
   A.SetLine(30, 70, 60);
end;
destructor TTestScale.Destroy;
begin
   A.Free;
   B.Free;
   inherited Destroy;
end;
procedure TTestScale.Paint;
begin
   A.DrawLine(Canvas, 0, 0);
   A.DrawTicks(Canvas);
end;
END.
