unit SCKForm;

{******************************************************************************

 Project:  	TSCK 											[Shortcut Conflict Killer]
 File:			SCK.pas        						[GUI and actual algorithms]
 Created:		06/30/1997
 Latest:		03/26/1999

 Author:		Christophe Porteneuve aka TDD
 Email:			tdd@sweetness.com
 Copyright:	(c) 1999 Christophe Porteneuve / The Magix Team!
 Rights:		Freeware with sources. Commercial use must mention author's name in
						application's credits. Bryant H. McGill must register for US$10.00.

						*DISCLAIMER* This component is provided "as is." The author doesn't
						take any responsibility whatsoever for dysfunctionings resulting of
						the use of this component.

 Check out:	TSysInfo (the *real* one!), TInstallationReader, TFloatMinimizer and
						TConfigManager.

 ******************************************************************************}

interface

uses
	SysUtils, Classes, Controls, Forms, Dialogs,
	Menus, StdCtrls, ExtCtrls, ComCtrls, Buttons, SCK, ImgList;

type
	// List of detected shortcuts in a context

	TConflictViewMode = (cvDetect, cvDetectAndShow);

	// TSCRecord

	PSCRecord = ^TSCRecord;
	TSCRecord = record
		ShortCut: TShortCut;
		Component:  TComponent;
	end; // TSCRecord

	// TShortCutList

	TShortCutList = class(TList)
	private
		function 	GetItems(Index: Integer): PSCRecord;
		procedure SetItems(Index: Integer; Value: PSCRecord);
	public
		destructor Destroy; override;
		procedure ClearItem(Index: Integer);
		procedure ClearList;
		function	IndexOfShortCut(ShortCut: TShortCut): Integer;
		property Items[Index: Integer]: PSCRecord read GetItems write SetItems; default;
	end; // TShortCutList

	// List of conflicts in a context

	// TConflict

	PConflict = ^TConflict;
	TConflict = record
		ShortCut: 	TShortCut;
		Potential: Boolean;
		Components: array[0..1] of TComponent;
	end; // TConflict

	// TConflictList

	TConflictList = class(TList)
	private
		function 	GetItems(Index: Integer): PConflict;
		procedure SetItems(Index: Integer; Value: PConflict);
	public
		destructor Destroy; override;
		procedure ClearItem(Index: Integer);
		procedure ClearList;
		property Items[Index: Integer]: PConflict
			read 	GetItems
			write SetItems; default;
	end; // TConflictList

	// List of contexts

	TContextType = (ctGlobal, ctMenu);

	TContextList = class;

 	// TContext

	TContext = class
		OwnerName: string;
		ChildCtx:	 TContextList;
		ShortCuts: TShortCutList;
		Conflicts: TConflictList;
		Component: TComponent;
		constructor Create;
		destructor Destroy; override;
	end; // TContext

  // TContextList

	TContextList = class(TList)
	private
		function 	GetItems(Index: Integer): TContext;
		procedure SetItems(Index: Integer; Value: TContext);
	public
		FClear: Boolean;
		destructor Destroy; override;
		procedure ClearItem(Index: Integer);
		procedure ClearList;
		property Items[Index: Integer]: TContext
			read 	GetItems
			write SetItems; default;
	end; // TContextList

	TSelectEvent = procedure(Sender: TObject; ComponentName: string) of object;

	//*** The Form Itself ********************************************************

	TfrmConflictsViewer = class(TForm)
		pnlSelects: TPanel;
		btnSelect: TButton;
		imgState: TImageList;
		stbMain: TStatusBar;
		chkCloseOnSel: TCheckBox;
		trvConflicts: TTreeView;
		pnlDetails: TPanel;
		lsvConflicts: TListView;
		lsvShortCuts: TListView;
		splMain: TSplitter;
		imgConflicts: TImageList;
    btnAbout: TButton;
		procedure btnSelectClick(Sender: TObject);
		procedure trvConflictsChange(Sender: TObject; Node: TTreeNode);
		procedure FormCreate(Sender: TObject);
		procedure chkCloseOnSelClick(Sender: TObject);
		procedure FormDestroy(Sender: TObject);
    procedure btnAboutClick(Sender: TObject);
	private
		FCmpt: TSCK;
		FOldOnHint: TNotifyEvent;
		FOnSelect: TSelectEvent;
		FSorted: Boolean;
		procedure DisplayCTL(OwnerNode: TTreeNode; CTL: TContextList);
		procedure HandleHint(Sender: TObject);
	public
		Contexts:	TContextList;
		property OnSelect: TSelectEvent read FOnSelect write FOnSelect;
		class function Execute(Sender: TSCK; AMode: TConflictViewMode;
			ASelProc: TSelectEvent): Boolean;
	end;

var
	frmConflictsViewer: TfrmConflictsViewer;
	GCfInGC:	Boolean;

implementation

{$R *.DFM}

uses
	TypInfo;

//******************************************************************************
// TShortCutList
//******************************************************************************

destructor TShortCutList.Destroy;
begin
	ClearList;
	inherited;
end; // Destroy

function TShortCutList.GetItems(Index: Integer): PSCRecord;
begin
	Result := PSCRecord(inherited Items[Index]);
end; // GetItems

procedure TShortCutList.SetItems(Index: Integer; Value: PSCRecord);
begin
	inherited Items[Index] := Value;
end; // SetItems

procedure TShortCutList.ClearItem(Index: Integer);
var
	P: PSCRecord;
begin
	P := PSCRecord(inherited Items[Index]);
	remove(P);
	dispose(P);
end; // ClearItem

procedure TShortCutList.ClearList;
begin
	while (Count > 0) do
		ClearItem(0);
end; // ClearList

function TShortCutList.IndexOfShortCut(ShortCut: TShortCut): Integer;
var
	i: Integer;
begin
	Result := -1;
	for i := 0 to Count-1 do
		if (Items[i]^.ShortCut = ShortCut) then
		begin
			Result := i;
			exit;
		end;
end; // IndexOfShortCut

//******************************************************************************
// TConflictList
//******************************************************************************

destructor TConflictList.Destroy;
begin
	ClearList;
	inherited;
end; // Destroy

function TConflictList.GetItems(Index: Integer): PConflict;
begin
	Result := PConflict(inherited Items[Index]);
end; // GetItems

procedure TConflictList.SetItems(Index: Integer; Value: PConflict);
begin
	inherited Items[Index] := Value;
end; // SetItems

procedure TConflictList.ClearItem(Index: Integer);
var
	P: PConflict;
begin
	P := PConflict(inherited Items[Index]);
	remove(P);
	dispose(P);
end; // ClearItem

procedure TConflictList.ClearList;
begin
	while (Count > 0) do
		ClearItem(0);
end; // ClearList

//******************************************************************************
// TContext
//******************************************************************************

constructor TContext.Create;
begin
	inherited;
	ShortCuts := TShortCutList.Create;
	Conflicts := TConflictList.Create;
	ChildCtx 	:= nil;
end; // Create

destructor TContext.Destroy;
begin
	ChildCtx.Free;
	ShortCuts.Free;
	Conflicts.Free;
	inherited;
end; // Destroy

//******************************************************************************
// TContextList
//******************************************************************************

destructor TContextList.Destroy;
begin
	if FClear then
		ClearList;
	inherited;
end; // Destroy

function TContextList.GetItems(Index: Integer): TContext;
begin
	Result := TContext(inherited Items[Index]);
end; // GetItems

procedure TContextList.SetItems(Index: Integer; Value: TContext);
begin
	inherited Items[Index] := Value;
end; // SetItems

procedure TContextList.ClearItem(Index: Integer);
var
	Ctx: TContext;
begin
	Ctx := TContext(inherited Items[Index]);
	Remove(Ctx);
	if FClear then
		Ctx.Free;
end; // ClearItem

procedure TContextList.ClearList;
begin
	while Count > 0 do
		ClearItem(0);
end; // ClearList

//******************************************************************************
// Global context stuff and tool routines
//******************************************************************************

// IsValidComponent -- Returns True if AComp publishes a Caption property

function IsValidComponent(AComp: TComponent): Boolean;
begin
	Result := False;
	if AComp = nil then Exit;
	if GetPropInfo(PTypeInfo(AComp.ClassInfo), 'Caption') <> nil then
		Result := True;
end; // IsValidComponent

// GetCaptionShortCut -- Returns the actual shortcut in the Caption prop. of
// the components passed as AComp

function GetCaptionShortCut(AComp: TComponent): TShortCut;
var
	PTI: 	PTypeInfo;
	CompCaption: string;
	CapLen, AmpPos:	Integer;
	AmpIndex:	Integer;
begin
	Result := 0;
	if (AComp = nil) then
		Exit;
	PTI := PTypeInfo(AComp.ClassInfo);
	CompCaption := GetStrProp(AComp, GetPropInfo(PTI, 'Caption'));
	CapLen := Length(CompCaption);
	// Purging double-&
	AmpPos := Pos('&&', CompCaption);
	while AmpPos <> 0 do
	begin
		Delete(CompCaption, AmpPos, 2);
		Dec(CapLen, 2);
		AmpPos := Pos('&&', CompCaption);
	end;
	// Searching '&'-made shortcuts, from end of the string (last has priority)
	for AmpIndex := CapLen-1 downto 1 do
		if (CompCaption[AmpIndex] = '&') then
		begin
			Result := ShortCut(Word(UpCase(CompCaption[AmpIndex + 1])), [ssAlt]);
			Break;
		end;
end; // GetCaptionShortCut

// CreateComponentsList -- Creates list of valid components for global context

function CreateComponentsList(AForm: TForm): TList;
var
	CompIndex: Integer;
	Cmpnt: TComponent;
begin
	Result := TList.Create;
	try
		with AForm do
			for CompIndex := 0 to ComponentCount-1 do
			begin
				Cmpnt := Components[CompIndex];
				if IsValidComponent(Cmpnt) then
				begin
					// The following components are ignored, because handled by further
					// routines: TMenuItem (if not main option), TTabSheet, and all those,
					// parent of which is a TTabSheet or a TPage.
					if ((Menu <> nil) and (Cmpnt is TMenuItem) and
							(TMenuItem(Cmpnt).Parent <> Menu.Items)) or
						 (Cmpnt is TTabSheet) or
						 ((Cmpnt is TControl) and
							((TControl(Cmpnt).Parent is TTabSheet) or
							 (TControl(Cmpnt).Parent is TPage))) then
						Continue;
					Result.Add(Cmpnt);
				end;
			end;
	except
		Result.Free;
		Result := nil;
	end;
end; // CreateComponentsList

//******************************************************************************
// Parsing the global context
//******************************************************************************

// ParseGlobalCaptionContext -- Determines a list of Global Context's shortcuts,
//															and conflicts by the same time (Captions only)

procedure ParseGlobalCaptionContext(CmpList: TList;
	AGlobalContext: TContext);
var
	CompIndex, Idx: Integer;
	PSCut: PSCRecord;
	PCnfl: PConflict;
	SCut: TShortCut;
begin
	with CmpList, AGlobalContext do
		for CompIndex := 0 to Count-1 do
		begin
			SCut := GetCaptionShortCut(Items[CompIndex]);
			if SCut <> 0 then
			begin
				// Add a shortcut entry
				New(PSCut);
				PSCut^.ShortCut := SCut;
				PSCut^.Component	:= Items[CompIndex];
				// Conflict test
				Idx := ShortCuts.IndexOfShortCut(PSCut^.ShortCut);
				if Idx <> -1 then
				begin
					// Add a conflict entry
					New(PCnfl);
					PCnfl^.ShortCut 			:= PSCut^.ShortCut;
					PCnfl^.Components[0]	:= PSCut^.Component;
					PCnfl^.Components[1]	:= ShortCuts[Idx]^.Component;
					PCnfl^.Potential			:= False;
					Conflicts.Add(PCnfl);
				end;
				ShortCuts.Add(PSCut);
			end;
		end;
end; // ParseGlobalCaptionContext

//******************************************************************************
// Menus & menu items handling
//******************************************************************************

// ParseMenuContext -- Parses a menu. Recurses for submenus

procedure ParseMenuContext(AMenu: TMenuItem;
	OwnerCtx: TContext; GlobalContext: TContext);
var
	MnuIndex, Idx: Integer;
	SCut:		TShortCut;
	Ctx:		TContext;
	PSC:		PSCRecord;
	PCnfl: 	PConflict;
begin
	// If necessary, initialize child-contexts list of the father menu
	if AMenu.Count > 0 then
	begin
		Ctx := TContext.Create;
		if AMenu.Parent <> nil then
			Ctx.OwnerName := AMenu.Name // MainMenu's main option
		else
			Ctx.OwnerName := AMenu.Owner.Name; // PopupMenu option
	end else
		Exit;
	if OwnerCtx.ChildCtx = nil then
		OwnerCtx.ChildCtx := TContextList.Create;
	OwnerCtx.ChildCtx.Add(Ctx);
	with AMenu do
		for MnuIndex := 0 to Count - 1 do
		begin
			// Caption handling
			SCut := GetCaptionShortCut(Items[MnuIndex]);
			if SCut <> 0 then
			begin
				// Add shortcut entry
				New(PSC);
				PSC^.ShortCut := SCut;
				PSC^.Component	:= Items[MnuIndex];
				// Conflict test
				Idx := Ctx.ShortCuts.IndexOfShortCut(PSC^.ShortCut);
				if Idx <> -1 then
				begin
					New(PCnfl);
					PCnfl^.ShortCut 			:= PSC^.ShortCut;
					PCnfl^.Components[0]	:= PSC^.Component;
					PCnfl^.Components[1]	:= Ctx.ShortCuts[Idx]^.Component;
					PCnfl^.Potential			:= False;
					Ctx.Conflicts.Add(PCnfl);
				end;
				Ctx.ShortCuts.Add(PSC);
			end; // if SCut <> 0

			// Handling the ShortCut property of the TMenuItem.
			// Identical handling, but detected and stored in the Global Context,
			// instead of the current one.
			SCut := Items[MnuIndex].ShortCut;
			if SCut <> 0 then
			begin
				// Recording shortcut
				New(PSC);
				PSC^.ShortCut	:= SCut;
				PSC^.Component  := Items[MnuIndex];
				// Conflict test
				Idx := GlobalContext.ShortCuts.IndexOfShortCut(PSC^.ShortCut);
				if Idx <> -1 then
				begin
					New(PCnfl);
					PCnfl^.ShortCut 			:= PSC^.ShortCut;
					PCnfl^.Components[0]	:= PSC^.Component;
					PCnfl^.Components[1]	:= GlobalContext.ShortCuts[Idx]^.Component;
					PCnfl^.Potential			:= False;
					if GCFInGC then
						GlobalContext.Conflicts.Add(PCnfl)
					else
						Ctx.Conflicts.Add(PCnfl);
				end;
				GlobalContext.ShortCuts.Add(PSC);
			end; // if SCut <> 0

			// Submenus handling
			if Items[MnuIndex].Count <> 0 then
				ParseMenuContext(Items[MnuIndex], Ctx, GlobalContext);
		end; // for MnuIndex
end; // ParseMenuContext

// ParseMainMenuContext -- Complete menu-bar handling

procedure ParseMainMenuContext(AMenu: TMainMenu;
	GlobalContext: TContext);
var
	MenuIndex: Integer;
begin
	with AMenu do
		for MenuIndex := 0 to Items.Count-1 do
			ParseMenuContext(Items[MenuIndex], GlobalContext, GlobalContext);
end; // ParseMainMenuContext

// ParsePopupMenusContext -- Popup menus handling

procedure ParsePopupMenusContext(AForm: TForm; GlobalContext: TContext);
var
	CompIndex: Integer;
begin
	with AForm do
		for CompIndex := 0 to ComponentCount - 1 do
			if Components[CompIndex] is TPopupMenu then
				with Components[CompIndex] as TPopupMenu do
					ParseMenuContext(Items, GlobalContext, GlobalContext);
end; // ParseMainMenuContext

//******************************************************************************
// Multiple page controls handling
//******************************************************************************

// ParseContainerControl -- Parses a TTabSheet/TPage. Ctx = current context

procedure ParseContainerControl(AContainer: TWinControl;
	GlobalContext, Ctx: TContext);
var
	ControlIndex, Idx: Integer;
	SCut:	TShortCut;
	PSC: PSCRecord;
	PCnfl: PConflict;
begin
	with AContainer do
		for ControlIndex := 0 to ControlCount - 1 do
		begin
			if not IsValidComponent(Controls[ControlIndex]) then
				Continue;
			// Caption handling
			SCut := GetCaptionShortCut(Controls[ControlIndex]);
			if SCut <> 0 then
			begin
				// Add shortcut entry
				New(PSC);
				PSC^.ShortCut := SCut;
				PSC^.Component	:= Controls[ControlIndex];
				// Conflict test (local)
				Idx := Ctx.ShortCuts.IndexOfShortCut(PSC^.ShortCut);
				if Idx <> -1 then
				begin
					New(PCnfl);
					PCnfl^.ShortCut 			:= PSC^.ShortCut;
					PCnfl^.Components[0]	:= PSC^.Component;
					PCnfl^.Components[1]	:= Ctx.ShortCuts[Idx]^.Component;
					PCnfl^.Potential			:= False;
					Ctx.Conflicts.Add(PCnfl);
				end;
				// Conflicts test (Global Context)
				Idx := GlobalContext.ShortCuts.IndexOfShortCut(PSC^.ShortCut);
				if Idx <> -1 then
				begin
					New(PCnfl);
					PCnfl^.ShortCut 			:= PSC^.ShortCut;
					PCnfl^.Components[0]	:= PSC^.Component;
					PCnfl^.Components[1]	:= GlobalContext.ShortCuts[Idx]^.Component;
					PCnfl^.Potential			:= True;
					if GCFInGC then
						GlobalContext.Conflicts.Add(PCnfl)
					else
						Ctx.Conflicts.Add(PCnfl);
				end;
				Ctx.ShortCuts.Add(PSC);
			end; // if SCut <> 0
		end; // for ControlIndex
end; // ParseContainerControl

// ParseTabSheet -- Parses components in a TTabSheet

procedure ParseTabSheet(ATabSheet: TTabSheet;
	GlobalContext, OwnerCtx: TContext);
var
	Ctx: TContext;
begin
	Ctx := TContext.Create;
	Ctx.OwnerName := ATabSheet.Name;
	if OwnerCtx.ChildCtx = nil then
		OwnerCtx.ChildCtx := TContextList.Create;
	OwnerCtx.ChildCtx.Add(Ctx);
	ParseContainerControl(ATabSheet, GlobalContext, Ctx);
end; // ParseTabSheet

// ParsePageControl -- Parses a TPageControl

function ParsePageControl(APC: TPageControl; GlobalContext: TContext): TContext;
var
	PageIndex: Integer;
	Ctx: TContext;
begin
	Result := nil;
	// Void PageControl implies void context...
	if APC.PageCount = 0 then
		Exit;
	// Otherwise, a context is created, and added to the GlobalContext's
	// subcontexts list
	Ctx := TContext.Create;
	Ctx.OwnerName := APC.Name;
	Ctx.Component	:= APC;
	if GlobalContext.ChildCtx = nil then
		GlobalContext.ChildCtx := TContextList.Create;
	GlobalContext.ChildCtx.Add(Ctx);

	// Parse pages
	with APC do
		for PageIndex := 0 to PageCount - 1 do
			ParseTabSheet(Pages[PageIndex], GlobalContext, Ctx);

	Result := Ctx;
end; // ParsePageControl

// ParsePage -- Parses components in a TPage

procedure ParsePage(APage: TPage; GlobalContext, OwnerCtx: TContext);
var
	Ctx: TContext;
begin
	Ctx := TContext.Create;
	Ctx.OwnerName := APage.Caption;
	if OwnerCtx.ChildCtx = nil then
		OwnerCtx.ChildCtx := TContextList.Create;
	OwnerCtx.ChildCtx.Add(Ctx);
	ParseContainerControl(APage, GlobalContext, Ctx);
end; // ParsePage

// ParseNoteBook -- Parses a TNoteBook

function ParseNoteBook(ANB: TNoteBook; GlobalContext: TContext): TContext;
var
	ControlIndex: Integer;
	Ctx: TContext;
begin
	Result := nil;
	// Void NoteBook implies void context...
	if ANB.Pages.Count = 0 then
		Exit;
	// Otherwise, a context is created, and Added to the GlobalContext's
	// subcontexts list
	Ctx := TContext.Create;
	Ctx.OwnerName := ANB.Name;
	Ctx.Component := ANB;
	if GlobalContext.ChildCtx = nil then
		GlobalContext.ChildCtx := TContextList.Create;
	GlobalContext.ChildCtx.Add(Ctx);

	// Parse pages
	with ANB do
		for ControlIndex := 0 to ControlCount - 1 do
			if Controls[ControlIndex] is TPage then
				ParsePage(TPage(Controls[ControlIndex]), GlobalContext, Ctx);

	Result := Ctx;
end; // ParseNoteBook

// DetectTSPToTSPConflicts -- Detects conflicts between two TabSheets/Pages

procedure DetectTSPToTSPConflicts(Ctx1, Ctx2: TContext);
var
	SCutIndex, Idx:	Integer;
	PCnfl: PConflict;
begin
	for SCutIndex := 0 to Ctx1.ShortCuts.Count - 1 do
	begin
		Idx := Ctx2.ShortCuts.IndexOfShortCut(Ctx1.ShortCuts[SCutIndex].ShortCut);
		if Idx <> -1 then
		begin
			New(PCnfl);
			PCnfl^.ShortCut 			:= Ctx1.ShortCuts[SCutIndex].ShortCut;
			PCnfl^.Components[0]	:= Ctx1.ShortCuts[SCutIndex].Component;
			PCnfl^.Components[1]	:= Ctx2.ShortCuts[Idx]^.Component;
			PCnfl^.Potential			:= True;
			Ctx2.Conflicts.Add(PCnfl);
		end;
	end;
end; // DetectTSPToTSPConflicts

// GetInterPagesConflicts -- Detects the conflicts between 2 multipage controls

procedure GetInterPagesConflicts(Ctx1, Ctx2: TContext);
var
	Ctx1ChildIndex, Ctx2ChildIndex: Integer;
begin
	for Ctx1ChildIndex := 0 to Ctx1.ChildCtx.Count - 1 do
		for Ctx2ChildIndex := 0 to Ctx2.ChildCtx.Count - 1 do
			DetectTSPToTSPConflicts(
				Ctx1.ChildCtx[Ctx1ChildIndex],
				Ctx2.ChildCtx[Ctx2ChildIndex]);
end; // GetInterPagesConflicts

// ParseMultiPages -- Parses all the multipage controls on a form

procedure ParseMultiPages(AForm: TForm; GlobalContext: TContext);
var
	PCCtxList: TContextList;
	NBCtxList: TContextList;
	Index: Integer;
	PgCtrl: TPageControl;
	NBk: TNoteBook;
	Parents: TList;

	procedure BuildParentList(ACtrl: TWinControl; ParentList: TList);
	var
		WCtrl: TWinControl;
	begin
		ParentList.Clear;
		WCtrl := ACtrl;
		while True do
		begin
			if (WCtrl is TTabSheet) or (WCtrl is TPageControl) or (WCtrl is TPage) or
				 (WCtrl is TNoteBook) or (WCtrl is TForm) then
				ParentList.Add(WCtrl);
			if WCtrl is TForm then
				Break;
			WCtrl := WCtrl.Parent;
		end;
	end; // BuildParentList

	// Sustained thinking (as said Voltaire... :-)) made me come up with the
	// following general condition: two multiple-page controls can "see
	// each other", i.e. conclict, ONLY if their closest common parent is NOT
	// a multipage control, OR that multipage control would be one of
	// them (i.e., here: Ref or Ctrl)

	function UnExclusive(Ref, Ctrl: TWinControl; ParentList: TList): Boolean;
	var
		WCtrl: TWinControl;
	begin
		Result := True;
		WCtrl := Ctrl.Parent;
		while not (WCtrl is TForm) do
		begin
			if (WCtrl is TPageControl) or (WCtrl is TNoteBook) then
				if (ParentList.IndexOf(WCtrl) <> -1) and (WCtrl <> Ref) then
				begin
					Result := False;
					Break;
				end;
			WCtrl := WCtrl.Parent;
		end;
	end; // UnExclusive

begin // ParseMultiPage
	Parents	:= nil;
	PCCtxList := nil;
	NBCtxList := nil;
	try
		PCCtxList := TContextList.Create;
		NBCtxList := TContextList.Create;
		PCCtxList.FClear := False;
		NBCtxList.FClear := False;
		// Obtain contexts for all multiple page components (PageControls and
		// Notebooks)
		with AForm do
			for Index := 0 to ComponentCount - 1 do
				if Components[Index] is TPageControl then
					PCCtxList.Add(
						ParsePageControl(TPageControl(Components[Index]), GlobalContext))
				else
					if Components[Index] is TNoteBook then
						NBCtxList.Add(
							ParseNoteBook(TNoteBook(Components[Index]), GlobalContext));

		Parents := TList.Create;
		while PCCtxList.Count > 0 do
		begin
			PgCtrl := PCCtxList[0].Component as TPageControl;
			BuildParentList(PgCtrl, Parents);
			for Index := 1 to PCCtxList.Count - 1 do
				if UnExclusive(
						PCCtxList[Index].Component as TWinControl, PgCtrl, Parents) then
					GetInterPagesConflicts(PCCtxList[0], PCCtxList[Index]);
			for Index := 0 to NBCtxList.Count - 1 do
				if UnExclusive(
						NBCtxList[Index].Component as TWinControl, PgCtrl, Parents) then
					GetInterPagesConflicts(PCCtxList[0], NBCtxList[Index]);
			PCCtxList.ClearItem(0);
		end;
		while NBCtxList.Count > 0 do
		begin
			NBk := NBCtxList[0].Component as TNoteBook;
			BuildParentList(NBk, Parents);
			for Index := 1 to NBCtxList.Count-1 do
				if UnExclusive(
						NBCtxList[Index].Component as TWinControl, NBk, Parents) then
					GetInterPagesConflicts(NBCtxList[0], NBCtxList[Index]);
			NBCtxList.ClearItem(0);
		end;
	finally
		PCCtxList.Free;
		NBCtxList.Free;
		Parents.Free;
	end;
end; // ParseMultiPages

//******************************************************************************
// Main parsing routine
//******************************************************************************

// ParseAllContexts -- Detects all conflicts

function ParseAllContexts(AForm: TForm; ACompList: TList): TContextList;
var
	Ctx: TContext;
begin
	Result := TContextList.Create;
	Ctx := TContext.Create;
	Ctx.OwnerName := AForm.Name;
	ParseGlobalCaptionContext(ACompList, Ctx);
	Result.Add(Ctx);
	if AForm.Menu <> nil then
		ParseMainMenuContext(AForm.Menu, Ctx);
	ParsePopupMenusContext(AForm, Ctx);
	ParseMultiPages(AForm, Ctx);
end; // ParseAllContexts

//******************************************************************************
// TfrmConflictsViewer
//******************************************************************************

// TfrmConflictsViewer - "Execute" Class function (entry point to the form)

class function TfrmConflictsViewer.Execute(Sender: TSCK;
	AMode: TConflictViewMode; ASelProc: TSelectEvent): Boolean;
var
	CmpList: TList;
begin
	Result 	:= False;
	GCfInGC := Sender.GlobalsAtRoot;
	frmConflictsViewer := TfrmConflictsViewer.Create(Application);
	try
		with frmConflictsViewer do
		begin
			FCmpt := Sender;
			FSorted := FCmpt.Sorted;
			chkCloseOnSel.Checked := FCmpt.FCloseOnSel;
			trvConflicts.Items.Clear;
			// Obtain list of components
			CmpList := CreateComponentsList(FCmpt.Owner as TForm);
			try
				// Dismiss potential former contexts list
				Contexts.Free;
				// Obtain list of all contexts with their conflicts
				Contexts 	:= ParseAllContexts(FCmpt.Owner as TForm, CmpList);
				// Display contexts and the current one's conflicts
				with trvConflicts do
				begin
					DisplayCTL(nil, Contexts);
					if Items.Count > 0 then
						Selected := Items[0];
					FullExpand;
					if FSorted then
						AlphaSort;
					// Set routine to be called when the "Select" button is pressed
					OnSelect := ASelProc;
				end; // with
			finally
				CmpList.Free;
			end;
			if trvConflicts.Items.Count > 0 then
			begin
				if AMode = cvDetectAndShow then
					ShowModal;
				Result := True;
			end;
		end; // with
	finally
		frmConflictsViewer.Free;
		frmConflictsViewer := nil;
	end;
end; // Execute

// TfrmConflictsViewer - DisplayCTL

procedure TfrmConflictsViewer.DisplayCTL(OwnerNode: TTreeNode;
	CTL: TContextList);
var
	TN: TTreeNode;
	CtxIndex:	Integer;

	function GetCtxStr(ACtx: TContext): string;
	var
		CflIndex, NbPot: Integer;
	begin
		with ACtx do
			if (Conflicts.Count = 0) then
				Result := OwnerName
			else
			begin
				NbPot := 0;
				for CflIndex := 0 to Conflicts.Count-1 do
					if Conflicts[CflIndex].Potential then
						Inc(NbPot);
				// Contexts with potential conflicts get noted as "(total-potentials)"
				if NbPot > 0 then
					Result := Format('%s (%d-%d)', [OwnerName, Conflicts.Count, NbPot])
				else
					Result := Format('%s (%d)', [OwnerName, Conflicts.Count]);
			end;
	end; // GetCtxStr

begin
	// No context list? Go away!
	if CTL = nil then
		Exit;
	with trvConflicts do
		for CtxIndex := 0 to CTL.Count-1 do
		begin
			// No owner node? Add at the root level. Otherwise, add as child of node
			if OwnerNode = nil then
				TN := Items.AddObject(OwnerNode,
					GetCtxStr(CTL[CtxIndex]), CTL[CtxIndex])
			else
				TN := Items.AddChildObject(OwnerNode,
					GetCtxStr(CTL[CtxIndex]), CTL[CtxIndex]);
			// Set StateIndex (1 = No conflict, 2 = Conflict) for node
			TN.StateIndex := Ord(CTL[CtxIndex].Conflicts.Count > 0) + 1;
			// Basic recursive behavior for subcontexts
			if CTL[CtxIndex].ChildCtx <> nil then
				DisplayCTL(TN, CTL[CtxIndex].ChildCtx);
		end;
end; // DisplayCTL

procedure TfrmConflictsViewer.HandleHint(Sender: TObject);
begin
	stbMain.SimpleText := Application.Hint;
end; // HandleHint

// TfrmConflictsViewer - Event handlers

procedure TfrmConflictsViewer.FormCreate(Sender: TObject);
begin
	FOldOnHint := Application.OnHint;
	Application.OnHint := HandleHint;
end; // FormCreate

procedure TfrmConflictsViewer.FormDestroy(Sender: TObject);
begin
	Application.OnHint := FOldOnHint;
	Contexts.Free;
end; // FormDestroy

procedure TfrmConflictsViewer.trvConflictsChange(Sender: TObject;
	Node: TTreeNode);
var
	Index: Integer;
begin
	// Refresh the conflicts list
	with lsvConflicts, TContext(Node.Data) do
	begin
		Items.Clear;
		for Index := 0 to Conflicts.Count-1 do
			with Items.Add, Conflicts[Index]^ do
			begin
				if Potential then
					StateIndex := 1;
				Caption := ShortCutToText(ShortCut);
				SubItems.Add(Components[0].Name);
				SubItems.Add(Components[1].Name);
			end;
	end;
	// Refresh the shortcuts list (all shortcuts in the context for the node)
	with lsvShortcuts, TContext(Node.Data) do
	begin
		Items.Clear;
		for Index := 0 to ShortCuts.Count-1 do
			with Items.Add, ShortCuts[Index]^ do
			begin
				Caption := ShortCutToText(ShortCut);
				SubItems.Add(Component.Name);
			end;
	end;
end; // trvConflictsChange

procedure TfrmConflictsViewer.btnSelectClick(Sender: TObject);
begin
	if lsvConflicts.Selected = nil then
	begin
		Application.MessageBox('Select a conflict', 'Missing selection', 0);
		Exit;
	end;
	if Assigned(FOnSelect) then
		FOnSelect(Self, lsvConflicts.Selected.SubItems[0])
	else
		ShowMessage('No Defined Select Routine');
	if chkCloseOnSel.Checked then
		Close;
end; // btnSelectClick

procedure TfrmConflictsViewer.chkCloseOnSelClick(Sender: TObject);
begin
	FCmpt.FCloseOnSel := chkCloseOnSel.Checked;
end; // chkCloseOnSelClick

procedure TfrmConflictsViewer.btnAboutClick(Sender: TObject);
begin
	MessageDlg(
		'ShortCut Conflict Killer v.' + VERSION_ID + #13#10 +
		'(c) 1999 by Christophe Porteneuve / The Magix Team!'#13#10#13#10 +
		'Freeware (Commercial use must mention author''s name in'#13#10 +
		'application credits) but for Bryant H. McGill: US$10.00 Fee.',
		mtInformation, [mbOk], 0);
end; // btnAboutClick

end.
