unit SCK;

{******************************************************************************

 Project:  	TSCK 											[Shortcut Conflict Killer]
 File:			SCK.pas        						[Main file]
 Created:		07/10/1997
 Latest:		03/26/1999

 Author:		Christophe Porteneuve aka TDD
 Email:			tdd@sweetness.com
 Copyright:	(c) 1999 Christophe Porteneuve / The Magix Team!
 Rights:		Freeware with sources. Commercial use must mention author's name in
						application's credits. Bryant H. McGill must register for US$10.00.

						*DISCLAIMER* This component is provided "as is." The author doesn't
						take any responsibility whatsoever for dysfunctionings resulting of
						the use of this component.

 Check out:	TSysInfo (the *real* one!), TInstallationReader, TFloatMinimizer and
            TConfigManager.

 ******************************************************************************}

interface

uses
	Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
	DsgnIntf;

const
	VERSION_ID = '1.0';

type
	// TSCK

	TSCK = class(TComponent)
	private
		FConflicts:	string;
		FCRTL:			Boolean;
		FGCfInGC: 	Boolean;
		FSorted:		Boolean;
	protected
		// Component Editor Setup
		procedure ReadCOS(Reader: TReader);
		procedure WriteCOS(Writer: TWriter);
		procedure DefineProperties(Filer: TFiler); override;
		procedure ValidateContainer(AComponent: TComponent); override;
	public
		FCloseOnSel: Boolean;
		constructor Create(AOwner: TComponent); override;
		destructor Destroy; override;
		procedure	Loaded; override;
		function AreThereConflicts: Boolean;
	published
		property CheckRunTimeLoad: boolean read FCRTL write FCRTL default True;
		property Conflicts: string read FConflicts write FConflicts;
		property GlobalsAtRoot: boolean read FGCfInGC write FGCfInGC default False;
		property Sorted: boolean read FSorted write FSorted default True;
	end; // TSCK

	// TConflictsProperty

	TConflictsProperty = class(TStringProperty)
	public
		procedure DoSelect(Sender: TObject; ComponentName: string);
		function GetAttributes: TPropertyAttributes; override;
		function GetValue: string; override;
		procedure Edit; override;
	end; // TConflictsProperty

	// TSCKEditor

	TSCKEditor = class(TComponentEditor)
	public
		procedure DoSelect(Sender: TObject; ComponentName: string);
		function GetVerbCount: Integer; override;
		function GetVerb(Index: Integer): string; override;
		procedure ExecuteVerb(Index: Integer); override;
		procedure Edit; override;
	end; // TSCKEditor

procedure Register;

implementation

uses SCKForm;

var
	SafeDesigner: IFormDesigner = nil;

//******************************************************************************
// TSCK
//******************************************************************************

constructor TSCK.Create(AOwner: TComponent);
begin
	inherited;
	FCRTL 			:= True;
	FGCfInGC		:= False;
	FSorted			:= True;
	FCloseOnSel	:= False; // Component Editor setup property
end; // Create

destructor TSCK.Destroy;
begin
	inherited;
end; // Destroy

procedure	TSCK.Loaded;
begin
	inherited;
	if FCRTL and not (csDesigning in ComponentState) then
		if AreThereConflicts then
			Application.MessageBox(
				PChar('There are conflicts in '+(Owner as TForm).Name),
				'SCK Run-Time Check Warning', MB_ICONEXCLAMATION + MB_OK);
end; // Loaded

procedure TSCK.ReadCOS(Reader: TReader);
begin
	FCloseOnSel := Reader.ReadBoolean;
end; // ReadCOS

procedure TSCK.WriteCOS(Writer: TWriter);
begin
	Writer.WriteBoolean(FCloseOnSel);
end; // WriteCOS

procedure TSCK.DefineProperties(Filer: TFiler);
begin
	inherited;
	Filer.DefineProperty('CloseOnSel', ReadCOS, WriteCOS, True);
end; // DefineProperties

procedure TSCK.ValidateContainer(AComponent: TComponent);
var
	CompIndex: Integer;
begin
	inherited;
	if AComponent is TForm then
		with AComponent as TForm do
			for CompIndex := 0 to ComponentCount - 1 do
				if (Components[CompIndex] is Self.ClassType) then
					raise Exception.Create(
						'There can be only one ' + Self.ClassName + ' by form.'#13#10+
						'(' + Components[CompIndex].Name + ' is already present)');
end; // ValidateContainer

function TSCK.AreThereConflicts: Boolean;
begin
	Result := TfrmConflictsViewer.Execute(Self, cvDetect, nil);
end; // AreThereConflicts

//******************************************************************************
// TConflictsProperty
//******************************************************************************

procedure TConflictsProperty.DoSelect(Sender: TObject; ComponentName: string);
begin
	// Selecting a component through the Designer makes us "lose it:" the
	// Designer property does not contain the right pointer anymore. That is
	// the reason why we save it on the first use, and then use it later.
	if SafeDesigner = nil then
		SafeDesigner := Designer;
	with SafeDesigner do
		SelectComponent(GetComponent(ComponentName));
end; // DoSelect

function TConflictsProperty.GetAttributes: TPropertyAttributes;
begin
	Result := inherited GetAttributes + [paReadOnly, paDialog];
end; // GetAttributes

function TConflictsProperty.GetValue: string;
begin
	Result := '(Double-click here)';
end; // GetValue

procedure TConflictsProperty.Edit;
begin
	if not TfrmConflictsViewer.Execute(GetComponent(0) as TSCK, cvDetectAndShow,
		DoSelect)
	then
		MessageDlg('No conflict in the form.', mtInformation, [mbOk], 0);
end; // Edit

//******************************************************************************
// TSCKEditor
//******************************************************************************

procedure TSCKEditor.DoSelect(Sender: TObject; ComponentName: string);
begin
	// Selecting a component through the Designer makes us "lose it:" the
	// Designer property does not contain the right pointer anymore. That is
	// the reason why we save it on the first use, and then use it later.
	if SafeDesigner = nil then
		SafeDesigner := Designer;
	with SafeDesigner do
		SelectComponent(GetComponent(ComponentName));
end; // DoSelect

function TSCKEditor.GetVerbCount: Integer;
begin
	Result := 4;
end; // GetVerbCount

function TSCKEditor.GetVerb(Index: Integer): string;
begin
	case Index of
		0: Result := 'Shorcut Conflict Killer v. ' + VERSION_ID;
		1: Result := '(c) 1999 by Christophe Porteneuve';
		2: Result := '-';
		3: Result := 'Detect conflicts...';
	end;
end; // GetVerb

procedure TSCKEditor.ExecuteVerb(Index: Integer);
begin
	if Index = 3 then
		Edit;
end; // ExecuteVerb

procedure TSCKEditor.Edit;
begin
	if not TfrmConflictsViewer.Execute(Component as TSCK, cvDetectAndShow,
		DoSelect)
	then
		MessageDlg('No conflict in the form.', mtInformation, [mbOk], 0);
end; // Edit

procedure Register;
begin
	RegisterComponents('TDDians', [TSCK]);
	RegisterPropertyEditor(TypeInfo(string), TSCK, 'Conflicts', TConflictsProperty);
	RegisterComponentEditor(TSCK, TSCKEditor);
end; // Register

end.
