unit MGraph;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ScGraph, ExtCtrls, StdCtrls, Printers;

type
  TForm1 = class(TForm)
    Panel1: TPanel;
    ScGraph1: TScGraph;
    ScGraph2: TScGraph;
    Button1: TButton;
    Button2: TButton;
    Button3: TButton;
    Button4: TButton;
    procedure FormResize(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure Button3Click(Sender: TObject);
    procedure Button1Click(Sender: TObject);
    procedure Button2Click(Sender: TObject);
    procedure Button4Click(Sender: TObject);
    procedure PrintGraph(LeftMM, TopMM, WidthMM, HeightMM: longint);
    procedure ScaleGraph(f: single);
  private
    { Private declarations }
  public
    { Public declarations }
  end;

var
  Form1: TForm1;

implementation

{$R *.DFM}

procedure TForm1.FormCreate(Sender: TObject);
var i: integer;
    x, d, f: double;
begin
  // derive form size from screen
  Width:= Screen.Width * 3 div 5;
  Height:= Screen.Height * 3 div 4;
  // create some nice simulated data
  ScGraph1.SetSeries( 2, true, true, clRed, 'measured data');
  ScGraph2.SetSeries( 1, true, true, clRed, 'Residuals');
  for i := 1 to 50 do
  begin
    x:= random(290) + 1;
    f:= 70 * (1 - exp(-0.2 * x )) * exp(-0.01 * x); // get a value
    d:= f + (20 * random - 10);                     // get some deviation
    ScGraph1.AddPoint( 2, x, d, -1, 5 * random + d * 0.1 );
    ScGraph2.AddPoint( 1, x, d - f, -1, -1 );
  end;
  ScGraph1.SetSeries( 1, true, true, clBlue, 'model fit');
  for i := 1 to 290 do
  begin
    d:= 70 * (1 - exp(-0.2 * i )) * exp(-0.01 * i);
    ScGraph1.AddPoint( 1, i, d, -1, -1 );
  end;

  // set points, lines and error bars
  ScGraph1.SetSeriesLine( 1, true, pmLine, 2, psSolid);
  ScGraph1.SetSeriesPoints( 2, true, psCircle, 10 );
  ScGraph1.SetSeriesErrBars( 2, false, true, 0, 14 );
  ScGraph2.SetSeriesPoints( 1, true, psCircleFilled, 10 );
  ScGraph2.SetSeriesLine( 1, true, pmLine, 0, psDash);
end;

procedure TForm1.FormResize(Sender: TObject);
begin
  // adjust position and sizes of components
  Panel1.Width:= ClientWidth - 16;
  Panel1.Height:= ClientHeight - 48;
  Button1.Top:= Panel1.Height + 16;
  Button2.Top:= Button1.Top;
  Button3.Top:= Button1.Top;
  Button4.Top:= Button1.Top;

  ScGraph1.Height:= Panel1.Height div 3 * 2;
  ScGraph1.PlotBox.FixedHorPos:= false;
  ScGraph1.Paint; // paint first to get plotbox size of first graph

  // sizing the second graph according to the master
  with ScGraph2 do
  begin
    PlotBox.FixedHorPos:= true;
    PlotBox.Left := ScGraph1.PlotBox.Left;
    PlotBox.Right:= ScGraph1.PlotBox.Right;
    XAxis.Assign(ScGraph1.XAxis);
    XAxis.TitleEnabled:= true;   // restore some props
    XAxis.TickLabelEnabled:= true;
    XAxis.Caption:= 'Time, sec';
    GridVertical.Assign(ScGraph1.GridVertical);
    XScale.Assign(ScGraph1.XScale);
    Paint;      // paint second graph
  end;
end;

/////// toggle x-scale ////////////////////////
procedure TForm1.Button1Click(Sender: TObject);
begin
  if ScGraph1.XScale.ScaleType = stLin then
  begin   // switch to log
    ScGraph1.XScale.ScaleType:= stLog;
    ScGraph1.XScale.Min:= 0.7;
    Button1.Caption:= 'X - lin scale';
  end
  else
  begin   // swith to lin
    ScGraph1.XScale.ScaleType:= stLin;
    ScGraph1.XScale.Min:= 0.0;
    Button1.Caption:= 'X - log scale';
  end;
  ScGraph1.XScale.Intersection:= ScGraph1.XScale.Min;

  FormResize(Self);
end;

/////// toggle y-scale ////////////////////////
procedure TForm1.Button2Click(Sender: TObject);
begin
  if ScGraph1.YScale.ScaleType = stLin then
  begin
    ScGraph1.YScale.ScaleType:= stLog;
    ScGraph1.YScale.Min:= 0.1;
    Button2.Caption:= 'Y1 - lin scale';
  end
  else
  begin
    ScGraph1.YScale.ScaleType:= stLin;
    ScGraph1.YScale.Min:= 0;
    Button2.Caption:= 'Y1 - log scale';
  end;
  // restore intersection
  ScGraph1.YScale.Intersection:= ScGraph1.YScale.Min;
  FormResize(Self);
end;

procedure TForm1.Button3Click(Sender: TObject);
begin
  Close;
end;

procedure TForm1.Button4Click(Sender: TObject);
begin
  // print centered, size 140x110 mm
  PrintGraph(-1, -1, 140, 110);
end;

procedure TForm1.PrintGraph(LeftMM, TopMM, WidthMM, HeightMM: longint);
// a fast hack for a printing routine
// prints the two stacked graphs in a rectangle given by
// LeftMM, TopMM, WidthMM, HeightMM in mm.
// 70% percent of height is used for the first graph
// appearance not as predictable as with 'autometrics' on
var
  MF: TMetafile;
  PrWidth, PrHeight, OffsetX, OffsetY, HorSize, VerSize,
  L, W, H, T: longint;
  ScaleX, ScaleY, fs: single;
begin
  Printer.BeginDoc;
  try
    {get total printer width and height in pixels/device units}
    PrWidth := GetDeviceCaps(Printer.Handle, PHYSICALWIDTH);
    PrHeight:= GetDeviceCaps(Printer.Handle, PHYSICALHEIGHT);
    {get printer offsets in pixels/device units}
    OffsetX := GetDeviceCaps(Printer.Handle, PHYSICALOFFSETX) * 2;
    OffsetY := GetDeviceCaps(Printer.Handle, PHYSICALOFFSETY) * 2;
    {get available printer paper size in mm}
    HorSize := GetDeviceCaps(Printer.Handle, HORZSIZE);
    VerSize := GetDeviceCaps(Printer.Handle, VERTSIZE);
    {scales in pixels/device units per mm}
    ScaleX := (PrWidth  - OffsetX) / HorSize;
    ScaleY := (PrHeight - OffsetY) / VerSize;

    {adjust position and size}
    {check on -1: center graph }
    if WidthMM >= 0 then W:= round(WidthMM * ScaleX)
                    else W:= PrWidth - OffsetX;
    if LeftMM >= 0 then L:= round(LeftMM * ScaleX) - OffsetX div 2
                   else L:= (PrWidth - W - OffsetX) div 2;
    if L < 0 then L := 0;
    if (L + W) > (PrWidth - OffsetX) then
      W:= PrWidth - L - OffsetX;

    if HeightMM >= 0 then H:= round(HeightMM * ScaleY)
                     else H:= PrHeight - OffsetY;
    if TopMM >= 0 then T:= round(TopMM * ScaleY) - OffsetY div 2
                  else T:= (PrHeight -H - OffsetY) div 2;
    if T < 0 then T:= 0;
    if (T + H) > (PrHeight - OffsetY) then H:= PrHeight - T - OffsetY;

    // scale both graphs the hard way
    fs:= (H + W) / 2 / 1000;
    ScaleGraph(fs);

    Printer.Title := 'ScGraph-Plot';
    // fool the printer to get the right handle
    Printer.Canvas.TextOut(0, 0, ' ');

    MF:= TMetafile.Create;
    // 70% of height for first graph
    ScGraph1.CopyGraphToMetafile(MF, true, W, round(H * 0.7), '');
    Printer.Canvas.Draw(L, T, MF);

    with ScGraph2 do
    begin
      PlotBox.FixedHorPos:= true;
      PlotBox.Left := ScGraph1.PlotBox.Left;
      PlotBox.Right:= ScGraph1.PlotBox.Right;
      // remaining 30% for the second graph
      CopyGraphToMetafile(MF, true, W, round(H * 0.3), '');
      Printer.Canvas.Draw(L, T + round(H * 0.7), MF);
    end;

    // scaling back to original
    ScaleGraph(1/fs);
  finally
    MF.Free;
    Printer.EndDoc;
  end;
end;

procedure TForm1.ScaleGraph(f: single);
  // scaling a graph the hard way
  // autmetrics false -->> no scaling of grids, FrameWidths,
  // distances, etc !!!!
var ti, ti1: integer;
    ts, ts1: boolean;
    tSh: TPointShape;
    tPm: TPlotMode;
    TLs: TPenStyle;
begin
  with ScGraph1 do
  begin
    AxisTitleFont.Size := round(f * AxisTitleFont.Size);
    GraphTitleFont.Size:= round(f * GraphTitleFont.Size);
    LegendFont.Size    := round(f * LegendFont.Size);
    TickLabelFont.Size := round(f * TickLabelFont.Size);
    AxisPen.Width := round(f * AxisPen.Width);
    FramePen.Width := round(f * FramePen.Width);
    GetSeriesPoints(2, ts, tSh, ti );
    SetSeriesPoints(2, ts, tSh, round(f * ti));
    GetSeriesErrBars(2, ts, ts1, ti, ti1);
    SetSeriesErrBars(2, ts, ts1, round(f * ti), round(f * ti1));
    GetSeriesLine(1, ts, tPm, ti, tLs);
    SetSeriesLine(1, ts, tPm, round(f * ti), tLs);
  end;
  with ScGraph2 do
  begin
    AxisTitleFont.Size := round(f * AxisTitleFont.Size);
    GraphTitleFont.Size:= round(f * GraphTitleFont.Size);
    LegendFont.Size    := round(f * LegendFont.Size);
    TickLabelFont.Size := round(f * TickLabelFont.Size);
    AxisPen.Width := round(f * AxisPen.Width);
    FramePen.Width := round(f * FramePen.Width);
    GetSeriesPoints(1, ts, tSh, ti );
    SetSeriesPoints(1, ts, tSh, round(f * ti));
    GetSeriesLine(1, ts, tPm, ti, tLs);
    SetSeriesLine(1, ts, tPm, round(f * ti), tLs);
  end;
end;

end.


