// **************************************************************************************************
//                                         uRuler Version 1.0
// **************************************************************************************************

//      Robert N. Kelly
//      email: Who_Dares_Wins@msn.com

//      Copyright  June 1999, Robert N. Kelly

// **************************************************************************************************
//                                        DESCRIPTION OF UNIT
// **************************************************************************************************

// The unit is freeware, or whatever else you want to call it. It was an experiment to emulate
// Microsoft WORD for Windows ruler. It was designed to be used with a TRichEdit component and
// should give the look and feel of the ruler that is found in Word although the interface section
// with the TRichEdit component has not been provided here

// It allows the programmer complete control over colour and presentation, and unlike word, does
// have a bug when reversing the margin positions. For example. moving the right indent, first line
// indent and left indent beyond the right hand margin, and then moving the left margin beyond the
// right hand margin can produce some unusual results. This is prevented here by limiting the distance
// between left handed indents and margins and right hand indents and margins. This should be
// maintained. I use the variables, WHOLEUNIT, HALFUNIT, SMALLUNIT and VERYSMALLUNIT.

//                    WholeUnit     := 40;                { = 40 }
//                    HalfUnit      := WholeUnit div 2;   { = 20 }
//                    SmallUnit     := WholeUnit div 4;   { = 10 }
//                    VerySmallUnit := WholeUnit div 8    { = 05 }

// It is also important to note that the height if the ruler must be 25. I should really change the
// way in which the image is drawn, but you can do this your self. For example, when you position
// either the HangingIndent image, LeftIndent image or RightIndent image, it measures 6 units less
// than the total height of the ruler image. If the image is larger than 25 units, these indents are
// not placed in the correct Y axis of the image.

// I have also included a tab box on the left hand side of the ruler to allow the user to select
// between left, right, centre and decimal tabs. Here the programmer must provide routines to cover
// these. I have yet to include the placing of tabs onto the ruler itself. This should be done at
// the MOUSEDOWN event in order to capture and test where the mouse pointer is and passing its
// position to the DRAWTAB routine with the TabType which is automatically updated and the WithShadow
// parameter set to FALSE. If it is set to TRUE, the program assumes the use has clicked on the Tab
// box itself and so changes the type, otherwise it draw a tab on the ruler itself.

// I think the best way to store user tabs is to user a variant array storing a record containing a
// TIMAGE component and a variable identifying the tab type and position. For example,

//            TTabInfo = record
//                         Image    : TImage;
//                         TabType  : TTabType;
//                       end;

// This is because the TIMAGE can then store the image of the ruler before the tab is drawn, and
// replace the image later if the tab is moved or removed altogether. The TAG variable of the TIMAGE
// can also be used to store the position of the tab.

// If you want to add Tabs to the ruler, remember that these should be removed first and drawn last
// when changes occur otherwise you may copy a background back onto the ruler that has changed because
// I use extensive use of off screen bitmapped images. (See later).

// Hints are also dynamic. There is really only one hint, but the text is changed depending on where
// the mouse pointer is. Whilst it works, I don't feel it works very well... I cannot figure out why.

// In order to produce flicker free images, I use extensive use of off screen bitmaps from the TIMAGE
// component. All drawing is done on these bitmaps and then copied into the main canvas. I tried to
// convert the whole thing to a component, but failed miserably. I didn't really have the time to do
// and ensure flicker free operation.

// If you look at the FORM.CREATE procedure, you will see that the ruler can be extensively modified
// in terms of colour and tick marks etc. It should be noted that the whole thing using the default
// mapping mode. It does not allow for scaling. This is something that could be added in time. I am
// not sure how Microsoft have achieved this. I have experimented using MM_LOMETRIC, MM_LONGLISH and
// MM_TWIPS, but I have never managed to produce acceptable levels of scaling. Maybe someone can
// figure this out.

// As I have said at the start, this was an experiment in emulation. Please feel free to use this as
// you wish, but I would recommend that you test it thoroughly before releasing it in any of your own
// software. I would also be interested to hear from anyone who converts it to a usable component. All
// I ask is that you give me a copy with source.

// If anyone uses this or modifies it, but the underlying code is mainly mine, I would appreciate a
// mention for my efforts. All comments from users are welcome. Enjoy.
// **************************************************************************************************

unit uRuler;
interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls, StdCtrls, ComCtrls;

type
  TDragObject = (doLeftMargin, doRightMargin, doFirstIndent, doHangIndent, doLeftIndent, doRightIndent, doNone);
  TTabType    = (ttLeft, ttCentre, ttRight, ttDecimal);
  TTickType   = (tkLong, tkShort, tkBoth, tkNone);
  TTickShape  = (tsLine, tsDiamond, tsSquare, tsTriangle);

  TForm1 = class(TForm)
    Panel1: TPanel;
    imRuler: TImage;
    imFirstLineIndent: TImage;
    imLeftIndent: TImage;
    imHangingIndent: TImage;
    imRightIndent: TImage;
    imGhostFirst: TImage;
    imGhostLeft: TImage;
    RichEdit1: TRichEdit;
    imRulerTab: TImage;
    procedure FormCreate(Sender: TObject);
    procedure imRulerMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure imRulerMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
    procedure imRulerMouseUp(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure FormDestroy(Sender: TObject);
    procedure FormResize(Sender: TObject);
  private
    { Private declarations }
    Paper        : TRichEdit;
    PHDC         : HDC;
    PenLine      : HPEN;
    GhostVisible : boolean;
    TabType      : TTabType;
    Tolerance    : integer;
    procedure EraseIndents;
    procedure DrawTickShape(X : integer; cl: TColor);
    procedure DrawRmHighlight;
    procedure DrawTab(TabType: TTabType; X: integer; WithShadow: boolean);
    procedure ShowTab;
    procedure DrawTabBlock;
    procedure DrawIndents;
    procedure DrawGhosts;
    procedure CopyIndents;
    procedure DrawTicks(LeftTicks: boolean; StartPos, EndPos: integer);
    procedure DrawLeftMarginNumbers;
    procedure DrawRightMarginNumbers(StartPos, EndPos: integer);
    procedure DrawTabTicks(Erase: boolean; StartPos, EndPos: integer);
    procedure MoveLeftMargin(X: Integer);
    procedure MoveRightMargin(X: Integer);
    procedure MoveFirstIndent(X: Integer);
    procedure MoveLeftIndent(X: Integer);
    procedure MoveHangIndent(X: Integer);
    procedure MoveRightIndent(X: Integer);
    procedure DrawIndentLine(IndentLine: integer);
    function CheckHints(X, Y: integer): TDragObject;
    procedure CheckDragType;
  public
    { Public declarations }
    Dragging          : boolean;
    DragType          : TDragObject;
    DragOffset        : integer;
    Offset            : integer;
    LeftMargin        : integer;
    RightMargin       : integer;
    PaperWidth        : integer;
    TabDistance       : integer;
    WholeUnit         : integer;
    HalfUnit          : integer;
    SmallUnit         : integer;
    VerySmallUnit     : integer;
    MinimumPaperSpace : integer;
    FirstLineIndent   : integer;
    HangingIndent     : integer;
    LeftIndent        : integer;
    RightIndent       : integer;
    TickType          : TTickType;
    TickShape         : TTickShape;
    ShowNumbers       : boolean;
    BorderColor       : TColor;
    MarginColor       : TColor;
    MarginTickColor   : TColor;
    PaperColor        : TColor;
    LongTickColor     : TColor;
    ShortTickColor    : TColor;
    NumberTickColor   : TColor;
    TabMarkerColor    : TColor;
    MarginEdgeColor   : TColor;
    IndentFace        : TColor;
    IndentHighlight   : TColor;
    IndentShadow      : TColor;
    IndentLine        : TColor;
    GhostFace         : TColor;
    GhostHighLight    : TColor;
    IndentBorder      : TColor;
    TabBlockFace      : TColor;
    TabBlockHighlight : TColor;
    TabBlockShadow    : TColor;
    TabImage          : TColor;
  end;

var
  Form1: TForm1;

implementation

{$R *.DFM}

procedure TForm1.EraseIndents;
  var
       hw   : integer;
       OldX : integer;

  begin
    with imRuler do
    begin
      hw := imFirstLineIndent.Width div 2 -1;
      Oldx := FirstLineIndent +Offset -hw;
      // replace the original ruler
      Canvas.CopyRect(rect(OldX, 1, OldX +imFirstLineIndent.Width -1, 1 +imFirstLineIndent.Height -1),
                           imFirstLineIndent.Canvas,
                           rect(0, 0, imFirstLineIndent.Width -1, imFirstLineIndent.Height -1));
      hw := imHangingIndent.Width div 2 -1;
      Oldx := HangingIndent +Offset -hw;
      // replace the original ruler
      Canvas.CopyRect(rect(OldX, 10, OldX +imHangingIndent.Width -1, 10 +imHangingIndent.Height -1),
                           imHangingIndent.Canvas,
                           rect(0, 0, imHangingIndent.Width -1, imHangingIndent.Height -1));
      hw := imLeftIndent.Width div 2 -1;
      Oldx := LeftIndent +Offset -hw;
      // replace the original ruler
      Canvas.CopyRect(rect(OldX, 18, OldX +imLeftIndent.Width -1, 18 +imLeftIndent.Height -1),
                           imLeftIndent.Canvas,
                           rect(0, 0, imLeftIndent.Width -1, imLeftIndent.Height -1));
      hw := imRightIndent.Width div 2 -1;
      Oldx := RightIndent +Offset -hw;
      // replace the original ruler
      Canvas.CopyRect(rect(OldX, 10, OldX +imRightIndent.Width -1, 10 +imRightIndent.Height -1),
                           imRightIndent.Canvas,
                           rect(0, 0, imRightIndent.Width -1, imRightIndent.Height -1));
      Oldx := Offset -imRulerTab.Width *2;
      // replace the original ruler
      Canvas.CopyRect(rect(OldX, 0, OldX +imRulerTab.Width, 0 +imRulerTab.Height -1),
                           imRulerTab.Canvas,
                           rect(0, 0, imRulerTab.Width -1, imRulerTab.Height -1))
    end
  end;

procedure TForm1.DrawTickShape(X : integer; cl: TColor);
  begin
    with imRuler do
    begin
      Canvas.Pen.Color := cl;
      Canvas.Brush.Color := cl;
      case TickShape of
        tsLine     : begin
                       Canvas.MoveTo(X, 9);
                       Canvas.LineTo(X, 15)
                     end;
        tsDiamond  : begin
                       Canvas.MoveTo(X -1, 10); Canvas.LineTo(X +2, 10);
                       Canvas.MoveTo(X -2, 11); Canvas.LineTo(X +3, 11);
                       Canvas.MoveTo(X -1, 12); Canvas.LineTo(X +2, 12);
                       Canvas.MoveTo(X, 9); Canvas.LineTo(X, 14)
                     end;
        tsSquare   : Canvas.FillRect(rect(X -1, 10, X +2, 15));
        tsTriangle : begin
                       Canvas.MoveTo(X -2, 11); Canvas.LineTo(X +3, 11);
                       Canvas.MoveTo(X -1, 12); Canvas.LineTo(X +2, 12);
                       Canvas.MoveTo(X, 12); Canvas.LineTo(X, 14)
                     end
      end
    end
  end;

procedure TForm1.DrawRmHighlight;
  begin
    if not (DragType in [doLeftMargin, doRightMargin, doNone]) then
      exit;
    with imRuler do
    begin
      Canvas.Brush.Color := MarginEdgeColor;
      Canvas.FillRect(rect(Offset +RightMargin -1, 5, Offset +RightMargin +2, Height -6));
      Canvas.Pen.Color := MarginTickColor;
      if (RightMargin -LeftMargin) mod WholeUnit = 0 then
        exit;
      if TickType <> tkNone then
      begin
        if (RightMargin -LeftMargin) mod HalfUnit = 0 then
          begin
            if TickType in[tkLong, tkBoth] then
              DrawTickShape(Offset +RightMargin, MarginTickColor)
          end
        else
          if TickType in[tkShort, tkBoth] then
          begin
            Canvas.MoveTo(Offset +RightMargin, 11);
            Canvas.LineTo(Offset +RightMargin, 13)
          end
      end
    end
  end;

procedure TForm1.DrawTab(TabType: TTabType; X: integer; WithShadow: boolean);
  var
       y : integer;

  begin
    with imRuler do
    begin
      Canvas.Pen.Width := 2;
      Canvas.Pen.Color := TabBlockShadow;
      Canvas.Brush.Color := TabBlockShadow;
      y := 4;
      if withShadow then
      begin
        y := 0;
        Case TabType of
          ttLeft    : begin
                        Canvas.MoveTo(x +10,  y +9);
                        Canvas.LineTo(x +10,  y +14);
                        Canvas.LineTo(x +14, y +14)
                      end;
          ttCentre,
          ttDecimal : begin
                        Canvas.MoveTo(x +8, y +14);
                        Canvas.LineTo(x +15, y +14);
                        Canvas.LineTo(x +8, y +14);
                        Canvas.MoveTo(x +12, y +14);
                        Canvas.LineTo(x +12, y +9);
                        if TabType = ttDecimal then
                          Canvas.FillRect(rect(x +14, y +10, x + 16, y +12))
                      end;
        ttRight   : begin
                      Canvas.MoveTo(x +14, y +9);
                      Canvas.LineTo(x +14, y +14);
                      Canvas.LineTo(x +9, y +14)
                    end
        end
      end;
      Canvas.Pen.Color := TabImage;
      Canvas.Brush.Color := TabImage;
      Case TabType of
        ttLeft    : begin
                      Canvas.MoveTo(x +9, y +8);
                      Canvas.LineTo(x +9, y +13);
                      Canvas.LineTo(x +13, y +13)
                    end;
        ttCentre,
        ttDecimal : begin
                      Canvas.MoveTo(x +7, y +13);
                      Canvas.LineTo(x +14, y +13);
                      Canvas.LineTo(x +7, y +13);
                      Canvas.MoveTo(x +11, y +13);
                      Canvas.LineTo(x +11, y +8);
                      if TabType = ttDecimal then
                        Canvas.FillRect(rect(x +13, y +9, x + 15, y +11))
                    end;
        ttRight   : begin
                      Canvas.MoveTo(x +13, y +8);
                      Canvas.LineTo(x +13, y +13);
                      Canvas.LineTo(x +8, y +13)
                    end
      end;
      Canvas.Pen.Width := 1
    end
  end;

procedure TForm1.ShowTab;
  var
       x : integer;

  begin
    with imRuler do
    begin
      x := Offset -imRulerTab.Width *2;
      Canvas.Brush.Color := TabBlockFace;
      Canvas.FillRect(rect(x +5, 5, x +17, 17));
      Canvas.Pen.Width := 2;
      DrawTab(TabType, X, TRUE)
    end
  end;

procedure TForm1.DrawTabBlock;
  var
       x : integer;

  begin
    with imRulerTab do
    begin
      x := Offset -Width *2;
      imRuler.Canvas.Pen.Color := TabBlockShadow;
      imRuler.Canvas.Brush.Style := bsSolid;
      imRuler.Canvas.Brush.Color := TabBlockFace;


      // now draw the indent
      imRuler.Canvas.FillRect(rect(x, 0, x +Width, Height));
      imRuler.Canvas.Rectangle(x +3, 3, x +19, 19);
      imRuler.Canvas.Pen.Color := TabBlockHighLight;
      imRuler.Canvas.MoveTo(x +17, 4); imRuler.Canvas.LineTo(x +4, 4); imRuler.Canvas.LineTo(x +4, 18);
      imRuler.Canvas.MoveTo(x +4,19); imRuler.Canvas.LineTo(x +19, 19); imRuler.Canvas.LineTo(x +19, 3);
      ShowTab
    end
  end;

procedure TForm1.DrawIndents;
  var
       hw   : integer;
       NewX : integer;

  begin
    with imRuler do
    begin
      if GhostVisible then
        case DragType of
          doFirstIndent : if FirstLineIndent = imGhostFirst.Tag then
                            DrawGhosts;
          doHangIndent  : if HangingIndent = imGhostLeft.Tag then
                            DrawGhosts;
          doLeftIndent  : if HangingIndent = imGhostLeft.Tag then
                            DrawGhosts;
          doRightIndent : if RightIndent = imGhostFirst.Tag then
                            DrawGhosts;
        end;
      DrawRmHighLight;
      Canvas.Pen.Color := IndentBorder;
      Canvas.Brush.Style := bsSolid;
      Canvas.Brush.Color := IndentFace;


      // copy the new ruler - tab block should be done before any other
      NewX := Offset -imRulerTab.Width *2;
      imRulerTab.Canvas.CopyRect(rect(0, 0, imRulerTab.Width -1, imRulerTab.Height -1),
                                 Canvas,
                                 rect(NewX, 0, NewX +imRulerTab.Width -1, imRulerTab.Height -1));


      hw := imFirstLineIndent.Width div 2 -1;
      NewX := FirstLineIndent +Offset -hw;
      // copy the new ruler
      imFirstLineIndent.Canvas.CopyRect(rect(0, 0, imFirstLineIndent.Width -1, imFirstLineIndent.Height -1),
                                        Canvas,
                                        rect(NewX, 1, NewX +imFirstLineIndent.Width -1, 1 +imFirstLineIndent.Height -1));
      // now draw the indent
      Canvas.Polygon([point(NewX, 1), point(NewX +8, 1), point(NewX +8, 4), point(NewX +4, 8), point(NewX, 4)]);
      Canvas.Pen.Color := IndentHighLight;
      Canvas.MoveTo(NewX +7, 2); Canvas.LineTo(NewX +1, 2); Canvas.LineTo(NewX +1, 4); Canvas.LineTo(NewX +4, 7);
      Canvas.Pen.Color := IndentShadow;
      Canvas.MoveTo(NewX +7, 3); Canvas.LineTo(NewX +7, 4); Canvas.LineTo(NewX +3, 8);

      hw := imHangingIndent.Width div 2 -1;
      NewX := HangingIndent +Offset -hw;
      Canvas.Pen.Color := IndentBorder;
      // copy the new ruler
      imHangingIndent.Canvas.CopyRect(rect(0, 0, imHangingIndent.Width -1, imHangingIndent.Height -1),
                                      Canvas,
                                      rect(NewX, 10, NewX +imHangingIndent.Width -1, 10 +imHangingIndent.Height -1));
      // now draw the indent
      Canvas.Polygon([point(NewX, 17), point(NewX, 14), point(NewX +4, 10), point(NewX +8, 14), point(NewX +8, 17)]);
      Canvas.Pen.Color := IndentHighLight;
      Canvas.MoveTo(NewX +1, 16); Canvas.LineTo(NewX +1, 14); Canvas.LineTo(NewX +5, 10);
      Canvas.Pen.Color := IndentShadow;
      Canvas.MoveTo(NewX +2, 16); Canvas.LineTo(NewX +7, 16); Canvas.LineTo(NewX +7, 14); Canvas.LineTo(NewX +4, 11);

      hw := imLeftIndent.Width div 2 -1;
      NewX := LeftIndent +Offset -hw;
      Canvas.Pen.Color := IndentBorder;
      // copy the new ruler
      imLeftIndent.Canvas.CopyRect(rect(0, 0, imLeftIndent.Width -1, imLeftIndent.Height -1),
                                   Canvas,
                                   rect(NewX, 18, NewX +imLeftIndent.Width -1, 18 +imLeftIndent.Height -1));
      // now draw the indent
      Canvas.Polygon([point(NewX, 17), point(NewX, 23), point(NewX +8, 23), point(NewX +8, 17)]);
      Canvas.Pen.Color := IndentHighLight;
      Canvas.MoveTo(NewX +7, 18); Canvas.LineTo(NewX +1, 18); Canvas.LineTo(NewX +1, 23);
      Canvas.Pen.Color := IndentShadow;
      Canvas.MoveTo(NewX +7, 19); Canvas.LineTo(NewX +7, 22); Canvas.LineTo(NewX +1, 22);

      hw := imRightIndent.Width div 2 -1;
      NewX := RightIndent +Offset -hw;
      Canvas.Pen.Color := IndentBorder;
      // copy the new ruler
      imRightIndent.Canvas.CopyRect(rect(0, 0, imRightIndent.Width -1, imRightIndent.Height -1),
                                    Canvas,
                                    rect(NewX, 10, NewX +imRightIndent.Width -1, 10 +imRightIndent.Height -1));
      // now draw the indent
      Canvas.Polygon([point(NewX, 17), point(NewX, 14), point(NewX +4, 10), point(NewX +8, 14), point(NewX +8, 17)]);
      Canvas.Pen.Color := IndentHighLight;
      Canvas.MoveTo(NewX +1, 16); Canvas.LineTo(NewX +1, 14); Canvas.LineTo(NewX +5, 10);
      Canvas.Pen.Color := IndentShadow;
      Canvas.MoveTo(NewX +2, 16); Canvas.LineTo(NewX +7, 16); Canvas.LineTo(NewX +7, 14); Canvas.LineTo(NewX +3, 10);

      DrawTabBlock
    end
  end;

procedure TForm1.DrawGhosts;
  var
       hw   : integer;
       NewX : integer;

  begin
    with imRuler do
    begin
      Canvas.Pen.Color := GhostHighLight;
      Canvas.Brush.Style := bsSolid;
      Canvas.Brush.Color := GhostFace;
      if DragType in [doFirstIndent, doLeftIndent] then
      begin
        // if ghost is visible, then repaint ruler face
        hw := imFirstLineIndent.Width div 2 -1;
        if GhostVisible then
          begin
            NewX := imGhostFirst.Tag +Offset -hw;
            // replace the original ruler
            Canvas.CopyRect(rect(NewX, 1, NewX +imGhostFirst.Width -1, 1 +imGhostFirst.Height -1),
                            imGhostFirst.Canvas,
                            rect(0, 0, imGhostFirst.Width -1, imGhostFirst.Height -1))
          end
        else
          begin
            // copy ruler under Ghost
            NewX := FirstLineIndent +Offset -hw;
            imGhostFirst.Canvas.CopyRect(rect(0, 0, imGhostFirst.Width -1, imGhostFirst.Height -1),
                                         Canvas,
                                         rect(NewX, 1, NewX +imGhostFirst.Width -1, 1 +imGhostFirst.Height -1));
            // draw Ghost indent
            Canvas.Polygon([point(NewX, 1), point(NewX +8, 1), point(NewX +8, 4), point(NewX +4, 8), point(NewX, 4)]);
            // save its position in the tag field
            imGhostFirst.Tag := FirstLineIndent
          end
      end;
      if DragType in[doHangIndent, doLeftIndent] then
      begin
        // if ghost is visible, then repaint ruler face
        hw := imHangingIndent.Width div 2 -1;
        if GhostVisible then
          begin
            NewX := imGhostLeft.Tag +Offset -hw;
            // replace the original ruler
            Canvas.CopyRect(rect(NewX, 10, NewX +imGhostLeft.Width -1, 10 +imGhostLeft.Height -1),
                            imGhostLeft.Canvas,
                            rect(0, 0, imGhostLeft.Width -1, imGhostLeft.Height -1))
          end
        else
          begin
            // copy ruler under Ghost
            NewX := HangingIndent +Offset -hw;
            imGhostLeft.Canvas.CopyRect(rect(0, 0, imGhostLeft.Width -1, imGhostLeft.Height -1),
                                        Canvas,
                                        rect(NewX, 10, NewX +imGhostLeft.Width -1, 10 +imGhostLeft.Height -1));
            // draw Ghost indent of hanging indent
            Canvas.Polygon([point(NewX, 17), point(NewX, 14), point(NewX +4, 10), point(NewX +8, 14), point(NewX +8, 17)]);
            // draw Ghost indent of left indent
            Canvas.Polygon([point(NewX, 17), point(NewX, 23), point(NewX +8, 23), point(NewX +8, 17)]);
            // save its positioin in the tag field
            imGhostLeft.Tag := HangingIndent
          end
      end;
      if DragType = doRightIndent then
      begin
        // if ghost is visible, then repaint ruler face
        hw := imRightIndent.Width div 2 -1;
        if GhostVisible then
          begin
            NewX := imGhostFirst.Tag +Offset -hw;
            // replace the original ruler
            Canvas.CopyRect(rect(NewX, 10, NewX +imGhostFirst.Width -1, 10 +imGhostFirst.Height -1),
                            imGhostFirst.Canvas,
                            rect(0, 0, imGhostFirst.Width -1, imGhostFirst.Height -1))
          end
        else
          begin
            // copy ruler under Ghost
            NewX := RightIndent +Offset -hw;
            // draw Ghost indent
            imGhostFirst.Canvas.CopyRect(rect(0, 0, imGhostFirst.Width -1, imGhostFirst.Height -1),
                                          Canvas,
                                          rect(NewX, 10, NewX +imGhostFirst.Width -1, 10 +imGhostFirst.Height -1));
            // now draw Ghost indent
            Canvas.Polygon([point(NewX, 17), point(NewX, 14), point(NewX +4, 10), point(NewX +8, 14), point(NewX +8, 17)]);
            // save its positioin in the tag field
            imGhostFirst.Tag := RightIndent
          end
      end
    end;
    GhostVisible := not GhostVisible
  end;

procedure TForm1.CopyIndents;
  var
       x : integer;

  begin
    // copy the areas of the ruler under the proposed indents
    x := Offset +FirstLineIndent -imFirstLineIndent.Width div 2;
    imFirstLineIndent.Canvas.CopyRect(rect(0, 0, imFirstLineIndent.Width -1, imFirstLineIndent.Height -1),
                                      imRuler.Canvas,
                                      rect(x, 1, x +imFirstLineIndent.Width -1, 1 +imFirstLineIndent.Height -1));

    x := Offset +HangingIndent -imHangingIndent.Width div 2;
    imHangingIndent.Canvas.CopyRect(rect(0, 0, imHangingIndent.Width -1, imHangingIndent.Height -1),
                                      imRuler.Canvas,
                                      rect(x, 10, x +imHangingIndent.Width -1, 10 +imHangingIndent.Height -1));

    x := Offset +LeftIndent -imLeftIndent.Width div 2;
    imLeftIndent.Canvas.CopyRect(rect(0, 0, imLeftIndent.Width -1, imLeftIndent.Height -1),
                                      imRuler.Canvas,
                                      rect(x, 18, x +imLeftIndent.Width -1, 18 +imLeftIndent.Height -1));

    x := Offset +RightIndent -imRightIndent.Width div 2;
    imRightIndent.Canvas.CopyRect(rect(0, 0, imRightIndent.Width -1, imRightIndent.Height -1),
                                      imRuler.Canvas,
                                      rect(x, 10, x +imRightIndent.Width -1, 10 +imRightIndent.Height -1));

    x := Offset- imRulerTab.Width *2;
    imRulerTab.Canvas.CopyRect(rect(0, 0, imRulerTab.Width -1, imRulerTab.Height -1),
                                      imRuler.Canvas,
                                      rect(x, 0, x +imRulerTab.Width-1, imRulerTab.Height -1));
    DrawIndents
  end;

procedure TForm1.DrawTicks(LeftTicks: boolean; StartPos, EndPos: integer);
  var
       x   : integer;

  procedure DrawTickMark;
    begin
      with imRuler do
        if x mod WholeUnit = 0 then                            // if zero then we are over a number
          begin
            if not ShowNumbers then
              DrawTickShape(x +StartPos, NumberTickColor)
          end
        else
          if (x mod HalfUnit) = 0 then                        // must be the middle, so draw a long tick mark
            begin
              if TickType in [tkLong, tkBoth] then
                DrawTickShape(x +StartPos, LongTickColor)
            end
          else
            if TickType in [tkShort, tkBoth] then
            begin                                       // must be a short tick mark
              Canvas.Pen.Color := ShortTickColor;
              Canvas.MoveTo(x +StartPos, 11);
              Canvas.LineTo(x +StartPos, 13)
            end
    end;

  begin
    x := -SmallUnit;                                         // this is the first tick to the left side of the left margin
    if LeftTicks then                                 // check if it is necessary to draw these
      while x +StartPos > 0 do
      begin
        DrawTickMark;
        dec(x, SmallUnit)
      end;
    x := 0;
    while x < EndPos -StartPos do                     // these are tick marks to the right side of the left margin
    begin
      DrawTickMark;
      inc(x, SmallUnit)
    end
  end;

procedure TForm1.DrawLeftMarginNumbers;

  var
       x : integer;
       n : integer;
       s : string;

  begin
    if not ShowNumbers then
      exit;
    n := 1;
    x := Offset +LeftMargin -WholeUnit;
    imRuler.Canvas.Brush.Style := bsClear;
    while x >= 0 do
    begin
      with imRuler do
      begin
        s := IntToStr(n);
        Canvas.TextOut(x -TPoint(imRuler.Canvas.TextExtent(s)).X div 2, (Height -TPoint(imRuler.Canvas.TextExtent(s)).Y) div 2, s)
      end;
      inc(n);
      dec(x, WholeUnit)
    end
  end;

procedure TForm1.DrawRightMarginNumbers(StartPos, EndPos: integer);

  var
       x : integer;
       n : integer;
       s : string;

  begin
    if not ShowNumbers then
      exit;
    n := (StartPos -LeftMargin) div WholeUnit;
    x := Offset +StartPos;
    imRuler.Canvas.Brush.Style := bsClear;
    while x <= Offset +EndPos do
    begin
      if n <> 0 then
      if x <> Offset +RightMargin then
      with imRuler do
      begin
        s := IntToStr(n);
        Canvas.TextOut(x -TPoint(imRuler.Canvas.TextExtent(s)).X div 2, (Height -TPoint(imRuler.Canvas.TextExtent(s)).Y) div 2, s)
      end;
      inc(n);
      inc(x, WholeUnit)
    end
  end;

procedure TForm1.DrawTabTicks(Erase: boolean; StartPos, EndPos: integer);
  var
       x : integer;

  begin
    with imRuler do
    begin
      if Erase then
        Canvas.Pen.Color := BorderColor
      else
        Canvas.Pen.Color := TabMarkerColor;
      x := StartPos;
      repeat
        if x <> Offset +LeftIndent then
        begin
          Canvas.MoveTo(x, Height -4);
          Canvas.LineTo(x, Height)
        end;
        inc(x, TabDistance)
      until x > EndPos
    end
  end;

procedure TForm1.FormCreate(Sender: TObject);
  var
       n : integer;

  begin
    WholeUnit := 40;
    HalfUnit  := WholeUnit div 2;
    SmallUnit := WholeUnit div 4;
    VerySmallUnit := WholeUnit div 8;
    MinimumPaperSpace := WholeUnit;

    imRuler.Canvas.Font.Name := 'Arial Narrow';
    imRuler.Canvas.Font.Size := 8;
    Tolerance := imRuler.Canvas.TextWidth('88') div 2; // make sure numbers are painted out at boundaries of margins
    imRulerTab.Visible := FALSE;
    imRulerTab.Left :=0;
    imRulerTab.Top := 0;
    imRulerTab.Width := 25;
    imRulerTab.Height := 25;

    Offset := 1000;
    Paper := nil;
    Paper := RichEdit1;
    n := Paper.Width -Paper.ClientWidth;
    Paper.BorderStyle := bsNone;
    PaperWidth := 21 *WholeUnit;
    Paper.Width := PaperWidth +1;
    Paper.Left := imRulerTab.Width*2 +n div 2;
    Paper.Top := Panel1.Height;
    imRuler.Left := -Offset +imRulerTab.Width *2;
    LeftMargin := WholeUnit;
    RightMargin := WholeUnit *8;

    FirstLineIndent := LeftMargin;
    HangingIndent := LeftMargin;
    LeftIndent := LeftMargin;
    RightIndent := RightMargin;

    imFirstLineIndent.Visible := FALSE;
    imHangingIndent.Visible := FALSE;
    imLeftIndent.Visible := FALSE;
    imRightIndent.Visible := FALSE;
    imGhostFirst.Visible := FALSE;
    imGhostLeft.Visible := FALSE;
    GhostVisible := FALSE;

    TickType          := tkBoth;
    TickShape         := tsLine;
    ShowNumbers       := TRUE;
    DragType          := doNone;
    TabDistance       := (WholeUnit *3) div 2;
    LongTickColor     := clBlack;
    ShortTickColor    := clBlack;
    NumberTickColor   := clBlack;
    TabMarkerColor    := clBtnShadow;
    BorderColor       := clBtnFace;
    MarginColor       := clBtnShadow;
    MarginTickColor   := clRed;
    PaperColor        := clWindow;
    MarginEdgeColor   := clBtnFace;
    IndentFace        := clBtnFace;
    IndentHighlight   := clBtnHighlight;
    IndentShadow      := clBtnShadow;
    IndentBorder      := clBlack;
    IndentLine        := clBlack;
    GhostFace         := clBtnFace;
    GhostHighLight    := clBtnShadow;
    TabBlockFace      := clBtnFace;
    TabBlockHighlight := clBtnHighLight;
    TabBlockshadow    := clBtnShadow;
    TabImage          := clBlack;
    TabType           := ttLeft;

    with imRuler do
    begin
      Canvas.Brush.Style := bsSolid;
      Canvas.Brush.Color := MarginColor;
      Canvas.FillRect(rect(0, 5, Offset +LeftMargin, Height -6));
      Canvas.FillRect(rect(Offset +RightMargin, 5, Width, Height -6));
      Canvas.Brush.Color := PaperColor;
      Canvas.FillRect(rect(Offset +LeftMargin, 5, Offset +RightMargin, Height -6));
      Canvas.Brush.Color := BorderColor;
      Canvas.FillRect(rect(0, 0, Width, 5));
      Canvas.FillRect(rect(0, Height -6, Width, Height));
      DrawTicks(TRUE, Offset +LeftMargin, Width);
      Canvas.Brush.Color := MarginEdgeColor;
      Canvas.FillRect(rect(Offset +LeftMargin -1, 5, Offset +LeftMargin +2, Height -6));
      DrawRmHighLight;
      DrawLeftMarginNumbers;
      DrawRightMarginNumbers(LeftMargin +WholeUnit, Width -Offset);
      DrawTabTicks(FALSE, Offset +LeftMargin, Offset +RightMargin)
    end;
    CopyIndents;
    PenLine :=  CreatePen(PS_DOT, 0, IndentLine);
    PHDC := GetDCEx(Paper.Handle, 0, DCX_CACHE or DCX_CLIPSIBLINGS or DCX_LOCKWINDOWUPDATE);
    SetROP2 (PHDC, R2_NOTXORPEN);
    HorzScrollBar.Margin := WholeUnit *2
  end;

function TForm1.CheckHints(X, Y: integer): TDragObject;
  var
       dt      : TDragObject;
       r       : TRect;
       pt      : TPoint;
       s       : string;
       Divisor : real;

  begin
    imRuler.ShowHint := FALSE;
    Divisor := WholeUnit *1.0;
    dt := doNone;

    r := rect(0, 0, 9, 8);
    pt.Y := Y;

    s := '';
    pt.X := X-(Offset +FirstLineIndent -4);
    if PtInRect(r, pt) then
    begin
      s := 'First line indent '+Format('(%f)', [FirstLineIndent / Divisor]);
      dt := doFirstIndent
    end;

    r.Top := 11; r.Bottom := 18;
    pt.X := X-(Offset +HangingIndent -4);
    if PtInRect(r, pt) then
    begin
      s := 'Hanging indent '+Format('(%f)', [HangingIndent / Divisor]);
      dt := doHangIndent
    end;

    pt.X := X-(Offset +RightIndent -4);
    if PtInRect(r, pt) then
    begin
      s := 'Right indent '+Format('(%f)', [(PaperWidth -RightIndent) / Divisor]);
      dt := doRightIndent
    end;

    r.Top := 18; r.Bottom := 23;
    pt.X := X-(Offset +LeftIndent -4);
    if PtInRect(r, pt) then
    begin
      s := 'Left indent '+Format('(%f)', [LeftIndent / Divisor]);
      dt := doLeftIndent
    end;

    if dt = doNone then
    begin
      r := rect(0, 5, 5, imRuler.Height -6);
      pt.X := X -(Offset +LeftMargin -2);
      if PtInRect(r, pt) then
      begin
        s := 'Left margin ' +Format('(%f)', [LeftMargin / Divisor]);
        dt := doLeftMargin
      end;
      pt.X := X -(Offset +RightMargin -2);
      if PtInRect(r, pt) then
      begin
        s := 'Right margin '+Format('(%f)', [(PaperWidth -RightMargin) / Divisor]);
        dt := doRightMargin
      end
    end;
    imRuler.Hint := s;
    CheckHints := dt
  end;

procedure TForm1.CheckDragType;
  begin
    case DragType of
      doLeftMargin  : DrawIndentLine(LeftMargin);
      doRightMargin : DrawIndentLine(RightMargin);
      doLeftIndent,
      doHangIndent  : DrawIndentLine(HangingIndent);
      doFirstIndent : DrawIndentLine(FirstLineIndent);
      doRightIndent : DrawIndentLine(RightIndent)
    end
  end;

procedure TForm1.imRulerMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
  var
       xx : integer;

  begin
    xx := Offset -imRulerTab.Width *2;
    if PtInRect(rect(xx +3, 3, xx +imRulerTab.Width -4, 19), Point(X, Y)) then
    begin
      TabType := TTabType((ord(TabType) +1) mod 4);
      ShowTab;
      exit
    end;

    DragType := CheckHints(X, Y);
    if DragType = doNone then
    begin
//      X := (X -LeftMargin -9) div 5 * 5 +LeftMargin +2;
//      DrawTab(TabType, X, FALSE);
      exit;
    end;
    Dragging := TRUE;
    DragOffset := X;
    CheckDragType
  end;

procedure TForm1.MoveLeftMargin(X: Integer);
  Var
       TabStart : integer;
       TabEnd   : integer;
       TabState : boolean;
       xx       : integer;
       n        : integer;

  begin
    n := round((X -DragOffset)/(SmallUnit *1.0)) *SmallUnit;
    if n = 0 then
      exit;
    if n > 0 then // moving LeftMargin to the right
      begin
        xx := LeftMargin;
        if FirstLineIndent >= xx then
          xx := FirstLineIndent;
        if LeftIndent >= xx then
          xx := LeftIndent;
        inc(xx, n +MinimumPaperspace);
        if (xx >RightIndent) or (xx >RightMargin) then
          exit;
        DrawIndentLine(LeftMargin);
        EraseIndents;
        imRuler.Left := imRuler.Left +n;
        dec(Offset, n);
        inc(LeftMargin, n);
        imRuler.Canvas.Brush.Style := bsSolid;
        imRuler.Canvas.Brush.Color := MarginColor;
        imRuler.Canvas.FillRect(rect(Offset +RightMargin, 5, Offset +RightMargin +n +2, imRuler.Height -6));
        imRuler.Canvas.Brush.Color := PaperColor;
        imRuler.Canvas.FillRect(rect(Offset +RightMargin -Tolerance, 5, Offset +RightMargin, imRuler.Height -6));
        if abs(n) < WholeUnit then
          begin
            DrawTicks(FALSE, Offset +(RightMargin -LeftMargin) div WholeUnit *WholeUnit +LeftMargin, Offset +RightMargin +WholeUnit);
            DrawRightMarginNumbers(RightMargin -((RightMargin -LeftMargin) mod WholeUnit), RightMargin +WholeUnit)
          end
        else
          begin
            DrawTicks(FALSE, Offset +(RightMargin -LeftMargin) div WholeUnit *WholeUnit +LeftMargin, Offset +RightMargin +n +SmallUnit);
            DrawRightMarginNumbers(RightMargin -((RightMargin -LeftMargin) mod WholeUnit), RightMargin +n)
          end;
        TabState := TRUE;
        TabStart := Offset +TabDistance -(RightMargin -LeftMargin) mod TabDistance +RightMargin;
        TabEnd := Offset +RightMargin +n
      end
    else
      begin
        if LeftMargin +n < 0 then
          exit;
        DrawIndentLine(LeftMargin);
        EraseIndents;
        imRuler.Left := imRuler.Left +n;
        dec(Offset, n);
        inc(LeftMargin, n);
        imRuler.Canvas.Brush.Style := bsSolid;
        imRuler.Canvas.Brush.Color := PaperColor;
        imRuler.Canvas.FillRect(rect(Offset +RightMargin +n -2, 5, Offset +RightMargin, imRuler.Height -6));
        imRuler.Canvas.Brush.Color := MarginColor;
        imRuler.Canvas.FillRect(rect(Offset +RightMargin, 5, Offset +RightMargin +Tolerance, imRuler.Height -6));
        if abs(n) < WholeUnit then
          begin
            DrawTicks(FALSE, Offset +(RightMargin -LeftMargin) div WholeUnit *WholeUnit +LeftMargin -WholeUnit, Offset +RightMargin);
            DrawRightMarginNumbers(RightMargin -((RightMargin -LeftMargin) mod WholeUnit) -WholeUnit, RightMargin)
          end
        else
          begin
            DrawTicks(FALSE, Offset +(RightMargin -LeftMargin +n) div WholeUnit *WholeUnit +LeftMargin, Offset +RightMargin);
            DrawRightMarginNumbers(LeftMargin +(RightMargin +n -LeftMargin) div WholeUnit *WholeUnit, RightMargin)
          end;
        TabState := FALSE;
        TabStart := Offset +LeftMargin;
        TabEnd := Offset +RightMargin
      end;
    inc(LeftIndent, n);
    inc(FirstLineIndent, n);
    HangingIndent := LeftIndent;
    DrawTabTicks(TabState, TabStart, TabEnd);
    DrawIndents;
    DrawIndentLine(LeftMargin)
  end;

procedure TForm1.MoveRightMargin(X: Integer);
  var
       StartPos : integer;
       EndPos   : integer;
       TabStart : integer;
       TabEnd   : integer;
       TabState : boolean;
       n        : integer;

  begin
    n := round((X -DragOffset)/(SmallUnit *1.0)) *SmallUnit;
    if n = 0 then
      exit;
    if n > 0 then // moving RightMargin to the right
      begin
        DrawIndentLine(RightMargin);
        EraseIndents;
        imRuler.Canvas.Brush.Style := bsSolid;
        imRuler.Canvas.Brush.Color := PaperColor;
        imRuler.Canvas.FillRect(rect(Offset +RightMargin -2, 5, Offset +RightMargin +n, imRuler.Height -6));
        imRuler.Canvas.Brush.Color := MarginColor;
        imRuler.Canvas.FillRect(rect(Offset +RightMargin +n, 5, Offset +RightMargin +n +Tolerance, imRuler.Height -6));
        DrawTicks(FALSE, Offset +LeftMargin +(RightMargin -LeftMargin) div WholeUnit *WholeUnit -WholeUnit, Offset +RightMargin +n);
        StartPos := RightMargin -((RightMargin -LeftMargin) mod WholeUnit);
        inc(RightMargin, n);
        DrawRightMarginNumbers(StartPos, RightMargin);
        TabState := FALSE;
        TabStart := Offset +LeftMargin;
        TabEnd := Offset +RightMargin
      end
    else
      begin
        if (RightMargin -MinimumPaperSpace +n < LeftMargin) or
           (RightIndent -MinimumPaperSpace +n < FirstLineIndent) or
           (RightIndent -MinimumPaperSpace +n < LeftIndent) then
          exit;
        DrawIndentLine(RightMargin);
        EraseIndents;
        imRuler.Canvas.Brush.Style := bsSolid;
        imRuler.Canvas.Brush.Color := MarginColor;
        imRuler.Canvas.FillRect(rect(Offset +RightMargin +n, 5, Offset +RightMargin +2, imRuler.Height -6));
        imRuler.Canvas.Brush.Color := PaperColor;
        imRuler.Canvas.FillRect(rect(Offset +RightMargin +n, 5, Offset +RightMargin +n -Tolerance, imRuler.Height -6));
        DrawTicks(FALSE, Offset +(RightMargin +n) div WholeUnit *WholeUnit, Offset +RightMargin +WholeUnit);
        EndPos := RightMargin +WholeUnit;
        inc(RightMargin, n);
        DrawRightMarginNumbers(RightMargin div WholeUnit *WholeUnit, EndPos);
        TabState := TRUE;
        TabStart := Offset +TabDistance -(RightMargin -LeftMargin) mod TabDistance +RightMargin;
        TabEnd := Offset +RightMargin
      end;
    inc(RightIndent, n);
    DrawTabTicks(TabState, TabStart, TabEnd);
    DrawIndentLine(RightMargin);
    DrawIndents;
    DragOffset := Offset +RightMargin
  end;

procedure TForm1.MoveFirstIndent(X: Integer);
  var
       n  : integer;

  begin
    n := round((X -DragOffset)/(SmallUnit *1.0)) *SmallUnit;
    if n = 0 then
      exit;
    if (FirstLineIndent +MinimumPaperSpace +n >RightMargin) or
       (FirstLineIndent +MinimumPaperspace +n >RightIndent) then
      exit;
    EraseIndents;
    if not GhostVisible then
      DrawGhosts;
    DrawIndentLine(FirstLineIndent);
    FirstLineIndent := FirstLineIndent +n;
    DrawIndentLine(FirstLineIndent);
    DrawIndents;
    DragOffset := Offset +FirstLineIndent
  end;

procedure TForm1.MoveLeftIndent(X: Integer);
  var
       Rm : integer;
       n  : integer;

  begin
    n := round((X -DragOffset)/(SmallUnit *1.0)) *SmallUnit;
    if n = 0 then
      exit;
    if RightIndent < RightMargin then
      Rm := RightIndent -MinimumPaperSpace -n
    else
      Rm := RightMargin -MinimumPaperSpace -n;
    if (FirstLineIndent >Rm) or
       (LeftIndent >Rm) then
      exit;
    EraseIndents;
    if not GhostVisible then
      DrawGhosts;
    DrawIndentLine(LeftIndent);
    inc(LeftIndent, n);
    HangingIndent := LeftIndent;
    inc(FirstLineIndent, n);
    DrawIndentLine(LeftIndent);
    DrawIndents;
    DragOffset := Offset +LeftIndent
  end;

procedure TForm1.MoveHangIndent(X: Integer);
  var
       Rm : integer;
       n  : integer;

  begin
    n := round((X -DragOffset)/(SmallUnit *1.0)) *SmallUnit;
    if n = 0 then
      exit;
    if RightIndent < RightMargin then
      Rm := RightIndent -MinimumPaperSpace -n
    else
      Rm := RightMargin -MinimumPaperSpace -n;
    if LeftIndent >Rm then
      exit;
    EraseIndents;
    if not GhostVisible then
      DrawGhosts;
    DrawIndentLine(LeftIndent);
    inc(LeftIndent, n);
    HangingIndent := LeftIndent;
    DrawIndentLine(LeftIndent);
    DrawIndents;
    DragOffset := Offset +LeftIndent
  end;

procedure TForm1.MoveRightIndent(X: Integer);
  var
       Lm : integer;
       n  : integer;

  begin
    n := round((X -DragOffset)/(SmallUnit *1.0)) *SmallUnit;
    if n = 0 then
      exit;
    Lm := LeftMargin;
    if FirstLineIndent >Lm then
      Lm := FirstLineIndent;
    if LeftIndent >Lm then
      Lm := LeftIndent;
    if Lm >RightIndent -MinimumPaperSpace +n then
      Exit;
    EraseIndents;
    if not GhostVisible then
      DrawGhosts;
    DrawIndentLine(RightIndent);
    inc(RightIndent, n);
    DrawIndentLine(RightIndent);
    DrawIndents;
    DragOffset := Offset +RightIndent
  end;

procedure TForm1.imRulerMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
  var
       r        : TRect;
       p1, p2   : TPoint;
       dt       : TDragObject;

  begin
    if not Dragging then
    begin
      r := rect(0, 5, 5, imRuler.Height -5);
      p1.X := X -(Offset +LeftMargin -2); p1.Y := Y;
      p2.X := X -(Offset +RightMargin -2); p2.y := Y;
      dt := CheckHints(X, Y);
      if dt in [doLeftMargin, doRightMargin] then
        imRuler.Cursor:= crSizeWE
      else
        imRuler.Cursor:= crDefault;
      imRuler.ShowHint := dt in [doLeftMargin, doRightMargin, doFirstIndent, doHangIndent, doLeftIndent, doRightIndent];
      exit
    end;
    imRuler.ShowHint := FALSE;
    case DragType of
      doLeftMargin  : MoveLeftMargin(X);
      doRightMargin : MoveRightMargin(X);
      doFirstIndent : MoveFirstIndent(X);
      doLeftIndent  : MoveLeftIndent(X);
      doHangIndent  : MoveHangIndent(X);
      doRightIndent : MoveRightIndent(X)
    end;
  end;

procedure TForm1.imRulerMouseUp(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
  begin
    Dragging := FALSE;
    CheckDragType;
    if GhostVisible then
      DrawGhosts;
    DragType := doNone
  end;

procedure TForm1.DrawIndentLine(IndentLine: integer);
  var
       Oldpen : HPEN;
       tp     : ppoint;

  begin
    tp := nil;
    Oldpen := SelectObject(PHDC, PenLine);
    MoveToEx(PHDC, IndentLine, 0, tp);
    LineTo(PHDC, IndentLine, Paper.Height);
    SelectObject(PHDC, Oldpen)
  end;

procedure TForm1.FormDestroy(Sender: TObject);
  begin
     DeleteObject(PenLine);
     ReleaseDC (0, PHDC);
     inherited
  end;

procedure TForm1.FormResize(Sender: TObject);
  begin
    Paper.Height := ClientHeight -imRuler.Height -4
  end;

end.
