{
***************************************************
* A binary compatible RIPEMD-160 implementation   *
* written by Dave Barton (davebarton@bigfoot.com) *
***************************************************
* 160bit hash                                     *
***************************************************
}
unit RMD160;

interface
uses
  Windows, SysUtils;

type
  TRMD160Context= record
    Hash: array[0..4] of DWord;
    Index: integer;
    LenHi, LenLo: integer;
    case integer of
      1: (Buf: array[0..63] of byte);
      2: (X: array[0..15] of DWord);
  end;
  TRMD160Digest= array[0..19] of byte;

function RMD160SelfTest: boolean;
procedure RMD160Init(var Context: TRMD160Context);
procedure RMD160Update(var Context: TRMD160Context; Buffer: pointer; Len: integer);
procedure RMD160Final(var Context: TRMD160Context; var Digest: TRMD160Digest);

//******************************************************************************
implementation

function RMD160SelfTest: boolean;
const
  s: string= '12345678901234567890123456789012345678901234567890123456789012345678901234567890';
  OutDigest: TRMD160Digest=
    ($9b,$75,$2e,$45,$57,$3d,$4b,$39,$f4,$db,$d3,$32,$3c,$ab,$82,$bf,$63,$32,$6b,$fb);
var
  Context: TRMD160Context;
  Digest: TRMD160Digest;
begin
  RMD160Init(Context);
  RMD160Update(Context,@s[1],length(s));
  RMD160Final(Context,Digest);
  if CompareMem(@Digest,@OutDigest,Sizeof(Digest)) then
    Result:= true
  else
    Result:= false;
end;

//******************************************************************************
procedure RMD160Init(var Context: TRMD160Context);
begin
  Context.Hash[0]:= $67452301;
  Context.Hash[1]:= $efcdab89;
  Context.Hash[2]:= $98badcfe;
  Context.Hash[3]:= $10325476;
  Context.Hash[4]:= $c3d2e1f0;
  Context.Index:= 0;
  Context.LenHi:= 0;
  Context.LenLo:= 0;
end;

//******************************************************************************
function ROL(x: DWord; n: DWord): DWord; assembler;
asm
  mov ecx,n
  rol &x,cl
end;

//******************************************************************************
function F1(x, y, z: DWord): DWord;
begin
  Result:= x xor y xor z;
end;
function F2(x, y, z: DWord): DWord;
begin
  Result:= (x and y) or ((x xor $FFFFFFFF) and z);
end;
function F3(x, y, z: DWord): DWord;
begin
  Result:= (x or (y xor $FFFFFFFF)) xor z;
end;
function F4(x, y,  z: DWord): DWord;
begin
  Result:= (x and z) or (y and (z xor $FFFFFFFF));
end;
function F5(x, y, z: DWord): DWord;
begin
  Result:= x xor (y or (z xor $FFFFFFFF));
end;

//******************************************************************************
procedure FF1(var a, b, c: DWord; d, e, x, s: DWord);
begin
  a:= a + F1(b, c, d) + x;
  a:= ROL(a,s) + e;
  c:= ROL(c,10);
end;
procedure FF2(var a, b, c: DWord; d, e, x, s: DWord);
begin
  a:= a + F2(b, c, d) + x + $5a827999;
  a:= ROL(a,s) + e;
  c:= ROL(c,10);
end;
procedure FF3(var a, b, c: DWord; d, e, x, s: DWord);
begin
  a:= a + F3(b, c, d) + x + $6ed9eba1;
  a:= ROL(a,s) + e;
  c:= ROL(c,10);
end;
procedure FF4(var a, b, c: DWord; d, e, x, s: DWord);
begin
  a:= a + F4(b, c, d) + x + $8f1bbcdc;
  a:= ROL(a,s) + e;
  c:= ROL(c,10);
end;
procedure FF5(var a, b, c: DWord; d, e, x, s: DWord);
begin
  a:= a + F5(b, c, d) + x + $a953fd4e;
  a:= ROL(a,s) + e;
  c:= ROL(c,10);
end;
procedure FFF1(var a, b, c: DWord; d, e, x, s: DWord);
begin
  a:= a + F1(b, c, d) + x;
  a:= ROL(a,s) + e;
  c:= ROL(c,10);
end;
procedure FFF2(var a, b, c: DWord; d, e, x, s: DWord);
begin
  a:= a + F2(b, c, d) + x + $7a6d76e9;
  a:= ROL(a,s) + e;
  c:= ROL(c,10);
end;
procedure FFF3(var a, b, c: DWord; d, e, x, s: DWord);
begin
  a:= a + F3(b, c, d) + x + $6d703ef3;
  a:= ROL(a,s) + e;
  c:= ROL(c,10);
end;
procedure FFF4(var a, b, c: DWord; d, e, x, s: DWord);
begin
  a:= a + F4(b, c, d) + x + $5c4dd124;
  a:= ROL(a,s) + e;
  c:= ROL(c,10);
end;
procedure FFF5(var a, b, c: DWord; d, e, x, s: DWord);
begin
  a:= a + F5(b, c, d) + x + $50a28be6;
  a:= ROL(a,s) + e;
  c:= ROL(c,10);
end;

//******************************************************************************
procedure RMD160Compress(var Context: TRMD160Context);
var
  aa, bb, cc, dd, ee, aaa, bbb, ccc, ddd, eee: DWord;
begin
  aa:= Context.Hash[0];
  aaa:= Context.Hash[0];
  bb:= Context.Hash[1];
  bbb:= Context.Hash[1];
  cc:= Context.Hash[2];
  ccc:= Context.Hash[2];
  dd:= Context.Hash[3];
  ddd:= Context.Hash[3];
  ee:= Context.Hash[4];
  eee:= Context.Hash[4];
  with Context do
  begin
    FF1(aa, bb, cc, dd, ee, X[ 0], 11);
    FF1(ee, aa, bb, cc, dd, X[ 1], 14);
    FF1(dd, ee, aa, bb, cc, X[ 2], 15);
    FF1(cc, dd, ee, aa, bb, X[ 3], 12);
    FF1(bb, cc, dd, ee, aa, X[ 4],  5);
    FF1(aa, bb, cc, dd, ee, X[ 5],  8);
    FF1(ee, aa, bb, cc, dd, X[ 6],  7);
    FF1(dd, ee, aa, bb, cc, X[ 7],  9);
    FF1(cc, dd, ee, aa, bb, X[ 8], 11);
    FF1(bb, cc, dd, ee, aa, X[ 9], 13);
    FF1(aa, bb, cc, dd, ee, X[10], 14);
    FF1(ee, aa, bb, cc, dd, X[11], 15);
    FF1(dd, ee, aa, bb, cc, X[12],  6);
    FF1(cc, dd, ee, aa, bb, X[13],  7);
    FF1(bb, cc, dd, ee, aa, X[14],  9);
    FF1(aa, bb, cc, dd, ee, X[15],  8);

    FF2(ee, aa, bb, cc, dd, X[ 7],  7);
    FF2(dd, ee, aa, bb, cc, X[ 4],  6);
    FF2(cc, dd, ee, aa, bb, X[13],  8);
    FF2(bb, cc, dd, ee, aa, X[ 1], 13);
    FF2(aa, bb, cc, dd, ee, X[10], 11);
    FF2(ee, aa, bb, cc, dd, X[ 6],  9);
    FF2(dd, ee, aa, bb, cc, X[15],  7);
    FF2(cc, dd, ee, aa, bb, X[ 3], 15);
    FF2(bb, cc, dd, ee, aa, X[12],  7);
    FF2(aa, bb, cc, dd, ee, X[ 0], 12);
    FF2(ee, aa, bb, cc, dd, X[ 9], 15);
    FF2(dd, ee, aa, bb, cc, X[ 5],  9);
    FF2(cc, dd, ee, aa, bb, X[ 2], 11);
    FF2(bb, cc, dd, ee, aa, X[14],  7);
    FF2(aa, bb, cc, dd, ee, X[11], 13);
    FF2(ee, aa, bb, cc, dd, X[ 8], 12);

    FF3(dd, ee, aa, bb, cc, X[ 3], 11);
    FF3(cc, dd, ee, aa, bb, X[10], 13);
    FF3(bb, cc, dd, ee, aa, X[14],  6);
    FF3(aa, bb, cc, dd, ee, X[ 4],  7);
    FF3(ee, aa, bb, cc, dd, X[ 9], 14);
    FF3(dd, ee, aa, bb, cc, X[15],  9);
    FF3(cc, dd, ee, aa, bb, X[ 8], 13);
    FF3(bb, cc, dd, ee, aa, X[ 1], 15);
    FF3(aa, bb, cc, dd, ee, X[ 2], 14);
    FF3(ee, aa, bb, cc, dd, X[ 7],  8);
    FF3(dd, ee, aa, bb, cc, X[ 0], 13);
    FF3(cc, dd, ee, aa, bb, X[ 6],  6);
    FF3(bb, cc, dd, ee, aa, X[13],  5);
    FF3(aa, bb, cc, dd, ee, X[11], 12);
    FF3(ee, aa, bb, cc, dd, X[ 5],  7);
    FF3(dd, ee, aa, bb, cc, X[12],  5);

    FF4(cc, dd, ee, aa, bb, X[ 1], 11);
    FF4(bb, cc, dd, ee, aa, X[ 9], 12);
    FF4(aa, bb, cc, dd, ee, X[11], 14);
    FF4(ee, aa, bb, cc, dd, X[10], 15);
    FF4(dd, ee, aa, bb, cc, X[ 0], 14);
    FF4(cc, dd, ee, aa, bb, X[ 8], 15);
    FF4(bb, cc, dd, ee, aa, X[12],  9);
    FF4(aa, bb, cc, dd, ee, X[ 4],  8);
    FF4(ee, aa, bb, cc, dd, X[13],  9);
    FF4(dd, ee, aa, bb, cc, X[ 3], 14);
    FF4(cc, dd, ee, aa, bb, X[ 7],  5);
    FF4(bb, cc, dd, ee, aa, X[15],  6);
    FF4(aa, bb, cc, dd, ee, X[14],  8);
    FF4(ee, aa, bb, cc, dd, X[ 5],  6);
    FF4(dd, ee, aa, bb, cc, X[ 6],  5);
    FF4(cc, dd, ee, aa, bb, X[ 2], 12);

    FF5(bb, cc, dd, ee, aa, X[ 4],  9);
    FF5(aa, bb, cc, dd, ee, X[ 0], 15);
    FF5(ee, aa, bb, cc, dd, X[ 5],  5);
    FF5(dd, ee, aa, bb, cc, X[ 9], 11);
    FF5(cc, dd, ee, aa, bb, X[ 7],  6);
    FF5(bb, cc, dd, ee, aa, X[12],  8);
    FF5(aa, bb, cc, dd, ee, X[ 2], 13);
    FF5(ee, aa, bb, cc, dd, X[10], 12);
    FF5(dd, ee, aa, bb, cc, X[14],  5);
    FF5(cc, dd, ee, aa, bb, X[ 1], 12);
    FF5(bb, cc, dd, ee, aa, X[ 3], 13);
    FF5(aa, bb, cc, dd, ee, X[ 8], 14);
    FF5(ee, aa, bb, cc, dd, X[11], 11);
    FF5(dd, ee, aa, bb, cc, X[ 6],  8);
    FF5(cc, dd, ee, aa, bb, X[15],  5);
    FF5(bb, cc, dd, ee, aa, X[13],  6);

    FFF5(aaa, bbb, ccc, ddd, eee, X[ 5],  8);
    FFF5(eee, aaa, bbb, ccc, ddd, X[14],  9);
    FFF5(ddd, eee, aaa, bbb, ccc, X[ 7],  9);
    FFF5(ccc, ddd, eee, aaa, bbb, X[ 0], 11);
    FFF5(bbb, ccc, ddd, eee, aaa, X[ 9], 13);
    FFF5(aaa, bbb, ccc, ddd, eee, X[ 2], 15);
    FFF5(eee, aaa, bbb, ccc, ddd, X[11], 15);
    FFF5(ddd, eee, aaa, bbb, ccc, X[ 4],  5);
    FFF5(ccc, ddd, eee, aaa, bbb, X[13],  7);
    FFF5(bbb, ccc, ddd, eee, aaa, X[ 6],  7);
    FFF5(aaa, bbb, ccc, ddd, eee, X[15],  8);
    FFF5(eee, aaa, bbb, ccc, ddd, X[ 8], 11);
    FFF5(ddd, eee, aaa, bbb, ccc, X[ 1], 14);
    FFF5(ccc, ddd, eee, aaa, bbb, X[10], 14);
    FFF5(bbb, ccc, ddd, eee, aaa, X[ 3], 12);
    FFF5(aaa, bbb, ccc, ddd, eee, X[12],  6);

    FFF4(eee, aaa, bbb, ccc, ddd, X[ 6],  9);
    FFF4(ddd, eee, aaa, bbb, ccc, X[11], 13);
    FFF4(ccc, ddd, eee, aaa, bbb, X[ 3], 15);
    FFF4(bbb, ccc, ddd, eee, aaa, X[ 7],  7);
    FFF4(aaa, bbb, ccc, ddd, eee, X[ 0], 12);
    FFF4(eee, aaa, bbb, ccc, ddd, X[13],  8);
    FFF4(ddd, eee, aaa, bbb, ccc, X[ 5],  9);
    FFF4(ccc, ddd, eee, aaa, bbb, X[10], 11);
    FFF4(bbb, ccc, ddd, eee, aaa, X[14],  7);
    FFF4(aaa, bbb, ccc, ddd, eee, X[15],  7);
    FFF4(eee, aaa, bbb, ccc, ddd, X[ 8], 12);
    FFF4(ddd, eee, aaa, bbb, ccc, X[12],  7);
    FFF4(ccc, ddd, eee, aaa, bbb, X[ 4],  6);
    FFF4(bbb, ccc, ddd, eee, aaa, X[ 9], 15);
    FFF4(aaa, bbb, ccc, ddd, eee, X[ 1], 13);
    FFF4(eee, aaa, bbb, ccc, ddd, X[ 2], 11);

    FFF3(ddd, eee, aaa, bbb, ccc, X[15],  9);
    FFF3(ccc, ddd, eee, aaa, bbb, X[ 5],  7);
    FFF3(bbb, ccc, ddd, eee, aaa, X[ 1], 15);
    FFF3(aaa, bbb, ccc, ddd, eee, X[ 3], 11);
    FFF3(eee, aaa, bbb, ccc, ddd, X[ 7],  8);
    FFF3(ddd, eee, aaa, bbb, ccc, X[14],  6);
    FFF3(ccc, ddd, eee, aaa, bbb, X[ 6],  6);
    FFF3(bbb, ccc, ddd, eee, aaa, X[ 9], 14);
    FFF3(aaa, bbb, ccc, ddd, eee, X[11], 12);
    FFF3(eee, aaa, bbb, ccc, ddd, X[ 8], 13);
    FFF3(ddd, eee, aaa, bbb, ccc, X[12],  5);
    FFF3(ccc, ddd, eee, aaa, bbb, X[ 2], 14);
    FFF3(bbb, ccc, ddd, eee, aaa, X[10], 13);
    FFF3(aaa, bbb, ccc, ddd, eee, X[ 0], 13);
    FFF3(eee, aaa, bbb, ccc, ddd, X[ 4],  7);
    FFF3(ddd, eee, aaa, bbb, ccc, X[13],  5);

    FFF2(ccc, ddd, eee, aaa, bbb, X[ 8], 15);
    FFF2(bbb, ccc, ddd, eee, aaa, X[ 6],  5);
    FFF2(aaa, bbb, ccc, ddd, eee, X[ 4],  8);
    FFF2(eee, aaa, bbb, ccc, ddd, X[ 1], 11);
    FFF2(ddd, eee, aaa, bbb, ccc, X[ 3], 14);
    FFF2(ccc, ddd, eee, aaa, bbb, X[11], 14);
    FFF2(bbb, ccc, ddd, eee, aaa, X[15],  6);
    FFF2(aaa, bbb, ccc, ddd, eee, X[ 0], 14);
    FFF2(eee, aaa, bbb, ccc, ddd, X[ 5],  6);
    FFF2(ddd, eee, aaa, bbb, ccc, X[12],  9);
    FFF2(ccc, ddd, eee, aaa, bbb, X[ 2], 12);
    FFF2(bbb, ccc, ddd, eee, aaa, X[13],  9);
    FFF2(aaa, bbb, ccc, ddd, eee, X[ 9], 12);
    FFF2(eee, aaa, bbb, ccc, ddd, X[ 7],  5);
    FFF2(ddd, eee, aaa, bbb, ccc, X[10], 15);
    FFF2(ccc, ddd, eee, aaa, bbb, X[14],  8);

    FFF1(bbb, ccc, ddd, eee, aaa, X[12] ,  8);
    FFF1(aaa, bbb, ccc, ddd, eee, X[15] ,  5);
    FFF1(eee, aaa, bbb, ccc, ddd, X[10] , 12);
    FFF1(ddd, eee, aaa, bbb, ccc, X[ 4] ,  9);
    FFF1(ccc, ddd, eee, aaa, bbb, X[ 1] , 12);
    FFF1(bbb, ccc, ddd, eee, aaa, X[ 5] ,  5);
    FFF1(aaa, bbb, ccc, ddd, eee, X[ 8] , 14);
    FFF1(eee, aaa, bbb, ccc, ddd, X[ 7] ,  6);
    FFF1(ddd, eee, aaa, bbb, ccc, X[ 6] ,  8);
    FFF1(ccc, ddd, eee, aaa, bbb, X[ 2] , 13);
    FFF1(bbb, ccc, ddd, eee, aaa, X[13] ,  6);
    FFF1(aaa, bbb, ccc, ddd, eee, X[14] ,  5);
    FFF1(eee, aaa, bbb, ccc, ddd, X[ 0] , 15);
    FFF1(ddd, eee, aaa, bbb, ccc, X[ 3] , 13);
    FFF1(ccc, ddd, eee, aaa, bbb, X[ 9] , 11);
    FFF1(bbb, ccc, ddd, eee, aaa, X[11] , 11);
  end;
  ddd:= ddd + cc + Context.Hash[1];
  Context.Hash[1]:= Context.Hash[2] + dd + eee;
  Context.Hash[2]:= Context.Hash[3] + ee + aaa;
  Context.Hash[3]:= Context.Hash[4] + aa + bbb;
  Context.Hash[4]:= Context.Hash[0] + bb + ccc;
  Context.Hash[0]:= ddd;
end;

//******************************************************************************
procedure RMD160UpdateLen(var Context: TRMD160Context; Len: integer);
var
  i, k: integer;
begin
  for k:= 0 to 7 do
  begin
    i:= Context.LenLo;
    Inc(Context.LenLo,Len);
    if Context.LenLo< i then
      Inc(Context.LenHi);
  end;
end;

//******************************************************************************
procedure RMD160Update(var Context: TRMD160Context; Buffer: pointer; Len: integer);
begin
  RMD160UpdateLen(Context,Len);
  while Len> 0 do
  begin
    Context.Buf[Context.Index]:= PByte(Buffer)^;
    Inc(PByte(Buffer));
    Inc(Context.Index);
    Dec(Len);
    if Context.Index= 64 then
    begin
      Context.Index:= 0;
      RMD160Compress(Context);
    end;
  end;
end;

//******************************************************************************
procedure RMD160Final(var Context: TRMD160Context; var Digest: TRMD160Digest);
var
  i, j, len: integer;
  mask: byte;
  xl: array[0..15] of DWord;
begin
  FillChar(xl,Sizeof(xl),0);
  len:= ((Context.LenLo and 511)+7) div 8;
  if (Context.LenLo and 7)<> 0 then
    mask:= 1 shl (Context.LenLo and 7)
  else
    mask:= $ff;
  j:= 0;
  for i:= 0 to len-1 do
  begin
    if i= (Len-1) then
      xl[i shr 2]:= xl[i shr 2] xor ((Context.Buf[j] and Mask) shl (8*(i and 3)))
    else
      xl[i shr 2]:= xl[i shr 2] xor (Context.Buf[j] shl (8*(i and 3)));
    Inc(j);
  end;
  xl[(Context.LenLo shr 5) and 15]:= xl[(Context.LenLo shr 5) and 15] xor (DWord(1) shl (8*((Context.LenLo shr 3) and 3)+7-(Context.LenLo and 7)));
  Move(xl,Context.X,Sizeof(xl));
  if (Context.LenLo and 511)> 447 then
  begin
    RMD160Compress(Context);
    FillChar(Context.X,Sizeof(Context.X),0);
  end;
  Context.X[14]:= Context.LenLo;
  Context.X[15]:= Context.LenHi;
  RMD160Compress(Context);
  Move(Context.Hash,Digest,Sizeof(Digest));
  FillChar(Context,Sizeof(Context),$FF);
end;

end.
