unit rjMime;
{
Unit:           rjMime
Version:        1.01
Last Modified:  09. April 2000
Author:         Ralf Junker <ralfjunker@gmx.de>

Description:    Ligtening fast Mime (Base64) Encoding and Decoding routines.

Legal:          This software is provided 'as-is', without any express or
                implied warranty. In no event will the author be held liable
                for any  damages arising from the use of this software.

                Permission is granted to anyone to use this software for any
                purpose, including commercial applications, and to alter it
                and redistribute it freely, subject to the following
                restrictions:

                1. The origin of this software must not be misrepresented,
                    you must not claim that you wrote the original software.
                    If you use this software in a product, an acknowledgment
                    in the product documentation would be appreciated but is
                    not required.

                2. Altered source versions must be plainly marked as such, and
                   must not be misrepresented as being the original software.

                3. This notice may not be removed or altered from any source
                   distribution.

History:        Version 1.00

17.01.2000      Initial Public Release

                Version 1.01

09.04.2000      Fixed a bug in MimeDecodeTable which caused wrong results
                decoding binary files.

Copyright (c) 2000 Ralf Junker
}

interface

{.$DEFINE RangChecking}
{ RangeChecking defines if an EMime exception will be raised if either
  InBuf and OutBuf are nil or if InBytes is not greater than zero.
  If defined, unit SysUtils will be included. }

{$IFDEF RangeChecking}
uses SysUtils;
{$ENDIF}

{$IFDEF RangeChecking}
resourcestring
 SInBufNil = 'MIME (Base64) Conversion: Input Buffer must not be NIL.';
 SOutBufNil = 'MIME (Base64) Conversion: Output Buffer must not be NIL.';
 SInBytesZero = 'MIME (Base64) Conversion: InBytes must be greater than zero.';
{$ENDIF}
 
type
{$IFDEF RangeChecking}
 EMime = class (Exception);
{$ENDIF}
 
 PByte4 = ^TByte4;
 TByte4 = packed record
  a: Byte;
  b: Byte;
  c: Byte;
  d: Byte;
 end;
 
 PByte3 = ^TByte3;
 TByte3 = packed record
  a: Byte;
  b: Byte;
  c: Byte;
 end;
 
const
 MimeEncodeTable: array[0..63] of Byte = (
  065, 066, 067, 068, 069, 070, 071, 072, // 00 - 07
  073, 074, 075, 076, 077, 078, 079, 080, // 08 - 15
  081, 082, 083, 084, 085, 086, 087, 088, // 16 - 23
  089, 090, 097, 098, 099, 100, 101, 102, // 24 - 31
  103, 104, 105, 106, 107, 108, 109, 110, // 32 - 39
  111, 112, 113, 114, 115, 116, 117, 118, // 40 - 47
  119, 120, 121, 122, 048, 049, 050, 051, // 48 - 55
  052, 053, 054, 055, 056, 057, 043, 047); // 56 - 63
 
 MimeDecodeTable: array[Byte] of Cardinal = (
  255, 255, 255, 255, 255, 255, 255, 255, //  00 -  07
  255, 255, 255, 255, 255, 255, 255, 255, //  08 -  15
  255, 255, 255, 255, 255, 255, 255, 255, //  16 -  23
  255, 255, 255, 255, 255, 255, 255, 255, //  24 -  31
  255, 255, 255, 255, 255, 255, 255, 255, //  32 -  39
  255, 255, 255, 062, 255, 255, 255, 063, //  40 -  47
  052, 053, 054, 055, 056, 057, 058, 059, //  48 -  55
  060, 061, 255, 255, 255, 255, 255, 255, //  56 -  63
  255, 000, 001, 002, 003, 004, 005, 006, //  64 -  71
  007, 008, 009, 010, 011, 012, 013, 014, //  72 -  79
  015, 016, 017, 018, 019, 020, 021, 022, //  80 -  87
  023, 024, 025, 255, 255, 255, 255, 255, //  88 -  95
  255, 026, 027, 028, 029, 030, 031, 032, //  96 - 103
  033, 034, 035, 036, 037, 038, 039, 040, // 104 - 111
  041, 042, 043, 044, 045, 046, 047, 048, // 112 - 119
  049, 050, 051, 255, 255, 255, 255, 255, // 120 - 127
  255, 255, 255, 255, 255, 255, 255, 255,
  255, 255, 255, 255, 255, 255, 255, 255,
  255, 255, 255, 255, 255, 255, 255, 255,
  255, 255, 255, 255, 255, 255, 255, 255,
  255, 255, 255, 255, 255, 255, 255, 255,
  255, 255, 255, 255, 255, 255, 255, 255,
  255, 255, 255, 255, 255, 255, 255, 255,
  255, 255, 255, 255, 255, 255, 255, 255,
  255, 255, 255, 255, 255, 255, 255, 255,
  255, 255, 255, 255, 255, 255, 255, 255,
  255, 255, 255, 255, 255, 255, 255, 255,
  255, 255, 255, 255, 255, 255, 255, 255,
  255, 255, 255, 255, 255, 255, 255, 255,
  255, 255, 255, 255, 255, 255, 255, 255,
  255, 255, 255, 255, 255, 255, 255, 255,
  255, 255, 255, 255, 255, 255, 255, 255);
 
function MimeEncodedSize (const i: Integer): Integer;
{ Calculates the output size of i MimeEncoded bytes. Use for MimeEncode only. }

procedure MimeEncode (const InBuf: Pointer; const OutBuf: Pointer; const InBytes: Integer);
{ CAUTTION: OutBuf must have enough memory allocated to take all encoded output.
  MimeEncodedSize (InBytes) calculates this amount in bytes. MimeEncode will
  then fill all OutBuf, so there is no OutBytes with this procedure.
  Preallocating all memory at once (as required by MimeEncode)
  avoids the time-cosuming process of reallocation. }

procedure MimeDecode (const InBuf: Pointer; const InBytes: Integer; const OutBuf: Pointer; out OutBytes: Integer);
{ CAUTTION: OutBuf must have enough memory allocated to take all output.
  Its size should be at least as large as InBytes bytes.
  There is therefore no need to calculate OutBuf size using MimeEncodedSize.
  OutBytes then returns acutal number of bytes written to OutBuf.
  Preallocating all memory at once (as required by MimeDecode)
  avoids the time-cosuming process of reallocation. After calling
  MimeDecode, simply cut the allocated memory down to OutBytes,
  i.e. SetLength (OutString, OutBytes). }

implementation

{ **************************************************************************** }

function MimeEncodedSize (const i: Integer): Integer;
begin
 Result := ((i + 2) div 3) * 4;
end;

{ ********** }

procedure MimeEncode (const InBuf: Pointer; const OutBuf: Pointer; const InBytes: Integer);
var
 b: Cardinal;
 InMax3: Integer;
 pIn, PInLimit: ^Byte;
 pOut: PByte4;
begin
{$IFDEF RangeChecking}
 if InBuf = nil then raise EMime.Create (SInBufNil);
 if OutBuf = nil then raise EMime.Create (SOutBufNil);
 if InBytes = 0 then raise EMime.Create (SInBytesZero);
{$ENDIF}
 InMax3 := InBytes div 3 * 3;
 
 pIn := InBuf;
 
 PInLimit := pIn;
 Inc (PInLimit, InMax3);
 
 pOut := OutBuf;
 
 while pIn <> PInLimit do
  begin
   b := b or pIn^; // Read 3 bytes from InBuf.
   Inc (pIn);
   b := b shl 8;
   b := b or pIn^;
   Inc (pIn);
   b := b shl 8;
   b := b or pIn^;
   Inc (pIn);

   // Write 4 bytes to OutBuf (in reverse order).
   pOut.d := MimeEncodeTable[b and $3F];
   b := b shr 6;
   pOut.c := MimeEncodeTable[b and $3F];
   b := b shr 6;
   pOut.b := MimeEncodeTable[b and $3F];
   b := b shr 6;
   pOut.a := MimeEncodeTable[b and $3F];
   b := b shr 6;

   Inc (pOut);
  end;
 
 InMax3 := InBytes - InMax3;
 
 if InMax3 <> 0 then // At this point InMax3 can be either 0, 1, or 2.
  begin // At this point InMax3 equals either 1 or 2.
   b := 0;
   if InMax3 = 1 then
    begin
     b := b or pIn^;

     b := b shl 4;

     pOut.b := MimeEncodeTable[b and $3F];
     b := b shr 6;
     pOut.a := MimeEncodeTable[b and $3F];

     pOut.c := Ord ('='); // Fill remaining 2 bytes.
     pOut.d := Ord ('=');
    end
   else // At this point InMax3 equals 2.
    begin
     b := b or pIn^;
     Inc (pIn);
     b := b shl 8;
     b := b or pIn^;

     b := b shl 2;

     pOut.c := MimeEncodeTable[b and $3F];
     b := b shr 6;
     pOut.b := MimeEncodeTable[b and $3F];
     b := b shr 6;
     pOut.a := MimeEncodeTable[b and $3F];

     pOut.d := Ord ('='); // Fill remaining byte.
    end;
  end;
end;

{ ********** }

procedure MimeDecode (const InBuf: Pointer; const InBytes: Integer; const OutBuf: Pointer; out OutBytes: Integer);
var
 b, i, j: Cardinal;
 pIn, PInLimit: ^Byte;
 pOut: PByte3;
begin
{$IFDEF RangeChecking}
 if InBuf = nil then raise EMime.Create (SInBufNil);
 if OutBuf = nil then raise EMime.Create (SOutBufNil);
 if InBytes = 0 then raise EMime.Create (SInBytesZero);
{$ENDIF}
 
 pIn := InBuf;
 Integer (PInLimit) := Integer (pIn) + InBytes;
 pOut := OutBuf;
 
 j := 4;
 b := 0;
 while pIn <> PInLimit do
  begin
   i := MimeDecodeTable[pIn^]; // Read from InBuf.
   if i <> $FF then
    begin
     b := b shl 6;
     b := b or i;
     Dec (j);
     if j = 0 then // 4 bytes read from InBuf?
      begin
       pOut.c := Byte (b); // Write 3 bytes to OutBuf (in reverse order).
       b := b shr 8;
       pOut.b := Byte (b);
       b := b shr 8;
       pOut.a := Byte (b);
       b := b shr 8;
       Inc (pOut);

       j := 4;
      end;
    end;
   Inc (pIn);
  end;
 
 OutBytes := Cardinal (pOut) - Cardinal (OutBuf);
 
 { At this point j can be either 0, 1, 2, or 3. }
 if j = 2 then
  begin // Write 2 Bytes.
   b := b shl 4;
   pOut.b := Byte (b);
   b := b shr 8;
   pOut.a := Byte (b);
   Inc (OutBytes);
  end
 else
  if j = 1 then
   begin // Write 3 Bytes.
    b := b shl 6;
    pOut.c := Byte (b);
    b := b shr 8;
    pOut.b := Byte (b);
    b := b shr 8;
    pOut.a := Byte (b);
    Inc (OutBytes, 2);
   end;
end;

end.

