unit main;
{ Delphi 2.0 sample to demonstrate run-time establishment of referential
  integrity (RI) between two Paradox tables.  As a bonus it also shows how
  to append fields to a table at run time.

 The code here works but isn't necessarily the best approach.  I had only a
 couple of hours available and very little to go on besides ..\doc\bde.int
 and ..\bde\bde32.hlp.

 To illustrate cascading modify I've allowed parent key (originally ftAutoinc)
 to be ftInteger instead, assigning values manually (see pkeys[]).  This
 enables you to change the parent values and see cascading modify in action.

 The project creates its own local alias and tables, so just build and run.
 Hope you find it helpful.  See also AUTOINC.ZIP at Delphi Super Page.

 Rev 1.1, 19 April 1997.  Freeware by Grant Walker, gw@enternet.com.au
}

interface
uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Grids, DBGrids, DB, DBTables, StdCtrls, DBCtrls, ExtCtrls;

type
  TMainForm = class(TForm)
    ParentTable:  TTable;
    ChildTable:   TTable;
    DataSource1:  TDataSource;
    DataSource2:  TDataSource;
    RIDatabase: TDatabase;
    ParentGrid:   TDBGrid;
    ChildGrid:    TDBGrid;
    DBMemo:       TDBMemo;
    UseAutoinc:   TCheckBox;
    CreateButton: TButton;
    Bevel:        TBevel;
    RIButton:     TButton;
    AppendButton: TButton;
    Comment:      TLabel;
    procedure FormCreate(Sender: TObject);
    procedure CreateButtonClick(Sender: TObject);
    procedure RIButtonClick(Sender: TObject);
    procedure ParentTableAfterPost(DataSet: TDataSet);
    procedure AppendButtonClick(Sender: TObject);
  private
    function  Qualify(const s:string):string;
    procedure RemoveOrphans;
    procedure ApplyRI;
  end;

var
  MainForm: TMainForm;


implementation
uses BDE, orphan;

{$R *.DFM}

//--------------------------------------------------------------------------
// ** This is a quick hack to allow demo to run without setting up an alias.
// In general putting tables in the same directory as the EXE is a Bad Idea.

procedure TMainForm.FormCreate(Sender: TObject);
begin
  if not Session.IsAlias('RIDEMO') then
    Session.AddStandardAlias('RIDEMO',ExtractFilePath(Application.ExeName),'PARADOX');
  DBMemo.DataField := 'Quote';                                  //**
  Comment.Caption := 'Press Create to make two sample tables in the current directory.';
end;

function  TMainForm.Qualify(const s:string):string;
begin
  Result := ExtractFilePath(Application.ExeName)+s;
end;                                  //**


//--------------------------------------------------------------------------
// Create two tables with sample data but without referential integrity
//--------------------------------------------------------------------------
//
procedure TMainForm.CreateButtonClick(Sender: TObject);
var
  pkeys:array[1..5] of Variant;
  s:TStringList;
  i:integer;
begin
  s := TStringList.Create;
  Screen.Cursor := crHourGlass;
  try
    // It won't let you recreate an existing master table that has dependents:
    // you must first relax the RI or delete the children.  I do the latter.
    ParentTable.Close;
    ChildTable.Close;
    if FileExists(Qualify(ChildTable.TableName)) then ChildTable.DeleteTable;
    with ParentTable do
      begin
        FieldDefs.Clear;
        if UseAutoinc.Checked then
          begin
            FieldDefs.Add('Parent#', ftAutoinc, 0, False);  // it's best to leave ftAutoinc False
            for i := 1 to 5 do pkeys[i] := Null;            // don't assign autoinc key values
          end
        else
          begin
            FieldDefs.Add('Parent#', ftInteger, 0, True);
            for i := 1 to 5 do pkeys[i] := i;               // sample integer key values
          end;
        FieldDefs.Add('Name', ftString, 10, True);
        FieldDefs.Add('Age', ftSmallInt, 0, False);
        FieldDefs.Add('Quote', ftMemo, 30, False);
        IndexDefs.Clear;
        IndexDefs.Add('ParentPK','Parent#',[ixPrimary]);
        IndexDefs.Add('ByName','Name',[ixCaseInsensitive]); // for example (unused here)
        CreateTable;
        Open;
        DisableControls;
        try
          // Ok, we've made the parent table, now let's add some sample data.
          AppendRecord([pkeys[1],'Marge',37,'Bart!!'#13#10'Go to your room!']);
          //------ other ways of doing the same thing... ------------
          Append;
          if not UseAutoinc.Checked then Fields[0].Value := pkeys[2];
          Fields[1].AsString := 'Homer';
          Fields[2].Value := 41;
          s.Clear; s.Add('Mmmmm donuts!'); s.Add(''); s.Add('Almost as tasty as Pork Rinds with Duff Beer...');
          FieldByName('Quote').Assign(s);
          Post;
          //----------------------------------------------------------
          AppendRecord([pKeys[3],'Barney',Null,'>> Blaarrrp <<']);
          AppendRecord([pKeys[4],'Mr Burns',92,'Excellent!']);
          AppendRecord([pKeys[5],'Smithers',34,'Right away, Mr Burns.']);
          First;
        finally
          EnableControls;
        end;  
      end;
    with ChildTable do
      begin
        Close;
        FieldDefs.Clear;
        FieldDefs.Add('Child#', ftAutoinc, 0, False);
        FieldDefs.Add('Name', ftString, 10, True);
        // In following line, ParentFK is foreign key -> Parent.db
        // It must be ftInteger (not ftSmallInt) otherwise it won't be
        // compatible with Parent# (ftAutoinc) when RI is applied.
        // I've made ParentFK value optional here but in some applications
        // you'll want it to be a required field.
        FieldDefs.Add('ParentFK', ftInteger, 0, False);
        IndexDefs.Clear;
        IndexDefs.Add('ChildPK','Child#',[ixPrimary]);
        // RI will not work unless you have an index on the foreign key (ie. ParentFK).
        // In TIndexOptions below I would have used [] but it doesn't like it, so I'm using ixCaseInsensitive as a workaround.
        IndexDefs.Add('ByParent','ParentFK',[ixCaseInsensitive]);
        CreateTable;
        Open;
        AppendRecord([Null,'Bart',2]);   // use Null to skip over the (readonly) ftAutoinc field
        AppendRecord([Null,'Lisa',1]);
        AppendRecord([Null,'Gates',666]);  // a couple of children don't have parents
        AppendRecord([Null,'Maggie',1]);
        AppendRecord([Null,'Bond',007]);
        First;
      end;
    // The rest is aesthetics...
    CreateButton.Enabled := False;
    UseAutoinc.Enabled := False;
    AppendButton.Enabled := True;
    RIButton.Enabled := True;
    Comment.Caption := 'Top table is parent (master), bottom is child (dependant).'#13#10#13#10'ParentFK (foreign key) RI is not yet enforced, so you can edit ParentFK values freely.';
    ActiveControl := ParentGrid;
  finally
    s.Free;
    Screen.Cursor := crDefault;
  end;
end;

//--------------------------------------------------------------------------
// Establish referential integrity
//--------------------------------------------------------------------------
{ The DbiDoRestructure call applying the RI link fails with "master record missing"
 if orphans exist, even if I give it a key violation table name.  I haven't
 worked out whether this is my fault or a limitation of the BDE; perhaps
 Paradox manually extracts orphans before restructuring.  That's what I'm
 having to do here - I first call RemoveOrphans(), followed by ApplyRI().

 It's not all bad news, however.  Paradox's automatically generated KEYVIOL
 duplicates the child structure completely, right down to the ftAutoInc
 Child#.  This means that the orphans' original Child# values get lost.
 By writing RemoveOrphans myself I can retain these values, which can
 be very handy if preparing a text report of the extracted child details
 and the children in turn have detail records.
}
procedure TMainForm.RIButtonClick(Sender: TObject);
begin
  RemoveOrphans;
  ApplyRI;
  // The rest is aesthetics...
  RIButton.Enabled := False;
  Comment.Caption := 'If you try changing ParentFK now it will only accept valid parent entries.'#13#10#13#10;
  with Comment do if UseAutoinc.Checked
    then Caption := Caption + 'You can''t modify Parent# as it is ftAutoinc, which is read-only.'
    else Caption := Caption + 'If you modify Parent# you will observe cascaded modify in child.'#13#10#13#10'Eg. change Parent# 1 to 6.';
end;


procedure TMainForm.RemoveOrphans;
// The following removes any existing children which would break RI.
// Once RI has been applied new orphans cannot be added.
var
  bm:TBookMark;
  cf,i:integer;
  orphans:boolean;
  OrphanTable:TTable;
begin
  Screen.Cursor := crHourglass;
  orphans := false;                  // only create keyviol table if orphans are found
  try
    ParentTable.DisableControls;
    ParentTable.Open;
    bm := ParentTable.GetBookMark;   // preserve parent location while deleting children
    with ChildTable do
      begin
        DisableControls;
        Open;
        First;
        cf := FindField('ParentFK').Index;  // for efficiency
        while not EOF do
          if ParentTable.Locate('Parent#',Fields[cf].AsInteger,[]) then
            Next   // if parent exists then child is ok
          else     // otherwise we extract them into an orphan table
            begin
              if not orphans then
                begin
                  OrphanTable := TTable.Create(Self);
                  with OrphanTable do
                    begin
                      DatabaseName := ChildTable.DatabaseName;
                      TableName := 'ORPHAN.DB';
                    //  FieldDefs.Assign(ChildTable.FieldDefs);  // << this is what Paradox does
                      //------ replaced by --------
                      FieldDefs.Add('Child#', ftInteger, 0, False);  // change ftAutoinc -> ftInteger to retain original value
                      for i := 1 to ChildTable.FieldCount-1 do
                        with ChildTable.FieldDefs[i] do
                          FieldDefs.Add(Name, DataType, Size, Required);
                      //---------------------------
                      CreateTable;
                      Open;
                    end;
                  orphans := true;
                end;
              // copying record between tables must be done field-by-field AFAIK
              OrphanTable.Append;
              for i := 0 to FieldCount-1 do
                OrphanTable.Fields[i].Assign(Fields[i]);
              OrphanTable.Post;
              Delete;   // remove orphan from child table
            end;
      end;
  finally
    ParentTable.GotoBookMark(bm);
    ParentTable.FreeBookMark(bm);
    ParentTable.EnableControls;
    ChildTable.EnableControls;
    Screen.Cursor := crDefault;
    if orphans then
      begin
        // if you were generating a text report on the orphans you'd
        // probably do it here using OrphanTable before freeing it.
        with OrphanForm do
          begin
            DataSource.Dataset := OrphanTable;
            ShowModal;
            DataSource.Dataset := nil;
          end;
        OrphanTable.Close;
        OrphanTable.Free;
      end;
  end;
end;


procedure TMainForm.ApplyRI;
{ If writing your own BDE calls you may find isolated discrepancies in field
  names between BDE32.HLP and BDE.INT.  Remember that BDE originated as a C
  API with case-sensitive identifiers; Delphi has occasionally had to rename
  identifiers slightly to avoid conflicts.  Use the HLP for quick reference
  but for definitive identifier names you should refer to the INT header. }
var
  tabledesc:  CRTblDesc;
  szProblem:  DbiName;
  singleop:   CROpType;
  singleRI:   RIntDesc;
  szKeyViol:  DbiName;
begin
  ParentTable.Close;  // both must be closed in this case
  ChildTable.Close;
  try
    // When establishing referential integrity, you apply it to the child
    // and refer to the master/parent in the process.  You can't do it
    // the other way - restructuring the parent and referring to the child.
    FillChar(tabledesc,sizeof(CRTblDesc),#0);
    with tabledesc do
      begin
        StrPCopy(szTblName,ChildTable.TableName);
        // StrCopy(szTblType,szParadox);  << not needed if .db suffix given
        // bPack := True;                 << optional
        iRintCount := 1;   // our RI spec contains a single entry only
        singleop := crADD;  // add a new dependency
        pecrRintOp := @singleop;
        FillChar(singleRI,sizeof(singleRI),#0);
        with singleRI do
          begin
            iRintNum := 1;  // AFAIK these must be unique for each RI per child
            StrPCopy(szRIntName,'RefParent');  // give the RI a name
            eType := rintDEPENDENT;            // we are the child
            StrPCopy(szTblName,ParentTable.TableName); // here is the master
            eModOp := rintCASCADE;             // use cascading modify
            eDelOp := rintRESTRICT; // Pdox does not support cascading delete
            iFldCount := 1;         // foreign key is a single field
            aiThisTabFld[0] := 3;   // for us it's the 3rd field (ParentFK)
            aiOthTabFld[0] := 1;    // in Parent.db it's the 1st field
          end;
        printDesc := @singleRI;
      end;
    Check(DbiDoRestructure(RIDatabase.Handle,1,@tabledesc,nil,
            nil, // StrPCopy(szKeyViol,'orphan.db'),  << doesn't work  :(
            nil, False));
  finally
    ParentTable.Open;
    ChildTable.Open;
  end;
end;
{ Question:
  Does anyone know where Paradox's "strict referential integrity"
  checkbox option gets specified?  At the moment it's defaulting
  'off', which means early DOS Paradox versions could connect to
  the tables and trash the RI.  The STRICT flag appears in table's
  cursor once opened, and helpfile suggests the flag is set in the
  RI header when restructured but there's no such bit in RIntDesc.
  Maybe it's a CRTblDesc optional parameter? ("STRICT" = 1 etc)
  I haven't time to experiment with it.
}

procedure TMainForm.ParentTableAfterPost(DataSet: TDataSet);
begin
  with ChildTable do
    if Active then Refresh;  // show results of any cascading modify
end;



//--------------------------------------------------------------------------
// Restructure the child table, appending two new fields.
//--------------------------------------------------------------------------
{ This has nothing to do with referential integrity,
  but several people have asked how it's done.
}

procedure TMainForm.AppendButtonClick(Sender: TObject);
type
  FLDDescs = array [1..100] of FLDDesc;   // these are kludges for ease of addressing
  PFLDDescs = ^FLDDescs;
  CROpTypes = array [1..100] of CROpType;
  PCROpTypes = ^CROpTypes;
var
  props:     CURProps;    // table properties
  descs:     pFLDDescs;   // field descriptors
  ops:       pCROpTypes;  // field operators
  tabledesc: CRTblDesc;   // table descriptor for restructure
  i:         integer;
begin
  Screen.Cursor := crHourglass;
  // If RI already exists you must close the parent table here because
  // restructure needs to open the child's master(s) exclusively.
  ParentTable.Close;
  with ChildTable do
    try
      // Get details of existing fields and use them to produce a
      // revised field description for the new restructured table.
      Open;
     // if FindField('Initials') <> nil then raise Exception.Create('You can only add the extra fields once');
      Check(DbiGetCursorProps(Handle,props));
      // make arrays big enough to hold existing fields, plus 2 new fields
      descs := AllocMem(sizeof(FLDDesc) * (props.iFields + 2));
      ops := AllocMem(sizeof(CROpType) * (props.iFields + 2));
      try
        Check(DbiGetFieldDescs(Handle,@descs[1]));
        for i := 1 to props.iFields do   // retain existing fields
          ops^[i] := crNOOP;
        i := props.iFields + 1;
        //------ add first new field --------------
        ops^[i] := crADD;
        with descs^[i] do
          begin
          //  iFldNum := i;  (leave 0 when performing crADD)
            StrCopy(szName,'Initials');
            iFldType := fldZSTRING;
            iUnits1 := 3;   // number of characters
          end;
        //------ add second new field --------------
        inc(i);
        ops^[i] := crADD;
        with descs^[i] do
          begin
            StrCopy(szName,'Age');
            iFldType := fldINT16;
          end;
        // Finally, feed the new field desciption into a new table descriptor.
        FillChar(tabledesc,sizeof(CRTblDesc),#0);
        with tabledesc do
          begin
            StrPCopy(szTblName,TableName);  // ie. ChildTable.TableName
          // StrCopy(szTblType,szParadox);
          // bPack := True;
            iFldCount := props.iFields+2;
            pecrFldOp := @ops^[1];
            pfldDesc := @descs^[1];
          end;
        Close;  // table must be closed as restructure requires exclusive access
        Check(DbiDoRestructure(RIDatabase.Handle,1,@tabledesc,nil,nil,nil,False));
      finally
        FreeMem(descs);
        FreeMem(ops);
      end;
      AppendButton.Enabled := False;
    finally
      Open;
      ParentTable.Open;
      Screen.Cursor := crDefault;
    end;
end;

end.

