unit REdit;

{Description:  REdit: RichEdit descendent component with added events, methods and properties}
{Author:       Michael Lam (wy2lam@undergrad.math.uwaterloo.ca)}
{Created:           19 Mar 1998}
{Last Modified:     04 Dec 1998}
{Created on  Delphi 2.0                 }
{        for All Delphi 2.X, 3.X and 4.X}
{            C++ Builder 1.X, and 3.X   }

{note: a workaround of the repaint upon resize / horizontal scroll bar vertical redraw bug is
       not to hide the scrollbar - i.e. HideScrollBars := false always
       and repaint the control upon owner resizing.
}

{$H+}   // ansistring

interface

{please check if StdCtrls is used.  I haven't removed using this unit}
uses
    Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
    StdCtrls, ComCtrls, Printers;

type
    TPosChange = procedure(InRow, InCol: integer) of object;
    TModeChange = procedure(IsStateOverwrite: boolean) of object;
    TNotifyEvent = procedure(Sender: TObject) of object;

///////////////////////////////////////////////////////////////////////////////

    TREdit = class(TRichEdit)
    private
        // new events
        FOnPosChange: TPosChange;       // new event
        FOnModeChange: TModeChange;     // new event
        FOnVScroll: TNotifyEvent;       // new event
        FOnHScroll: TNotifyEvent;       // new event
        // variables
        FStream: TMemoryStream;         // the actual RTF stream
        FIndentSpace: integer;          // width of an indentation level, 0 = use true tabs
        FBackSpaceUnIndent: boolean;    // toggles backspace unindent
        FAutoIndent: boolean;           // toggles autoindent
        FIndentString: string;          // for autoindent - the "spaces" to be inserted on the next line
        FProcessBSOnKeyUp: boolean;     // for KeyDown to dictate the action of KeyUp on BS
        FIsOverwrite: boolean;          // true if in overwrite mode
        FRecursion: boolean;            // true if recursion is allowed
        FRow: integer;                  // to store current row
        FCol: integer;                  // to store current column
        FLastLine: string;              // to store the content of any line above the current line
        FCurLine: string;               // to store the content of current line
        // private functions related to new events
        function GetRTF: string;            // get the RTF string
        procedure SetRTF(InRTF: string); // set the RTF string
        function GetRow: integer;           // get the current caret row
        procedure SetRow(InRow: integer);   // set the current caret row
        function GetCol: integer;           // get the current caret column
        procedure SetCol(InCol: integer);   // set the current caret column
        function GetTop: integer;           // get the top visible line
        procedure SetTop(InRow: integer);   // set the top visible line
        function GetMaxVisible: integer;    // get the maximum number of visible lines
        procedure WMVScroll(var InMessage: TWMVScroll); message WM_VSCROLL;    // message handler
        procedure WMHScroll(var InMessage: TWMHScroll); message WM_HSCROLL;    // message handler
        // other property handling
        procedure SetIndentSpace(InIndentSpace: integer);
        // other private functions
        function SpacesTillLastIndent: integer; // calculates the number of spaces until last indent mark
        function SpacesTillNextIndent: integer; // calculates the number of spaces until next indent mark
        procedure PutSpaces;                    // send SpaceTillNextIndent spaces to Handle
    protected
        // overridden event-handler-handlers
        procedure KeyDown(var Key: word; Shift: TShiftState); override;
        procedure KeyPress(var Key: char); override;
        procedure KeyUp(var Key: word; Shift: TShiftState); override;
        procedure SelectionChange; override;
    public
        // overridden constructor and destructors
        constructor Create(AOwner: TComponent); override;
        destructor Destroy; override;
        // new methods
        procedure LoadFromFile(const FileName: string);
        procedure InsertFromFile(const FileName: string);
        procedure SaveToDOS(const FileName: string);
        procedure SaveToUnix(const FileName: string);
        procedure SaveToMac(const FileName: string);
        procedure TrimAllTrail;
        function Find(InPattern: string; InStart: integer;
                      IsCaseSen: boolean; IsRegExp: boolean): integer;
        // runtime properties
        property CurrentRow: integer read GetRow write SetRow;
        property CurrentCol: integer read GetCol write SetCol;
        property CurrentTop: integer read GetTop write SetTop;
        property MaxVisible: integer read GetMaxVisible;
        property IsOverwrite: boolean read FIsOverwrite;
    published
        property RTFText: string read GetRTF write SetRTF;
        property IndentSpace: integer read FIndentSpace write SetIndentSpace;
        property BackSpaceUnIndent: boolean read FBackSpaceUnIndent write FBackSpaceUnIndent;
        property AutoIndent: boolean read FAutoIndent write FAutoIndent;
        property OnChange;
        property OnKeyDown;
        property OnKeyPress;
        property OnKeyUp;
        property OnSelectionChange;
        property OnPosChange: TPosChange read FOnPosChange write FOnPosChange;
        property OnModeChange: TModeChange read FOnModeChange write FOnModeChange;
        property OnScrollV: TNotifyEvent read FOnVScroll write FOnVScroll;
        property OnScrollH: TNotifyEvent read FOnHScroll write FOnHScroll;
    end;

procedure Register;

implementation

// regular expression engine (but not included - search for it...)
// uses Regexp;

///////////////////////////////////////////////////////////////////////////////
constructor TREdit.Create(AOwner: TComponent);
begin
    inherited Create(AOwner);
    FIsOverwrite := false;              // in insert mode
    FRecursion := true;                 // allow recursion
    FStream := TMemoryStream.Create;    // RTF stream
end; //ok

///////////////////////////////////////////////////////////////////////////////
destructor TREdit.Destroy;
begin
    FStream.Free;                   // free the RTF stream
    inherited Destroy;
end; //ok

///////////////////////////////////////////////////////////////////////////////
procedure TREdit.LoadFromFile(const FileName: string);
begin
    // by default, the RichEdit control displays the Unix and Mac format
    // as if a conversion is made, but in fact, no conversion took place.
    // so if a file in Mac format is opened using Lines.LoadFromFile and saved
    // using SaveToMac, we'll have created a strange new format.
    // so, conversion to DOS format after each loading is necessary.
    Lines.LoadFromFile(FileName);
    // convert to DOS format
    Text := AdjustLineBreaks(Text);
end; //ok

///////////////////////////////////////////////////////////////////////////////
procedure TREdit.InsertFromFile(const FileName: string);
const
    ZERO: char = #0;
var
    theStream: TMemoryStream;
begin
    // insert a file at the selection point
    theStream := TMemoryStream.Create;
    theStream.LoadFromFile(FileName);
    theStream.Seek(0, soFromEnd);
    // null-terminate the buffer
    theStream.Write(ZERO, 1);
    theStream.Seek(0, soFromBeginning);
    SetSelTextBuf(theStream.Memory);
    theStream.Free;
    // convert to DOS format
    Text := AdjustLineBreaks(Text);
end; //ok

///////////////////////////////////////////////////////////////////////////////
procedure TREdit.SaveToDOS(const FileName: string);
begin
    // save the content to a DOS-formatted file
    // provided that the file is opened by TREdit.ReadFromFile and not
    // TREdit.Lines.ReadFromFile
    Lines.SaveToFile(FileName);
end; //ok

///////////////////////////////////////////////////////////////////////////////
procedure TREdit.SaveToUnix(const FileName: string);
const
    LF: char = #10;
var
    f: TextFile;
    i: integer;
    lastLine: integer;
    walkLine: string;
begin
    // save the current file in Unix format
    // provided that the file is opened by TREdit.ReadFromFile and not
    // TREdit.Lines.ReadFromFile
    AssignFile(f, FileName);
    Rewrite(f);
    lastLine := Lines.Count - 1;
    for i := 0 to lastLine do begin
        walkLine := Lines[i];
        Write(f, walkLine);
        Write(f, LF);
    end;
    CloseFile(f);
end; //ok

///////////////////////////////////////////////////////////////////////////////
procedure TREdit.SaveToMac(const FileName: string);
const
    CR: char = #13;
var
    f: TextFile;
    i: integer;
    lastLine: integer;
    walkLine: string;
begin
    // save the current file in Mac format
    // provided that the file is opened by TREdit.ReadFromFile and not
    // TREdit.Lines.ReadFromFile
    AssignFile(f, FileName);
    Rewrite(f);
    lastLine := Lines.Count - 1;
    for i := 0 to lastLine do begin
        walkLine := Lines[i];
        Write(f, walkLine);
        Write(f, CR);
    end;
    CloseFile(f);
end; //ok

///////////////////////////////////////////////////////////////////////////////
procedure TREdit.TrimAllTrail;
var
    i: integer;
    lastLine: integer;
    tempSelStart: integer;
begin
    // trim all trailing spaces
    // save selstart
    tempSelStart := SelStart;
    // trim all trailing spaces
    lastLine := Lines.Count - 1;
    // hide
    Visible := false;
    // action
    for i := 0 to lastLine do begin
        Lines[i] := TrimRight(Lines[i]);
    end;
    // unhide
    Visible := true;
    // restore selstart but scrap sellength
    SelStart := tempSelStart;
    SelLength := 0;
end; //ok

///////////////////////////////////////////////////////////////////////////////
function TREdit.Find(InPattern: string; InStart: integer;
                     IsCaseSen: boolean; IsRegExp: boolean): integer;
var
    foundLength: integer;
begin
{
    This is useful after linking with RegExp.pas (available in the SWAG archive)

    // advanced text search
    // InPattern - pattern to search,
    // InStart - position from which to start searching
    // IsCaseSen - case sensitivity
    // IsRegExp - regular expression?
    // the first string found will be highlighted, and its position (1-based) returned.
    // if it's not found, 0 is returned.
    if IsRegExp then
        InPattern := RegExp.MakePattern(InPattern, 1)
    else
        InPattern := RegExp.MakePatternNoRegEx(InPattern, 1);
    result := RegExp.Find(Text, InStart, InPattern, foundLength, IsCaseSen);
    if result <> 0 then begin
        SelStart := result - 1;
        SelLength := foundLength;
    end;
}
end; //test

///////////////////////////////////////////////////////////////////////////////
function TREdit.GetRTF: string;
begin
    // get the RTF string
    FStream.Clear;
    Lines.SaveToStream(FStream);
    Result := PChar(FStream.Memory);
end; //ok

///////////////////////////////////////////////////////////////////////////////
procedure TREdit.SetRTF(InRTF: string);
begin
    // set the RTF string
    FStream.Clear;
    FStream.WriteBuffer(InRTF[1], Length(InRTF));
    FStream.Position := 0;
    Lines.LoadFromStream(FStream);
end; //ok

///////////////////////////////////////////////////////////////////////////////
function TREdit.GetRow: integer;
begin
    // return the current row
    result := Perform(EM_LINEFROMCHAR, SelStart, 0);
end; //ok

///////////////////////////////////////////////////////////////////////////////
procedure TREdit.SetRow(InRow: integer);
begin
    // set current row
    // Restrict range to available lines
    if InRow < 0 then
        InRow := 0;
    if InRow >= Lines.Count then
        InRow := Lines.Count - 1;
    SelLength := 0;
    SelStart := Perform(EM_LINEINDEX, InRow, 0);
    FRow := InRow;
end; //ok

///////////////////////////////////////////////////////////////////////////////
function TREdit.GetCol: integer;
begin
    // return the current column
    result := SelStart - Perform(EM_LINEINDEX, Perform(EM_LINEFROMCHAR, SelStart, 0), 0);
end; //ok

///////////////////////////////////////////////////////////////////////////////
procedure TREdit.SetCol(InCol: integer);
begin
    // set current column
    // Value must be within range
    FRow := GetRow;
    if InCol < 0 then
        InCol := 0;
    if (InCol > Length(Lines[FRow])) then
        InCol := Length(Lines[FRow]);
    SelLength := 0;
    SelStart := Perform(EM_LINEINDEX, FRow, 0) + InCol;
    FCol := InCol;
end; //ok

///////////////////////////////////////////////////////////////////////////////
function TREdit.GetTop: integer;
begin
    // get the top visible line number
    result := Perform(EM_GETFIRSTVISIBLELINE, 0, 0);
end; //ok

///////////////////////////////////////////////////////////////////////////////
procedure TREdit.SetTop(InRow: integer);
begin
    // set the top visible line number
    if InRow < 0 then
        InRow := 0;
    if InRow >= Lines.Count then
        InRow := Lines.Count - 1;
    Perform(EM_LINESCROLL, 0, InRow - GetTop);
    // handle FOnVScroll
    if Assigned(FOnVScroll) then FOnVScroll(Self);
end; //ok

///////////////////////////////////////////////////////////////////////////////
function TREdit.GetMaxVisible: integer;
var
    richDC: HDC;        // temporary device context
    tm: TTextMetric;
begin
    // get the maximum number of visible lines
    if Visible then begin
        richDC := GetDC(Handle);
        SelectObject(richDC, Font.Handle);
        GetTextMetrics(richDC, tm);
        ReleaseDC(Handle, richDC);
        result := ClientHeight div tm.tmHeight;
    end else
        result := -1;
end; //ok

///////////////////////////////////////////////////////////////////////////////
procedure TREdit.WMVScroll(var InMessage: TWMVScroll);
begin
    // call event if assigned on vertical scroll
    inherited;
    if Assigned(FOnVScroll) then FOnVScroll(Self);
end; //ok

///////////////////////////////////////////////////////////////////////////////
procedure TREdit.WMHScroll(var InMessage: TWMHScroll);
begin
    // call event if assigned on horizontal scroll
    inherited;
    if Assigned(FOnHScroll) then FOnHScroll(Self);
end; //ok

///////////////////////////////////////////////////////////////////////////////
procedure TREdit.SetIndentSpace(InIndentSpace: integer);
begin
    // filter negative integers to IndentSpace
    if InIndentSpace >= 0 then
        FIndentSpace := InIndentSpace;
end; //ok

///////////////////////////////////////////////////////////////////////////////
function TREdit.SpacesTillLastIndent: integer;
// pre: FIndentSpace is non-zero
begin
    // calculates the number of spaces until last indent mark
    FCol := GetCol;
    result := FCol - (FCol div FIndentSpace) * FIndentSpace;
end; //ok

///////////////////////////////////////////////////////////////////////////////
function TREdit.SpacesTillNextIndent: integer;
// pre: FIndentSpace is non-zero
begin
    // calculates the number of spaces until next Indent mark
    FCol := GetCol;
    result := ((FCol div FIndentSpace) + 1) * FIndentSpace - FCol;
end; //ok

///////////////////////////////////////////////////////////////////////////////
procedure TREdit.PutSpaces;
const ONLY_ONE_SPACE = $00390001;
var
    i: integer;
    numSpaces: integer;
begin
    // send FIndentSpace number of spaces to Handle
    numSpaces := SpacesTillNextIndent;
    HideCaret(Handle);
    for i := 1 to numSpaces do
        PostMessage(Handle, WM_CHAR, VK_SPACE, ONLY_ONE_SPACE);
    ShowCaret(Handle);
end; //ok

///////////////////////////////////////////////////////////////////////////////
procedure TREdit.KeyDown(var Key: word; Shift: TShiftState);
var c: char;
begin if FRecursion then begin

    FRow := GetRow;
    FCol := GetCol;
    {optional stuffs that could be bypassed in this procedure}
        // disable Shift-Ins (i.e. OLD) style paste
        // and Ctrl-Ins (i.e. OLD) style copy
        // disable Shift-Delete (i.e. OLD) style cut
        // disable Alt-Backspace (i.e. OLD) style undo
        // (Alt-Ins does nothing - save to ignore)
        // if (ssShift in Shift) or (ssCtrl in Shift) then Key := 0;
    {end of optional bypassing stuffs}

    // and determines whether KeyUp should further look at VK_BACK
    if (Key = VK_BACK) then begin
        if (FRow > 1) or (FCol > 1) {avoid GPF} then begin
            // no need to look further if selection is not empty
            c := Text[SelStart];
            FProcessBSOnKeyUp := FBackSpaceUnIndent and (FIndentSpace > 0) and (c = ' ') and (SelLength = 0);
        end else
            FProcessBSOnKeyUp := false;
    end else

    // change mode (insert / overwrite) and call event if assigned
    if (Key = VK_INSERT) then begin
        // handle mode change
        if not (ssCtrl in Shift) and not (ssAlt in Shift) and not (ssShift in Shift) then begin
            FIsOverwrite := not FIsOverwrite;
            if Assigned(FOnModeChange) then FOnModeChange(FIsOverwrite);
        end;
    end else

    // scroll up without moving cursor
    if (Key = VK_UP) and (ssCtrl in Shift) then begin
        SetTop(GetTop - 1);
        Key := 0;
    end else

    // scroll down without moving cursor
    if (Key = VK_DOWN) and (ssCtrl in Shift) then begin
        SetTop(GetTop + 1);
        Key := 0;
    end else

    // page up without moving cursor
    if (Key = VK_PRIOR) and (ssCtrl in Shift) then begin
        SetTop(GetTop - GetMaxVisible);
        Key := 0;
    end else

    // page down without moving cursor
    if (Key = VK_NEXT) then begin and (ssCtrl in Shift) then begin
        SetTop(GetTop + GetMaxVisible);
        Key := 0;
    end;

    inherited KeyDown(Key, Shift);
end end; //ok

///////////////////////////////////////////////////////////////////////////////
procedure TREdit.KeyPress(var Key: char);
begin
    // handles "tab to next indent mark with spaces"
    // Ctrl-Tab and Alt-Tab get thru so don't worry.
    if Key = #9 then
        if FIndentSpace <> 0 then begin    // use true tab if FIndentSpace = 0
            PutSpaces;
            Key := #0;
        end;

    inherited KeyPress(Key);
end; //ok

///////////////////////////////////////////////////////////////////////////////
procedure TREdit.KeyUp(var Key: word; Shift: TShiftState);
const ONLY_ONE_BS = $000E0001;
var
    indentLength: integer;      // length of indentation string
    growSpace: string;          // total number of spaces
    spacesTillLast: integer;    // spaces to go back to the last indent mark
    isAllSpace: boolean;        // for backspace handling - check if the last (spacesTillLast - 1) characters are spaces
    i: integer;
begin

    FRow := GetRow;
    FCol := GetCol;

    // handles autoindent
    // ssAlt is filtered since Alt-Enter doesn't move the caret
    if (Key = VK_RETURN) and not (ssAlt in Shift) then begin
        if FAutoIndent then begin
            begin
                if (FRow = 0) then exit;  // no last line !
                i := FRow - 1;
                // if line is empty (after trim), follow the
                // indentation of the last non-empty (after trim) line
                repeat
                    FLastLine := Lines[i];
                    Dec(i);
                until (Trim(FLastLine) <> '') or (i < 0);
                indentLength := Pos(TrimLeft(FLastLine), FLastLine) - 1;
                FIndentString := Copy(FLastLine, 1, indentLength);
            end;
            SelText := FIndentString + SelText;
        end;
    end else

    // handles "backspace unindent"
    if (Key = VK_BACK) and FProcessBSOnKeyUp then begin
        // if the last SpacesTillLastIndent characters are spaces, delete them
        growSpace := '';
        spacesTillLast := SpacesTillLastIndent;
        for i := 1 to spacesTillLast do growSpace := growSpace + ' ';
        FCurLine := Lines[FRow];
        isAllSpace := (growSpace = Copy(FCurLine, FCol - spacesTillLast + 1, spacesTillLast));
        if isAllSpace then begin
            HideCaret(Handle);
            FRecursion := false;    // stop KeyDown from trapping these backspaces
            for i := 1 to spacesTillLast do
                PostMessage(Handle, WM_KEYDOWN, VK_BACK, ONLY_ONE_BS);
            FRecursion := true;
            ShowCaret(Handle);
            Key := 0;
        end;
    end else

    inherited KeyUp(Key, Shift);
end; //ok

///////////////////////////////////////////////////////////////////////////////
procedure TREdit.SelectionChange;
begin
    // position change handler: get the current position, and call event if assigned
    // do NOT call anything that changes selection here

    FRow := GetRow;
    FCol := GetCol;

    if Assigned(FOnPosChange) then FOnPosChange(FRow, FCol);
    // change of position is placed before change of selection
    // so the change in position is handled before calling user's OnSelectionChange
    inherited SelectionChange;

end; //ok

///////////////////////////////////////////////////////////////////////////////
procedure Register;
begin
    RegisterComponents('Samples', [TREdit]);
end; //ok

end.