{
***************************************************
* A binary compatible RC5 implementation          *
* written by Dave Barton (davebarton@bigfoot.com) *
***************************************************
* 64bit block encryption                          *
* Variable size key - up to 2048bit               *
***************************************************
}
unit RC5;

interface
uses
  Sysutils, Tools;

const
  NUMROUNDS= 12;   { number of rounds must be between 12-16 }

type
  TRC5Data= record
    InitBlock: array[0..7] of byte;    { initial IV }
    LastBlock: array[0..7] of byte;    { current IV }
    Key: array[0..((NUMROUNDS*2)+1)] of DWord;
  end;

function RC5SelfTest: boolean;
  { performs a self test on this implementation }
procedure RC5Init(var Data: TRC5Data; Key: pointer; Len: integer; IV: pointer);
  { initializes the TRC5Data structure with the key information and IV if applicable }
procedure RC5Burn(var Data: TRC5Data);
  { erases all information about the key }

procedure RC5EncryptECB(var Data: TRC5Data; InData, OutData: pointer);
  { encrypts the data in a 64bit block using the ECB mode }
procedure RC5EncryptCBC(var Data: TRC5Data; InData, OutData: pointer);
  { encrypts the data in a 64bit block using the CBC chaining mode }
procedure RC5EncryptOFB(var Data: TRC5Data; InData, OutData: pointer);
  { encrypts the data in a 64bit block using the OFB chaining mode }
procedure RC5EncryptCFB(var Data: TRC5Data; InData, OutData: pointer; Len: integer);
  { encrypts Len bytes of data using the CFB chaining mode }
procedure RC5EncryptOFBC(var Data: TRC5Data; InData, OutData: pointer; Len: integer);
  { encrypts Len bytes of data using the OFB counter chaining mode }

procedure RC5DecryptECB(var Data: TRC5Data; InData, OutData: pointer);
  { decrypts the data in a 64bit block using the ECB mode }
procedure RC5DecryptCBC(var Data: TRC5Data; InData, OutData: pointer);
  { decrypts the data in a 64bit block using the CBC chaining mode }
procedure RC5DecryptOFB(var Data: TRC5Data; InData, OutData: pointer);
  { decrypts the data in a 64bit block using the OFB chaining mode }
procedure RC5DecryptCFB(var Data: TRC5Data; InData, OutData: pointer; Len: integer);
  { decrypts Len bytes of data using the CFB chaining mode }
procedure RC5DecryptOFBC(var Data: TRC5Data; InData, OutData: pointer; Len: integer);
  { decrypts Len bytes of data using the OFB counter chaining mode }

procedure RC5Reset(var Data: TRC5Data);
  { resets the chaining mode information }

{******************************************************************************}
implementation

const
   sBox: array[0..33] of dword= (
    $B7E15163,$5618CB1C,$F45044D5,$9287BE8E,$30BF3847,$CEF6B200,
    $6D2E2BB9,$0B65A572,$A99D1F2B,$47D498E4,$E60C129D,$84438C56,
    $227B060F,$C0B27FC8,$5EE9F981,$FD21733A,$9B58ECF3,$399066AC,
    $D7C7E065,$75FF5A1E,$1436D3D7,$B26E4D90,$50A5C749,$EEDD4102,
    $8D14BABB,$2B4C3474,$C983AE2D,$67BB27E6,$05F2A19F,$A42A1B58,
    $42619511,$E0990ECA,$7ED08883,$1D08023C);

function RC5SelfTest;
const
  Key: array[0..15] of byte=
    ($91,$5F,$46,$19,$BE,$41,$B2,$51,$63,$55,$A5,$01,$10,$A9,$CE,$91);
  InBlock: array[0..1] of DWord=
    ($EEDBA521,$6D8F4B15);
  OutBlock: array[0..1] of DWord=
    ($AC13C0F7,$52892B5B);
var
  Block: array[0..7] of byte;
  Data: TRC5Data;
begin
  RC5Init(Data,@Key,Sizeof(Key),nil);
  RC5EncryptECB(Data,@InBlock,@Block);
  Result:= CompareMem(@Block,@OutBlock,Sizeof(Block)) or not (NUMROUNDS=12); { test vector not valid for any other than 12 rounds }
  RC5DecryptECB(Data,@Block,@Block);
  Result:= Result and CompareMem(@Block,@InBlock,Sizeof(Block));
  RC5Burn(Data);
end;

procedure RC5Init;
var
  xKeyD: array[0..63] of DWord;
  i, j, k, xKeyLen: integer;
  A, B: DWord;
begin
  if (Len<= 0) or (Len> 256) then
    raise Exception.Create('RC5: Key length must be between 1 and 256 bytes');
  if IV= nil then
  begin
    FillChar(Data.InitBlock,8,0);
    FillChar(Data.LastBlock,8,0);
  end
  else
  begin
    Move(IV^,Data.InitBlock,8);
    Move(IV^,Data.LastBlock,8);
  end;
  FillChar(xKeyD,Sizeof(xKeyD),0);
  Move(Key^,xKeyD,Len);
  with Data do
  begin
    xKeyLen:= Len div 4;
    if (Len mod 4)<> 0 then
      Inc(xKeyLen);
    Move(sBox,Key,(NUMROUNDS+1)*8);
    i:= 0; j:= 0;
    A:= 0; B:= 0;
    if xKeyLen> ((NUMROUNDS+1)*2) then
      k:= xKeyLen*3
    else
      k:= (NUMROUNDS+1)*6;
    for k:= k downto 1 do
    begin
      A:= LRot32(Key[i]+A+B,3);
      Key[i]:= A;
      B:= LRot32(xKeyD[j]+A+B,A+B);
      xKeyD[j]:= B;
      i:= (i+1) mod ((NUMROUNDS+1)*2);
      j:= (j+1) mod xKeyLen;
    end;
    FillChar(xKeyD,Sizeof(xKeyD),0);
  end;
end;

procedure RC5Burn;
begin
  FillChar(Data,Sizeof(Data),0);
end;

procedure RC5EncryptECB;
var
  A, B: DWord;
  i: integer;
begin
  Move(InData^,A,4);
  Move(pointer(integer(InData)+4)^,B,4);
  A:= A + Data.Key[0];
  B:= B + Data.Key[1];
  for i:= 1 to NUMROUNDS do
  begin
    A:= A xor B;
    A:= LRot32(A,B)+Data.Key[2*i];
    B:= B xor A;
    B:= LRot32(B,A)+Data.Key[(2*i)+1];
  end;
  Move(A,OutData^,4);
  Move(B,pointer(integer(OutData)+4)^,4);
end;

procedure RC5DecryptECB;
var
  A, B: DWord;
  i: integer;
begin
  Move(InData^,A,4);
  Move(pointer(integer(InData)+4)^,B,4);
  for i:= NUMROUNDS downto 1 do
  begin
    B:= RRot32(B-Data.Key[(2*i)+1],A);
    B:= B xor A;
    A:= RRot32(A-Data.Key[2*i],B);
    A:= A xor B;
  end;
  B:= B - Data.Key[1];
  A:= A - Data.Key[0];
  Move(A,OutData^,4);
  Move(B,pointer(integer(OutData)+4)^,4);
end;

procedure RC5EncryptCBC;
begin
  XorBlock(InData,@Data.LastBlock,OutData,8);
  RC5EncryptECB(Data,OutData,OutData);
  Move(OutData^,Data.LastBlock,8);
end;

procedure RC5DecryptCBC;
var
  TempBlock: array[0..7] of byte;
begin
  Move(InData^,TempBlock,8);
  RC5DecryptECB(Data,InData,OutData);
  XorBlock(OutData,@Data.LastBlock,OutData,8);
  Move(TempBlock,Data.LastBlock,8);
end;

procedure RC5EncryptCFB;
var
  i: integer;
  TempBlock: array[0..7] of byte;
begin
  for i:= 0 to Len-1 do
  begin
    RC5EncryptECB(Data,@Data.LastBlock,@TempBlock);
    PByteArray(OutData)[i]:= PByteArray(InData)[i] xor TempBlock[0];
    Move(Data.LastBlock[1],Data.LastBlock[0],7);
    Data.LastBlock[7]:= PByteArray(OutData)[i];
  end;
end;

procedure RC5DecryptCFB;
var
  i: integer;
  TempBlock: array[0..7] of byte;
  b: byte;
begin
  for i:= 0 to Len-1 do
  begin
    b:= PByteArray(InData)[i];
    RC5EncryptECB(Data,@Data.LastBlock,@TempBlock);
    PByteArray(OutData)[i]:= PByteArray(InData)[i] xor TempBlock[0];
    Move(Data.LastBlock[1],Data.LastBlock[0],7);
    Data.LastBlock[7]:= b;
  end;
end;

procedure RC5EncryptOFB;
begin
  RC5EncryptECB(Data,@Data.LastBlock,@Data.LastBlock);
  XorBlock(@Data.LastBlock,InData,OutData,8);
end;

procedure RC5DecryptOFB;
begin
  RC5EncryptECB(Data,@Data.LastBlock,@Data.LastBlock);
  XorBlock(@Data.LastBlock,InData,OutData,8);
end;

procedure RC5EncryptOFBC;
var
  i: integer;
  TempBlock: array[0..7] of byte;
begin
  for i:= 0 to Len-1 do
  begin
    RC5EncryptECB(Data,@Data.LastBlock,@TempBlock);
    PByteArray(OutData)[i]:= PByteArray(InData)[i] xor TempBlock[0];
    IncBlock(@Data.LastBlock,8);
  end;
end;

procedure RC5DecryptOFBC;
var
  i: integer;
  TempBlock: array[0..7] of byte;
begin
  for i:= 0 to Len-1 do
  begin
    RC5EncryptECB(Data,@Data.LastBlock,@TempBlock);
    PByteArray(OutData)[i]:= PByteArray(InData)[i] xor TempBlock[0];
    IncBlock(@Data.LastBlock,8);
  end;
end;

procedure RC5Reset;
begin
  Move(Data.InitBlock,Data.LastBlock,8);
end;


end.
