#define STRICT
#include <windows.h>
#include <stdio.h>
#include <ctype.h>
#include "unrar.h"

enum { EXTRACT, TEST, PRINT };

void ExtractArchive(char *ArcName,int Mode);
void ListArchive(char *ArcName);
void ShowComment(char *CmtBuf);
void OutHelp(void);
void OutOpenArchiveError(int Error,char *ArcName);
void OutProcessFileError(int Error);
int ChangeVolProc(char *ArcName,int Mode);
int ProcessDataProc(unsigned char *Addr,int Size);

main(int Argc, char *Argv[])
{
  if (Argc!=3)
  {
    OutHelp();
    return(0);
  }

  switch(toupper(Argv[1][0]))
  {
    case 'X':
      ExtractArchive(Argv[2],EXTRACT);
      break;
    case 'T':
      ExtractArchive(Argv[2],TEST);
      break;
    case 'P':
      ExtractArchive(Argv[2],PRINT);
      break;
    case 'L':
      ListArchive(Argv[2]);
      break;
    default:
      OutHelp();
      return(0);
  }

  return(0);
}


void ExtractArchive(char *ArcName,int Mode)
{
  HANDLE hArcData;
  int RHCode,PFCode;
  char CmtBuf[16384];
  struct RARHeaderData HeaderData;
  struct RAROpenArchiveData OpenArchiveData;

  OpenArchiveData.ArcName=ArcName;
  OpenArchiveData.CmtBuf=CmtBuf;
  OpenArchiveData.CmtBufSize=sizeof(CmtBuf);
  OpenArchiveData.OpenMode=RAR_OM_EXTRACT;
  hArcData=RAROpenArchive(&OpenArchiveData);

  if (OpenArchiveData.OpenResult!=0)
  {
    OutOpenArchiveError(OpenArchiveData.OpenResult,ArcName);
    return;
  }

  if (OpenArchiveData.CmtState==1)
    ShowComment(CmtBuf);

  RARSetChangeVolProc(hArcData,ChangeVolProc);

  if (Mode==PRINT)
    RARSetProcessDataProc(hArcData,ProcessDataProc);

  HeaderData.CmtBuf=NULL;

  while ((RHCode=RARReadHeader(hArcData,&HeaderData))==0)
  {
    switch(Mode)
    {
      case EXTRACT:
        printf("\nExtracting %-45s",HeaderData.FileName);
        break;
      case TEST:
        printf("\nTesting %-45s",HeaderData.FileName);
        break;
      case PRINT:
        printf("\nPrinting %-45s\n",HeaderData.FileName);
        break;
    }
    PFCode=RARProcessFile(hArcData,(Mode==EXTRACT) ? RAR_EXTRACT:RAR_TEST,
                          NULL,NULL);
    if (PFCode==0)
      printf(" Ok");
    else
    {
      OutProcessFileError(PFCode);
      break;
    }
  }

  if (RHCode==ERAR_BAD_DATA)
    printf("\nFile header broken");

  RARCloseArchive(hArcData);
}


void ListArchive(char *ArcName)
{
  HANDLE hArcData;
  int RHCode,PFCode;
  char CmtBuf[16384];
  struct RARHeaderData HeaderData;
  struct RAROpenArchiveData OpenArchiveData;

  OpenArchiveData.ArcName=ArcName;
  OpenArchiveData.CmtBuf=CmtBuf;
  OpenArchiveData.CmtBufSize=sizeof(CmtBuf);
  OpenArchiveData.OpenMode=RAR_OM_LIST;
  hArcData=RAROpenArchive(&OpenArchiveData);

  if (OpenArchiveData.OpenResult!=0)
  {
    OutOpenArchiveError(OpenArchiveData.OpenResult,ArcName);
    return;
  }

  if (OpenArchiveData.CmtState==1)
    ShowComment(CmtBuf);

  RARSetChangeVolProc(hArcData,ChangeVolProc);

  HeaderData.CmtBuf=CmtBuf;
  HeaderData.CmtBufSize=sizeof(CmtBuf);

  printf("\nFile                       Size");
  printf("\n-------------------------------");
  while ((RHCode=RARReadHeader(hArcData,&HeaderData))==0)
  {
    printf("\n%-20s %10d ",HeaderData.FileName,HeaderData.UnpSize);
    if (HeaderData.CmtState==1)
      ShowComment(CmtBuf);
    if ((PFCode=RARProcessFile(hArcData,RAR_SKIP,NULL,NULL))!=0)
    {
      OutProcessFileError(PFCode);
      break;
    }
  }

  if (RHCode==ERAR_BAD_DATA)
    printf("\nFile header broken");

  RARCloseArchive(hArcData);
}


void ShowComment(char *CmtBuf)
{
  printf("\nComment:\n%s\n",CmtBuf);
}


void OutHelp(void)
{
  printf("\nUNRDLL.   This is a simple example of UNRAR.DLL usage\n");
  printf("\nSyntax:\n");
  printf("\nUNRDLL X <Archive>     extract archive contents");
  printf("\nUNRDLL T <Archive>     test archive contents");
  printf("\nUNRDLL P <Archive>     print archive contents to stdout");
  printf("\nUNRDLL L <Archive>     view archive contents\n");
}


void OutOpenArchiveError(int Error,char *ArcName)
{
  switch(Error)
  {
    case ERAR_NO_MEMORY:
      printf("\nNot enough memory");
      break;
    case ERAR_EOPEN:
      printf("\nCannot open %s",ArcName);
      break;
    case ERAR_BAD_ARCHIVE:
      printf("\n%s is not RAR archive",ArcName);
      break;
    case ERAR_BAD_DATA:
      printf("\n%s: archive header broken",ArcName);
      break;
  }
}

void OutProcessFileError(int Error)
{
  switch(Error)
  {
    case ERAR_UNKNOWN_FORMAT:
      printf("Unknown archive format");
      break;
    case ERAR_BAD_ARCHIVE:
      printf("Bad volume");
      break;
    case ERAR_ECREATE:
      printf("File create error");
      break;
    case ERAR_EOPEN:
      printf("Volume open error");
      break;
    case ERAR_ECLOSE:
      printf("File close error");
      break;
    case ERAR_EREAD:
      printf("Read error");
      break;
    case ERAR_EWRITE:
      printf("Write error");
      break;
    case ERAR_BAD_DATA:
      printf("CRC error");
      break;
  }
}


int ChangeVolProc(char *ArcName,int Mode)
{
  int Ch;
  if (Mode==RAR_VOL_ASK)
  {
    printf("\nInsert disk with %s and press 'Enter' or enter 'Q' to exit ",ArcName);
    Ch=getchar();
    return(toupper(Ch)!='Q');
  }
  return(1);
}


int ProcessDataProc(unsigned char *Addr,int Size)
{
  fflush(stdout);
  fwrite(Addr,1,Size,stdout);
  fflush(stdout);
  return(1);
}
