{*******************************************************************************
    TRangeBox & TDBRangeBox v1.0 (My first written component!)
    Copyright  1997 Tom Deprez
    (code of Bill Menees : Copyright  1996 Bill Menees)
    Tom.Deprez@uz.kuleuven.ac.be

    10x to Bill Menees who gave his approval for distributing this component as
    freeware.

    This is a VCL component that checks if a value is in a certain range or not.
    There is a non dataaware and an dataaware component. Range is given trough
    Max and Min property. You can check if a value lies between Max & Min, below
    Max or above Min. Specified by the Range property. User is able to alter
    the check if certain property is set to true.
    There are three states : InRange, OutRange, NoCheck. Designer can use its
    own images for the different states.

    A lot of the source code (almost everything) comes directly form the
    TBitCheckBox component of Bill Menees. I only made some changes to make
    the TBitCheckBox do what I wanted it to do. For instance the Paint
    procedure is only altered so it can be used with other types, just like
    other procedures and properties.
    I made it a new component because it's no longer a CheckBox, now it's a
    checkbox with range check capabilities.
    The TBitCheckBox component is Copyright  1996 Bill Menees,
    bmenees@usit.net, http://www.public.usit.net/bmenees

    You're free to use and redistribute this component as long as this header
    stays attached.  This component is provided AS IS with no warrenties or
    guarantees implied.

    It would be great if you could send me the improvements you make to this
    component. Also, all possible bugs are welcome. If you want, you could
    send me why and for what you've used this component.
    That will be appreciated.

*******************************************************************************}

unit Range;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs
  , DB, DBTables;

type
  TRangeControlState = (rbInRange, rbOutRange, rbNoCheck);
  TRangeType     = (ctMin, ctMax);

  TRange         = Set of TRangeType;
  TRangeControl = class(TCustomControl)
  private
    { Private declarations }
   fInRangeBitmap: TBitmap;
   fOutRangeBitmap: TBitmap;
   fNoCheckBitmap: TBitmap;
   fAlignment: TLeftRight;
   fState: TRangeControlState;
   fInRangeNumGlyphs: Integer;
   fOutRangeNumGlyphs: Integer;
   fNoCheckNumGlyphs: Integer;
   fMaxRangeValue: Double;
   fMinRangeValue: Double;
   fCheckValue: Double;
   fRange: TRange;
   fCanModify: Boolean;

   fAfterRangeCheck: TNotifyEvent;
   fBeforeRangeCheck: TNotifyEvent;

   function GetInRange: Boolean;

   procedure SetInRangeNumGlyphs(Value: Integer);
   procedure SetOutRangeNumGlyphs(Value: Integer);
   procedure SetNoCheckNumGlyphs(Value: Integer);
   procedure SetInRangeBitmap(Value : TBitmap);
   procedure SetOutRangeBitmap(Value : TBitmap);
   procedure SetNoCheckBitmap(Value : TBitmap);
   procedure SetState(Value: TRangeControlState);
   procedure SetAlignment(Value : TLeftRight);
   procedure SetInRange(Value: Boolean);
   procedure SetMaxRangeValue(Value: Double);
   procedure SetMinRangeValue(Value: Double);
   procedure SetRange(Value: TRange);
   procedure CMEnabledChanged(var Msg: TMessage); message CM_ENABLEDCHANGED;
   procedure CMDialogChar(var Msg: TCMDialogChar); message CM_DIALOGCHAR;
   procedure CMFontChanged(var Msg: TMessage); message CM_FONTCHANGED;
   procedure CMTextChanged(var Msg: TMessage); message CM_TEXTCHANGED;
   procedure CMSysColorChange(var Msg: TMessage); message CM_SYSCOLORCHANGE;
   procedure CMCtl3DChanged(var Msg: TMessage); message CM_CTL3DCHANGED;
   procedure WMSetFocus(var Message: TWMSetFocus); message WM_SETFOCUS;
   procedure WMKillFocus(var Message: TWMKillFocus); message WM_KILLFOCUS;
  protected
    { Protected declarations }
    procedure Paint; override;
    procedure Click; override;
    procedure KeyPress(var Key: Char); override;
    procedure ValueInRange;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Toggle; virtual;
  published
    { Published declarations }
    property Alignment: TLeftRight read fAlignment write SetAlignment default taRightJustify;
    property Caption;
    property CanModify: Boolean read fCanModify write fCanModify default False;
    property Range: TRange read fRange write SetRange default [];
    property Color;
    property Ctl3D;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
    property InRangeGlyph: TBitmap read fInRangeBitmap write SetInRangeBitmap;
    property InRangeNumGlyphs: Integer read fInRangeNumGlyphs write SetInRangeNumGlyphs default 1;
    property MaxRangeValue: Double read fMaxRangeValue write SetMaxRangeValue;
    property MinRangeValue: Double read fMinRangeValue write SetMinRangeValue;
    property NoCheckGlyph: TBitmap read fNoCheckBitmap write SetNoCheckBitmap;
    property NoCheckNumGlyphs: Integer read fNoCheckNumGlyphs write SetNoCheckNumGlyphs default 1;
    property OutRangeGlyph: TBitmap read fOutRangeBitmap write SetOutRangeBitmap;
    property OutRangeNumGlyphs: Integer read fOutRangeNumGlyphs write SetOutRangeNumGlyphs default 1;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property State: TRangeControlState read fState write SetState default rbNoCheck;
    property TabOrder;
    property TabStop;
    property Visible;

    property AfterRangeCheck: TNotifyEvent read fAfterRangeCheck write fAfterRangeCheck;
    property BeforeRangeCheck: TNotifyEvent read fBeforeRangeCheck write fBeforeRangeCheck;
    property OnClick;
    property OnDragDrop;
    property OnEndDrag;
    property OnEnter;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseUp;
    property OnStartDrag;
  end;

  TRangeBox = class(TRangeControl)
  private
    { Private declarations }
   procedure SetCheckValue(Value: Double);
  protected
    { Protected declarations }
  public
    { Public declarations }
  published
    { Published declarations }
    property CheckValue: Double read fCheckValue write SetCheckValue;
  end;

  TDBRangeBox = class(TRangeControl)
  private
    { Private declarations }
   fDataLink : TFieldDataLink;
   function GetDataField: string;
   function GetDataSource: TDataSource;
   procedure SetDataField(const Value: String);
   procedure SetDataSource(Value: TDataSource);
   procedure DataChange(Sender: TObject);
  protected
    { Protected declarations }
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    { Published declarations }
    property DataField: string read GetDataField write SetDataField;
    property DataSource: TDataSource read GetDataSource write SetDataSource;
  end;

procedure Register;

implementation

{$R RangeControl.res}

{******************************************************************************}
{*** Non-Member Functions *****************************************************}
{******************************************************************************}

procedure Register;
begin
  RegisterComponents('Samples', [TRangeBox]);
  RegisterComponents('Samples', [TDBRangeBox])
end;

{******************************************************************************}
{*** TRangeControl Public Methods ********************************************}
{******************************************************************************}

constructor TRangeControl.Create(AOwner: TComponent);
begin
 inherited Create(AOwner);
 fInRangeBitmap  := TBitmap.Create;
 fOutRangeBitmap := TBitmap.Create;
 fNoCheckBitmap  := TBitmap.Create;

 ControlStyle := [csSetCaption, csClickEvents];
 Alignment    := taRightJustify;
 State        := rbNoCheck;
 Width        := 97;
 Height       := 17;
 TabStop      := False;
 ParentCtl3D  := True;

 InRangeGlyph  := nil;
 OutRangeGlyph := nil;
 NoCheckGlyph  := nil;
 InRangeNumGlyphs   := 1;
 OutRangeNumGlyphs := 1;
 NoCheckNumGlyphs  := 1;
 fMinRangeValue    := 0;
 fMaxRangeValue    := 0;
 fCheckValue       := 0;
 Range        := [];
end;

destructor TRangeControl.Destroy;
begin
 fInRangeBitmap.Free;
 fOutRangeBitmap.Free;
 fNoCheckBitmap.Free;
 inherited Destroy;
end;

procedure TRangeControl.Toggle;
begin
 if CanModify then begin
  case State of
   rbInRange  : State := rbOutRange;
   rbOutRange : State := rbNoCheck;
   rbNoCheck  : State := rbInRange;
  end;
 end;
end;

{******************************************************************************}
{*** TRangeControl Protected Methods *****************************************}
{******************************************************************************}

procedure TRangeControl.Paint;
const
     DrawOptions = DT_LEFT or DT_VCENTER or DT_SINGLELINE;
var
   Rc, BitRect, FocusRect: TRect;
   Bitmap: TBitmap;
   HOffset, VOffset, NumGlyphs, GlyphWidth: Integer;
   OldColor: TColor;
begin
     with Canvas do
     begin
          {First, blank our client area.}
          Rc := ClientRect;
          Brush.Color := Self.Color;
          Brush.Style := bsSolid;
          FillRect(Rc);

          {Now make the brush clear so our fonts get
          drawn correctly (like 3D disabled text).}
          Brush.Style := bsClear;

          {Determine which bitmap will be used.}
          case State of
               rbInRange:
               begin
                    Bitmap:=InRangeGlyph;
                    NumGlyphs:=InRangeNumGlyphs;
               end;
               rbOutRange:
               begin
                    Bitmap:=OutRangeGlyph;
                    NumGlyphs:=OutRangeNumGlyphs;
               end;
               else
               begin
                    Bitmap:=NoCheckGlyph;
                    NumGlyphs:=NoCheckNumGlyphs;
               end;
          end;
          GlyphWidth:=Bitmap.Width div NumGlyphs;

          {Draw the text (after determining the FocusRect).}
          Rc:=ClientRect;
          if Alignment = taLeftJustify then
             HOffset:=1
          else
              HOffset:=GlyphWidth+4;
          Rc.Left:=HOffset;
          Rc.Right:=Rc.Left+TextWidth(Caption)+1;
          {Subtract out the width of the underscoring character.}
          if Pos('&', Caption) > 0 then Rc.Right:=Rc.Right-TextWidth('&');
          VOffset:=(ClientHeight-TextHeight(Caption)) div 2 - 2;
          Rc.Top:=VOffset;
          Rc.Bottom:=Rc.Top+TextHeight(Caption)+3;
          IntersectRect(FocusRect, Rc, ClientRect);

          OldColor:=Font.Color;
          {DrawText is used because it handles
          the underscored accelerator key.}
          if Enabled then
          begin
               DrawText(Handle, PChar(Caption), Length(Caption), FocusRect, DrawOptions);
          end
          else
          begin
               if Ctl3D then
               begin
                    {This draws disabled text in 3D.}
                    OffsetRect(FocusRect, 1, 1);
                    Font.Color := clBtnHighlight;
                    DrawText(Handle, PChar(Caption), Length(Caption), FocusRect, DrawOptions);
                    OffsetRect(FocusRect, -1, -1);
               end;
               {This draws disabled text like SQL6.}
               Font.Color:=clGrayText;
               DrawText(Handle, PChar(Caption), Length(Caption), FocusRect, DrawOptions);
          end;
          Font.Color:=OldColor;

          {Now we need make the brush solid again for the BrushCopy to work.}
          Brush.Color := Self.Color;
          Brush.Style := bsSolid;

          {Draw the focused rectangle.}
          if Focused then DrawFocusRect(FocusRect);

          {Now, draw the bitmap.}
          if Alignment = taLeftJustify then
             HOffset:=ClientWidth-GlyphWidth
          else
              HOffset:=0;
          VOffset:=(ClientHeight - Bitmap.Height) div 2;
          {Figure out where to draw the image.}
          Rc.Top:=VOffset;
          Rc.Bottom:=Bitmap.Height+VOffset;
          Rc.Left:=HOffset;
          Rc.Right:=GlyphWidth+HOffset;
          {Choose the correct bitmap.}
          {If we're disabled choose the second bitmap.}
          if not Enabled and (NumGlyphs = 2) then
             BitRect.Left:=GlyphWidth
          else
              BitRect.Left:=0;
          BitRect.Right:=BitRect.Left+GlyphWidth;
          BitRect.Top:=0;
          BitRect.Bottom:=Bitmap.Height;

          BrushCopy(Rc, Bitmap, BitRect, Bitmap.TransparentColor);
     end;
end;

procedure TRangeControl.Click;
begin
  Toggle;
  inherited Click;
  if Showing and CanFocus then SetFocus;
end;

procedure TRangeControl.KeyPress(var Key: Char);
begin
  inherited KeyPress(Key);
  if Key = ' ' then Click;
end;

{******************************************************************************}
{** TBitCheckBox Private Methods **********************************************}
{******************************************************************************}

procedure TRangeControl.SetAlignment(Value: TLeftRight);
begin
     if Value <> fAlignment then
     begin
          fAlignment:=Value;
          Invalidate;
     end;
end;

procedure TRangeControl.CMEnabledChanged(var Msg: TMessage);
begin
     inherited;
     Invalidate;
end;

procedure TRangeControl.CMDialogChar(var Msg: TCMDialogChar);
begin
     with Msg do
          if IsAccel(CharCode, Caption) and Enabled and CanFocus then
          begin
               Click;
               Result := 1;
          end
          else
              inherited;
end;

procedure TRangeControl.CMFontChanged(var Msg: TMessage);
begin
     inherited;
     Canvas.Font:=Font;
     Invalidate;
end;

procedure TRangeControl.CMTextChanged(var Msg: TMessage);
begin
     inherited;
     Invalidate;
end;

procedure TRangeControl.CMSysColorChange(var Msg: TMessage);
begin
     inherited;
     Invalidate;
end;

procedure TRangeControl.CMCtl3DChanged(var Msg: TMessage);
begin
     inherited;
     Invalidate;
     Realign;
end;

procedure TRangeControl.WMSetFocus(var Message: TWMSetFocus);
begin
     inherited;
     Invalidate;
end;

procedure TRangeControl.WMKillFocus(var Message: TWMKillFocus);
begin
     inherited;
     Invalidate;
end;

procedure TRangeControl.SetState(Value: TRangeControlState);
begin
     if fState <> Value then
     begin
          fState := Value;
          Invalidate;
     end;
end;

function TRangeControl.GetInRange: Boolean;
begin
 Result := (State = rbInRange);
end;

procedure TRangeControl.SetInRange(Value: Boolean);
begin
 if Value then
   State := rbInRange
  else
    State := rbOutRange;
end;

procedure TRangeControl.SetInRangeBitmap(Value: TBitmap);
begin
 if Value = nil then
     fInRangeBitmap.LoadFromResourceName(HInstance, 'INRANGE')
 else
     fInRangeBitmap.Assign(Value);
 if (fInRangeBitmap.Width mod fInRangeBitmap.Height) = 0 then
     InRangeNumGlyphs := fInRangeBitmap.Width div fInRangeBitmap.Height;
 Invalidate;
end;

procedure TRangeControl.SetOutRangeBitmap(Value: TBitmap);
begin
 if Value = nil then
     fOutRangeBitmap.LoadFromResourceName(HInstance, 'OUTRANGE')
 else
     fOutRangeBitmap.Assign(Value);
 if (fOutRangeBitmap.Width mod fOutRangeBitmap.Height) = 0 then
     OutRangeNumGlyphs := fOutRangeBitmap.Width div fOutRangeBitmap.Height;
 Invalidate;
end;

procedure TRangeControl.SetNoCheckBitmap(Value: TBitmap);
begin
 if Value = nil then
     fNoCheckBitmap.LoadFromResourceName(HInstance, 'NOCHECK')
 else
     fNoCheckBitmap.Assign(Value);
 if (fNoCheckBitmap.Width mod fNoCheckBitmap.Height) = 0 then
     NoCheckNumGlyphs := fNoCheckBitmap.Width div fNoCheckBitmap.Height;
 Invalidate;
end;

procedure TRangeControl.SetInRangeNumGlyphs(Value: Integer);
begin
 if Value < 1 then fInRangeNumGlyphs := 1
 else if Value > 2 then fInRangeNumGlyphs := 2
 else fInRangeNumGlyphs := Value;
 Invalidate;
end;

procedure TRangeControl.SetOutRangeNumGlyphs(Value: Integer);
begin
 if Value < 1 then fOutRangeNumGlyphs := 1
 else if Value > 2 then fOutRangeNumGlyphs := 2
 else fOutRangeNumGlyphs := Value;
 Invalidate;
end;

procedure TRangeControl.SetNoCheckNumGlyphs(Value: Integer);
begin
 if Value < 1 then fNoCheckNumGlyphs := 1
 else if Value > 2 then fNoCheckNumGlyphs := 2
 else fNoCheckNumGlyphs := Value;
 Invalidate;
end;

procedure TRangeControl.SetRange(Value: TRange);
begin
 if Value <> fRange then begin
  fRange := Value;
  ValueInRange;
 end;
end;

procedure TRangeControl.ValueInRange;
begin
 if Assigned(fBeforeRangeCheck) then fBeforeRangeCheck(Self);
 if (ctMin in fRange) and (ctMax in fRange) then begin
  if (fCheckValue > fMinRangeValue) and (fCheckValue < fMaxRangeValue) then
   SetState(rbInRange)
    else SetState(rbOutRange);
 end
  else if (ctMax in fRange) then begin
        if (fCheckValue < fMaxRangeValue) then SetState(rbInRange)
         else SetState(rbOutRange);
       end
        else if (ctMin in fRange) then begin
              if (fCheckValue > fMinRangeValue) then SetState(rbInRange)
               else SetState(rbOutRange)
             end
              else SetState(rbNoCheck);
 if Assigned(fAfterRangeCheck) then fAfterRangeCheck(Self);
end;

procedure TRangeControl.SetMaxRangeValue(Value: Double);
begin
 if Value <> fMaxRangeValue then begin
  fMaxRangeValue := Value;
  ValueInRange;
 end;
end;

procedure TRangeControl.SetMinRangeValue(Value: Double);
begin
 if (Value <> fMinRangeValue) then begin
  fMinRangeValue := Value;
  ValueInRange;
 end;
end;

{******************************************************************************}
{** TRANGEBOX *****************************************************************}
{******************************************************************************}

procedure TRangeBox.SetCheckValue(Value: Double);
begin
 if Value <> fCheckValue then begin
  fCheckValue := Value;
  ValueInRange;
 end;
end;

{******************************************************************************}
{*** TDBRANGEBOX **************************************************************}
{******************************************************************************}

constructor TDBRangeBox.Create(AOwner: TComponent);
begin
 inherited Create(AOwner);
 fDataLink := TFieldDataLink.Create;
 fDataLink.OnDataChange := DataChange;
end;

destructor TDBRangeBox.Destroy;
begin
 fDataLink.OnDataChange := nil;
 fDataLink.Free;
 inherited Destroy;
end;

function TDBRangeBox.GetDataField: string;
begin
 Result := FDataLink.FieldName;
end;

function TDBRangeBox.GetDataSource: TDataSource;
begin
 Result := FDataLink.DataSource;
end;

procedure TDBRangeBox.SetDataField(const Value: string);
begin
 FDataLink.FieldName := Value;
end;

procedure TDBRangeBox.SetDataSource(Value: TDataSource);
begin
 FDataLink.DataSource := Value;
end;

procedure TDBRangeBox.DataChange(Sender: TObject);
begin
 if fDataLink.Field <> nil then begin
  fCheckValue := fDataLink.Field.AsFloat;
  ValueInRange;
 end;
end;

end.
