unit qrSwRect;

interface
Uses Graphics, Classes, Windows, QReport, QuickRpt;

type
  TQRShadowRect = class(TQRPrintable)
  private
    FBrush : TBrush;
    FPen : TPen;
    FCurve: Integer;
    FShadowWidth :Integer;
    FShadowBrush : TBrush;
    FRounded :Boolean;
    procedure SetShadowBrush(Value : TBrush);
    procedure SetShadowWidth(Value: Integer);
    procedure SetRounded(Value: Boolean);
    procedure SetBrush(Value : TBrush);
    procedure SetPen(Value : TPen);
    procedure SetCurve(Value: Integer);
  protected
    procedure Paint; override;
    procedure Print(OfsX, OfsY : integer); override;
    procedure StyleChanged(sender : TObject);
    procedure DefineProperties(Filer: TFiler); override;
    procedure ReadVisible(Reader : TReader); virtual;
    procedure WriteDummy(Writer : TWriter); virtual;
  public
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
  published
    property Brush : TBrush read FBrush write SetBrush;
    property Height default 65;
    property Pen : TPen read FPen write Setpen;
    property Width default 65;
    property Curve: Integer READ FCurve WRITE SetCurve;
    property ShadowBrush : TBrush read FShadowBrush write SetShadowBrush;
    property ShadowWidth: Integer READ FShadowWidth WRITE SetShadowWidth;
    property Rounded: Boolean READ FRounded WRITE SetRounded;
  end;


procedure Register;

implementation

constructor TQRShadowRect.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Width           := 100;
  Height          := 50;
  FCurve          := 10;
  FShadowWidth    :=4;
  FPen            := TPen.Create;
  FBrush          := TBrush.Create;
  FShadowBrush    := TBrush.Create;
  FBrush.OnChange := StyleChanged;
  FPen.OnChange   := StyleChanged;
  FShadowBrush.OnChange := StyleChanged;
  FShadowBrush.Color:=clGray;
  Rounded:=True;
end;

procedure TQRShadowRect.DefineProperties(Filer: TFiler);
begin
  Filer.DefineProperty('Visible', ReadVisible, WriteDummy, false); { <-- do not resource }
  INHERITED DefineProperties(Filer);
end;

procedure TQRShadowRect.ReadVisible(Reader : TReader);
begin
  Enabled := Reader.ReadBoolean;
end;

procedure TQRShadowRect.WriteDummy(Writer : TWriter);
begin
end;

procedure TQRShadowRect.StyleChanged(Sender : TObject);
begin
  Invalidate;
end;

procedure TQRShadowRect.SetBrush(Value: TBrush);
begin
  FBrush.Assign(Value);
end;

procedure TQRShadowRect.SetPen(Value: TPen);
begin
  FPen.Assign(Value);
end;

procedure TQRShadowRect.SetCurve(Value: Integer);
begin
  IF Value<>FCurve THEN BEGIN
     FCurve := Value;
     Invalidate;
  END;
end;

procedure TQRShadowRect.SetShadowBrush(Value: TBrush);
begin
  FShadowBrush.Assign(Value);
end;

procedure TQRShadowRect.SetShadowWidth(Value: Integer);
begin
  FShadowWidth:=Value;
  Invalidate;
end;

procedure TQRShadowRect.SetRounded(Value: Boolean);
begin
  FRounded:=Value;
  Invalidate;
end;

procedure TQRShadowRect.Paint;
begin
  inherited paint;
  with Canvas do begin
    Pen.Assign(FPen);
    Brush.Assign(FShadowBrush);
    Pen.Color:=Brush.Color;
    if FRounded then
     RoundRect(0+FShadowWidth,0+FShadowWidth,Width,Height,FCurve,FCurve)
    else
     Rectangle(0+FShadowWidth,0+FShadowWidth,Width,Height);

    Pen.Assign(FPen);
    Brush.Assign(FBrush);
    if FRounded then
      RoundRect(0,0,Width-FShadowWidth,Height-FShadowWidth,FCurve,FCurve)
    else
      Rectangle(0,0,Width-FShadowWidth,Height-FShadowWidth);
  end
end;

procedure TQRShadowRect.Print(OfsX,OfsY : Integer);
var
 DecalX,DecalY :Word;
begin
  IF ParentReport.FinalPass and Enabled then begin
     if FShadowWidth <> 0 then
      begin
        DecalX:=Round(QRPrinter.XFactor*FShadowWidth*2.54);
        DecalY:=Round(QRPrinter.YFactor*FShadowWidth*2.54);
      end
     else
      begin DecalX:=0; DecalY:=0; End;

     QRPrinter.Canvas.Brush := FShadowBrush;
     QRPrinter.Canvas.Pen.Color := FShadowBrush.Color;
     WITH QRPrinter DO BEGIN
        WITH Canvas DO
         If FRounded then
           RoundRect(XPos(OfsX + Size.Left+DecalX), YPos(OfsY + Size.Top+DecalY),
                     XPos(OfsX+Size.Left + Size.Width), YPos(OfsY + Size.Top + Size.Height),
                     Round(QRPrinter.XFactor*FCurve*2.54),
                     Round(QRPrinter.YFactor*FCurve*2.54))
         else
           Rectangle(XPos(OfsX + Size.Left+DecalX), YPos(OfsY + Size.Top+DecalY),
                     XPos(OfsX+Size.Left + Size.Width), YPos(OfsY + Size.Top + Size.Height));
     END;

     QRPrinter.Canvas.Brush := FBrush;
     QRPrinter.Canvas.Pen   := FPen;
     WITH QRPrinter DO BEGIN
        WITH Canvas DO
         If FRounded then
           RoundRect(XPos(OfsX + Size.Left), YPos(OfsY + Size.Top),
                     XPos(OfsX+Size.Left + Size.Width-DecalX), YPos(OfsY + Size.Top + Size.Height-DecalY),
                     Round(QRPrinter.XFactor*FCurve*2.54),
                     Round(QRPrinter.YFactor*FCurve*2.54))
         else
           Rectangle(XPos(OfsX + Size.Left), YPos(OfsY + Size.Top),
                     XPos(OfsX+Size.Left + Size.Width-DecalX), YPos(OfsY + Size.Top + Size.Height-DecalY));
     END;
  end;
end;

destructor TQRShadowRect.Destroy;
begin
  FPen.Free;
  FBrush.Free;
  FShadowBrush.Free;
  inherited Destroy;
end;

procedure Register;
begin
  RegisterComponents('QReportWWW', [TQRShadowRect]);
{$ifdef ver100}
  RegisterNonActiveX([TQRShadowRect],axrComponentOnly);
{$endif}
end;

end.
