{------------------------------------------------------------------------------}
UNIT XControl;     {                                last revision: Apr 15 1999 }
{------------------------------------------------------------------------------}
{
   TXTabbedListBox(TXCustomListBox)
     PROPERTY Header: THeader;
     PROPERTY ColWidths[Index: Longint]: INTEGER;
}
INTERFACE

USES
  SysUtils, WinTypes, Messages, Classes, Controls, Graphics, Forms,
  Menus, StdCtrls, ExtCtrls;

TYPE
  TXCustomListBox = CLASS(TCustomListBox)
    PRIVATE
      FOnChange          : TNotifyEvent;
      FOnSelectionChange : TNotifyEvent;
      FLastSel  : INTEGER;
      FReadOnly : BOOLEAN;
      FModified : BOOLEAN;
      PROCEDURE Click; OVERRIDE;
      FUNCTION  GetCaretIndex: INTEGER;
    PROTECTED
      PROCEDURE Change; DYNAMIC;                   { list contents has changed }
      PROCEDURE SelectionChanged; DYNAMIC;
      PROCEDURE SetUpdateState(Updating: BOOLEAN); DYNAMIC;
      PROCEDURE WndProc(VAR Message: TMessage); OVERRIDE;
    PUBLIC
      CONSTRUCTOR Create(AOwner : TComponent); OVERRIDE;
      PROCEDURE BeginUpdate;
      PROCEDURE EndUpdate;
      PROPERTY  OnChange  : TNotifyEvent READ FOnChange WRITE FOnChange;
      PROPERTY  OnSelectionChange : TNotifyEvent READ FOnSelectionChange WRITE FOnSelectionChange;
      PROPERTY  ReadOnly  : BOOLEAN READ FReadOnly WRITE FReadOnly DEFAULT FALSE;
      PROPERTY  Modified  : BOOLEAN READ FModified WRITE FModified DEFAULT FALSE;
      PROPERTY  CaretIndex: INTEGER READ GetCaretIndex;
    PUBLISHED
  END;

  TXListBox = CLASS(TXCustomListBox)
  PUBLISHED
    PROPERTY Align;
    PROPERTY BorderStyle;
    PROPERTY Color;
    PROPERTY Columns;
    PROPERTY Ctl3D;
    PROPERTY DragCursor;
    PROPERTY DragMode;
    PROPERTY Enabled;
    PROPERTY ExtendedSelect;
    PROPERTY Font;
    PROPERTY IntegralHeight;
    PROPERTY ItemHeight;
    PROPERTY Items;
    PROPERTY MultiSelect;
    PROPERTY ParentColor;
    PROPERTY ParentCtl3D;
    PROPERTY ParentFont;
    PROPERTY ParentShowHint;
    PROPERTY PopupMenu;
    PROPERTY ReadOnly;
    PROPERTY ShowHint;
    PROPERTY Sorted;
    PROPERTY Style;
    PROPERTY TabOrder;
    PROPERTY TabStop;
    PROPERTY Visible;
    PROPERTY OnChange;
    PROPERTY OnClick;
    PROPERTY OnDblClick;
    PROPERTY OnDragDrop;
    PROPERTY OnDragOver;
    PROPERTY OnDrawItem;
    PROPERTY OnEndDrag;
    PROPERTY OnEnter;
    PROPERTY OnExit;
    PROPERTY OnKeyDown;
    PROPERTY OnKeyPress;
    PROPERTY OnKeyUp;
    PROPERTY OnMeasureItem;
    PROPERTY OnMouseDown;
    PROPERTY OnMouseMove;
    PROPERTY OnMouseUp;
    PROPERTY OnSelectionChange;
  END{TXListBox};

{------------------------------------------------------------------------------}
CONST
  MaxColWidth = 32;                                 { maximum count of columns }
  MinShowWidth = 8;
                            { columns less then 8 pixel would not be displayed }
TYPE
  TXCustomTabbedListBox = CLASS(TXCustomListBox)
  PRIVATE
    FHeader  : THeader;
    FColCount: LONGINT;
    FColWidth: ARRAY[0..MaxColWidth-1] OF LONGINT;
    PROCEDURE SetHeader(AHeader: THeader);
    PROCEDURE DoSized(Sender: TObject; ASection, AWidth: Integer);
    PROCEDURE DoSizing(Sender: TObject; ASection, AWidth: Integer);
    FUNCTION  GetColWidth(Index: LONGINT): INTEGER;
    PROCEDURE SetColWidth(Index: LONGINT; ColWidth: INTEGER);
    FUNCTION  SubItem(CONST Str:STRING;Pos: INTEGER): STRING;
  PROTECTED
    PROCEDURE DrawItem(Index: Integer;Rect: TRect;State: TOwnerDrawState); OVERRIDE;
  PUBLIC
    CONSTRUCTOR Create(AOwner: TComponent); OVERRIDE;
    PROPERTY Header: THeader READ FHeader WRITE SetHeader;
    PROPERTY ColCount: LONGINT READ FColCount WRITE FColCount;
    PROPERTY ColWidths[Index: LONGINT]: INTEGER READ GetColWidth WRITE SetColWidth;
  END;

CONST Tab = CHAR(9);

TYPE
  TXTabbedListBox = CLASS(TXCustomTabbedListBox)
  PUBLISHED
    PROPERTY Align;
    PROPERTY BorderStyle;
    PROPERTY Color;
{    PROPERTY Columns; }
    PROPERTY ColCount; { new }
    PROPERTY Ctl3D;
    PROPERTY DragCursor;
    PROPERTY DragMode;
    PROPERTY Enabled;
    PROPERTY ExtendedSelect;
    PROPERTY Font;
    PROPERTY Header; { new }
    PROPERTY IntegralHeight;
    PROPERTY ItemHeight;
    PROPERTY Items;
    PROPERTY MultiSelect;
    PROPERTY ParentColor;
    PROPERTY ParentCtl3D;
    PROPERTY ParentFont;
    PROPERTY ParentShowHint;
    PROPERTY PopupMenu;
    PROPERTY ReadOnly;
    PROPERTY ShowHint;
    PROPERTY Sorted;
 {   PROPERTY Style; }
    PROPERTY TabOrder;
    PROPERTY TabStop;
    PROPERTY Visible;
    PROPERTY OnChange;
    PROPERTY OnClick;
    PROPERTY OnDblClick;
    PROPERTY OnDragDrop;
    PROPERTY OnDragOver;
    PROPERTY OnDrawItem;
    PROPERTY OnEndDrag;
    PROPERTY OnEnter;
    PROPERTY OnExit;
    PROPERTY OnKeyDown;
    PROPERTY OnKeyPress;
    PROPERTY OnKeyUp;
    PROPERTY OnMeasureItem;
    PROPERTY OnMouseDown;
    PROPERTY OnMouseMove;
    PROPERTY OnMouseUp;
    PROPERTY OnSelectionChange;
  END{TXTabbedListBox};

{------------------------------------------------------------------------------}

PROCEDURE Register;

IMPLEMENTATION

USES   WinProcs;

{------------------------------------------------------------------------------}
CONSTRUCTOR TXCustomListBox.Create(AOwner : TComponent);
BEGIN
  INHERITED Create(AOwner);
  FLastSel := -1;
  ReadOnly := FALSE;
  Modified := FALSE;
END{ Create };

{------------------------------------------------------------------------------}
PROCEDURE TXCustomListBox.Change;
BEGIN
  IF Assigned(FOnChange) THEN BEGIN
    FOnChange(SELF);
  END;
END{ Change };

{------------------------------------------------------------------------------}
PROCEDURE TXCustomListBox.SelectionChanged;
BEGIN
  FLastSel := ItemIndex;
  IF Assigned(FOnSelectionChange) THEN BEGIN
    FOnSelectionChange(SELF);
  END;
END{ SelectionChanged };

{------------------------------------------------------------------------------}
PROCEDURE TXCustomListBox.Click;
BEGIN
  INHERITED Click;
  IF (FLastSel <> ItemIndex) THEN BEGIN
    SelectionChanged;
  END;
END{ Click };

{------------------------------------------------------------------------------}
FUNCTION TXCustomListBox.GetCaretIndex: INTEGER;
BEGIN
  Result := SendMessage(Handle, LB_GETCARETINDEX, 0, 0);
END{ GetCaretIndex };

{------------------------------------------------------------------------------}
PROCEDURE TXCustomListBox.SetUpdateState(Updating: BOOLEAN);
BEGIN
  SendMessage(Handle, WM_SETREDRAW, ORD(NOT Updating), 0);
  IF NOT Updating THEN BEGIN
    Refresh;
  END;
END{ SetUpdateState };

{------------------------------------------------------------------------------}
PROCEDURE TXCustomListBox.BeginUpdate;
BEGIN
  SetUpdateState(TRUE);
END{ BeginUpdate };

{------------------------------------------------------------------------------}
PROCEDURE TXCustomListBox.EndUpdate;
BEGIN
  SetUpdateState(FALSE);
END{ EndUpdate};

{------------------------------------------------------------------------------}
PROCEDURE TXCustomListBox.WndProc(VAR Message: TMessage);
{
   capure all messages which can modify the listbox contents
}
BEGIN
       INHERITED WndProc(Message);
       IF ((Message.Msg = LB_ADDSTRING) OR
           (Message.Msg = LB_SETITEMDATA) OR
           (Message.Msg = LB_INSERTSTRING) OR
           (Message.Msg = LB_DELETESTRING) OR
           (Message.Msg = LB_RESETCONTENT)) THEN BEGIN
         IF NOT Modified THEN BEGIN
           Modified := TRUE;
         END;
         IF (Message.Msg = LB_DELETESTRING) AND (ItemIndex=-1) THEN BEGIN
           SelectionChanged;
         END;
         Change;
       END;
END{ Wndproc };

{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
CONSTRUCTOR TXCustomTabbedListBox.Create(AOwner : TComponent);
BEGIN
  INHERITED Create(AOwner);
  Style     := lbOwnerDrawFixed;
  FColCount := 1;
END{ Create };

{------------------------------------------------------------------------------}
PROCEDURE TXCustomTabbedListBox.SetHeader(AHeader: THeader);
VAR i: INTEGER;
BEGIN
  IF (FHeader<>NIL) THEN BEGIN                       { remove from old control }
    FHeader.OnSized  := NIL;
    FHeader.OnSizing := NIL;
  END;
  FHeader := AHeader;
  IF (FHeader<>NIL) THEN BEGIN
    FColCount := FHeader.Sections.Count;
    FOR i:=0 TO FColCount-1 DO BEGIN
      FColWidth[i] := FHeader.SectionWidth[i]
    END{for};
    Top   := FHeader.Top + FHeader.Height;
    Left  := FHeader.Left;
    Width := FHeader.Width;
    FHeader.OnSized  := DoSized;
    FHeader.OnSizing := DoSizing;
    IF (Visible) THEN BEGIN                                           { redraw }
      Invalidate;
    END;
  END;
END{ SetHeader };

{------------------------------------------------------------------------------}
PROCEDURE TXCustomTabbedListBox.DoSized(Sender: TObject; ASection, AWidth: Integer);
BEGIN
  FColWidth[ASection] := AWidth;
  IF (Visible) THEN BEGIN                                             { redraw }
    Invalidate;
  END;
END{ DoSized };

{------------------------------------------------------------------------------}
PROCEDURE TXCustomTabbedListBox.DoSizing(Sender: TObject; ASection, AWidth: Integer);
BEGIN
  FColWidth[ASection] := AWidth;
  IF (Visible) THEN BEGIN                                             { redraw }
    Invalidate;
  END;
END{ DoSizing };

{------------------------------------------------------------------------------}
FUNCTION  TXCustomTabbedListBox.GetColWidth(Index: LONGINT): INTEGER;
BEGIN
  Result :=  FColWidth[Index];
END{ GetColWidth };

{------------------------------------------------------------------------------}
PROCEDURE TXCustomTabbedListBox.SetColWidth(Index: LONGINT; ColWidth: INTEGER);
BEGIN
  FColWidth[Index] := ColWidth;
  IF Assigned(FHeader) THEN BEGIN
    FHeader.SectionWidth[Index] := ColWidth;
  END;
  IF (Visible) THEN BEGIN                                             { redraw }
    Invalidate;
  END;
END{ GetColWidth };

{------------------------------------------------------------------------------}
FUNCTION TXCustomTabbedListBox.SubItem(CONST Str:STRING;Pos: INTEGER): STRING;
{
  calculates the nth section
  Pos = 0..n-1
}
VAR i,p: INTEGER;
BEGIN
  i := 1;
  p := Pos;
  WHILE (i<=LENGTH(Str)) AND (p>0) DO BEGIN
    IF (Str[i]=Tab) THEN BEGIN
      DEC(p);
    END;
    INC(i);
  END{while};
  Result := '';
  p := 0;
  WHILE (i<=LENGTH(Str)) AND (Str[i]<>Tab) DO BEGIN
    INC(p);
    Result := Result + Str[i];
    INC(i);
  END{while};
  SetLength(Result,p);
END{ SubItem };

{------------------------------------------------------------------------------}
PROCEDURE TXCustomTabbedListBox.DrawItem(Index: Integer;Rect: TRect;State: TOwnerDrawState);
VAR Rc: TRect;
    i : INTEGER;
BEGIN
  Canvas.FillRect(Rect);
  IF (Index<Items.Count) THEN BEGIN
    Rc := Rect;
    i := 0;
    WHILE (i<FColCount) AND (Rc.Left<Rect.Right) DO BEGIN
      Rc.Right := Rc.Left + FColWidth[i];

      IF (FColWidth[i]>=MinShowWidth) THEN BEGIN
        Canvas.TextRect(Rc,Rc.Left,Rc.Top,SubItem(Items[Index],i));
      END;
      Rc.Left := Rc.Right;
      INC(i);
    END{while};
  END;                                  
END{ DrawItem };

{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}

PROCEDURE Register;
BEGIN
  RegisterComponents('XControls',[TXListBox,TXTabbedListBox]);
END;

END.
