unit PowerArc;

{| PowerArc 1.2.1  /1 Mar 2001/
 | Copyright (c) 2000 SoftLab, Andre N Belokon
 | Web    http://softlab.od.ua
 | Email  support@softlab.od.ua
 | Data compression library for Delphi and C++ Builder
 |}

interface

uses SysUtils, Windows, Classes;

const
  PowerArcModuleSignature = 'AA6F3C60-37D7-11D4-B4BF-D80DBEC04C01';

type
  EPowerArcError = class(Exception);
  TPowerArcModuleInfo = packed record
    Signature:   PChar; // must be eq to PowerArcModuleSignature
    Name:        PChar; // short name
    Description: PChar; // full description
    Options:     PChar; // opt list delimited with #0
    // bit per char on calgary corpus *100
    DefaultBPC:  integer;
    MaxBPC:      integer;
    case integer of     // unique
      0: ( ModuleID:    packed array[0..7] of Char );
      1: ( ModuleIDW:   packed array[0..1] of integer );
  end;
  PPowerArcModuleInfo = ^TPowerArcModuleInfo;

  TProgressCallback = procedure (Total, Current: integer) of object;

  TReadFunc = function (Data: Pointer; var Buffer; Size: integer): integer; stdcall;
  TWriteFunc = function (Data: Pointer; const Buffer; Size: integer): integer; stdcall;

  TPowerArcSetOptions =  procedure (Opt: PChar); stdcall;
  TPowerArcCompress =    procedure (Data: Pointer; Opt: PChar; ReadFunc: TReadFunc;
                                    WriteFunc: TWriteFunc); stdcall;
  TPowerArcCompressMem = procedure (Data: Pointer; Opt: PChar; Mem: Pointer;
                                    MemSize: integer; WriteFunc: TWriteFunc); stdcall;
  TPowerArcDecompress =  function (Data: Pointer; ReadFunc: TReadFunc;
                                   WriteFunc: TWriteFunc): Boolean; stdcall;

  TPowerArcModule = record
    Name:        string;
    hLib:        THandle;
    Info:        PPowerArcModuleInfo;
    Options:     TStringList;
    SetOptions:  TPowerArcSetOptions;
    Compress:    TPowerArcCompress;
    CompressMem: TPowerArcCompressMem;
    Decompress:  TPowerArcDecompress;
  end;

var
  PowerArcModules: array of TPowerArcModule;
  iPowerZIP: integer = -1;
  iPowerBZIP: integer = -1;
  iPowerRANK: integer = -1;
  iPowerPPM: integer = -1;

function RegisterPowerArcModule(const Name: string): integer;

procedure SetOptions(ArcIdx: integer; const ArcOpt: string);

function Compress(ArcIdx: integer;
  InStream,OutStream: TStream; const ArcOpt: string = '';
  ProgressCallback: TProgressCallback = nil): Boolean; overload;

function Compress(ArcIdx: integer; const Buffer; Size: integer;
  OutStream: TStream; const ArcOpt: string = ''): Boolean; overload;

function Decompress(InStream,OutStream: TStream;
  ProgressCallback: TProgressCallback = nil): Boolean;

//======================== Stream interface ==============================

type   
  TPowerArcCompressStream = class(TStream)
  private
    Base: TStream;
    ArcIdx: integer;
    ArcOpt: string;
    Thread: TThread;
    hReadPipe,
    hWritePipe: THandle;
  public
    constructor Create(BaseStream: TStream; FArcIdx: integer;
      const FArcOpt: string = '');
    destructor Destroy; override;
    function Read(var Buffer; Count: Longint): Longint; override;
    function Write(const Buffer; Count: Longint): Longint; override;
    function Seek(Offset: Longint; Origin: Word): Longint; override;
  end;

  TPowerArcDecompressStream = class(TStream)
  private
    Base: TStream;
    Thread: TThread;
    hReadPipe,
    hWritePipe: THandle;
  public
    constructor Create(BaseStream: TStream);
    destructor Destroy; override;
    function Read(var Buffer; Count: Longint): Longint; override;
    function Write(const Buffer; Count: Longint): Longint; override;
    function Seek(Offset: Longint; Origin: Word): Longint; override;
  end;

implementation

type
  TPowerArcData = record
    InStream,OutStream: TStream;
    Total,Current: integer;
    ProgressCallback: TProgressCallback;
  end;

function ReadFunc(Data: Pointer; var Buffer; Size: integer): integer; stdcall;
begin
  Result:=TPowerArcData(Data^).InStream.Read(Buffer,Size);
  if Assigned(TPowerArcData(Data^).ProgressCallback) then begin
    Inc(TPowerArcData(Data^).Current,Result);
    TPowerArcData(Data^).ProgressCallback(TPowerArcData(Data^).Total,
                                          TPowerArcData(Data^).Current);
  end;
end;

function WriteFunc(Data: Pointer; const Buffer; Size: integer): integer; stdcall;
begin
  Result:=TPowerArcData(Data^).OutStream.Write(Buffer,Size);
end;

function ValidArcIdx(ArcIdx: integer): Boolean;
begin
  Result:=(ArcIdx >= 0) and (ArcIdx < Length(PowerArcModules));
end;

procedure SetOptions(ArcIdx: integer; const ArcOpt: string);
begin
  if ValidArcIdx(ArcIdx) then
    PowerArcModules[ArcIdx].SetOptions(PChar(ArcOpt));
end;

function Compress(ArcIdx: integer;
  InStream,OutStream: TStream; const ArcOpt: string;
  ProgressCallback: TProgressCallback): Boolean;
var Data: TPowerArcData;
begin
  Result:=False;
  if ValidArcIdx(ArcIdx) then try
    Data.InStream:=InStream;
    Data.OutStream:=OutStream;
    Data.ProgressCallback:=ProgressCallback;
    if Assigned(ProgressCallback) then
      Data.Total:=InStream.Size;
    Data.Current:=0;
    OutStream.Write(PowerArcModules[ArcIdx].Info^.ModuleID[0],8);
    PowerArcModules[ArcIdx].Compress(@Data,PChar(ArcOpt),ReadFunc,WriteFunc);
    Result:=True;
  except
  end;
end;

function Compress(ArcIdx: integer;
  const Buffer; Size: integer; OutStream: TStream; const ArcOpt: string): Boolean;
var Data: TPowerArcData;
begin
  Result:=False;
  if ValidArcIdx(ArcIdx) then try
    Data.OutStream:=OutStream;
    OutStream.Write(PowerArcModules[ArcIdx].Info^.ModuleID[0],8);
    PowerArcModules[ArcIdx].CompressMem(@Data,PChar(ArcOpt),@Buffer,Size,WriteFunc);
    Result:=True;
  except
  end;
end;

function Decompress(InStream,OutStream: TStream;
  ProgressCallback: TProgressCallback): Boolean;
var ModuleID: packed array[0..7] of Char;
    j: integer;
    Data: TPowerArcData;
begin
  Result:=False;
  InStream.Read(ModuleID[0],8);
  for j:=0 to Length(PowerArcModules)-1 do
    if PowerArcModules[j].Info^.ModuleID = ModuleID then begin
      Data.InStream:=InStream;
      Data.OutStream:=OutStream;
      Data.ProgressCallback:=ProgressCallback;
      if Assigned(ProgressCallback) then
        Data.Total:=InStream.Size;
      Data.Current:=0;
      PowerArcModules[j].Decompress(@Data,ReadFunc,WriteFunc);
      Result:=True;
      Exit;
    end;
end;

function RegisterPowerArcModule(const Name: string): integer;
type TGetPowerArcModuleInfo = function: PPowerArcModuleInfo;
var PowerArcModule: TPowerArcModule;
    GetPowerArcModuleInfo: TGetPowerArcModuleInfo;
    POpt: PChar;
begin
  Result:=-1;
  PowerArcModule.hLib:=LoadLibrary(PChar(Name));
  if PowerArcModule.hLib <> 0 then begin
    PowerArcModule.Name:=Name;
    GetPowerArcModuleInfo:=TGetPowerArcModuleInfo(GetProcAddress(PowerArcModule.hLib,
      'GetPowerArcModuleInfo'));
    PowerArcModule.Info:=GetPowerArcModuleInfo;
    PowerArcModule.SetOptions:=TPowerArcSetOptions(GetProcAddress(PowerArcModule.hLib,'SetOptions'));
    PowerArcModule.Compress:=TPowerArcCompress(GetProcAddress(PowerArcModule.hLib,'Compress'));
    PowerArcModule.CompressMem:=TPowerArcCompressMem(GetProcAddress(PowerArcModule.hLib,'CompressMem'));
    PowerArcModule.Decompress:=TPowerArcDecompress(GetProcAddress(PowerArcModule.hLib,'Decompress'));
    if Assigned(GetPowerArcModuleInfo) and
       (PowerArcModule.Info^.Signature = PowerArcModuleSignature) and
       Assigned(PowerArcModule.SetOptions) and
       Assigned(PowerArcModule.Compress) and
       Assigned(PowerArcModule.CompressMem) and
       Assigned(PowerArcModule.Decompress) then begin
      PowerArcModule.Options:=TStringList.Create;
      POpt:=PowerArcModule.Info^.Options;
      while POpt^ <> #0 do begin
        PowerArcModule.Options.Add(POpt);
        POpt:=POpt+StrLen(POpt)+1;
      end;
      SetLength(PowerArcModules,Length(PowerArcModules)+1);
      PowerArcModules[Length(PowerArcModules)-1]:=PowerArcModule;
      Result:=Length(PowerArcModules)-1;
    end else
      FreeLibrary(PowerArcModule.hLib);
  end;
end;

procedure UnregisterPowerArcModules;
var j: integer;
begin
  for j:=0 to Length(PowerArcModules)-1 do
    FreeLibrary(PowerArcModules[j].hLib);
  PowerArcModules:=nil;
end;

{ TCompressThread }

type
  TCompressThread = class(TThread)
  private
    Done: Boolean;
    CompressStream: TPowerArcCompressStream;
  protected
    procedure Execute; override;
  end;

{ TCompressThread }

function ReadCompressFunc(Data: Pointer; var Buffer; Size: integer): integer; stdcall;
begin
  if not Windows.ReadFile(TPowerArcCompressStream(Data).hReadPipe,Buffer,Size,DWORD(Result),nil) then
    Result:=-1;
end;

function WriteCompressFunc(Data: Pointer; const Buffer; Size: integer): integer; stdcall;
begin
  Result:=TPowerArcCompressStream(Data).Base.Write(Buffer,Size);
end;

procedure TCompressThread.Execute;
begin
  try
    CompressStream.Base.Write(PowerArcModules[CompressStream.ArcIdx].Info^.ModuleID[0],8);
    PowerArcModules[CompressStream.ArcIdx].Compress(CompressStream,
      PChar(CompressStream.ArcOpt),ReadCompressFunc,WriteCompressFunc);
  except
  end;
  CloseHandle(CompressStream.hReadPipe);
  Done:=True;
end;

{ TPowerArcCompressStream }

constructor TPowerArcCompressStream.Create(BaseStream: TStream;
  FArcIdx: integer; const FArcOpt: string);
begin
  inherited Create;
  Base:=BaseStream;
  ArcIdx:=FArcIdx;
  ArcOpt:=FArcOpt;
  Thread:=nil;
  if not ValidArcIdx(ArcIdx) then
    raise EPowerArcError.Create('Invalid acrhive index');
end;

destructor TPowerArcCompressStream.Destroy;
begin
  if Thread <> nil then begin
//    CloseHandle(hReadPipe);
    CloseHandle(hWritePipe);
    while not TCompressThread(Thread).Done do Sleep(0);
    Thread.Free;
  end;
  inherited;
end;

function TPowerArcCompressStream.Read(var Buffer; Count: Integer): Longint;
begin
  raise EPowerArcError.Create('Invalid stream operation');
end;

function TPowerArcCompressStream.Seek(Offset: Integer;
  Origin: Word): Longint;
begin
  raise EPowerArcError.Create('Invalid stream operation');
end;

function TPowerArcCompressStream.Write(const Buffer;
  Count: Integer): Longint;
var Ret: Boolean;
    ActualWrite: DWORD;
    P: PChar;
begin
  if Count > 0 then begin
    if Thread = nil then begin
      CreatePipe(hReadPipe,hWritePipe,nil,4096);
      Thread:=TCompressThread.Create(True);
      TCompressThread(Thread).CompressStream:=Self;
      TCompressThread(Thread).Done:=False;
      Thread.FreeOnTerminate:=False;
      Thread.Resume;
    end;
    //Windows.WriteFile(hWritePipe,Buffer,Count,DWORD(Result),nil);
    Result:=0;
    P:=PChar(@Buffer);
    while Count > 0 do begin
      Ret:=Windows.WriteFile(hWritePipe,P^,Count,ActualWrite,nil);
      if not Ret or (Ret and (ActualWrite = 0)) then begin
        Result:=-1;
        Exit;
      end;
      Dec(Count,ActualWrite);
      Inc(Result,ActualWrite);
      Inc(P,ActualWrite);
      Sleep(0);
    end;
  end else
    Result:=0;
end;

{ TDecompressThread }

type
  TDecompressThread = class(TThread)
  private
    Done: Boolean;
    DecompressStream: TPowerArcDecompressStream;
  protected
    procedure Execute; override;
  end;

{ TDecompressThread }

function ReadDecompressFunc(Data: Pointer; var Buffer; Size: integer): integer; stdcall;
begin
  Result:=TPowerArcDecompressStream(Data).Base.Read(Buffer,Size);
end;

function WriteDecompressFunc(Data: Pointer; const Buffer; Size: integer): integer; stdcall;
begin
  if not Windows.WriteFile(TPowerArcDecompressStream(Data).hWritePipe,Buffer,Size,DWORD(Result),nil) then
    Result:=-1;
end;

procedure TDecompressThread.Execute;
var ModuleID: packed array[0..7] of Char;
    j: integer;
begin
  try
    DecompressStream.Base.Read(ModuleID[0],8);
    for j:=0 to Length(PowerArcModules)-1 do
      if PowerArcModules[j].Info^.ModuleID = ModuleID then begin
        PowerArcModules[j].Decompress(DecompressStream,
          ReadDecompressFunc,WriteDecompressFunc);
        Break;
      end;
  except
  end;
  CloseHandle(DecompressStream.hWritePipe);
  Done:=True;
end;

{ TPowerArcDecompressStream }

constructor TPowerArcDecompressStream.Create(BaseStream: TStream);
begin
  inherited Create;
  Base:=BaseStream;
  Thread:=nil;
end;

destructor TPowerArcDecompressStream.Destroy;
begin
  if Thread <> nil then begin
    CloseHandle(hReadPipe);
    while not TDecompressThread(Thread).Done do Sleep(0);
    Thread.Free;
  end;
  inherited;
end;

function TPowerArcDecompressStream.Read(var Buffer;
  Count: Integer): Longint;
var Ret: Boolean;
    ActualRead: DWORD;
    P: PChar;
begin
  if Count > 0 then begin
    if Thread = nil then begin
      CreatePipe(hReadPipe,hWritePipe,nil,4096);
      Thread:=TDecompressThread.Create(True);
      TDecompressThread(Thread).DecompressStream:=Self;
      TDecompressThread(Thread).Done:=False;
      Thread.FreeOnTerminate:=False;
      Thread.Resume;
    end;
    Result:=0;
    P:=PChar(@Buffer);
    while Count > 0 do begin
      Ret:=Windows.ReadFile(hReadPipe,P^,Count,ActualRead,nil);
      if not Ret or (Ret and (ActualRead = 0)) then begin
        Result:=-1;
        Exit;
      end;
      Dec(Count,ActualRead);
      Inc(Result,ActualRead);
      Inc(P,ActualRead);
      Sleep(0);
    end;
  end else
    Result:=0;
end;

function TPowerArcDecompressStream.Seek(Offset: Integer;
  Origin: Word): Longint;
begin
  raise EPowerArcError.Create('Invalid stream operation');
end;

function TPowerArcDecompressStream.Write(const Buffer;
  Count: Integer): Longint;
begin
  raise EPowerArcError.Create('Invalid stream operation');
end;

initialization
  iPowerRANK:=RegisterPowerArcModule('PowerRANK.dll');
  iPowerZIP:=RegisterPowerArcModule('PowerZIP.dll');
  iPowerBZIP:=RegisterPowerArcModule('PowerBZIP.dll');
  iPowerPPM:=RegisterPowerArcModule('PowerPPM.dll');
finalization
  UnregisterPowerArcModules;
end.
