{ ##
  @FILE                     PJHotLabel.pas
  @COMMENTS                 Hot Label Component source file.
  @PROJECT_NAME             Hot Label Component
  @PROJECT_DESC             Label component that can access a URL when the label
                            is clicked.
  @OWNER                    delphiDabbler
  @AUTHOR                   Peter Johnson, LLANARTH, Ceredigion, Wales, UK
  @EMAIL                    peter.johnson@openlink.org
  @WEBSITE                  http://www.delphidabbler.com/
  @COPYRIGHT                 Peter D Johnson, 1999-2003.
  @LEGAL_NOTICE             This component and its source code are placed in the
                            public domain. It may be freely copied and
                            circulated on a not-for-profit basis providing that
                            the code is unmodified and this notice and
                            information about the author and his copyright
                            remains attached to the source code.
  @OTHER_NAMES              + Original unit name was HotLabel.pas
                            + Changed to PJHotLabel.pas at v2.0
  @HISTORY(
    @REVISION(
      @VERSION              1.0
      @DATE                 24/10/1999
      @COMMENTS             Original version.
    )
    @REVISION(
      @VERSION              2.0
      @DATE                 02/11/2003
      @COMMENTS             + Added separate URL property to enable caption to
                              be different to URL accessed.
                            + Added ability to highlight label text when cursor
                              passes over it.
                            + Added ability to display URL in hint or to display
                              custom hints set by handling the new OnCustomHint
                              event.
                            + Now uses built in hand point cursor for Delphi 4
                              and higher while still loading custom cursor from
                              resources for earlier compilers. Value of custom
                              cursor constant changed from $0A to $630B.
                            + Moved error messages to resource strings for
                              Delphi 3 and later while using constants for
                              Delphi 2.
                            + Added support for https:// protocol.
                            + Changed default URL to http://localhost/
    )
  )
}


unit PJHotLabel;


interface


// Determine conditional symbols based on compiler
{$IFDEF VER90}    // --- Delphi 2
  {$DEFINE NO_RES_STRINGS}
  {$DEFINE NO_HAND_CURSOR}
{$ENDIF}
{$IFDEF VER100}   // --- Delphi 3
  {$DEFINE NO_HAND_CURSOR}
{$ENDIF}


uses
  // Delphi
  SysUtils, Classes, Graphics, Messages, Controls, StdCtrls;


const
  // We define hand cursor (not supported natively by all Delphi versions):
  // + where hand cursor is defined in Delphi, we use it;
  // + where Delphi doesn't provide cursor we use provided custom cursor and
  //   define crHandPoint constant.
  // note crHand provided for backwards compatibility: we now use crHandPoint.
  {$IFDEF NO_HAND_CURSOR}
    // no built in hand cursor: define crHand as custom cursor: also define
    //   crHandPoint alias
    crHand = $630B;
    crHandPoint = crHand;
  {$ELSE}
    // crHandPoint defined by Delphi: crHand as same as crHandPoint
    crHand = crHandPoint;
  {$ENDIF}


type

  {
  EPJURLError:
    Exception triggered by TPJHotLabel when URL problems encountered.
  }
  EPJURLError = class(Exception);


  {
  TPJHLHintStyle:
    The various hint styles used by TPJHotLabel.
  }
  TPJHLHintStyle = (
    hsNormal,   // display normal hints per Hint property
    hsURL,      // display URL property in hint and ignore Hint property
    hsCustom    // display a custom hint by handling OnCustomHint event
                //   if event not handled Hint property is used
  );


  {
  TPJHLCustomHintEvent:
    Type of event triggered just before hint is displayed when HintStyle
    property is set to hsCustom.
  }
  TPJHLCustomHintEvent = procedure(Sender: TObject; var HintStr: string)
    of object;


  {
  TPJHotLabel:
    Label component that can use its caption text as a hotlink to default web
    browser or e-mail program.
  }
  TPJHotLabel = class(TLabel)
  private // properties
    fValidateURL: Boolean;
    fURL: string;
    fCaptionIsURL: Boolean;
    fHighlightFont: TFont;
    fHighlightURL: Boolean;
    fHintStyle: TPJHLHintStyle;
    fOnCustomHint: TPJHLCustomHintEvent;
    procedure SetValidateURL(const Value: Boolean);
    procedure SetURL(const Value: string);
    procedure SetCaptionIsURL(const Value: Boolean);
    procedure SetHighlightFont(const Value: TFont);
  protected // properties
    procedure SetCaption(const Value: TCaption); virtual;
    function GetCaption: TCaption; virtual;
    function GetFont: TFont; virtual;
    procedure SetFont(const Value: TFont); virtual;
  private
    fBackupFont: TFont;
      {Font used to store Font property while it is being used to display
      highlight font}
    fBackupFontUsed: Boolean;
      {Flag set true when backup font is storing font property: used to enable
      actual (backup of) Font value to be accessed when Font property is being
      used to display highlight}
    procedure CMMouseEnter(var Msg: TMessage); message CM_MOUSEENTER;
      {Handles mouse enter event and highlight label text if required}
    procedure CMMouseLeave(var Msg: TMessage); message CM_MOUSELEAVE;
      {Handles mouse leave event and un-highlight label text if required}
    procedure CMHintShow(var Msg: TMessage); message CM_HINTSHOW;
      {Message handler that intercepts hints before they are displayed and
      sets hint as required by the HintStyle property. Triggers OnCustomHint
      event when HintStyle = hsCustom}
  protected
    procedure SetDefaultURL; virtual;
      {Sets the URL to the default value. Override this method in descendant
      classes to change this default}
    procedure CheckURL(const URL: string); virtual;
      {Checks the given URL for validity (i.e. has file:, http://, https://,
      ftp:// or mailto: protocols) and raises exception if URL is not for one of
      these protocols. Over-ride this method to add additional protocols in a
      descendant component}
    procedure Click; override;
      {Overrides inherited Click method to start default browser or e-mail
      client when label clicked and Enabled property is true. Also performs
      default click processing for label}
    procedure Loaded; override;
      {Loaded property called after component has been loaded: ensures Caption
      matches URL property if CaptionIsURL property is true}
  public
    constructor Create(AOwner: TComponent); override;
      {Class constructor: creates owned objects, loads cursor if required and
      sets required defaults}
    destructor Destroy; override;
      {Class destructor: frees owned objects}
  published
    // New properties
    property CaptionIsURL: Boolean
      read fCaptionIsURL write SetCaptionIsURL default True;
      {When true the Caption displays the URL per the URL property. Setting the
      Caption updates the URL property and vice versa}
    property HighlightFont: TFont read fHighlightFont write SetHighlightFont;
      {Font displayed by label when cursor is over the component. This font is
      only used if HighlightURL property is true}
    property HighlightURL: Boolean
      read fHighlightURL write fHighlightURL default False;
      {When true the HighlightFont is displayed by the label when the mouse
      cursor moves over it. When this property is false the label is not
      highlighted and HighlightFont is ignored}
    property HintStyle: TPJHLHintStyle
      read fHintStyle write fHintStyle default hsNormal;
      {Determines how the component's hint is displayed: this can be displayed
      as normal (using the Hint property), display the URL or display a custom
      hint message set by handling the OnCustomHint event}
    property URL: string
      read fURL write SetURL;
      {Stores the URL to be accessed when the label is clicked}
    property ValidateURL: Boolean
      read fValidateURL write SetValidateURL default True;
      {Causes URL property to be checked for valid URLs when true. Setting this
      property true when there is an invalid URL in the URL property raises an
      exception and restores the default URL}
    property OnCustomHint: TPJHLCustomHintEvent
      read fOnCustomHint write fOnCustomHint;
      {Event triggered just before the component's hint is displayed when the
      HintStyle property is hsCustom. The user can set the hint that is
      displayed by chanfing the value of the HintStr parameter to the event
      handler. HintStr is set to Hint before triggering the event}
    // Overridden inherited properties
    property Caption: TCaption read GetCaption write SetCaption;
      {Caption property: changed so that, when CaptionIsURL is true, the
      property is equivalent to the URL property - i.e. the value is validated
      when ValidateURL is true and updating Caption also updates URL and vice
      versa}
    property Cursor default crHandPoint;
      {Change default cursor of inherited Cursor property from crDefault to
      crHandPoint (which is equivalent crHand)}
    property Font: TFont
      read GetFont write SetFont;
      {Font property is used to display both normal and highlighted text (by
      taking on value of the HighlightFont property when label is highlighted).
      We need to override the read and write accessor methods to ensure that the
      original attributes of the Font property are always accessible for reading
      and update even when the Font property has beeb assigned the HighlightFont
      attributes. This is done by keeping a backup of Font and accessing the
      backup when the highlight is active}
    property ParentFont default False;
      {ParentFont property is set false since the default caption font style is
      different to the parent font}
  end;


procedure Register;
  {Component registration routine}


implementation


uses
  // Delphi
  Windows, ShellAPI, Forms;


// Only link this resource for Delphi version with no built in hand point cursor
{$IFDEF NO_HAND_CURSOR}
  {$R PJHotLabel.res}
{$ENDIF}


const
  cDefaultURL = 'http://localhost/';
    {URL used as default caption property by SetDefaultURL method}


// Message strings: uses resource strings where supported, and const if not
{$IFDEF NO_RES_STRINGS}
  const
{$ELSE}
  resourcestring
{$ENDIF}
  sCantAccessURL = 'Can''t access URL "%s"';
  sBadProtocol = 'Protocol not recognised for URL "%s"';


{ Component registration }

procedure Register;
  {Registers the component}
begin
  RegisterComponents('DelphiDabbler', [TPJHotLabel]);
end;


{ TPJHotLabel }

procedure TPJHotLabel.CheckURL(const URL: string);
  {Checks the given URL for validity (i.e. has file:, http://, https://, ftp://
  or mailto: protocols) and raises exception if URL is not for one of these
  protocols. Over-ride this method to add additional protocols in a descendant
  component}
const
  // List of recognised URLs
  cValidURLS: array[1..5] of string
    = ('http://', 'https://', 'mailto', 'file:', 'ftp://');
var
  I: Integer; // loops thru all recognised URLs
begin
  // Allow '': we don't ever try to jump to URL!
  if URL = '' then Exit;
  // Check caption against array of valid protocols: exit if we recognise one
  for I := Low(cValidURLS) to High(cValidURLS) do
    if CompareText(cValidURLS[I],Copy(URL, 1, Length(cValidURLS[I]))) = 0 then
      Exit;
  // If we get here, we haven't recognised the protocol so raise exception
  raise EPJURLError.CreateFmt(sBadProtocol, [URL]);
end;

procedure TPJHotLabel.Click;
  {Overrides inherited Click method to start default browser or e-mail client
  when label clicked and Enabled property is true. Also performs default click
  processing for label}
var
  URL: string;  // the URL we're going to access
begin
  // Get URL
  URL := fURL;
  // Only try to access URL if it's not empty string and component is enabled
  if Enabled and (URL <> '')then
  begin
   // Get windows to decode URL and execute browser or e-mail client etc
    if ShellExecute(ValidParentForm(Self).Handle, nil, PChar(URL), nil,
      nil, SW_SHOW) < 32 then
      // and raise excpetion if there was a problem
      raise EPJURLError.CreateFmt(sCantAccessURL, [URL]);
  end;
  // Now, if we got thru all that successfully, call inherited click stuff
  inherited Click;
end;

procedure TPJHotLabel.CMHintShow(var Msg: TMessage);
  {Message handler that intercepts hints before they are displayed and sets hint
  as required by the HintStyle property. Triggers OnCustomHint event when
  HintStyle = hsCustom}
var
  HintStr: string;  // the hint to be displayed when custom hints being used
begin
  // Change hint to URL if ShowURLInHint
  case fHintStyle of
    hsNormal:
      // Hint property displays as normal
      {Do nothing};
    hsURL:
      // Display URL in hint
      PHintInfo(Msg.LParam)^.HintStr := fURL;
    hsCustom:
    begin
      // Custom hint: user sets in event handler: if not handled display Hint
      // set default hint text to Hint property
      HintStr := Hint;
      // trigger event if assigned
      if Assigned(fOnCustomHint) then
        fOnCustomHint(Self, HintStr);
      // store new hint text
      PHintInfo(Msg.LParam)^.HintStr := HintStr;
    end;
  end;
  // Call inherited handler
  inherited;
end;

procedure TPJHotLabel.CMMouseEnter(var Msg: TMessage);
  {Handles mouse enter event and highlight label text if required}
begin
  inherited;
  if HighlightURL then
  begin
    // Highlight the label by using the highlight font, saving Font
    fBackupFont.Assign(Font);
    // Note that Font is backed up and being used for highligh
    fBackupFontUsed := True;
    inherited Font.Assign(HighlightFont);
  end;
end;

procedure TPJHotLabel.CMMouseLeave(var Msg: TMessage);
  {Handles mouse leave event and un-highlight label text if required}
begin
  inherited;
  if HighlightURL then
  begin
    // Un-highlight the label by restoring saved Font property
    inherited Font.Assign(fBackupFont);
    // Note that font is no longer backed up: it has been restored
    fBackupFontUsed := False;
  end;
end;

constructor TPJHotLabel.Create(AOwner: TComponent);
  {Class constructor: creates owned objects, loads cursor if required and sets
  required defaults}
begin
  inherited Create(AOwner);

  // Where hand cursor not built in: load hand cursor from resources
  {$IFDEF NO_HAND_CURSOR}
    Screen.Cursors[crHand] := LoadCursor(HInstance, 'HOTLABEL_HANDCUR');
  {$ENDIF}
  // Set hand point cursor as default
  Cursor := crHandPoint;

  // Create backup and highlight fonts
  fBackupFont := TFont.Create;
  fHighlightFont := TFont.Create;

  // Set default values
  // set default Font and HighlightFont colour and style
  Font.Color := clNavy;         // sets ParentFont to False
  Font.Style := [fsUnderline];
  HighlightFont := Font;
  HighlightFont.Color := clRed;
  // we validate URLs by default
  SetValidateURL(True);
  // we use caption for URL by default (for backwards compatibility)
  fCaptionIsURL := True;
  // set default URL and caption to default URL
  // (caption set to URL since CaptionIsURL defaults to true)
  SetDefaultURL;
  // we don't highlight URL by default (for backwards compatibility)
  fHighlightURL := False;
  // we display hints normally
  fHintStyle := hsNormal;
  inherited Caption := fURL;
end;

destructor TPJHotLabel.Destroy;
  {Class destructor: frees owned objects}
begin
  fHighlightFont.Free;
  fBackupFont.Free;
  inherited;
end;

function TPJHotLabel.GetCaption: TCaption;
  {"Overridden" read access method for Caption property - references inherited
  Caption property}
begin
  Result := inherited Caption;
end;

function TPJHotLabel.GetFont: TFont;
  {"Overridden" read access method for Font property: returns Font property
  value from one of two places according to whether Font is in use to display
  highlight}
begin
  if fBackupFontUsed then
    // Font being used to display highlight: return its value from backup
    Result := fBackupFont
  else
    // Font has its true value: return inherited value
    Result := inherited Font;
end;

procedure TPJHotLabel.Loaded;
  {Loaded property called after component has been loaded: ensures Caption
  matches URL property if CaptionIsURL property is true}
begin
  inherited;
  if fCaptionIsURL then
    SetCaption(fURL);
end;

procedure TPJHotLabel.SetCaption(const Value: TCaption);
  {"Overridden" write access method for Caption property: if CaptionIsURL
  property is true then checks Value is a valid URL and updated URL property to
  same value. If CaptionIsURL is false then we simply set caption with no
  checks}
begin
  if fCaptionIsURL then
  begin
    // Caption is a URL
    // check URL if we're validating, but not if component loading
    if fValidateURL and not (csLoading in ComponentState) then
      CheckURL(Value);  // raises exception on bad URL
    // update URL property with same value
    fURL := Value;
  end;
  // Now update actual caption
  inherited Caption := Value;
end;

procedure TPJHotLabel.SetCaptionIsURL(const Value: Boolean);
  {Write access method for CaptionIsURL property: when set true we set Caption
  property to same value as URL property}
begin
  fCaptionIsURL := Value;
  if Value then
    SetCaption(fURL);
end;

procedure TPJHotLabel.SetDefaultURL;
  {Sets the URL to the default value. Override this method in descendant classes
  to change this default}
begin
  try
    // Try to set URL to default
    SetURL(cDefaultURL)
  except on EPJURLError do
    // Catch exception thrown if default URL not valid
    // we also set URL to '': aren't we being cautious!
    SetURL('');
  end;
end;

procedure TPJHotLabel.SetFont(const Value: TFont);
  {"Overridden" write access method for Font property: sets either the Font
  property itself to new value or, when Font is displaying highlight, stores new
  Font value in fBackupFont since this will be restored to the Font property
  once highlight is removed => Font will be updated with its true value at a
  later time. This lag doesn't matter since the GetFont method will also access
  fBackupFont for Font value while Font is displaying highlight}
begin
  if fBackupFontUsed then
    // Font displaying highlight: store new value in backup font for later
    // restoration to Font
    fBackupFont.Assign(Value)
  else
    // Font displaying normally: simply update the actual font
    inherited Font := Value;
end;

procedure TPJHotLabel.SetHighlightFont(const Value: TFont);
  {Write access method for HighlightFont property}
begin
  fHighlightFont.Assign(Value);
end;

procedure TPJHotLabel.SetURL(const Value: string);
  {Write access method for URL property: validates URL if required by
  ValidateURL property and also stores URL in Caption if CaptionIsURL property
  is true}
begin
  // We only perform validation and setting Caption is component isn't loading
  if not (csLoading in ComponentState) then
  begin
    // Check URL for validity if required
    if fValidateURL then
      CheckURL(Value);  // raises exception on error
    // Store URL in Caption if it is displaying URL
    if fCaptionIsURL then
      Caption := Value;
  end;
  // Finally record new value in URL
  fURL := Value;
end;

procedure TPJHotLabel.SetValidateURL(const Value: Boolean);
  {Write access method for ValidateURL property - ensures that current URL
  property value is validated if this property is set to true}
begin
  // Record new value
  fValidateURL := Value;
  if fValidateURL then
    try
      // ValidateURL changed to true => validate URL property
      CheckURL(fURL);
    except
      on E: EPJURLError do
      begin
        // Not a valid URL - replace it with default and re-raise
        // exception so user gets to know about error
        SetDefaultURL;
        raise;
      end;
    end;
end;

end.
