{$J+,Z4}
unit X509Funcs; // PGP 7.X and later only

{------------------------------------------------------------------------------}
{                                                                              }
{                 This unit is partly based on Steve Heller's                  }
{         SPGP sources available from http://www.oz.net/~srheller/spgp/        }
{                                                                              }
{                Portions created by Michael in der Wiesche are                }
{              Copyright (C) 2004-2006 by Michael in der Wiesche               }
{                                                                              }
{------------------------------------------------------------------------------}

interface

uses
  Windows,
  Classes,
  SysUtils,
  StrTools,
  KeyPropTypes,
  X509Types,
  UTF8,
  pgpBase,
  pgpErrors,
  pgpPubTypes,
  pgpUtilities,
  pgpOptionList,
  pgpMemoryMgr,
  pgpKeys,
  KeyFuncs,
  PGPDialogs;

// returns the last valid certificate issued by SignKey (if specified)
function GetValidX509CertFromKey(const CertUserData: String; CertKey, SignKey: pPGPKeyDBObj;
				 KeySetToSearch: pPGPKeySet): pPGPKeyDBObj;

// check whether all keys in the specified keyset have valid certificates
function CheckValidX509CertifierSet(const KeySetToCheck: pPGPKeySet): PGPError;

// create a key DB with X509 certified keys from the specified keyset
function GetX509CertKeyDB(Context: pPGPContext; KeySetMain: pPGPKeySet; var KeyDBX509: pPGPKeyDB): PGPError;

// filter the specified keyset for signing keys with valid certificates
function GetValidX509CertifierSet(const KeySetToFilter: pPGPKeySet;
				  var KeySetValidCertifiers: pPGPKeySet;
				  RootOnly, ValidOnly: Longbool): PGPError;

// dialog for retrieving a signing key and it's passphrase
function X509GetCertAndPassphraseDialog(Context: pPGPContext; KeySetMain: pPGPKeySet;
					var SignKey, SignCert: pPGPKeyDBObj; var Passphrase: PChar;
					const SignCertUser, SignCertIASN, SignKeyAndPassPrompt: String;
					WinHandle: Integer): Integer;
{ specifying ...Prompt strings will present the respective PGP dialogs regardless of specifying any ...ID(s) }

// if CertAttributes' CommonName or Email are undefined they are taken from the CertKey's primary user ID
function CreateX509CertificateRequest(const CertKeyAndPassPrompt, CertKeyHexID: String; ReqKeyPassphrase: PChar;
				      var CertAttributes: TX509CertAttributes; Armor: Longbool;
				      var Request: String; var ReqKeyProps: TKeyPropsRec;
				      KeyPropFlags, WinHandle: Integer): Integer;
// specifying a unique CertAttributes' SerialNumber is mandatory, CommonName and Email are taken from the request
// if the SignCertIASN string contains a SignKeyHexID the key's last valid certificate will be used for certifying
function CreateX509CertificateFromRequest(const SignKeyAndPassPrompt, SignCertIASN: String; IssuerPassphrase: PChar;
					  var CertAttributes: TX509CertAttributes; ValidDays: Integer;
					  Armor: Longbool; const Request: String; var Certificate: String;
					  var CertPropsList: TX509CertPropsList; WinHandle: Integer): Integer;
// if the RevokeCertIASN string contains valid CertKeyHexID(s) all certificates on the respective key(s) will be revoked
// if no RevokeCertIASN string is specified all (non-self) certificates issued by the respective SignKey will be revoked
// if the SignCertIASN string contains a SignKeyHexID it's last valid certificate will be used for revoking certificates
// if SignCertIASN isn't provided and RevokeCertIASN contains a unique IASN the certifying key can be found automatically
function CreateX509CertificateRevocationList(const CertKeyPrompt, SignKeyAndPassPrompt, RevokeCertIASN, SignCertIASN: String;
					     RevokerPassphrase: PChar; ValidDays: Integer;
					     var CertPropsList: TX509CertPropsList;
					     WinHandle: Integer): Integer;
// specifying a unique CertAttributes.SerialNumber is mandatory
// splitting a user ID into CommonName and Email will certify it
// self-signed certificates cannot be created for an existing user ID
function CreateX509Certificate(const CertKeyPrompt, SignKeyAndPassPrompt, CertKeyHexID, SignCertIASN: String;
			       IssuerPassphrase: PChar; var CertAttributes: TX509CertAttributes; ValidDays: Integer;
			       SelfSigned: Longbool; var CertPropsList: TX509CertPropsList; WinHandle: Longint): Integer;
// self-signed certificates can only be removed including the user ID they are attached to
// user IDs of certificates can only be removed if they don't have any other signatures assigned
// if there's only a single user ID left on a key the key will be removed from the keyring all together
function RemoveX509Certificate(const CertKeyPrompt, RemoveCertIASN: String; IncludeUserID: Longbool;
			       var CertPropsList: TX509CertPropsList; WinHandle: Integer): Integer;
// import binary or armored standard X509 certificates or PGP keys to keyring(s)
function ImportX509Certificate(const CertKeyPrompt, CertPassPrompt, CertData: String; PKCS12Passphrase: PChar;
			       FromFile: Longbool; var CertPropsList: TX509CertPropsList; WinHandle: Integer): Integer;
// either export a single certificate or a certificate chain with parent certificate(s) included
// chains suitable for VerifyX509CertificateChain can be created with Armor disabled
function ExportX509Certificate(const CertKeyOrKeyAndPassPrompt, ExportCertIASN: String; SecKeyPassphrase: PChar;
			       Armor, Chain, Secret, ToFile: Longbool; var CertData: String;
			       var CertPropsList: TX509CertPropsList; WinHandle: Integer): Integer;
// the required certificate chain can be created using ExportX509Certificate() with Chain enabled and Armor disabled
function VerifyX509CertificateChain(const CertChain: String; FromFile: Longbool): Integer;
// verify certificate chain in given key DB
function VerifyX509CertificateChainDB(CertKeyDB: pPGPKeyDB): TX509ChainValidity;
// select certificate and return it's properties
function SelectX509CertificateDialog(const CertKeyPrompt: String;
				     var CertPropsList: TX509CertPropsList;
				     WinHandle: Integer): Integer;
// returns number of certificates found or error
function GetX509CertificateProperties(const CertData: String; FromFile: Longbool;
				      var CertPropsList: TX509CertPropsList): Integer;
// returns number of keyring certificates or error
// KeyFilterFlags will be applied to any kind of search
// SignKeyID can narrow the search with CertKey...IDs specified
// IncludeSigner is only valid without any CertKey...IDs specified
// in this case you will get the following results depending on SignKeyHexID and IncludeSigner:
// (SignKey = nil) and (IncludeSigner = false)  => all certified keys on keyring
// (SignKey = nil) and (IncludeSigner = true)   => all certifier keys on keyring (signing keys only)
// (SignKey <> nil) and (IncludeSigner = false) => all SignKey-certified keys on keyring without certifier
// (SignKey <> nil) and (IncludeSigner = true)  => all SignKey-certified keys on keyring including certifier
function FindX509CertProps(const CertKeyUserID, CertKeyHexID, SignKeyHexID: String; KeyFilterFlags: DWord;
			   IncludeSigner: Longbool; var CertPropsList: TX509CertPropsList): Integer;

implementation

function IsSerialNumberUnique(Context: pPGPContext; KeySetToCheck: pPGPKeySet; SignKey: pPGPKeyDBObj;
			      const SerialNumber: String; var Unique: Longbool): Integer; forward;
function GetX509CertPropsRec(Certificate: pPGPKeyDBObj; var CertPropsRec: TX509CertPropsRec): Integer; forward;
function GetX509CertPropsList(Certificate: pPGPKeyDBObj; var CertPropsList: TX509CertPropsList): Integer; forward;
function GetX509CertSetProps(CertSet, KeySetToSearch: pPGPKeySet; SignKey: pPGPKeyDBObj; const UserID: String;
			     Order: PGPKeyOrdering; var CertPropsList: TX509CertPropsList): Integer; forward;

// convert X509 errors to PGP errors with corresponding messages
function TranslateX509ErrorToPGPError(X509Error: PGPError): PGPError;
begin
  case X509Error of
    k509Error_NeededCertNotAvailable: Result := kPGPError_X509NeededCertNotAvailable;
    k509Error_SelfSignedCert: Result := kPGPError_X509SelfSignedCert;
    k509Error_InvalidCertificateSignature: Result := kPGPError_X509InvalidCertificateSignature;
    k509Error_InvalidCertificateFormat: Result := kPGPError_X509InvalidCertificateFormat;
  else
    Result := X509Error;
  end;
end;

// create a key DB with X509 certified keys from the specified keyset
function GetX509CertKeyDB(Context: pPGPContext; KeySetMain: pPGPKeySet; var KeyDBX509: pPGPKeyDB): PGPError;
var
  CertSetTemp	: pPGPKeySet;
  KeySetX509	: pPGPKeySet;
begin
  KeyDBX509 := nil;
  Result := PGPNewKeyDB(Context, KeyDBX509);
  if Result <> 0 then Exit;
  CertSetTemp := nil;
  try
    Result := GetCertSetFromKeySet(KeySetMain, CertSetTemp);
    if Result <> 0 then Exit;
    Result := PGPCopyKeys(CertSetTemp, KeyDBX509, KeySetX509);
    if Result <> 0 then Exit;
    PGPFreeKeySet(KeySetX509);
  finally
    PGPFreeKeySet(CertSetTemp);
    if Result <> 0 then PGPFreeKeyDB(KeyDBX509);
  end;
end;

// filter the specified keyset for keys with X509 certificates
function GetX509CertKeySet(Context: pPGPContext; KeySetMain: pPGPKeySet; var KeySetX509: pPGPKeySet): PGPError;
var
  X509Filter	: pPGPFilter;
begin
  Result := PGPNewKeyDBObjBooleanFilter(Context, kPGPSigProperty_IsX509, PGPTrue, X509Filter);
  if Result <> 0 then Exit;
  try
    Result := PGPFilterKeySet(KeySetMain, X509Filter, KeySetX509);
  finally
    PGPFreeFilter(X509Filter);
  end;
end;

// init main keyset and return filtered set of keys with X509 certificates
function InitX509CertKeySet(var Context: pPGPContext; var KeySetMain, KeySetX509: pPGPKeySet): PGPError;
begin
  Context := nil;
  KeySetMain := nil;
  KeySetX509 := nil;
  Result := KeyRings.InitKeyRings(Context, KeySetMain);
  if Result <> 0 then Exit;
  try
    Result := GetX509CertKeySet(Context, KeySetMain, KeySetX509);
  finally
    if Result <> 0 then begin
      KeyRings.FreeKeyRings;
      KeySetMain := nil;
      Context := nil;
    end;
  end;
end;

// finit X509 keyset and free keyrings, convert X509 specific errors
procedure FinitX509CertKeySet(var KeySetX509: pPGPKeySet);
begin
  PGPFreeKeySet(KeySetX509);
  KeyRings.FreeKeyRings;
  KeySetX509 := nil;
end;

// retrieve a key's primary user ID for using them as CommonName and Email on a certificate
function GetPrimaryUserIDAsAttributes(CertKey: pPGPKeyDBObj; var Name, Address: String): PGPError;
var
  StrPos, StrLen: Integer;
begin
  try
    Name := GetKeyPropUserID(CertKey);
    StrPos := LastPos(' ', Name);
    if StrPos <> 0 then Address := Copy(Name, succ(StrPos), MAXINT);
    Delete(Name, StrPos, MAXINT);
    StrLen := Length(Address);
    if StrLen > 2 then begin
      if Address[StrLen] = '>' then Delete(Address, StrLen, 1);
      if Address[1] = '<' then Delete(Address, 1, 1);
    end;
  finally
    Result := ord((Name = '') and (Address = '')) * kPGPError_InvalidDistinguishedName;
  end;
end;

// convert Attributes to Utf8
procedure GetUtf8Attributes(var CertAttributes: TX509CertAttributes);
begin
  with CertAttributes do begin
    CommonName := AnsiToUtf8(CommonName);
    Email := AnsiToUtf8(Email);
    OrganizationName := AnsiToUtf8(OrganizationName);
    OrganizationalUnitName := AnsiToUtf8(OrganizationalUnitName);
    SurName := AnsiToUtf8(SurName);
    Country := AnsiToUtf8(Country);
    Locality := AnsiToUtf8(Locality);
    State := AnsiToUtf8(State);
    StreetAddress := AnsiToUtf8(StreetAddress);
    Title := AnsiToUtf8(Title);
    Description := AnsiToUtf8(Description);
    PostalCode := AnsiToUtf8(PostalCode);
    POBOX := AnsiToUtf8(POBOX);
    PhysicalDeliveryOfficeName := AnsiToUtf8(PhysicalDeliveryOfficeName);
    TelephoneNumber := AnsiToUtf8(TelephoneNumber);
    X121Address := AnsiToUtf8(X121Address);
    ISDN := AnsiToUtf8(ISDN);
    DestinationIndicator := AnsiToUtf8(DestinationIndicator);
    Name := AnsiToUtf8(Name);
    GivenName := AnsiToUtf8(GivenName);
    Initials := AnsiToUtf8(Initials);
    HouseIdentifier := AnsiToUtf8(HouseIdentifier);
    DirectoryManagementDomain := AnsiToUtf8(DirectoryManagementDomain);
    DomainComponent := AnsiToUtf8(DomainComponent);
    UnstructuredName := AnsiToUtf8(UnstructuredName);
    UnstructuredAddress := AnsiToUtf8(UnstructuredAddress);
    RFC822Name := AnsiToUtf8(RFC822Name);
    DNSName := AnsiToUtf8(DNSName);
    AnotherName := AnsiToUtf8(AnotherName);
    IPAddress := AnsiToUtf8(IPAddress);
    CertificateExtension := AnsiToUtf8(CertificateExtension);
  end;
end;

// convert attribute record to option list parameter using risky type cast ...
function GetAttributeListFromRecord(var CertAttributes: TX509CertAttributes;
				    var AttributeList: TX509CertAttrArray;
				    var AttributeCount: Longint): PGPError;
var
  AttributeAddr	: ^PChar;
  AttributeIndex: PGPUInt32;
begin
  Result := 0;
  AttributeCount := 0;
  try
    AttributeAddr := @CertAttributes;
    GetUtf8Attributes(CertAttributes);
    FillChar(AttributeList, SizeOf(TX509CertAttrArray), 0);
    for AttributeIndex := 0 to pred(SizeOf(CertAttributes) div SizeOf(Pointer)) do begin
      with AttributeList[AttributeCount] do begin
	if AttributeAddr^ <> '' then begin
	  Attribute := PGPAVAttribute(AttributeIndex);
	  Value.PointerValue := AttributeAddr^;
	  Size := Length(Value.PointerValue);
	  inc(AttributeCount);
	end;
	inc(AttributeAddr);
      end;
    end;
  except
    on EAccessViolation do
      Result := kPGPError_BadMemAddress
    else Result := kPGPError_UnknownError;
  end;
end;

// search for certificates and their signing keys
function FindCertificate(KeyIter: pPGPKeyIter; KeySetToSearch: pPGPKeySet; var Certificate, SignKey: pPGPKeyDBObj): PGPError;
var
  X509Prop	: PGPBoolean;
begin
  SignKey := nil;
  repeat
    Result := PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_Any, Certificate);
    if (Result = 0) and (PGPGetKeyDBObjBooleanProperty(Certificate, kPGPSigProperty_IsX509, X509Prop) = 0)
    and (X509Prop = PGPTrue) then begin
      PGPGetSigCertifierKey(Certificate, KeySetToSearch, SignKey);
      Break;
    end;
  until Result <> 0;
end;

// check whether a user ID already has a certificate by the signing key(s)
function HasUserIDCertifierCert(CertKey: pPGPKeyDBObj; const CertKeyUserID: String;
				KeySetToSearch: pPGPKeySet; SignKey: pPGPKeyDBObj): Longbool;
var
  KeySet	: pPGPKeySet;
  KeyList	: pPGPKeyList;
  KeyIter	: pPGPKeyIter;
  UTF8ID	: UTF8String;
  UIDFound	: pPGPKeyDBObj;
  UserIDBuf	: TUserID;
  IDSize	: PGPSize;
  SigFound	: pPGPKeyDBObj;
  X509Prop	: PGPBoolean;
  KeyFound	: pPGPKeyDBObj;
begin
  Result := false;
  if GetSingleKeyDBObjIter(CertKey, KeySet, KeyList, KeyIter) = 0 then begin
    try
      UTF8ID := AnsiToUtf8(CertKeyUserID);
      while PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_UserID, UIDFound) = 0 do begin
	if (PGPGetKeyDBObjDataProperty(UIDFound, kPGPUserIDProperty_Name, @UserIDBuf, SizeOf(TUserID), IDSize) = 0)
	and ((PGPCompareUserIDStrings(UserIDBuf, PChar(CertKeyUserID)) = 0) or
	     (PGPCompareUserIDStrings(UserIDBuf, PChar(UTF8ID)) = 0)) then begin
	  while PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_Signature, SigFound) = 0 do begin
	    if (PGPGetKeyDBObjBooleanProperty(SigFound, kPGPSigProperty_IsX509, X509Prop) = 0)
	    and (X509Prop = PGPTrue) and (PGPGetSigCertifierKey(SigFound, KeySetToSearch, KeyFound) = 0)
	    and ((KeyFound = SignKey) or ((SignKey = nil) and GetKeyPropHasCertificate(KeyFound))) then begin
	      Result := true;
	      Exit;
	    end;
	  end;
	end;
      end;
    finally
      PGPFreeKeyIter(KeyIter);
      PGPFreeKeyIter(KeyList);
      PGPFreeKeySet(KeySet);
    end;
  end;
end;

// count valid (SignKey signed) certificates of a key
function ValidKeyCertCount(CertKey, SignKey: pPGPKeyDBObj): PGPUInt32;
var
  KeySet	: pPGPKeySet;
  KeyList	: pPGPKeyList;
  KeyIter	: pPGPKeyIter;
  RootKeySet	: pPGPKeySet;
  Certificate	: pPGPKeyDBObj;
  KeyFound	: pPGPKeyDBObj;
  X509Prop	: PGPBoolean;
begin
  Result := 0;
  if GetSingleKeyDBObjIter(CertKey, KeySet, KeyList, KeyIter) <> 0 then Exit;
  try
    RootKeySet := PGPPeekKeyDBRootKeySet(PGPPeekKeyDBObjKeyDB(CertKey));
    while (PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_Any, Certificate) = 0) do begin
      if (PGPGetKeyDBObjBooleanProperty(Certificate, kPGPSigProperty_IsX509, X509Prop) = 0)
      and (X509Prop = PGPTrue) and GetCertPropIsValid(Certificate)
      and ((SignKey = nil) or ((PGPGetSigCertifierKey(Certificate, RootKeySet, KeyFound) = 0) and (KeyFound = SignKey)))
      then begin
	inc(Result);
      end;
    end;
  finally
    PGPFreeKeyIter(KeyIter);
    PGPFreeKeyIter(KeyList);
    PGPFreeKeySet(KeySet);
  end;
end;

// returns the last valid certificate issued by SignKey (if specified)
function GetValidX509CertFromKey(const CertUserData: String; CertKey, SignKey: pPGPKeyDBObj;
				 KeySetToSearch: pPGPKeySet): pPGPKeyDBObj;
var
  KeySet	: pPGPKeySet;
  KeyList	: pPGPKeyList;
  KeyIter	: pPGPKeyIter;
  Certificate	: pPGPKeyDBObj;
  KeyFound	: pPGPKeyDBObj;
begin
  Result := nil;
  if GetSingleKeyDBObjIter(CertKey, KeySet, KeyList, KeyIter) = 0 then begin
    try
      while FindCertificate(KeyIter, KeySetToSearch, Certificate, KeyFound) = 0 do begin
	if GetCertPropIsValid(Certificate) and ((SignKey = nil) or (KeyFound = SignKey))
	and ((CertUserData = '') or
	     (pos(CertUserData, GetCertPropOwnerName(Certificate)) <> 0) or
	     (pos(CertUserData, GetCertPropOwnerUserID(Certificate)) <> 0))
	then begin
	  Result := Certificate;
	end;
      end;
    finally
      PGPFreeKeyIter(KeyIter);
      PGPFreeKeyIter(KeyList);
      PGPFreeKeySet(KeySet);
    end;
  end;
end;

// check whether all keys in the specified keyset have valid certificates
function CheckValidX509CertifierSet(const KeySetToCheck: pPGPKeySet): PGPError;
var
  KeyCount	: PGPUInt32;
  KeyIter	: pPGPKeyIter;
  Certificate	: pPGPKeyDBObj;
  SignKey	: pPGPKeyDBObj;
begin
  Result := PGPCountKeys(KeySetToCheck, KeyCount);
  if (Result = 0) and (KeyCount <> 0) then begin
    Result := PGPNewKeyIterFromKeySet(KeySetToCheck, KeyIter);
    if Result <> 0 then Exit;
    try
      while (FindCertificate(KeyIter, KeySetToCheck, Certificate, SignKey) = 0)
      and GetCertPropIsValid(Certificate) do begin
	dec(KeyCount);
      end;
      if KeyCount > 0 then Result := kPGPError_X509NeededCertNotAvailable;
    finally
      PGPFreeKeyIter(KeyIter);
    end;
  end;
end;

// filter the specified keyset for signing keys with valid certificates
function GetValidX509CertifierSet(const KeySetToFilter: pPGPKeySet;
				  var KeySetValidCertifiers: pPGPKeySet;
				  RootOnly, ValidOnly: Longbool): PGPError;
var
  KeyIter	: pPGPKeyIter;
  Certificate	: pPGPKeyDBObj;
  SignKey	: pPGPKeyDBObj;
  CertKey	: pPGPKeyDBObj;
begin
  KeySetValidCertifiers := nil;
  Result := PGPNewEmptyKeySet(PGPPeekKeySetKeyDB(KeySetToFilter), KeySetValidCertifiers);
  if Result <> 0 then Exit;
  Result := PGPNewKeyIterFromKeySet(KeySetToFilter, KeyIter);
  if Result <> 0 then begin
    PGPFreeKeySet(KeySetValidCertifiers);
    KeySetValidCertifiers := nil;
    Exit;
  end;
  try
    while FindCertificate(KeyIter, KeySetToFilter, Certificate, SignKey) = 0 do begin
      CertKey := PGPPeekKeyDBObjKey(Certificate);
      if not ValidOnly or (GetKeyPropCanSign(CertKey) and GetCertPropIsValid(Certificate)) then begin
	if not RootOnly or (SignKey = CertKey) then PGPAddKey(CertKey, KeySetValidCertifiers);
      end;
    end;
  finally
    PGPFreeKeyIter(KeyIter);
  end;
end;

// filter the specified keyset for signing keys with certificates
function GetX509CertifierSet(const KeySetToFilter: pPGPKeySet; var KeySetCertifiers: pPGPKeySet): PGPError;
var
  KeyIter	: pPGPKeyIter;
  Certificate	: pPGPKeyDBObj;
  SignKey	: pPGPKeyDBObj;
  CertKey	: pPGPKeyDBObj;
begin
  KeySetCertifiers := nil;
  Result := PGPNewEmptyKeySet(PGPPeekKeySetKeyDB(KeySetToFilter), KeySetCertifiers);
  if Result <> 0 then Exit;
  Result := PGPNewKeyIterFromKeySet(KeySetToFilter, KeyIter);
  if Result <> 0 then Exit;
  try
    while FindCertificate(KeyIter, KeySetToFilter, Certificate, SignKey) = 0 do begin
      CertKey := PGPPeekKeyDBObjKey(Certificate);
      if GetKeyPropCanSign(CertKey) then PGPAddKey(CertKey, KeySetCertifiers);
    end;
  finally
    PGPFreeKeyIter(KeyIter);
  end;
end;

// retrieve keys with certificates - options:
// (SignKey = nil) and (IncludeSigner = false)  => all certified keys on keyring
// (SignKey = nil) and (IncludeSigner = true)   => all certifier keys on keyring (signing keys only)
// (SignKey <> nil) and (IncludeSigner = false) => all SignKey-certified keys on keyring without certifier
// (SignKey <> nil) and (IncludeSigner = true)  => all SignKey-certified keys on keyring including certifier
function GetCertKeySetBySignKey(Context: pPGPContext; KeySetToFilter: pPGPKeySet; SignKey: pPGPKeyDBObj;
				IncludeSigner, ValidKeysOnly: Longbool; var KeySetFiltered: pPGPKeySet): PGPError;
var
  SignKeyID	: TPGPKeyID7;
  SignKeyFilter	: pPGPFilter;
  ExcludeFilter	: pPGPFilter;
  CertKeyFilter	: pPGPFilter;
  KeyCount	: PGPUInt32;
  KeyIter	: pPGPKeyIter;
  Certificate	: pPGPKeyDBObj;
  KeyFound	: pPGPKeyDBObj;
  KeySetChecked	: pPGPKeySet;
begin
  KeySetFiltered := nil;
  if SignKey = nil then begin
    if IncludeSigner then begin
      // signing keys with certificates only
      Result := GetX509CertifierSet(KeySetToFilter, KeySetFiltered);
    end
    else begin
      // all certified keys in keyset (as is)
      Result := PGPIncKeySetRefCount(KeySetToFilter);
      if Result = 0 then KeySetFiltered := KeySetToFilter;
    end;
  end
  else begin
    SignKeyFilter := nil;
    ExcludeFilter := nil;
    CertKeyFilter := nil;
    try
      // filter by signing key
      Result := PGPGetKeyIDFromKey(SignKey, SignKeyID);
      if Result <> 0 then Exit;
      Result := PGPNewKeyDBObjDataFilter(Context, kPGPSigProperty_KeyID, @SignKeyID,
					 SizeOf(TPGPKeyID7), kPGPMatchEqual, SignKeyFilter);
      if Result <> 0 then Exit;
      // remove signing key?
      if not IncludeSigner then begin
	ExcludeFilter := GetKeyExcludeFilter(Context, SignKey);
	Result := PGPIntersectFilters(SignKeyFilter, ExcludeFilter, CertKeyFilter);
	SignKeyFilter := nil;
	ExcludeFilter := nil;
	if Result <> 0 then Exit;
      end
      else begin
	CertKeyFilter := SignKeyFilter;
	SignKeyFilter := nil;
      end;
      Result := PGPFilterKeySet(KeySetToFilter, CertKeyFilter, KeySetFiltered);
    finally
      PGPFreeFilter(SignKeyFilter);
      PGPFreeFilter(ExcludeFilter);
      PGPFreeFilter(CertKeyFilter);
    end;
  end;
  if Result <> 0 then Exit;
  Result := PGPCountKeys(KeySetFiltered, KeyCount);
  if Result <> 0 then Exit;
  if KeyCount <> 0 then begin
    // remove keys without (valid) SignKey certificates
    if SignKey <> nil then begin
      KeySetChecked := nil;
      KeyIter := nil;
      try
	Result := PGPNewEmptyKeySet(PGPPeekKeySetKeyDB(KeySetFiltered), KeySetChecked);
	if Result = 0 then begin
	  Result := PGPNewKeyIterFromKeySet(KeySetFiltered, KeyIter);
	  if Result = 0 then begin
	    while FindCertificate(KeyIter, KeySetFiltered, Certificate, KeyFound) = 0 do begin
	      if (KeyFound = SignKey) and (not ValidKeysOnly or GetCertPropIsValid(Certificate)) then begin
		PGPAddKey(PGPPeekKeyDBObjKey(Certificate), KeySetChecked);
	      end;
	    end;
	    if Result = 0 then begin
	      Result := PGPCountKeys(KeySetChecked, KeyCount);
	      if (Result = 0) and (KeyCount = 0) then Result := kPGPError_X509NeededCertNotAvailable;
	    end;
	  end;
	end;
      finally
	PGPFreeKeyIter(KeyIter);
	PGPFreeKeySet(KeySetFiltered);
	if Result = 0 then
	  KeySetFiltered := KeySetChecked
	else begin
	  PGPFreeKeySet(KeySetChecked);
	  KeySetFiltered := nil;
	end;
      end;
    end;
  end
  else begin
    Result := kPGPError_X509NeededCertNotAvailable;
    PGPFreeKeySet(KeySetFiltered);
    KeySetFiltered := nil;
  end;
end;

function EqualIASN(const IASN1, IASN2: String): Longbool; register; assembler;
asm	// EAX = @IASN1, EDX = @IASN2
  PUSH	EBX
  OR	EAX,EAX
  JE	@EXIT
  OR	EDX,EDX
  JE	@EXIT
  CMP	EAX,EDX
  JZ	@EQUAL
  MOV	ECX,[EAX - 04h]
  CMP	ECX,[EDX - 04h]
  JNE	@EXIT
  PUSH	ECX
  SHR	ECX,2
  JE	@REST
  @DWLOOP:
  MOV	EBX,[EAX]
  CMP	EBX,[EDX]
  JNE	@DWEXIT
  ADD	EAX,4
  ADD	EDX,4
  DEC	ECX
  JNE	@DWLOOP
  @REST:
  POP	ECX
  AND	ECX,3
  JE	@EQUAL
  MOV	EBX,[EAX + ECX - 04h]
  CMP	EBX,[EDX + ECX - 04h]
  JNE	@EXIT
  @EQUAL:
  MOV	EAX,TRUE
  POP	EBX
  RET
  @DWEXIT:
  POP	ECX
  @EXIT:
  XOR	EAX,EAX
  POP	EBX
end;

// search the specified keyset for a certificate with the specified CertificateIASN
function GetCertByCertIASN(KeySetToSearch: pPGPKeySet; const CertificateIASN: String;
			   var SignKey: pPGPKeyDBObj): pPGPKeyDBObj;
var
  KeyIter	: pPGPKeyIter;
  Certificate	: pPGPKeyDBObj;
begin
  Result := nil;
  SignKey := nil;
  KeyIter := nil;
  if PGPNewKeyIterFromKeySet(KeySetToSearch, KeyIter) <> 0 then Exit;
  try
    while FindCertificate(KeyIter, KeySetToSearch, Certificate, SignKey) = 0 do begin
      if EqualIASN(GetCertPropIASN(Certificate), CertificateIASN) then begin
	Result := Certificate;
	Break;
      end;
    end;
  finally
    PGPFreeKeyIter(KeyIter);
  end;
end;

// retrieve a set containing a single certificate
function GetCertSetByCertIASN(Context: pPGPContext; KeySetToSearch: pPGPKeySet; const CertificateIASN: String;
			      var SingleCertSet: pPGPKeySet; var SignKey: pPGPKeyDBObj): Integer;
var
  Certificate	: pPGPKeyDBObj;
begin
  Certificate := GetCertByCertIASN(KeySetToSearch, CertificateIASN, SignKey);
  if Certificate <> nil then
    Result := PGPNewSingletonKeySet(Certificate, SingleCertSet)
  else Result := kPGPError_X509NeededCertNotAvailable;
end;

// a simple file read function
function ReadDataFromFile(const CertFile: String; var CertData: String): Integer;
var
  CertStream	: TMemoryStream;
begin
  Result := 0;
  CertData := '';
  try
    CertStream := TMemoryStream.Create;
    if CertStream <> nil then with CertStream do begin
      try
	LoadFromFile(CertFile);
	if Size <> 0 then begin
	  SetLength(CertData, Size);
	  ReadBuffer(Pointer(CertData)^, Size);
	end;
      finally
	CertStream.Free;
      end;
    end;
  except
    Result := kPGPError_ReadFailed;
  end;
end;

// a simple file write function
function WriteDataToFile(const CertData, CertFile: String): Integer;
var
  CertStream	: TMemoryStream;
begin
  Result := 0;
  try
    CertStream := TMemoryStream.Create;
    if CertStream <> nil then with CertStream do begin
      try
	WriteBuffer(Pointer(CertData)^, Length(CertData));
	SaveToFile(CertFile);
      finally
	CertStream.Free;
      end;
    end
    else Result := kPGPError_OutOfMemory;
  except
    Result := kPGPError_WriteFailed;
  end;
end;

// copy (binary) data to string
function CopyDataToString(const InData: PChar; var OutData: String; Size: DWord): Integer;
begin
  Result := 0;
  try
    SetLength(OutData, Size);
    Move(InData^, Pointer(OutData)^, Size);
  except
    on EOutOfMemory do begin
      Result := kPGPError_OutOfMemory;
      OutData := '';
    end;
  end;
end;

// dialog for retrieving a single (to get) certified key
function X509GetKeyDialog(Context: pPGPContext; KeySetMain: pPGPKeySet; CertsOnly: Longbool;
			  var CertOrKey: pPGPKeyDBObj; const CertKeyPrompt: String;
			  WinHandle: Integer): Integer;
var
  CertSetMain	: pPGPKeySet;
  CertSetSel	: pPGPKeySet;
begin
  if CertsOnly then begin
    Result := GetCertSetFromKeySet(KeySetMain, CertSetMain);
    if Result <> 0 then Exit;
  end
  else CertSetMain := KeySetMain;
  try
    Result := SelectX509CertifiedKeysDialog(Context, CertSetMain, CertSetSel, CertOrKey,
					    CertKeyPrompt, CertsOnly, true, WinHandle);
    if Result <> 0 then Exit;
    try
      if not (PGP81 and CertsOnly) then Result := GetKeyFromKeySet(CertSetSel, CertOrKey);
    finally
      PGPFreeKeySet(CertSetSel);
    end;
  finally
    if CertsOnly then PGPFreeKeySet(CertSetMain);
  end;
end;

// dialog for retrieving a signing key and it's passphrase
function X509GetCertAndPassphraseDialog(Context: pPGPContext; KeySetMain: pPGPKeySet;
					var SignKey, SignCert: pPGPKeyDBObj; var Passphrase: PChar;
					const SignCertUser, SignCertIASN, SignKeyAndPassPrompt: String;
					WinHandle: Integer): Integer;
var
  KeySetFiltered: pPGPKeySet;
  UserEmail	: String;
  UserIDFilter	: pPGPFilter;
  KeyCount	: PGPUInt32;
  KeyDBTemp	: pPGPKeyDB;
  KeySetTemp	: pPGPKeySet;
  CertSetTemp	: pPGPKeySet;
  Dummy		: PGPUInt32;
  CertPrompt	: String;
begin
  SignKey := nil;
  SignCert := nil;
  if SignKeyAndPassPrompt <> '' then begin // prompt for signing key and passphrase
    Passphrase := nil;
    KeySetFiltered := nil;
    if SignCertUser <> '' then begin
      UserEmail := GetUserFromLongName(SignCertUser, true);
      if UserEmail <> '' then UserEmail := ExtractEmail(UserEmail);
      if UserEmail = '' then UserEmail := ExtractEmail(SignCertUser);
      if UserEmail <> '' then begin
	if PGPNewKeyDBObjDataFilter(Context, kPGPUserIDProperty_EmailAddress, PChar(UserEmail),
				    Length(UserEmail), kPGPMatchEqual, UserIDFilter) = 0 then begin
	  try
	    PGPFilterKeySet(KeySetMain, UserIDFilter, KeySetFiltered);
	  finally
	    PGPFreeFilter(UserIDFilter);
	  end;
	end;
      end;
    end;
    if (KeySetFiltered = nil) or (PGPCountKeys(KeySetFiltered, KeyCount) <> 0) or (KeyCount = 0) then begin
      KeySetFiltered := KeySetMain;
      UserEmail := '';
    end;
    try
      Result := PGPNewKeyDB(Context, KeyDBTemp);
      if Result <> 0 then Exit;
      try
	Result := PGPCopyKeys(KeySetFiltered, KeyDBTemp, KeySetTemp);
	if Result <> 0 then Exit;
	try
	  Result := GetCertSetFromKeySet(KeySetTemp, CertSetTemp);
	  if Result <> 0 then Exit;
	  try
	    Result := SigningPassphraseDialog(Context, CertSetTemp, SignKey, Passphrase, SignCertIASN,
					      true, soNo, Dummy, Dummy, Dummy, SignKeyAndPassPrompt, WinHandle);
	  finally
	    PGPFreeKeySet(CertSetTemp);
	  end;
	  if Result <> 0 then begin
	    if Result = kPGPError_SecretKeyNotFound then Result := kPGPError_X509NeededCertNotAvailable;
	    Exit;
	  end;
	  Result := GetKeyFromNewSet(SignKey, KeySetMain, SignKey);
	  if Result <> 0 then Exit;
	  if PGP81 and ((SignCertUser = '') or (UserEmail = '')) and (ValidKeyCertCount(SignKey, nil) > 1) then begin
	    Result := PGPNewSingletonKeySet(SignKey, CertSetTemp);
	    if Result <> 0 then Exit;
	    try
	      CertPrompt := Copy(SignKeyAndPassPrompt, 1, FirstPos(' ', SignKeyAndPassPrompt));
	      if CertPrompt <> '' then CertPrompt[Length(CertPrompt)] := ':';
	      Result := X509GetKeyDialog(Context, CertSetTemp, true, SignCert, CertPrompt, WinHandle);
	    finally
	      PGPFreeKeySet(CertSetTemp);
	    end;
	  end
	  else SignCert := GetValidX509CertFromKey(SignCertUser, SignKey, nil, KeySetMain);
	finally
	  PGPFreeKeySet(KeySetTemp);
	end;
      finally
	PGPFreeKeyDB(KeyDBTemp);
      end;
    finally
      if KeySetFiltered <> KeySetMain then PGPFreeKeySet(KeySetFiltered);
    end;
  end
  else begin // do we have a hex ID or an IASN?
    Result := GetKeyByHexID(KeySetMain, SignCertIASN, SignKey);
    if Result <> 0 then begin
      SignCert := GetCertByCertIASN(KeySetMain, SignCertIASN, SignKey);
      if SignCert <> nil then Result := 0;
    end
    else begin
      if GetKeyPropCanSign(SignKey) then
	SignCert := GetValidX509CertFromKey('', SignKey, nil, KeySetMain)
      else Result := kPGPError_KeyUnusableForSignature;
    end;
    if Result <>0 then Exit;
    if PGPPassphraseIsValid(SignKey, PGPOPassphrase(Context, Passphrase), PGPOLastOption(Context)) = PGPFalse then begin
      Result := kPGPError_BadPassphrase;
    end;
  end;
  if (SignCert = nil) and (Result = 0) then Result := kPGPError_X509NeededCertNotAvailable;
end;

// if CertAttributes' CommonName or Email are undefined they are taken from the CertKey's primary user ID
function CreateX509CertificateRequest(const CertKeyAndPassPrompt, CertKeyHexID: String; ReqKeyPassphrase: PChar;
				      var CertAttributes: TX509CertAttributes; Armor: Longbool;
				      var Request: String; var ReqKeyProps: TKeyPropsRec;
				      KeyPropFlags, WinHandle: Integer): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  KeyToCertify	: pPGPKeyDBObj;
  Dummy		: pPGPKeyDBObj;
  AttributeCount: Longint;
  AttributeList	: TX509CertAttrArray;
  OptionList	: pPGPOptionList;
  CertOutput	: PChar;
  CertLength	: PGPSize;
begin
  if PGP7X then begin
    Result := KeyRings.InitKeyRings(Context, KeySetMain);
    if Result <> 0 then Exit;
    try
      Result := X509GetCertAndPassphraseDialog(Context, KeySetMain, KeyToCertify, Dummy, ReqKeyPassphrase,
					       '', CertKeyHexID, CertKeyAndPassPrompt, WinHandle);
      if (Result <> 0) and (Result <> kPGPError_X509NeededCertNotAvailable) then Exit;
      Result := GetKeyProps(KeySetMain, KeyToCertify, KeyPropFlags, ReqKeyProps);
      with CertAttributes do begin
	if (CommonName = '') or (Email = '') then begin
	  Result := GetPrimaryUserIDAsAttributes(KeyToCertify, CommonName, Email);
	  if Result <> 0 then Exit;
	end;
      end;
      Result := GetAttributeListFromRecord(CertAttributes, AttributeList, AttributeCount);
      if Result <> 0 then Exit;
      OptionList := nil;
      CertOutput := nil;
      Result := PGPBuildOptionList(Context, OptionList,
	[
	  PGPOExportKey(Context, KeyToCertify),
	  PGPOPassphrase(Context, ReqKeyPassphrase),
	  PGPOArmorOutput(Context, PGPBoolean(Armor) and 1),
	  PGPOExportFormat(Context, kPGPExportFormat_X509CertReq),
	  PGPOAttributeValue(Context, @AttributeList, AttributeCount),
	  PGPOAllocatedOutputBuffer(Context, CertOutput, $FFFFFFFF, CertLength)
	]);
      if Result <> 0 then Exit;
      try
	Result := PGPExport(Context, OptionList, PGPOLastOption(Context));
	if (Result = 0) and (CertLength <> 0) then begin
	  Result := CopyDataToString(CertOutput, Request, CertLength);
	  if Result <> 0 then Exit;
	end
	else begin
	  Request := '';
	  Exit;
	end;
      finally
	PGPFreeData(CertOutput);
	PGPFreeOptionList(OptionList);
      end;
    finally
      if CertKeyAndPassPrompt <> '' then PGPFreeData(ReqKeyPassphrase);
      KeyRings.FreeKeyRings;
      Result := TranslateX509ErrorToPGPError(Result);
    end;
  end
  else Result := kPGPError_LazyProgrammer;
end;

// specifying a unique CertAttributes' SerialNumber is mandatory, CommonName and Email are taken from the request
// if the SignCertIASN string contains a SignKeyHexID the key's last valid certificate will be used for certifying
function CreateX509CertificateFromRequest(const SignKeyAndPassPrompt, SignCertIASN: String; IssuerPassphrase: PChar;
					  var CertAttributes: TX509CertAttributes; ValidDays: Integer;
					  Armor: Longbool; const Request: String; var Certificate: String;
					  var CertPropsList: TX509CertPropsList; WinHandle: Integer): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  KeySetX509	: pPGPKeySet;
  KeySetX509Sign: pPGPKeySet;
  SignKey	: pPGPKeyDBObj;
  SignCert	: pPGPKeyDBObj;
  SNUnique	: Longbool;
  AttributeCount: Longint;
  AttributeList	: TX509CertAttrArray;
  OptionList	: pPGPOptionList;
  KeyDBTemp	: pPGPKeyDB;
  KeySetTemp	: pPGPKeySet;
  RequestCert	: pPGPKeyDBObj;
  RequestKey	: pPGPKeyDBObj;
  UserIDFound	: pPGPKeyDBObj;
  KeyFound	: pPGPKeyDBObj;
  UserID	: TUserID;
  IDSize	: PGPSize;
  RequestCertSet: pPGPKeySet;
  CertOutput	: PChar;
  CertLength	: PGPSize;
begin
  if PGP7X then begin
    Result := InitX509CertKeySet(Context, KeySetMain, KeySetX509);
    if Result <> 0 then Exit;
    try
      Result := GetValidX509CertifierSet(KeySetX509, KeySetX509Sign, false, true);
      if Result <> 0 then Exit;
      try
	Result := X509GetCertAndPassphraseDialog(Context, KeySetX509Sign, SignKey, SignCert, IssuerPassphrase,
						 '', SignCertIASN, SignKeyAndPassPrompt, WinHandle);
      finally
	PGPFreeKeySet(KeySetX509Sign);
      end;
      if Result <> 0 then Exit;
      Result := IsSerialNumberUnique(Context, KeySetMain, SignKey, CertAttributes.SerialNumber, SNUnique);
      if Result <> 0 then Exit;
      if not SNUnique then begin
	Result := kPGPError_DuplicateCert;
	Exit;
      end;
      Result := GetAttributeListFromRecord(CertAttributes, AttributeList, AttributeCount);
      if Result <> 0 then Exit;
      OptionList := nil;
      Result := PGPBuildOptionList(Context, OptionList,
	[
	  PGPOInputBuffer(Context, PChar(Request), Length(Request)),
	  PGPOAttributeValue(Context, @AttributeList, AttributeCount),
	  PGPOPassphrase(Context, IssuerPassphrase),
	  PGPOExpiration(Context, ValidDays)
	]);
      if Result <> 0 then Exit;
      try
	Result := PGPNewKeyDB(Context, KeyDBTemp);
	if Result <> 0 then Exit;
	try
	  Result := PGPCopyKeys(KeySetX509, KeyDBTemp, KeySetTemp);
	  if Result <> 0 then Exit;
	  try
	    Result := PGPCreateX509CertificateFromRequest(SignCert, RequestCert, OptionList, PGPOLastOption(Context));
	    if Result <> 0 then Exit;
	    Result := KeyRings.UpdateKeyRings;
	    if Result <> 0 then Exit;
	    Result := GetX509CertPropsList(RequestCert, CertPropsList);
	    if Result <> 0 then Exit;
	    RequestKey := PGPPeekKeyDBObjKey(RequestCert);
	    if RequestKey = SignKey then begin
	      Result := kPGPError_X509SelfSignedCert;
	      if GetKeyFromNewSet(SignKey, KeySetTemp, SignKey) = 0 then begin
		RemoveKeyFromKeySet(RequestKey, KeySetX509);
		AddKeyToKeySet(SignKey, KeySetX509);
	      end;
	    end
	    else begin
	      UserIDFound := PGPPeekKeyDBObjUserID(RequestCert);
	      if (GetKeyFromNewSet(SignKey, KeySetTemp, SignKey) = 0)
	      and (GetKeyFromNewSet(RequestKey, KeySetTemp, KeyFound) = 0)
	      and (PGPGetKeyDBObjDataProperty(UserIDFound, kPGPUserIDProperty_Name, @UserID, SizeOf(TUserID), IDSize) = 0)
	      and HasUserIDCertifierCert(KeyFound, UserID, KeySetTemp, SignKey) then begin
		Result := kPGPError_DuplicateUserID;
		RemoveKeyFromKeySet(RequestKey, KeySetX509);
		AddKeyToKeySet(KeyFound, KeySetX509);
	      end;
	    end;
	  finally
	    PGPFreeKeySet(KeySetTemp);
	  end;
	finally
	  PGPFreeKeyDB(KeyDBTemp);
	end;
      finally
	PGPFreeOptionList(OptionList);
      end;
      if Result <> 0 then Exit;
      Result := PGPNewSingletonKeySet(RequestCert, RequestCertSet);
      if Result <> 0 then Exit;
      try
        OptionList := nil;
	Result := PGPBuildOptionList(Context, OptionList,
	  [
	    PGPOExportKeySet(Context, RequestCertSet),
	    PGPOExportFormat(Context, kPGPExportFormat_X509Cert),
	    PGPOArmorOutput(Context, PGPBoolean(Armor) and 1),
	    PGPOAllocatedOutputBuffer(Context, CertOutput, $FFFFFFFF, CertLength)
	  ]);
	if Result <> 0 then Exit;
	try
	  Result := PGPExport(Context, OptionList, PGPOLastOption(Context));
	  if Result <> 0 then Exit;
	  try
	    Result := CopyDataToString(CertOutput, Certificate, CertLength);
	  finally
	    PGPFreeData(CertOutput);
	  end;
	finally
	  PGPFreeOptionList(OptionList);
	end;
      finally
	PGPFreeKeySet(RequestCertSet);
      end;
    finally
      if SignKeyAndPassPrompt <> '' then PGPFreeData(IssuerPassphrase);
      FinitX509CertKeySet(KeySetX509);
      Result := TranslateX509ErrorToPGPError(Result);
    end;
  end
  else Result := kPGPError_LazyProgrammer;
end;

// if the RevokeCertIASN string contains valid CertKeyHexID(s) all certificates on the respective key(s) will be revoked
// if no RevokeCertIASN string is specified all (non-self) certificates issued by the respective SignKey will be revoked
// if the SignCertIASN string contains a SignKeyHexID it's last valid certificate will be used for revoking certificates
// if SignCertIASN isn't provided and RevokeCertIASN contains a unique IASN the certifying key can be found automatically
function CreateX509CertificateRevocationList(const CertKeyPrompt, SignKeyAndPassPrompt, RevokeCertIASN, SignCertIASN: String;
					     RevokerPassphrase: PChar; ValidDays: Integer;
					     var CertPropsList: TX509CertPropsList;
					     WinHandle: Integer): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  KeySetX509	: pPGPKeySet;
  KeySetX509Sign: pPGPKeySet;
  SignKey	: pPGPKeyDBObj;
  SignCert	: pPGPKeyDBObj;
  SignKeyCertSet: pPGPKeySet;
  RevokeCertSet	: pPGPKeySet;
  RevokeCert	: pPGPKeyDBObj;
  SignKeyFound	: pPGPKeyDBObj;
  OptionList	: pPGPOptionList;
begin
  if PGP7X then begin
    Result := InitX509CertKeySet(Context, KeySetMain, KeySetX509);
    if Result <> 0 then Exit;
    try
      Result := GetValidX509CertifierSet(KeySetX509, KeySetX509Sign, true, true);
      if Result <> 0 then Exit;
      if (SignKeyAndPassPrompt <> '') or (SignCertIASN <> '') then begin
	try
	  Result := X509GetCertAndPassphraseDialog(Context, KeySetX509Sign, SignKey, SignCert, RevokerPassphrase,
						   '', SignCertIASN, SignKeyAndPassPrompt, WinHandle);
	finally
	  PGPFreeKeySet(KeySetX509Sign);
	end;
	if Result <> 0 then Exit;
	if not GetCertPropIsRoot(SignCert) then begin
	  Result := kPGPError_MissingX509Certificate;
	  Exit;
	end;
      end
      else SignCert := nil;
      RevokeCertSet := nil;
      try
	if CertKeyPrompt <> '' then begin // prompt for certified key(s)
	  Result := GetCertKeySetBySignKey(Context, KeySetX509, SignKey, true, true, SignKeyCertSet);
	  if Result <> 0 then Exit;
	  try
	    Result := SelectX509CertifiedKeysDialog(Context, SignKeyCertSet, RevokeCertSet, RevokeCert,
						    CertKeyPrompt, true, true, WinHandle);
	  finally
	    PGPFreeKeySet(SignKeyCertSet);
	  end;
	end
	else begin
	  if RevokeCertIASN <> '' then begin // do we have hex ID(s) or an IASN?
	    Result := GetKeySetByAnyIDs(Context, KeySetX509, RevokeCertIASN, RevokeCertSet);
	    if Result <> 0 then begin
	      Result := GetCertSetByCertIASN(Context, KeySetX509, RevokeCertIASN, RevokeCertSet, SignKeyFound);
	      if (Result = 0) and (SignCert = nil) then begin
		SignCert := GetValidX509CertFromKey('', SignKeyFound, SignKeyFound, KeySetX509);
		if SignCert <> nil then SignKey := SignKeyFound;
	      end;
	    end;
	  end
	  else begin // get key(s) certified by signing key
	    Result := GetCertKeySetBySignKey(Context, KeySetX509, SignKey, false, true, RevokeCertSet);
	  end;
	end;
	if Result <> 0 then Exit;
	OptionList := nil;
	Result := PGPBuildOptionList(Context, OptionList,
	  [
	    PGPOPassphrase(Context, RevokerPassphrase),
	    PGPOExpiration(Context, ValidDays)
	  ]);
	if Result <> 0 then Exit;
	try
	  Result := PGPCreateX509CRL(SignCert, RevokeCertSet, OptionList, PGPOLastOption(Context));
	  if Result <> 0 then Exit;
	  Result := KeyRings.UpdateKeyRings;
	  if Result <> 0 then Exit;
	  Result := GetX509CertSetProps(RevokeCertSet, KeySetX509, SignKey, '', kPGPKeyOrdering_UserID, CertPropsList);
	  if Result > 0 then Result := 0;
	finally
	  PGPFreeOptionList(OptionList);
	end;
      finally
	PGPFreeKeySet(RevokeCertSet);
      end;
    finally
      if SignKeyAndPassPrompt <> '' then PGPFreeData(RevokerPassphrase);
      FinitX509CertKeySet(KeySetX509);
      Result := TranslateX509ErrorToPGPError(Result);
    end;
  end
  else Result := kPGPError_LazyProgrammer;
end;

// specifying a unique CertAttributes.SerialNumber is mandatory
// splitting a user ID into CommonName and Email will certify it
// self-signed certificates cannot be created for an existing user ID
function CreateX509Certificate(const CertKeyPrompt, SignKeyAndPassPrompt, CertKeyHexID, SignCertIASN: String;
			       IssuerPassphrase: PChar; var CertAttributes: TX509CertAttributes; ValidDays: Integer;
			       SelfSigned: Longbool; var CertPropsList: TX509CertPropsList; WinHandle: Longint): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  KeySetX509	: pPGPKeySet;
  KeySetX509Sign: pPGPKeySet;
  SignKey	: pPGPKeyDBObj;
  SignCert	: pPGPKeyDBObj;
  SNUnique	: Longbool;
  KeySetFiltered: pPGPKeySet;
  KeyToCertify	: pPGPKeyDBObj;
  AttributeCount: Longint;
  AttributeList	: TX509CertAttrArray;
  OptionList	: pPGPOptionList;
  Certificate	: pPGPKeyDBObj;
begin
  if PGP7X then begin
    Result := InitX509CertKeySet(Context, KeySetMain, KeySetX509);
    if Result <> 0 then Exit;
    try
      if SelfSigned then
	KeySetX509Sign := KeySetMain
      else begin
	Result := GetValidX509CertifierSet(KeySetX509, KeySetX509Sign, false, true);
	if Result <> 0 then Exit;
      end;
      try
	Result := X509GetCertAndPassphraseDialog(Context, KeySetX509Sign, SignKey, SignCert, IssuerPassphrase,
						 '', SignCertIASN, SignKeyAndPassPrompt, WinHandle);
      finally
	if SelfSigned then begin
	  if Result = kPGPError_X509NeededCertNotAvailable then Result := 0;
	end
	else PGPFreeKeySet(KeySetX509Sign);
      end;
      if Result <> 0 then Exit;
      Result := IsSerialNumberUnique(Context, KeySetMain, SignKey, CertAttributes.SerialNumber, SNUnique);
      if Result <> 0 then Exit;
      if not SNUnique then begin
	Result := kPGPError_DuplicateCert;
	Exit;
      end;
      if not SelfSigned then begin
	GetKeySetWithoutSigner(Context, KeySetMain, SignKey, KeySetFiltered);
	try
	  if CertKeyPrompt <> '' then
	    Result := X509GetKeyDialog(Context, KeySetFiltered, false, KeyToCertify, CertKeyPrompt, WinHandle)
	  else Result := GetKeyByHexID(KeySetFiltered, CertKeyHexID, KeyToCertify);
	finally
	  if KeySetFiltered <> KeySetMain then PGPFreeKeySet(KeySetFiltered);
	end;
      end
      else KeyToCertify := SignKey;
      if Result <> 0 then Exit;
      with CertAttributes do begin
	if KeyToCertify <> SignKey then begin
	  if (CommonName = '') or (Email = '') then begin
	    Result := GetPrimaryUserIDAsAttributes(KeyToCertify, CommonName, Email);
	    if Result <> 0 then Exit;
	  end;
	  // as we already know which certifier key will be used we only have to check with the selected SignKey
	  if HasUserIDCertifierCert(KeyToCertify, CommonName + ' <' + Email + '>', KeySetMain, SignKey) then begin
	    Result := kPGPError_DuplicateUserID;
	    Exit;
	  end;
	end
	else begin
	  if (CommonName = '') or (Email = '') then begin
	    Result := kPGPError_InvalidDistinguishedName;
	    Exit;
	  end
	  else if HasKeyUserID(KeyToCertify, CommonName + ' <' + Email + '>') then begin
	    Result := kPGPError_DuplicateUserID;
	    Exit;
	  end;
	end;
      end;
      Result := GetAttributeListFromRecord(CertAttributes, AttributeList, AttributeCount);
      if Result <> 0 then Exit;
      OptionList := nil;
      Result := PGPBuildOptionList(Context, OptionList,
	[
	  PGPOAttributeValue(Context, @AttributeList, AttributeCount),
	  PGPOPassphrase(Context, IssuerPassphrase),
	  PGPOExpiration(Context, ValidDays)
	]);
      if Result <> 0 then Exit;
      try
	if KeyToCertify = SignKey then
	  Result := PGPCreateSelfSignedX509Certificate(SignKey, Certificate, OptionList, PGPOLastOption(Context))
	else Result := PGPCreateX509Certificate(SignCert, KeyToCertify, Certificate, OptionList, PGPOLastOption(Context));
	if Result <> 0 then Exit;
	Result := KeyRings.UpdateKeyRings;
	if Result <> 0 then Exit;
	Result := GetX509CertPropsList(Certificate, CertPropsList);
      finally
	PGPFreeOptionList(OptionList);
      end;
    finally
      if SignKeyAndPassPrompt <> '' then PGPFreeData(IssuerPassphrase);
      FinitX509CertKeySet(KeySetX509);
      Result := TranslateX509ErrorToPGPError(Result);
    end;
  end
  else Result := kPGPError_LazyProgrammer;
end;

// self-signed certificates can only be removed including the user ID they are attached to
// user IDs of certificates can only be removed if they don't have any other signatures assigned
// if there's only a single user ID left on a key the key will be removed from the keyring all together
function RemoveX509Certificate(const CertKeyPrompt, RemoveCertIASN: String; IncludeUserID: Longbool;
			       var CertPropsList: TX509CertPropsList; WinHandle: Integer): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  KeySetX509	: pPGPKeySet;
  RemoveCertKey : pPGPKeyDBObj;
  RemoveCert	: pPGPKeyDBObj;
  SignKey	: pPGPKeyDBObj;
  RemoveUserID	: pPGPKeyDBObj;
  SigRemoved	: Longbool;
  KeySet	: pPGPKeySet;
  KeyList	: pPGPKeyList;
  KeyIter	: pPGPKeyIter;
  UserIDFound	: pPGPKeyDBObj;
  SignatureFound: pPGPKeyDBObj;
begin
  if PGP7X then begin
    Result := InitX509CertKeySet(Context, KeySetMain, KeySetX509);
    if Result <> 0 then Exit;
    try
      if CertKeyPrompt <> '' then begin // prompt for a certified key
	Result := X509GetKeyDialog(Context, KeySetX509, true, RemoveCertKey, CertKeyPrompt, WinHandle);
	if Result <> 0 then Exit;
	if PGP81 then begin
	  RemoveCert := RemoveCertKey;
	  RemoveCertKey := PGPPeekKeyDBObjKey(RemoveCert);
	end
	else RemoveCert := GetValidX509CertFromKey('', RemoveCertKey, nil, KeySetX509);
      end
      else begin // do we have a hex ID or an IASN?
	Result := GetKeyByHexID(KeySetX509, RemoveCertIASN, RemoveCertKey);
	if Result = 0 then
	  RemoveCert := GetValidX509CertFromKey('', RemoveCertKey, nil, KeySetX509)
	else begin
	  RemoveCert := GetCertByCertIASN(KeySetX509, RemoveCertIASN, SignKey);
	  if RemoveCert <> nil then RemoveCertKey := PGPPeekKeyDBObjKey(RemoveCert);
	end;
      end;
      RemoveUserID := nil;
      if RemoveCert <> nil then begin
	RemoveUserID := PGPPeekKeyDBObjUserID(RemoveCert);
	Result := GetX509CertPropsList(RemoveCert, CertPropsList);
      end
      else if Result = 0 then Result := kPGPError_X509NeededCertNotAvailable;
      if Result <> 0 then Exit;
      with CertPropsList do begin
	if not X509CertProps[pred(Count)].x509IsRoot then begin
	  Result := PGPRemoveSig(RemoveCert);
	  if Result <> 0 then Exit;
	  SigRemoved := true;
	end
	else SigRemoved := false;
      end;
      if IncludeUserID then begin
	Result := GetSingleKeyDBObjIter(RemoveUserID, KeySet, KeyList, KeyIter);
	if Result = 0 then begin
	  try
	    while PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_UserID, UserIDFound) = 0 do begin
	      if UserIDFound = RemoveUserID then begin
		if (PGPKeyIterRewindKeyDBObj(KeyIter, kPGPKeyDBObjType_Signature) <> 0)
		or (PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_Signature, SignatureFound) <> 0)
		or (
		  (SignatureFound = RemoveCert) and
		  (PGPKeyIterNextKeyDBObj(KeyIter, kPGPKeyDBObjType_Signature, SignatureFound) <> 0)
		) then begin
		  Result := PGPRemoveUserID(RemoveUserID);
		  if Result <> 0 then begin
		    PGPFreeKeySet(KeySet);
		    KeySet := nil;
		    if (PGPNewSingletonKeySet(RemoveCertKey, KeySet) = 0) and (CountUserIDs(KeySet) = 1) then begin
		      Result := PGPRemoveKeys(KeySet, KeySetMain);
		    end;
		  end;
		  if Result = 0 then SigRemoved := true
		end;
		Break;
	      end;
	    end;
	  finally
	    PGPFreeKeyIter(KeyIter);
	    PGPFreeKeyList(KeyList);
	    PGPFreeKeySet(KeySet);
	  end;
	end;
      end;
      if SigRemoved then
	Result := KeyRings.UpdateKeyRings
      else if not IncludeUserID then Result := kPGPError_X509SelfSignedCert;
    finally
      FinitX509CertKeySet(KeySetX509);
      Result := TranslateX509ErrorToPGPError(Result);
    end;
  end
  else Result := kPGPError_LazyProgrammer;
end;

// import binary or armored standard X509 certificates or PGP keys to keyring(s)
function ImportX509Certificate(const CertKeyPrompt, CertPassPrompt, CertData: String; PKCS12Passphrase: PChar;
			       FromFile: Longbool; var CertPropsList: TX509CertPropsList; WinHandle: Integer): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  KeyPropsList	: TKeyPropsList;
  ImportKeySet	: pPGPKeySet;
begin
  if PGP7X then begin
    if CertData <> '' then begin
      KeyPropsList := nil;
      try
	Result := KeyImportDialog(CertKeyPrompt, CertPassPrompt, CertData, PKCS12Passphrase, KeyPropsList, FromFile,
				  IgnoreFlag_None, spgpKeyPropFlag_KeyID, WinHandle);
	if Result <> 0 then Exit;
	Result := KeyRings.InitKeyRings(Context, KeySetMain);
	if Result <> 0 then Exit;
	try
	  Result := GetKeySetByAnyIDs(Context, KeySetMain, KeyPropsList.CommaText, ImportKeySet);
	  if Result <> 0 then Exit;
	  try
	    Result := GetX509CertSetProps(ImportKeySet, KeySetMain, nil, '', kPGPKeyOrdering_UserID, CertPropsList);
	    if Result > 0 then Result := 0;
	  finally
	    PGPFreeKeySet(ImportKeySet);
	  end;
	finally
	  KeyRings.FreeKeyRings;
	end;
      finally
	KeyPropsList.Free;
      end;
    end
    else Result := kPGPError_InvalidInputFormat;
    Result := TranslateX509ErrorToPGPError(Result);
  end
  else Result := kPGPError_LazyProgrammer;
end;

// write certificate(s) and size information to chain
function WriteDataToChain(const CertData: PChar; var Chain: String;
			  var ChainPos: PGPSize; CertLen: PGPSize): Integer;
var
  pChain	: PChar;
begin
  Result := 0;
  pChain := Pointer(Chain);
  if ChainPos = 0 then begin
    pChain[0]:=chr(CertLen shr 16);
    pChain[1]:=chr(CertLen shr 8);
    pChain[2]:=chr(CertLen);
    ChainPos := 3;
  end
  else begin
    SetLength(Chain, ChainPos + CertLen + 3);
    pChain := Pointer(Chain);
    pChain[ChainPos + 0]:=chr(CertLen shr 16);
    pChain[ChainPos + 1]:=chr(CertLen shr 8);
    pChain[ChainPos + 2]:=chr(CertLen);
    Move(CertData^, pChain[ChainPos + 3], CertLen);
    inc(ChainPos, CertLen + 3);
    if ChainPos > $FFFFFF then begin
      Result := kPGPError_LDAPSizelimitExceeded;
      Chain := '';
    end;
  end;
end;

// create a certificate chain from any keyset
function CreateCertChainFromKeySet(CertKeySet: pPGPKeySet; var CertChain: String): Integer;
var
  KeyCount	: PGPUInt32;
  KeyIter	: pPGPKeyIter;
  CertList	: TStringList;
  Certificate	: pPGPKeyDBObj;
  KeyFound	: pPGPKeyDBObj;
  CreaTime	: PGPTime;
  ChainPos	: PGPSize;
  CertData	: TX509CertData;
  CertIndex	: Integer;
  CertLength	: PGPSize;
begin
  Result := PGPCountKeys(CertKeySet, KeyCount);
  if Result <> 0 then Exit;
  if KeyCount = 0 then begin
    Result := kPGPError_X509NeededCertNotAvailable;
    Exit;
  end;
  Result := PGPNewKeyIterFromKeySet(CertKeySet, KeyIter);
  if Result <> 0 then Exit;
  try
    CertList := TStringList.Create;
    if CertList <> nil then begin
      try
	while FindCertificate(KeyIter, CertKeySet, Certificate, KeyFound) = 0 do begin
	  Result := PGPGetKeyDBObjTimeProperty(Certificate, kPGPSigProperty_Creation, CreaTime);
	  if Result <> 0 then Exit;
	  CertList.AddObject(IntToHex(CreaTime, 8), TObject(Certificate));
	end;
	CertList.Sort;
	ChainPos := 0;
	CertData[0] := #0;
	SetLength(CertChain, 3);
	Result := WriteDataToChain(CertData, CertChain, ChainPos, 0);
	if Result <> 0 then Exit;
	CertIndex := pred(CertList.Count);
	while CertIndex >= 0 do begin
	  Result := PGPGetKeyDBObjDataProperty(pPGPKeyDBObj(CertList.Objects[CertIndex]), kPGPSigProperty_X509Certificate,
					       @CertData, MAX_509_CERT_SIZE, CertLength);
	  if Result <> 0 then Exit;
	  try
	    Result := WriteDataToChain(CertData, CertChain, ChainPos, CertLength);
	    if Result <> 0 then Exit;
	  except
	    on EOutOfMemory do begin
	      Result := kPGPError_OutOfMemory;
	      CertChain := '';
	      Exit;
	    end;
	  end;
	  dec(CertIndex);
	end;
	if ChainPos > 3 then begin
	  CertLength := 0;
	  CertData[0] := #0;
	  WriteDataToChain(CertData, CertChain, CertLength, ChainPos - 3);
	end
	else begin
	  Result := kPGPError_X509NeededCertNotAvailable;
	  Exit;
	end;
      finally
	CertList.Free;
      end;
    end
    else Result := kPGPError_OutOfMemory;
  finally
    PGPFreeKeyIter(KeyIter);
  end;
end;

// either export a single certificate or a certificate chain with parent certificate(s) included
// chains suitable for VerifyX509CertificateChain can be created with Armor disabled
function ExportX509Certificate(const CertKeyOrKeyAndPassPrompt, ExportCertIASN: String; SecKeyPassphrase: PChar;
			       Armor, Chain, Secret, ToFile: Longbool; var CertData: String;
			       var CertPropsList: TX509CertPropsList; WinHandle: Integer): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  KeySetX509	: pPGPKeySet;
  ExportCert	: pPGPKeyDBObj;
  ExportCertKey	: pPGPKeyDBObj;
  SignKey	: pPGPKeyDBObj;
  CertFile	: String;
  FileExtension	: String;
  ExportCertSet	: pPGPKeySet;
  OptionList	: pPGPOptionList;
  PassOption	: pPGPOptionList;
  CertOutput	: PChar;
  CertLength	: PGPSize;
  OptionListCopy: pPGPOptionList;
  CertSigner	: pPGPKeyDBObj;
begin
  if PGP7X then begin
    if Secret then Chain := false;
    Result := InitX509CertKeySet(Context, KeySetMain, KeySetX509);
    if Result <> 0 then Exit;
    try
      ExportCert := nil;
      if CertKeyOrKeyAndPassPrompt <> '' then begin // prompt for a certified key
	if Secret then begin
	  Result := X509GetCertAndPassphraseDialog(Context, KeySetX509, ExportCertKey, ExportCert, SecKeyPassphrase,
						   '', ExportCertIASN, CertKeyOrKeyAndPassPrompt, WinHandle);
	end
	else begin
	  Result := X509GetKeyDialog(Context, KeySetX509, true, ExportCertKey, CertKeyOrKeyAndPassPrompt, WinHandle);
	  if Result <> 0 then Exit;
	  if PGP81 then begin
	    ExportCert := ExportCertKey;
	    ExportCertKey := PGPPeekKeyDBObjKey(ExportCert);
	  end
	  else ExportCert := GetValidX509CertFromKey('', ExportCertKey, nil, KeySetX509);
	end;
      end
      else begin // do we have a hex ID or an IASN?
	Result := GetKeyByHexID(KeySetX509, ExportCertIASN, ExportCertKey);
	if Result = 0 then
	  ExportCert := GetValidX509CertFromKey('', ExportCertKey, nil, KeySetX509)
	else begin
	  ExportCert := GetCertByCertIASN(KeySetX509, ExportCertIASN, SignKey);
	  if ExportCert <> nil then begin
	    ExportCertKey := PGPPeekKeyDBObjKey(ExportCert);
	    Result := 0;
	  end;
	end;
	if ExportCert = nil then Result := kPGPError_X509NeededCertNotAvailable;
      end;
      if Result <> 0 then Exit;
      Secret := (Secret and GetKeyPropIsSecret(ExportCertKey));
      if ToFile then begin
	CertFile := CertData;
	FileExtension := LowerCase(ExtractFileExt(CertFile));
	if Armor then begin
	  if FileExtension <> '.pem' then CertFile := CertFile + '.pem';
	end
	else if Secret then begin
	  if FileExtension <> '.p12' then CertFile := CertFile + '.p12';
	end
	else if Chain then begin
	  if FileExtension <> '.chain' then CertFile := CertFile + '.chain';
	end
	else if (FileExtension <> '.cer') then CertFile := CertFile + '.cer';
      end;
      ExportCertSet := nil;
      OptionList := nil;
      try
	if not Chain then begin
	  if Secret then begin
	    if SecKeyPassphrase <> nil then
	      PassOption := PGPOPassphrase(Context, SecKeyPassphrase)
	    else begin
	      Result := kPGPError_BadPassphrase;
	      Exit;
	    end;
	  end
	  else PassOption := PGPONullOption(Context);
	  Result := PGPBuildOptionList(Context, OptionList,
	    [
	      PassOption,
	      PGPOExportUserID(Context, ExportCert),
	      PGPOArmorOutput(Context, PGPBoolean(Armor) and 1),
	      PGPOExportFormat(Context, kPGPExportFormat_X509Cert),
	      PGPOExportPrivateKeys(Context, PGPBoolean(Secret) and 1),
	      PGPOExportPrivateSubKeys(Context, PGPBoolean(Secret) and 1),
	      PGPOAllocatedOutputBuffer(Context, CertOutput, $FFFFFFFF, CertLength)
	    ]);
	  if Result <> 0 then Exit;
	  try
	    Result := PGPExport(Context, OptionList, PGPOLastOption(Context));
	    if Result <> 0 then Exit;
	    try
	      Result := CopyDataToString(CertOutput, CertData, CertLength);
	      if Result <> 0 then Exit;
	      Result := GetX509CertPropsList(ExportCert, CertPropsList);
	    finally
	      PGPFreeData(CertOutput);
	    end;
	  finally
	    PGPFreeOptionList(OptionList);
	  end;
	end
	else begin
	  Result := PGPNewEmptyKeySet(PGPPeekKeySetKeyDB(KeySetX509), ExportCertSet);
	  if Result <> 0 then Exit;
	  if Armor then begin
	    Result := PGPBuildOptionList(Context, OptionList,
	      [
		PGPOArmorOutput(Context, PGPTrue),
		PGPOExportFormat(Context, kPGPExportFormat_X509Cert),
		PGPOAllocatedOutputBuffer(Context, CertOutput, $FFFFFFFF, CertLength)
	      ]);
	    if Result <> 0 then Exit;
	  end;
	  try
	    CertData := '';
	    repeat
	      if ExportCert <> nil then begin
		Result := PGPAddKey(ExportCert, ExportCertSet);
		if Result <> 0 then Exit;
		if Armor then begin
		  Result := PGPCopyOptionList(OptionList, OptionListCopy);
		  if Result <> 0 then Exit;
		  try
		    Result := PGPAppendOptionList(OptionListCopy, [PGPOExportUserID(Context, ExportCert)]);
		    if Result <> 0 then Exit;
		    Result := PGPExport(Context, OptionListCopy, PGPOLastOption(Context));
		    if Result <> 0 then Exit;
		    try
		      CertData := CertData + CertOutput + CRLF;
		    finally
		      PGPFreeData(CertOutput);
		    end;
		  finally
		    PGPFreeOptionList(OptionListCopy);
		  end;
		end;
		Result := PGPGetSigCertifierKey(ExportCert, KeySetX509, CertSigner);
		if (Result <> 0) and (Result <> kPGPError_ItemNotFound) then Exit;
		if (Result = 0) and (CertSigner <> ExportCertKey) then
		  ExportCertKey := CertSigner
		else Chain := false;
	      end
	      else begin
		Result := kPGPError_X509NeededCertNotAvailable;
		Exit;
	      end;
	      ExportCert := GetValidX509CertFromKey(GetCertPropIssuerName(ExportCert), ExportCertKey, nil, KeySetX509);
	    until not Chain;
	  finally
	    PGPFreeOptionList(OptionList);
	  end;
	  if not Armor then begin
	    Result := CreateCertChainFromKeySet(ExportCertSet, CertData);
	    if Result <> 0 then Exit;
	  end
	  else Delete(CertData, pred(Length(CertData)), Length(CRLF));
	  Result := GetX509CertSetProps(ExportCertSet, KeySetX509, nil, '', kPGPKeyOrdering_CreationDate, CertPropsList);
	  if Result > 0 then Result := 0;
	end;
      finally
	PGPFreeKeySet(ExportCertSet);
      end;
      if ToFile then begin
	Result := WriteDataToFile(CertData, CertFile);
	CertData := CertFile;
      end;
    finally
      if CertKeyOrKeyAndPassPrompt <> '' then PGPFreeData(SecKeyPassphrase);
      FinitX509CertKeySet(KeySetX509);
      if Result <> 0 then CertData := '';
      Result := TranslateX509ErrorToPGPError(Result);
    end;
  end
  else Result := kPGPError_LazyProgrammer;
end;

// create a temporary keyDB for checking certificates
function GetX509CertImportDB(Context: pPGPContext; const CertData: String;
			     var CertImportDB: pPGPKeyDB; FromFile: Longbool): Integer;
var
  CertInput	: String;
  LoopCount	: Integer;
  OptionList	: pPGPOptionList;
begin
  Result := 0;
  OptionList := nil;
  CertImportDB := nil;
  if FromFile then
    Result := ReadDataFromFile(CertData, CertInput)
  else CertInput := CertData;
  if Result <> 0 then Exit;
  LoopCount := 0;
  repeat
    inc(LoopCount);
    if Result <> kPGPError_InvalidPKCS7Encoding then begin
      Result := PGPBuildOptionList(Context, OptionList,
	[
	  PGPOInputFormat(Context, kPGPInputFormat_X509DataInPKCS7),
	  PGPOInputBuffer(Context, PChar(CertInput), Length(CertInput))
	]);
    end
    else begin
      Result := PGPBuildOptionList(Context, OptionList,
	[
	  PGPOInputFormat(Context, kPGPInputFormat_PEMEncodedX509Cert),
	  PGPOInputBuffer(Context, PChar(CertInput), Length(CertInput))
	]);
    end;
    if Result <> 0 then Exit;
    try
      Result := PGPImport(Context, CertImportDB, OptionList, PGPOLastOption(Context));
    finally
      PGPFreeOptionList(OptionList);
      OptionList := nil;
    end;
  until (Result = 0) or (LoopCount = 2);
  if Result < 0 then Result := kPGPError_X509CertificateParseError;
end;

// the required certificate chain can be created using ExportX509Certificate() with Chain enabled and Armor disabled
function VerifyX509CertificateChain(const CertChain: String; FromFile: Longbool): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  KeySetX509	: pPGPKeySet;
  CertDataBuf	: String;
  RootCertsBuf	: String;
  CertDataDB	: pPGPKeyDB;
begin
  if PGP7X then begin
    Result := InitX509CertKeySet(Context, KeySetMain, KeySetX509);
    if Result <> 0 then Exit;
    try
      Result := CreateCertChainFromKeySet(KeySetX509, RootCertsBuf);
      if Result <> 0 then Exit;
      if FromFile then
	Result := ReadDataFromFile(CertChain, CertDataBuf)
      else CertDataBuf := CertChain;
      if Result <> 0 then Exit;
      Result := PGPVerifyX509CertificateChain(Context, PChar(CertDataBuf), PChar(RootCertsBuf));
      if Result = kPGPError_X509SelfSignedCert then Exit;
      if Result <> 0 then begin
	if GetX509CertImportDB(Context, CertDataBuf, CertDataDB, false) = 0 then begin
	  Result := CreateCertChainFromKeySet(PGPPeekKeyDBRootKeySet(CertDataDB), CertDataBuf);
	  if Result = 0 then Result := PGPVerifyX509CertificateChain(Context, PChar(CertDataBuf), PChar(RootCertsBuf));
	end;
      end;
    finally
      FinitX509CertKeySet(KeySetX509);
      Result := TranslateX509ErrorToPGPError(Result);
    end;
  end
  else Result := kPGPError_LazyProgrammer;
end;

// verify certificate chain in given key DB
function VerifyX509CertificateChainDB(CertKeyDB: pPGPKeyDB): TX509ChainValidity;
var
  CertChain	: String;
begin
  Result := CertChain_Invalid;
  if PGP7X and (CreateCertChainFromKeySet(PGPPeekKeyDBRootKeySet(CertKeyDB), CertChain) = 0) then begin
    case PGPVerifyX509CertificateChain(PGPPeekKeyDBContext(CertKeyDB), PChar(CertChain), PChar(CertChain)) of
      0: Result := CertChain_Valid;
      kPGPError_X509SelfSignedCert: Result := CertChain_Root;
    end;
  end;
end;

// select certificate and return it's properties
function SelectX509CertificateDialog(const CertKeyPrompt: String;
				     var CertPropsList: TX509CertPropsList;
				     WinHandle: Integer): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  KeySetX509	: pPGPKeySet;
  CertKey	: pPGPKeyDBObj;
  Certificate	: pPGPKeyDBObj;
begin
  if PGP7X then begin
    Result := InitX509CertKeySet(Context, KeySetMain, KeySetX509);
    if Result <> 0 then Exit;
    try
      Result := X509GetKeyDialog(Context, KeySetX509, true, CertKey, CertKeyPrompt, WinHandle);
      if Result <> 0 then Exit;
      if PGP81 then
        Certificate := CertKey
      else Certificate := GetValidX509CertFromKey('', CertKey, nil, KeySetX509);
      if Certificate <> nil then
	Result := GetX509CertPropsList(Certificate, CertPropsList)
      else Result := kPGPError_X509NeededCertNotAvailable;
    finally
      FinitX509CertKeySet(KeySetX509);
      Result := TranslateX509ErrorToPGPError(Result);
    end;
  end
  else Result := kPGPError_LazyProgrammer;
end;

// check whether a certificate with the given serial number already exists on the signer's keyring
function IsSerialNumberUnique(Context: pPGPContext; KeySetToCheck: pPGPKeySet; SignKey: pPGPKeyDBObj;
			      const SerialNumber: String; var Unique: Longbool): Integer;
var
  SignKeyCertSet: pPGPKeySet;
  KeyIter	: pPGPKeyIter;
  Certificate	: pPGPKeyDBObj;
  KeyFound	: pPGPKeyDBObj;
begin
  Result := 0;
  Unique := false;
  if IntToStr(StrToIntDef(SerialNumber, 0)) <> SerialNumber then Exit;
  Result := GetCertKeySetBySignKey(Context, KeySetToCheck, SignKey, true, false, SignKeyCertSet);
  if Result = kPGPError_X509NeededCertNotAvailable then begin
    Unique := true;
    Result := 0;
    Exit;
  end
  else if Result <> 0 then Exit;
  try
    Result := PGPNewKeyIterFromKeySet(SignKeyCertSet, KeyIter);
    if Result <> 0 then Exit;
    try
      Unique := true;
      while (FindCertificate(KeyIter, KeySetToCheck, Certificate, KeyFound) = 0) and Unique do begin
	if (KeyFound = SignKey) and (GetCertPropOwnerSerialNumber(Certificate) = SerialNumber) then Unique := false;
      end;
    finally
      PGPFreeKeyIter(KeyIter);
    end;
  finally
    PGPFreeKeySet(SignKeyCertSet);
  end;
end;

// retrieve property values of certificate
function GetX509CertPropsRec(Certificate: pPGPKeyDBObj; var CertPropsRec: TX509CertPropsRec): Integer;
begin
  Result := 0;
  try
    with CertPropsRec do begin
      x509CertificateID := GetCertIssuerInfos(Certificate, x509IssuerKeyAlgorithm, x509IssuerHashAlgorithm);
      x509IssuerLongName := GetCertPropIssuerName(Certificate);
      x509IssuerSerialNumber := GetCertPropIssuerSerialNumber(Certificate);
      x509OwnerKeyHexID := GetCertPropOwnerKeyID(Certificate);
      x509OwnerCertUserID := GetCertPropOwnerUserID(Certificate);
      x509OwnerLongName := GetCertPropOwnerName(Certificate);
      x509OwnerSerialNumber := GetCertPropOwnerSerialNumber(Certificate);
      x509CreaTimeStr := GetCertPropCertCreationTimeStr(Certificate);
      x509ExpTimeStr := GetCertPropCertExpirationTimeStr(Certificate);
      x509CreaTimeNum := GetCertPropCertCreationTime(Certificate);
      x509ExpTimeNum := GetCertPropCertExpirationTime(Certificate);
      x509IsVerified := GetCertPropIsVerified(Certificate);
      x509IsCorrupt := GetCertPropIsCorrupt(Certificate);
      x509IsRevoked := GetCertPropIsRevoked(Certificate);
      x509IsExpired := GetCertPropIsExpired(Certificate);
      x509IsRoot := GetCertPropIsRoot(Certificate);
    end;
  except
    Result := -1;
  end;
end;

// retrieve properties and IASN of single certificate
function GetX509CertPropsList(Certificate: pPGPKeyDBObj; var CertPropsList: TX509CertPropsList): Integer;
begin
  if CertPropsList = nil then CertPropsList := TX509CertPropsList.Create;
  if CertPropsList <> nil then with CertPropsList do begin
    Result := GetX509CertPropsRec(Certificate, pX509CertPropsRec(Objects[Add(GetCertPropIASN(Certificate))])^);
  end
  else Result := kPGPError_OutOfMemory;
end;

// returns number of certificates found or error
// strings contain the supposedly unique IssuerAndSerialNumber (binary) structure
function GetX509CertSetProps(CertSet, KeySetToSearch: pPGPKeySet; SignKey: pPGPKeyDBObj; const UserID: String;
			     Order: PGPKeyOrdering; var CertPropsList: TX509CertPropsList): Integer;
var
  CertList	: pPGPKeyList;
  CertIter	: pPGPKeyIter;
  Certificate	: pPGPKeyDBObj;
  KeyFound	: pPGPKeyDBObj;
begin
  if PGP7X then begin
    if CertPropsList = nil then CertPropsList := TX509CertPropsList.Create;
    if CertPropsList = nil then begin
      Result := kPGPError_OutOfMemory;
      Exit;
    end;
    Result := PGPOrderKeySet(CertSet, Order, PGPFalse, CertList);
    if Result <> 0 then Exit;
    CertIter := nil;
    try
      Result := PGPNewKeyIter(CertList, CertIter);
      if Result <> 0 then Exit;
      while FindCertificate(CertIter, KeySetToSearch, Certificate, KeyFound) = 0 do begin
	if ((SignKey = nil) or (KeyFound = SignKey))
	and ((UserID = '') or (GetCertPropOwnerUserID(Certificate) = UserID)) then begin
	  with CertPropsList do begin
	    Result := GetX509CertPropsRec(Certificate, pX509CertPropsRec(Objects[Add(GetCertPropIASN(Certificate))])^);
	  end;
	end;
      end;
      if Result = 0 then Result := CertPropsList.Count;
    finally
      PGPFreeKeyIter(CertIter);
      PGPFreeKeyList(CertList);
    end;
  end
  else Result := kPGPError_LazyProgrammer;
end;

// returns number of certificates found or error
function GetX509CertificateProperties(const CertData: String; FromFile: Longbool;
				      var CertPropsList: TX509CertPropsList): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  CertDBToCheck	: pPGPKeyDB;
  CertSetToCheck: pPGPKeySet;
begin
  if PGP7X then begin
    Result := KeyRings.InitKeyRings(Context, KeySetMain);
    if Result <> 0 then Exit;
    try
      Result := GetX509CertImportDB(Context, CertData, CertDBToCheck, FromFile);
      if Result <> 0 then Exit;
      try
	Result := PGPNewKeySet(CertDBToCheck, CertSetToCheck);
	if Result <> 0 then Exit;
	try
	  Result := GetX509CertSetProps(CertSetToCheck, KeySetMain, nil, '', kPGPKeyOrdering_UserID, CertPropsList);
	finally
	  PGPFreeKeySet(CertSetToCheck);
	end;
      finally
	PGPFreeKeyDB(CertDBToCheck);
      end;
    finally
      KeyRings.FreeKeyRings;
      Result := TranslateX509ErrorToPGPError(Result);
    end;
  end
  else Result := kPGPError_LazyProgrammer;
end;

// returns number of certificates found or error
// KeyFilterFlags is applied to any kind of search
// SignKeyID can narrow down the search with CertKey...IDs
// IncludeSigner is only valid without CertKey...IDs specified
// in this case you will get the following results depending on SignKeyHexID and IncludeSigner:
// (SignKey = nil) and (IncludeSigner = false)  => all certified keys on keyring
// (SignKey = nil) and (IncludeSigner = true)   => all certifier keys on keyring (signing keys only)
// (SignKey <> nil) and (IncludeSigner = false) => all SignKey-certified keys on keyring without certifier
// (SignKey <> nil) and (IncludeSigner = true)  => all SignKey-certified keys on keyring including certifier
function FindX509CertProps(const CertKeyUserID, CertKeyHexID, SignKeyHexID: String; KeyFilterFlags: DWord;
			   IncludeSigner: Longbool; var CertPropsList: TX509CertPropsList): Integer;
var
  Context	: pPGPContext;
  KeySetMain	: pPGPKeySet;
  CertSet	: pPGPKeySet;
  KeyFilter	: pPGPFilter;
  KeySetToCheck	: pPGPKeySet;
  KeySetRoots	: pPGPKeySet;
  SignKey	: pPGPKeyDBObj;
  CertKey	: pPGPKeyDBObj;
  CertSetBuf	: pPGPKeySet;
  UserIDFilter	: pPGPFilter;
  KeyCount	: PGPUInt32;
begin
  if PGP7X then begin
    Result := KeyRings.InitKeyRings(Context, KeySetMain);
    if Result <> 0 then Exit;
    CertSet := nil;
    try
      if KeyFilterFlags <> KeyFilterFlag_AllKeys then begin
	Result := GetKeyFilter(Context, KeyFilterFlags, KeyFilter);
	if Result <>0 then Exit;
	try
	  Result := PGPFilterKeySet(KeySetMain, KeyFilter, KeySetToCheck);
	  if Result <> 0 then Exit;
	finally
	  PGPFreeFilter(KeyFilter);
	end;
	if KeyFilterFlags and KeyFilterFlag_X509Root <> 0 then begin
	  Result := GetValidX509CertifierSet(KeySetToCheck, KeySetRoots, true, false);
	  if Result <> 0 then Exit;
	  PGPFreeKeySet(KeySetToCheck);
	  KeySetToCheck := KeySetRoots;
	end;
      end
      else KeySetToCheck := KeySetMain;
      try
	if SignKeyHexID <> '' then begin
	  Result := GetKeyByHexID(KeySetMain, SignKeyHexID, SignKey);
	  if (Result = 0) and not GetKeyPropCanSign(SignKey) then Result := kPGPError_KeyUnusableForSignature;
	  if Result <> 0 then Exit;
	end
	else SignKey := nil;
	if (CertKeyUserID <> '') or (CertKeyHexID <> '') then begin
	  if CertKeyUserID <> '' then begin
	    Result := GetUserIDFilter(Context, CertKeyUserID, true, UserIDFilter);
	    if Result <> 0 then Exit;
	  end
	  else UserIDFilter := nil;
	  try
	    if CertKeyHexID <> '' then begin
	      Result := GetKeyByHexID(KeySetToCheck, CertKeyHexID, CertKey);
	      if Result <> 0 then Exit;
	      Result := PGPNewSingletonKeySet(CertKey, CertSet);
	      if Result <> 0 then Exit;
	      if UserIDFilter <> nil then begin
		CertSetBuf := CertSet;
		CertSet := nil;
		try
		  Result := PGPFilterKeySet(CertSetBuf, UserIDFilter, CertSet);
		  if Result <> 0 then Exit;
		finally
		  PGPFreeKeySet(CertSetBuf);
		end;
	      end;
	    end
	    else begin
	      Result := PGPFilterKeySet(KeySetToCheck, UserIDFilter, CertSet);
	      if Result <> 0 then Exit;
	    end;
	  finally
	    PGPFreeFilter(UserIDFilter);
	  end;
	end
	else begin
	  Result := GetCertKeySetBySignKey(Context, KeySetToCheck, SignKey, IncludeSigner, false, CertSet);
	  if Result <> 0 then Exit;
	  Result := PGPCountKeys(CertSet, KeyCount);
	  if (Result <> 0) or (KeyCount = 0) then Exit;
	end;
	Result := GetX509CertSetProps(CertSet, KeySetMain, SignKey, CertKeyUserID, kPGPKeyOrdering_UserID, CertPropsList);
      finally
	if KeySetToCheck <> KeySetMain then PGPFreeKeySet(KeySetToCheck);
      end;
    finally
      PGPFreeKeySet(CertSet);
      KeyRings.FreeKeyRings;
      Result := TranslateX509ErrorToPGPError(Result);
    end;
  end
  else Result := kPGPError_LazyProgrammer;
end;

end.

