{$A+,B-,C-,D+,E-,F-,G+,H+,I+,J+,K-,L+,M-,N+,O+,P+,Q-,R-,S-,T-,U-,V+,W-,X+,Y+,Z4}
unit X509Comp;

{------------------------------------------------------------------------------}
{                                                                              }
{                 This unit is partly based on Steve Heller's                  }
{         SPGP sources available from http://www.oz.net/~srheller/spgp/        }
{                                                                              }
{                Portions created by Michael in der Wiesche are                }
{              Copyright (C) 2004-2006 by Michael in der Wiesche               }
{                                                                              }
{------------------------------------------------------------------------------}

interface

uses
  Forms,
  Windows,
  Classes,
  TypInfo,
  SysUtils,
  StdCtrls,
  UTF8,
  pgpBase,
  pgpErrors,
  pgpPubTypes,
  pgpMemoryMgr,
  PGP2Comp,
  X509Types,
  X509Funcs;

// Register --------------------------------------------------------------------

procedure Register;


// Common types ----------------------------------------------------------------

type
  TCertAttributes = Class(TPersistent)
  private
    function GetSerialNumber: Longint;
    procedure SetSerialNumber(Value: Longint);
  public
    Values: TX509CertAttributes;
  published
    property CommonName: String
      read Values.CommonName
      write Values.CommonName;
    property Email: String
      read Values.Email
      write Values.Email;
    property OrganizationName: String
      read Values.OrganizationName
      write Values.OrganizationName;
    property OrganizationalUnitName: String
      read Values.OrganizationalUnitName
      write Values.OrganizationalUnitName;
    property SurName: String
      read Values.SurName
      write Values.SurName;
    property SerialNumber: Longint
      read GetSerialNumber
      write SetSerialNumber;
    property Country: String
      read Values.Country
      write Values.Country;
    property Locality: String
      read Values.Locality
      write Values.Locality;
    property State: String
      read Values.State
      write Values.State;
    property StreetAddress: String
      read Values.StreetAddress
      write Values.StreetAddress;
    property Title: String
      read Values.Title
      write Values.Title;
    property Description: String
      read Values.Description
      write Values.Description;
    property PostalCode: String
      read Values.PostalCode
      write Values.PostalCode;
    property POBOX: String
      read Values.POBOX
      write Values.POBOX;
    property PhysicalDeliveryOfficeName: String
      read Values.PhysicalDeliveryOfficeName
      write Values.PhysicalDeliveryOfficeName;
    property TelephoneNumber: String
      read Values.TelephoneNumber
      write Values.TelephoneNumber;
    property X121Address: String
      read Values.X121Address
      write Values.X121Address;
    property ISDN: String
      read Values.ISDN
      write Values.ISDN;
    property DestinationIndicator: String
      read Values.DestinationIndicator
      write Values.DestinationIndicator;
    property Name: String
      read Values.Name
      write Values.Name;
    property GivenName: String
      read Values.GivenName
      write Values.GivenName;
    property Initials: String
      read Values.Initials
      write Values.Initials;
    property HouseIdentifier: String
      read Values.HouseIdentifier
      write Values.HouseIdentifier;
    property DirectoryManagementDomain: String
      read Values.DirectoryManagementDomain
      write Values.DirectoryManagementDomain;
    property DomainComponent: String
      read Values.DomainComponent
      write Values.DomainComponent;
    property UnstructuredName: String
      read Values.UnstructuredName
      write Values.UnstructuredName;
    property UnstructuredAddress: String
      read Values.UnstructuredAddress
      write Values.UnstructuredAddress;
    property RFC822Name: String
      read Values.RFC822Name
      write Values.RFC822Name;
    property DNSName: String
      read Values.DNSName
      write Values.DNSName;
    property AnotherName: String
      read Values.AnotherName
      write Values.AnotherName;
    property IPAddress: String
      read Values.IPAddress
      write Values.IPAddress;
    property CertificateExtension: String
      read Values.CertificateExtension
      write Values.CertificateExtension;
  end;


// Common Procs ----------------------------------------------------------------

type
  TOnFailure = procedure(ErrorCode: Longint; const ErrorMsg: String) of Object;
  TOnEnterPassphrase = procedure(const Passphrase: PChar; var Cancel: Longbool) of Object;


// TX509RequestHandler ---------------------------------------------------------

type
  TX509RequestHandler = class(TComponent)
  private
    FParentHandle: THandle;
    FArmor: Longbool;
    FCertAttributes: TCertAttributes;
    FCertificate: String;
    FCertPropsRec: TX509CertPropsRec;
    FKeyAndPassPrompt: String;
    FKeyHexIDorIASN: String;
    FKeyProps: TKeyProps;
    FKeyPropsRec: TKeyPropsRec;
    FRequestData: String;
    FValidDays: Longint;
    FOnFailure: TOnFailure;
    FOnEnterPassphrase: TOnEnterPassphrase;
  protected
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function CreateCertRequest: Longint;
    function CreateCertFromRequest: Longint;
    property ParentHandle: THandle
      read FParentHandle
      write FParentHandle;
    property Certificate: String
      read FCertificate;
    property CertPropsRec: TX509CertPropsRec
      read FCertPropsRec;
    property KeyPropsRec: TKeyPropsRec
      read FKeyPropsRec;
    property RequestData: String
      read FRequestData
      write FRequestData;
  published
    property Armor: Longbool
      read FArmor
      write FArmor;
    property CertAttributes: TCertAttributes
      read FCertAttributes
      write FCertAttributes;
    property KeyAndPassPrompt: String
      read FKeyAndPassPrompt
      write FKeyAndPassPrompt;
    property KeyHexIDorIASN: String
      read FKeyHexIDorIASN
      write FKeyHexIDorIASN;
    property KeyProps: TKeyProps
      read FKeyProps
      write FKeyProps;
    property ValidDays: Longint
      read FValidDays
      write FValidDays;
    property OnFailure: TOnFailure
      read FOnFailure
      write FOnFailure;
    property OnEnterPassphrase: TOnEnterPassphrase
      read FOnEnterPassphrase
      write FOnEnterPassphrase;
  end;


// TX509GetCertProps -----------------------------------------------------------

type
  TCertificateFilter = CertFilter_X509Cert..CertFilter_X509Root;

type
  TRingCertProps = class(TComponent)
  private
    FCertIDs: TStringList;
    FComboBox: TComboBox;
    FListBox: TListBox;
    FRingAlgorithmFilter: TAlgorithmKeyFilter;
    FRingBoolFilter: TBooleanKeyFilter;
    FRingCertFilter: TCertificateFilter;
    FCertPropsList: TX509CertPropsList;
    FOnFailure: TOnFailure;
    function GetCertIDs: Longint;
  protected
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function Update: Longint;
    property CertPropsList: TX509CertPropsList
      read FCertPropsList;
  published
    property RingAlgorithmFilter: TAlgorithmKeyFilter
      read FRingAlgorithmFilter
      write FRingAlgorithmFilter;
    property RingBoolFilter: TBooleanKeyFilter
      read FRingBoolFilter
      write FRingBoolFilter;
    property RingCertFilter: TCertificateFilter
      read FRingCertFilter
      write FRingCertFilter;
    property ComboBox: TComboBox
      read FComboBox
      write FComboBox;
    property ListBox: TListBox
      read FListBox
      write FListBox;
    property OnFailure: TOnFailure
      read FOnFailure
      write FOnFailure;
  end;

type
  TOnGetCertProps = procedure(const CertPropsList: TX509CertPropsList) of Object;
  TX509GetCertProps = class(TRingCertProps)
  private
    FAlgorithmFilter: TAlgorithmKeyFilter;
    FBoolFilter: TBooleanKeyFilter;
    FCertFilter: TCertificateFilter;
    FCertData: String;
    FCertKeyHexID: String;
    FCertKeyUserID: String;
    FIncludeSigner: Longbool;
    FSignKeyHexID: String;
    FOnFailure: TOnFailure;
    FOnGetCertProps: TOnGetCertProps;
  protected
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function VerifyCertChain(FileInput: Longbool): Longint;
    function GetCertProps(FileInput: Longbool): Longint;
    function GetRingCertProps: Longint;
    function GetCertProp(CertPropsList: TX509CertPropsList; CertIndex: Longint; CertProp: TX509CertProp): Variant;
    property CertData: String
      read FCertData
      write FCertData;
  published
    property AlgorithmFilter: TAlgorithmKeyFilter
      read FAlgorithmFilter
      write FAlgorithmFilter;
    property BoolFilter: TBooleanKeyFilter
      read FBoolFilter
      write FBoolFilter;
    property CertFilter: TCertificateFilter
      read FCertFilter
      write FCertFilter;
    property CertKeyHexID: String
      read FCertKeyHexID
      write FCertKeyHexID;
    property CertKeyUserID: String
      read FCertKeyUserID
      write FCertKeyUserID;
    property IncludeSigner: Longbool
      read FIncludeSigner
      write FIncludeSigner;
    property SignKeyHexID: String
      read FSignKeyHexID
      write FSignKeyHexID;
    property OnFailure: TOnFailure
      read FOnFailure
      write FOnFailure;
    property OnGetCertProps: TOnGetCertProps
      read FOnGetCertProps
      write FOnGetCertProps;
end;


// TX509SetCertProps -----------------------------------------------------------

type
  TOnCertPropsChanged = procedure(const CertPropsList: TX509CertPropsList) of Object;
  TX509SetCertProps = class(TComponent)
  private
    FParentHandle: THandle;
    FCertKeyPrompt: String;
    FSignCertIASN: String;
    FSignKeyAndPassPrompt: String;
    FTargetCertIASN: String;
    FOnFailure: TOnFailure;
    FOnCertPropsChanged: TOnCertPropsChanged;
    FOnEnterRevokerPassphrase: TOnEnterPassphrase;
  protected
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function RevokeCertificate(ValidDays: Longint): Longint;
    function RemoveCertificate(IncludeUserID: Longbool): Longint;
    property ParentHandle: THandle
      read FParentHandle
      write FParentHandle;
  published
    property CertKeyPrompt: String
      read FCertKeyPrompt
      write FCertKeyPrompt;
    property SignCertIASN: String
      read FSignCertIASN
      write FSignCertIASN;
    property SignKeyAndPassPrompt: String
      read FSignKeyAndPassPrompt
      write FSignKeyAndPassPrompt;
    property TargetCertIASN: String
      read FTargetCertIASN
      write FTargetCertIASN;
    property OnFailure: TOnFailure
      read FOnFailure
      write FOnFailure;
    property OnCertPropsChanged: TOnCertPropsChanged
      read FOnCertPropsChanged
      write FOnCertPropsChanged;
    property OnEnterRevokerPassphrase: TOnEnterPassphrase
      read FOnEnterRevokerPassphrase
      write FOnEnterRevokerPassphrase;
end;


// TX509CertGenerate -----------------------------------------------------------

type
  TX509CertGenerate = class(TComponent)
  private
    FParentHandle: THandle;
    FCertAttributes: TCertAttributes;
    FCertKeyHexID: String;
    FCertKeyPrompt: String;
    FCertPropsList: TX509CertPropsList;
    FSignCertIASN: String;
    FSignKeyAndPassPrompt: String;
    FValidDays: Longint;
    FOnFailure: TOnFailure;
    FOnEnterIssuerPassphrase: TOnEnterPassphrase;
  protected
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function CreateCertificate(SelfSigned: Longbool): Longint;
    property ParentHandle: THandle
      read FParentHandle
      write FParentHandle;
    property CertPropsList: TX509CertPropsList
      read FCertPropsList;
  published
    property CertAttributes: TCertAttributes
      read FCertAttributes
      write FCertAttributes;
    property CertKeyHexID: String
      read FCertKeyHexID
      write FCertKeyHexID;
    property CertKeyPrompt: String
      read FCertKeyPrompt
      write FCertKeyPrompt;
    property SignCertIASN: String
      read FSignCertIASN
      write FSignCertIASN;
    property SignKeyAndPassPrompt: String
      read FSignKeyAndPassPrompt
      write FSignKeyAndPassPrompt;
    property ValidDays: Longint
      read FValidDays
      write FValidDays;
    property OnFailure: TOnFailure
      read FOnFailure
      write FOnFailure;
    property OnEnterIssuerPassphrase: TOnEnterPassphrase
      read FOnEnterIssuerPassphrase
      write FOnEnterIssuerPassphrase;
end;


// TX509CertImport -------------------------------------------------------------

type
  TOnCertImported = procedure(const CertPropsList: TX509CertPropsList; CertsImported: Longint) of Object;
  TX509CertImport = class(TComponent)
  private
    FParentHandle: THandle;
    FCertData: String;
    FCertKeyPrompt: String;
    FCertPassPrompt: String;
    FOnFailure: TOnFailure;
    FOnCertImported: TOnCertImported;
    FOnEnterPKCS12Passphrase: TOnEnterPassphrase;
  protected
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function ImportCertificate(FileInput: Longbool): Longint;
    property ParentHandle: THandle
      read FParentHandle
      write FParentHandle;
    property CertData: String
      read FCertData
      write FCertData;
  published
    property CertKeyPrompt: String
      read FCertKeyPrompt
      write FCertKeyPrompt;
    property CertPassPrompt: String
      read FCertPassPrompt
      write FCertPassPrompt;
    property OnFailure: TOnFailure
      read FOnFailure
      write FOnFailure;
    property OnCertImported: TOnCertImported
      read FOnCertImported
      write FOnCertImported;
    property OnEnterPKCS12Passphrase: TOnEnterPassphrase
      read FOnEnterPKCS12Passphrase
      write FOnEnterPKCS12Passphrase;
end;


// TX509CertExport -------------------------------------------------------------

type
  TOnCertExported = procedure(const CertPropsList: TX509CertPropsList; CertChain, ToFile: Longbool) of Object;
  TX509CertExport = class(TComponent)
  private
    FParentHandle: THandle;
    FArmorCert: Longbool;
    FCertChain: Longbool;
    FCertData: String;
    FCertKeyOrKeyAndPassPrompt: String;
    FExportPrivate: Longbool;
    FKeyHexIDorIASN: String;
    FOnFailure: TOnFailure;
    FOnCertExported: TOnCertExported;
    FOnEnterSecKeyPassphrase: TOnEnterPassphrase;
    procedure SetArmorCert(Value: Longbool);
    procedure SetCertChain(Value: Longbool);
    procedure SetExportPrivate(Value: Longbool);
  protected
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function ExportCertificate(FileOutput: Longbool): Longint;
    property ParentHandle: THandle
      read FParentHandle
      write FParentHandle;
    property CertData: String
      read FCertData
      write FCertData;
  published
    property ArmorCert: Longbool
      read FArmorCert
      write SetArmorCert;
    property CertChain: Longbool
      read FCertChain
      write SetCertChain;
    property CertKeyOrKeyAndPassPrompt: String
      read FCertKeyOrKeyAndPassPrompt
      write FCertKeyOrKeyAndPassPrompt;
    property ExportPrivate: Longbool
      read FExportPrivate
      write SetExportPrivate;
    property KeyHexIDorIASN: String
      read FKeyHexIDorIASN
      write FKeyHexIDorIASN;
    property OnFailure: TOnFailure
      read FOnFailure
      write FOnFailure;
    property OnCertExported: TOnCertExported
      read FOnCertExported
      write FOnCertExported;
    property OnEnterSecKeyPassphrase: TOnEnterPassphrase
      read FOnEnterSecKeyPassphrase
      write FOnEnterSecKeyPassphrase;
  end;


implementation


// String constants

const
  UnknownCertReqErr = 'Unknown CertRequest error';
  UnknownCertPropsErr = 'Unknown CertProps error';
  UnknownCertRemoveErr = 'Unknown CertRemove error';
  UnknownCertRevokeErr = 'Unknown CertRevoke error';
  UnknownCertGenErr = 'Unknown CertGenerate error';
  UnknownCertImportErr = 'Unknown CertImport error';
  UnknownCertExportErr = 'Unknown CertExport error';


// Passphrase event handler ----------------------------------------------------

function GetPassphrase(var Passphrase: PChar; const PassphraseProc: TOnEnterPassphrase): Longint;
var
  Cancel: Longbool;
begin
  Passphrase := PGPNewSecureData(PGPGetDefaultMemoryMgr, 256 * UTF8Factor, kPGPMemoryMgrFlags_Clear);
  if Passphrase <> nil then begin
    Cancel := false;
    Result := kPGPError_UserAbort;
    PassphraseProc(Passphrase, Cancel);
    if not Cancel then begin
      if PGP8X then SecureAnsiToUtf8PChar(Passphrase, Passphrase, 256 * UTF8Factor);
      Result := kPGPError_NoErr;
    end;
  end
  else Result := kPGPError_OutOfMemory;
end;


// TCertAttributes -------------------------------------------------------------

function TCertAttributes.GetSerialNumber: Longint;
begin
  Result := StrToIntDef(Values.SerialNumber, 0);
end;

procedure TCertAttributes.SetSerialNumber(Value: Longint);
begin
  Values.SerialNumber := IntToStr(Value);
end;


// TX509RequestHandler ---------------------------------------------------------

constructor TX509RequestHandler.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FCertAttributes := TCertAttributes.Create;
  if AOwner is TForm then FParentHandle := TForm(AOwner).Handle;
end;

destructor TX509RequestHandler.Destroy;
begin
  FCertAttributes.Free;
  inherited Destroy;
end;

function TX509RequestHandler.CreateCertRequest: Longint;
var
  CertKeyPassphrase: PChar;
begin
  Result := kPGPError_BadParams;
  try
    if (FKeyAndPassPrompt <> EMPTY) or Assigned(FOnEnterPassphrase) then begin
      repeat
	CertKeyPassphrase := nil;
	if (FKeyAndPassPrompt = EMPTY) and Assigned(FOnEnterPassphrase) then begin
	  Result := GetPassphrase(CertKeyPassphrase, FOnEnterPassphrase);
	end;
	try
	  if (FKeyAndPassPrompt <> EMPTY) or (Result = 0) then begin
	    Result := CreateX509CertificateRequest(FKeyAndPassPrompt, FKeyHexIDorIASN, CertKeyPassphrase,
						   FCertAttributes.Values, FArmor, FRequestData,
						   FKeyPropsRec, GetKeyPropsFlags(FKeyProps), FParentHandle);
	  end;
	finally
	  PGPFreeData(CertKeyPassphrase);
	end;
      until (Result <> kPGPError_BadPassphrase) or not Assigned(FOnEnterPassphrase);
    end;
    if Result <> 0 then ShowError(FOnFailure, Result, EMPTY);
  except
    on EFailException do Raise;
    else Result := ShowError(FOnFailure, -1, UnknownCertReqErr);
  end;
end;

function TX509RequestHandler.CreateCertFromRequest: Longint;
var
  SignKeyPassphrase: PChar;
  CertPropsList: TX509CertPropsList;
begin
  Result := kPGPError_BadParams;
  try
    if (FKeyAndPassPrompt <> EMPTY) or Assigned(FOnEnterPassphrase) then begin
      repeat
	SignKeyPassphrase := nil;
	if (FKeyAndPassPrompt = EMPTY) and Assigned(FOnEnterPassphrase) then begin
	  Result := GetPassphrase(SignKeyPassphrase, FOnEnterPassphrase);
	end;
	try
	  if (FKeyAndPassPrompt <> EMPTY) or (Result = 0) then begin
	    CertPropsList := nil;
	    try
	      Result := CreateX509CertificateFromRequest(FKeyAndPassPrompt, FKeyHexIDorIASN, SignKeyPassphrase,
							 FCertAttributes.Values, FValidDays, FArmor, FRequestData,
							 FCertificate, CertPropsList, FParentHandle);
	      if (Result = 0) and (CertPropsList.Count > 0) then begin
		FCertPropsRec := PX509CertPropsRec(CertPropsList.Objects[0])^;
	      end;
	    finally
	      CertPropsList.Free;
	    end;
	  end;
	finally
	  PGPFreeData(SignKeyPassphrase);
	end;
      until (Result <> kPGPError_BadPassphrase) or not Assigned(FOnEnterPassphrase);
    end;
    if Result <> 0 then ShowError(FOnFailure, Result, EMPTY);
  except
    on EFailException do Raise;
    else Result := ShowError(FOnFailure, -1, UnknownCertReqErr);
  end;
end;


// TX509GetCertProps -----------------------------------------------------------

constructor TRingCertProps.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FCertIDs := TStringList.Create;
  FCertPropsList := TX509CertPropsList.Create;
end;

destructor TRingCertProps.Destroy;
begin
  FCertPropsList.Free;
  FCertIDs.Free;
  inherited Destroy;
end;

function TRingCertProps.GetCertIDs: Longint;
var
  CertIndex	: Longint;
begin
  Result := 0;
  if (FCertIDs <> nil) and (FCertPropsList <> nil) and (FCertPropsList.Count <> 0) then begin
    FCertIDs.Clear;
    FCertIDs.Sorted:=true;
    with FCertPropsList do begin
      for CertIndex := 0 to pred(Count) do with pX509CertPropsRec(Objects[CertIndex])^ do begin
	Move(CertIndex, FCertIDs.Add(x509OwnerLongName));
	inc(Result);
      end;
    end;
    if Assigned(FComboBox) then FComboBox.Items.Assign(FCertIDs);
    if Assigned(FListBox) then FListBox.Items.Assign(FCertIDs);
  end;
end;

function TRingCertProps.Update: Longint;
begin
  try
    FCertPropsList.Clear;
    Result := FindX509CertProps(ALL, ALL, EMPTY,
    				GetKeyFilter(FRingAlgorithmFilter, FRingBoolFilter, TCertKeyFilter(FRingCertFilter)),
				false, FCertPropsList);
    if Result >= 0 then
      Result := GetCertIDs
    else begin
      if Assigned(FComboBox) then FComboBox.Items.Clear;
      if Assigned(FListBox) then FListBox.Items.Clear;
      ShowError(FOnFailure, Result, EMPTY);
    end;
  except
    on EFailException do Raise;
    else Result := ShowError(FOnFailure, -1, UnknownCertPropsErr);
    if Assigned(FComboBox) then FComboBox.Items.Clear;
    if Assigned(FListBox) then FListBox.Items.Clear;
  end;
end;

constructor TX509GetCertProps.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
end;

destructor TX509GetCertProps.Destroy;
begin
  inherited Destroy;
end;

function TX509GetCertProps.VerifyCertChain(FileInput: Longbool): Longint;
begin
  try
    Result := VerifyX509CertificateChain(FCertData, FileInput);
    if Result <> 0 then ShowError(FOnFailure, Result, EMPTY);
  except
    on EFailException do Raise;
    else Result := ShowError(FOnFailure, -1, UnknownCertPropsErr);
  end;
end;

function TX509GetCertProps.GetCertProps(FileInput: Longbool): Longint;
var
  CertPropsList: TX509CertPropsList;
begin
  try
    CertPropsList := nil;
    try
      Result := GetX509CertificateProperties(FCertData, FileInput, CertPropsList);
      if Result >= 0 then begin
	if Assigned(FOnGetCertProps) then FOnGetCertProps(CertPropsList);
      end
      else ShowError(FOnFailure, Result, EMPTY);
    finally
      CertPropsList.Free;
    end;
  except
    on EFailException do Raise;
    else Result := ShowError(FOnFailure, -1, UnknownCertPropsErr);
  end;
end;

function TX509GetCertProps.GetRingCertProps: Longint;
var
  CertPropsList: TX509CertPropsList;
begin
  try
    CertPropsList := nil;
    try
      Result := FindX509CertProps(FCertKeyUserID, FCertKeyHexID, FSignKeyHexID,
				  GetKeyFilter(FAlgorithmFilter, FBoolFilter, TCertKeyFilter(FCertFilter)),
				  FIncludeSigner, CertPropsList);
      if Result >= 0 then begin
	if Assigned(FOnGetCertProps) then FOnGetCertProps(CertPropsList);
      end
      else ShowError(FOnFailure, Result, EMPTY);
    finally
      CertPropsList.Free;
    end;
  except
    on EFailException do Raise;
    else Result := ShowError(FOnFailure, -1, UnknownCertPropsErr);
  end;
end;

function TX509GetCertProps.GetCertProp(CertPropsList: TX509CertPropsList; CertIndex: Longint;
				       CertProp: TX509CertProp): Variant;
begin
  Result := kPGPError_NoErr;
  if (CertPropsList <> nil) and (CertIndex >= 0) and (CertIndex < CertPropsList.Count) then begin
    with PX509CertPropsRec(CertPropsList.Objects[CertIndex])^ do begin
      case CertProp of
	CertProp_CertSerialNumber: Result := x509CertificateID;
	CertProp_IssuerKeyAlgorithm: Result := x509IssuerKeyAlgorithm;
	CertProp_IssuerHashAlgorithm: Result := x509IssuerHashAlgorithm;
	CertProp_IssuerLongName: Result := x509IssuerLongName;
	CertProp_IssuerSerialNumber: Result := x509IssuerSerialNumber;
	CertProp_OwnerHexID: Result := x509OwnerKeyHexID;
	CertProp_OwnerLongName: Result := x509OwnerLongName;
	CertProp_OwnerSerialNumber: Result := x509OwnerSerialNumber;
	CertProp_CreaTimeStr: Result := x509CreaTimeStr;
	CertProp_ExpTimeStr: Result := x509ExpTimeStr;
	CertProp_CreaTimeNum: Result := x509CreaTimeNum;
	CertProp_ExpTimeNum: Result := x509ExpTimeNum;
	CertProp_Verified: Result := x509IsVerified;
	CertProp_Corrupt: Result := x509IsCorrupt;
	CertProp_Revoked: Result := x509IsRevoked;
	CertProp_Expired: Result := x509IsExpired;
	CertProp_Root: Result := x509IsRoot;
      else
	ShowError(FOnFailure, kPGPError_BadParams, EMPTY);
      end;
    end;
  end
  else ShowError(FOnFailure, kPGPError_BadParams, EMPTY);
end;


// TX509SetCertProps -----------------------------------------------------------

constructor TX509SetCertProps.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  if AOwner is TForm then FParentHandle := TForm(AOwner).Handle;
end;

destructor TX509SetCertProps.Destroy;
begin
  inherited Destroy;
end;

function TX509SetCertProps.RevokeCertificate(ValidDays: Longint): Longint;
var
  SignKeyPassphrase: PChar;
  CertPropsList: TX509CertPropsList;
begin
  Result := kPGPError_BadParams;
  try
    if (FSignKeyAndPassPrompt <> EMPTY) or Assigned(FOnEnterRevokerPassphrase) then begin
      repeat
	SignKeyPassphrase := nil;
	if (FSignKeyAndPassPrompt = EMPTY) and Assigned(FOnEnterRevokerPassphrase) then begin
	  Result := GetPassphrase(SignKeyPassphrase, FOnEnterRevokerPassphrase);
	end;
	try
	  if (FSignKeyAndPassPrompt <> EMPTY) or (Result = 0) then begin
	    CertPropsList := nil;
	    try
	      Result := CreateX509CertificateRevocationList(FCertKeyPrompt, FSignKeyAndPassPrompt,
							    FTargetCertIASN, FSignCertIASN,
							    SignKeyPassphrase, ValidDays,
							    CertPropsList, FParentHandle);
	      if (Result = 0) and (CertPropsList.Count > 0) and Assigned(FOnCertPropsChanged) then begin
		FOnCertPropsChanged(CertPropsList);
	      end;
	    finally
	      CertPropsList.Free;
	    end;
	  end;
	finally
	  PGPFreeData(SignKeyPassphrase);
	end;
      until (Result <> kPGPError_BadPassphrase) or not Assigned(FOnEnterRevokerPassphrase);
    end;
    if Result <> 0 then ShowError(FOnFailure, Result, EMPTY);
  except
    on EFailException do Raise;
    else Result := ShowError(FOnFailure, -1, UnknownCertRevokeErr);
  end;
end;

function TX509SetCertProps.RemoveCertificate(IncludeUserID: Longbool): Longint;
var
  CertPropsList: TX509CertPropsList;
begin
  try
    CertPropsList := TX509CertPropsList.Create;
    try
      Result := RemoveX509Certificate(FCertKeyPrompt, FTargetCertIASN, IncludeUserID, CertPropsList, ParentHandle);
      if (Result = 0) and (CertPropsList.Count > 0) and Assigned(FOnCertPropsChanged) then begin
	FOnCertPropsChanged(CertPropsList);
      end;
    finally
      CertPropsList.Free;
    end;
    if Result <> 0 then ShowError(FOnFailure, Result, EMPTY);
  except
    on EFailException do Raise;
    else Result := ShowError(FOnFailure, -1, UnknownCertRemoveErr);
  end;
end;


// TX509CertGenerate -----------------------------------------------------------

constructor TX509CertGenerate.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FCertAttributes := TCertAttributes.Create;
  FCertPropsList := TX509CertPropsList.Create;
  if AOwner is TForm then FParentHandle := TForm(AOwner).Handle;
end;

destructor TX509CertGenerate.Destroy;
begin
  FCertAttributes.Free;
  FCertPropsList.Free;
  inherited Destroy;
end;

function TX509CertGenerate.CreateCertificate(SelfSigned: Longbool): Longint;
var
  SignKeyPassphrase: PChar;
begin
  Result := kPGPError_BadParams;
  try
    if (FSignKeyAndPassPrompt <> EMPTY) or Assigned(FOnEnterIssuerPassphrase) then begin
      repeat
	SignKeyPassphrase := nil;
	if (FSignKeyAndPassPrompt = EMPTY) and Assigned(FOnEnterIssuerPassphrase) then begin
	  Result := GetPassphrase(SignKeyPassphrase, FOnEnterIssuerPassphrase);
	end;
	try
	  if (FSignKeyAndPassPrompt <> EMPTY) or (Result = 0) then begin
	    FCertPropsList.Clear;
	    Result := CreateX509Certificate(FCertKeyPrompt, FSignKeyAndPassPrompt, FCertKeyHexID, FSignCertIASN,
					    SignKeyPassphrase, FCertAttributes.Values, FValidDays,
					    SelfSigned, FCertPropsList, FParentHandle);
	  end;
	finally
	  PGPFreeData(SignKeyPassphrase);
	end;
      until (Result <> kPGPError_BadPassphrase) or not Assigned(FOnEnterIssuerPassphrase);
    end;
    if Result <> 0 then ShowError(FOnFailure, Result, EMPTY);
  except
    on EFailException do Raise;
    else Result := ShowError(FOnFailure, -1, UnknownCertGenErr);
  end;
end;


// TX509CertImport -------------------------------------------------------------

constructor TX509CertImport.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  if AOwner is TForm then FParentHandle := TForm(AOwner).Handle;
end;

destructor TX509CertImport.Destroy;
begin
  inherited Destroy;
end;

function TX509CertImport.ImportCertificate(FileInput: Longbool): Longint;
var
  CertPropsList: TX509CertPropsList;
  ImportCertPassphrase: PChar;
begin
  Result := kPGPError_NoErr;
  try
    repeat
      ImportCertPassphrase := nil;
      if (FCertPassPrompt = EMPTY) and Assigned(FOnEnterPKCS12Passphrase) then begin
	Result := GetPassphrase(ImportCertPassphrase, FOnEnterPKCS12Passphrase);
      end;
      try
	if Result = 0 then begin
	  CertPropsList := nil;
	  try
	    Result := ImportX509Certificate(FCertKeyPrompt, FCertPassPrompt, FCertData, ImportCertPassphrase,
					    FileInput, CertPropsList, FParentHandle);
	    if (Result = 0) or ((Result = kPGPError_FilePermissions) and (CertPropsList.Count > 0)) then begin
	      if Assigned(FOnCertImported) then FOnCertImported(CertPropsList, CertPropsList.Count);
	    end;
	  finally
	    CertPropsList.Free;
	  end;
	end;
      finally
	PGPFreeData(ImportCertPassphrase);
      end;
    until (Result <> kPGPError_BadPassphrase) or not Assigned(FOnEnterPKCS12Passphrase);
    if Result <> 0 then ShowError(FOnFailure, Result, EMPTY);
  except
    on EFailException do Raise;
    else Result := ShowError(FOnFailure, -1, UnknownCertImportErr);
  end;
end;


// TX509CertExport -------------------------------------------------------------

constructor TX509CertExport.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  if AOwner is TForm then FParentHandle := TForm(AOwner).Handle;
end;

destructor TX509CertExport.Destroy;
begin
  inherited Destroy;
end;

procedure TX509CertExport.SetArmorCert(Value: Longbool);
begin
  FArmorCert := Value and not FExportPrivate;
end;

procedure TX509CertExport.SetCertChain(Value: Longbool);
begin
  FCertChain := Value;
  if FCertChain then FExportPrivate := false;
end;

procedure TX509CertExport.SetExportPrivate(Value: Longbool);
begin
  FExportPrivate := Value;
  if FExportPrivate then begin
    FArmorCert := false;
    FCertChain := false;
  end;
end;

function TX509CertExport.ExportCertificate(FileOutput: Longbool): Longint;
var
  ExportCertPassphrase: PChar;
  CertPropsList: TX509CertPropsList;
begin
  Result := kPGPError_BadParams;
  try
    if (FCertKeyOrKeyAndPassPrompt <> EMPTY)
    or ((FKeyHexIDorIASN <> EMPTY) and (FCertChain or (not FExportPrivate) or Assigned(FOnEnterSecKeyPassphrase))) then begin
      if not FileOutput or (FCertData <> EMPTY) then begin
	repeat
	  ExportCertPassphrase := nil;
	  if not FExportPrivate then
	    Result := kPGPError_NoErr
	  else if (FCertKeyOrKeyAndPassPrompt = EMPTY) and Assigned(FOnEnterSecKeyPassphrase) then begin
	    Result := GetPassphrase(ExportCertPassphrase, FOnEnterSecKeyPassphrase);
	  end;
	  try
	    if (FCertKeyOrKeyAndPassPrompt <> EMPTY) or (Result = 0) then begin
	      CertPropsList := nil;
	      try
		Result := ExportX509Certificate(FCertKeyOrKeyAndPassPrompt, FKeyHexIDorIASN, ExportCertPassphrase,
						FArmorCert, FCertChain, FExportPrivate, FileOutput,
						FCertData, CertPropsList, FParentHandle);
		if (Result = 0) and Assigned(FOnCertExported) then FOnCertExported(CertPropsList, FCertChain, FileOutput);
	      finally
		CertPropsList.Free;
	      end;
	    end;
	  finally
	    PGPFreeData(ExportCertPassphrase);
	  end;
	until (Result <> kPGPError_BadPassphrase) or not Assigned(FOnEnterSecKeyPassphrase);
      end;
    end;
    if Result <> 0 then ShowError(FOnFailure, Result, EMPTY);
  except
    on EFailException do Raise;
    else Result := ShowError(FOnFailure, -1, UnknownCertExportErr);
  end;
end;


// Register --------------------------------------------------------------------

procedure Register;
begin
  RegisterComponents('X509', [TX509RequestHandler, TX509GetCertProps, TX509SetCertProps,
			      TX509CertGenerate, TX509CertImport, TX509CertExport]);
end;


end.

