{$J+,Z4}
unit UTF8;
{$IFDEF CONDITIONALEXPRESSIONS} {$WARN SYMBOL_PLATFORM OFF} {$ENDIF}

{------------------------------------------------------------------------------}
{                                                                              }
{          UTF8 utilities - especially for versions prior to Delphi 6          }
{                                                                              }
{                                 This code is                                 }
{              Copyright (C) 2003-2006 by Michael in der Wiesche               }
{                                                                              }
{------------------------------------------------------------------------------}

// These functions are only required when using PGP 8.x or later
// UTF8Sec.dll provides them for using Delphi versions prior to 6.x
// Secure memory management prevents passphrases from being swapped to disk

interface

type UTF8String = type String;

function SecureAnsiToUtf8PChar(const Ansi, Utf8: PChar; Len: Cardinal): Longint;
function SecureUtf8ToAnsiPChar(const Utf8, Ansi: PChar; Len: Cardinal): Longint;

{$IFNDEF CONDITIONALEXPRESSIONS}
function AnsiToUtf8(const S: String): UTF8String;
function Utf8ToAnsi(const S: UTF8String): String;
{$ENDIF}

function Utf8OrAnsi(const S: UTF8String): String;

const UTF8Factor = 4;

implementation

uses Windows, SysUtils, pgpMemoryMgr, pgpBase;

const // from Winnls.h
  CP_UTF7		= 65000;	// UTF-7 translation
  CP_UTF8		= 65001;	// UTF-8 translation

  MB_PRECOMPOSED	= $00000001;	// use precomposed chars
  MB_COMPOSITE		= $00000002;	// use composite chars
  MB_USEGLYPHCHARS	= $00000004;	// use glyph chars, not ctrl chars
  MB_ERR_INVALID_CHARS	= $00000008;	// error for invalid chars

  WC_COMPOSITECHECK	= $00000200;	// convert composite to precomposed
  WC_DISCARDNS		= $00000010;	// discard non-spacing chars
  WC_SEPCHARS		= $00000020;	// generate separate chars
  WC_DEFAULTCHAR	= $00000040;	// replace w/ default char

function SecureAnsiToUtf8PChar(const Ansi, Utf8: PChar; Len: Cardinal): Longint;
var
  Wide: PWideChar;
  WideLen: Cardinal;
begin
  Result := 0;
  if (Ansi <> nil) and (Utf8 <> nil) and (Len > 0) then begin
    WideLen := succ(Len div Utf8Factor);
    Wide := PGPNewSecureData(PGPGetDefaultMemoryMgr, WideLen shl 1, kPGPMemoryMgrFlags_Clear);
    try
      WideLen := MultiByteToWideChar(CP_ACP, MB_PRECOMPOSED or MB_ERR_INVALID_CHARS, Ansi, -1, Wide, WideLen);
      if WideLen > 0 then Result := WideCharToMultiByte(CP_UTF8, 0, Wide, -1, Utf8, Len, nil, nil);
    finally
      PGPFreeData(Wide);
    end;
  end;
end;

function SecureUtf8ToAnsiPChar(const Utf8, Ansi: PChar; Len: Cardinal): Longint;
var
  Wide: PWideChar;
  WideLen: Cardinal;
begin
  Result := 0;
  if (Utf8 <> nil) and (Ansi <> nil) and (Len > 0) then begin
    WideLen := succ(Len);
    Wide := PGPNewSecureData(PGPGetDefaultMemoryMgr, WideLen shl 1, kPGPMemoryMgrFlags_Clear);
    try
      WideLen := MultiByteToWideChar(CP_UTF8, 0, Utf8, -1, Wide, WideLen);
      if WideLen > 0 then Result := WideCharToMultiByte(CP_ACP, 0, Wide, -1, Ansi, Len, nil, nil);
    finally
      PGPFreeData(Wide);
    end;
  end;
end;

{$IFNDEF CONDITIONALEXPRESSIONS}

var AnsiToUtf8PChar: function(const Ansi, Utf8: PChar; Len: Cardinal): Longint; stdcall;
var Utf8ToAnsiPChar: function(const Utf8, Ansi: PChar; Len: Cardinal): Longint; stdcall;

function AnsiToUtf8(const S: String): UTF8String;
var
  Len: Cardinal;
begin
  Len := succ(Length(S) shl 2);
  SetLength(Result, Len);
  Len := AnsiToUtf8PChar(PChar(S), PChar(Result), Len);
  Delete(Result, succ(Len), MAXINT);
end;

function Utf8ToAnsi(const S: UTF8String): String;
var
  Len: Cardinal;
begin
  Len := succ(Length(S));
  SetLength(Result, Len);
  Len := Utf8ToAnsiPChar(PChar(S), PChar(Result), Len);
  Delete(Result, succ(Len), MAXINT);
end;

{$ENDIF}

function Utf8OrAnsi(const S: UTF8String): String;
begin
  Result := Utf8ToAnsi(S);
  if Result = '' then Result := S;
end;

initialization

{$IFNDEF CONDITIONALEXPRESSIONS}
  if hUTF8Lib <> 0 then begin
    AnsiToUtf8PChar := GetProcAddress(hUTF8Lib, ptr(1));
    Utf8ToAnsiPChar := GetProcAddress(hUTF8Lib, ptr(2));
  end;
{$ENDIF}

finalization

end.

