{$J+,Z4}
unit PrefFuncs;

{------------------------------------------------------------------------------}
{                                                                              }
{       This unit is based on Steve Heller's spgpPreferences.pas from his      }
{         SPGP sources available from http://www.oz.net/~srheller/spgp/        }
{                                                                              }
{                Portions created by Michael in der Wiesche are                }
{              Copyright (C) 2001-2006 by Michael in der Wiesche               }
{                                                                              }
{------------------------------------------------------------------------------}

interface

uses
  Windows,
  Classes,
  SysUtils,
  StrTools,
  KeyPropTypes,
  UTF8,
  pgpBase,
  pgpErrors,
  pgpPubTypes,
  pgpUtilities,
  pgpOptionList,
  pgpMemoryMgr,
  pgpKeyServer,
  pgpSDKPrefs,
  pgpKeys,
  pgpCL,
  KeyFuncs;

const
  AppDataRegPath = 'Software\Microsoft\Windows\CurrentVersion\Explorer\Shell Folders';
  PrefsFileXml = 'PGPprefs.xml';
  PrefsFileTxt = 'PGPprefs.txt';
  GroupsFileName = 'PGPgroup.pgr';

  tagKey = '<key>';
  tagCKey = '</key>';
  tagData = '<data>';
  tagCData = '</data>';
  tagString = '<string>';
  tagCString = '</string>';
  tagInteger = '<integer>';
  tagCInteger = '</integer>';
  tagFalse = '<false>';
  tagCFalse = '</false>';
  tagTrue = '<true>';
  tagCTrue = '</true>';
  tagDict = '<dict>';
  tagCDict = '</dict>';
  tagArray = '<array>';
  tagCArray = '</array>';

  PubFile = 'PublicKeyringFile';
  SecFile = 'PrivateKeyringFile';
  RNGFile = 'RNGSeedFile';                      	// blank since 8.X
  DefKeyID7 = 'DefaultKeyID';
  DefKeyID9 = 'LastSigningKeyID';
  KeyServers9 = 'KeyServers';
  KeyServers7 = 'KeyServerList';
  WipeWarning9 = 'WipeWarning';
  WipeWarning7 = 'WarnOnWipe';
  FileWipeCount = 'FileWipePasses';
  PrefAlgorithm = 'PreferredAlgorithm';
  AllowedAlgList = 'AllowedAlgorithmsList';
  CacheType9 = 'PassphraseCacheType';
  CacheType7 = 'CacheType';
  CacheSeconds9 = 'PassphraseCacheSeconds';
  CacheSeconds7 = 'CacheSeconds';
  ShareCache9 = 'PassphraseShareCache';
  ShareCache7 = 'ShareCache';

  PrefsFlag_Default		= $00;
  PrefsFlag_PublicKeyring	= $01;
  PrefsFlag_PrivateKeyring	= $02;
  PrefsFlag_RandomSeedFile	= $04;
  PrefsFlag_DefaultKeyID	= $08;
  PrefsFlag_GroupsFile		= $10;

type
  TPassCacheSeconds = -1..MaxLongint;

type
  pPreferenceRec = ^TPreferenceRec;
  TPreferenceRec = Record
    PublicKeyring: String;
    PrivateKeyring: String;
    RandomSeedFile: String;
    GroupsFile: String;
    DefaultKeyHexID: String;
  end;

type
  TPrefsList = class(TStringList)
  private
    function GetStringPos(I: Integer): Integer;
    procedure SetStringPos(I, Value: Integer);
  public
    function XmlIndexOf(const S: String): Integer;
    property StringPos[I: Integer]: Integer
      read GetStringPos
      write SetStringPos;
  end;

type
  TPrefsFile = Class
    PrefsData: TPrefsList;
    PrefsFilePath: String;
    GroupsFilePath: String;
    CacheAge: Integer;
    constructor Create;
    destructor Destroy; override;
    function LoadPrefs: Longbool;
    function SavePrefs: Longbool;
    function GetPrefDict(var StartIndex: Integer): String;
    function GetPrefArray(var StartIndex: Integer): String;
    function GetPrefValue(const PrefName: String; var PrefIndex: Integer): String;
    function SetPrefDict(const PrefDict: String; var StartIndex: Integer): Longbool;
    function SetPrefArray(const PrefArray: String; var StartIndex: Integer): Longbool;
    function SetPrefValue(const PrefName, PrefValue: String; var PrefIndex: Integer): Longbool;
    function GetPGPKeyID(const PGPKeyIDString: String): TPGPKeyID7;
    function GetPGPKeyIDString(const PGPKeyID: TPGPKeyID7): String;
    function GetAppPathRegEntry(const RegPath, RegKey: String): String;
  end;

function GetPreferences(var Prefs: TPreferenceRec; Flags: Longint): Longint;
function SetPreferences(const Prefs: TPreferenceRec; Flags: Longint): Longint;
function GetServerList(var ServerList: TStringList): Longint;
function GetPrefWarnOnWipe(var WarnOnWipe: Longbool): Longint;
function SetPrefWarnOnWipe(WarnOnWipe: Longbool): Longint;
function GetPrefWipeCount(var WipeCount: PGPUInt32): Longint;
function SetPrefWipeCount(WipeCount: PGPUInt32): Longint;
function GetPrefCacheTime(var Seconds: Longint): Longint;
function GetPrefShareCache(var Share: Longbool): Longint;
function GetPreferredCipherAlgorithm(var Algorithm: PGPCipherAlgorithm): Longint;
function GetAllowedCipherAlgorithms(var AlgorithmList: TPGPCipherAlgorithms; var Count: Longint): Longint;

const PrefsFile: TPrefsFile = nil;

implementation

function ExtractLine(var Str: String): String;
var
  iPos		 : Longint;
begin
  iPos := FirstStrPos(CRLF, Str);
  if iPos <> 0 then begin
    Result := Copy(Str, 1, pred(iPos));
    Delete(Str, 1, succ(iPos));
  end
  else begin
    Result := TrimRight(Str);
    Str := E;
  end;
end;

function ExtractStr(const Str: String): String;
var
  iPos		 : Longint;
begin
  Result := Str;
  iPos := Length(Result);
  if iPos > 1 then begin
    if (Result[iPos] = BS) and (Result[pred(iPos)] = SP) then begin
      Delete(Result, pred(iPos), 2);
      dec(iPos, 2);
    end;
    if (iPos <> 0) and (Result[1] = QU) and (Result[iPos] = QU) then begin
      Delete(Result, iPos, 1);
      Delete(Result, 1, 1);
    end;
  end;
end;

function DeleteTags(const Str, Tag: String; Pos: Integer): String;
var
  sCloseTag	 : String;
  iLen, iPos	 : Integer;
begin
  Result := Str;
  iLen:= Length(Tag);
  if FastLoStrPos(Tag, Result, Pos) then begin
    Delete(Result, 1, pred(Pos + iLen));
    sCloseTag := Tag;
    Insert(SL, sCloseTag, 2);
    iPos:= Length(Result) - iLen;
    if FastLoStrPos(sCloseTag, Result, iPos) then Delete(Result, iPos, MAXINT);
  end;
end;

function TPrefsList.XmlIndexOf(const S: String): Integer;
begin
  for Result := 0 to pred(Count) do if FastLoStrPos(S, Strings[Result], StringPos[Result]) then Exit;
  Result := -1;
end;

function TPrefsList.GetStringPos(I: Integer): Integer;
begin
  Result := Integer(Objects[I]);
end;

procedure TPrefsList.SetStringPos(I, Value: Integer);
begin
  Objects[I] := TObject(Value);
end;

constructor TPrefsFile.Create;
var
  sFilePath	 : String;
begin
  inherited;
  PrefsData := TPrefsList.Create;
  PrefsData.BeginUpdate;
  if PGP8X then
    sFilePath := GetAppPathRegEntry(AppDataRegPath, 'AppData') + PathPGP + '\'
  else sFilePath := GetAppPathRegEntry(AppDataRegPath, 'AppData') + PathNAI + '\';
  if PGP9X then
    PrefsFilePath := sFilePath + PrefsFileXml
  else PrefsFilePath := sFilePath + PrefsFileTxt;
  GroupsFilePath := sFilePath + GroupsFileName;
  LoadPrefs;
end;

destructor TPrefsFile.Destroy;
begin
  PrefsData.Free;
  inherited;
end;

function TPrefsFile.LoadPrefs: Longbool;
var
  sBuffer	 : String;
  iIndex	 : Longint;
begin
  Result := false;
  if (PrefsData <> nil) and (PrefsFilePath <> E) then with PrefsData do begin
    if FileAge(PrefsFilePath) >= CacheAge then begin
      try
	LoadFromFile(PrefsFilePath);
	if PGP9X then with PrefsData do begin
	  for iIndex := 0 to pred(Count) do StringPos[iIndex] := FirstPos(LT, Strings[iIndex]);
	end
	else begin
	  sBuffer := Text;
	  iIndex := 1;
	  repeat
	    iIndex := ShortLoStrPos(SP + BS + CRLF, sBuffer, iIndex);
	    if iIndex > 1 then begin
	      if (sBuffer[pred(iIndex)] = QU) then
		System.Delete(sBuffer, pred(iIndex), Length(QU + SP + BS + CRLF + QU))
	      else System.Delete(sBuffer, iIndex, Length(SP + BS + CRLF));
	    end;
	  until iIndex = 0;
	  Text := sBuffer;
	end;
	Result := (Count <> 0);
	if Result then CacheAge := DateTimeToFileDate(Now);
      except
      end;
    end
    else Result := true;
  end;
end;

function TPrefsFile.SavePrefs: Longbool;
begin
  Result := false;
  if (PrefsData <> nil) and (PrefsFilePath <> E) then with PrefsData do begin
    try
      PrefsData.SaveToFile(PrefsFilePath);
      Result := true;
    except
    end;
  end;
end;

function TPrefsFile.GetPrefDict(var StartIndex: Integer): String;
var
  iIndex	 : Integer;
begin
  Result := E;
  if PrefsData <> nil then with PrefsData do begin
    iIndex := succ(StartIndex);
    while (iIndex < Count) and not FastLoStrPos(tagCDict, Strings[iIndex], StringPos[iIndex]) do begin
      if FastLoStrPos(tagKey, Strings[iIndex], StringPos[iIndex]) then
	Result := Result + CM + QU + DeleteTags(Strings[iIndex], tagKey, StringPos[iIndex])
      else Result := Result + EQ + GetPrefValue(E, iIndex) + QU;
      inc(iIndex);
    end;
    if Result <> E then System.Delete(Result, 1, Length(CM));
    StartIndex := iIndex;
  end;
end;

function TPrefsFile.GetPrefArray(var StartIndex: Integer): String;
var
  iIndex	 : Integer;
begin
  Result := E;
  if PrefsData <> nil then with PrefsData do begin
    if not FastLoStrPos(tagArray + tagCArray, Strings[StartIndex], StringPos[StartIndex]) then begin
      iIndex := succ(StartIndex);
      while (iIndex < Count) and not FastLoStrPos(tagCArray, Strings[iIndex], StringPos[iIndex]) do begin
	if not FastLoStrPos(tagKey, Strings[iIndex], StringPos[iIndex]) then begin
	  Result := Result + CRLF + GetPrefValue(E, iIndex);
	end;
	inc(iIndex);
      end;
      if Result <> E then System.Delete(Result, 1, Length(CRLF));
      StartIndex := iIndex;
    end;
  end;
end;

function TPrefsFile.GetPrefValue(const PrefName: String; var PrefIndex: Integer): String;
var
  sPrefValue	 : String;
  iPrefPos	 : Integer;
begin
  Result := E;
  if PrefsData <> nil then with PrefsData do begin
    if PrefName <> E then PrefIndex := succ(XmlIndexOf(tagKey + PrefName + tagCKey));
    if (PrefIndex > -1) and (PrefIndex < pred(Count)) then begin
      sPrefValue := Strings[PrefIndex];
      iPrefPos := StringPos[PrefIndex];
      if FastLoStrPos(tagData, sPrefValue, iPrefPos) then
	Result := DeleteTags(sPrefValue, tagData, iPrefPos)
      else if FastLoStrPos(tagString, sPrefValue, iPrefPos) then
	Result := Utf8OrAnsi(DeleteTags(sPrefValue, tagString, iPrefPos))
      else if FastLoStrPos(tagInteger, sPrefValue, iPrefPos) then
	Result := DeleteTags(sPrefValue, tagInteger, iPrefPos)
      else if FastLoStrPos(tagFalse, sPrefValue, iPrefPos) then
	Result := 'FALSE'
      else if FastLoStrPos(tagTrue, sPrefValue, iPrefPos) then
	Result := 'TRUE'
      else if FastLoStrPos(tagDict, sPrefValue, iPrefPos) then
	Result := GetPrefDict(PrefIndex)
      else if FastLoStrPos(tagArray, sPrefValue, iPrefPos) then Result := GetPrefArray(PrefIndex);
    end;
  end;
end;

function TPrefsFile.SetPrefDict(const PrefDict: String; var StartIndex: Integer): Longbool;
var
  iIndex	 : Integer;
  slPrefDict	 : TStringList;
  sPrefName	 : String;
  sPrefValue	 : String;
begin
  Result := false;
  if PrefsData <> nil then with PrefsData do begin
    if PrefDict <> E then begin
      slPrefDict := TStringList.Create;
      if slPrefDict <> nil then begin
	try
	  slPrefDict.CommaText := PrefDict;
	  iIndex := succ(StartIndex);
	  while (iIndex < Count) and not FastLoStrPos(tagCDict, Strings[iIndex], StringPos[iIndex]) do begin
	    if FastLoStrPos(tagKey, Strings[iIndex], StringPos[iIndex]) then begin
	      sPrefName := DeleteTags(Strings[iIndex], tagKey, StringPos[iIndex]);
	      sPrefValue := slPrefDict.Values[sPrefName];
	      inc(iIndex);
	      Result := SetPrefValue(E, sPrefValue, iIndex);
	    end;
	    inc(iIndex);
	  end;
	finally
	  slPrefDict.Free;
	end;
      end;
    end
    else while (iIndex < Count) and not FastLoStrPos(tagCDict, Strings[iIndex], StringPos[iIndex]) do inc(iIndex);
    StartIndex := iIndex;
  end;
end;

function TPrefsFile.SetPrefArray(const PrefArray: String; var StartIndex: Integer): Longbool;
var
  iIndex	 : Integer;
  sPrefArray	 : String;
begin
  Result := false;
  sPrefArray := PrefArray;
  if PrefsData <> nil then with PrefsData do begin
    if not FastLoStrPos(tagArray + tagCArray, Strings[StartIndex], StringPos[StartIndex]) then begin
      if PrefArray <> E then begin
	iIndex := succ(StartIndex);
	while (iIndex < Count) and not FastLoStrPos(tagCArray, Strings[iIndex], StringPos[iIndex]) do begin
	  Result := SetPrefValue(E, ExtractLine(sPrefArray), iIndex);
	  inc(iIndex);
	end;
      end
      else while (iIndex < Count) and not FastLoStrPos(tagCArray, Strings[iIndex], StringPos[iIndex]) do inc(iIndex);
      StartIndex := iIndex;
    end;
  end;
end;

function TPrefsFile.SetPrefValue(const PrefName, PrefValue: String; var PrefIndex: Integer): Longbool;
var
  sPrefLine	 : String;
  iPrefPos	 : Integer;
begin
  Result := false;
  if PrefsData <> nil then with PrefsData do begin
    if PrefName <> E then PrefIndex := succ(XmlIndexOf(tagKey + PrefName + tagCKey));
    if (PrefIndex > -1) and (PrefIndex < pred(Count)) then begin
      sPrefLine := Strings[PrefIndex];
      iPrefPos := StringPos[PrefIndex];
      if FastLoStrPos(tagData, sPrefLine, iPrefPos) then begin
	System.Delete(sPrefLine, iPrefPos + Length(tagData), MAXINT);
	PrefsData[PrefIndex] := sPrefLine + PrefValue + tagCData;
	Result := true;
      end
      else if FastLoStrPos(tagString, sPrefLine, iPrefPos) then begin
	System.Delete(sPrefLine, iPrefPos + Length(tagString), MAXINT);
	if PrefValue = BS then
	  PrefsData[PrefIndex] := sPrefLine + tagCString
	else PrefsData[PrefIndex] := sPrefLine + AnsiToUtf8(PrefValue) + tagCString;
	Result := true;
      end
      else if FastLoStrPos(tagInteger, sPrefLine, iPrefPos) then begin
	System.Delete(sPrefLine, iPrefPos + Length(tagInteger), MAXINT);
	PrefsData[PrefIndex] := sPrefLine + PrefValue + tagCInteger;
	Result := true;
      end
      else if FastLoStrPos(tagFalse, sPrefLine, iPrefPos) or FastLoStrPos(tagTrue, sPrefLine, iPrefPos) then begin
	System.Delete(sPrefLine, iPrefPos, MAXINT);
	if Uppercase(PrefValue) = 'FALSE' then
	  PrefsData[PrefIndex] := sPrefLine + tagFalse + tagCFalse
	else if Uppercase(PrefValue) = 'TRUE' then
	  PrefsData[PrefIndex] := sPrefLine + tagTrue + tagCTrue
	else Exit;
	Result := true;
      end
      else if FastLoStrPos(tagDict, sPrefLine, iPrefPos) then
	Result := SetPrefDict(PrefValue, PrefIndex)
      else if FastLoStrPos(tagArray, sPrefLine, iPrefPos) then Result := SetPrefArray(PrefValue, PrefIndex);
    end;
  end;
end;

function TPrefsFile.GetPGPKeyID(const PGPKeyIDString: String): TPGPKeyID7; register; assembler;
asm	// EAX = @Self, EDX = @PGPKeyIDString, ECX = @Result
  PUSH	EBX
  OR	EDX,EDX
  JE	@ERROR
  CMP	DWORD PTR [EDX - 4],TYPE TPGPKeyID7 * 2 + 2
  JNE	@ERROR
  AND	BYTE PTR [EDX + 1],0DFh
  CMP	WORD PTR [EDX],'X0'
  JNE	@ERROR
  MOV	EBX,[EDX - 4]
  SHR	EBX,1
  DEC	EBX
  @START:
  MOV	AX,[EDX + EBX * 2]
  @FIRSTBYTE:
  SUB	AL,'0'
  JB	@ERROR
  CMP	AL,09h
  JA	@FIRSTCHAR
  JMP	@NEXTBYTE
  @FIRSTCHAR:
  AND	AL,0DFh
  SUB	AL,'A' - '0'
  JB	@ERROR
  CMP	AL,05h
  JA	@ERROR
  ADD	AL,0Ah
  @NEXTBYTE:
  SUB	AH,'0'
  JB	@ERROR
  CMP	AH,09h
  JA	@NEXTCHAR
  JMP	@DECODE
  @NEXTCHAR:
  AND	AH,0DFh
  SUB	AH,'A' - '0'
  JB	@ERROR
  CMP	AH,05h
  JA	@ERROR
  ADD	AH,0Ah
  @DECODE:
  SHL	AL,4
  ADD	AL,AH
  MOV	[ECX + EBX - 1],AL
  DEC	EBX
  JNZ	@START
  POP	EBX
  RET
  @ERROR:
  POP	EBX
  MOV	DWORD PTR [ECX],0
end;

function TPrefsFile.GetPGPKeyIDString(const PGPKeyID: TPGPKeyID7): String; register; assembler;
asm	// EAX = @Self, EDX = @PGPKeyID, ECX = @Result
  PUSH	EBX
  PUSH	EDI
  PUSH	ESI
  MOV	ESI,EDX
  MOV	EDI,ECX
  MOV	EAX,ECX
  CALL	SYSTEM.@LSTRCLR
  MOV	EAX,TYPE TPGPKeyID7 * 2
  CALL	SYSTEM.@NEWANSISTRING
  MOV	[EDI],EAX
  MOV	EDI,EAX
  MOV	ECX,TYPE TPGPKeyID7 - 1
  @LOOP:
  MOVZX	EAX,BYTE PTR[ESI + ECX]
  MOV	EDX,EAX
  AND	EAX,0Fh
  SHR	EDX,4
  MOV	AH,BYTE PTR[@HEXCHARS + EAX]
  MOV	AL,BYTE PTR[@HEXCHARS + EDX]
  MOV	[EDI + ECX * 2],AX
  DEC	ECX
  JNS	@LOOP
  @END:
  POP	ESI
  POP	EDI
  POP	EBX
  RET
  @HEXCHARS:
  DB	'0123456789ABCDEF';
end;

function TPrefsFile.GetAppPathRegEntry(const RegPath, RegKey: String): String;
var
  hRegKey	 : hKey;
  dwBufSize	 : DWord;
  aFilePath	 : Array[0..MAX_PATH] of Char;
begin
  dwBufSize := MAX_PATH;
  if RegOpenKeyEx(HKEY_CURRENT_USER, PChar(RegPath), 0, KEY_QUERY_VALUE, hRegKey) = ERROR_SUCCESS then begin
    try
      if RegQueryValueEx(hRegKey, PChar(RegKey), nil, nil, @aFilePath, @dwBufSize) = ERROR_SUCCESS then begin
	Result := aFilePath;
	Exit;
      end;
    finally
      RegCloseKey(hRegKey);
    end;
  end;
  Result := E;
end;

function GetPreferences(var Prefs: TPreferenceRec; Flags: Longint): Longint;
var
  Index		 : Integer;
  Context	 : pPGPContext;
  FileSpec	 : pPGPFileSpec;
  FileOut	 : PChar;
  KeyOut	 : PChar;
  PGPKeyID	 : TPGPKeyID7;
  IDSize	 : PGPSize;
  KeyID		 : TKeyID;
begin
  Result := kPGPError_PrefNotFound;
  if Flags <> 0 then begin
    FillChar(Prefs, SizeOf(Prefs), 0);
    if PGP7X then begin
      if PrefsFile <> nil then with PrefsFile do if LoadPrefs then begin
	try
	  if PGP9X then begin
	    if Flags and PrefsFlag_PublicKeyring <> 0 then Prefs.PublicKeyring := GetPrefValue(PubFile, Index);
	    if Flags and PrefsFlag_PrivateKeyring <> 0 then Prefs.PrivateKeyring := GetPrefValue(SecFile, Index);
	    if Flags and PrefsFlag_RandomSeedFile <> 0 then Prefs.RandomSeedFile := GetPrefValue(RNGFile, Index);
	    if Flags and PrefsFlag_GroupsFile <> 0 then Prefs.GroupsFile := PrefsFile.GroupsFilePath;
	    if Flags and PrefsFlag_DefaultKeyID <> 0 then Prefs.DefaultKeyHexID := GetPrefValue(DefKeyID9, Index);
	  end
	  else begin
	    if Flags and PrefsFlag_PublicKeyring <> 0 then Prefs.PublicKeyring := ExtractStr(PrefsData.Values[PubFile]);
	    if Flags and PrefsFlag_PrivateKeyring <> 0 then Prefs.PrivateKeyring := ExtractStr(PrefsData.Values[SecFile]);
	    if Flags and PrefsFlag_RandomSeedFile <> 0 then Prefs.RandomSeedFile := ExtractStr(PrefsData.Values[RNGFile]);
	    if Flags and PrefsFlag_GroupsFile <> 0 then Prefs.GroupsFile := PrefsFile.GroupsFilePath;
	    if Flags and PrefsFlag_DefaultKeyID <> 0 then begin
	      PGPKeyID := GetPGPKeyID(PrefsData.Values[DefKeyID7]);
	      if PGPGetKeyIDString(PGPKeyID, kPGPKeyIDString_Full, KeyID) = 0 then Prefs.DefaultKeyHexID := KeyID;
	    end;
	  end;
	  Result := 0;
	except
	end;
      end;
    end
    else begin
      Context := nil;
      FileSpec := nil;
      Result := PGPNewContext(kPGPsdkAPIVersion, Context);
      if Result <> 0 then Exit;
      try
	Result := PGPsdkLoadDefaultPrefs(Context);
	if Result <> 0 then Exit;

	// Pubring
	if Flags and PrefsFlag_PublicKeyring <> 0 then begin
	  Result := PGPsdkPrefGetFileSpec(Context, kPGPsdkPref_PublicKeyring, FileSpec);
	  try
	    if Result = 0 then begin
	      FileOut := nil;
	      Result := PGPGetFullPathFromFileSpec(FileSpec, FileOut);
	      try
		Prefs.PublicKeyring := FileOut;
	      finally
		PGPFreeData(FileOut);
	      end;
	    end;
	  finally
	    PGPFreeFileSpec(FileSpec);
	  end;
	end;
	if Result <> 0 then Exit;

	// Secring
	if Flags and PrefsFlag_PrivateKeyring <> 0 then begin
	  Result := PGPsdkPrefGetFileSpec(Context, kPGPsdkPref_PrivateKeyring, FileSpec);
	  try
	    if Result = 0 then begin
	      FileOut := nil;
	      Result := PGPGetFullPathFromFileSpec(FileSpec, FileOut);
	      try
		Prefs.PrivateKeyring := FileOut;
	      finally
		PGPFreeData(FileOut);
	      end;
	    end;
	  finally
	    PGPFreeFileSpec(FileSpec);
	  end;
	end;
	if Result <> 0 then Exit;

	// Randseed file
	if Flags and PrefsFlag_RandomSeedFile <> 0 then begin
	  Result := PGPsdkPrefGetFileSpec(Context, kPGPsdkPref_RandomSeedFile, FileSpec);
	  try
	    if Result = 0 then begin
	      FileOut := nil;
	      Result := PGPGetFullPathFromFileSpec(FileSpec, FileOut);
	      try
		Prefs.RandomSeedFile := FileOut;
	      finally
		PGPFreeData(FileOut);
	      end;
	    end;
	  finally
	    PGPFreeFileSpec(FileSpec);
	  end;
	end;
	if Result <> 0 then Exit;

	// Groups file
	if Flags and PrefsFlag_GroupsFile <> 0 then begin
	  Result := PGPsdkPrefGetFileSpec(Context, kPGPsdkPref_GroupsFile, FileSpec);
	  try
	    if Result = 0 then begin
	      FileOut := nil;
	      Result := PGPGetFullPathFromFileSpec(FileSpec, FileOut);
	      try
		Prefs.GroupsFile := FileOut;
	      finally
		PGPFreeData(FileOut);
	      end;
	    end;
	  finally
	    PGPFreeFileSpec(FileSpec);
	  end;
	end;
	if Result <> 0 then Exit;

	// Private key
	if Flags and PrefsFlag_DefaultKeyID <> 0 then begin
	  KeyOut := nil;
	  Result := PGPsdkPrefGetData(Context, kPGPsdkPref_DefaultKeyID, @KeyOut, IDSize);
	  if Result <> 0 then Exit;
	  try
	    Result := PGPImportKeyID(KeyOut, PGPKeyID);
	  finally
	    PGPFreeData(KeyOut);
	  end;
	  if Result <> 0 then Exit;
	  Result := PGPGetKeyIDString(PGPKeyID, kPGPKeyIDString_Full, KeyID);
	  if Result = 0 then Prefs.DefaultKeyHexID := KeyID;
	end;
      finally
	PGPFreeContext(Context);
      end;
    end;
  end;
end;

function SetPreferences(const Prefs: TPreferenceRec; Flags: Longint): Longint;
var
  RingData	 : TStringList;
  NewPrefValue	 : String;
  OldPrefValue	 : String;
  Index		 : Integer;
  Context	 : pPGPContext;
  FileSpec	 : pPGPFileSpec;
  KeyPropsList	 : TKeyPropsList;
  RingContext	 : pPGPContext;
  MainKeySet	 : pPGPKeySet;
  Key		 : pPGPKey;
  KeyIn		 : TPGPKeyID7;
  PGPKeyID	 : TPGPKeyID7;
  IDSize	 : PGPSize;
begin
  Result := kPGPError_PrefNotFound;
  if Flags <> 0 then begin
    if PGP7X then begin
      if PrefsFile <> nil then with PrefsFile do if LoadPrefs then begin
	try
	  RingData := TStringList.Create;
	  if RingData <> nil then begin
	    try
	      if Flags and PrefsFlag_PublicKeyring <> 0 then begin
		if not FileExists(Prefs.PublicKeyring) then begin
		  Result := kPGPError_FileNotFound;
		  Exit;
		end;
		if PGP9X then begin
		  NewPrefValue := E;
		  OldPrefValue := GetPrefValue('fileKeyrings', Index);
		  with RingData do while OldPrefValue <> E do begin
		    CommaText := ExtractLine(OldPrefValue);
		    if Values['default'] = 'TRUE' then Values['pubring'] := Prefs.PublicKeyring;
		    NewPrefValue := NewPrefValue + CRLF + CommaText;
		  end;
		  Delete(NewPrefValue, 1, Length(CRLF));
		  SetPrefValue(PubFile, Prefs.PublicKeyring, Index);
		  SetPrefValue('fileKeyrings', NewPrefValue, Index);
		end
		else PrefsData.Values[PubFile] := QU + Prefs.PublicKeyring + QU;
		Result := 0;
	      end;
	      if Flags and PrefsFlag_PrivateKeyring <> 0 then begin
		if not FileExists(Prefs.PrivateKeyring) then begin
		  Result := kPGPError_FileNotFound;
		  Exit;
		end;
		if PGP9X then begin
		  NewPrefValue := E;
		  OldPrefValue := GetPrefValue('fileKeyrings', Index);
		  with RingData do while OldPrefValue <> E do begin
		    CommaText := ExtractLine(OldPrefValue);
		    if Values['default'] = 'TRUE' then Values['secring'] := Prefs.PrivateKeyring;
		    NewPrefValue := NewPrefValue + CRLF + CommaText;
		  end;
		  Delete(NewPrefValue, 1, Length(CRLF));
		  SetPrefValue(SecFile, Prefs.PrivateKeyring, Index);
		  SetPrefValue('fileKeyrings', NewPrefValue, Index);
		end
		else PrefsData.Values[SecFile] := QU + Prefs.PrivateKeyring + QU;
		Result := 0;
	      end;
	      if Flags and PrefsFlag_RandomSeedFile <> 0 then begin
		if not FileExists(Prefs.RandomSeedFile) then begin
		  Result := kPGPError_FileNotFound;
		  Exit;
		end;
		if PGP9X then
		  SetPrefValue(RNGFile, Prefs.RandomSeedFile, Index)
		else PrefsData.Values[RNGFile] := QU + Prefs.RandomSeedFile + QU;
		Result := 0;
	      end;
	      if Flags and PrefsFlag_GroupsFile <> 0 then begin
		MessageBox(0, 'PGP 7.X & later don''t support changing the default groups file name',
			   'PGP Components error:', MB_ICONERROR);
	      end;
	      if Flags and PrefsFlag_DefaultKeyID <> 0 then begin
		KeyPropsList := nil;
		try
		  Result := KeyRings.InitKeyRings(RingContext, MainKeySet);
		  if Result <> 0 then Exit;
		  try
		    if FindKeyProps(Prefs.DefaultKeyHexID, KeyPropsList, 0, KeyFilterFlag_CanSign, Any_Ordering) = 1 then begin
		      Result := GetKeyByHexID(MainKeySet, Prefs.DefaultKeyHexID, Key);
		      if Result <> 0 then Exit;
		      Result := PGPGetKeyIDFromKey(Key, PGPKeyID);
		      if Result <> 0 then Exit;
		      if PGP9X then
			SetPrefValue(DefKeyID9, Prefs.DefaultKeyHexID, Index)
		      else PrefsData.Values[DefKeyID7] := '0x' + GetPGPKeyIDString(PGPKeyID);
		    end
		    else Result := kPGPError_SecretKeyNotFound;
		  finally
		    KeyRings.FreeKeyRings;
		  end;
		finally
		  KeyPropsList.Free;
		end;
	      end;
	      if (Result = 0) and not SavePrefs then Result := kPGPError_WriteFailed;
	    finally
	      RingData.Free;
	    end;
	  end
	  else Result := kPGPError_OutOfMemory;
	except
	end;
      end;
    end
    else begin
      Context := nil;
      FileSpec := nil;
      Result := PGPNewContext(kPGPsdkAPIVersion, Context);
      if Result <> 0 then Exit;
      try
	Result := PGPsdkLoadDefaultPrefs(Context);
	if Result <> 0 then Exit;

	// Pubring
	if Flags and PrefsFlag_PublicKeyring <> 0 then begin
	  if not FileExists(Prefs.PublicKeyring) then begin
	    Result := kPGPError_FileNotFound;
	    Exit;
	  end;
	  Result := PGPNewFileSpecFromFullPath(Context, PChar(Prefs.PublicKeyring), FileSpec);
	  try
	    if Result = 0 then Result := PGPsdkPrefSetFileSpec(Context, kPGPsdkPref_PublicKeyring, FileSpec);
	    if Result <> 0 then Exit;
	  finally
	    PGPFreeFileSpec(FileSpec);
	  end;
	end;

	// Secring
	if Flags and PrefsFlag_PrivateKeyring <> 0 then begin
	  if not FileExists(Prefs.PrivateKeyring) then begin
	    Result := kPGPError_FileNotFound;
	    Exit;
	  end;
	  Result := PGPNewFileSpecFromFullPath(Context, PChar(Prefs.PrivateKeyring), FileSpec);
	  try
	    if Result = 0 then Result := PGPsdkPrefSetFileSpec(Context, kPGPsdkPref_PrivateKeyring, FileSpec);
	    if Result <> 0 then Exit;
	  finally
	    PGPFreeFileSpec(FileSpec);
	  end;
	end;

	// Randseed file
	if Flags and PrefsFlag_RandomSeedFile <> 0 then begin
	  if not FileExists(Prefs.RandomSeedFile) then begin
	    Result := kPGPError_FileNotFound;
	    Exit;
	  end;
	  Result := PGPNewFileSpecFromFullPath(Context, PChar(Prefs.RandomSeedFile), FileSpec);
	  try
	    if Result = 0 then Result := PGPsdkPrefSetFileSpec(Context, kPGPsdkPref_RandomSeedFile, FileSpec);
	    if Result <> 0 then Exit;
	  finally
	    PGPFreeFileSpec(FileSpec);
	  end;
	end;

	// Groups file
	if Flags and PrefsFlag_GroupsFile <> 0 then begin
	  if not FileExists(Prefs.GroupsFile) then begin
	    Result := kPGPError_FileNotFound;
	    Exit;
	  end;
	  Result := PGPNewFileSpecFromFullPath(Context, PChar(Prefs.GroupsFile), FileSpec);
	  try
	    if Result = 0 then Result := PGPsdkPrefSetFileSpec(Context, kPGPsdkPref_GroupsFile, FileSpec);
	    if Result <> 0 then Exit;
	  finally
	    PGPFreeFileSpec(FileSpec);
	  end;
	end;

	// Private key
	if Flags and PrefsFlag_DefaultKeyID <> 0 then begin
	  KeyPropsList := nil;
	  try
	    if FindKeyProps(Prefs.DefaultKeyHexID, KeyPropsList, 0, KeyFilterFlag_CanSign, Any_Ordering) = 1 then begin
	      Result := KeyRings.InitKeyRings(RingContext, MainKeySet);
	      if Result <> 0 then Exit;
	      try
		Result := GetKeyByHexID(MainKeySet, Prefs.DefaultKeyHexID, Key);
		if Result <> 0 then Exit;
		Result := PGPGetKeyIDFromString(PChar(Prefs.DefaultKeyHexID), kPGPPublicKeyAlgorithm_Invalid, KeyIn);
		if Result <> 0 then Exit;
		Result := PGPExportKeyID(@KeyIn, PGPKeyID, IDSize);
		if Result <> 0 then Exit;
	      finally
		KeyRings.FreeKeyRings;
	      end;
	      Result := PGPsdkPrefSetData(Context, kPGPsdkPref_DefaultKeyID, @PGPKeyID, IDSize);
	    end
	    else Result := kPGPError_SecretKeyNotFound;
	    if Result <> 0 then Exit;
	  finally
	    KeyPropsList.Free;
	  end;
	end;
	Result := PGPsdkSavePrefs(Context);
      finally
	PGPFreeContext(Context);
      end;
    end;
    if Result = 0 then begin
      PGPclNotifyKeyringChanges(GetCurrentProcessID);
      PGPclNotifyPrefsChanges(GetCurrentProcessID);
    end;
  end;
end;

function GetServerList(var ServerList: TStringList): Longint;
const
  Protocols: Array[kPGPKeyServerProtocol_Invalid..kPGPKeyServerProtocol_HTTPS] of String = (
    E, 'ldap', 'http', 'ldaps', 'https'
  );
var
  Servers	 : String;
  Entry		 : TStringList;
  Index		 : Integer;
  Port		 : String;
  PGPPref	 : pPGPPref;
  KeyServerList	 : pPGPKeyServerEntry;
  KeyServerEntry : pPGPKeyServerEntry;
  ServerCount	 : PGPUInt32;
  ServerURL	 : String;
begin
  if PGP7X then begin
    Result := kPGPError_PrefNotFound;
    if PrefsFile <> nil then with PrefsFile do if LoadPrefs then begin
      try
	Entry := TStringList.Create;
	ServerList := TStringList.Create;
	if (Entry <> nil) and (ServerList <> nil) then with Entry do begin
	  try
	    if PGP9X then begin
	      Servers := GetPrefValue(KeyServers9, Index);
	      repeat
		CommaText := ExtractLine(Servers);
		if Count >= 8 then begin
		  ServerURL := Protocols[StrToIntDef(Values[kPGPPrefKeyServerProtocol], 0)] + '://' +
			       Values[kPGPPrefKeyServerHostname];
		  Port := Values[kPGPPrefKeyServerPort];
		  if (ServerURL <> E) and (Port <> '0') then ServerURL := ServerURL + ':' + Port;
		  ServerList.Add(ServerURL);
		end;
	      until Servers = E;
	    end
	    else begin
	      ServerCount := 0;
	      repeat
		inc(ServerCount);
		CommaText := PrefsData.Values[KeyServers7 + IntToStr(ServerCount)];
		if Count >= 8 then begin
		  ServerURL := Protocols[StrToIntDef(Strings[0], 0)] + '://' + Strings[3];
		  Port := Strings[4];
		  if (ServerURL <> E) and (Port <> '0') then ServerURL := ServerURL + ':' + Port;
		  ServerList.Add(ServerURL);
		end;
	      until Count = 0;
	    end;
	  finally
	    Entry.Free;
	  end;
	  Result := 0;
	end;
      except
      end;
    end;
  end
  else begin
    PGPPref := nil;
    KeyServerList := nil;
    ServerList := TStringList.Create;
    if ServerList <> nil then begin
      Result := PGPclOpenClientPrefs(PGPGetDefaultMemoryMgr, PGPPref);
      if Result = 0 then begin
	try
	  Result := PGPGetKeyServerPrefs(PGPPref, KeyServerList, ServerCount);
	  if Result = 0 then begin
	    try
	      if ServerCount <> 0 then begin
		KeyServerEntry := KeyServerList;
		for ServerCount := ServerCount downto 1 do with KeyServerEntry^ do begin
		  if Protocol <> kPGPKeyServerProtocol_Invalid then ServerURL := Protocols[Protocol] + '://';
		  if ServerDNS <> E then begin
		    ServerURL := ServerURL + ServerDNS;
		    if ServerPort <> 0 then ServerURL := ServerURL + ':' + IntToStr(ServerPort);
		    ServerList.Add(ServerURL);
		  end;
		  inc(KeyServerEntry);
		end;
	      end
	      else Result := kPGPError_PrefNotFound;
	    finally
	      PGPFreeData(KeyServerList);
	    end;
	  end;
	finally
	  PGPclCloseClientPrefs(PGPPref, false);
	end;
      end;
    end
    else Result := kPGPError_OutOfMemory;
  end;
end;

function GetPrefWarnOnWipe(var WarnOnWipe: Longbool): Longint;
var
  Index		 : Integer;
  PGPPref	 : pPGPPref;
  Pref		 : PGPBoolean;
begin
  WarnOnWipe := false;
  if PGP7X then begin
    Result := kPGPError_PrefNotFound;
    if PrefsFile <> nil then with PrefsFile do if LoadPrefs then begin
      try
	if PGP9X then
	  WarnOnWipe := (CompareText(GetPrefValue(WipeWarning9, Index), 'TRUE') = 0)
	else WarnOnWipe := (CompareText(PrefsData.Values[WipeWarning7], 'TRUE') = 0);
	Result := 0;
      except
      end;
    end;
  end
  else begin
    PGPPref := nil;
    Result := PGPclOpenClientPrefs(PGPGetDefaultMemoryMgr, PGPPref);
    if Result = 0 then begin
      try
	Result := PGPclGetPrefBoolean(PGPPref, kPGPPrefWarnOnWipe, Pref);
	if Result = 0 then WarnOnWipe := Boolean(Pref);
      finally
	PGPclCloseClientPrefs(PGPPref, false);
      end;
    end;
  end;
end;

function SetPrefWarnOnWipe(WarnOnWipe: Longbool): Longint;
var
  Index		 : Integer;
  PGPPref	 : pPGPPref;
begin
  if PGP7X then begin
    Result := kPGPError_PrefNotFound;
    if PrefsFile <> nil then with PrefsFile do if LoadPrefs then begin
      try
	if PGP9X then begin
	  if WarnOnWipe then
	    SetPrefValue(WipeWarning9, 'TRUE', Index)
	  else SetPrefValue(WipeWarning9, 'FALSE', Index)
	end
	else begin
	  if WarnOnWipe then
	    PrefsData.Values[WipeWarning7] := 'TRUE'
	  else PrefsData.Values[WipeWarning7] := 'FALSE';
	 end;
	if SavePrefs then Result := 0;
      except
      end;
    end;
  end
  else begin
    PGPPref := nil;
    Result := PGPclOpenClientPrefs(PGPGetDefaultMemoryMgr, PGPPref);
    if Result = 0 then begin
      try
	Result := PGPclSetPrefBoolean(PGPPref, kPGPPrefWarnOnWipe, PGPBoolean(WarnOnWipe) and 1);
      finally
	PGPclCloseClientPrefs(PGPPref, true);
      end;
    end;
  end;
  if Result = 0 then PGPclNotifyPrefsChanges(GetCurrentProcessID);
end;

function GetPrefWipeCount(var WipeCount: PGPUInt32): Longint;
var
  Index		 : Integer;
  PGPPref	 : pPGPPref;
begin
  WipeCount := 0;
  if PGP7X then begin
    Result := kPGPError_PrefNotFound;
    if PrefsFile <> nil then with PrefsFile do if LoadPrefs then begin
      try
	if PGP9X then
	  WipeCount := StrToIntDef(GetPrefValue(FileWipeCount, Index), 0)
	else WipeCount := StrToIntDef(PrefsData.Values[FileWipeCount], 0);
	Result := 0;
      except
      end;
    end;
  end
  else begin
    PGPPref := nil;
    Result := PGPclOpenClientPrefs(PGPGetDefaultMemoryMgr, PGPPref);
    if Result = 0 then begin
      try
	Result := PGPclGetPrefNumber(PGPPref, kPGPPrefFileWipePasses, WipeCount);
      finally
	PGPclCloseClientPrefs(PGPPref, false);
      end;
    end;
  end;
end;

function SetPrefWipeCount(WipeCount: PGPUInt32): Longint;
var
  Index		 : Integer;
  PGPPref	 : pPGPPref;
begin
  if PGP7X then begin
    Result := kPGPError_PrefNotFound;
    if PrefsFile <> nil then with PrefsFile do if LoadPrefs then begin
      try
	if PGP9X then
	  SetPrefValue(FileWipeCount, IntToStr(WipeCount), Index)
	else PrefsData.Values[FileWipeCount] := IntToStr(WipeCount);
	if SavePrefs then Result := 0;
      except
      end;
    end;
  end
  else begin
    PGPPref := nil;
    Result := PGPclOpenClientPrefs(PGPGetDefaultMemoryMgr, PGPPref);
    if Result = 0 then begin
      try
	Result := PGPclSetPrefNumber(PGPPref, kPGPPrefFileWipePasses, WipeCount);
      finally
	PGPclCloseClientPrefs(PGPPref, true);
      end;
    end;
  end;
  if Result = 0 then PGPclNotifyPrefsChanges(GetCurrentProcessID);
end;

function GetPrefCacheTime(var Seconds: Longint): Longint;
var
  CacheType	 : String;
  Index		 : Integer;
  PGPPref	 : pPGPPref;
  DPref,SPref	 : PGPBoolean;
  DSecs, sSecs	 : PGPTime;
begin
  Seconds := 0;
  if PGP7X then begin
    Result := kPGPError_PrefNotFound;
    if PrefsFile <> nil then with PrefsFile do if LoadPrefs then begin
      try
	if PGP9X then
	  CacheType := GetPrefValue(CacheType9, Index)
	else CacheType := PrefsData.Values[CacheType7];
	case StrToIntDef(CacheType, 0) of
	  1: if PGP9X then
	       Seconds := StrToIntDef(GetPrefValue(CacheSeconds9, Index), 0)
	     else Seconds := StrToIntDef(PrefsData.Values[CacheSeconds7], 0);
	  2: Seconds := Longint(kPGPMaxTimeInterval);
	end;
	Result := 0;
      except
      end;
    end;
  end
  else begin
    PGPPref := nil;
    Result := PGPclOpenClientPrefs(PGPGetDefaultMemoryMgr, PGPPref);
    if Result = 0 then begin
      try
	Result := PGPclGetPrefBoolean(PGPPref, kPGPPrefDecryptCacheEnable, DPref);
	if Result <> 0 then Exit;
	Result := PGPclGetPrefBoolean(PGPPref, kPGPPrefSignCacheEnable, SPref);
	if Result <> 0 then Exit;
	if (DPref = PGPTrue) and (SPref = PGPTrue) then begin
	  Result := PGPclGetPrefNumber(PGPPref, kPGPPrefDecryptCacheSeconds, DSecs);
	  if Result <> 0 then Exit;
	  Result := PGPclGetPrefNumber(PGPPref, kPGPPrefSignCacheSeconds, SSecs);
	  if Result <> 0 then Exit;
	  Seconds := (DSecs + SSecs) div 2;
	end;
      finally
	PGPclCloseClientPrefs(PGPPref, false);
      end;
    end;
  end;
end;

function GetPrefShareCache(var Share: Longbool): Longint;
var
  Index		 : Integer;
begin
  Share := false;
  if PGP7X then begin
    Result := kPGPError_PrefNotFound;
    if PrefsFile <> nil then with PrefsFile do if LoadPrefs then begin
      try
	if PGP9X then
	  Share := (CompareText(GetPrefValue(ShareCache9, Index), 'TRUE') = 0)
	else Share := (CompareText(PrefsData.Values[ShareCache7], 'TRUE') = 0);
	Result := 0;
      except
      end;
    end;
  end
  else Result := 0;
end;

function GetPreferredCipherAlgorithm(var Algorithm: PGPCipherAlgorithm): Longint;
var
  Index		 : Integer;
  PGPPref	 : pPGPPref;
  Pref		 : PGPUInt32;
begin
  // PGP's default
  Algorithm := kPGPCipherAlgorithm_CAST5;
  if PGP7X then begin
    Result := kPGPError_PrefNotFound;
    if PrefsFile <> nil then with PrefsFile do if LoadPrefs then begin
      try
	if PGP9X then
	  Algorithm := PGPCipherAlgorithm(StrToIntDef(GetPrefValue(PrefAlgorithm, Index), kPGPCipherAlgorithm_CAST5))
	else Algorithm := PGPCipherAlgorithm(StrToIntDef(PrefsData.Values[PrefAlgorithm], kPGPCipherAlgorithm_CAST5));
	Result := 0;
      except
      end;
    end;
  end
  else begin
    PGPPref := nil;
    Result := PGPclOpenClientPrefs(PGPGetDefaultMemoryMgr, PGPPref);
    if Result = 0 then begin
      try
	Result := PGPclGetPrefNumber(PGPPref, kPGPPrefPreferredAlgorithm, Pref);
	if Result = 0 then Algorithm := PGPCipherAlgorithm(Pref);
      finally
	PGPclCloseClientPrefs(PGPPref, false);
      end;
    end;
  end;
end;

function GetAllowedCipherAlgorithms(var AlgorithmList: TPGPCipherAlgorithms; var Count: Longint): Longint;
var
  Pref		 : String;
  Index		 : Integer;
  PGPPref	 : pPGPPref;
  DataLength	 : PGPSize;
  PrefData	 : pPGPCipherAlgorithm;
  PrefIndex	 : pPGPCipherAlgorithm;
  AlgIndex	 : Longint;
begin
  Count := 0;
  FillChar(AlgorithmList, SizeOf(TPGPCipherAlgorithms), 0);
  if PGP7X then begin
    Result := kPGPError_PrefNotFound;
    if PrefsFile <> nil then with PrefsFile do if LoadPrefs then begin
      try
	if PGP9X then begin
	  Pref := GetPrefValue(AllowedAlgList, Index);
	  for AlgIndex := 0 to High(AlgorithmList) do begin
	    if Pref <> E then
	      AlgorithmList[AlgIndex] := PGPCipherAlgorithm(StrToIntDef(ExtractLine(Pref), 0))
	    else Break;
	  end;
	end
	else begin
	  for AlgIndex := 0 to High(AlgorithmList) do begin
	    Pref := PrefsData.Values[AllowedAlgList + IntToStr(succ(AlgIndex))];
	    if Pref <> E then
	      AlgorithmList[AlgIndex] := PGPCipherAlgorithm(StrToIntDef(Pref, 0))
	    else Break;
	  end;
	end;
	Count := AlgIndex;
	Result := 0;
      except
      end;
    end;
  end
  else begin
    PGPPref := nil;
    Result := PGPclOpenClientPrefs(PGPGetDefaultMemoryMgr, PGPPref);
    if Result = 0 then begin
      try
	Result := PGPclGetPrefData(PGPPref, kPGPPrefAllowedAlgorithmsList, DataLength, Pointer(PrefData));
	if Result = 0 then begin
	  PrefIndex := PrefData;
	  try
	    Count := DataLength div SizeOf(PGPCipherAlgorithm);
	    if Count > succ(High(AlgorithmList)) then Count := succ(High(AlgorithmList));
	    for AlgIndex := 0 to pred(Count) do begin
	      AlgorithmList[AlgIndex] := PrefIndex^;
	      inc(PrefIndex);
	    end;
	  finally
	    PGPFreeData(PrefData);
	  end;
	end;
      finally
	PGPclCloseClientPrefs(PGPPref, false);
      end;
    end;
  end;
end;

initialization
  PrefsFile := TPrefsFile.Create;

finalization
  PrefsFile.Free;
  PrefsFile := nil;

end.

