{$J+,Z4}
unit PSMimeTools;

{------------------------------------------------------------------------------}
{                                                                              }
{                                 This code is                                 }
{              Copyright (C) 2005-2006 by Michael in der Wiesche               }
{                                                                              }
{------------------------------------------------------------------------------}

interface

uses
  Isaac,
  Base64,
  Windows,
  SysUtils,
  StrTools,
  KeyPropTypes;

type
  TPSMime = (psNone, psPGP, psSigned, psEncrypted, psClearSigned);

function GetPSMimeType(const MimeMessage: String; var Headers, Body, Signature: String; AnalyseOnly: Longbool): TPSMime;
function SMimeBuild(const MimeMessage, Signature: String; var Separator: String; var BodyOffset: Longint;
		    HashAlgorithm: THashAlgorithm; Encrypted, OmitMimeVersion: Longbool): String;
function PGPMimeBuild(const MimeMessage: String; var Separator: String; var BodyOffset: Longint;
		      HashAlgorithm: THashAlgorithm; Encrypted, OmitMimeVersion: Longbool): String;

implementation

const
  APPDATA	= 'application/';
  MULTIPART	= 'multipart/';

  BASE64	= 'base64';
  BOUNDS	= 'boundary';

  CD_HEADER	= 'Content-Disposition:';
  CT_HEADER	= 'Content-Type:';
  CTE_HEADER	= 'Content-Transfer-Encoding:';

  PGP		= 'pgp-';
  PM_ENC	= 'encrypted';
  PM_SIG	= 'signature';
  PM_KEYS	= 'keys';

  MIME		= 'mime';
  PKCS7		= 'pkcs7-';
  SM_TYPE	= 'smime-type';
  SM_ENC	= 'enveloped';
  SM_SIG	= 'signed';
  SM_CERT	= 'certs';
  DATA		= '-data';

  PGP_BEG	= '-----BEGIN PGP';
  PGP_END	= '-----END PGP';

  DASH		= '-';
  DD		= DASH + DASH;

  MAX_HEAD_LEN	= 76;

var
  RNG: TIsaac;

function ExtractMimeHeaders(var MimeMessage: String; CopyOnly: Longbool): String;
var
  iLen, iPos: Integer;
begin
  iPos := pos(PARA, MimeMessage);
  if iPos > 0 then begin
    Result := Copy(MimeMessage, 1, succ(iPos));
    if not CopyOnly then begin
      iLen := Length(MimeMessage);
      while (iPos <= iLen) and (MimeMessage[iPos] <= SP) do inc(iPos);
      Delete(MimeMessage, 1, pred(iPos));
    end;
  end
  else begin
    if (MimeMessage <> E) and not FastStrPos(CRLF, Result, pred(Length(MimeMessage))) then
      Result := MimeMessage + CRLF
    else Result := MimeMessage;
    MimeMessage := E;
  end;
end;

function DeleteHeaders(const HeaderNames, Headers: String): String;
var
  sNameBuffer, sName : String;
  iCommaPos, iFullLen, iNameLen, iBegPos, iEndPos: Integer;
begin
  sNameBuffer := HeaderNames;
  Result := Headers;
  repeat
    iCommaPos := LastShortPos(CM, HeaderNames, Length(sNameBuffer));
    sName := Trim(Copy(sNameBuffer, succ(iCommaPos), MAXINT));
    Delete(sNameBuffer, iCommaPos, MAXINT);
    iFullLen := Length(Result);
    iNameLen := Length(sName);
    iBegPos := 0;
    repeat
      iBegPos := ShortLoStrPos(sName, Result, succ(iBegPos));
      if (iBegPos = 1) or ((iBegPos > 1) and (Result[pred(iBegPos)] = LF)) then begin
	iEndPos := ShortLoStrPos(LF, Result, iBegPos + iNameLen);
	while (iEndPos < iFullLen) and (Result[succ(iEndPos)] in [SP, TAB]) do begin
	  iEndPos := FirstShortPos(LF, Result, succ(iEndPos));
	end;
	if iEndPos = 0 then iEndPos := iFullLen;
	if iEndPos > iBegPos then begin
	  Delete(Result, iBegPos, succ(iEndPos - iBegPos));
	  Break;
	end;
      end;
    until iBegPos = 0;
  until iCommaPos = 0;
end;

function GetHeaderData(const HeaderName, Headers: String): String;
var
  iFullLen, iHdrLen, iBegPos, iEndPos: Integer;
begin
  iFullLen := Length(Headers);
  iHdrLen := Length(HeaderName);
  iBegPos := 0;
  Result := E;
  repeat
    iBegPos := ShortLoStrPos(HeaderName, Headers, succ(iBegPos));
    if (iBegPos = 1) or ((iBegPos > 1) and (Headers[pred(iBegPos)] = LF)) then begin
      iEndPos := ShortLoStrPos(LF, Headers, iBegPos + iHdrLen);
      while (iEndPos < iFullLen) and (Headers[succ(iEndPos)] in [SP, TAB]) do begin
	iEndPos := FirstShortPos(LF, Headers, succ(iEndPos));
      end;
      if iEndPos = 0 then iEndPos := iFullLen;
      if iEndPos > iBegPos then begin
	inc(iEndPos);
	inc(iBegPos, iHdrLen);
	Result := Trim(Copy(Headers, iBegPos, iEndPos - iBegPos));
      end;
    end;
  until (Result <> E) or (iBegPos = 0);
end;

function GetApplicationType(const HeaderData: String; var AppTypePos: Integer): String;
var
  iBegPos, iEndPos: Integer;
begin
  Result := E;
  iBegPos := FirstLoStrPos(APPDATA, HeaderData);
  if iBegPos > 0 then begin
    inc(iBegPos, Length(APPDATA));
    iEndPos := FirstShortPos(SC, HeaderData, iBegPos);
    if iEndPos = 0 then iEndPos := Length(HeaderData);
    Result := Trim(Copy(HeaderData, iBegPos, iEndPos - iBegPos));
    iEndPos := pos(QU, Result);
    if iEndPos > 0 then begin
      dec(iEndPos);
      while (iEndPos > 0) and (Result[iEndPos] <= SP) do dec(iEndPos);
      if iEndPos > 0 then
	Delete(Result, succ(iEndPos), MAXINT)
      else Result := E;
    end;
  end;
  AppTypePos := iBegPos;
end;

function GetBoundary(const HeaderData: String): String;
var
  iBegPos, iEndPos, iQUPos: Integer;
begin
  Result := E;
  iBegPos := ShortLoStrPos(BOUNDS, HeaderData, succ(Length(MULTIPART)));
  if iBegPos > 0 then begin
    iBegPos := FirstShortPos(EQ, HeaderData, iBegPos + Length(BOUNDS));
    if iBegPos > 0 then begin
      inc(iBegPos);
      iEndPos := FirstShortPos(SC, HeaderData, succ(iBegPos));
      if iEndPos <> 0 then
	dec(iEndPos)
      else iEndPos := Length(HeaderData);
      if iEndPos > iBegPos then begin
	iQUPos := iBegPos;
	while (iQUPos < iEndPos) and (HeaderData[iQUPos] <> QU) do inc(iQUPos);
	if iQUPos < iEndPos then begin
	  while (iEndPos > iQUPos) and (HeaderData[iEndPos] <> QU) do dec(iEndPos);
	  if iQUPos < iEndPos then begin
	    iBegPos := succ(iQUPos);
	    dec(iEndPos);
	  end;
	end;
	while (iBegPos < iEndPos) and (HeaderData[iBegPos] <= SP) do inc(iBegPos);
	while (iEndPos > iBegPos) and (HeaderData[iEndPos] <= SP) do dec(iEndPos);
	if iEndPos > iBegPos then Result := '--' + Copy(HeaderData, iBegPos, succ(iEndPos - iBegPos));
      end;
    end;
  end;
end;

function FindValidBoundary(const Boundary, Data: String; StartPos: Integer): Integer;
begin
  Result := ShortStrPos(Boundary, Data, StartPos);
  while (Result > 1) and (Data[pred(Result)] <> LF) do begin
    Result := ShortStrPos(Boundary, Data, Result + Length(Boundary));
  end;
end;

function GetMimePart(const Boundary: String; var MimeBody: String): String;
var
  iBegPos, iEndPos: Integer;
begin
  Result := E;
  iBegPos := FindValidBoundary(Boundary, MimeBody, 1);
  if iBegPos > 0 then begin
    iEndPos := FindValidBoundary(Boundary, MimeBody, iBegPos + Length(Boundary));
    if iEndPos > iBegPos then begin
      dec(iEndPos, Length(CRLF));
      inc(iBegPos, Length(Boundary) + Length(CRLF));
      Result := Copy(MimeBody, iBegPos, iEndPos - iBegPos);
      Delete(MimeBody, 1, iEndPos + pred(Length(CRLF)));
    end;
  end;
end;

function DecodeMimePart(const MimePart: String): String;
var
  sHeaders: String;
begin
  Result := MimePart;
  sHeaders := ExtractMimeHeaders(Result, false);
  if LowerCase(GetHeaderData(CTE_HEADER, sHeaders)) = BASE64 then begin
    Result := MimeDecodeString(Trim(Result));
  end;
end;

function GetPGPData(const BodyPart: String): String;
var
  iBegPos, iEndPos: Integer;
begin
  Result := E;
  if FastStrPos(PGP_BEG, BodyPart, 1) then
    iBegPos := 1
  else begin
    iBegPos := FirstStrPos(LF + PGP_BEG, BodyPart);
    if iBegPos > 0 then inc(iBegPos);
  end;
  if iBegPos > 0 then begin
    iEndPos := ShortStrPos(LF + PGP_END, BodyPart, iBegPos + Length(PGP_BEG));
    if iEndPos > iBegPos then begin
      inc(iEndPos, Length(PGP_END) + succ(Length(CRLF)));
      Result := Copy(BodyPart, iBegPos, iEndPos - iBegPos);
    end;
  end;
end;

function WrapHeaderLine(const Header: String): String;
var
  iLen, iResPos, iBegPos, iEndPos, iCopyLen: Integer;
  pResult: PChar;
begin
  iLen := Length(Header);
  if (iLen > MAX_HEAD_LEN) and (FirstStrPos(CRLF, Header) = 0) then begin
    SetLength(Result, iLen + (GetCharCount(SC, Header) * (Length(CRLF) + Length(SP))));
    pResult := Pointer(Result);
    iResPos := 0;
    iBegPos := 1;
    repeat
      iEndPos := FirstShortPos(SC, Header, iBegPos);
      iCopyLen := succ(iEndPos - iBegPos);
      if (iCopyLen > 0) and (iCopyLen + (iLen - iEndPos) > MAX_HEAD_LEN) then begin
	Move(Header[iBegPos], pResult[iResPos], iCopyLen);
	inc(iResPos, iCopyLen);
	pResult[iResPos] := CR;
	inc(iResPos);
	pResult[iResPos] := LF;
	inc(iResPos);
	iBegPos := succ(iEndPos);
	if (iBegPos <= iLen) and not (Header[iBegPos] in [TAB, SP]) then begin
	  pResult[iResPos] := SP;
	  inc(iResPos);
	end;
	Continue;
      end
      else begin
	iCopyLen := succ(iLen - iBegPos);
	if iCopyLen > 0 then begin
	  Move(Header[iBegPos], pResult[iResPos], iCopyLen);
	  inc(iResPos, iCopyLen);
	end;
	Break;
      end;
    until iBegPos > iLen;
    Delete(Result, succ(iResPos), MAXINT);
  end
  else Result := Header;
end;

function GetPSMimeType(const MimeMessage: String; var Headers, Body, Signature: String; AnalyseOnly: Longbool): TPSMime;
var
  sMimeParts, sCTHeader, sAppType, sBoundary: String;
  iAppTypePos, iDataPos: Integer;
begin
  Result := psNone;
  Signature := E;
  Body := AdjustLineBreaks(MimeMessage);
  Headers := ExtractMimeHeaders(Body, AnalyseOnly);
  sCTHeader := GetHeaderData(CT_HEADER, Headers);
  sAppType := GetApplicationType(sCTHeader, iAppTypePos);
  if FirstLoStrPos(PKCS7, sAppType) > 0 then begin
    if FirstLoStrPos(MULTIPART + SM_SIG, sCTHeader) > 0 then begin
      if not AnalyseOnly then begin
	sBoundary := GetBoundary(sCTHeader);
	if sBoundary <> E then begin
	  sMimeParts := Body;
	  Headers := DeleteHeaders(CT_HEADER + CM + CTE_HEADER + CM + CD_HEADER, Headers);
	  Body := GetMimePart(sBoundary, sMimeParts);
	  Signature := DecodeMimePart(GetMimePart(sBoundary, sMimeParts));
	end;
      end;
      Result := psClearSigned;
    end
    else begin
      iDataPos := ShortLoStrPos(SM_TYPE, sCTHeader, iAppTypePos + succ(Length(PKCS7 + MIME)));
      if iDataPos > 0 then begin
	inc(iDataPos, succ(Length(SM_TYPE)));
	if (ShortLoStrPos(SM_ENC, sCTHeader, iDataPos) > 0) then begin
	  if not AnalyseOnly then begin
	    Headers := DeleteHeaders(CT_HEADER + CM + CTE_HEADER + CM + CD_HEADER, Headers);
	    Body := DecodeMimePart(MimeMessage);
	  end;
	  Result := psEncrypted;
	end
	else if (ShortLoStrPos(SM_SIG, sCTHeader, iDataPos) > 0)
	or (ShortLoStrPos(SM_CERT, sCTHeader, iDataPos) > 0) then begin
	  if not AnalyseOnly then begin
	    Headers := DeleteHeaders(CT_HEADER + CM + CTE_HEADER + CM + CD_HEADER, Headers);
	    Body := DecodeMimePart(MimeMessage);
	  end;
	  Result := psSigned;
	end;
      end;
    end;
  end
  else if FirstLoStrPos(PGP, sAppType) > 0 then begin
    if not AnalyseOnly then begin
      if FirstLoStrPos(PM_SIG, sAppType) > 0 then begin
	sBoundary := GetBoundary(sCTHeader);
	if sBoundary <> E then begin
	  Headers := DeleteHeaders(CT_HEADER + CM + CTE_HEADER + CM + CD_HEADER, Headers);
	  Body := WrapHeaderLine(CT_HEADER + SP + sCTHeader) + PARA + Body;
	end;
      end
      else if (FirstLoStrPos(PM_ENC, sAppType) > 0) or (FirstLoStrPos(PM_KEYS, sAppType) > 0) then begin
	Headers := DeleteHeaders(CT_HEADER + CM + CTE_HEADER + CM + CD_HEADER, Headers);
	Body := GetPGPData(Body);
      end;
    end;
    Result := psPGP;
  end;
end;

function SMimeBuild(const MimeMessage, Signature: String; var Separator: String; var BodyOffset: Longint;
		    HashAlgorithm: THashAlgorithm; Encrypted, OmitMimeVersion: Longbool): String;
var
  sMimeVersion, sBoundary, sMainHeaders, sAlgorithm, sSigHeaders: String;
begin
  if OmitMimeVersion then
    sMimeVersion := E
  else sMimeVersion := 'Mime-Version: 1.0' + CRLF;
  if Encrypted then begin
    sBoundary := E;
    sMainHeaders := sMimeVersion +
		    CT_HEADER + SP + APPDATA + PKCS7 + MIME + SC + CRLF +
		    SP + SM_TYPE + EQ + SM_ENC + DATA + CRLF +
		    SP + 'name=SMime.p7m' + CRLF +
		    CTE_HEADER + SP + BASE64 + CRLF +
		    CD_HEADER + SP + 'attachment; filename=SMime.p7m';
    Result := sMainHeaders + PARA + MimeEncodeString(MimeMessage) + CRLF;
  end
  else begin
    sBoundary := 'SMime=Boundary-' + IntToHex(RNG.Val, 8);
    case HashAlgorithm of
      HashAlgorithm_MD5: sAlgorithm := 'micalg=md5';
      HashAlgorithm_SHA: sAlgorithm := 'micalg=sha1';
    else
      sAlgorithm := E;
    end;
    sMainHeaders:= sMimeVersion +
		   CT_HEADER + SP + MULTIPART + SM_SIG + SC + SP +
		   'protocol=' + QU + APPDATA + 'x-' + PKCS7 + PM_SIG + QU + SC + CRLF +
		   SP + sAlgorithm + SC + SP + BOUNDS + EQ + QU + sBoundary + QU;
    sSigHeaders := CT_HEADER + SP + QU + APPDATA + 'x-' + PKCS7 + PM_SIG + QU + SC + SP + 'name=SMime.p7s' + CRLF +
		   CTE_HEADER + SP + BASE64 + CRLF + CD_HEADER + SP + 'attachment; filename=SMime.p7s';
    Result := sMainHeaders + PARA + DD + sBoundary + CRLF +
	      MimeMessage + CRLF + DD + sBoundary + CRLF +
	      sSigHeaders + PARA + MimeEncodeString(Signature) + CRLF + DD + sBoundary + DD + CRLF;
  end;
  BodyOffset := Length(sMainHeaders) + pred(Length(PARA));
  Separator := sBoundary;
end;

function PGPMimeBuild(const MimeMessage: String; var Separator: String; var BodyOffset: Longint;
		      HashAlgorithm: THashAlgorithm; Encrypted, OmitMimeVersion: Longbool): String;
var
  iSepLen, iMsgLen, iBoundsLen, iHeaderLen, iCopyLen: Integer;
  iDestPos, iSourcePos, iBoundsPos: Integer;
  sMimeVersion, sBoundary, sMainHeaders, sAlgorithm: String;
  pResult: PChar;
begin
  iSepLen := Length(Separator);
  iMsgLen := Length(MimeMessage);
  if OmitMimeVersion then
    sMimeVersion := E
  else sMimeVersion := 'Mime-Version: 1.0' + CRLF;
  sBoundary := 'PGPMime=Boundary-' + IntToHex(RNG.Val, 8);
  iBoundsLen := Length(sBoundary);
  if Encrypted then begin
    sMainHeaders := sMimeVersion +
		    CT_HEADER + SP + MULTIPART + PM_ENC + SC + SP +
		    'protocol=' + QU + APPDATA + PGP + PM_ENC + QU + SC + CRLF +
		    SP + BOUNDS + EQ + QU + sBoundary + QU + PARA;
  end
  else begin
    case HashAlgorithm of
      HashAlgorithm_MD5: sAlgorithm := 'micalg=md5';
      HashAlgorithm_SHA: sAlgorithm := 'micalg=sha1';
      HashAlgorithm_RIPEMD160: sAlgorithm := 'micalg=ripemd160';
    else
      sAlgorithm := E;
    end;
    sMainHeaders := sMimeVersion +
		    CT_HEADER + SP + MULTIPART + SM_SIG + SC + SP +
		    'protocol=' + QU + APPDATA + PGP + PM_SIG + QU + SC + CRLF +
		    SP + sAlgorithm + SC + SP + BOUNDS + EQ + QU + sBoundary + QU + PARA;
  end;
  SetLength(Result, iMsgLen + (iBoundsLen shl 2));
  iHeaderLen := Length(sMainHeaders);
  pResult := Pointer(Result);
  iDestPos := 0;
  Move(Pointer(sMainHeaders)^, pResult[iDestPos], iHeaderLen);
  inc(iDestPos, iHeaderLen);
  iSourcePos := succ(BodyOffset);
  BodyOffset := 0;
  repeat
    iBoundsPos := ShortStrPos(Separator, MimeMessage, iSourcePos);
    if iBoundsPos > 0 then begin
      iCopyLen := iBoundsPos - iSourcePos;
      Move(MimeMessage[iSourcePos], pResult[iDestPos], iCopyLen);
      if BodyOffset = 0 then BodyOffset := iDestPos;
      inc(iDestPos, iCopyLen);
      Move(Pointer(sBoundary)^, pResult[iDestPos], iBoundsLen);
      inc(iDestPos, iBoundsLen);
      inc(iSourcePos, iCopyLen + iSepLen);
    end;
  until iBoundsPos = 0;
  pResult[iDestPos] := DASH;
  inc(iDestPos);
  pResult[iDestPos] := DASH;
  inc(iDestPos);
  pResult[iDestPos] := CR;
  inc(iDestPos);
  pResult[iDestPos] := LF;
  inc(iDestPos);
  Delete(Result, succ(iDestPos), MAXINT);
  Separator := sBoundary;
end;

procedure RNGInit;
var
  stLocal: TSystemTime;
  ptCursor: TPoint;
begin
  GetLocalTime(stLocal);
  GetCursorPos(ptCursor);
  with stLocal, ptCursor do begin
    RNG := TIsaac.Create;
    RNG.Seed([MakeLong(wMonth, wSecond), MakeLong(wHour, wDay),
	      MakeLong(wMilliseconds, wYear), MakeLong(wDayOfWeek, wMinute),
	      MakeLong(X, Y), GetMessageTime, GetMessagePos, GetTickCount], true);
  end;
end;

procedure RNGFinit;
begin
  RNG.Free;
end;

initialization
  RNGInit;

finalization
  RNGFinit;

end.

