{$J+,Z4}
unit PGPEncode;

{------------------------------------------------------------------------------}
{                                                                              }
{                 This unit is partly based on Steve Heller's                  }
{         SPGP sources available from http://www.oz.net/~srheller/spgp/        }
{                                                                              }
{                Portions created by Michael in der Wiesche are                }
{              Copyright (C) 2001-2006 by Michael in der Wiesche               }
{                                                                              }
{------------------------------------------------------------------------------}

interface

uses
  Windows,
  Classes,
  SysUtils,
  PSMimeTools,
  KeyPropTypes,
  UTF8,
  pgpBase,
  pgpErrors,
  pgpPubTypes,
  pgpUtilities,
  pgpOptionList,
  pgpMemoryMgr,
  pgpEvents,
  pgpKeys,
  pgpTLS,
  KeyFuncs,
  PrefFuncs,
  X509Funcs,
  PGPDialogs;

type
  TSignAlgorithm = (
    HashAlgorithm_Default,
    HashAlgorithm_MD5,
    HashAlgorithm_SHA,
    HashAlgorithm_RIPEMD160,
    HashAlgorithm_SHA256,
    HashAlgorithm_SHA384,
    HashAlgorithm_SHA512
  );
  TConventionalAlgorithm = (
    CipherAlgorithm_IDEA,
    CipherAlgorithm_3DES,
    CipherAlgorithm_CAST5,
    CipherAlgorithm_AES128,
    CipherAlgorithm_AES192,
    CipherAlgorithm_AES256,
    CipherAlgorithm_Twofish256
  );
  TFormatOption = (
    Format_Armor,
    Format_MIME,
    Format_Textmode
  );
  TFormatOptions = Set of TFormatOption;
  TOnGetInputFileName = procedure(var SuggestedName: String) of Object;
  TOnGetOutputFileName = procedure(var SuggestedName: String) of Object;
  TOnEnterPassphrase = procedure(const Passphrase: PChar;
				 const SigningKeyList: TKeyPropsList;
				 var SelectedKey: Longint;
				 BadPassphrase: Longbool;
				 var Cancel: Longbool) of Object;
  TOnShowProgress = procedure(BytesProcessed, BytesTotal: Longint) of Object;
  TOnWipePassphrase = procedure(const Passphrase: PChar) of Object;
  TPGPEncodeCustom = class(TComponent)
  private
    // internal
    FContext: pPGPContext;
    FKeySetMain: pPGPKeySet;
    FPreferences: TPreferenceRec;
    FAllocatedOutputBuffer: PChar;
    FActualOutputSize: PGPSize;
    FInputSize: PGPSize;
    FInputBuffer: PChar;
    FInputFileName: String;
    FPassphrase: PChar;
    FEncryptKeySet: pPGPKeySet;
    FSigningKey: pPGPKey;
    FClear: Longbool;
    FCompress: Longbool;
    FConventional: Longbool;
    FDetachedSign: Longbool;
    FEncrypt: Longbool;
    FFileToFile: Longbool;
    FSMime: Longbool;
    FSign: Longbool;
    // protected properties
    FOutputBuffer: String;
    // public properties
    FMimeBodyOffset: Longint;
    FMimeSeparator: String;
    FParentHandle: THandle;
    // published properties
    FComment: String;
    FConventionalAlgorithm: TConventionalAlgorithm;
    FEncryptToSelf: Longbool;
    FEyesOnly: Longbool;
    FFileOutput: Longbool;
    FFormatOptions: TFormatOptions;
    FOmitMimeVersion: Longbool;
    FPassCacheSeconds: TPassCacheSeconds;
    FPassCacheShare : Longbool;
    FProgressInterval: Cardinal;
    FSignAlgorithm: TSignAlgorithm;
    FEncryptKeyIDs: TStrings;
    FSignKeyID: String;
    FSignCertUserName: String;
    FKeyDlgPrompt: String;
    FPassDlgPrompt: String;
    FOutputFileName: String;
    // events
    FOnGetInputFileName: TOnGetInputFileName;
    FOnGetOutputFileName: TOnGetOutputFileName;
    FOnEnterPassphrase: TOnEnterPassphrase;
    FOnShowProgress: TOnShowProgress;
    FOnWipePassphrase: TOnWipePassphrase;
    procedure SetEncryptKeyIDs(const Value: TStrings);
    procedure SetClear(Value: Longbool);
    procedure SetConventional(Value: Longbool);
    procedure SetEncrypt(Value: Longbool);
    procedure SetSign(Value: Longbool);
    function  InitEncode: PGPError;
    procedure FinitEncode;
    function  AddSignOptions(const OptionList: pPGPOptionList): PGPError;
    function  GetEncryptKeyIDs(const EncryptKeyIDs: TStrings; SignKeyID: String): PGPError;
    function  GetOptionList(var OptionList: pPGPOptionList): PGPError;
    function  ReadInputFile: PGPError;
    function  SetInputOption(IsFile: Longbool; var OptionList: pPGPOptionList): PGPError;
    function  SetOutputOption(var OptionList: pPGPOptionList): PGPError;
    function  WriteOutputFile: PGPError;
    function  Encode(IsFile: Longbool): Longint;
  protected
    property OutputBuffer: String
      read FOutputBuffer;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function KeyEncryptBuffer(const DataBuffer: String; Sign, SMime: Longbool): Longint; virtual;
    function KeyEncryptFile(const FileName: String; Sign, SMime: Longbool): Longint; virtual;
    function ConventionalEncryptBuffer(const DataBuffer: String): Longint; virtual;
    function ConventionalEncryptFile(const FileName: String): Longint; virtual;
    function ClearSignBuffer(const DataBuffer: String; SMime: Longbool): Longint; virtual;
    function ClearSignFile(const FileName: String; SMime: Longbool): Longint; virtual;
    function ArmorBuffer(const DataBuffer: String): Longint; virtual;
    function ArmorFile(const FileName: String): Longint; virtual;
    function DetachedSignBuffer(const DataBuffer: String): Longint; virtual;
    function DetachedSignFile(const FileName: String): Longint; virtual;
    property MimeBodyOffset: Longint
      read FMimeBodyOffset;
    property MimeSeparator: String
      read FMimeSeparator;
    property ParentHandle: THandle
      read FParentHandle
      write FParentHandle;
  published
    property Comment: String
      read FComment
      write FComment;
    property ConventionalAlgorithm: TConventionalAlgorithm
      read FConventionalAlgorithm
      write FConventionalAlgorithm;
    property EncryptToSelf: Longbool
      read FEncryptToSelf
      write FEncryptToSelf;
    property EyesOnly: Longbool
      read FEyesOnly
      write FEyesOnly;
    property FileOutput: Longbool
      read FFileOutput
      write FFileOutput;
    property FormatOptions: TFormatOptions
      read FFormatOptions
      write FFormatOptions;
    property OmitMimeVersion: Longbool
      read FOmitMimeVersion
      write FOmitMimeVersion;
    property PassCacheSeconds: TPassCacheSeconds
      read FPassCacheSeconds
      write FPassCacheSeconds;
    property PassCacheShare : Longbool
      read FPassCacheShare
      write FPassCacheShare;
    property ProgressInterval: Cardinal
      read FProgressInterval
      write FProgressInterval;
    property SignAlgorithm: TSignAlgorithm
      read FSignAlgorithm
      write FSignAlgorithm;
    property EncryptKeyIDs: TStrings
      read FEncryptKeyIDs
      write SetEncryptKeyIDs;
    property SignKeyID: String
      read FSignKeyID
      write FSignKeyID;
    property SignCertUserName: String
      read FSignCertUserName
      write FSignCertUserName;
    property KeyDlgPrompt: String
      read FKeyDlgPrompt
      write FKeyDlgPrompt;
    property PassDlgPrompt: String
      read FPassDlgPrompt
      write FPassDlgPrompt;
    property OutputFileName: String
      read FOutputFileName
      write FOutputFileName;
    property OnGetInputFileName: TOnGetInputFileName
      read FOnGetInputFileName
      write FOnGetInputFileName;
    property OnGetOutputFileName: TOnGetOutputFileName
      read FOnGetOutputFileName
      write FOnGetOutputFileName;
    property OnEnterPassphrase: TOnEnterPassphrase
      read FOnEnterPassphrase
      write FOnEnterPassphrase;
    property OnShowProgress: TOnShowProgress
      read FOnShowProgress
      write FOnShowProgress;
    property OnWipePassphrase: TOnWipePassphrase
      read FOnWipePassphrase
      write FOnWipePassphrase;
  end;

implementation

function EventHandler(Context: pPGPContext; Event: pPGPEvent; UserValue: PGPUserValue): PGPError; cdecl;
begin
  Result := 0;
  with TPGPEncodeCustom(UserValue) do begin
    case Event^.EType of
      kPGPEvent_NullEvent:	if not FClear and (TMethod(FOnShowProgress).Code <> nil) then begin
				    // BytesTotal always stays 0 => use FInputSize (at least in 6.5.X)
				  with Event^.EData.NullData do FOnShowProgress(BytesWritten, FInputSize);
				  ProcessMessages;
				end;
      kPGPEvent_InitialEvent:	;
      kPGPEvent_FinalEvent:	;
      kPGPEvent_ErrorEvent:	Result := Event^.EData.ErrorData.Error;
      kPGPEvent_WarningEvent:	;
    end;
  end;
end;

constructor TPGPEncodeCustom.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FEncryptKeyIDs := TStringList.Create;
  FProgressInterval := 1000;
end;

destructor TPGPEncodeCustom.Destroy;
begin
  EncryptKeyIDs.Free;
  inherited Destroy;
end;

procedure TPGPEncodeCustom.SetEncryptKeyIDs(const Value: TStrings);
begin
  FEncryptKeyIDs.Assign(Value);
end;

procedure TPGPEncodeCustom.SetClear(Value: Longbool);
begin
  if Value <> FClear then FClear := Value;
  if FClear then begin
    FSign := true;
    FEncrypt := false;
    FCompress := false;
    FConventional := false;
    FDetachedSign := false;
    Include(FFormatOptions, Format_Armor);
    Include(FFormatOptions, Format_Textmode);
  end;
end;

procedure TPGPEncodeCustom.SetConventional(Value: Longbool);
begin
  if Value <> FConventional then FConventional := Value;
  if FConventional then begin
    FDetachedSign := false;
    FEncrypt := false;
    FClear := false;
    FSign := false;
  end;
end;

procedure TPGPEncodeCustom.SetEncrypt(Value: Longbool);
begin
  if Value <> FEncrypt then FEncrypt := Value;
  if FEncrypt then begin
    FDetachedSign := false;
    FConventional := false;
    FClear := false;
  end;
end;

procedure TPGPEncodeCustom.SetSign(Value: Longbool);
begin
  if Value <> FSign then FSign := Value;
  if FSign then FConventional := false;
end;

function TPGPEncodeCustom.InitEncode: PGPError;
begin
  FSigningKey := nil;
  FPassphrase := nil;
  FOutputBuffer := '';
  FActualOutputSize := 0;
  FAllocatedOutputBuffer := nil;
  Result := KeyRings.InitKeyRings(FContext, FKeySetMain);
  GetPreferences(FPreferences, PrefsFlag_GroupsFile or PrefsFlag_DefaultKeyID);
end;

procedure TPGPEncodeCustom.FinitEncode;
begin
  KeyRings.FreeKeyRings;
end;

function GetPGPHashAlgorithm(SignAlgorithm: TSignAlgorithm): PGPHashAlgorithm;
begin
  Result := PGPHashAlgorithm(
    ord(SignAlgorithm) +
    ord(SignAlgorithm > HashAlgorithm_RIPEMD160) * ord(pred(kPGPHashAlgorithm_SHA256 - kPGPHashAlgorithm_RIPEMD160))
  );
end;

function TPGPEncodeCustom.AddSignOptions(const OptionList: pPGPOptionList): PGPError;
var
  SignKeySet	: pPGPKeySet;
  SigningCert	: pPGPKeyDBObj;
  KeyPropsList	: TKeyPropsList;
  Armor		: PGPUInt32;
  MIME		: PGPUInt32;
  Textmode	: PGPUInt32;
  SignKeyHexID	: String;
  SelectedKey	: Longint;
  Cancel	: Longbool;
  BadPassphrase	: Longbool;
  RipeMD	: PGPUInt32;
  SignOption	: TSignOption;
  SMimeOptions	: pPGPOptionList;
begin
  if FSMime then begin
    SignKeySet := nil;
    Result := GetValidX509CertifierSet(FKeySetMain, SignKeySet, false, true);
    if Result <> 0 then Exit;
    SigningCert := nil;
  end
  else SignKeySet := FKeySetMain;
  KeyPropsList := nil;
  try
    Armor := ord(Format_Armor in FFormatOptions);
    MIME := ord(Format_MIME in FFormatOptions);
    Textmode := ord(Format_Textmode in FFormatOptions);
    if Assigned(FOnEnterPassphrase) then begin
      if FindKeyProps('', KeyPropsList, spgpKeyPropFlag_IDComplete,
		      KeyFilterFlag_CanSign, UserID_Ordering) > 0 then begin
	Result := GetHexIDByAnyID(FContext, SignKeySet, FSignKeyID, SignKeyHexID);
	if Result <> 1 then with FPreferences do begin
	  if DefaultKeyHexID <> '' then begin
	    FSignKeyID := DefaultKeyHexID;
	    Result := 1;
	  end;
	end
	else FSignKeyID := SignKeyHexID;
	if Result = 1 then with KeyPropsList do begin
	  SelectedKey := IndexOf(FSignKeyID);
	  if SelectedKey > 0 then Move(SelectedKey, 0);
	end;
      end
      else begin
	Result := kPGPError_SecretKeyNotFound;
	Exit;
      end;
      if (GetKeyByHexID(SignKeySet, PChar(FSignKeyID), FSigningKey) <> 0) or (FPassCacheSeconds = 0)
      or not GetCachedPassphrase(FContext, FSigningKey, nil, FPassphrase) then begin
	FPassphrase := PGPNewSecureData(PGPGetDefaultMemoryMgr, 256 * UTF8Factor, kPGPMemoryMgrFlags_Clear);
	if FPassphrase <> nil then begin
	  SelectedKey := 0;
	  Cancel := false;
	  BadPassphrase := false;
	  Result := kPGPError_UserAbort;
	  repeat
	    if FPassphrase[0] <> #0 then FillChar(FPassphrase^, 256 * UTF8Factor, 0);
	    FOnEnterPassphrase(FPassphrase, KeyPropsList, SelectedKey, BadPassphrase, Cancel);
	    if not Cancel then begin
	      if PGP8X then SecureAnsiToUtf8PChar(FPassphrase, FPassphrase, 256 * UTF8Factor);
	      if (SelectedKey >= 0) and (SelectedKey < KeyPropsList.Count) then begin
		FSignKeyID := KeyPropsList[SelectedKey];
		BadPassphrase := not PassphraseIsValid(FContext, SignKeySet, PChar(FSignKeyID), FPassphrase);
	      end
	      else begin
		Result := kPGPError_SecretKeyNotFound;
		Exit;
	      end;
	    end
	    else Exit;
	  until Cancel or not BadPassphrase;
	end
	else begin
	  Result := kPGPError_OutOfMemory;
	  Exit;
	end;
	Result := GetKeyByHexID(SignKeySet, PChar(FSignKeyID), FSigningKey);
	if Result <> 0 then Exit;
      end;
    end
    else begin
      if (FSignKeyID = '')
      or (FindKeyProps(FSignKeyID, KeyPropsList,
		       spgpKeyPropFlag_KeyID,
		       KeyFilterFlag_CanSign,
		       Creation_Ordering) <> 1) then begin
	FSignKeyID := FPreferences.DefaultKeyHexID;
	if (FSignKeyID = '') and (KeyPropsList.Count > 0) then FSignKeyID := KeyPropsList[0];
      end
      else FSignKeyID := KeyPropsList[0];
      if (GetKeyByHexID(SignKeySet, PChar(FSignKeyID), FSigningKey) <> 0) or (FPassCacheSeconds = 0)
      or not GetCachedPassphrase(FContext, FSigningKey, nil, FPassphrase) then begin
	if FSMime then
	  SignOption := soNo
	else SignOption := TSignOption((ord(FClear) and 1) * ord(soClear) + (ord(FDetachedSign) and 1) * ord(soDetached));
	RipeMD := ord(FSignAlgorithm = HashAlgorithm_RIPEMD160);
	if FSMime then begin
	  Result := X509GetCertAndPassphraseDialog(FContext, SignKeySet, FSigningKey, SigningCert, FPassphrase,
						   FSignCertUserName, FSignKeyID, FPassDlgPrompt, FParentHandle);
	end
	else begin
	  Result := SigningPassphraseDialog(FContext, SignKeySet, FSigningKey, FPassphrase, FSignKeyID, true,
					    SignOption, Armor, MIME, RipeMD, FPassDlgPrompt, FParentHandle);
	end;
	if Result <> 0 then Exit;
	if not FEncrypt then begin
	  FFormatOptions := [];
	  if Armor <> 0 then Include(FFormatOptions, Format_Armor);
	  if MIME <> 0 then Include(FFormatOptions, Format_MIME);
	  if Textmode <> 0 then Include(FFormatOptions, Format_Textmode);
	end;
	if FSMime and not (FSignAlgorithm in [HashAlgorithm_MD5, HashAlgorithm_SHA]) then
	  FSignAlgorithm := HashAlgorithm_Default
	else if RipeMD <> 0 then FSignAlgorithm := HashAlgorithm_RIPEMD160;
      end;
    end;
    if FSignAlgorithm = HashAlgorithm_Default then begin
      case GetKeyPropKeyAlg(FSigningKey) of
	KeyAlgorithm_RSA..KeyAlgorithm_RSASignOnly: if GetKeyPropLegacy(FSigningKey) then
	  FSignAlgorithm := HashAlgorithm_MD5
	else FSignAlgorithm := HashAlgorithm_SHA;
	KeyAlgorithm_DH..KeyAlgorithm_DHDSS: FSignAlgorithm := HashAlgorithm_SHA;
      else
	Result := kPGPError_UnknownPublicKeyAlgorithm;
	Exit;
      end;
    end;
    if FSMime then begin
      if SigningCert = nil then SigningCert := GetValidX509CertFromKey(FSignCertUserName, FSigningKey, nil, SignKeySet);
      Result := PGPBuildOptionList(FContext, SMimeOptions,
	[
	  PGPOPassphrase(FContext, FPassphrase),
	  PGPOSMIMESigner(FContext, SigningCert)
	]);
      if Result <> 0 then Exit;
      try
	Result := PGPAppendOptionList(OptionList,
	  [
	    PGPOSignWithKey(FContext, FSigningKey, SMimeOptions, PGPOLastOption(FContext)),
	    PGPOHashAlgorithm(FContext, GetPGPHashAlgorithm(FSignAlgorithm)),
	    PGPODetachedSig(FContext, PGPOLastOption(FContext), nil)
	  ]);
      finally
	PGPFreeOptionList(SMimeOptions);
      end;
    end
    else begin
      Result := PGPAppendOptionList(OptionList,
	[
	  PGPOSignWithKey(FContext, FSigningKey, PGPOPassphrase(FContext, FPassphrase), PGPOLastOption(FContext)),
	  PGPOHashAlgorithm(FContext, GetPGPHashAlgorithm(FSignAlgorithm))
	]);
    end;
  finally
    KeyPropsList.Free;
    if FSMime then PGPFreeKeySet(SignKeySet);
  end;
end;

function TPGPEncodeCustom.GetEncryptKeyIDs(const EncryptKeyIDs: TStrings; SignKeyID: String): PGPError;
var
  RecipientKeys	: Pointer;
  KeyPropsList	: TKeyPropsList;
  RecipientDB	: pPGPKeyDB;
  Armor		: PGPUInt32;
  MIME		: PGPUInt32;
  Textmode	: PGPUInt32;
  KeyCount	: PGPUInt32;
begin
  RecipientKeys := nil;
  with KeyRings do if GroupsFile = '' then GroupsFile := FPreferences.GroupsFile;
  if FEncryptToSelf then begin
    if not FSign then begin
      KeyPropsList := nil;
      try
	if FindKeyProps(SignKeyID, KeyPropsList,
			spgpKeyPropFlag_KeyID or
			spgpKeyPropFlag_CanEncrypt,
			KeyFilterFlag_CanSign,
			Creation_Ordering) <> 1 then begin
	  SignKeyID := FPreferences.DefaultKeyHexID;
	  if (SignKeyID = '') and (KeyPropsList.Count > 0) then SignKeyID := KeyPropsList[0];
	end;
      finally
	KeyPropsList.Free;
      end;
      if SignKeyID = '' then begin
	Result := kPGPError_PublicKeyNotFound;
	Exit;
      end;
    end;
  end
  else SignKeyID := '';
  Result := GetKeySetByAnyIDs(FContext, FKeySetMain, EncryptKeyIDs.CommaText, pPGPKeySet(RecipientKeys));
  if Result <> 0 then begin
    if PGP7X then begin
      RecipientDB := nil;
      try
	EncryptKeyIDs.Clear;
	Armor := ord(Format_Armor in FFormatOptions);
	MIME := ord(Format_MIME in FFormatOptions);
	Textmode := ord(Format_Textmode in FFormatOptions);
	if FSMime then begin
	  Result := GetX509CertKeyDB(FContext, FKeySetMain, RecipientDB);
	  if Result <> 0 then Exit;
	end
	else RecipientDB := PGPPeekKeySetKeyDB(FKeySetMain);
	Result := RecipientsDialog(FContext, PGPPeekKeyDBRootKeySet(RecipientDB), true, FSMime, Armor, Textmode,
				   PGPUInt32(FEyesOnly), MIME, SignKeyID, pPGPKeyDB(RecipientKeys),
				   FKeyDlgPrompt, FParentHandle);
	if Result = 0 then begin
	  FFormatOptions := [];
	  if Armor <> 0 then Include(FFormatOptions, Format_Armor);
	  if MIME <> 0 then Include(FFormatOptions, Format_MIME);
	  if Textmode <> 0 then Include(FFormatOptions, Format_Textmode);
	  KeyPropsList := nil;
	  try
	    KeyCount := GetKeySetProps(FContext, PGPPeekKeyDBRootKeySet(pPGPKeyDB(RecipientKeys)),
				       KeyPropsList, spgpKeyPropFlag_KeyID,
				       KeyFilterFlag_CanEncrypt,
				       UserID_Ordering);
	    if KeyCount > 0 then EncryptKeyIDs.Text := TrimRight(KeyPropsList.Text);
	  finally
	    KeyPropsList.Free;
	  end;
	end;
      finally
	PGPFreeKeyDB(pPGPKeyDB(RecipientKeys));
	if FSMime then PGPFreeKeyDB(RecipientDB);
      end;
    end
    else begin
      try
	EncryptKeyIDs.Clear;
	Armor := ord(Format_Armor in FFormatOptions);
	MIME := ord(Format_MIME in FFormatOptions);
	Textmode := ord(Format_Textmode in FFormatOptions);
	Result := RecipientsDialog(FContext, FKeySetMain, true, FSMime, Armor, PGPUInt32(FEyesOnly), MIME, Textmode,
				   SignKeyID, pPGPKeySet(RecipientKeys), FKeyDlgPrompt, FParentHandle);
	if Result = 0 then begin
	  FFormatOptions := [];
	  if Armor <> 0 then Include(FFormatOptions, Format_Armor);
	  if MIME <> 0 then Include(FFormatOptions, Format_MIME);
	  if Textmode <> 0 then Include(FFormatOptions, Format_Textmode);
	  KeyPropsList := nil;
	  try
	    KeyCount := GetKeySetProps(FContext, pPGPKeySet(RecipientKeys),
				       KeyPropsList, spgpKeyPropFlag_KeyID,
				       KeyFilterFlag_CanEncrypt,
				       UserID_Ordering);
	    if KeyCount > 0 then EncryptKeyIDs.Text := TrimRight(KeyPropsList.Text);
	  finally
	    KeyPropsList.Free;
	  end;
	end;
      finally
	PGPFreeKeySet(pPGPKeySet(RecipientKeys));
      end;
    end;
  end
  else if FEncryptToSelf and (EncryptKeyIDs.Count <> 0) then EncryptKeyIDs.Add(SignKeyID);
end;

function TPGPEncodeCustom.GetOptionList(var OptionList: pPGPOptionList): PGPError;
var
  SelectedKey	: Longint;
  Cancel	: Longbool;
  BadPassphrase	: Longbool;
  KeyPropsList	: TKeyPropsList;
  AlgorithmList	: TPGPCipherAlgorithms;
  AlgCount	: Longint;
begin
  if FSMime then begin
    if FEncrypt then begin
      Result := PGPBuildOptionList(FContext, OptionList,
	[
	  PGPOSendNullEvents(FContext, FProgressInterval),
	  PGPOEventHandler(FContext, EventHandler, Self),
	  PGPODataIsASCII(FContext, PGPTrue),
	  PGPOOutputFormat(FContext, kPGPOutputFormat_SMIMEBodyEncryptedData)
	]);
    end
    else begin
      Result := PGPBuildOptionList(FContext, OptionList,
	[
	  PGPOEventHandler(FContext, EventHandler, Self),
	  PGPODataIsASCII(FContext, PGPTrue),
	  PGPOOutputFormat(FContext, kPGPOutputFormat_SMIMEBodySignedData)
	]);
    end;
  end
  else if FDetachedSign then begin
    Result := PGPBuildOptionList(FContext, OptionList,
      [
	PGPOEventHandler(FContext, EventHandler, Self),
	PGPOArmorOutput(FContext, PGPBoolean(Format_Armor in FFormatOptions)),
	PGPODetachedSig(FContext, PGPOLastOption(FContext), nil),
	PGPOCommentString(FContext, PChar(FComment)),
	PGPOVersionString(FContext, MyVersion)
      ]);
  end
  else begin
    Result := PGPBuildOptionList(FContext, OptionList,
      [
	PGPOSendNullEvents(FContext, FProgressInterval),
	PGPOEventHandler(FContext, EventHandler, Self),
	PGPOClearSign(FContext, PGPBoolean(FClear) and 1),
	PGPOCompression(FContext, PGPBoolean(FCompress) and 1),
	PGPOCommentString(FContext, PChar(FComment)),
	PGPOVersionString(FContext, MyVersion)
      ]);
  end;
  if Result <> 0 then Exit;
  if FConventional then begin
    Result := PGPAppendOptionList(OptionList,
      [
	PGPOCipherAlgorithm(FContext, PGPCipherAlgorithm(succ(FConventionalAlgorithm)))
      ]);
    if Result <> 0 then Exit;
    if Assigned(FOnEnterPassphrase) then begin
      FPassphrase := PGPNewSecureData(PGPGetDefaultMemoryMgr, 256 * UTF8Factor, kPGPMemoryMgrFlags_Clear);
      if FPassphrase <> nil then begin
	SelectedKey := -1;
	Cancel := false;
	BadPassphrase := false;
	Result := kPGPError_UserAbort;
	KeyPropsList := TKeyPropsList.Create(spgpKeyPropFlag_IDComplete);
	try
	  FOnEnterPassphrase(FPassphrase, KeyPropsList, SelectedKey, BadPassphrase, Cancel);
	finally
	  KeyPropsList.Free;
	end;
	if not Cancel then begin
	  if PGP8X then SecureAnsiToUtf8PChar(FPassphrase, FPassphrase, 256 * UTF8Factor);
	end
	else Exit;
      end
      else begin
	Result := kPGPError_OutOfMemory;
	Exit;
      end;
    end
    else begin
      Result := ConvEncPassphraseDialog(FContext, FPassphrase, FPassDlgPrompt, FParentHandle);
      if Result <> 0 then Exit;
    end;
    Result := PGPAppendOptionList(OptionList,
      [
	PGPOConventionalEncrypt(FContext, PGPOPassphrase(FContext, FPassphrase), PGPOLastOption(FContext))
      ]);
    if Result <> 0 then Exit;
  end
  else begin
    if FSign or FDetachedSign then begin
      Result := AddSignOptions(OptionList);
      if Result <> 0 then Exit;
    end;
    if FEncrypt and not (FSMime and FSign) then begin
      if FEncryptToSelf and not GetKeyPropCanEncrypt(FSigningKey) then
	Result := GetEncryptKeyIDs(FEncryptKeyIDs, '')
      else Result := GetEncryptKeyIDs(FEncryptKeyIDs, GetKeyPropKeyID(FSigningKey));
      if Result <> 0 then Exit;
      Result := GetKeySetByAnyIDs(FContext, FKeySetMain, FEncryptKeyIDs.CommaText, FEncryptKeySet);
      if Result <> 0 then Exit;
      if FSMime then begin
	Result := CheckValidX509CertifierSet(FEncryptKeySet);
	if Result <> 0 then Exit;
      end;
      Result := PGPAppendOptionList(OptionList, [PGPOEncryptToKeySet(FContext, FEncryptKeySet)]);
      if Result <> 0 then Exit;
      if GetAllowedCipherAlgorithms(AlgorithmList, AlgCount) = 0 then begin
	PGPAppendOptionList(OptionList, [PGPOPreferredAlgorithms(FContext, AlgorithmList, AlgCount)]);
      end;
    end;
  end;
  if not (FDetachedSign or FSMime) then begin
    Result := PGPAppendOptionList(OptionList,
      [
	PGPOForYourEyesOnly(FContext, PGPBoolean(FEyesOnly) and 1),
	PGPOArmorOutput(FContext, PGPBoolean(Format_Armor in FFormatOptions)),
	PGPODataIsASCII(FContext, PGPBoolean(Format_Textmode in FFormatOptions))
      ]);
    if Result <> 0 then Exit;
    if Format_Mime in FFormatOptions then begin
      SetLength(FMimeSeparator, SizeOf(TMimeSeparator));
      FMimeBodyOffset := 0;
      FMimeSeparator[1] := #0;
      Result := PGPAppendOptionList(OptionList,
	[
	  PGPOPGPMIMEEncoding(FContext, PGPTrue, PGPSize(FMimeBodyOffset), @FMimeSeparator[1]),
	  PGPOOmitMIMEVersion(fContext, PGPBoolean(FOmitMimeVersion) and 1)
	]);
      if Result <> 0 then Exit;
    end;
  end;
end;

function TPGPEncodeCustom.ReadInputFile: PGPError;
var
  InputFile	: THandle;
  BytesRead	: DWord;
begin
  FInputBuffer := PGPNewSecureData(PGPGetDefaultMemoryMgr, succ(FInputSize), kPGPMemoryMgrFlags_None);
  if FInputBuffer <> nil then begin
    FInputBuffer[FInputSize] := #0;
    InputFile := CreateFile(PChar(FInputFileName), GENERIC_READ, FILE_SHARE_READ, nil, OPEN_EXISTING,
			    FILE_FLAG_SEQUENTIAL_SCAN, 0);
    if InputFile <> INVALID_HANDLE_VALUE then begin
      try
	if not ReadFile(InputFile, FInputBuffer[0], FInputSize, BytesRead, nil) or (BytesRead <> FInputSize) then
	  Result := kPGPError_ReadFailed
	else Result := kPGPError_NoErr;
      finally
	CloseHandle(InputFile);
      end;
    end
    else Result := kPGPError_CantOpenFile;
  end
  else begin
    Result := kPGPError_OutOfMemory;
    Exit;
  end;
end;

function TPGPEncodeCustom.SetInputOption(IsFile: Longbool; var OptionList: pPGPOptionList): PGPError;
var
  FileHandle	: THandle;
  FileSizeHigh	: DWord;
  InFileSpec	: pPGPFileSpec;
begin
  Result := 0;
  if IsFile then begin
    FileHandle := FileOpen(FInputFileName, fmOpenRead or fmShareDenyNone);
    if FileHandle <> INVALID_HANDLE_VALUE then begin
      FInputSize := GetFileSize(FileHandle, @FileSizeHigh);
      FileClose(FileHandle);
    end
    else FInputSize := $FFFFFFFF;
    if (FInputSize = 0) or (FInputSize = $FFFFFFFF) or (FileSizeHigh <> 0) then begin
      Result := kPGPError_ReadFailed;
      Exit;
    end;
    if not FSMime then begin
      InFileSpec := nil;
      Result := PGPNewFileSpecFromFullPath(FContext, PChar(FInputFileName), InFileSpec);
      if Result <> 0 then Exit;
      try
	Result := PGPAppendOptionList(OptionList, [PGPOInputFile(FContext, InFileSpec)]);
      finally
	PGPFreeFileSpec(InFileSpec);
      end;
    end
    else Result := ReadInputFile;
    if Result <> 0 then Exit;
  end;
  if not IsFile or FSMime then begin
    Result := PGPAppendOptionList(OptionList, [PGPOInputBuffer(FContext, FInputBuffer, FInputSize)]);
  end;
end;

function TPGPEncodeCustom.SetOutputOption(var OptionList: pPGPOptionList): PGPError;
var
  FileSpec	: pPGPFileSpec;
begin
  if FFileOutput then begin
    if (FOutputFileName = '') or FileExists(FOutputFileName) then begin
      if FOutputFileName = '' then begin
	if FDetachedSign then
	  FOutputFileName := FInputFileName + '.sig'
	else begin
	  if FSMime or (Format_Armor in FFormatOptions) then
	    FOutputFileName := FInputFileName + '.asc'
	  else FOutputFileName := FInputFileName + '.pgp';
	end;
      end;
      if Assigned(FOnGetOutputFileName) then begin
	FOnGetOutputFileName(FOutputFileName);
    	DeleteFile(FOutputFileName);
      end
      else FOutputFileName := '';
    end;
    if FOutputFileName = '' then begin
      Result := kPGPError_CantOpenFile;
      Exit;
    end;
  end;
  if FFileToFile then begin
    FileSpec := nil;
    Result := PGPNewFileSpecFromFullPath(FContext, PChar(FOutputFileName), FileSpec);
    try
      if Result = 0 then Result := PGPAppendOptionList(OptionList, [PGPOOutputFile(FContext, FileSpec)]);
    finally
      PGPFreeFileSpec(FileSpec);
    end;
  end
  else begin
    Result := PGPAppendOptionList(OptionList,
      [
	PGPOAllocatedOutputBuffer(FContext, FAllocatedOutputBuffer, $FFFFFFFF, FActualOutputSize)
      ]);
  end;
end;

function TPGPEncodeCustom.WriteOutputFile: PGPError;
var
  OutputFile	: THandle;
  BytesWritten	: DWord;
begin
  OutputFile := CreateFile(PChar(FOutputFileName), GENERIC_WRITE, 0, nil, CREATE_ALWAYS,
			   FILE_ATTRIBUTE_ARCHIVE or FILE_FLAG_WRITE_THROUGH, 0);
  if OutputFile <> INVALID_HANDLE_VALUE then begin
    try
      if not WriteFile(OutputFile, FAllocatedOutputBuffer[0], FActualOutputSize, BytesWritten, nil)
      or (BytesWritten <> DWord(FActualOutputSize)) then
	Result := kPGPError_WriteFailed
      else Result := kPGPError_NoErr;
    finally
      CloseHandle(OutputFile);
    end;
  end
  else Result := kPGPError_CantOpenFile;
end;

function TPGPEncodeCustom.Encode(IsFile: Longbool): Longint;
var
  OptionList	: pPGPOptionList;
begin
  Result := 0;
  if IsFile then begin
    FInputFileName := FInputBuffer;
    if not FileExists(FInputFileName) then begin
      if Assigned(FOnGetInputFileName) then begin
	FOnGetInputFileName(FInputFileName);
	if FInputFileName = '' then Result := kPGPError_FileNotFound;
      end
      else Result := kPGPError_FileNotFound;
    end;
    if PGP7X and not (FSMime or (Format_Mime in FFormatOptions)) then
      FFileToFile := FFileOutput
    else FFileToFile := false;
  end
  else if FInputSize = 0 then Result := kPGPError_ItemNotFound;
  if Result = 0 then begin
    OptionList := nil;
    FEncryptKeySet := nil;
    Result := InitEncode;
    if Result <> 0 then Exit;
    try
      Result := GetOptionList(OptionList);
      try
	if Result <> 0 then Exit;
	Result := SetInputOption(IsFile, OptionList);
	if Result <> 0 then Exit;
	Result := SetOutputOption(OptionList);
	if Result <> 0 then Exit;
	Result := pgpEvents.PGPEncode(FContext, OptionList, PGPOLastOption(FContext));
	if (Result = 0) and (FActualOutputSize <> 0) then begin
	  if FFileOutput and not (FSMime or (Format_Mime in FFormatOptions)) then
	    Result := WriteOutputFile
	  else begin
	    try
	      SetLength(FOutputBuffer, FActualOutputSize);
	      Move(FAllocatedOutputBuffer^, Pointer(FOutputBuffer)^, FActualOutputSize);
	    except
	      on EOutOfMemory do Result := kPGPError_OutOfMemory;
	    end;
	  end;
	end;
      finally
	try
	  SetLength(FMimeSeparator, StrLen(PChar(FMimeSeparator)));
	  PGPFreeOptionList(OptionList);
	  PGPFreeKeySet(FEncryptKeySet);
	  if Assigned(FOnWipePassphrase) then begin
	    if PGP8X then SecureUtf8ToAnsiPChar(FPassphrase, FPassphrase, 256 * UTF8Factor);
	    FOnWipePassphrase(FPassphrase);
	  end;
	  if FPassCacheSeconds <> 0 then CachePassphrase(FContext, FSigningKey, FPassCacheSeconds, FPassCacheShare, FPassphrase);
	finally
	  PGPFreeData(FAllocatedOutputBuffer);
	  FAllocatedOutputBuffer := nil;
	  PGPFreeData(FPassphrase);
	  FPassphrase := nil;
	end;
      end;
    finally
      FinitEncode;
    end;
  end;
end;

function SignToHashAlgorithm(SignAlgorithm: TSignAlgorithm): THashAlgorithm;
begin
  Result := THashAlgorithm(ord(SignAlgorithm) + ord(SignAlgorithm > HashAlgorithm_Default));
end;

function TPGPEncodeCustom.KeyEncryptBuffer(const DataBuffer: String; Sign, SMime: Longbool): Longint;
var
  SMimeSigned: String;
begin
  if SMime and not PGP81 then begin
    Result := kPGPError_FeatureNotAvailable;
    Exit;
  end;
  SetSign(Sign);
  SetEncrypt(true);
  FCompress := true;
  FSMime := SMime;
  if SMime and Sign then FEncrypt := false;
  FInputBuffer := PChar(DataBuffer);
  FInputSize := Length(DataBuffer);
  Result := Encode(false);
  if (Result = 0) and (FSMime or (Format_Mime in FFormatOptions)) then begin
    if FSMime then begin
      if not FEncrypt then begin
	SMimeSigned := SMimeBuild(FInputBuffer, FOutputBuffer, FMimeSeparator, FMimeBodyOffset,
				  SignToHashAlgorithm(FSignAlgorithm), false, true);
	FInputBuffer := PChar(SMimeSigned);
	FInputSize := Length(SMimeSigned);
	FEncrypt := true;
	FSign := false;
	Result := Encode(false);
	if Result <> 0 then Exit;
      end;
      FOutputBuffer := SMimeBuild(FOutputBuffer, '', FMimeSeparator, FMimeBodyOffset,
				  HashAlgorithm_Invalid, true, FOmitMimeVersion);
    end
    else begin
      FOutputBuffer := PGPMimeBuild(FOutputBuffer, FMimeSeparator, FMimeBodyOffset,
				    HashAlgorithm_Invalid, true, FOmitMimeVersion);
    end;
    if FFileOutput then begin
      FAllocatedOutputBuffer := PChar(FOutputBuffer);
      FActualOutputSize := Length(FOutputBuffer);
      Result := WriteOutputFile;
      FOutputBuffer := ''
    end;
  end;
  // zero based FMimeBodyOffset => string based
  if FMimeBodyOffset > 0 then inc(FMimeBodyOffset);
end;

function TPGPEncodeCustom.KeyEncryptFile(const FileName: String; Sign, SMime: Longbool): Longint;
var
  SMimeSigned: String;
begin
  if SMime and not PGP81 then begin
    Result := kPGPError_FeatureNotAvailable;
    Exit;
  end;
  SetSign(Sign);
  SetEncrypt(true);
  FCompress := true;
  FSMime := SMime;
  if SMime and Sign then FEncrypt := false;
  FInputBuffer := PChar(FileName);
  FInputSize := Length(FileName);
  Result := Encode(true);
  if (Result = 0) and (FSMime or (Format_Mime in FFormatOptions)) then begin
    if FSMime then begin
      if not FEncrypt then begin
	SMimeSigned := SMimeBuild(FInputBuffer, FOutputBuffer, FMimeSeparator, FMimeBodyOffset,
				  SignToHashAlgorithm(FSignAlgorithm), false, true);
	FInputBuffer := PChar(SMimeSigned);
	FInputSize := Length(SMimeSigned);
	FEncrypt := true;
	FSign := false;
	Result := Encode(false);
	if Result <> 0 then Exit;
      end;
      FOutputBuffer := SMimeBuild(FOutputBuffer, '', FMimeSeparator, FMimeBodyOffset,
      				  HashAlgorithm_Invalid, true, FOmitMimeVersion);
    end
    else begin
      FOutputBuffer := PGPMimeBuild(FOutputBuffer, FMimeSeparator, FMimeBodyOffset,
				    HashAlgorithm_Invalid, true, FOmitMimeVersion);
    end;
    if FFileOutput then begin
      FAllocatedOutputBuffer := PChar(FOutputBuffer);
      FActualOutputSize := Length(FOutputBuffer);
      Result := WriteOutputFile;
      FOutputBuffer := ''
    end;
  end;
  // zero based FMimeBodyOffset => string based
  if FMimeBodyOffset > 0 then inc(FMimeBodyOffset);
end;

function TPGPEncodeCustom.ConventionalEncryptBuffer(const DataBuffer: String): Longint;
begin
  FCompress := true;
  SetConventional(true);
  FInputBuffer := PChar(DataBuffer);
  FInputSize := Length(DataBuffer);
  Result := Encode(false);
end;

function TPGPEncodeCustom.ConventionalEncryptFile(const FileName: String): Longint;
begin
  FCompress := true;
  SetConventional(true);
  FInputBuffer := PChar(FileName);
  FInputSize := Length(FileName);
  Result := Encode(true);
end;

function TPGPEncodeCustom.ClearSignBuffer(const DataBuffer: String; SMime: Longbool): Longint;
begin
  if SMime and not PGP81 then begin
    Result := kPGPError_FeatureNotAvailable;
    Exit;
  end;
  SetClear(true);
  FSMime := SMime;
  FInputBuffer := PChar(DataBuffer);
  FInputSize := Length(DataBuffer);
  Result := Encode(false);
  if (Result = 0) and (FSMime or (Format_Mime in FFormatOptions)) then begin
    if FSMime then begin
      FOutputBuffer := SMimeBuild(FInputBuffer, FOutputBuffer, FMimeSeparator, FMimeBodyOffset,
				  SignToHashAlgorithm(FSignAlgorithm), false, FOmitMimeVersion);
    end
    else begin
      FOutputBuffer := PGPMimeBuild(FOutputBuffer, FMimeSeparator, FMimeBodyOffset,
				    SignToHashAlgorithm(FSignAlgorithm), false, FOmitMimeVersion);
    end;
    if FFileOutput then begin
      FAllocatedOutputBuffer := PChar(FOutputBuffer);
      FActualOutputSize := Length(FOutputBuffer);
      Result := WriteOutputFile;
      FOutputBuffer := ''
    end;
  end;
  // zero based FMimeBodyOffset => string based
  if FMimeBodyOffset > 0 then inc(FMimeBodyOffset);
end;

function TPGPEncodeCustom.ClearSignFile(const FileName: String; SMime: Longbool): Longint;
begin
  if SMime and not PGP81 then begin
    Result := kPGPError_FeatureNotAvailable;
    Exit;
  end;
  SetClear(true);
  FSMime := SMime;
  FInputBuffer := PChar(FileName);
  FInputSize := Length(FileName);
  Result := Encode(true);
  if (Result = 0) and (FSMime or (Format_Mime in FFormatOptions)) then begin
    if FSMime then begin
      FOutputBuffer := SMimeBuild(FInputBuffer, FOutputBuffer, FMimeSeparator, FMimeBodyOffset,
				  SignToHashAlgorithm(FSignAlgorithm), false, FOmitMimeVersion);
    end
    else begin
      FOutputBuffer := PGPMimeBuild(FOutputBuffer, FMimeSeparator, FMimeBodyOffset,
				    SignToHashAlgorithm(FSignAlgorithm), false, FOmitMimeVersion);
    end;
    if FFileOutput then begin
      FAllocatedOutputBuffer := PChar(FOutputBuffer);
      FActualOutputSize := Length(FOutputBuffer);
      Result := WriteOutputFile;
      FOutputBuffer := ''
    end;
  end;
  // zero based FMimeBodyOffset => string based
  if FMimeBodyOffset > 0 then inc(FMimeBodyOffset);
end;

function TPGPEncodeCustom.ArmorBuffer(const DataBuffer: String): Longint;
begin
  FClear := false;
  FCompress := true;
  FDetachedSign := false;
  FEncrypt := false;
  FSign := false;
  Include(FFormatOptions, Format_Armor);
  FInputBuffer := PChar(DataBuffer);
  FInputSize := Length(DataBuffer);
  Result := Encode(false);
end;

function TPGPEncodeCustom.ArmorFile(const FileName: String): Longint;
begin
  FClear := false;
  FCompress := true;
  FDetachedSign := false;
  FEncrypt := false;
  FSign := false;
  Include(FFormatOptions, Format_Armor);
  FInputBuffer := PChar(FileName);
  FInputSize := Length(FileName);
  Result := Encode(true);
end;

function TPGPEncodeCustom.DetachedSignBuffer(const DataBuffer: String): Longint;
begin
  FClear := false;
  FConventional := false;
  FDetachedSign := true;
  FEncrypt := false;
  FSign := false;
  FInputBuffer := PChar(DataBuffer);
  FInputSize := Length(DataBuffer);
  Result := Encode(false);
end;

function TPGPEncodeCustom.DetachedSignFile(const FileName: String): Longint;
begin
  FClear := false;
  FConventional := false;
  FDetachedSign := true;
  FEncrypt := false;
  FSign := false;
  FInputBuffer := PChar(FileName);
  FInputSize := Length(FileName);
  Result := Encode(true);
end;

end.

