{$J+,Z4}
unit PGPDialogs;

{------------------------------------------------------------------------------}
{                                                                              }
{                 This unit is partly based on Steve Heller's                  }
{         SPGP sources available from http://www.oz.net/~srheller/spgp/        }
{                                                                              }
{                Portions created by Michael in der Wiesche are                }
{              Copyright (C) 2001-2006 by Michael in der Wiesche               }
{                                                                              }
{------------------------------------------------------------------------------}

interface

uses
  Windows,
  Classes,
  SysUtils,
  StrTools,
  KeyPropTypes,
  X509Types,
  UTF8,
  pgpBase,
  pgpErrors,
  pgpPubTypes,
  pgpUtilities,
  pgpOptionList,
  pgpMemoryMgr,
  pgpKeyServer,
  pgpGroups,
  pgpKeys,
  pgpTLS,
  pgpUI,
  pgpCL,
  KeyFuncs,
  PrefFuncs;

type
  TSignOption = (soNo, soEnc, soClear, soDetached);

{ "public" functions }
function PreferencesDialog(PrefsPage, WinHandle: Integer): Integer;
function SelectKeysDialog(const Prompt: String; var KeyPropsList: TKeyPropsList;
			  SingleKey: Longbool; PropertyFlags, FilterFlags, WinHandle: Integer): Integer;
function SendKeyToServerDialog(const Prompt: String; var KeyPropsList: TKeyPropsList;
			       KeyPropFlags, WinHandle: Integer): Integer;
function GetKeyFromServerDialog(const Prompt, KeyData: String; var KeyPropsList: TKeyPropsList;
				KeyPropFlags, WinHandle: Integer): Integer;
function KeyImportDialog(const KeyPrompt, CertPassPrompt, KeyData: String; CertPassphrase: PChar;
			 var KeyPropsList: TKeyPropsList; FromFile: Longbool; IgnoreKnownFlag: Integer;
			 KeyPropFlags, WinHandle: Integer): Integer;
function KeyExportDialog(Prompt: String; const KeyIDs: TStrings; var KeyData: String; var KeyPropsList: TKeyPropsList;
			 ToFile, ExportCompatible, ExportPrivate: Longbool; KeyPropFlags, WinHandle: Integer): Integer;
function KeyRevokeDialog(const KeyHexID, PassPrompt: String; WinHandle: Integer): Integer;
function KeyPassChangeDialog(const KeyHexID, OldPrompt, NewPrompt: String;
			     MinPassLength, MinPassQuality, WinHandle: Integer): Integer;

{ "private" functions }
function SelectX509CertifiedKeysDialog(Context: pPGPContext; KeySetX509CertifiedKeys: pPGPKeySet;
				       var KeySetSelected: pPGPKeySet; var CertSelected: pPGPKeyDBObj;
				       const Prompt: String; CertsOnly, SingleKey: Longbool;
				       WinHandle: Integer): Integer;
function SelectAddKeysToKeyRing(Context: pPGPContext; KeysToAdd: pPGPKeySet; KeySetMain: pPGPKeySet;
				var KeyPropsList: TKeyPropsList; const Prompt: String; SingleKey: Longbool;
				KeyPropFlags, WinHandle: Integer): PGPError;
function RecipientsDialog(Context: pPGPContext; AllKeys: pPGPKeySet; AlwaysDisplayDialog, SMIME: Longbool;
			  var Armor, Textmode, EyesOnly, MIME: PGPUInt32; const DefaultKey: String;
			  var RecipientKeys: Pointer; const Prompt: String; WinHandle: Integer): PGPError;
function KeyServerDialog(Context: pPGPContext; KeySetMain: pPGPKeySet; const KeyData: String;
			 var KeysFound: Pointer; WinHandle: Integer): PGPError;
function CollectRandomDataDialog(Context: pPGPContext; NeededEntropyBits: PGPUInt32; WinHandle: Integer): PGPError;
function PassphraseDialog(Context: pPGPContext; const Prompt: String; var Passphrase: PChar; WinHandle: Integer): Integer;
function ConvEncPassphraseDialog(Context: pPGPContext; var Passphrase: PChar;
				 const Prompt: String; WinHandle: Integer): PGPError;
function ConvDecPassphraseDialog(Context: pPGPContext; var Passphrase: PChar;
				 const Prompt: String; WinHandle: Integer): PGPError;
function KeyPassphraseDialog(Context: pPGPContext; KeySet: pPGPKeySet; var Passphrase: PChar;
			     const Prompt: String; WinHandle: Integer): PGPError;
function SigningPassphraseDialog(Context: pPGPContext; KeySetMain: pPGPKeySet;
				 var SigningKey: pPGPKey; var SignKeyPass: PChar;
				 const DefaultKeyID: String; FindMatchingKey: Longbool; SignOption: TSignOption;
				 var Armor, MIME, RipeMD: PGPUInt32; const Prompt: String; WinHandle: Integer): PGPError;
function DecryptionPassphraseDialog(Context: pPGPContext; RecipientKeys: pPGPKeySet;
				    KeyIDCount: PGPUInt32; const KeyIDList: pPGPKeyID7;
				    var DecryptionKey: pPGPKey; var DecryptKeyPass: PChar;
				    FindMatchingKey: Longbool; const Prompt: String; WinHandle: Integer): PGPError;
function ConfirmationPassphraseDialog(Context: pPGPContext; var Passphrase: PChar;
				      MinPassLength, MinPassQuality: Integer;
				      ShowPassphraseQuality: Longbool;
				      const Prompt: String; WinHandle: Integer): PGPError;

implementation

function PreferencesDialog(PrefsPage, WinHandle: Integer): Integer;
var
  Context	 : pPGPContext;
  KeySetMain	 : pPGPKeySet;
begin
  Result := KeyRings.InitKeyRings(Context, KeySetMain);
  if Result <> 0 then Exit;
  Result := kPGPError_PrefNotFound;
  try
    if PGP9X then begin
      case PrefsPage of
	kPGPclGeneralPrefs	: ;
	kPGPclKeyringPrefs	: PrefsPage := kPGPclKeysPrefs;
	kPGPclEmailPrefs	: PrefsPage := kPGPclMessagingPrefs;
	kPGPclHotkeyPrefs	: PrefsPage := kPGPclHotkeysPrefs;
	kPGPclAdvancedPrefs	: PrefsPage := kPGPclWipingPrefs;
      else
	Exit;
      end;
    end;
    Result := PGPclPreferences(Context, WinHandle, PrefsPage, KeySetMain);
    ProcessMessages;
  finally
    KeyRings.FreeKeyRings;
  end;
end;

function SelectKeysDialog(const Prompt: String; var KeyPropsList: TKeyPropsList;
			  SingleKey: Longbool; PropertyFlags, FilterFlags, WinHandle: Integer): Integer;
var
  Context	 : pPGPContext;
  KeySetMain	 : pPGPKeySet;
  KeyFilter	 : pPGPFilter;
  KeySetFiltered : pPGPKeySet;
  KeySetSelected : pPGPKeySet;
  KeyCount	 : PGPUInt32;
begin
  KeyFilter := nil;
  KeySetFiltered := nil;
  KeySetSelected := nil;
  Result := KeyRings.InitKeyRings(Context, KeySetMain);
  if Result <> 0 then Exit;
  try
    if FilterFlags <> KeyFilterFlag_AllKeys then begin
      Result := GetKeyFilter(Context, FilterFlags, KeyFilter);
      if Result <> 0 then Exit;
      try
	Result := PGPFilterKeySet(KeySetMain, KeyFilter, KeySetFiltered);
      finally
	PGPFreeFilter(KeyFilter);
      end;
      if Result <> 0 then Exit;
    end
    else KeySetFiltered := KeySetMain;
    try
      Result := PGPclSelectKeys(Context, WinHandle, Prompt, KeySetFiltered, KeySetMain,
				(ord(SingleKey) and 1) * PGPCL_SINGLESELECTION, KeySetSelected);
      ProcessMessages;
      if Result <> 0 then Exit;
    finally
      if KeySetFiltered <> KeySetMain then PGPFreeKeySet(KeySetFiltered);
    end;
    try
      Result := PGPCountKeys(KeySetSelected, KeyCount);
      if Result <> 0 then Exit;
      if KeyCount <> 0 then begin
	Result := GetKeySetProps(Context, KeySetSelected, KeyPropsList, PropertyFlags,
				 KeyFilterFlag_AllKeys, UserID_Ordering);
	if Result > 0 then Result := 0;
      end;
    finally
      PGPFreeKeySet(KeySetSelected);
    end;
  finally
    KeyRings.FreeKeyRings;
  end;
end;

function InitKeyServer(Context: pPGPContext; var ServerThreadStorage: pPGPKeyServerThreadStorage;
		       var KeyServerSpecList: pPGPKeyServerSpec; var KeyServerCount: PGPUInt32): PGPError;
var
  KeyServerEntry : pPGPKeyServerSpec;
  KeyServerList	 : TStringList;
  KeyServerIndex : Integer;
begin
  ServerThreadStorage := nil;
  KeyServerSpecList := nil;
  KeyServerCount := 0;
  KeyServerList := nil;
  Result := PGPKeyServerInit;
  if Result <> 0 then Exit;
  Result := PGPKeyServerCreateThreadStorage(ServerThreadStorage);
  if Result <> 0 then Exit;
  try
    Result := GetServerList(KeyServerList);
    if (Result = 0) and (KeyServerList <> nil) then with KeyServerList do begin
      KeyServerSpecList := AllocMem(SizeOf(TPGPKeyServerSpec) * Count);
      if KeyServerSpecList <> nil then begin
	KeyServerEntry := KeyServerSpecList;
	for KeyServerIndex := 0 to pred(Count) do begin
	  Result := PGPNewKeyServer(Context, kPGPKeyServerClass_PGP, KeyServerEntry.Server,
				    PGPONetURL(Context, PChar(Strings[KeyServerIndex])), PGPOLastOption(Context));
	  if Result <> 0 then Exit;
	  inc(KeyServerCount);
	  if PGP7X then
	    KeyServerEntry := ptr(Longint(KeyServerEntry) + KeyServerSpecSize7)
	  else KeyServerEntry := ptr(Longint(KeyServerEntry) + KeyServerSpecSize6);
	end;
      end
      else Result := kPGPError_OutOfMemory;
    end;
  finally
    KeyServerList.Free;
  end;
end;

procedure FreeKeyServer(ServerThreadStorage: pPGPKeyServerThreadStorage;
			KeyServerSpecList: pPGPKeyServerSpec; KeyServerCount: PGPUInt32);
var
  KeyServerEntry : pPGPKeyServerSpec;
begin
  KeyServerEntry := KeyServerSpecList;
  try
    for KeyServerCount := KeyServerCount downto 1 do begin
      PGPFreeKeyServer(KeyServerEntry.Server);
      inc(KeyServerEntry);
    end;
  finally
    if KeyServerSpecList <> nil then FreeMem(KeyServerSpecList);
    PGPKeyServerDisposeThreadStorage(ServerThreadStorage);
    PGPKeyServerCleanup;
  end;
end;

function SendKeyToServerDialog(const Prompt: String; var KeyPropsList: TKeyPropsList;
			       KeyPropFlags, WinHandle: Integer): Integer;
var
  Context	 : pPGPContext;
  KeySetMain	 : pPGPKeySet;
  tlsContext	 : pPGPtlsContext;
  KeySetToSend	 : pPGPKeySet;
  ThreadStorage	 : pPGPKeyServerThreadStorage;
  KeyServerSpecs : pPGPKeyServerSpec;
  KeyServerCount : PGPUInt32;
  KeySetFailed	 : pPGPKeySet;
begin
  KeySetToSend := nil;
  KeySetFailed := nil;
  tlsContext := nil;
  Result := KeyRings.InitKeyRings(Context, KeySetMain);
  if Result <> 0 then Exit;
  try
    Result := PGPclSelectKeys(Context, WinHandle, Prompt, KeySetMain, KeySetMain, 0, KeySetToSend);
    ProcessMessages;
    if Result <> 0 then Exit;
    try
      Result := PGPNewTLSContext(Context, tlsContext);
      if Result <> 0 then Exit;
      try
	Result := InitKeyServer(Context, ThreadStorage, KeyServerSpecs, KeyServerCount);
	if Result <> 0 then Exit;
	try
	  Result := PGPSendToKeyServerDialog(Context, KeyServerSpecs^, tlsContext, KeySetToSend, KeySetFailed,
					     PGPOUIParentWindowHandle(Context, WinHandle), PGPOLastOption(Context));
	  Sleep(100); // trying to prevent a strange cancel crash
	  ProcessMessages;
	finally
	  FreeKeyServer(ThreadStorage, KeyServerSpecs, KeyServerCount);
	end;
	if (KeySetFailed <> nil) and (KeySetToSend <> nil) then PGPRemoveKeys(KeySetFailed, KeySetToSend);
	GetKeySetProps(Context, KeySetToSend, KeyPropsList, KeyPropFlags, KeyFilterFlag_AllKeys, UserID_Ordering);
      finally
	PGPFreeTLSContext(tlsContext);
      end;
    finally
      PGPFreeKeySet(KeySetToSend);
      PGPFreeKeySet(KeySetFailed);
    end;
  finally
    KeyRings.FreeKeyRings;
  end;
end;

function SearchKeyServerOptions(Context: pPGPContext; const KeyData: String; IncludeSubKeys, IncludeUTF8: Longbool;
				WinHandle: Integer; var OptionList: pPGPOptionList): Integer;
var
  KeyFilter	 : pPGPFilter;
begin
  OptionList := nil;
  Result := GetKeyFilterByAnyID(Context, KeyData, true, IncludeUTF8, KeyFilter);
  if Result = 0 then begin
    try
      Result := PGPBuildOptionList(Context, OptionList,
	[
	  PGPOKeyServerSearchFilter(Context, KeyFilter),
	  PGPOUIParentWindowHandle(Context, WinHandle)
	]);
    finally
      PGPFreeFilter(KeyFilter);
    end;
  end;
end;

function GetKeyFromServerDialog(const Prompt, KeyData: String; var KeyPropsList: TKeyPropsList;
				KeyPropFlags, WinHandle: Integer): Integer;
var
  Context	 : pPGPContext;
  KeySetMain	 : pPGPKeySet;
  tlsContext	 : pPGPtlsContext;
  ThreadStorage	 : pPGPKeyServerThreadStorage;
  KeyServerSpecs : pPGPKeyServerSpec;
  KeyServerCount : PGPUInt32;
  OptionList	 : pPGPOptionList;
  KeysFound	 : Pointer;
begin
  tlsContext := nil;
  OptionList := nil;
  KeysFound := nil;
  Result := KeyRings.InitKeyRings(Context, KeySetMain);
  if Result <> 0 then Exit;
  try
    Result := PGPNewTLSContext(Context, tlsContext);
    if Result <> 0 then Exit;
    try
      Result := InitKeyServer(Context, ThreadStorage, KeyServerSpecs, KeyServerCount);
      if Result <> 0 then Exit;
      try
	Result := SearchKeyServerOptions(Context, KeyData, false, false, WinHandle, OptionList);
	if Result <> 0 then Exit;
	try
	  Result := PGPSearchKeyServerDialog(Context, KeyServerCount, KeyServerSpecs, tlsContext, PGPFalse,
					     KeysFound, OptionList, PGPOLastOption(Context));
	  Sleep(100); // trying to prevent a strange cancel crash
	  ProcessMessages;
	finally
	  PGPFreeOptionList(OptionList);
	end;
	if Result <> 0 then Exit;
	if PGP7X then begin
	  try
	    Result := SelectAddKeysToKeyRing(Context, PGPPeekKeyDBRootKeySet(pPGPKeyDB(KeysFound)),
					     KeySetMain, KeyPropsList, Prompt, false, KeyPropFlags, WinHandle);
	  finally
	    PGPFreeKeyDB(pPGPKeyDB(KeysFound));
	  end;
	end
	else begin
	  try
	    Result := SelectAddKeysToKeyRing(Context, pPGPKeySet(KeysFound), KeySetMain,
					     KeyPropsList, Prompt, false, KeyPropFlags, WinHandle);
	  finally
	    PGPFreeKeySet(pPGPKeySet(KeysFound));
	  end;
	end;
      finally
	FreeKeyServer(ThreadStorage, KeyServerSpecs, KeyServerCount);
      end;
    finally
      PGPFreeTLSContext(tlsContext);
    end;
  finally
    KeyRings.FreeKeyRings;
  end;
end;

function KeyImportDialog(const KeyPrompt, CertPassPrompt, KeyData: String; CertPassphrase: PChar;
			 var KeyPropsList: TKeyPropsList; FromFile: Longbool; IgnoreKnownFlag: Integer;
			 KeyPropFlags, WinHandle: Integer): Integer;
type
  TInputFormat = (ifNone, ifPKCS_12, ifPKCS_7, ifPEM, ifKEY);
var
  Context	 : pPGPContext;
  KeySetMain	 : pPGPKeySet;
  FileHandle	 : THandle;
  InputBuffer	 : String;
  InputSize	 : Longint;
  InputFormat	 : TInputFormat;
  OptionList	 : pPGPOptionList;
  PosKeyData	 : Longint;
  LenKeyData	 : Longint;
  KeysToImport	 : Pointer;
  KeySetSelected : pPGPKeySet;
  OptionListCopy : pPGPOptionList;
  KeysToAdd	 : pPGPKeyDB;
  CertsToImport	 : pPGPKeySet;
begin
  Result := KeyRings.InitKeyRings(Context, KeySetMain);
  if Result <> 0 then Exit;
  OptionList := nil;
  KeysToImport := nil;
  KeySetSelected := nil;
  try
    if FromFile then begin
      FileHandle := FileOpen(KeyData, fmOpenReadWrite);
      if FileHandle <> THandle(-1) then begin
	InputSize := FileSeek(FileHandle, 0, 2);
	SetLength(InputBuffer, InputSize);
	FileSeek(FileHandle, 0, 0);
	FileRead(FileHandle, InputBuffer[1], InputSize);
	FileClose(FileHandle);
      end
      else InputSize := 0;
    end
    else begin
      InputSize := Length(KeyData);
      InputBuffer := KeyData;
    end;
    if PGP7X then
      InputFormat := ifNone
    else InputFormat := ifKEY;
    repeat
      PosKeyData := 1;
      LenKeyData := InputSize;
      case InputFormat of
	ifNone: begin
	  InputFormat := ifPKCS_12;
	  Result := PGPBuildOptionList(Context, OptionList, [PGPOInputFormat(Context, kPGPInputFormat_PKCS12)]);
	end;
	ifPKCS_12: begin
	  InputFormat := ifPKCS_7;
	  Result := PGPBuildOptionList(Context, OptionList, [PGPOInputFormat(Context, kPGPInputFormat_X509DataInPKCS7)]);
	end;
	ifPKCS_7 : begin
	  InputFormat := ifPEM;
	  PosKeyData := FirstStrPos(X509Cert_Beg, InputBuffer);
	  if PosKeyData > 0 then
	    LenKeyData := InputSize - pred(PosKeyData)
	  else Continue;
	  Result := PGPBuildOptionList(Context, OptionList, [PGPOInputFormat(Context, kPGPInputFormat_PEMEncodedX509Cert)]);
	end
      else
	InputFormat := ifKEY;
	Result := PGPBuildOptionList(Context, OptionList, [PGPONullOption(Context)]);
      end;
      if Result <> 0 then Exit;
      try
	Result := PGPAppendOptionList(OptionList, [PGPOInputBuffer(Context, @InputBuffer[PosKeyData], LenKeyData)]);
	if Result <> 0 then Exit;
	if PGP7X then begin
	  Result := PGPImport(Context, KeysToImport, OptionList, PGPOLastOption(Context));
	  if Result = kPGPError_BadPassphrase then begin
	    repeat
	      Result := PGPCopyOptionList(OptionList, OptionListCopy);
	      if Result <> 0 then Exit;
	      try
		if CertPassPrompt <> '' then begin
		  CertPassphrase := nil;
		  Result := PassphraseDialog(Context, CertPassPrompt, CertPassphrase, WinHandle);
		  if Result <> 0 then Exit;
		end;
		try
		  Result := PGPAppendOptionList(OptionListCopy, [PGPOPassphrase(Context, CertPassphrase)]);
		  if Result <> 0 then Exit;
		  Result := PGPImport(Context, KeysToImport, OptionListCopy, PGPOLastOption(Context));
		finally
		  if CertPassPrompt <> '' then PGPFreeData(CertPassphrase);
		end;
	      finally
		PGPFreeOptionList(OptionListCopy);
	      end;
	    until (Result <> kPGPError_BadPassphrase) or (CertPassPrompt = '');
	  end
	  else if InputFormat = ifPEM then begin
	    KeySetSelected := PGPPeekKeyDBRootKeySet(pPGPKeyDB(KeysToImport));
	    repeat
	      PosKeyData := ShortStrPos(X509Cert_Beg, InputBuffer, PosKeyData + Length(X509Cert_Beg) + Length(CRLF));
	      if PosKeyData > 0 then begin
		LenKeyData := InputSize - pred(PosKeyData);
		PGPFreeOptionList(OptionList);
		OptionList := nil;
		Result := PGPBuildOptionList(Context, OptionList,
		  [
		    PGPOInputFormat(Context, kPGPInputFormat_PEMEncodedX509Cert),
		    PGPOInputBuffer(Context, @InputBuffer[PosKeyData], LenKeyData)
		  ]);
		if Result <> 0 then Exit;
		KeysToAdd := nil;
		Result := PGPImport(Context, KeysToAdd, OptionList, PGPOLastOption(Context));
		if Result <> 0 then Exit;
		try
		  Result := PGPCopyKeys(PGPPeekKeyDBRootKeySet(KeysToAdd), KeysToImport, KeySetSelected);
		  if Result <> 0 then Exit;
		finally
		  PGPFreeKeyDB(KeysToAdd);
		end;
	      end;
	    until PosKeyData = 0;
	  end;
	end
	else Result := PGPImportKeySet(Context, KeysToImport, OptionList, PGPOLastOption(Context));
	if Result = 0 then begin
	  try
	    if PGP7X then begin
	      if KeySetSelected = nil then KeySetSelected := PGPPeekKeyDBRootKeySet(pPGPKeyDB(KeysToImport));
	      Result := GetExclusiveKeySet(KeySetSelected, KeySetMain, Context, IgnoreKnownFlag);
	      if Result <> 0 then Exit;
	      if KeyPrompt = '' then begin
		Result := AddKeysToKeyRing(Context, KeySetMain, KeySetSelected, KeyPropsList, KeyPropFlags);
		if Result = 0 then
		  Result := kPGPError_InvalidInputFormat
		else if Result > 0 then Result := 0;
	      end
	      else begin
		CertsToImport := nil;
		if InputFormat = ifKEY then
		  CertsToImport := KeySetSelected
		else begin
		  Result :=GetCertSetFromKeySet(KeySetSelected, CertsToImport);
		  if Result <> 0 then Exit;
		end;
		try
		  Result := SelectAddKeysToKeyRing(Context, CertsToImport, KeySetMain, KeyPropsList,
						   KeyPrompt, false, KeyPropFlags, WinHandle);
		finally
		  if InputFormat <> ifKEY then PGPFreeKeySet(CertsToImport);
		end;
	      end;
	    end
	    else begin
	      Result := GetExclusiveKeySet(pPGPKeySet(KeysToImport), KeySetMain, Context, IgnoreKnownFlag);
	      if Result <> 0 then Exit;
	      if KeyPrompt = '' then begin
		Result := AddKeysToKeyRing(Context, KeySetMain, KeysToImport, KeyPropsList, KeyPropFlags);
		if Result > 0 then Result := 0;
	      end
	      else begin
		Result := SelectAddKeysToKeyRing(Context, pPGPKeySet(KeysToImport), KeySetMain,
						 KeyPropsList, KeyPrompt, false, KeyPropFlags, WinHandle);
	      end;
	    end;
	  finally
	    if PGP7X then
	      PGPFreeKeyDB(pPGPKeyDB(KeysToImport))
	    else PGPFreeKeySet(pPGPKeySet(KeysToImport));
	    KeysToImport := nil;
	  end;
	end
	else if (InputFormat < ifKEY) and (Result <> kPGPError_BadPassphrase) then Result := kPGPError_InvalidInputFormat;
      finally
	PGPFreeOptionList(OptionList);
	OptionList := nil;
      end;
    until (Result <> kPGPError_InvalidInputFormat) or (InputFormat = ifKEY);
    if (Result < 0) and (Result >= -40) then Result := kPGPError_InvalidInputFormat;
  finally
    KeyRings.FreeKeyRings;
  end;
end;

function KeyExportDialog(Prompt: String; const KeyIDs: TStrings; var KeyData: String; var KeyPropsList: TKeyPropsList;
			 ToFile, ExportCompatible, ExportPrivate: Longbool; KeyPropFlags, WinHandle: Integer): Integer;
var
  Context	 : pPGPContext;
  KeySetMain	 : pPGPKeySet;
  KeySetSelected : pPGPKeySet;
  OptionList	 : pPGPOptionList;
  FileSpec	 : pPGPFileSpec;
  KeyCount	 : PGPUInt32;
  ExportFormat	 : PGPExportFormat;
  KeyOutput	 : PChar;
  KeyLength	 : PGPUInt32;
begin
  Result := KeyRings.InitKeyRings(Context, KeySetMain);
  if Result <> 0 then Exit;
  KeySetSelected := nil;
  OptionList := nil;
  FileSpec := nil;
  try
    if Prompt = '' then
      Result := GetKeySetByAnyIDs(Context, KeySetMain, KeyIDs.CommaText, KeySetSelected)
    else begin
      Result := PGPclSelectKeys(Context, WinHandle, Prompt, KeySetMain, KeySetMain, 0, KeySetSelected);
      ProcessMessages;
    end;
    if Result <> 0 then Exit;
    try
      Result := PGPCountKeys(KeySetSelected, KeyCount);
      if Result <> 0 then Exit;
      if KeyCount <> 0 then begin
	if ExportCompatible then
	  ExportFormat := kPGPExportFormat_Basic
	else ExportFormat := kPGPExportFormat_Complete;
	Result := PGPBuildOptionList(Context, OptionList,
	  [
	    PGPOVersionString(Context, MyVersion),
	    PGPOExportFormat(Context, ExportFormat),
	    PGPOExportPrivateKeys(Context, PGPBoolean(ExportPrivate) and 1),
	    PGPOExportPrivateSubKeys(Context, PGPBoolean(ExportPrivate) and 1)
	  ]);
	if Result <> 0 then Exit;
	try
	  if ToFile then begin
	    Result := PGPNewFileSpecFromFullPath(Context, PChar(KeyData), FileSpec);
	    if Result <> 0 then Exit;
	    try
	      if Result <> 0 then Exit;
	      Result := PGPAppendOptionList(OptionList, [PGPOOutputFile(Context, FileSpec)]);
	      if Result <> 0 then Exit;
	      if PGP7X then begin
		Result := PGPAppendOptionList(OptionList, [PGPOExportKeySet(Context, KeySetSelected)]);
		if Result = 0 then Result := PGPExport(Context, OptionList, PGPOLastOption(Context));
	      end
	      else Result := PGPExportKeySet(KeySetSelected, OptionList, PGPOLastOption(Context));
	      if Result = 0 then Result := GetKeySetProps(Context, KeySetSelected, KeyPropsList, KeyPropFlags,
							  KeyFilterFlag_AllKeys, UserID_Ordering);
	      if Result > 0 then Result := 0;
	    finally
	      PGPFreeFileSpec(FileSpec);
	    end;
	  end
	  else begin
	    Result := PGPAppendOptionList(OptionList, [PGPOAllocatedOutputBuffer(Context, KeyOutput, $FFFFFFFF, KeyLength)]);
	    if Result <> 0 then Exit;
	    if PGP7X then begin
	      Result := PGPAppendOptionList(OptionList, [PGPOExportKeySet(Context, KeySetSelected)]);
	      if Result = 0 then Result := PGPExport(Context, OptionList, PGPOLastOption(Context));
	    end
	    else Result := PGPExportKeySet(KeySetSelected, OptionList, PGPOLastOption(Context));
	    if (Result = 0) and (KeyLength <> 0) then begin
	      try
		SetLength(KeyData, KeyLength);
		StrPLCopy(PChar(KeyData), KeyOutput, KeyLength);
	      finally
		PGPFreeData(KeyOutput);
	      end;
	    end
	    else KeyData := '';
	    if Result = 0 then Result := GetKeySetProps(Context, KeySetSelected, KeyPropsList, KeyPropFlags,
							KeyFilterFlag_AllKeys, UserID_Ordering);
	    if Result > 0 then Result := 0;
	  end;
	finally
	  PGPFreeOptionList(OptionList);
	end;
      end;
    finally
      PGPFreeKeySet(KeySetSelected);
    end;
  finally
    KeyRings.FreeKeyRings;
  end;
end;

function KeyRevokeDialog(const KeyHexID, PassPrompt: String; WinHandle: Integer): Integer;
var
  Context	 : pPGPContext;
  KeySetMain	 : pPGPKeySet;
  KeySetFound	 : pPGPKeySet;
  Passphrase	 : PChar;
begin
  KeySetFound := nil;
  Passphrase := nil;
  Result := KeyRings.InitKeyRings(Context, KeySetMain);
  if Result <> 0 then Exit;
  try
    Result := GetKeySetByAnyIDs(Context, KeySetMain, KeyHexID, KeySetFound);
    if Result <> 0 then Exit;
    try
      Result := KeyPassphraseDialog(Context, KeySetFound, Passphrase, PassPrompt, WinHandle);
      ProcessMessages;
      if Result <> 0 then Exit;
    finally
      PGPFreeKeySet(KeySetFound);
    end;
    try
      KeyRevoke(PChar(KeyHexID), Passphrase);
    finally
      PGPFreeData(Passphrase);
    end;
  finally
    KeyRings.FreeKeyRings;
  end;
end;

function KeyPassChangeDialog(const KeyHexID, OldPrompt, NewPrompt: String;
			     MinPassLength, MinPassQuality, WinHandle: Integer): Integer;
var
  Context	 : pPGPContext;
  KeySetMain	 : pPGPKeySet;
  KeySetFound	 : pPGPKeySet;
  OldPassphrase	 : PChar;
  NewPassphrase  : PChar;
begin
  KeySetFound := nil;
  OldPassphrase := nil;
  NewPassphrase := nil;
  Result := KeyRings.InitKeyRings(Context, KeySetMain);
  if Result <> 0 then Exit;
  try
    Result := GetKeySetByAnyIDs(Context, KeySetMain, KeyHexID, KeySetFound);
    if Result <> 0 then Exit;
    try
      Result := KeyPassphraseDialog(Context, KeySetFound, OldPassphrase, OldPrompt, WinHandle);
      ProcessMessages;
      if Result <> 0 then Exit;
    finally
      PGPFreeKeySet(KeySetFound);
    end;
    try
      Result := ConfirmationPassphraseDialog(Context, NewPassphrase,
					     MinPassLength, MinPassQuality, true,
					     NewPrompt, WinHandle);
      ProcessMessages;
      if Result <> 0 then Exit;
      try
	ChangePassphrase(PChar(KeyHexID), OldPassphrase, NewPassphrase);
      finally
	PGPFreeData(NewPassphrase);
      end;
    finally
      PGPFreeData(OldPassphrase);
    end;
  finally
    KeyRings.FreeKeyRings;
  end;
end;

function SelectX509CertifiedKeysDialog(Context: pPGPContext; KeySetX509CertifiedKeys: pPGPKeySet;
				       var KeySetSelected: pPGPKeySet; var CertSelected: pPGPKeyDBObj;
				       const Prompt: String; CertsOnly, SingleKey: Longbool;
				       WinHandle: Integer): Integer;
begin
  KeySetSelected := nil;
  if PGP81 and CertsOnly and SingleKey then begin
    Result := PGPclSelectX509Cert(Context, WinHandle, Prompt, KeySetX509CertifiedKeys,
				  kPGPclDefaultX509SelectionFlags, CertSelected);
    if Result = 0 then Result := PGPNewSingletonKeySet(CertSelected, KeySetSelected);
  end
  else begin
    Result := PGPclSelectKeys(Context, WinHandle, Prompt, KeySetX509CertifiedKeys, nil,
			      (ord(SingleKey) and 1) * PGPCL_SINGLESELECTION, KeySetSelected);
  end;
  ProcessMessages;
end;

function SelectAddKeysToKeyRing(Context: pPGPContext; KeysToAdd: pPGPKeySet; KeySetMain: pPGPKeySet;
				var KeyPropsList: TKeyPropsList; const Prompt: String; SingleKey: Longbool;
				KeyPropFlags, WinHandle: Integer): PGPError;
var
  KeyCount	 : PGPUInt32;
  KeyDBTemp	 : pPGPKeyDB;
  KeySetTemp	 : pPGPKeySet;
  KeySetSelected : pPGPKeySet;
  KeySetSelToAdd : pPGPKeySet;
begin
  KeySetSelected := nil;
  KeySetSelToAdd := nil;
  Result := PGPCountKeys(KeysToAdd, KeyCount);
  if Result <> 0 then Exit;
  if KeyCount <> 0 then begin
    if PGP7X then begin
      KeyDBTemp := nil;
      Result := PGPNewKeyDB(Context, KeyDBTemp);
      if Result <> 0 then Exit;
      try
	KeySetTemp := nil;
	Result := PGPCopyKeys(KeySetMain, KeyDBTemp, KeySetTemp);
	if Result <> 0 then Exit;
	PGPFreeKeySet(KeySetTemp);
	Result := PGPCopyKeys(KeysToAdd, KeyDBTemp, KeySetTemp);
	if Result <> 0 then Exit;
	try
	  // KeyDBTemp and KeySetTemp are needed for displaying "web of trust" validities
	  Result := PGPclSelectKeys(Context, WinHandle, Prompt, KeySetTemp, nil,
	  			    (ord(SingleKey) and 1) * PGPCL_SINGLESELECTION, KeySetSelected);
	  ProcessMessages;
	  if Result <> 0 then Exit;
	  try
	    Result := PGPCopyKeys(KeySetSelected, PGPPeekKeySetKeyDB(KeySetMain), KeySetSelToAdd);
	  finally
	    PGPFreeKeySet(KeySetSelected);
	  end;
	finally
	  PGPFreeKeySet(KeySetTemp);
	end;
      finally
	PGPFreeKeyDB(KeyDBTemp);
      end;
    end
    else begin
      Result := PGPclSelectKeys(Context, WinHandle, Prompt, KeysToAdd, KeySetMain,
      				(ord(SingleKey) and 1) * PGPCL_SINGLESELECTION, KeySetSelected);
      ProcessMessages;
      if Result <> 0 then Exit;
      KeySetSelToAdd := KeySetSelected;
      Result := PGPAddKeys(KeySetSelToAdd, KeySetMain);
    end;
    if Result <> 0 then Exit;
    try
      Result := KeyRings.UpdateKeyRings;
      if Result <> 0 then Exit;
      Result := GetKeySetProps(Context, KeySetSelToAdd, KeyPropsList, KeyPropFlags,
			       KeyFilterFlag_AllKeys, UserID_Ordering);
      if Result > 0 then Result := 0;
    finally
      PGPFreeKeySet(KeySetSelToAdd);
    end;
  end
  else Result := kPGPError_PublicKeyNotFound;
end;

function RecipientsDialog(Context: pPGPContext; AllKeys: pPGPKeySet; AlwaysDisplayDialog, SMime: Longbool;
			  var Armor, Textmode, EyesOnly, MIME: PGPUInt32; const DefaultKey: String;
			  var RecipientKeys: Pointer; const Prompt: String; WinHandle: Integer): PGPError;
var
  Utf8Prompt	 : String;
  GroupFileSpec	 : pPGPFileSpec;
  GroupSet	 : pPGPGroupSet;
  GroupOption	 : pPGPOptionList;
  FormOptions	 : pPGPOptionList;
  OptionList	 : pPGPOptionList;
  RecipientSpec	 : TPGPRecipientSpec;
begin
  if PGP9X then
    Utf8Prompt := AnsiToUtf8(Prompt)
  else Utf8Prompt := Prompt;
  RecipientKeys := nil;
  GroupFileSpec := nil;
  GroupSet := nil;
  OptionList := nil;
  FormOptions := nil;
  if SMime then
    Result := kPGPError_FeatureNotAvailable
  else Result := PGPNewFileSpecFromFullPath(Context, PChar(KeyRings.GroupsFile), GroupFileSpec);
  try
    if Result = 0 then Result := PGPNewGroupSetFromFile(Context, GroupFileSpec, GroupSet);
    try
      if Result = 0 then
	GroupOption := PGPOUIRecipientGroups(Context, GroupSet)
      else GroupOption := PGPONullOption(Context);
      if SMime then
	FormOptions := PGPONullOption(Context)
      else begin
	Result := PGPBuildOptionList(Context, FormOptions,
	  [
	    PGPOUICheckbox(Context, 801, '&Armor', nil, Armor, Armor, PGPOLastOption(Context)),
	    PGPOUICheckbox(Context, 802, '&MIME', nil, MIME, MIME, PGPOLastOption(Context)),
	    PGPOUICheckbox(Context, 803, '&Textmode', nil, Textmode, Textmode, PGPOLastOption(Context)),
	    PGPOUICheckbox(Context, 804, 'For your &eyes only', nil, EyesOnly, EyesOnly, PGPOLastOption(Context))
	  ]);
	if Result <> 0 then Exit;
      end;
      try
	Result := PGPBuildOptionList(Context, OptionList,
	  [
	    PGPOUIDialogOptions(Context, FormOptions, PGPOLastOption(Context)),
	    PGPOUIDisplayMarginalValidity(Context, PGPTrue),
	    GroupOption,
	    PGPOUIWindowTitle(Context, 'PGP'),
	    PGPOUIDialogPrompt(Context, PChar(Utf8Prompt)),
	    PGPOUIParentWindowHandle(Context, WinHandle)
	  ]);
	if Result <> 0 then Exit;
	if DefaultKey <> '' then begin
	  FillChar(RecipientSpec, SizeOf(TPGPRecipientSpec), 0);
	  RecipientSpec.AType := kPGPRecipientSpecType_Key;
	  Result := GetKeyByHexID(AllKeys, DefaultKey, RecipientSpec.ID.Key);
	  if Result = 0 then PGPAppendOptionList(OptionList, [PGPOUIDefaultRecipients(Context, 1, @RecipientSpec)]);
	end;
	try
	  if PGP7X then begin
	    Result := PGPRecipientDialog(Context, PGPPeekKeySetKeyDB(AllKeys), PGPBoolean(ord(AlwaysDisplayDialog) and 1),
					 pPGPKeyDB(RecipientKeys) ,OptionList, PGPOLastOption(Context));
	  end
	  else begin
	    Result := PGPRecipientDialog(Context, AllKeys, PGPBoolean(ord(AlwaysDisplayDialog) and 1),
					 pPGPKeySet(RecipientKeys), OptionList, PGPOLastOption(Context));
	  end;
	  ProcessMessages;
	finally
	  PGPFreeOptionList(OptionList);
	end;
      finally
	if not SMime then PGPFreeOptionList(FormOptions);
      end;
    finally
      if GroupSet <> nil then PGPFreeGroupSet(GroupSet);
    end;
  finally
    PGPFreeFileSpec(GroupFileSpec);
  end;
end;

function KeyServerDialog(Context: pPGPContext; KeySetMain: pPGPKeySet; const KeyData: String;
			 var KeysFound: Pointer; WinHandle: Integer): PGPError;
var
  tlsContext	 : pPGPtlsContext;
  ThreadStorage	 : pPGPKeyServerThreadStorage;
  KeyServerSpecs : pPGPKeyServerSpec;
  KeyServerCount : PGPUInt32;
  OptionList	 : pPGPOptionList;
begin
  KeysFound := nil;
  tlsContext := nil;
  OptionList := nil;
  Result := PGPNewTLSContext(Context, tlsContext);
  if Result <> 0 then Exit;
  try
    Result := InitKeyServer(Context, ThreadStorage, KeyServerSpecs, KeyServerCount);
    if Result <> 0 then Exit;
    try
      Result := SearchKeyServerOptions(Context, KeyData, false, false, WinHandle, OptionList);
      if Result <> 0 then Exit;
      try
	Result := PGPSearchKeyServerDialog(Context, KeyServerCount, KeyServerSpecs, tlsContext, PGPFalse,
					   KeysFound, OptionList, PGPOLastOption(Context));
	Sleep(100); // trying to prevent a strange cancel crash
	ProcessMessages;
      finally
	PGPFreeOptionList(OptionList);
      end;
    finally
      FreeKeyServer(ThreadStorage, KeyServerSpecs, KeyServerCount);
    end;
  finally
    PGPFreeTLSContext(tlsContext);
  end;
end;

function CollectRandomDataDialog(Context: pPGPContext; NeededEntropyBits: PGPUInt32; WinHandle: Integer): PGPError;
begin
  Result := PGPCollectRandomDataDialog(Context,
				       NeededEntropyBits,
				       PGPOUIParentWindowHandle(Context, WinHandle),
				       PGPOLastOption(Context));
  ProcessMessages;
end;

function PassphraseDialog(Context: pPGPContext; const Prompt: String; var Passphrase: PChar; WinHandle: Integer): Integer;
var
  Utf8Prompt	 : String;
  OptionList	 : pPGPOptionList;
begin
  if PGP9X then
    Utf8Prompt := AnsiToUtf8(Prompt)
  else Utf8Prompt := Prompt;
  Passphrase := nil;
  OptionList := nil;
  Result := PGPBuildOptionList(Context, OptionList,
    [
      PGPOUIWindowTitle(Context, 'PGP'),
      PGPOUIDialogPrompt(Context, PChar(Utf8Prompt)),
      PGPOUIOutputPassphrase(Context, Passphrase),
      PGPOUIParentWindowHandle(Context, WinHandle)
    ]);
  if Result <> 0 then Exit;
  try
    Result := PGPPassphraseDialog(Context, OptionList, PGPOLastOption(Context));
    ProcessMessages;
  finally
    PGPFreeOptionList(OptionList);
  end;
end;

function ConvEncPassphraseDialog(Context: pPGPContext; var Passphrase: PChar;
				 const Prompt: String; WinHandle: Integer): PGPError;
var
  Utf8Prompt	 : String;
  OptionList	 : pPGPOptionList;
begin
  if PGP9X then
    Utf8Prompt := AnsiToUtf8(Prompt)
  else Utf8Prompt := Prompt;
  Passphrase := nil;
  OptionList := nil;
  Result := PGPBuildOptionList(Context, OptionList,
    [
      PGPOUIWindowTitle(Context, 'PGP'),
      PGPOUIDialogPrompt(Context, PChar(Utf8Prompt)),
      PGPOUIOutputPassphrase(Context, Passphrase),
      PGPOUIParentWindowHandle(Context, WinHandle)
    ]);
  if Result <> 0 then Exit;
  try
    Result := PGPConventionalEncryptionPassphraseDialog(Context, OptionList, PGPOLastOption(Context));
    ProcessMessages;
  finally
    PGPFreeOptionList(OptionList);
  end;
end;

function ConvDecPassphraseDialog(Context: pPGPContext; var Passphrase: PChar;
				 const Prompt: String; WinHandle: Integer): PGPError;
var
  Utf8Prompt	 : String;
  OptionList	 : pPGPOptionList;
begin
  if PGP9X then
    Utf8Prompt := AnsiToUtf8(Prompt)
  else Utf8Prompt := Prompt;
  Passphrase := nil;
  OptionList := nil;
  Result := PGPBuildOptionList(Context, OptionList,
    [
      PGPOUIWindowTitle(Context, 'PGP'),
      PGPOUIDialogPrompt(Context, PChar(Utf8Prompt)),
      PGPOUIOutputPassphrase(Context, Passphrase),
      PGPOUIParentWindowHandle(Context, WinHandle)
    ]);
  if Result <> 0 then Exit;
  try
    Result := PGPConventionalDecryptionPassphraseDialog(Context, OptionList, PGPOLastOption(Context));
    ProcessMessages;
  finally
    PGPFreeOptionList(OptionList);
  end;
end;

function KeyPassphraseDialog(Context: pPGPContext; KeySet: pPGPKeySet; var Passphrase: PChar;
			     const Prompt: String; WinHandle: Integer): PGPError;
var
  Utf8Prompt	 : String;
  Key		 : pPGPKey;
  OptionList	 : pPGPOptionList;
begin
  if PGP9X then
    Utf8Prompt := AnsiToUtf8(Prompt)
  else Utf8Prompt := Prompt;
  Passphrase := nil;
  OptionList := nil;
  Result := GetKeyFromKeySet(KeySet, Key);
  if Result <> 0 then Exit;
  Result := PGPBuildOptionList(Context, OptionList,
    [
      PGPOUIWindowTitle(Context, 'PGP'),
      PGPOUIDialogPrompt(Context, PChar(Utf8Prompt)),
      PGPOUIOutputPassphrase(Context, Passphrase),
      PGPOUIParentWindowHandle(Context, WinHandle)
    ]);
  if Result <> 0 then Exit;
  try
    Result := PGPKeyPassphraseDialog(Context, Key, OptionList, PGPOLastOption(Context));
    ProcessMessages;
  finally
    PGPFreeOptionList(OptionList);
  end;
end;

function SigningPassphraseDialog(Context: pPGPContext; KeySetMain: pPGPKeySet;
				 var SigningKey: pPGPKey; var SignKeyPass: PChar;
				 const DefaultKeyID: String; FindMatchingKey: Longbool; SignOption: TSignOption;
				 var Armor, MIME, RipeMD: PGPUInt32; const Prompt: String; WinHandle: Integer): PGPError;
var
  Utf8Prompt	 : String;
  DefaultKey	 : pPGPKey;
  OptionList	 : pPGPOptionList;
  DialogOptions	 : pPGPOptionList;
begin
  if PGP9X then
    Utf8Prompt := AnsiToUtf8(Prompt)
  else Utf8Prompt := Prompt;
  SigningKey := nil;
  SignKeyPass := nil;
  OptionList := nil;
  if (DefaultKeyID = '') or (GetKeyByHexID(KeySetMain, PChar(DefaultKeyID), DefaultKey) <> 0) then DefaultKey := nil;
  Result := PGPBuildOptionList(Context, OptionList,
    [
      PGPOUIWindowTitle(Context, 'PGP'),
      PGPOUIDialogPrompt(Context, PChar(Utf8Prompt)),
      PGPOUIDefaultKey(Context, DefaultKey),
      PGPOUIFindMatchingKey(Context, PGPBoolean(FindMatchingKey) and 1),
      PGPOUIVerifyPassphrase(Context, PGPTrue),
      PGPOUIOutputPassphrase(Context, SignKeyPass),
      PGPOUIParentWindowHandle(Context, WinHandle)
    ]);
  if Result <> 0 then Exit;
  if SignOption <> soNo then begin
    DialogOptions := nil;
    try
      case SignOption of
	soEnc: begin
	  Result := PGPBuildOptionList(Context, DialogOptions,
	    [
	      PGPOUICheckbox(Context, 803, '&RipeMD', nil, RipeMD, RipeMD, PGPOLastOption(Context))
	    ]);
	end;
	soClear: begin
	  Result := PGPBuildOptionList(Context, DialogOptions,
	    [
	      PGPOUICheckbox(Context, 802, '&MIME', nil, MIME, MIME, PGPOLastOption(Context)),
	      PGPOUICheckbox(Context, 803, '&RipeMD (RSA)', nil, RipeMD, RipeMD, PGPOLastOption(Context))
	    ]);
	end;
	soDetached: begin
	  Result := PGPBuildOptionList(Context, DialogOptions,
	    [
	      PGPOUICheckbox(Context, 801, '&Armor', nil, Armor, Armor, PGPOLastOption(Context)),
	      PGPOUICheckbox(Context, 803, '&RipeMD', nil, RipeMD, RipeMD, PGPOLastOption(Context))
	    ]);
	end;
      end;
      if Result <> 0 then Exit;
      Result := PGPAppendOptionList(OptionList, [PGPOUIDialogOptions(Context, DialogOptions, PGPOLastOption(Context))]);
    finally
      PGPFreeOptionList(DialogOptions);
    end;
    if Result <> 0 then Exit;
  end;
  try
    if PGP7X then begin
      Result := PGPSigningPassphraseDialog(Context, PGPPeekKeySetKeyDB(KeySetMain), SigningKey,
					   OptionList, PGPOLastOption(Context));
    end
    else Result := PGPSigningPassphraseDialog(Context, KeySetMain, SigningKey, OptionList, PGPOLastOption(Context));
    ProcessMessages;
  finally
    PGPFreeOptionList(OptionList);
  end;
end;

function DecryptionPassphraseDialog(Context: pPGPContext; RecipientKeys: pPGPKeySet;
				    KeyIDCount: PGPUInt32; const KeyIDList: pPGPKeyID7;
				    var DecryptionKey: pPGPKey; var DecryptKeyPass: PChar;
				    FindMatchingKey: Longbool; const Prompt: String; WinHandle: Integer): PGPError;
var
  Utf8Prompt	 : String;
  OptionList	 : pPGPOptionList;
begin
  if PGP9X then
    Utf8Prompt := AnsiToUtf8(Prompt)
  else Utf8Prompt := Prompt;
  DecryptionKey := nil;
  DecryptKeyPass := nil;
  OptionList := nil;
  Result := PGPBuildOptionList(Context, OptionList,
    [
      PGPOUIWindowTitle(Context, 'PGP'),
      PGPOUIDialogPrompt(Context, PChar(Utf8Prompt)),
      PGPOUIFindMatchingKey(Context, PGPBoolean(FindMatchingKey) and 1),
      PGPOUIVerifyPassphrase(Context, PGPTrue),
      PGPOUIOutputPassphrase(Context, DecryptKeyPass),
      PGPOUIParentWindowHandle(Context, WinHandle)
    ]);
  if Result <> 0 then Exit;
  try
    Result := PGPDecryptionPassphraseDialog(Context, RecipientKeys, KeyIDCount, KeyIDList,
					    DecryptionKey, OptionList, PGPOLastOption(Context));
    ProcessMessages;
  finally
    PGPFreeOptionList(OptionList);
  end;
end;

function ConfirmationPassphraseDialog(Context: pPGPContext; var Passphrase: PChar;
				      MinPassLength, MinPassQuality: Integer;
				      ShowPassphraseQuality: Longbool;
				      const Prompt: String; WinHandle: Integer): PGPError;
var
  Utf8Prompt	 : String;
  OptionList	 : pPGPOptionList;
begin
  if PGP9X then
    Utf8Prompt := AnsiToUtf8(Prompt)
  else Utf8Prompt := Prompt;
  Passphrase := nil;
  OptionList := nil;
  Result := PGPBuildOptionList(Context, OptionList,
    [
      PGPOUIWindowTitle(Context, 'PGP'),
      PGPOUIDialogPrompt(Context, PChar(Utf8Prompt)),
      PGPOUIMinimumPassphraseLength(Context, MinPassLength),
      PGPOUIMinimumPassphraseQuality(Context, MinPassQuality),
      PGPOUIShowPassphraseQuality(Context, PGPBoolean(ShowPassphraseQuality) and 1),
      PGPOUIOutputPassphrase(Context, Passphrase),
      PGPOUIParentWindowHandle(Context, WinHandle)
    ]);
  if Result <> 0 then Exit;
  try
    Result := PGPConfirmationPassphraseDialog(Context, OptionList, PGPOLastOption(Context));
    ProcessMessages;
  finally
    PGPFreeOptionList(OptionList);
  end;
end;

end.

