{$J+,Z4}
unit PGPDecode;

{------------------------------------------------------------------------------}
{                                                                              }
{                 This unit is partly based on Steve Heller's                  }
{         SPGP sources available from http://www.oz.net/~srheller/spgp/        }
{                                                                              }
{                Portions created by Michael in der Wiesche are                }
{              Copyright (C) 2001-2006 by Michael in der Wiesche               }
{                                                                              }
{------------------------------------------------------------------------------}

interface

uses
  Windows,
  Classes,
  SysUtils,
  PSMimeTools,
  KeyPropTypes,
  X509Types,
  UTF8,
  pgpBase,
  pgpErrors,
  pgpPubTypes,
  pgpUtilities,
  pgpOptionList,
  pgpMemoryMgr,
  pgpEvents,
  pgpKeys,
  pgpTLS,
  pgpSC,
  KeyFuncs,
  PrefFuncs,
  X509Funcs,
  PGPDialogs;

type
  TSigStatus = (
    SIGNED_NOT,
    SIGNED_BAD,
    SIGNED_GOOD,
    SIGNED_NO_KEY,
    SIGNED_UNKNOWN_ALG,
    SIGNED_SIG_EXPIRED
  );
  TSigPropsRec = Record
    sStatus: TSigStatus;
    sHexID: String;
    sUserID: String;
    sCreaTimeStr: String;
    sCreaTimeNum: DWord;
    sChecked: Longbool;
    sVerified: Longbool;
    sValidity: TValidityLevel;
    sRevoked: Longbool;
    sDisabled: Longbool;
    sExpired: Longbool;
    sDetached: Longbool;
    sAxiomatic: Longbool;
    sExpTimeNum: Longint;
    sExpTimeStr: String;
    sHashAlgorithm: THashAlgorithm;
    sValidCertChain: TX509ChainValidity;
  end;
  TAnalysis = (
    ENCRYPTED,
    SIGNED,
    DETACHED_SIGNATURE,
    KEY,
    UNKNOWN,
    X509_CERTIFICATE,
    X509_SECRETKEY,
    SMIME_UNRECOGNIZED,
    SMIME_ENCRYPTED,
    SMIME_SIGNED
  );
  TPassThroughOption = (
    PassThrough_Keys,
    PassThrough_ClearSigned,
    PassThrough_Unrecognized
  );
  TPassOption = (
    AnalyseOnly,
    AnalysePass,
    DecodePass
  );
  TX509Mode = (
    X509Off,
    X509Secret,
    X509Binary,
    X509Armored,
    X509IsMime,
    X509Signed,
    X509Encrypted,
    X509Processed
  );
  TPassThroughOptions = Set of TPassThroughOption;
  TOnAnalysis = procedure(SectionNumber: Longint; SectionType: TAnalysis; var SkipSection: Longbool) of Object;
  TOnGetInputFileName = procedure(var SuggestedName: String) of Object;
  TOnGetOutputFileName = procedure(var SuggestedName: String) of Object;
  TOnGetSignedFileName = procedure(var SuggestedName: String) of Object;
  TOnEnterPassphrase = procedure(const Passphrase: PChar;
				 const DecryptionKeyList: TKeyPropsList;
				 Conventional, BadPassphrase: Longbool;
				 var Cancel: Longbool) of Object;
  TOnGetSignature = procedure(const SigProps: TSigPropsRec) of Object;
  TOnShowProgress = procedure(BytesProcessed, BytesTotal: Longint) of Object;
  TOnWipePassphrase = procedure(const Passphrase: PChar) of Object;
  TPGPDecodeCustom = class(TComponent)
  private
    // internal
    FContext: pPGPContext;
    FKeySetMain: pPGPKeySet;
    FAllocatedOutputBuffer: PChar;
    FActualOutputSize: PGPSize;
    FOutputOptions: pPGPOptionList;
    FInputSize: PGPSize;
    FInputBuffer: PChar;
    FInputFileName: String;
    FFileNameBuffer: String;
    FAnalysis: TAnalysis;
    FClearSigned: Longbool;
    FEyesOnly: Longbool;
    FFileToFile: Longbool;
    FOutputExists: Longbool;
    FIsPGPMime: Longbool;
    FRecognized: Longbool;
    FRecurring: Longbool;
    FPassOption: TPassOption;
    FPassphrase: PChar;
    FPassSection: PGPUInt32;
    FRecipientsData: TPGPEventRecipientsData;
    FSectionNumber: PGPUInt32;
    FSignedDataBuffer: String;
    FDecryptionKey: pPGPKey;
    FSecureBuffer: PChar;
    FSecureBufferSize: PGPSize;
    FValidCertChain: TX509ChainValidity;
    // protected properties
    FOutputBuffer: String;
    FDetachedSigVerify: Longbool;
    FIgnoreKnownFlag: Longint;
    FSigPropsRec: TSigPropsRec;
    FKeyPropsList: TKeyPropsList;
    // public properties
    FKeyAddResult: Longint;
    FParentHandle: THandle;
    // published properties
    FFileOutput: Longbool;
    FGetKeyFromServer: Longbool;
    FLogBadSignature: String;
    FLogExpSignature: String;
    FLogKeyDisabled: String;
    FLogKeyExpired: String;
    FLogKeyRevoked: String;
    FLogKeyUnknown: String;
    FLogUnknownAlg: String;
    FPassCacheSeconds: TPassCacheSeconds;
    FPassCacheShare : Longbool;
    FPassThroughOptions: TPassThroughOptions;
    FProgressInterval: Cardinal;
    FQueryAddKeys: Longbool;
    FRecursivelyDecode: Longbool;
    FShowSigLog: Longbool;
    FX509Mode: TX509Mode;
    FKeyDlgPrompt: String;
    FPassDlgPrompt: String;
    FOutputFileName: String;
    FSignedFileName: String;
    // events
    FOnAnalysis: TOnAnalysis;
    FOnGetInputFileName: TOnGetInputFileName;
    FOnGetOutputFileName: TOnGetOutputFileName;
    FOnGetSignedFileName: TOnGetSignedFileName;
    FOnEnterPassphrase: TOnEnterPassphrase;
    FOnGetSignature: TOnGetSignature;
    FOnShowProgress: TOnShowProgress;
    FOnWipePassphrase: TOnWipePassphrase;
    procedure SetX509Mode(Value: TX509Mode);
    procedure ClearSigPropsRec;
    function  InitDecode: PGPError;
    procedure FinitDecode;
    function  HandleFileInput: PGPError;
    function  X509Test: TX509Mode;
    function  SecureMove: PGPError;
    function  ShowSigInfo: PGPError;
    function  MaxFileSize: DWord;
    function  FileSize(const FileName: String; var Bytes: DWord): PGPError;
    function  ReadInputFile: PGPError;
    function  GetOutputFileName: String;
    function  Analyze(InFileSpec: pPGPFileSpec): PGPError;
    function  FixNonAsciiBug(const Data: PChar): PGPError;
    function  GetOptionList(var OptionList: pPGPOptionList): PGPError;
    function  SetOutputOption(var OptionList: pPGPOptionList): PGPError;
    function  PrepareOutput(const Input: PChar; InputSize: DWord; const MimeHeaders, MimeBody: String): PGPError;
    function  DoRecursion(const Input: PChar; InputSize: DWord): PGPError;
    function  SecureView(const Output: PChar; OutputSize: DWord): PGPError;
    function  WriteOutputFile(const Output: PChar; OutputSize: DWord): PGPError;
    function  Decode(IsFile: Longbool): Longint;
    function  AnalyzeHandler(Event: pPGPEvent): PGPError;
    function  RecipientsHandler(Event: pPGPEvent): PGPError;
    function  PassphraseHandler(Event: pPGPEvent): PGPError;
    function  KeyFoundHandler(Event: pPGPEvent): PGPError;
    function  OutputHandler(Event: pPGPEvent): PGPError;
    function  SignatureHandler(Event: pPGPEvent): PGPError;
    function  EndLexHandler(Event: pPGPEvent): PGPError;
  protected
    property OutputBuffer: String
      read FOutputBuffer;
    property DetachedSignature: Longbool
      read FDetachedSigVerify;
    property IgnoreKnownFlag: Longint
      write FIgnoreKnownFlag;
    property SigPropsRec: TSigPropsRec
      read FSigPropsRec;
    property KeyPropsList: TKeyPropsList
      read FKeyPropsList;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function AnalyseBuffer(const DataBuffer: String): Longint; virtual;
    function AnalyseFile(const FileName: String): Longint; virtual;
    function AnalyseMimeBuffer(const DataBuffer: String): Longint; virtual;
    function AnalyseMimeFile(const FileName: String): Longint; virtual;
    function DecodeBuffer(const DataBuffer: String): Longint; virtual;
    function DecodeFile(const FileName: String): Longint; virtual;
    function DecodeMimeBuffer(const DataBuffer: String): Longint; virtual;
    function DecodeMimeFile(const FileName: String): Longint; virtual;
    property KeyAddResult: Longint
      read FKeyAddResult;
    property ParentHandle: THandle
      read FParentHandle
      write FParentHandle;
  published
    property FileOutput: Longbool
      read FFileOutput
      write FFileOutput;
    property GetKeyFromServer: Longbool
      read FGetKeyFromServer
      write FGetKeyFromServer;
    property LogBadSignature: String
      read FLogBadSignature
      write FLogBadSignature;
    property LogExpSignature: String
      read FLogExpSignature
      write FLogExpSignature;
    property LogKeyDisabled: String
      read FLogKeyDisabled
      write FLogKeyDisabled;
    property LogKeyExpired: String
      read FLogKeyExpired
      write FLogKeyExpired;
    property LogKeyRevoked: String
      read FLogKeyRevoked
      write FLogKeyRevoked;
    property LogKeyUnknown: String
      read FLogKeyUnknown
      write FLogKeyUnknown;
    property LogUnknownAlg: String
      read FLogUnknownAlg
      write FLogUnknownAlg;
    property PassCacheSeconds: TPassCacheSeconds
      read FPassCacheSeconds
      write FPassCacheSeconds;
    property PassCacheShare : Longbool
      read FPassCacheShare
      write FPassCacheShare;
    property PassThroughOptions: TPassThroughOptions
      read FPassThroughOptions
      write FPassThroughOptions;
    property ProgressInterval: Cardinal
      read FProgressInterval
      write FProgressInterval;
    property RecursivelyDecode: Longbool
      read FRecursivelyDecode
      write FRecursivelyDecode;
    property QueryAddKeys: Longbool
      read FQueryAddKeys
      write FQueryAddKeys;
    property ShowSigLog: Longbool
      read FShowSigLog
      write FShowSigLog;
    property X509Mode: TX509Mode
      read FX509Mode
      write SetX509Mode;
    property KeyDlgPrompt: String
      read FKeyDlgPrompt
      write FKeyDlgPrompt;
    property PassDlgPrompt: String
      read FPassDlgPrompt
      write FPassDlgPrompt;
    property OutputFileName: String
      read FOutputFileName
      write FOutputFileName;
    property SignedFileName: String
      read FSignedFileName
      write FSignedFileName;
    property OnAnalysis: TOnAnalysis
      read FOnAnalysis
      write FOnAnalysis;
    property OnGetInputFileName: TOnGetInputFileName
      read FOnGetInputFileName
      write FOnGetInputFileName;
    property OnGetSignedFileName: TOnGetSignedFileName
      read FOnGetSignedFileName
      write FOnGetSignedFileName;
    property OnGetOutputFileName: TOnGetOutputFileName
      read FOnGetOutputFileName
      write FOnGetOutputFileName;
    property OnEnterPassphrase: TOnEnterPassphrase
      read FOnEnterPassphrase
      write FOnEnterPassphrase;
    property OnGetSignature: TOnGetSignature
      read FOnGetSignature
      write FOnGetSignature;
    property OnShowProgress: TOnShowProgress
      read FOnShowProgress
      write FOnShowProgress;
    property OnWipePassphrase: TOnWipePassphrase
      read FOnWipePassphrase
      write FOnWipePassphrase;
  end;

implementation

function DataCopy(var Output: PChar; const Input: PChar; InputSize: DWord): PGPError;
begin
  Result := kPGPError_OutOfMemory;
  if InputSize > 0 then begin
    Output := PGPNewData(PGPGetDefaultMemoryMgr, InputSize, kPGPMemoryMgrFlags_None);
    if Output <> nil then begin
      Move(Input[0], Output[0], InputSize);
      Result := kPGPError_NoErr;
    end;
  end;
end;

function EventHandler(Context: pPGPContext; Event: pPGPEvent; UserValue: PGPUserValue): PGPError; cdecl;
begin
  Result := 0;
  with TPGPDecodeCustom(UserValue) do begin
    case Event^.EType of
      kPGPEvent_NullEvent:		if (FPassOption = DecodePass)
					and not (FDetachedSigVerify or FClearSigned) then begin
					  if TMethod(FOnShowProgress).Code <> nil then begin
					    // BytesTotal always stays 0 => use FInputSize (at least in 6.5.X)
					    with Event^.EData.NullData do FOnShowProgress(BytesWritten, FInputSize);
					  end;
					  ProcessMessages;
					end;
      kPGPEvent_InitialEvent:		;
      kPGPEvent_FinalEvent:		;
      kPGPEvent_ErrorEvent:	  	Result := Event^.EData.ErrorData.Error;
      kPGPEvent_WarningEvent:		;
      kPGPEvent_PassphraseEvent:	Result := PassphraseHandler(Event);
      kPGPEvent_AnalyzeEvent:		Result := AnalyzeHandler(Event);
      kPGPEvent_RecipientsEvent:  	if FPassOption = DecodePass then Result := RecipientsHandler(Event);
      kPGPEvent_KeyFoundEvent:		if FPassOption = DecodePass then Result := KeyFoundHandler(Event);
      kPGPEvent_OutputEvent:		if FPassOption = DecodePass then OutputHandler(Event);
      kPGPEvent_SignatureEvent:		if FPassOption = DecodePass then Result := SignatureHandler(Event);
      kPGPEvent_BeginLexEvent:	  	FSectionNumber := succ(Event^.EData.BeginLexData.SectionNumber);
      kPGPEvent_EndLexEvent:      	Result := EndLexHandler(Event);
      kPGPEvent_DetachedSignatureEvent:	;
      kPGPEvent_DecryptionEvent:	;
    end;
  end;
end;

constructor TPGPDecodeCustom.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FKeyPropsList := TKeyPropsList.Create(spgpKeyPropFlag_IDFlags);
  FLogBadSignature := 'bad signature';
  FLogExpSignature := 'expired signature';
  FLogKeyDisabled := 'disabled key';
  FLogKeyExpired := 'expired key';
  FLogKeyRevoked := 'revoked key';
  FLogKeyUnknown := 'unknown key';
  FProgressInterval := 1000;
end;

destructor TPGPDecodeCustom.Destroy;
begin
  FKeyPropsList.Free;
  inherited Destroy;
end;

procedure TPGPDecodeCustom.SetX509Mode(Value: TX509Mode);
begin
  if PGP7X and (FPassOption <> DecodePass) then FX509Mode := Value;
end;

procedure TPGPDecodeCustom.ClearSigPropsRec;
begin
  with FSigPropsRec do begin
    sStatus := SIGNED_NOT;
    sHexID := '';
    sUserID := '';
    sCreaTimeStr := '';
    sCreaTimeNum := 0;
    sChecked := false;
    sVerified := false;
    sValidity := Validity_Unknown;
    sRevoked := false;
    sDisabled := false;
    sExpired := false;
    sDetached := false;
    sAxiomatic := false;
    sExpTimeNum := 0;
    sExpTimeStr := '';
    sHashAlgorithm := HashAlgorithm_Invalid;
    sValidCertChain := CertChain_Invalid;
  end;
end;

function TPGPDecodeCustom.InitDecode: PGPError;
begin
  FAnalysis := UNKNOWN;
  FSectionNumber := 0;
  FOutputBuffer := '';
  FSecureBufferSize := 0;
  FSecureBuffer := nil;
  FActualOutputSize := 0;
  FAllocatedOutputBuffer := nil;
  FSignedDataBuffer := '';
  FDetachedSigVerify := false;
  FClearSigned := false;
  FIsPGPMime := false;
  FKeyAddResult := 0;
  FRecognized := false;
  if not FRecurring then FEyesOnly := false;
  FKeyPropsList.Clear;
  FPassSection := 0;
  FDecryptionKey := nil;
  FPassphrase := nil;
  ClearSigPropsRec;
  FValidCertChain := CertChain_Invalid;
  FillChar(FRecipientsData, SizeOf(TPGPEventRecipientsData), 0);
  Result := KeyRings.InitKeyRings(FContext, FKeySetMain);
end;

procedure TPGPDecodeCustom.FinitDecode;
begin
  FX509Mode := X509Off;
  if (FKeyPropsList.Count <> 0) or (FAnalysis = X509_CERTIFICATE) then KeyRings.UpdateKeyRings;
  with FRecipientsData do begin
    PGPFreeData(KeyIDArray);
    PGPFreeKeySet(RecipientSet);
  end;
  KeyRings.FreeKeyRings;
end;

function TPGPDecodeCustom.HandleFileInput: PGPError;
begin
  Result := 0;
  FInputFileName := FInputBuffer;
  if not FileExists(FInputFileName) then begin
    if Assigned(FOnGetInputFileName) then begin
      FOnGetInputFileName(FInputFileName);
      if FInputFileName = '' then Result := kPGPError_FileNotFound;
    end
    else Result := kPGPError_FileNotFound;
  end;
  if Result <> 0 then Exit;
  Result := FileSize(FInputFileName, FInputSize);
  if PGP7X and (FInputSize > MaxFileSize) then begin
    if FX509Mode >= X509IsMime then
      Result := kPGPError_FeatureNotAvailable
    else FFileToFile := FFileOutput;
  end
  else FFileToFile := false;
end;

function TPGPDecodeCustom.X509Test: TX509Mode;
var
  Pos		: DWord;
  Error		: PGPError;
  OptionList	: pPGPOptionList;
  TestDB	: pPGPKeyDB;
begin
  Result := X509Off;
  if PGP81 then begin
    Pos := DWord(StrPos(FInputBuffer, '-----BEGIN CERTIFICATE-----'));
    if Pos > 0 then begin
      dec(Pos, DWord(FInputBuffer));
      if ((Pos = 0) or (FInputBuffer[pred(Pos)] = #10)) then Result := X509Armored;
    end;
    if Result = X509Off then begin
      TestDB := nil;
      try
	// test for PKCS7 format
	OptionList := nil;
	try
	  Error := PGPBuildOptionList(FContext, OptionList,
	    [
	      PGPOInputFormat(FContext, kPGPInputFormat_X509DataInPKCS7),
	      PGPOInputBuffer(FContext, FInputBuffer, FInputSize)
	    ]);
	  if (Error = 0) and (PGPImport(FContext, TestDB, OptionList, PGPOLastOption(FContext)) = 0) then begin
	    Result := X509Binary;
	  end;
	finally
	  PGPFreeOptionList(OptionList);
	end;
	if Result = X509Off then begin
	  // test for PKCS12 format
	  OptionList := nil;
	  try
	    Error := PGPBuildOptionList(FContext, OptionList,
	      [
		PGPOInputFormat(FContext, kPGPInputFormat_PKCS12),
		PGPOInputBuffer(FContext, FInputBuffer, FInputSize)
	      ]);
	    if (Error = 0)
	    and (PGPImport(FContext, TestDB, OptionList, PGPOLastOption(FContext)) = kPGPError_BadPassphrase) then begin
	      Result := X509Secret;
	    end;
	  finally
	    PGPFreeOptionList(OptionList);
	  end;
	end;
      finally
	PGPFreeKeyDB(TestDB);
      end;
    end;
  end;
end;

function TPGPDecodeCustom.SecureMove: PGPError;
begin
  Result := kPGPError_NoErr;
  if FActualOutputSize <> 0 then begin
    try
      if FEyesOnly and (FSecureBuffer = nil) then begin
	Result := kPGPError_OutOfMemory;
	FSecureBuffer := PGPNewSecureData(PGPGetDefaultMemoryMgr,
					  succ(FSecureBufferSize + FActualOutputSize),
					  kPGPMemoryMgrFlags_None);
	if FSecureBuffer <> nil then Result := 0;
      end
      else begin
	Result := PGPReallocData(PGPGetDefaultMemoryMgr,
				 Pointer(FSecureBuffer),
				 succ(FSecureBufferSize + FActualOutputSize),
				 kPGPMemoryMgrFlags_None);
      end;
      if Result = 0 then begin
	Move(FAllocatedOutputBuffer[0], FSecureBuffer[FSecureBufferSize], succ(FActualOutputSize));
	if FEyesOnly then FillChar(FAllocatedOutputBuffer^, FActualOutputSize, 0);
	inc(FSecureBufferSize, FActualOutputSize);
      end;
    finally
      PGPFreeData(FAllocatedOutputBuffer);
      FAllocatedOutputBuffer := nil;
      FActualOutputSize := 0;
    end;
  end;
end;

function TPGPDecodeCustom.ShowSigInfo: PGPError;
var
  Status	: String;
  Source	: String;
  ID		: String;
  Validity	: String;
  Info		: String;
begin
  Result := 0;
  with FSigPropsRec do begin
    if sStatus <> SIGNED_NOT then begin
      // don't change these (constant) PGP strings, see SigEvent.c
      if sStatus = SIGNED_GOOD then
	Status := 'Good Signature' + #10
      else Status := 'Bad Signature' + #10;
      // these strings may be customized
      if FInputFileName <> '' then begin
	if FDetachedSigVerify then
	  Source := ExtractFileName(FSignedFileName) + #10
	else if FRecurring then
	  Source := ExtractFileName(FFileNameBuffer) + #10
	else Source := ExtractFileName(FInputFileName) + #10;
      end
      else Source := '(...)' + #10;
      if (sStatus = SIGNED_NO_KEY) or (sUserID = '') then begin
	if PGP8X then
	  ID := '(' + sHexID + ')' + #10 + ShortHexID(sHexID) + #10
	else ID := '(' + ShortHexID(sHexID) + ')' + #10;
      end
      else begin
	if PGP8X then
	  ID := sUserID + #10 + ShortHexID(sHexID) + #10
	else ID := sUserID + #10;
      end;
      // don't change these (constant) PGP strings, see SigEvent.c
      if sAxiomatic then
	Validity := 'Implicit Trust' + #10 + sCreaTimeStr
      else begin
	case sValidity of
	  Validity_Unknown, Validity_Invalid: Validity := 'Invalid Key' + #10 + sCreaTimeStr;
	  Validity_Marginal: Validity := 'Marginal Key' + #10 + sCreaTimeStr;
	  Validity_Complete: Validity := 'Valid Key' + #10 + sCreaTimeStr;
	end;
      end;
      // these strings may be customized
      if sUserID = '' then
	Info := FLogKeyUnknown
      else begin
	if sStatus <> SIGNED_UNKNOWN_ALG then begin
	  if sVerified then begin
	    if sStatus = SIGNED_SIG_EXPIRED then
	      Info := FLogExpSignature
	    else begin
	      if sRevoked then Info := FLogKeyRevoked;
	      if sDisabled then Info := FLogKeyDisabled;
	      if sExpired then Info := FLogKeyExpired;
	    end;
	  end
	  else Info := FLogBadSignature;
	end
	else Info := FLogUnknownAlg;
      end;
      if Info <> '' then  Info := ' (' + Info + ')';
      if not SendPGPlogMsg(FParentHandle, Status + Source + ID + Validity + Info) then begin
	Result := kPGPError_FeatureNotAvailable;
      end;
    end;
  end;
end;

function TPGPDecodeCustom.MaxFileSize: DWord;
var
  MemoryStatus	: TMemoryStatus;
begin
  MemoryStatus.dwLength := SizeOf(TMemoryStatus);
  GlobalMemoryStatus(MemoryStatus);
  Result := MemoryStatus.dwAvailPhys shr 2;
end;

function TPGPDecodeCustom.FileSize(const FileName: String; var Bytes: DWord): PGPError;
var
  FileHandle	: THandle;
  FileSizeHigh	: DWord;
begin
  Result := 0;
  FileHandle := FileOpen(FileName, fmOpenRead or fmShareDenyNone);
  if FileHandle <> INVALID_HANDLE_VALUE then begin
    Bytes := GetFileSize(FileHandle, @FileSizeHigh);
    FileClose(FileHandle);
  end
  else Bytes := $FFFFFFFF;
  if (Bytes = 0) or (Bytes = $FFFFFFFF) or (FileSizeHigh <> 0) then Result := kPGPError_ReadFailed;
end;

function TPGPDecodeCustom.ReadInputFile: PGPError;
var
  InputFile	: THandle;
  BytesRead	: DWord;
begin
  FInputBuffer := PGPNewSecureData(PGPGetDefaultMemoryMgr, succ(FInputSize), kPGPMemoryMgrFlags_None);
  if FInputBuffer <> nil then begin
    FInputBuffer[FInputSize] := #0;
    InputFile := CreateFile(PChar(FInputFileName), GENERIC_READ, FILE_SHARE_READ, nil, OPEN_EXISTING,
			    FILE_FLAG_SEQUENTIAL_SCAN, 0);
    if InputFile <> INVALID_HANDLE_VALUE then begin
      try
	if not ReadFile(InputFile, FInputBuffer[0], FInputSize, BytesRead, nil) or (BytesRead <> FInputSize) then
	  Result := kPGPError_ReadFailed
	else Result := kPGPError_NoErr;
      finally
	CloseHandle(InputFile);
      end;
    end
    else Result := kPGPError_CantOpenFile;
  end
  else begin
    Result := kPGPError_OutOfMemory;
    Exit;
  end;
end;

function TPGPDecodeCustom.GetOutputFileName: String;
var
  FileExtension	: String;
begin
  if FDetachedSigVerify then begin
    if FSignedFileName = '' then begin
      FSignedFileName := Trim(FInputFileName);
      FileExtension := LowerCase(ExtractFileExt(FSignedFileName));
      if (FileExtension = '.sig') or (FileExtension = '.asc') then begin
	Delete(FSignedFileName, Length(FSignedFileName) - 3, 4);
      end;
    end;
    if ((FSignedFileName = FInputFileName) or not FileExists(FSignedFileName))
    and Assigned(FOnGetSignedFileName) then begin
      FOnGetSignedFileName(FSignedFileName);
    end;
    Result := FSignedFileName;
  end
  else begin
    if FOutputFileName = '' then begin
      FOutputFileName := Trim(FInputFileName);
      FileExtension := LowerCase(ExtractFileExt(FOutputFileName));
      if (FileExtension = '.pgp') or (FileExtension = '.gpg') or (FileExtension = '.asc') then begin
	Delete(FOutputFileName, Length(FOutputFileName) - 3, 4);
      end;
    end;
    if (FOutputFileName = '') or (ExtractFileExt(FOutputFileName) = '') or FileExists(FOutputFileName) then begin
      if Assigned(FOnGetOutputFileName) then
	FOnGetOutputFileName(FOutputFileName)
      else FOutputFileName := '';
    end;
    Result := FOutputFileName;
    if Result <> '' then DeleteFile(Result);
  end;
end;

function TPGPDecodeCustom.Analyze(InFileSpec: pPGPFileSpec): PGPError;
var
  OptionList	: pPGPOptionList;
begin
  OptionList := nil;
  Result := PGPBuildOptionList(FContext, OptionList,
    [
      PGPODiscardOutput(FContext, PGPTrue),
      PGPOEventHandler(FContext, EventHandler, Self)
    ]);
  if Result <> 0 then Exit;
  if InFileSpec <> nil then begin
    Result := PGPAppendOptionList(OptionList, [PGPOInputFile(FContext, InFileSpec)]);
  end
  else if FInputSize <> 0 then begin
    Result := PGPAppendOptionList(OptionList, [PGPOInputBuffer(FContext, FInputBuffer, FInputSize)]);
  end;
  if Result <> 0 then Exit;
  try
    Result := pgpEvents.PGPDecode(FContext, OptionList, PGPOLastOption(FContext));
    if Result = kPGPError_UserAbort then Result := 0;
  finally
    PGPFreeOptionList(OptionList);
  end;
end;

function TPGPDecodeCustom.FixNonAsciiBug(const Data: PChar): PGPError;
begin
  // try to circumvent errors with some non-ASCII data
  inc(FInputSize);
  if FEyesOnly then
    FInputBuffer := PGPNewSecureData(PGPGetDefaultMemoryMgr, FInputSize + 2, kPGPMemoryMgrFlags_None)
  else FInputBuffer := PGPNewData(PGPGetDefaultMemoryMgr, FInputSize + 2, kPGPMemoryMgrFlags_None);
  if FInputBuffer <> nil then begin
    FInputBuffer[succ(FInputSize)] := #0;
    Move(Data[0], FInputBuffer[1], FInputSize);
    FInputBuffer[0] := '@';
    Result := 0;
  end
  else Result := kPGPError_OutOfMemory;
end;

function TPGPDecodeCustom.GetOptionList(var OptionList: pPGPOptionList): PGPError;
var
  X509Option	: pPGPOptionList;
begin
  Result := PGPBuildOptionList(FContext, OptionList,
    [
      PGPOKeySetRef(FContext, FKeySetMain),
      PGPOPassThroughKeys(FContext, PGPBoolean(PassThrough_Keys in PassThroughOptions)),
      // PGPOPassThroughClearSigned(FContext, PGPBoolean(PassThrough_ClearSigned in PassThroughOptions)), // buggy
      PGPOPassThroughIfUnrecognized(FContext, PGPBoolean(PassThrough_Unrecognized in PassThroughOptions)),
      PGPORecursivelyDecode(FContext, PGPBoolean(FRecursivelyDecode) and 1),
      PGPOSendEventIfKeyFound(FContext, PGPTrue),
      PGPOEventHandler(FContext, EventHandler, Self)
    ]);
  if Result <> 0 then Exit;
  case FX509Mode of
    X509Secret: X509Option := PGPOInputFormat(FContext, kPGPInputFormat_PKCS12);
    X509Binary: X509Option := PGPOInputFormat(FContext, kPGPInputFormat_X509DataInPKCS7);
    X509Armored: X509Option := PGPOInputFormat(FContext, kPGPInputFormat_PEMEncodedX509Cert);
    X509IsMime..X509Encrypted: X509Option := PGPOInputFormat(FContext, kPGPInputFormat_SMIMEBody);
  else
    // doesn't work properly with S/MIME
    Result := PGPAppendOptionList(OptionList, [PGPOSendNullEvents(FContext, FProgressInterval)]);
    Exit;
  end;
  if FX509Mode in [X509IsMime..X509Encrypted] then begin
    Result := PGPAppendOptionList(OptionList, [PGPOSMIMEMatchCriterion(FContext, kPGPSMIMEMatchCriterion_Any), X509Option]);
  end;
end;

function TPGPDecodeCustom.SetOutputOption(var OptionList: pPGPOptionList): PGPError;
var
  FileSpec	: pPGPFileSpec;
begin
  if FDetachedSigVerify then begin
    FFileToFile := false;
    if GetOutputFileName <> '' then begin
      FileSpec := nil;
      Result := PGPNewFileSpecFromFullPath(FContext, PChar(FSignedFileName), FileSpec);
      if Result <> 0 then Exit;
      try
	Result := PGPAppendOptionList(OptionList,
	  [
	    PGPODetachedSig(FContext, PGPOInputFile(FContext, FileSpec), PGPOLastOption(FContext))
	  ]);
      finally
	PGPFreeFileSpec(FileSpec);
      end;
    end
    else Result := kPGPError_DetachedSignatureFound;
  end
  else if FX509Mode = X509IsMime then begin
    Result := PGPAppendOptionList(OptionList,
      [
	PGPODetachedSig(FContext,
			PGPOInputBuffer(FContext, PChar(FSignedDataBuffer), Length(FSignedDataBuffer)),
			PGPOLastOption(FContext))
      ]);
  end
  else begin
    if FFileOutput then begin
      if GetOutputFileName = '' then begin
	Result := kPGPError_CantOpenFile;
	Exit;
      end;
    end;
    if FFileToFile then begin
      FileSpec := nil;
      Result := PGPNewFileSpecFromFullPath(FContext, PChar(FOutputFileName), FileSpec);
      try
	if Result = 0 then begin
	  Result := PGPAppendOptionList(OptionList,
	    [
	      PGPOOutputFile(FContext, FileSpec),
	      PGPOAppendOutput(FContext, PGPTrue)
	    ]);
	end;
      finally
	PGPFreeFileSpec(FileSpec);
      end;
    end
    else begin
      if FClearSigned and (PassThrough_ClearSigned in PassThroughOptions) then
	Result := PGPAppendOptionList(OptionList, [PGPODiscardOutput(FContext, PGPTrue)]) // for bugfix
      else begin
	Result := PGPAppendOptionList(OptionList,
	  [
	    PGPOAllocatedOutputBuffer(FContext, FAllocatedOutputBuffer, $FFFFFFFF, FActualOutputSize)
	  ]);
      end;
    end;
  end;
end;

function TPGPDecodeCustom.PrepareOutput(const Input: PChar; InputSize: DWord; const MimeHeaders, MimeBody: String): PGPError;
var
  HeaderSize	: DWord;
begin
  Result := 0;
  if not FFileToFile then begin
    if not (FDetachedSigVerify or FClearSigned)
    or (FIsPGPMime and not (PassThrough_ClearSigned in PassThroughOptions)) then begin
      Result := SecureMove;
      if Result <> 0 then Exit;
      HeaderSize := Length(MimeHeaders);
      if (FIsPGPMime or (FX509Mode = X509Encrypted)) and (HeaderSize > 0) then begin
	FActualOutputSize := HeaderSize + FSecureBufferSize;
	FAllocatedOutputBuffer := PGPNewSecureData(PGPGetDefaultMemoryMgr, FActualOutputSize, kPGPMemoryMgrFlags_None);
	if FAllocatedOutputBuffer <> nil then begin
	  Move(MimeHeaders[1], FAllocatedOutputBuffer[0], HeaderSize);
	  Move(FSecureBuffer[0], FAllocatedOutputBuffer[HeaderSize], FSecureBufferSize);
	  PGPFreeData(FSecureBuffer);
	  FSecureBuffer := nil;
	  FSecureBufferSize := 0;
	end
	else Result := kPGPError_OutOfMemory;
      end
      else begin
	FAllocatedOutputBuffer := FSecureBuffer;
	FActualOutputSize := FSecureBufferSize;
      end;
    end;
    // fixes for signed data
    if (FClearSigned or (FX509Mode = X509IsMime))
    and (PassThrough_ClearSigned in PassThroughOptions) then begin
      FActualOutputSize := InputSize;
      Result := DataCopy(FAllocatedOutputBuffer, Input, FActualOutputSize);
    end
    else if (FX509Mode in [X509IsMime..X509Signed]) and (FActualOutputSize = 0) then begin
      FActualOutputSize := Length(MimeHeaders) + Length(MimeBody);
      Result := DataCopy(FAllocatedOutputBuffer, PChar(MimeHeaders + MimeBody), FActualOutputSize);
    end;
  end;
end;

function TPGPDecodeCustom.DoRecursion(const Input: PChar; InputSize: DWord): PGPError;
begin
  FFileNameBuffer := FInputFileName;
  if FFileToFile then begin
    SetLength(FInputFileName, MAX_PATH);
    if GetTempFileName(PChar(ExtractFilePath(FOutputFileName)), 'PGP', 0, PChar(FInputFileName)) <> 0 then begin
      Delete(FInputFileName, succ(StrLen(Pointer(FInputFileName))), MAXINT);
      if MoveFileEx(PChar(FOutputFileName), PChar(FInputFileName), MOVEFILE_REPLACE_EXISTING) then begin
	FInputBuffer := PChar(FInputFileName);
	FInputSize := Length(FInputFileName);
	Result := Decode(true);
	if Result <> 0 then DeleteFile(FInputFileName);
      end
      else begin
	Result := kPGPError_FileOpFailed;
	DeleteFile(FOutputFileName);
	DeleteFile(FInputFileName);
	Exit;
      end;
    end
    else begin
      Result := kPGPError_FileOpFailed;
      DeleteFile(FOutputFileName);
      Exit;
    end;
  end
  else begin
    FInputBuffer := Input;
    FInputSize := InputSize;
    Result := Decode(false);
  end;
end;

function TPGPDecodeCustom.SecureView(const Output: PChar; OutputSize: DWord): PGPError;
var
  PGPsc		: pPGPsc;
  PGPtls	: pPGPtls;
  Utf8OutputSize: DWord;
  Utf8Output	: PChar;
begin
  PGPsc := nil;
  PGPtls := nil;
  if InitPGPsc(FParentHandle, PGPsc, PGPtls, 0) then begin
    try
      if PGP9X then begin
	Utf8OutputSize := OutputSize * UTF8Factor;
	Utf8Output := PGPNewSecureData(PGPGetDefaultMemoryMgr, Utf8OutputSize, kPGPMemoryMgrFlags_None);
	try
	  Utf8OutputSize := SecureAnsiToUtf8PChar(Output, Utf8Output, Utf8OutputSize);
	  Result := TempestViewer(PGPsc, FParentHandle, Utf8Output, Utf8OutputSize, true);
	finally
	  FillChar(Utf8Output^, Utf8OutputSize, 0);
	  PGPFreeData(Utf8Output);
	end;
      end
      else Result := TempestViewer(PGPsc, FParentHandle, Output, OutputSize, true);
      ProcessMessages;
    finally
      UninitPGPsc(FParentHandle, PGPsc, PGPtls);
    end;
  end
  else Result := kPGPError_FeatureNotAvailable;
end;

function TPGPDecodeCustom.WriteOutputFile(const Output: PChar; OutputSize: DWord): PGPError;
var
  OutputFile	: THandle;
  BytesWritten	: DWord;
begin
  OutputFile := CreateFile(PChar(FOutputFileName), GENERIC_WRITE, 0, nil, CREATE_ALWAYS,
			   FILE_ATTRIBUTE_ARCHIVE or FILE_FLAG_WRITE_THROUGH, 0);
  if OutputFile <> INVALID_HANDLE_VALUE then begin
    try
      if not WriteFile(OutputFile, Output[0], OutputSize, BytesWritten, nil) or (BytesWritten <> OutputSize) then
	Result := kPGPError_WriteFailed
      else Result := kPGPError_NoErr;
    finally
      CloseHandle(OutputFile);
    end;
  end
  else Result := kPGPError_CantOpenFile;
end;

function TPGPDecodeCustom.Decode(IsFile: Longbool): Longint;
var
  BugFix	: Longbool;
  Buffered	: Longbool;
  Recursively	: Longbool;
  OptionList	: pPGPOptionList;
  InFileSpec	: pPGPFileSpec;
  Output	: PChar;
  OutputSize	: DWord;
  Input		: PChar;
  InputSize	: DWord;
  MimeHeaders	: String;
  MimeBody	: String;
  MimeSig	: String;
  Dummy		: Longbool;
  OutputBuffer	: PChar;
  BufferSize	: DWord;
begin
  Result := 0;
  BugFix := false;
  Buffered := false;
  Output := nil;
  OutputSize := 0;
  if IsFile then
    Result := HandleFileInput
  else if FInputSize = 0 then Result := kPGPError_ItemNotFound;
  if Result <> 0 then Exit;
  Input := FInputBuffer;
  InputSize := FInputSize;
  try
    InFileSpec := nil;
    OptionList := nil;
    Result := InitDecode;
    if Result <> 0 then Exit;
    if IsFile then begin
      if PGP7X and not FFileToFile then begin
	Result := ReadInputFile;
	if Result <> 0 then Exit;
	Input := FInputBuffer;
	InputSize := FInputSize;
	Buffered := true;
	IsFile := false;
      end
      else begin
	Result := PGPNewFileSpecFromFullPath(FContext, PChar(FInputFileName), InFileSpec);
	if Result <> 0 then Exit;
      end;
    end;
    try
      Result := Analyze(InFileSpec);
      if FPassOption in [AnalyseOnly, AnalysePass] then begin
	if Result <> kPGPError_AsciiParseIncomplete then Result := 0; // assume FAnalysis = UNKNOWN on other errors
	if (Result = 0) and not FRecognized then begin
	  if (FX509Mode = X509IsMime) and not IsFile then begin
	    case GetPSMimeType(FInputBuffer, MimeHeaders, MimeBody, MimeSig, true) of
	      psClearSigned, psSigned: FAnalysis := SMIME_SIGNED;
	      psEncrypted: FAnalysis := SMIME_ENCRYPTED;
	    else
	      FX509Mode := X509Test;
	      case FX509Mode of
		X509Secret: FAnalysis := X509_SECRETKEY;
		X509Binary..X509Armored: FAnalysis := X509_CERTIFICATE;
	      end;
	    end;
	  end;
	  if Assigned(FOnAnalysis) then FOnAnalysis(0, FAnalysis, Dummy);
	end;
	FRecognized := (FAnalysis <> UNKNOWN);
      end;
      if FPassOption <> AnalyseOnly then begin
	if (FAnalysis = UNKNOWN) and not IsFile and (FInputSize > 0) and (FInputBuffer[0] > #127) then begin
	  Result := FixNonAsciiBug(Input);
	  if Result <> 0 then Exit;
	  BugFix := true;
	end;
	// X509 and S/MIME handling
	if PGP81 and not IsFile then begin
	  case FX509Mode of
	    X509Off: FX509Mode := X509Processed;
	    X509Secret..X509Armored: Exit;
	    X509IsMime..X509Encrypted: begin
	      case GetPSMimeType(FInputBuffer, MimeHeaders, MimeBody, MimeSig, false) of
		psClearSigned: begin
		  FAnalysis := SMIME_SIGNED;
		  FX509Mode := X509IsMime;
		  FSignedDataBuffer := MimeBody;
		  Input := FInputBuffer;
		  InputSize := FInputSize;
		  FInputSize := Length(MimeSig);
		  DataCopy(FInputBuffer, PChar(MimeSig), FInputSize);
		end;
		psEncrypted: begin
		  FAnalysis := SMIME_ENCRYPTED;
		  FX509Mode := X509Encrypted;
		  Input := FInputBuffer;
		  InputSize := FInputSize;
		  FInputSize := Length(MimeBody);
		  DataCopy(FInputBuffer, PChar(MimeBody), FInputSize);
		end;
		psSigned: begin
		  FAnalysis := SMIME_SIGNED;
		  FX509Mode := X509Signed;
		  Input := FInputBuffer;
		  InputSize := FInputSize;
		  FInputSize := Length(MimeBody);
		  DataCopy(FInputBuffer, PChar(MimeBody), FInputSize);
		end;
		psPGP: begin
		  FIsPGPMime := true;
		  FX509Mode := X509Processed;
		  Input := FInputBuffer;
		  InputSize := FInputSize;
		  FInputSize := Length(MimeBody);
		  DataCopy(FInputBuffer, PChar(MimeBody), FInputSize);
		end;
	      else
		FX509Mode := X509Processed;
	      end;
	    end;
	  end;
	end
	else FX509Mode := X509Processed;
	// PassThrough_ClearSigned may truncate signed data,
	// RecursivelyDecode enables PassThrough_ClearSigned,
	// if we already have clear signed data just disable it
	Recursively := FRecursivelyDecode;
	if FClearSigned then FRecursivelyDecode := false;
	Result := GetOptionList(OptionList);
	if Result <> 0 then Exit;
	if IsFile then
	  Result := PGPAppendOptionList(OptionList, [PGPOInputFile(FContext, InFileSpec)])
	else Result := PGPAppendOptionList(OptionList, [PGPOInputBuffer(FContext, FInputBuffer, FInputSize)]);
	try
	  if Result <> 0 then Exit;
	  if FDetachedSigVerify or FClearSigned or (FX509Mode = X509IsMime) then begin
	    Result := SetOutputOption(OptionList);
	    if Result <> 0 then Exit;
	  end
	  else FOutputExists := false;
	  if not FRecurring or FClearSigned or (FX509Mode in [X509IsMime..X509Signed]) then begin
	    FPassOption := DecodePass;
	    Result := pgpEvents.PGPDecode(FContext, OptionList, PGPOLastOption(FContext));
	    if Result <> 0 then Exit;
	    if FPassOption <> AnalyseOnly then begin
	      Result := PrepareOutput(Input, InputSize, MimeHeaders, MimeBody);
	      Output := FAllocatedOutputBuffer;
	      OutputSize := FActualOutputSize;
	      if BugFix and (PassThrough_Unrecognized in PassThroughOptions) then begin // remove leading AT
		dec(OutputSize);
		inc(Output);
	      end;
	    end;
	  end;
	finally
	  try
	    if Assigned(FOnWipePassphrase) then begin
	      if PGP8X then SecureUtf8ToAnsiPChar(FPassphrase, FPassphrase, 256 * UTF8Factor);
	      FOnWipePassphrase(FPassphrase);
	    end;
	    if FPassCacheSeconds <> 0 then CachePassphrase(FContext, FDecryptionKey, FPassCacheSeconds, FPassCacheShare, FPassphrase);
	  finally
	    PGPFreeData(FPassphrase);
	    FPassphrase := nil;
	  end;
	end;
	// reset after bugfixing
	FRecursivelyDecode := Recursively;
      end;
    finally
      PGPFreeFileSpec(InFileSpec);
      PGPFreeOptionList(OptionList);
      if BugFix then begin
	PGPFreeData(FInputBuffer);
	FInputBuffer := Input;
	FInputSize := InputSize;
      end;
    end;
    // workaround for verifying encrypted PGP/MIME & S/MIME signatures like PGP does internally with encrypted clear signed data
    if FRecursivelyDecode and not FRecurring and (FPassOption <> AnalyseOnly)
    and (FAnalysis in [ENCRYPTED, SMIME_ENCRYPTED]) and (FSigPropsRec.sStatus = SIGNED_NOT) then begin
      if FIsPGPMime then FX509Mode := X509IsMime;
      OutputBuffer := FAllocatedOutputBuffer;
      BufferSize := FActualOutputSize;
      FPassOption := AnalysePass;
      FRecurring := true;
      try
	Result := DoRecursion(Output, OutputSize);
      finally
	if FEyesOnly and (OutputBuffer <> nil) then FillChar(OutputBuffer^, BufferSize, 0);
	PGPFreeData(OutputBuffer);
	// reset to start value
	if FAnalysis in [SMIME_UNRECOGNIZED, SMIME_SIGNED] then
	  FAnalysis := SMIME_ENCRYPTED
	else FAnalysis := ENCRYPTED;
	FRecurring := false;
      end;
    end
    else if FFileToFile then begin
      if FRecurring then begin
	if FAnalysis = SIGNED then
	  DeleteFile(FInputFileName)
	else if not RenameFile(FInputFileName, FOutputFileName) then begin
	  Result := kPGPError_FileOpFailed;
	  DeleteFile(FInputFileName);
	end;
      end;
    end
    else begin
      if FRecurring then begin
	if FSigPropsRec.sStatus = SIGNED_NOT then begin
	  Output := Input;
	  OutputSize := InputSize;
	end
	else if FAnalysis = SMIME_UNRECOGNIZED then begin
	  Output := PChar(MimeHeaders + MimeBody);
	  OutputSize := Length(MimeHeaders) + Length(MimeBody);
	end;
      end;
      if FEyesOnly then
	SecureView(Output, OutputSize)
      else if not FFileOutput then begin
	try
	  SetLength(FOutputBuffer, OutputSize);
	  Move(Output^, Pointer(FOutputBuffer)^, OutputSize);
	except
	  on EOutOfMemory do Result := kPGPError_OutOfMemory;
	end;
      end
      else WriteOutputFile(Output, OutputSize);
    end;
  finally
    if Buffered then PGPFreeData(Input);
    try
      if FEyesOnly and (FAllocatedOutputBuffer <> nil) then FillChar(FAllocatedOutputBuffer^, FActualOutputSize, 0);
    finally
      try
	PGPFreeData(FAllocatedOutputBuffer);
	FAllocatedOutputBuffer := nil;
      finally
	FinitDecode;
      end;
    end;
  end;
end;

function TPGPDecodeCustom.AnalyseBuffer(const DataBuffer: String): Longint;
begin
  FPassOption := AnalyseOnly;
  FInputBuffer := PChar(DataBuffer);
  FInputSize := Length(DataBuffer);
  FInputFileName := '';
  FOutputOptions := nil;
  try
    Result := Decode(false);
  finally
    PGPFreeOptionList(FOutputOptions);
  end;
end;

function TPGPDecodeCustom.AnalyseFile(const FileName: String): Longint;
begin
  FPassOption := AnalyseOnly;
  FInputBuffer := PChar(FileName);
  FInputSize := Length(FileName);
  FOutputOptions := nil;
  try
    Result := Decode(true);
  finally
    PGPFreeOptionList(FOutputOptions);
  end;
end;

function TPGPDecodeCustom.AnalyseMimeBuffer(const DataBuffer: String): Longint;
begin
  FX509Mode := X509IsMime;
  FPassOption := AnalyseOnly;
  FInputBuffer := PChar(DataBuffer);
  FInputSize := Length(DataBuffer);
  FInputFileName := '';
  FOutputOptions := nil;
  try
    Result := Decode(false);
  finally
    PGPFreeOptionList(FOutputOptions);
  end;
end;

function TPGPDecodeCustom.AnalyseMimeFile(const FileName: String): Longint;
begin
  FX509Mode := X509IsMime;
  FPassOption := AnalyseOnly;
  FInputBuffer := PChar(FileName);
  FInputSize := Length(FileName);
  FOutputOptions := nil;
  try
    Result := Decode(true);
  finally
    PGPFreeOptionList(FOutputOptions);
  end;
end;

function TPGPDecodeCustom.DecodeBuffer(const DataBuffer: String): Longint;
begin
  FPassOption := AnalysePass;
  FInputBuffer := PChar(DataBuffer);
  FInputSize := Length(DataBuffer);
  FInputFileName := '';
  FOutputOptions := nil;
  try
    Result := Decode(false);
  finally
    PGPFreeOptionList(FOutputOptions);
  end;
end;

function TPGPDecodeCustom.DecodeFile(const FileName: String): Longint;
begin
  FPassOption := AnalysePass;
  FInputBuffer := PChar(FileName);
  FInputSize := Length(FileName);
  FOutputOptions := nil;
  try
    Result := Decode(true);
  finally
    PGPFreeOptionList(FOutputOptions);
  end;
end;

function TPGPDecodeCustom.DecodeMimeBuffer(const DataBuffer: String): Longint;
begin
  FX509Mode := X509IsMime;
  FPassOption := AnalysePass;
  FInputBuffer := PChar(DataBuffer);
  FInputSize := Length(DataBuffer);
  FInputFileName := '';
  FOutputOptions := nil;
  try
    Result := Decode(false);
  finally
    PGPFreeOptionList(FOutputOptions);
  end;
end;

function TPGPDecodeCustom.DecodeMimeFile(const FileName: String): Longint;
begin
  FX509Mode := X509IsMime;
  FPassOption := AnalysePass;
  FInputBuffer := PChar(FileName);
  FInputSize := Length(FileName);
  FOutputOptions := nil;
  try
    Result := Decode(true);
  finally
    PGPFreeOptionList(FOutputOptions);
  end;
end;

function TPGPDecodeCustom.AnalyzeHandler(Event: pPGPEvent): PGPError;
var
  SkipSection	: Longbool;
begin
  Result := 0;
  SkipSection := false;
  with Event^.EData.AnalyzeData do begin
    case FPassOption of
      AnalyseOnly: begin
	FRecognized := true;
	if Assigned(FOnAnalysis) then FOnAnalysis(FSectionNumber, TAnalysis(SectionType), SkipSection);
	if SectionType <> kPGPAnalyze_Key then Result := kPGPError_SkipSection; // avoid key data bug
      end;
      AnalysePass: begin
	FRecognized := (SectionType <> kPGPAnalyze_Unknown);
	case SectionType of
	  kPGPAnalyze_Signed: FClearSigned := true;
	  kPGPAnalyze_DetachedSignature: FDetachedSigVerify := true;
	end;
	if FRecognized then FAnalysis := TAnalysis(SectionType);
	if FDetachedSigVerify or FClearSigned or (FX509Mode >= X509IsMime) then
	  Result := kPGPError_UserAbort
	else if SectionType <> kPGPAnalyze_Key then Result := kPGPError_SkipSection; // avoid key data bug
      end;
      DecodePass: begin
	if (SectionType <> kPGPAnalyze_Unknown) and (FX509Mode < X509IsMime) then FAnalysis := TAnalysis(SectionType);
	if Assigned(FOnAnalysis) then begin
	  if FX509Mode >= X509IsMime then
	    FOnAnalysis(FSectionNumber, FAnalysis, SkipSection)
	  else FOnAnalysis(FSectionNumber, TAnalysis(SectionType), SkipSection)
	end;
	Result := (ord(SkipSection) and 1) * kPGPError_SkipSection;
      end;
    end;
  end;
end;

function TPGPDecodeCustom.RecipientsHandler(Event: pPGPEvent): PGPError;
var
  KeyIDArraySize: PGPSize;
begin
  Result := 0;
  if FAnalysis = SMIME_UNRECOGNIZED then FAnalysis := SMIME_ENCRYPTED;
  with Event^.EData.RecipientsData do begin
    PGPIncKeySetRefCount(RecipientSet);
    FRecipientsData.RecipientSet := RecipientSet;
    FRecipientsData.KeyCount := KeyCount;
    if PGP7X then
      KeyIDArraySize := SizeOf(TPGPKeyID7) * KeyCount
    else KeyIDArraySize := SizeOf(TPGPKeyID6) * KeyCount;
    FRecipientsData.KeyIDArray := PGPNewData(PGPGetDefaultMemoryMgr, KeyIDArraySize, kPGPMemoryMgrFlags_None);
    if FRecipientsData.KeyIDArray <> nil then Move(KeyIDArray^, FRecipientsData.KeyIDArray^, KeyIDArraySize)
  end;
end;

function TPGPDecodeCustom.PassphraseHandler(Event: pPGPEvent): PGPError;
var
  Cancel	: Longbool;
  KeyPropsList	: TKeyPropsList;
  BadPassphrase	: Longbool;
begin
  Result := 0;
  Cancel := false;
  with Event^.EData.PassphraseData do begin
    KeyPropsList := TKeyPropsList.Create(spgpKeyPropFlag_IDComplete);
    try
      if (FPassphrase = nil) or (FSectionNumber = FPassSection) then begin
	FPassSection := FSectionNumber;
	BadPassphrase := false;
	if FPassphrase <> nil then begin
	  PGPFreeData(FPassphrase);
	  FPassphrase := nil;
	  BadPassphrase := true;
	end;
	if Boolean(IsConventional) or (FPassphrase <> nil) or (FPassCacheSeconds = 0)
	or not GetCachedPassphrase(FContext, FDecryptionKey, KeySet, FPassphrase) then begin
	  if Assigned(FOnEnterPassphrase) then begin
	    if not Boolean(IsConventional) then begin
	      Result := GetKeySetProps(FContext, KeySet, KeyPropsList, spgpKeyPropFlag_IDComplete,
				       KeyFilterFlag_CanDecrypt, UserID_Ordering);
	      if Result = 0 then Result := kPGPError_SecretKeyNotFound;
	    end;
	    if Result >= 0 then begin
	      Result := kPGPError_UserAbort;
	      FPassphrase := PGPNewSecureData(PGPGetDefaultMemoryMgr, 256 * UTF8Factor, kPGPMemoryMgrFlags_Clear);
	      if FPassphrase <> nil then begin
		FOnEnterPassphrase(FPassphrase, KeyPropsList, Boolean(IsConventional), BadPassphrase, Cancel);
		if not Cancel then begin
		  if PGP8X then SecureAnsiToUtf8PChar(FPassphrase, FPassphrase, 256 * UTF8Factor);
		  if KeyPropsList.Count = 1 then
		    GetKeyFromKeySet(KeySet, FDecryptionKey)
		  else FDecryptionKey := GetMatchingKey(FContext, KeySet, FPassphrase);
		end
		else Exit;
	      end
	      else begin
		Result := kPGPError_OutOfMemory;
		Exit;
	      end;
	    end
	    else Exit;
	  end
	  else begin
	    if Boolean(IsConventional) then
	      Result := ConvDecPassphraseDialog(FContext, FPassphrase, FPassDlgPrompt, FParentHandle)
	    else with FRecipientsData do begin
	      Result := DecryptionPassphraseDialog(FContext, RecipientSet, KeyCount, KeyIDArray,
						   FDecryptionKey, FPassphrase, true, FPassDlgPrompt, FParentHandle);
	    end;
	    if Result <> 0 then Exit;
	  end;
	end;
      end
      else FPassSection := FSectionNumber;
    finally
      KeyPropsList.Free;
    end;
  end;
  Result := PGPAddJobOptions(Event^.Job, PGPOPassphrase(FContext, FPassphrase), PGPOLastOption(FContext));
end;

function TPGPDecodeCustom.KeyFoundHandler(Event: pPGPEvent): PGPError;
var
  IsSMime	: Longbool;
  KeySetSelected: pPGPKeySet;
  CertsToImport	: pPGPKeySet;
begin
  Result := 0; // don't cancel decoding whatever happens here
  with Event^.EData.KeyFoundData do begin
    IsSMime := (FX509Mode in [X509IsMime..X509Signed]);
    if IsSMime then FValidCertChain := VerifyX509CertificateChainDB(KeysFound);
    if FQueryAddKeys then begin
      if PGP7X then begin
	KeySetSelected := PGPPeekKeyDBRootKeySet(KeysFound);
	FKeyAddResult := GetExclusiveKeySet(KeySetSelected, FKeySetMain, FContext, FIgnoreKnownFlag);
	if FKeyAddResult <> 0 then Exit;
	if FKeyDlgPrompt <> '' then begin
	  if not IsSMime then
	    CertsToImport := PGPPeekKeyDBRootKeySet(KeySetSelected)
	  else begin
	    FKeyAddResult := GetCertSetFromKeySet(PGPPeekKeyDBRootKeySet(KeysFound), CertsToImport);
	    if FKeyAddResult <> 0 then Exit;
	  end;
	  try
	    FKeyAddResult := SelectAddKeysToKeyRing(FContext, CertsToImport, FKeySetMain,
						    FKeyPropsList, PChar(FKeyDlgPrompt),
						    false, spgpKeyPropFlag_IDFlags,
						    FParentHandle);
	  finally
	    if IsSMime then PGPFreeKeySet(CertsToImport);
	  end;
	end
	else FKeyAddResult := AddKeysToKeyRing(FContext, FKeySetMain, KeySetSelected, FKeyPropsList, spgpKeyPropFlag_IDFlags);
      end
      else begin
	FKeyAddResult := GetExclusiveKeySet(KeysFound, FKeySetMain, FContext, FIgnoreKnownFlag);
	if FKeyAddResult <> 0 then Exit;
	if FKeyDlgPrompt <> '' then begin
	  FKeyAddResult := SelectAddKeysToKeyRing(FContext, KeysFound, FKeySetMain,
						  FKeyPropsList, PChar(FKeyDlgPrompt),
						  false, spgpKeyPropFlag_IDFlags,
						  FParentHandle);
	end
	else FKeyAddResult := AddKeysToKeyRing(FContext, FKeySetMain, KeysFound, FKeyPropsList, spgpKeyPropFlag_IDFlags);
      end;
      if FKeyAddResult > 0 then FKeyAddResult := 0;
    end;
  end;
end;

// causes trouble when used with PassThrough_ClearSigned on clear signed input
// causes trouble with PassThrough_Unrecognized and local output options
function TPGPDecodeCustom.OutputHandler(Event: pPGPEvent): PGPError;
var
  FileSpec	: pPGPFileSpec;
begin
  Result := 0;
  with Event^.EData.OutputData do begin
    if (ForYourEyesOnly = PGPTrue) and not FFileToFile then FEyesOnly := true;
    if FOutputExists then begin
      if not FFileToFile then Result := SecureMove;
    end
    else if FFileOutput and not FEyesOnly then begin
      if (FOutputFileName = '') and (SuggestedName <> '') then begin
	FOutputFileName := ExtractFilePath(FInputFileName) + SuggestedName;
      end;
      if GetOutputFileName = '' then begin
	Result := kPGPError_CantOpenFile;
	Exit;
      end;
      if FFileToFile then begin
	FileSpec := nil;
	Result := PGPNewFileSpecFromFullPath(FContext, PChar(FOutputFileName), FileSpec);
	try
	  if Result = 0 then begin
	    Result := PGPBuildOptionList(FContext, FOutputOptions,
	      [
		PGPOOutputFile(FContext, FileSpec),
		PGPOAppendOutput(FContext, PGPTrue)
	      ]);
	  end;
	finally
	  PGPFreeFileSpec(FileSpec);
	end;
      end
      else begin
	Result := PGPBuildOptionList(FContext, FOutputOptions,
	  [
	    PGPOAllocatedOutputBuffer(FContext, FAllocatedOutputBuffer, $FFFFFFFF, FActualOutputSize)
	  ]);
      end;
    end
    else begin
      FFileToFile := false;
      Result := PGPBuildOptionList(FContext, FOutputOptions,
	[
	  PGPOAllocatedOutputBuffer(FContext, FAllocatedOutputBuffer, $FFFFFFFF, FActualOutputSize)
	]);
    end;
  end;
  if Result = 0 then begin
    Result := PGPAddJobOptions(Event^.Job, FOutputOptions, PGPOLastOption(FContext));
    FOutputExists := true;
  end;
end;

function GetTHashAlgorithm(HashAlg: PGPHashAlgorithm): THashAlgorithm;
begin
  Result := THashAlgorithm(
    HashAlg - ord(HashAlg > kPGPHashAlgorithm_RIPEMD160) * pred(kPGPHashAlgorithm_SHA256 - kPGPHashAlgorithm_RIPEMD160) + 1
  );
end;

function TPGPDecodeCustom.SignatureHandler(Event: pPGPEvent): PGPError;
var
  SignKeyType	: PGPInt32;
  SignKeyOnRing	: pPGPKeyDBObj;
  PGPSignKeyID	: TPGPKeyID7;
  SignKeyID	: TKeyID;
  KeysFound	: Pointer;
begin
  Result := 0; // don't cancel decoding whatever happens here
  ClearSigPropsRec;
  if FAnalysis = SMIME_UNRECOGNIZED then FAnalysis := SMIME_SIGNED;
  if PGP7X then with Event^.EData.SignatureData.PGPEventSignatureData7, FSigPropsRec do begin
    if PGP7X and (PGPGetKeyDBObjNumericProperty(SigningKey, kPGPKeyDBObjProperty_ObjectType, SignKeyType) = 0) then begin
      if SignKeyType = kPGPKeyDBObjType_SubKey then SigningKey := PGPPeekKeyDBObjKey(SigningKey);
    end;
    if Boolean(Checked) then begin
      sHexID := GetKeyPropKeyID(SigningKey);
      sUserID := GetKeyPropUserID(SigningKey);
      sCreaTimeNum := PGPGetStdTimeFromPGPTime(CreationTime);
      sCreaTimeStr := UnixTimeToLocalTimeStr(sCreaTimeNum);
      sChecked := Boolean(Checked);
      sVerified := Boolean(Verified);
      // workaround for S/MIME signatures
      if PGPKeySetIsMember(SigningKey, FKeySetMain) = PGPFalse then begin
	GetKeyFromNewSet(SigningKey, FKeySetMain, SignKeyOnRing);
	sValidity := GetKeyPropValidity(SignKeyOnRing);
      end
      else sValidity := TValidityLevel(KeyValidity);
      sRevoked := Boolean(KeyRevoked);
      sDisabled := Boolean(KeyDisabled);
      sExpired := Boolean(KeyExpired);
      sAxiomatic := GetKeyPropIsAxiomatic(SigningKey);
      if PGP8X and (ExpirationPeriod <> 0) then begin
	sExpTimeNum := sCreaTimeNum + ExpirationPeriod;
	sExpTimeStr := UnixTimeToLocalTimeStr(sExpTimeNum);
      end;
      if PGP9X then sHashAlgorithm := GetTHashAlgorithm(HashAlgorithm);
      if GetKeyPropIsSigningKey(SigningKey) then begin
	sStatus := TSigStatus(Checked + Verified);
	if PGP8X and (sStatus = SIGNED_GOOD) and (ExpirationPeriod <> 0)
	and (sExpTimeNum < SystemTimeToUnixTimeNum) then begin
	  sStatus := SIGNED_SIG_EXPIRED;
	end;
      end
      else sStatus := SIGNED_UNKNOWN_ALG;
      sValidCertChain := FValidCertChain;
    end
    else begin
      Move(SigningKeyID, PGPSignKeyID, SizeOf(TPGPKeyID7));
      Result := PGPGetKeyIDString(PGPSignKeyID, kPGPKeyIDString_Full, SignKeyID);
      if Result = 0 then begin
	sCreaTimeNum := PGPGetStdTimeFromPGPTime(CreationTime);
	sCreaTimeStr := UnixTimeToLocalTimeStr(sCreaTimeNum);
	sHexID := SignKeyID;
      end;
      sStatus := SIGNED_NO_KEY;
      if (Result = 0) and FGetKeyFromServer then begin
	KeysFound := nil;
	Result := KeyServerDialog(FContext, FKeySetMain, SignKeyID, pPGPKeyDB(KeysFound), FParentHandle);
	if (Result = 0) and (KeysFound <> nil) then begin
	  try
	    if FKeyDlgPrompt <> '' then begin
	      SelectAddKeysToKeyRing(FContext, PGPPeekKeyDBRootKeySet(pPGPKeyDB(KeysFound)), FKeySetMain,
				     FKeyPropsList, PChar(FKeyDlgPrompt),
				     false, spgpKeyPropFlag_IDFlags,
				     FParentHandle);
	    end;
	    PGPAddJobOptions(Event^.Job,
			     PGPOKeySetRef(FContext, PGPPeekKeyDBRootKeySet(pPGPKeyDB(KeysFound))),
			     PGPOLastOption(FContext));
	  finally
	    PGPFreeKeyDB(pPGPKeyDB(KeysFound));
	  end;
	end;
      end;
      Result := 0;
    end;
    sDetached := FDetachedSigVerify;
  end
  else with Event^.EData.SignatureData.PGPEventSignatureData6, FSigPropsRec do begin
    if Boolean(Checked) then begin
      sHexID := GetKeyPropKeyID(SigningKey);
      sUserID := GetKeyPropUserID(SigningKey);
      sCreaTimeNum := PGPGetStdTimeFromPGPTime(CreationTime);
      sCreaTimeStr := UnixTimeToLocalTimeStr(sCreaTimeNum);
      sChecked := Boolean(Checked);
      sVerified := Boolean(Verified);
      sValidity := TValidityLevel(KeyValidity);
      sRevoked := Boolean(KeyRevoked);
      sDisabled := Boolean(KeyDisabled);
      sExpired := Boolean(KeyExpired);
      sAxiomatic := GetKeyPropIsAxiomatic(SigningKey);
      if GetKeyPropIsSigningKey(SigningKey) then
	sStatus := TSigStatus(Checked + Verified)
      else sStatus := SIGNED_UNKNOWN_ALG;
    end
    else begin
      Move(SigningKeyID, PGPSignKeyID, SizeOf(TPGPKeyID6));
      Result := PGPGetKeyIDString(PGPSignKeyID, kPGPKeyIDString_Full, SignKeyID);
      if Result = 0 then begin
	sCreaTimeNum := PGPGetStdTimeFromPGPTime(CreationTime);
	sCreaTimeStr := UnixTimeToLocalTimeStr(sCreaTimeNum);
	sHexID := SignKeyID;
      end;
      sStatus := SIGNED_NO_KEY;
      if (Result = 0) and FGetKeyFromServer then begin
	KeysFound := nil;
	Result := KeyServerDialog(FContext, FKeySetMain, SignKeyID, pPGPKeySet(KeysFound), FParentHandle);
	if (Result = 0) and (KeysFound <> nil) then begin
	  try
	    if FKeyDlgPrompt <> '' then begin
	      SelectAddKeysToKeyRing(FContext, pPGPKeySet(KeysFound), FKeySetMain,
				     FKeyPropsList, PChar(FKeyDlgPrompt),
				     false, spgpKeyPropFlag_IDFlags,
				     FParentHandle);
	    end;
	    PGPAddJobOptions(Event^.Job,
			     PGPOKeySetRef(FContext, pPGPKeySet(KeysFound)),
			     PGPOLastOption(FContext));
	  finally
	    PGPFreeKeySet(pPGPKeySet(KeysFound));
	  end;
	end;
      end;
      Result := 0;
    end;
    sDetached := FDetachedSigVerify;
  end;
  if FShowSigLog then ShowSigInfo;
  if Assigned(FOnGetSignature) then FOnGetSignature(SigPropsRec);
end;

function TPGPDecodeCustom.EndLexHandler(Event: pPGPEvent): PGPError;
begin
  Result := 0;
  with FRecipientsData do begin
    PGPFreeData(KeyIDArray);
    PGPFreeKeySet(RecipientSet);
  end;
  FillChar(FRecipientsData, SizeOf(TPGPEventRecipientsData), 0);
end;

end.

